# -*- encoding: UTF-8 -*-
# Lightproof grammar checker for LibreOffice and OpenOffice.org
# http://launchpad.net/lightproof
# version 1.4.3 (2011-12-05)
#
# 2009-2011 (c) László Németh (nemeth at numbertext org), license: MPL 1.1 / GPLv3+ / LGPLv3+

import uno, unohelper, sys, traceback, re
from lightproof_impl_pt_BR import locales
from lightproof_impl_pt_BR import pkg
import lightproof_handler_pt_BR

from com.sun.star.linguistic2 import XProofreader, XSupportedLocales
from com.sun.star.linguistic2 import ProofreadingResult, SingleProofreadingError
from com.sun.star.lang import XServiceInfo, XServiceName, XServiceDisplayName
from com.sun.star.lang import Locale
from com.sun.star.text.TextMarkupType import PROOFREADING
from com.sun.star.beans import PropertyValue

# loaded rules
langrule = {}
# ignored rules
ignore = {}

# cache for morphogical analyses
analyses = {}
stems = {}
suggestions = {}

# assign Calc functions
calcfunc = None

# check settings
def option(lang, opt):
    return lightproof_handler_pt_BR.get_option(lang.Language + "_" + lang.Country, opt)

# filtering affix fields (ds, is, ts etc.)
def onlymorph(st):
    if st != None:
        st = re.sub(r"^.*(st:|po:)", r"\1", st) # keep last word part
        st = re.sub(r"\b(?=[dit][sp]:)","@", st) # and its affixes
        st = re.sub(r"(?<!@)\b\w\w:\w+","", st).replace('@','').strip()
    return st

# if the pattern matches all analyses of the input word, 
# return the last matched substring
def _morph(rLoc, word, pattern, all, onlyaffix):
    global analyses
    if word == None:
        return None
    if word not in analyses:
        x = spellchecker.spell(u"<?xml?><query type='analyze'><word>" + word + "</word></query>", rLoc, ())
        if not x:
            return None
        t = x.getAlternatives()
        if not t:
            t = [""]
        analyses[word] = t[0]
    a = analyses[word].split("</a>")[:-1]
    result = None
    p = re.compile(pattern)
    for i in a:
        if onlyaffix:
            i = onlymorph(i)
        result = p.search(i)
        if result:
            result = result.group(0)
            if not all:
                return result
        elif all:
            return None
    return result

def morph(rLoc, word, pattern, all=True):
    return _morph(rLoc, word, pattern, all, False)

def affix(rLoc, word, pattern, all=True):
    return _morph(rLoc, word, pattern, all, True)

def spell(rLoc, word):
    if word == None:
        return None
    return spellchecker.isValid(word, rLoc, ())

# get the tuple of the stem of the word or an empty array
def stem(rLoc, word):
    global stems
    if word == None:
        return []
    if not word in stems:
        x = spellchecker.spell(u"<?xml?><query type='stem'><word>" + word + "</word></query>", rLoc, ())
        if not x:
            return []
        t = x.getAlternatives()
        if not t:
            t = []
        stems[word] = list(t)
    return stems[word]

# get the tuple of the morphological generation of a word or an empty array
def generate(rLoc, word, example):
    if word == None:
        return []
    x = spellchecker.spell(u"<?xml?><query type='generate'><word>" + word + "</word><word>" + example + "</word></query>", rLoc, ())
    if not x:
        return []
    t = x.getAlternatives()
    if not t:
        t = []
    return list(t)

# get suggestions
def suggest(rLoc, word):
    global suggestions
    if word == None:
        return word
    if word not in suggestions:
        x = spellchecker.spell("_" + word, rLoc, ())
        if not x:
            return word
        t = x.getAlternatives()
        suggestions[word] = "\n".join(t)
    return suggestions[word]

# get the nth word of the input string or None
def word(s, n):
    a = re.match("(?u)( [-.\w%]+){" + str(n-1) + "}( [-.\w%]+)", s)
    if not a:
        return None
    return a.group(2)[1:]

# get the (-)nth word of the input string or None
def wordmin(s, n):
    a = re.search("(?u)([-.\w%]+ )([-.\w%]+ ){" + str(n-1) + "}$", s)
    if not a:
        return None
    return a.group(1)[:-1]

def calc(funcname, par):
    global calcfunc
    global SMGR
    if calcfunc == None:
        calcfunc = SMGR.createInstance( "com.sun.star.sheet.FunctionAccess")
        if calcfunc == None:
                return None
    return calcfunc.callFunction(funcname, par)

def proofread( nDocId, TEXT, LOCALE, nStartOfSentencePos, nSuggestedSentenceEndPos, rProperties ):
    global ignore
    aErrs = []
    s = TEXT[nStartOfSentencePos:nSuggestedSentenceEndPos]
    for i in get_rule(LOCALE):
        if i[0] and not str(i[0]) in ignore:
            for m in i[0].finditer(s):
              if not i[3] or eval(i[3]):
                aErr = uno.createUnoStruct( "com.sun.star.linguistic2.SingleProofreadingError" )
                aErr.nErrorStart        = nStartOfSentencePos + m.start(0) # nStartOfSentencePos
                aErr.nErrorLength       = m.end(0) - m.start(0)
                aErr.nErrorType         = PROOFREADING
                aErr.aRuleIdentifier    = str(i[0])
                iscap = (i[4] and m.group(0)[0:1].isupper())
                if i[1][0:1] == "=":
                        aErr.aSuggestions = tuple(cap(eval(i[1][1:]).split("\n"), iscap, LOCALE))
                else:
                        aErr.aSuggestions = tuple(cap(m.expand(i[1]).split("\n"), iscap, LOCALE))
                comment = i[2]
                if comment[0:1] == "=":
                        comment = eval(comment[1:])
                aErr.aShortComment      = comment.split("\\n")[0].strip()
                aErr.aFullComment       = comment.split("\\n")[-1].strip()
                if "://" in aErr.aFullComment:
                        p = PropertyValue()
                        p.Name = "FullCommentURL"
                        p.Value = aErr.aFullComment
                        aErr.aFullComment = aErr.aShortComment
                        aErr.aProperties        = (p,)
                else:
                        aErr.aProperties        = ()
                aErrs = aErrs + [aErr]
    return tuple(aErrs)

def cap(a, iscap, rLoc):
    if iscap:
        for i in range(0, len(a)):
            if a[i][0:1] == "i":
                if rLoc.Language == "tr" or rLoc.Language == "az":
                    a[i] = u"\u0130" + a[i][1:]
                elif a[i][1:2] == "j" and rLoc.Language == "nl":
                    a[i] = "IJ" + a[i][2:]
                else:
                    a[i] = "I" + a[i][1:]
            else:
                a[i] = a[i].capitalize()
    return a

def get_rule(rLocale):
        module = rLocale.Language
        if rLocale.Country != "":
                module = module + "_" + rLocale.Country
        try:
                return langrule[module]
        except:
                try:
                        module = rLocale.Language
                        return langrule[module]
                except:
                        try:
                                d = __import__("lightproof_" + pkg)
                        except:
                                print ("Error: missing language data: " + module)
                                return None
        # compile regular expressions
        for i in d.dic:
                try:
                        if re.compile("[(][?]iu[)]").match(i[0]):
                                i += [True]
                                i[0] = re.sub("[(][?]iu[)]", "(?u)", i[0])
                        else:
                                i += [False]
                        i[0] = re.compile(i[0])
                except:
                        print ("Lightproof: bad rule -- " +  str(i[0]))
                        i[0] = None
        langrule[module] = d.dic
        return langrule[module]

class Lightproof( unohelper.Base, XProofreader, XServiceInfo, XServiceName, XServiceDisplayName, XSupportedLocales):

    def __init__( self, ctx, *args ):
        global spellchecker
        global SMGR
        self.ctx = ctx
        self.ServiceName = "com.sun.star.linguistic2.Proofreader"
        self.ImplementationName = "org.openoffice.comp.pyuno.Lightproof." + pkg
        self.SupportedServiceNames = (self.ServiceName, )
        self.locales = []
        for i in locales:
            l = locales[i]
            self.locales += [Locale(l[0], l[1], l[2])]
        self.locales = tuple(self.locales)
        currentContext = uno.getComponentContext()
        SMGR = currentContext.ServiceManager
        spellchecker = SMGR.createInstanceWithContext("com.sun.star.linguistic2.SpellChecker", currentContext)
        lightproof_handler_pt_BR.load(currentContext)

    # XServiceName method implementations
    def getServiceName(self):
        return self.ImplementationName

    # XServiceInfo method implementations
    def getImplementationName (self):
                return self.ImplementationName

    def supportsService(self, ServiceName):
                return (ServiceName in self.SupportedServiceNames)

    def getSupportedServiceNames (self):
                return self.SupportedServiceNames

    # XSupportedLocales
    def hasLocale(self, aLocale):
        if aLocale in self.locales:
            return True
        for i in self.locales:
            if (i.Country == aLocale.Country or i.Country == "") and aLocale.Language == i.Language:
                return True
        return False

    def getLocales(self):
        return self.locales

    # XProofreader
    def isSpellChecker(self):
        return False

    def doProofreading(self, nDocId, rText, rLocale, nStartOfSentencePos, \
        nSuggestedSentenceEndPos, rProperties):
        aRes = uno.createUnoStruct( "com.sun.star.linguistic2.ProofreadingResult" )
        aRes.aDocumentIdentifier = nDocId
        aRes.aText = rText
        aRes.aLocale = rLocale
        aRes.nStartOfSentencePosition = nStartOfSentencePos
        aRes.nStartOfNextSentencePosition = nSuggestedSentenceEndPos
        l = rText[nSuggestedSentenceEndPos:nSuggestedSentenceEndPos+1]
        while l == " ":
            aRes.nStartOfNextSentencePosition = aRes.nStartOfNextSentencePosition + 1
            l = rText[aRes.nStartOfNextSentencePosition:aRes.nStartOfNextSentencePosition+1]
        if aRes.nStartOfNextSentencePosition == nSuggestedSentenceEndPos and l!="":
            aRes.nStartOfNextSentencePosition = nSuggestedSentenceEndPos + 1
        aRes.nBehindEndOfSentencePosition = aRes.nStartOfNextSentencePosition

        try:
            aRes.aErrors = proofread( nDocId, rText, rLocale, \
                nStartOfSentencePos, aRes.nBehindEndOfSentencePosition, rProperties)
        except:
            # traceback.print_exc(file=sys.stdout)
            aRes.aErrors = ()
        aRes.aProperties = ()
        aRes.xProofreader = self
        return aRes

    def ignoreRule(self, rid, aLocale):
        global ignore
        ignore[rid] = 1

    def resetIgnoreRules(self):
        global ignore
        ignore = {}

    # XServiceDisplayName
    def getServiceDisplayName(self, aLocale):
        return "Lightproof Verificador Gramatical (" + pkg + ")"

g_ImplementationHelper = unohelper.ImplementationHelper()
g_ImplementationHelper.addImplementation( Lightproof, \
        "org.openoffice.comp.pyuno.Lightproof." + pkg,
        ("com.sun.star.linguistic2.Proofreader",),)

g_ImplementationHelper.addImplementation( lightproof_handler_pt_BR.LightproofOptionsEventHandler, \
        "org.openoffice.comp.pyuno.LightproofOptionsEventHandler." + pkg,
        ("com.sun.star.awt.XContainerWindowEventHandler",),)
# pattern matching for common English abbreviations
abbrev = re.compile("(?i)\\b([a-z]|acct|approx|appt|apr|apt|assoc|asst|aug|ave|avg|co(nt|rp)?|ct|dec|defn|dept|dr|eg|equip|esp|est|etc|excl|ext|feb|fri|ft|govt?|hrs?|ib(id)?|ie|in(c|t)?|jan|jr|jul|lit|ln|mar|max|mi(n|sc)?|mon|Mrs?|mun|natl?|neg?|no(rm|s|v)?|nw|obj|oct|org|orig|pl|pos|prev|proj|psi|qty|rd|rec|rel|reqd?|resp|rev|sat|sci|se(p|pt)?|spec(if)?|sq|sr|st|subj|sun|sw|temp|thurs|tot|tues|univ|var|vs)\\.")

# pattern for paragraph checking
paralcap = re.compile(u"(?u)^[a-z].*[.?!] [A-Z].*[.?!][)\u201d]?$")


punct = { "?": "do ponto de interroga"+u"\u00E7"+u"\u00E3"+"o", "!": "do ponto de exclama"+u"\u00E7"+u"\u00E3"+"o",
  ",": "da v"+u"\u00ED"+"rgula", ":": "do dois pontos", ";": "do ponto e v"+u"\u00ED"+"rgula",
  "(": "do par"+u"\u00EA"+"ntese de abertura", ")": "do par"+u"\u00EA"+"ntese de fechamento",
  "[": "do colchete de abertura", "]": "do colchete de fechamento",
  u"\u201c": "das aspas de abertura", u"\u201d": "das aspas de fechamento"}

aaVer = { "quando": "Quando", "Quando": "Quando",
  "que": "Que", "Que": "Que", "se": "Se",
  "Se": "Se"}

aacento = {"abada": "abad"+u"\u00E1",
"abobora": "ab"+u"\u00F3"+"bora",
"aboboras": "ab"+u"\u00F3"+"boras",
"abundancia": "abund"+u"\u00E2"+"ncia",
"abundancias": "abund"+u"\u00E2"+"ncias",
"acaro": u"\u00E1"+"caro",
"acida": u"\u00E1"+"cida",
"acidas": u"\u00E1"+"cidas",
"acido": u"\u00E1"+"cido",
"adverbio": "adv"+u"\u00E9"+"rbio",
"Adverbio": "Adv"+u"\u00E9"+"rbio",
"Africa": u"\u00C1"+"frica",
"africa": u"\u00E1"+"frica",
"Africas": u"\u00C1"+"fricas",
"africas": u"\u00E1"+"fricas",
"africo": u"\u00E1"+"frico",
"Agencia": "Ag"+u"\u00EA"+"ncia",
"agencia": "ag"+u"\u00EA"+"ncia",
"Agencias": "Ag"+u"\u00EA"+"ncias",
"agencias": "ag"+u"\u00EA"+"ncias",
"agua": u"\u00E1"+"gua",
"aguas": u"\u00E1"+"guas",
"Agua": u"\u00C1"+"gua",
"Aguas": u"\u00C1"+"guas",
"alfandega": "alf"+u"\u00E2"+"ndega",
"alfandegas": "alf"+u"\u00E2"+"ndegas",
"alivio": "al"+u"\u00ED"+"vio",
"amilase": "am"+u"\u00ED"+"lase",
"angulo": u"\u00E2"+"ngulo",
"anodo": u"\u00E2"+"nodo",
"anonima": "an"+u"\u00F4"+"nima",
"anonimas": "an"+u"\u00F4"+"nimas",
"anonimo": "an"+u"\u00F4"+"nimo",
"antidoto": "ant"+u"\u00ED"+"doto",
"antiespirita": "antiesp"+u"\u00ED"+"rita",
"antifona": "ant"+u"\u00ED"+"fona",
"antifonas": "ant"+u"\u00ED"+"fonas",
"antifono": "ant"+u"\u00ED"+"fono",
"apostola": "ap"+u"\u00F3"+"stola",
"apostolas": "ap"+u"\u00F3"+"stolas",
"apostolo": "ap"+u"\u00F3"+"stolo",
"apostrofe": "ap"+u"\u00F3"+"strofe",
"apostrofes": "ap"+u"\u00F3"+"strofes",
"apostrofo": "ap"+u"\u00F3"+"strofo",
"apostrofos": "ap"+u"\u00F3"+"strofos",
"arbitro": u"\u00E1"+"rbitro",
"artificio": "artif"+u"\u00ED"+"cio",
"assedio": "ass"+u"\u00E9"+"dio",
"Assedio": "Ass"+u"\u00E9"+"dio",
"assuca": "a"+u"\u00E7"+u"\u00FA"+"car",
"assucar": "a"+u"\u00E7"+u"\u00FA"+"car",
"atico": u"\u00E1"+"tico",
"atona": u"\u00E1"+"tona",
"atonas": u"\u00E1"+"tonas",
"atono": u"\u00E1"+"tono",
"atoxica":  "at"+u"\u00F3"+"xica",
"atoxicas":  "at"+u"\u00F3"+"xicas",
"atoxico":  "at"+u"\u00F3"+"xico",
"auspicia": "ausp"+u"\u00ED"+"cia",
"auspicias": "ausp"+u"\u00ED"+"cias",
"auspicio": "ausp"+u"\u00ED"+"cio",
"autopsia":  "aut"+u"\u00F3"+"psia",
"autopsias":  "aut"+u"\u00F3"+"psias",
"avicola": "av"+u"\u00ED"+"cola",
"avicolas": "av"+u"\u00ED"+"colas",
"avicula": "av"+u"\u00ED"+"cola",
"aviculas": "av"+u"\u00ED"+"colas",
"avida": u"\u00E1"+"vida",
"avidas": u"\u00E1"+"vidas",
"avido": u"\u00E1"+"vido",
"avo": "av"+u"\u00F4",
"bau": "ba"+u"\u00FA",
"balaustre": "bala"+u"\u00FA"+"stre",
"balaustres": "bala"+u"\u00FA"+"stres",
"bibliografa": "bibli"+u"\u00F3"+"grafa",
"bibliografas": "bibli"+u"\u00F3"+"grafas",
"bibliografo": "bibli"+u"\u00F3"+"grafo",
"biopsia": "bi"+u"\u00F3"+"psia",
"biopsias": "bi"+u"\u00F3"+"psias",
"biotipo": "bi"+u"\u00F3"+"tipo",
"brecho": "brech"+u"\u00F3",
"bugues": "burgu"+u"\u00EA"+"s",
"bussola": "b"+u"\u00FA"+"ssola",
"bussolas": "b"+u"\u00FA"+"ssolas",
"buzio": "b"+u"\u00FA"+"zio",
"cadencia": "cad"+u"\u00EA"+"ncia",
"cafe": "caf"+u"\u00E9",
"cafes": "caf"+u"\u00E9"+"s",
"cafune": "cafun"+u"\u00E9",
"calorifica": "calor"+u"\u00ED"+"fica",
"calorificas": "calor"+u"\u00ED"+"ficas",
"calorifico": "calor"+u"\u00ED"+"fico",
"cambio": "c"+u"\u00E2"+"mbio",
"canada": "Canad"+u"\u00E1",
"Canada": "Canad"+u"\u00E1",
"canfora": "c"+u"\u00E2"+"nfora",
"capitulo": "cap"+u"\u00ED"+"tulo",
"Capitulo": "Cap"+u"\u00ED"+"tulo",
"CAPITULO": "CAP"+u"\u00CD"+"TULO",
"carate": "carat"+u"\u00EA",
"caricia": "car"+u"\u00ED"+"cia",
"caricias": "car"+u"\u00ED"+"cias",
"carie": "c"+u"\u00E1"+"rie",
"caries": "c"+u"\u00E1"+"ries",
"catalogo": "cat"+u"\u00E1"+"logo",
"catimbo": "catimb"+u"\u00F3",
"caustica": "c"+u"\u00E1"+"ustica",
"causticas": "c"+u"\u00E1"+"usticas",
"caustico": "c"+u"\u00E1"+"ustico",
"centigrada": "cent"+u"\u00ED"+"grada",
"centigradas": "cent"+u"\u00ED"+"gradas",
"centigrado": "cent"+u"\u00ED"+"grado",
"centrifuga": "centr"+u"\u00ED"+"fuga",
"centrifugas": "centr"+u"\u00ED"+"fugas",
"centrifugo": "centr"+u"\u00ED"+"fugo",
"chines": "chin"+u"\u00EA"+"s",
"cientifica": "cient"+u"\u00ED"+"fica",
"cientificas": "cient"+u"\u00ED"+"ficas",
"cientifico": "cient"+u"\u00ED"+"fico",
"cinematografo": "cinemat"+u"\u00F3"+"grafo",
"cipo": "cip"+u"\u00F3",
"circunstancia": "circunst"+u"\u00E2"+"ncia",
"ciume": "ci"+u"\u00FA"+"me",
"ciumes": "ci"+u"\u00FA"+"mes",
"CLAUSULA": "CL"+u"\u00C1"+"USULA",
"clausula": "cl"+u"\u00E1"+"usula",
"Clausula": "Cl"+u"\u00E1"+"usula",
"CLAUSULAS": "CL"+u"\u00C1"+"USULAS",
"clausulas": "cl"+u"\u00E1"+"usulas",
"Clausulas": "Cl"+u"\u00E1"+"usulas",
"clemencia": "clem"+u"\u00EA"+"ncia",
"clemencias": "clem"+u"\u00EA"+"ncias",
"clinica": "cl"+u"\u00ED"+"nica",
"clinicas": "cl"+u"\u00ED"+"nicas",
"clinico": "cl"+u"\u00ED"+"nico",
"coagulo": "co"+u"\u00E1"+"gulo",
"cocega": "c"+u"\u00F3"+"cega",
"cocegas": "c"+u"\u00F3"+"cegas",
"confidencia": "confid"+u"\u00EA"+"ncia",
"confidencias": "confid"+u"\u00EA"+"ncias",
"colonia": "col"+u"\u00F4"+"nia",
"colonias": "col"+u"\u00F4"+"nias",
"coloquio": "col"+u"\u00F3"+"quio",
"comedia": "com"+u"\u00E9"+"dia",
"Comedia": "Com"+u"\u00E9"+"dia",
"comedias": "com"+u"\u00E9"+"dias",
"Comedias": "Com"+u"\u00E9"+"dias",
"comercio": "com"+u"\u00E9"+"rcio",
"Comercio": "Com"+u"\u00E9"+"rcio",
"comissaria": "comiss"+u"\u00E1"+"ria",
"comissarias": "comiss"+u"\u00E1"+"rias",
"comissario": "comiss"+u"\u00E1"+"rio",
"comoda": "c"+u"\u00F4"+"moda",
"comodas": "c"+u"\u00F4"+"modas",
"comodo": "c"+u"\u00F4"+"modo",
"compendio": "comp"+u"\u00EA"+"ndio",
"compito": "c"+u"\u00F4"+"mpito",
"concava": "c"+u"\u00F4"+"ncava",
"concavas": "c"+u"\u00F4"+"ncavas",
"concavo": "c"+u"\u00F4"+"ncavo",
"conferencia": "confer"+u"\u00EA"+"ncia",
"conferencias": "confer"+u"\u00EA"+"ncias",
"congenere": "cong"+u"\u00EA"+"nere",
"congeneres": "cong"+u"\u00EA"+"neres",
"consocio": "cons"+u"\u00F3"+"rcio",
"consorcio": "cons"+u"\u00F3"+"rcio",
"contingencia": "conting"+u"\u00EA"+"ncia",
"contingencias": "conting"+u"\u00EA"+"ncias",
"convenio": "conv"+u"\u00EA"+"nio",
"Convenio": "Conv"+u"\u00EA"+"nio",
"coparticipe": "copart"+u"\u00ED"+"cipe",
"coparticipes": "copart"+u"\u00ED"+"cipes",
"cranio": "cr"+u"\u00E2"+"nio",
"cubica": "c"+u"\u00FA"+"bica",
"cubicas": "c"+u"\u00FA"+"bicas",
"cubico": "c"+u"\u00FA"+"bico",
"cumulo": "c"+u"\u00FA"+"mulo",
"curia": "c"+u"\u00FA"+"ria",
"Curia": "C"+u"\u00FA"+"ria",
"custodia": "cust"+u"\u00F3"+"dia",
"Custodia": "Cust"+u"\u00F3"+"dia",
"custodias": "cust"+u"\u00F3"+"dias",
"Custodio": "Cust"+u"\u00F3"+"dio",
"dadiva": "d"+u"\u00E1"+"diva",
"dadivas": "d"+u"\u00E1"+"divas",
"datilografo": "datil"+u"\u00F3"+"grafo",
"decima": "d"+u"\u00E9"+"cima",
"decimas": "d"+u"\u00E9"+"cimas",
"decimo": "d"+u"\u00E9"+"cimo",
"Dicionario": "Dicion"+u"\u00E1"+"rio",
"dicionario": "dicion"+u"\u00E1"+"rio",
"diligencia": "dilig"+u"\u00EA"+"ncia",
"domestica": "dom"+u"\u00E9"+"stica",
"domesticas": "dom"+u"\u00E9"+"sticas",
"domestico": "dom"+u"\u00E9"+"stico",
"domicilio": "domic"+u"\u00ED"+"lio",
"duble": "dubl"+u"\u00EA",
"dubles": "dubl"+u"\u00EA"+"s",
"duzia": "d"+u"\u00FA"+"zia",
"duzias": "d"+u"\u00FA"+"zias",
"ecografo": "ec"+u"\u00F3"+"grafo",
"eletrolise": "eletr"+u"\u00F3"+"lise",
"eminencia": "emin"+u"\u00EA"+"ncia",
"Eminencia": "Emin"+u"\u00EA"+"ncia",
"eminencias": "emin"+u"\u00EA"+"ncias",
"empresaria": "empres"+u"\u00E1"+"ria",
"empresarias": "empres"+u"\u00E1"+"rias",
"empresario": "empres"+u"\u00E1"+"rio",
"enfase": u"\u00EA"+"nfase",
"enfases": u"\u00EA"+"nfases",
"EPIGRAFE": "EP"+u"\u00CD"+"GRAFE",
"epigrafe": "ep"+u"\u00ED"+"grafe",
"Epigrafe": "Ep"+u"\u00ED"+"grafe",
"EPIGRAFES": "EP"+u"\u00CD"+"GRAFES",
"epigrafes": "ep"+u"\u00ED"+"grafes",
"Epigrafes": "Ep"+u"\u00ED"+"grafes",
"epigrafo": "ep"+u"\u00ED"+"grafo",
"epigramatica": "epigram"+u"\u00E1"+"tica",
"epigramaticas": "epigram"+u"\u00E1"+"ticas",
"epigramatico": "epigram"+u"\u00E1"+"tico",
"episodio": "epis"+u"\u00F3"+"dio",
"equivoco": "equ"+u"\u00ED"+"voco",
"escoria": "esc"+u"\u00F3"+"ria",
"escorias": "esc"+u"\u00F3"+"rias",
"espatula": "esp"+u"\u00E1"+"tula",
"espatulas": "esp"+u"\u00E1"+"tulas",
"especie": "esp"+u"\u00E9"+"cie",
"especies": "esp"+u"\u00E9"+"cies",
"espectroscopio": "espectrosc"+u"\u00F3"+"pio",
"espirita": "esp"+u"\u00ED"+"rita",
"espiritas": "esp"+u"\u00ED"+"ritas",
"espirito": "esp"+u"\u00ED"+"rito",
"esplendida": "espl"+u"\u00EA"+"ndida",
"esplendidas": "espl"+u"\u00EA"+"ndidas",
"esplendido": "espl"+u"\u00EA"+"ndido",
"espolio": "esp"+u"\u00F3"+"lio",
"estabulo": "est"+u"\u00E1"+"bulo",
"estadio": "est"+u"\u00E1"+"dio",
"estancia": "est"+u"\u00E2"+"ncia",
"Estancia": "Est"+u"\u00E2"+"ncia",
"estancias": "est"+u"\u00E2"+"ncias",
"estatua": "est"+u"\u00E1"+"tua",
"estatuas": "est"+u"\u00E1"+"tuas",
"estomago": "est"+u"\u00F4"+"mago",
"estrategia": "estrat"+u"\u00E9"+"gia",
"estrategias": "estrat"+u"\u00E9"+"gias",
"estupida": "est"+u"\u00FA"+"pida",
"estupidas": "est"+u"\u00FA"+"pidas",
"estupido": "est"+u"\u00FA"+"pido",
"exito": u"\u00EA"+"xito",
"experiencia": "experi"+u"\u00EA"+"ncia",
"experiencias": "experi"+u"\u00EA"+"ncias",
"explicitas": "expl"+u"\u00ED"+"citas",
"explicito": "expl"+u"\u00ED"+"cito",
"extravagancia": "extravag"+u"\u00E2"+"ncia",
"extravagancias": "extravag"+u"\u00E2"+"ncias",
"fabula": "f"+u"\u00E1"+"bula",
"fabulas": "f"+u"\u00E1"+"bulas",
"falacia": "fal"+u"\u00E1"+"cia",
"falacias": "fal"+u"\u00E1"+"cias",
"figado": "f"+u"\u00ED"+"gado",
"filosofa": "fil"+u"\u00F3"+"sofa",
"filosofas": "fil"+u"\u00F3"+"sofas",
"filosofo": "fil"+u"\u00F3"+"sofo",
"fonografo": "fon"+u"\u00F3"+"grafo",
"formica": "f"+u"\u00F3"+"rmica",
"formica": "f"+u"\u00F3"+"rmica",
"formicas": "f"+u"\u00F3"+"rmicas",
"fosforo": "f"+u"\u00F3"+"sforo",
"fotocopia": "fotoc"+u"\u00F3"+"pia",
"fotocopias": "fotoc"+u"\u00F3"+"pias",
"frigida": "fr"+u"\u00ED"+"gida",
"frigidas": "fr"+u"\u00ED"+"gidas",
"frigido": "fr"+u"\u00ED"+"gido",
"frigorifico": "frigor"+u"\u00ED"+"fico",
"fuba": "fub"+u"\u00E1",
"fubas": "fub"+u"\u00E1"+"s",
"fulgido": "f"+u"\u00FA"+"lgido",
"Galaxia": "Gal"+u"\u00E1"+"xia",
"galaxia": "gal"+u"\u00E1"+"xia",
"galaxias": "gal"+u"\u00E1"+"xias",
"ganancia": "gan"+u"\u00E2"+"ncia",
"ganancias": "gan"+u"\u00E2"+"ncias",
"geografa": "ge"+u"\u00F3"+"grafa",
"geografas": "ge"+u"\u00F3"+"grafas",
"geografo": "ge"+u"\u00F3"+"grafo",
"georreferencia": "georrefer"+u"\u00EA"+"ncia",
"georreferencias": "georrefer"+u"\u00EA"+"ncias",
"ginastica": "gin"+u"\u00E1"+"stica",
"ginasticas": "gin"+u"\u00E1"+"sticas",
"ginastico": "gin"+u"\u00E1"+"stico",
"gloria": "gl"+u"\u00F3"+"ria",
"Gloria": "Gl"+u"\u00F3"+"ria",
"glorias": "gl"+u"\u00F3"+"rias",
"gondola": "g"+u"\u00F4"+"ndola",
"gondolas": "g"+u"\u00F4"+"ndolas",
"gramatica": "gram"+u"\u00E1"+"tica",
"gramaticas": "gram"+u"\u00E1"+"ticas",
"gramatico": "gram"+u"\u00E1"+"tico",
"guarana": "guaran"+u"\u00E1",
"ha": "h"+u"\u00E1",
"halito": "h"+u"\u00E1"+"lito",
"hemodialise": "hemodi"+u"\u00E1"+"lise",
"hemodialises": "hemodi"+u"\u00E1"+"lises",
"hemodiatase": "hemodi"+u"\u00E1"+"stase",
"hemodiatases": "hemodi"+u"\u00E1"+"stases",
"heptafono": "hept"+u"\u00E1"+"fono",
"heptafonos": "hept"+u"\u00E1"+"fonos",
"hibrida": "h"+u"\u00ED"+"brida",
"hibridas": "h"+u"\u00ED"+"bridas",
"hibrido": "h"+u"\u00ED"+"brido",
"hidrogena": "hidr"+u"\u00F3"+"gena",
"hidrogenas": "hidr"+u"\u00F3"+"genas",
"hidrogeno": "hidr"+u"\u00F3"+"geno",
"hidrolise": "hidr"+u"\u00F3"+"lise",
"hidrolises": "hidr"+u"\u00F3"+"lises",
"hidrometro": "hidr"+u"\u00F4"+"metro",
"hilaria": "hil"+u"\u00E1"+"ria",
"hilarias": "hil"+u"\u00E1"+"rias",
"hilario": "hil"+u"\u00E1"+"rio",
"historia": "hist"+u"\u00F3"+"ria",
"historias": "hist"+u"\u00F3"+"rias",
"homofila": "hom"+u"\u00F3"+"fila",
"homofilas": "hom"+u"\u00F3"+"filas",
"homofilo": "hom"+u"\u00F3"+"filo",
"homofona": "hom"+u"\u00F3"+"fona",
"homofonas": "hom"+u"\u00F3"+"fonas",
"homofono": "hom"+u"\u00F3"+"fono",
"honorifica": "honor"+u"\u00ED"+"fica",
"honorificas": "honor"+u"\u00ED"+"ficas",
"honorifico": "honor"+u"\u00ED"+"fico",
"horoscopo": "hor"+u"\u00F3"+"scopo",
"ignominia": "ignom"+u"\u00ED"+"nia",
"ignominias": "ignom"+u"\u00ED"+"nias",
"ilegitima": "ileg"+u"\u00ED"+"tima",
"ilegitimas": "ileg"+u"\u00ED"+"timas",
"ilegitimo": "ileg"+u"\u00ED"+"timo",
"impar": u"\u00ED"+"mpar",
"impares": u"\u00ED"+"mpares",
"impertinencia": "impertin"+u"\u00EA"+"ncia",
"impertinencias": "impertin"+u"\u00EA"+"ncias",
"impeto": u"\u00ED"+"mpeto",
"impia": u"\u00ED"+"mpia",
"impias": u"\u00ED"+"mpias",
"impio": u"\u00ED"+"mpio",
"indicio": "ind"+u"\u00ED"+"cio",
"indulgencia": "indulg"+u"\u00EA"+"ncia",
"indulgencias": "indulg"+u"\u00EA"+"ncias",
"industria": "ind"+u"\u00FA"+"stria",
"industrias": "ind"+u"\u00FA"+"strias",
"ingenua": "ing"+u"\u00EA"+"nua",
"ingenuas": "ing"+u"\u00EA"+"nuas",
"ingenuo": "ing"+u"\u00EA"+"nuo",
"insignia": "ins"+u"\u00ED"+"gnia",
"insignias": "ins"+u"\u00ED"+"gnias",
"insipida": "ins"+u"\u00ED"+"pida",
"insipidas": "ins"+u"\u00ED"+"pidas",
"insipido": "ins"+u"\u00ED"+"pido",
"insonia": "ins"+u"\u00F4"+"nia",
"insonias": "ins"+u"\u00F4"+"nias",
"instancia": "inst"+u"\u00E2"+"ncia",
"instancias": "inst"+u"\u00E2"+"ncias",
"inteligencia": "intelig"+u"\u00EA"+"ncia",
"inteligencias": "intelig"+u"\u00EA"+"ncias",
"inventario": "invent"+u"\u00E1"+"rio",
"iridio": "ir"+u"\u00ED"+"dio",
"iris": u"\u00ED"+"ris",
"irreverencia": "irrever"+u"\u00EA"+"ncia",
"irreverencias": "irrever"+u"\u00EA"+"ncias",
"jatancia": "jat"+u"\u00E2"+"ncia",
"jatancias": "jat"+u"\u00E2"+"ncias",
"jornalistica": "jornal"+u"\u00ED"+"stica",
"jornalisticas": "jornal"+u"\u00ED"+"sticas",
"jornalistico": "jornal"+u"\u00ED"+"stico",
"lagrima": "l"+u"\u00E1"+"grima",
"lagrimas": "l"+u"\u00E1"+"grimas",
"lamina": "l"+u"\u00E2"+"mina",
"laminas": "l"+u"\u00E2"+"minas",
"legitimo": "leg"+u"\u00ED"+"timo",
"licitas": "l"+u"\u00ED"+"citas",
"licito": "l"+u"\u00ED"+"cito",
"lirica": "l"+u"\u00ED"+"rica",
"liricas": "l"+u"\u00ED"+"ricas",
"lirico": "l"+u"\u00ED"+"rico",
"litigio": "lit"+u"\u00ED"+"gio",
"logica": "l"+u"\u00F3"+"gica",
"logicas": "l"+u"\u00F3"+"gicas",
"logico": "l"+u"\u00F3"+"gico",
"lucifero": "luc"+u"\u00ED"+"fero",
"macunaima": "macuna"+u"\u00ED"+"ma",
"Macunaima": "Macuna"+u"\u00ED"+"ma",
"mafia": "m"+u"\u00E1"+"fia",
"mafias": "m"+u"\u00E1"+"fias",
"mafua": "mafu"+u"\u00E1",
"mafuas": "mafu"+u"\u00E1"+"s",
"magica": "m"+u"\u00E1"+"gica",
"magicas": "m"+u"\u00E1"+"gicas",
"magico": "m"+u"\u00E1"+"gico",
"magnifica": "magn"+u"\u00ED"+"fica",
"magnificas": "magn"+u"\u00ED"+"ficas",
"magnifico": "magn"+u"\u00ED"+"fico",
"manigancia": "manig"+u"\u00E2"+"ncia",
"manigancias": "manig"+u"\u00E2"+"ncias",
"maleficio": "malef"+u"\u00ED"+"cio",
"maluques": "maluqu"+u"\u00EA"+"s",
"maniaca": "man"+u"\u00ED"+"aca",
"maniacas": "man"+u"\u00ED"+"acas",
"maniaco": "man"+u"\u00ED"+"aco",
"maquina": "m"+u"\u00E1"+"quina",
"maquinas": "m"+u"\u00E1"+"quinas",
"matrimonio": "matrim"+u"\u00F4"+"nio",
"megametro": "meg"+u"\u00E2"+"metro",
"memoria": "mem"+u"\u00F3"+"ria",
"memorias": "memorias",
"mercancia": "merc"+u"\u00E2"+"ncia",
"mercancias": "merc"+u"\u00E2"+"ncias",
"metafisica": "metaf"+u"\u00ED"+"sica",
"metafisicas": "metaf"+u"\u00ED"+"sicas",
"metafisico": "metaf"+u"\u00ED"+"sico",
"microbio": "micr"+u"\u00F3"+"bio",
"micrografo": "micr"+u"\u00F3"+"grafo",
"milicia": "mil"+u"\u00ED"+"cia",
"milicias": "mil"+u"\u00ED"+"cias",
"milimetro": "mil"+u"\u00ED"+"metro",
"mimeografo": "mime"+u"\u00F3"+"grafo",
"mimica": "m"+u"\u00ED"+"mica",
"mimicas": "m"+u"\u00ED"+"micas",
"mimico": "m"+u"\u00ED"+"mico",
"minudencia": "minud"+u"\u00EA"+"ncia",
"minudencias": "minud"+u"\u00EA"+"ncias",
"minuscula": "min"+u"\u00FA"+"scula",
"minusculas": "min"+u"\u00FA"+"sculas",
"minusculo": "min"+u"\u00FA"+"sculo",
"misera": "m"+u"\u00ED"+"sera",
"miseras": "m"+u"\u00ED"+"seras",
"misero": "m"+u"\u00ED"+"sero",
"monossilaba": "monoss"+u"\u00ED"+"laba",
"monossilabas": "monoss"+u"\u00ED"+"labas",
"monossilabo": "monoss"+u"\u00ED"+"labo",
"murmurio": "murm"+u"\u00FA"+"rio",
"musica": "m"+u"\u00FA"+"sica",
"musicas": "m"+u"\u00FA"+"sicas",
"musico": "m"+u"\u00FA"+"sico",
"naufrago": "n"+u"\u00E1"+"ufrago",
"necropsia": "necr"+u"\u00F3"+"psia",
"necropsias": "necr"+u"\u00F3"+"psias",
"nevoa": "n"+u"\u00E9"+"voa",
"nevoas": "n"+u"\u00E9"+"voas",
"obliqua": "obl"+u"\u00ED"+"qua",
"obliquas": "obl"+u"\u00ED"+"quas",
"obliquo": "obl"+u"\u00ED"+"quo",
"obsequio": "obs"+u"\u00E9"+"quio",
"obstaculo": "obst"+u"\u00E1"+"culo",
"oficio": "of"+u"\u00ED"+"cio",
"omega": u"\u00F4"+"mega",
"oraculo": "or"+u"\u00E1"+"culo",
"osculo": u"\u00F3"+"sculo",
"pacifico": "pac"+u"\u00ED"+"fico",
"pagina": "p"+u"\u00E1"+"gina",
"paginas": "p"+u"\u00E1"+"ginas",
"paleofilo": "pale"+u"\u00F3"+"filo",
"paleografo": "pale"+u"\u00F3"+"grafo",
"paleto": "palet"+u"\u00F3",
"palmatoria": "palmat"+u"\u00F3"+"ria",
"palmatorias": "palmat"+u"\u00F3"+"rias",
"pandego": "p"+u"\u00E2"+"ndego",
"panico": "p"+u"\u00E2"+"nico",
"pantografo": "pant"+u"\u00F3"+"grafo",
"paralitica": "paral"+u"\u00ED"+"tica",
"paraliticas": "paral"+u"\u00ED"+"ticas",
"paralitico": "paral"+u"\u00ED"+"tico",
"Parana": "Paran"+u"\u00E1",
"parodia": "par"+u"\u00F3"+"dia",
"parodias": "par"+u"\u00F3"+"dias",
"Paroquia": "Par"+u"\u00F3"+"quia",
"paroquia": "par"+u"\u00F3"+"quia",
"Paroquias": "Par"+u"\u00F3"+"quias",
"paroquias": "par"+u"\u00F3"+"quias",
"pascoa": "p"+u"\u00E1"+"scoa",
"pascoas": "p"+u"\u00E1"+"scoas",
"pavia": "p"+u"\u00E1"+"via",
"pavias": "p"+u"\u00E1"+"vias",
"pendencia": "pend"+u"\u00EA"+"ncia",
"pendencias": "pend"+u"\u00EA"+"ncias",
"pendulo": "p"+u"\u00EA"+"ndulo",
"penes": "p"+u"\u00EA"+"nis",
"penitencia": "penit"+u"\u00EA"+"ncia",
"penitencias": "penit"+u"\u00EA"+"ncias",
"perdulario": "perdul"+u"\u00E1"+"rio",
"perfei"+u"\u00E7"+"oes": "perfei"+u"\u00E7"+u"\u00F5"+"es",
"perola": "p"+u"\u00E9"+"rola",
"perolas": "p"+u"\u00E9"+"rolas",
"pestilencia": "pestil"+u"\u00EA"+"ncia",
"pestilencias": "pestil"+u"\u00EA"+"ncias",
"piramide": "pir"+u"\u00E2"+"mide",
"piramides": "pir"+u"\u00E2"+"mides",
"pirografo": "pir"+u"\u00F3"+"grafo",
"placida": "pl"+u"\u00E1"+"cida",
"placidas": "pl"+u"\u00E1"+"cidas",
"placido": "pl"+u"\u00E1"+"cido",
"plagio": "pl"+u"\u00E1"+"gio",
"plastica": "pl"+u"\u00E1"+"stica",
"plasticas": "pl"+u"\u00E1"+"sticas",
"plastico": "pl"+u"\u00E1"+"stico",
"polemica": "pol"+u"\u00EA"+"mica",
"polemicas": "pol"+u"\u00EA"+"micas",
"polemico": "pol"+u"\u00EA"+"mico",
"poligrafa": "pol"+u"\u00ED"+"grafa",
"poligrafas": "pol"+u"\u00ED"+"grafas",
"poligrafo": "pol"+u"\u00ED"+"grafo",
"porem": "por"+u"\u00E9"+"m",
"porno": "porn"+u"\u00F4",
"pornos": "porn"+u"\u00F4"+"s",
"potencia": "pot"+u"\u00EA"+"ncia",
"potencias": "pot"+u"\u00EA"+"ncias",
"preambulo": "pre"+u"\u00E2"+"mbulo",
"prefacio": "pref"+u"\u00E1"+"cio",
"preferencia": "prefer"+u"\u00EA"+"ncia",
"preferencias": "prefer"+u"\u00EA"+"ncias",
"premio": "pr"+u"\u00EA"+"mio",
"prenuncio": "pren"+u"\u00FA"+"ncio",
"presagio": "press"+u"\u00E1"+"gio",
"presidio": "pres"+u"\u00ED"+"dio",
"pressagio": "press"+u"\u00E1"+"gio",
"principio": "princ"+u"\u00ED"+"pio",
"prodiga": "pr"+u"\u00F3"+"diga",
"prodigas": "pr"+u"\u00F3"+"digas",
"prodigo": "pr"+u"\u00F3"+"digo",
"prologo": "pr"+u"\u00F3"+"logo",
"pronostico": "pron"+u"\u00F3"+"stico",
"prontuario": "prontu"+u"\u00E1"+"rio",
"propicio": "prop"+u"\u00ED"+"cio",
"proposito": "prop"+u"\u00F3"+"sito",
"proverbio": "prov"+u"\u00E9"+"rbio",
"Proverbio": "Prov"+u"\u00E9"+"rbio",
"prudencia": "prud"+u"\u00EA"+"ncia",
"prudencias": "prud"+u"\u00EA"+"ncias",
"psicanalise": "psican"+u"\u00E1"+"lise",
"psicanalises": "psican"+u"\u00E1"+"lises",
"psicoanalise": "psicoan"+u"\u00E1"+"lise",
"psicoanalises": "psicoan"+u"\u00E1"+"lises",
"psicologa": "psic"+u"\u00F3"+"loga",
"psicologas": "psic"+u"\u00F3"+"logas",
"psicologo": "psic"+u"\u00F3"+"logo",
"quadrupede": "quadr"+u"\u00FA"+"pede",
"quadrupedes": "quadr"+u"\u00FA"+"pedes",
"quadrupla": "qu"+u"\u00E1"+"drupla",
"quadruplas": "qu"+u"\u00E1"+"druplas",
"quadruplo": "qu"+u"\u00E1"+"druplo",
"querencia": "quer"+u"\u00EA"+"ncia",
"querencias": "quer"+u"\u00EA"+"ncias",
"quilometro": "quil"+u"\u00F4"+"metro",
"quintessencia": "quintess"+u"\u00EA"+"ncia",
"quintessencias": "quintess"+u"\u00EA"+"ncias",
"radio": "r"+u"\u00E1"+"dio",
"reciproca": "rec"+u"\u00ED"+"proca",
"reciprocas": "rec"+u"\u00ED"+"procas",
"reciproco": "rec"+u"\u00ED"+"proco",
"reminiscencia": "reminisc"+u"\u00EA"+"ncia",
"reminiscencias": "reminisc"+u"\u00EA"+"ncias",
"residencia": "resid"+u"\u00EA"+"ncia",
"residencias": "resid"+u"\u00EA"+"ncias",
"reticencia": "retic"+u"\u00EA"+"ncia",
"reticencias": "retic"+u"\u00EA"+"ncias",
"retorica": "ret"+u"\u00F3"+"rica",
"retoricas": "ret"+u"\u00F3"+"ricas",
"retorico": "ret"+u"\u00F3"+"rico",
"retrograda": "retr"+u"\u00F3"+"grada",
"retrogradas": "retr"+u"\u00F3"+"gradas",
"retrogrado": "retr"+u"\u00F3"+"grado",
"retrogrado": "retr"+u"\u00F3"+"grado",
"reusa": "re"+u"\u00FA"+"sa",
"reusas": "re"+u"\u00FA"+"sas",
"reuso": "re"+u"\u00FA"+"so",
"rodizio": "rod"+u"\u00ED"+"zio",
"Romania": "Rom"+u"\u00E2"+"nia",
"rufio": "r"+u"\u00FA"+"fio",
"rustica": "r"+u"\u00FA"+"stica",
"rusticas": "r"+u"\u00FA"+"sticas",
"rustico": "r"+u"\u00FA"+"stico",
"sache": "sach"+u"\u00EA",
"saches": "sach"+u"\u00EA"+"s",
"salario": "sal"+u"\u00E1"+"rio",
"saltica": "s"+u"\u00E1"+"ltica",
"salticas": "s"+u"\u00E1"+"lticas",
"saltico": "s"+u"\u00E1"+"ltico",
"Secretario": "Secret"+u"\u00E1"+"rio",
"secretario": "secret"+u"\u00E1"+"rio",
"seculo": "s"+u"\u00E9"+"culo",
"sequencia": "sequ"+u"\u00EA"+"ncia",
"sequencias": "sequ"+u"\u00EA"+"ncias",
"sequio": "s"+u"\u00E9"+"quio",
"serie": "s"+u"\u00E9"+"rie",
"series": "s"+u"\u00E9"+"ries",
"serigrafo": "ser"+u"\u00ED"+"grafo",
"serio": "s"+u"\u00E9"+"rio",
"silaba": "s"+u"\u00ED"+"laba",
"silabas": "s"+u"\u00ED"+"labas",
"silabo": "s"+u"\u00ED"+"labo",
"simbolo": "s"+u"\u00ED"+"mbolo",
"sincope": "s"+u"\u00ED"+"ncope",
"sincopes": "s"+u"\u00ED"+"ncopes",
"sindica": "s"+u"\u00ED"+"ndica",
"sindicas": "s"+u"\u00ED"+"ndicas",
"sindico": "s"+u"\u00ED"+"ndico",
"sitio": "s"+u"\u00ED"+"tio",
"solida": "s"+u"\u00F3"+"lida",
"solidas": "s"+u"\u00F3"+"lidas",
"solido": "s"+u"\u00F3"+"lido",
"sonambula": "son"+u"\u00E2"+"mbula",
"sonambulas": "son"+u"\u00E2"+"mbulas",
"sonambulo": "son"+u"\u00E2"+"mbulo",
"soporifero": "sopor"+u"\u00ED"+"fero",
"subespecifica": "subespec"+u"\u00ED"+"fica",
"subespecificas": "subespec"+u"\u00ED"+"ficas",
"subespecifico": "subespec"+u"\u00ED"+"fico",
"Subsecretario": "Subsecret"+u"\u00E1"+"rio",
"subsecretario": "subsecret"+u"\u00E1"+"rio",
"subsidio": "subs"+u"\u00ED"+"dio",
"substancia": "subst"+u"\u00E2"+"ncia",
"substancias": "subst"+u"\u00E2"+"ncias",
"sudorifica": "sudor"+u"\u00ED"+"fica",
"sudorificas": "sudor"+u"\u00ED"+"ficas",
"sudorifico": "sudor"+u"\u00ED"+"fico",
"sumaria": "sum"+u"\u00E1"+"ria",
"sumarias": "sum"+u"\u00E1"+"rias",
"Sumario": "Sum"+u"\u00E1"+"rio",
"sumario": "sum"+u"\u00E1"+"rio",
"sumula": "s"+u"\u00FA"+"mula",
"sumulas": "s"+u"\u00FA"+"mulas",
"suplicio": "supl"+u"\u00ED"+"cio",
"taboa": "t"+u"\u00E1"+"bua",
"taboas": "t"+u"\u00E1"+"buas",
"tabua": "t"+u"\u00E1"+"bua",
"tabuas": "t"+u"\u00E1"+"buas",
"tendencia": "tend"+u"\u00EA"+"ncia",
"tendencias": "tend"+u"\u00EA"+"ncias",
"tipica": "t"+u"\u00ED"+"pica",
"tipicas": "t"+u"\u00ED"+"picas",
"tipico": "t"+u"\u00ED"+"pico",
"tipografo": "tip"+u"\u00F3"+"grafo",
"titulo": "t"+u"\u00ED"+"tulo",
"Titulo": "T"+u"\u00ED"+"tulo",
"tomografo": "tom"+u"\u00F3"+"grafo",
"toner": "t"+u"\u00F4"+"ner",
"topografo": "top"+u"\u00F3"+"grafo",
"toxica": "t"+u"\u00F3"+"xica",
"toxicas": "t"+u"\u00F3"+"xicas",
"toxico": "t"+u"\u00F3"+"xico",
"trafico": "tr"+u"\u00E1"+"fico",
"tragedia": "trag"+u"\u00E9"+"dia",
"tragedias": "trag"+u"\u00E9"+"dias",
"translucida": "transl"+u"\u00FA"+"cida",
"translucidas": "transl"+u"\u00FA"+"cidas",
"translucido": "transl"+u"\u00FA"+"cido",
"tremula": "tr"+u"\u00EA"+"mula",
"tremulas": "tr"+u"\u00EA"+"mulas",
"tremulo": "tr"+u"\u00EA"+"mulo",
"tremulo": "tr"+u"\u00EA"+"mulo",
"triangulo": "tri"+u"\u00E2"+"ngulo",
"tripe": "trip"+u"\u00E9",
"tropega": "tr"+u"\u00F4"+"pega",
"tropegas": "tr"+u"\u00F4"+"pegas",
"tropego": "tr"+u"\u00F4"+"pego",
"tropico": "tr"+u"\u00F3"+"pico",
"tumulo": "t"+u"\u00FA"+"mulo",
"ulcera": u"\u00FA"+"lcera",
"ulceras": u"\u00FA"+"lceras",
"ultima": u"\u00FA"+"ltima",
"ultimas": u"\u00FA"+"ltimas",
"ultimo": u"\u00FA"+"ltimo",
"unanime": "un"+u"\u00E2"+"nime",
"unanimes": "un"+u"\u00E2"+"nimes",
"urticaria": "urtic"+u"\u00E1"+"ria",
"veiculo": "ve"+u"\u00ED"+"culo",
"vermifugo": "verm"+u"\u00ED"+"fugo",
"virgula": "v"+u"\u00ED"+"rgula",
"virgulas": "v"+u"\u00ED"+"rgulas",
"vitima": "v"+u"\u00ED"+"tima",
"vitimas": "v"+u"\u00ED"+"timas",
"vitoria": "vit"+u"\u00F3"+"ria",
"vitorias": "vit"+u"\u00F3"+"rias",
"voluntaria": "volunt"+u"\u00E1"+"ria",
"voluntarias": "volunt"+u"\u00E1"+"rias",
"voluntario": "volunt"+u"\u00E1"+"rio",
"xilografo": "xil"+u"\u00F3"+"grafo",
"xoco": "xoc"+u"\u00F3",
"zoografo": "zo"+u"\u00F3"+"grafo"}


acAcento = set(["abada",
"abobora", "aboboras", "abundancia",
"abundancias", "acaro", "acida", "acidas", "acido",
"adverbio", "Adverbio", "Africa", "africa",
"Africas", "africas", "africo", "Agencia", "agencia", "Agencias", "agencias",
"alfandega","alfandegas", "alivio", "amilase", "angulo",
"anodo", "anonima", "anonimas", "anonimo",
"antidoto", "antiespirita", "antifona", "antifonas",
"antifono", "apostola", "apostolas", "apostolo",
"apostrofe", "apostrofes", "apostrofo", "apostrofos",
"arbitro", "artificio", "assedio", "Assedio",
"assuca", "assucar", "atico", "atona", "atonas", "atono",
"atoxica", "atoxicas", "atoxico", "auspicia",
"auspicias", "auspicio",  "autopsia", "autopsias",
"avicola", "avicolas", "avicula", "aviculas",
"avida", "avidas", "avido", "avo", "balaustre", "balaustres", "bau",
"bibliografa", "bibliografas", "bibliografo", "biopsia",
"biopsias", "biotipo", "brecho", "burgues", "bussola",
"bussolas", "buzio", "cadencia", "cafe",
"cafes", "cafune", "calorifica", "calorificas",
"calorifico", "cambio", "canada", "Canada",
"canfora", "CAPITULO", "Capitulo", "capitulo", "carate",
"caricia", "caricias", "carie", "caries", "catalogo", "catimbo",
"caustica", "causticas", "caustico", "centigrada",
"centigradas", "centigrado", "centrifuga", "centrifugas",
"centrifugo", "chines", "cientifica", "cientificas",
"cientifico", "cinematografo", "cipo", "circunstancia",
"ciume", "ciumes", "CLAUSULA", "clausula",
"Clausula", "CLAUSULAS", "clausulas", "Clausulas",
"clinica", "clinicas", "clinico", "coagulo",
"cocega", "cocegas", "colonia", "colonias",
"coloquio", "comedia", "Comedia", "comedias",
"Comedias", "comercio", "Comercio",
"comissaria", "comissarias", "comissario", "comoda",
"comodas", "comodo", "compendio", "compito",
"concava", "concavas", "concavo", "conferencia",
"conferencias", "congenere", "congeneres", "consocio",
"consorcio", "contingencia", "contingencias", "convenio",
"Convenio", "coparticipe", "coparticipes", "cranio",
"cubica", "cubicas", "cubico", "cumulo",
"curia", "Curia", "custodia", "Custodia",
"custodias", "Custodio", "datilografo",
"decima", "decimas", "decimo", "Dicionario", "dicionario", "diligencia",
"domestica", "domesticas", "domestico", "domicilio",
"duble", "dubles", "duzia", "duzias",
"ecografo", "eletrolise", "eminencia", "Eminencia",
"eminencias", "empresaria", "empresarias", "empresario",
"enfase", "enfases", "EPIGRAFE", "epigrafe",
"Epigrafe", "EPIGRAFES", "epigrafes", "Epigrafes",
"epigrafo", "epigramatica", "epigramaticas", "epigramatico",
"episodio", "equivoco", "escoria", "escorias",
"espatula", "espatulas", "especie", "especies",
"espectroscopio", "espirita", "espiritas", "espirito",
"esplendida", "esplendidas", "esplendido", "espolio",
"estabulo", "estadio", "estancia", "Estancia",
"estancias", "estatua", "estatuas", "estomago",
"estrategia", "estrategias", "estupida", "estupidas",
"estupido", "exito", "experiencia", "experiencias",
"explicita", "explicitas", "explicito", "fabula",
"fabulas", "falacia", "falacias", "figado",
"filosofa", "filosofas", "filosofo", "fonografo",
"formica", "formica", "formicas", "fosforo",
"fotocopia", "fotocopias", "frigida", "frigidas",
"frigido", "frigorifico", "fuba", "fubas",
"fulgido", "Galaxia", "galaxia", "galaxias", "geografa",
"geografas", "geografo", "georreferencia", "georreferencias",
"ginastica", "ginasticas", "ginastico", "gloria",
"Gloria", "glorias", "gondola", "gondolas",
"gramatica", "gramaticas", "gramatico", "guarana", "ha", "halito",
"hemodialise", "hemodialises", "hemodiastase", "hemodiastases",
"heptafono", "heptafonos", "hibrida", "hibridas",
"hibrido", "hidrogena", "hidrogenas", "hidrogeno",
"hidrolise", "hidrolises", "hidrometro", "hilaria",
"hilarias", "hilario", "historia", "historias",
"homofila", "homofilas", "homofilo", "homofona",
"homofonas", "homofono", "honorifica", "honorificas",
"honorifico", "horoscopo", "ignominia",
"ignominias", "ilegitima", "ilegitimas",
"ilegitimo", "impar", "impares", "impeto",
"impia", "impias", "impio", "indicio",
"indulgencia", "indulgencias", "industria", "industrias",
"ingenua", "ingenuas", "ingenuo", "insignia",
"insignias", "insipida", "insipidas", "insipido",
"insonia", "insonias", "instancia", "instancias",
"inteligencia", "inteligencias", "inventario", "iridio",
"iris", "irreverencia", "irreverencias", "jornalistica",
"jornalisticas", "jornalistico", "lagrima", "lagrimas",
"lamina", "laminas", "legitimo", "licitas",
"licito", "lirica", "liricas", "lirico",
"litigio", "logica", "logicas", "logico",
"lucifero", "macunaima", "Macunaima", "mafia",
"mafias", "mafua", "mafuas", "magica",
"magicas", "magico", "magnifica", "magnificas",
"magnifico", "maleficio", "maluques", "maniaca",
"maniacas", "maniaco", "maquina",
"maquinas", "matrimonio", "megametro",
"memoria", "memorias", "metafisica", "metafisicas",
"metafisico", "microbio", "micrografo", "milicia",
"milicias", "milimetro", "mimeografo", "mimica",
"mimicas", "mimico", "minuscula", "minusculas",
"minusculo", "misera", "miseras", "misero",
"monossilaba", "monossilabas", "monossilabo", "murmurio",
"musica", "musicas", "musico", "naufrago",
"necropsia", "necropsias", "nevoa", "nevoas",
"obsequio", "obstaculo", "oficio", "omega",
"oraculo", "osculo", "pacifico", "pagina",
"paginas", "paleofilo", "paleografo", "paleto",
"palmatoria", "palmatorias",
"pandego", "panico", "pantografo", "paralitica",
"paraliticas", "paralitico", "Parana",
"parodia", "parodias", "Paroquia", "paroquia",
"Paroquias", "paroquias", "pascoa", "pascoas",
"pavia", "pavias", "pendencia", "pendencias",
"pendulo", "penes", "penitencia", "penitencias",
"perdulario", "perfei"+u"\u00E7"+"oes",
"perola", "perolas", "piramide", "piramides",
"pirografo", "placida", "placidas", "placido",
"plagio", "plastica", "plasticas", "plastico",
"polemica", "polemicas", "polemico", "poligrafa",
"poligrafas", "poligrafo", "porem", "porno",
"pornos", "potencia", "potencias", "preambulo",
"prefacio", "preferencia", "preferencias", "premio",
"prenuncio", "presagio", "presidio", "pressagio",
"principio", "prodiga", "prodigas", "prodigo",
"prologo", "pronostico", "prontuario", "propicio",
"proposito", "proverbio", "Proverbio", "prudencia",
"prudencias", "psicanalise", "psicanalises", 
 "psicoanalise", "psicoanalises", "psicologa",
"psicologas", "psicologo", "quadrupla", "quadruplas",
"quadruplo", "quilometro", "radio", "reciproca",
"reciprocas", "reciproco", "residencia", "residencias",
"reticencia", "reticencias", "retorica", "retoricas",
"retorico", "retrograda", "retrogradas",
"retrogrado", "retrogrado", "reusa", "reusas",
"reuso", "rodizio", "Romania", "rufio",
"rustica", "rusticas", "rustico", "sache",
"saches", "salario", "saltica", "salticas",
"saltico", "Secretario", "secretario", "seculo",
"sequio", "serie", "series", "serigrafo", "serio",
"silaba", "silabas", "silabo", "simbolo",
"sincope", "sincopes", "sindica",
"sindicas", "sindico", "sitio", "solida",
"solidas", "solido", "sonambula", "sonambulas",
"sonambulo", "soporifero", "subespecifica", "subespecificas",
"subespecifico", "Subsecretario", "subsecretario", "subsidio",
"substancia", "substancias",
"sudorifica", "sudorificas", "sudorifico", "sumaria",
"sumarias", "Sumario", "sumario", "sumula",
"sumulas", "suplicio",
"taboa", "taboas", "tabua", "tabuas",
"tendencia", "tendencias", "tipica", "tipicas",
"tipico", "tipografo", "titulo", "Titulo",
"tomografo", "toner", "topografo", "toxica",
"toxicas", "toxico", "trafico", "tragedia",
"tragedias", "translucida", "translucidas", "translucido",
"tremula", "tremulas", "tremulo", "tremulo",
"triangulo", "tripe", "tropega",
"tropegas", "tropego", "tropico", "tumulo",
"ulcera", "ulceras", "ultima", "ultimas",
"ultimo", "unanime", "unanimes", "urticaria",
"veiculo", "vermifugo", "virgula", "virgulas",
"vitima", "vitimas",
"vitoria", "vitorias", "voluntaria", "voluntarias",
"voluntario", "xilografo", "xoco", "zoografo"])


aaGrafia = {"abita": "habita",
"abitam": "habitam",
"abitar": "habitar",
"abitaram": "habitaram",
"abitarem": "habitarem",
"abitaria": "habitaria",
"abitariam": "habitariam",
"abitasse": "habitasse",
"abitassem": "habitassem",
"abitava": "habitava",
"abitavam": "habitavam",
"abite": "habite",
"abitei": "habitei",
"abitem": "habitem",
"abito": "habitou",
"abitou": "habitou",
"arroxa": "arrocha",
"arroxam": "arrocham",
"arroxar": "arrochar",
"arroxaram": "arrocharam",
"arroxarem": "arrocharem",
"arroxasse": "arrochasse",
"arroxassem": "arrochassem",
"arroxava": "arrochava",
"arroxavam": "arrochavam",
"arroxe": "arroche",
"arroxei": "arrochei",
"arroxem": "arrochem",
"arroxo": "arrocho",
"arroxou": "arrochou", 
"asar": "azar",
"Asar": "Azar",
"asia": "azia",
"Asia": u"\u00C1"+"sia",
"asseita": "aceita",
"asseitam": "aceitam",
"asseitar": "aceitar",
"asseitaram": "aceitaram",
"asseitarem": "aceitarem",
"asseitasse": "aceitasse",
"asseitassem": "aceitassem",
"asseitava": "aceitava",
"asseitavam": "aceitavam",
"asseite": "aceite",
"asseitei": "aceitei",
"asseitem": "aceitem",
"asseito": "aceito",
"asseitou": "aceitou", 
"asserta": "acerta",
"assertam": "acertam",
"assertar": "acertar",
"assertaram": "acertaram",
"assertarem": "acertarem",
"assertasse": "acertasse",
"assertassem": "acertassem",
"assertava": "acertava",
"assertavam": "acertavam",
"asserte": "acerte",
"assertei": "acertei",
"assertem": "acertem",
"asserto": "acerto",
"assertou": "acertou",
"aza": "asa",
"azas": "asas",
"azinha": "asinha",
"azinhas": "asinhas",
"cerra": "serra",
"cerrar": "serrar",
"cerras": "serras",
"estrema": "extrema",
"estremas": "extremas",
"estremo": "extremo",
"estremos": "extremos",
"facha": "faixa",
"fachas": "faixas",
"jia": "gia",
"jias": "gias",
"liza": "lisa",
"lizas": "lisas",
"lizo": "liso",
"mofina": "morfina",
"maz": "mas",
"n"+u"\u00ED"+"veo": "n"+u"\u00ED"+"vel",
"pence": "pense",
"pences": "penses",
"rezina": "resina",
"rezinas": "resinas",
"sega": "cega",
"segas": "cegas",
"sego": "cego",
"sine": "cine",
"treno": "treino",
"trocha": "trouxa",
"trochas": "trouxas",
"X"+u"\u00E1"+"cara": "Ch"+u"\u00E1"+"cara",
"x"+u"\u00E1"+"cara": "ch"+u"\u00E1"+"cara",
"x"+u"\u00E1"+"caras": "Ch"+u"\u00E1"+"caras"}

acGrafia = set(["abita", "abitam", "abitar", "abitaram", "abitarem",
"abitaria", "abitariam", "abitasse", "abitassem", "abitava", "abitavam",
"abite", "abitei", "abitem", "abito", "abitou", "arroxa", "arroxam",
"arroxar", "arroxaram", "arroxarem", "arroxasse", "arroxassem",
"arroxava", "arroxavam", "arroxe", "arroxei", "arroxem", "arroxo",
"arroxou", "asar", "Asar", "asia", "Asia", "asseita", "asseitam",
"asseitar", "asseitaram",
"asseitarem", "asseitasse", "asseitassem", "asseitava", "asseitavam",
"asseite", "asseitei", "asseitem", "asseito", "asseitou", "asserta",
"assertam", "assertar", "assertaram", "assertarem", "assertasse",
"assertassem", "assertava", "assertavam", "asserte", "assertei",
"assertem", "asserto", "assertou", "aza",  "azas", "azinha", "azinhas",
"cerra", "cerrar", "cerras", "estrema", "estremas", "estremo", "estremos",
"facha", "fachas", "jia", "jias", "liza", "lizas", "lizo", "maz", "mofina",
"n"+u"\u00ED"+"veo", "pence", "pences", "rezina", "rezinas", "sega", "segas",
"sego", "treno", "trocha", "trochas", "X"+u"\u00E1"+"cara",
"x"+u"\u00E1"+"cara", "x"+u"\u00E1"+"caras"])


comCrase = set(["anterior", "anteriores", "compara", "comparada",
"comparadas", "equivale", "equivalente", "equivalentes",
"iguais", "igual", "inferior", "inferiores", "oposta",
"opostas", "paralela", "paralelas", "perpendicular", "perpendiculares", "posterior",
"posteriores", "semelhante", "semelhantes", "similar", "similares",
"superior", "superiores"])


semCrase = set(["acidente", "acidentes", "acolher", u"\u00E1"+"cool",
"algu"+u"\u00E9"+"m", "algum", "alguma", "algumas", "alguns", "ambas",
"ambiente", "ambientes", "bem", "bordoadas", "bra"+u"\u00E7"+"adas",
 "cabe"+u"\u00E7"+"adas", "cacetadas", "cada", "celular",
"celulares", "chibatadas", "convite", "convites", "crime", "crimes",
"cuja", "cujas", "Deus", "diesel", "dizem", "dois",
"ela", "elas", "ele", "eletricidade", "eles", "encher", "encolher",
"esc"+u"\u00E2"+"ncaras", "escolher", "essa",
"essas", "esse", "esses", "esta", "estas", "este",
"estes", "facadas", "faz", "fez", "filme", "filmes", "fim", "fiz", "galope",
"gasolina", "granel", "homem", "homens", "Jesus", "joelhadas", "jusante",
"l"+u"\u00E1"+"pis", "lenha", "leste", "lufadas", "lume", "mais", "mal",
"males",  "marteladas", "melhor", "mim", "montante", "nome",
"ningu"+u"\u00E9"+"m", "nomes", "noroeste", "nordeste", "norte",
"n"+u"\u00F3"+"s", "oeste", "outrem", "pauladas", "p"+u"\u00E3"+"o", "perfume",
"p"+u"\u00E9", "pilha", "pior", "pontap"+u"\u00E9"+"s", "porradas", "porretadas",
"preencher", "presta"+u"\u00E7"+u"\u00F5"+"es", "punhaladas", "querosene",
"recolher", "rigor", "rosas", "sangue", "sangues", "s"+u"\u00F3"+"s", "sudeste",
"sudoeste", "sul", "tem", "ti", "time", "times", "toda", "todas", "tolher",
"toque", "transporte", "transportes", "trote", "um", "uma", "vapor",
"volume", "volumes", "v"+u"\u00F3"+"s",
"vossa", "Vossa", "vossas", "Zeus"])


#femeCrase  - Palavras femininas terminadas em -o-
femeCrase = set(["acess"+u"\u00E3"+"o", "Acess"+u"\u00E3"+"o",
"acupress"+u"\u00E3"+"o", "Acupress"+u"\u00E3"+"o", "ades"+u"\u00E3"+"o", "Ades"+u"\u00E3"+"o",
"admiss"+u"\u00E3"+"o", "Admiss"+u"\u00E3"+"o", "adust"+u"\u00E3"+"o", "Adust"+u"\u00E3"+"o",
"agress"+u"\u00E3"+"o", "Agress"+u"\u00E3"+"o", "alus"+u"\u00E3"+"o", "Alus"+u"\u00E3"+"o",
"amarelid"+u"\u00E3"+"o", "Amarelid"+u"\u00E3"+"o", "amplid"+u"\u00E3"+"o", "Amplid"+u"\u00E3"+"o",
"animadvers"+u"\u00E3"+"o", "Animadvers"+u"\u00E3"+"o", "antevis"+u"\u00E3"+"o", "Antevis"+u"\u00E3"+"o",
"apreens"+u"\u00E3"+"o", "Apreens"+u"\u00E3"+"o", "aptid"+u"\u00E3"+"o", "Aptid"+u"\u00E3"+"o",
"ascens"+u"\u00E3"+"o", "Ascens"+u"\u00E3"+"o", "aspers"+u"\u00E3"+"o", "Aspers"+u"\u00E3"+"o",
"autogest"+u"\u00E3"+"o", "Autogest"+u"\u00E3"+"o", "avers"+u"\u00E3"+"o", "Avers"+u"\u00E3"+"o",
"bernard"+u"\u00E3"+"o", "Bernard"+u"\u00E3"+"o", "bioimpress"+u"\u00E3"+"o",
"Bioimpress"+u"\u00E3"+"o", "branquid"+u"\u00E3"+"o", "Branquid"+u"\u00E3"+"o",
"brusquid"+u"\u00E3"+"o", "Brusquid"+u"\u00E3"+"o", "certid"+u"\u00E3"+"o", "Certid"+u"\u00E3"+"o",
"cess"+u"\u00E3"+"o", "Cess"+u"\u00E3"+"o", "Cinco", "cinco", "circuncis"+u"\u00E3"+"o", "Circuncis"+u"\u00E3"+"o",
"cis"+u"\u00E3"+"o", "Cis"+u"\u00E3"+"o", "coaliz"+u"\u00E3"+"o", "Coaliz"+u"\u00E3"+"o",
"coes"+u"\u00E3"+"o", "Coes"+u"\u00E3"+"o", "colis"+u"\u00E3"+"o", "Colis"+u"\u00E3"+"o",
"combust"+u"\u00E3"+"o", "Combust"+u"\u00E3"+"o", "comich"+u"\u00E3"+"o", "Comich"+u"\u00E3"+"o",
"comiss"+u"\u00E3"+"o", "Comiss"+u"\u00E3"+"o", "compaix"+u"\u00E3"+"o", "Compaix"+u"\u00E3"+"o",
"compreens"+u"\u00E3"+"o", "Compreens"+u"\u00E3"+"o", "compress"+u"\u00E3"+"o", "Compress"+u"\u00E3"+"o",
"compuls"+u"\u00E3"+"o", "Compuls"+u"\u00E3"+"o", "comunh"+u"\u00E3"+"o", "Comunh"+u"\u00E3"+"o",
"concess"+u"\u00E3"+"o", "Concess"+u"\u00E3"+"o", "concis"+u"\u00E3"+"o", "Concis"+u"\u00E3"+"o",
"conclus"+u"\u00E3"+"o", "Conclus"+u"\u00E3"+"o", "concuss"+u"\u00E3"+"o", "Concuss"+u"\u00E3"+"o",
"conex"+u"\u00E3"+"o", "Conex"+u"\u00E3"+"o", "confiss"+u"\u00E3"+"o", "Confiss"+u"\u00E3"+"o",
"confus"+u"\u00E3"+"o", "Confus"+u"\u00E3"+"o", "congest"+u"\u00E3"+"o", "Congest"+u"\u00E3"+"o",
"contram"+u"\u00E3"+"o",  "Contarraz"+u"\u00E3"+"o", "contarraz"+u"\u00E3"+"o",
"contus"+u"\u00E3"+"o", "Contus"+u"\u00E3"+"o", "convers"+u"\u00E3"+"o",
"Convers"+u"\u00E3"+"o", "convex"+u"\u00E3"+"o", "Convex"+u"\u00E3"+"o", "convuls"+u"\u00E3"+"o",
"Convuls"+u"\u00E3"+"o", "corras"+u"\u00E3"+"o", "Corras"+u"\u00E3"+"o",
"corros"+u"\u00E3"+"o", "Corros"+u"\u00E3"+"o", "cosmovis"+u"\u00E3"+"o", "Cosmovis"+u"\u00E3"+"o",
"crucifix"+u"\u00E3"+"o", "Crucifix"+u"\u00E3"+"o", "Cubat"+u"\u00E3"+"o", "d"+u"\u00E3"+"o",
"D"+u"\u00E3"+"o", "decis"+u"\u00E3"+"o", "Decis"+u"\u00E3"+"o", "defens"+u"\u00E3"+"o",
"Defens"+u"\u00E3"+"o", "deflex"+u"\u00E3"+"o", "Deflex"+u"\u00E3"+"o", "demiss"+u"\u00E3"+"o",
"Demiss"+u"\u00E3"+"o", "densid"+u"\u00E3"+"o", "Densid"+u"\u00E3"+"o", "depress"+u"\u00E3"+"o",
"Depress"+u"\u00E3"+"o", "deris"+u"\u00E3"+"o", "Deris"+u"\u00E3"+"o", "descens"+u"\u00E3"+"o",
"Descens"+u"\u00E3"+"o", "descompress"+u"\u00E3"+"o", "Descompress"+u"\u00E3"+"o", "desconex"+u"\u00E3"+"o",
"Desconex"+u"\u00E3"+"o", "desilus"+u"\u00E3"+"o", "Desilus"+u"\u00E3"+"o", "Deso", "DESO",
"despretens"+u"\u00E3"+"o", "Despretens"+u"\u00E3"+"o", "desuni"+u"\u00E3"+"o",
"Desuni"+u"\u00E3"+"o", "devassid"+u"\u00E3"+"o", "Devassid"+u"\u00E3"+"o",
"Dezoito", "dezoito", "difus"+u"\u00E3"+"o", "Difus"+u"\u00E3"+"o", "digest"+u"\u00E3"+"o",
"Digest"+u"\u00E3"+"o", "digress"+u"\u00E3"+"o", "Digress"+u"\u00E3"+"o", "dimens"+u"\u00E3"+"o",
"Dimens"+u"\u00E3"+"o", "discuss"+u"\u00E3"+"o", "Discuss"+u"\u00E3"+"o", "dispers"+u"\u00E3"+"o",
"Dispers"+u"\u00E3"+"o", "dissuas"+u"\u00E3"+"o", "Dissuas"+u"\u00E3"+"o", "distens"+u"\u00E3"+"o",
"Distens"+u"\u00E3"+"o", "divers"+u"\u00E3"+"o", "Divers"+u"\u00E3"+"o", "divis"+u"\u00E3"+"o",
"Divis"+u"\u00E3"+"o", "do", "Do", "dos",
"Dos", "eclos"+u"\u00E3"+"o", "Eclos"+u"\u00E3"+"o", "efus"+u"\u00E3"+"o",
"Efus"+u"\u00E3"+"o", "egest"+u"\u00E3"+"o", "Egest"+u"\u00E3"+"o", "electrocuss"+u"\u00E3"+"o",
"Electrocuss"+u"\u00E3"+"o", "eletrofus"+u"\u00E3"+"o", "Eletrofus"+u"\u00E3"+"o", "emers"+u"\u00E3"+"o",
"Emers"+u"\u00E3"+"o", "emiss"+u"\u00E3"+"o", "Emiss"+u"\u00E3"+"o", "emuls"+u"\u00E3"+"o",
"Emuls"+u"\u00E3"+"o", "ent"+u"\u00E3"+"o", "entrevis"+u"\u00E3"+"o", "Entrevis"+u"\u00E3"+"o", "eros"+u"\u00E3"+"o",
"Eros"+u"\u00E3"+"o", "escans"+u"\u00E3"+"o", "Escans"+u"\u00E3"+"o",
"escravid"+u"\u00E3"+"o", "Escravid"+u"\u00E3"+"o", "escrupulid"+u"\u00E3"+"o",
"Escrupulid"+u"\u00E3"+"o", "escurid"+u"\u00E3"+"o", "Escurid"+u"\u00E3"+"o", "espessid"+u"\u00E3"+"o",
"Espessid"+u"\u00E3"+"o", "evas"+u"\u00E3"+"o", "Evas"+u"\u00E3"+"o", "Ex-pris"+u"\u00E3"+"o",
"exatid"+u"\u00E3"+"o", "Exatid"+u"\u00E3"+"o", "exaust"+u"\u00E3"+"o", "Exaust"+u"\u00E3"+"o",
"excis"+u"\u00E3"+"o", "Excis"+u"\u00E3"+"o", "exclus"+u"\u00E3"+"o", "Exclus"+u"\u00E3"+"o",
"excomunh"+u"\u00E3"+"o", "Excomunh"+u"\u00E3"+"o", "excurs"+u"\u00E3"+"o", "Excurs"+u"\u00E3"+"o",
"expans"+u"\u00E3"+"o", "Expans"+u"\u00E3"+"o", "explos"+u"\u00E3"+"o", "Explos"+u"\u00E3"+"o",
"express"+u"\u00E3"+"o", "Express"+u"\u00E3"+"o", "expuls"+u"\u00E3"+"o", "Expuls"+u"\u00E3"+"o",
"extens"+u"\u00E3"+"o", "Extens"+u"\u00E3"+"o", "extors"+u"\u00E3"+"o", "Extors"+u"\u00E3"+"o",
"extrovers"+u"\u00E3"+"o", "Extrovers"+u"\u00E3"+"o", "extrus"+u"\u00E3"+"o", "Extrus"+u"\u00E3"+"o",
"f"+u"\u00E3"+"o", "F"+u"\u00E3"+"o", "fiss"+u"\u00E3"+"o", "Fiss"+u"\u00E3"+"o",
"flex"+u"\u00E3"+"o", "Flex"+u"\u00E3"+"o", "fonofoto", "Fonofoto", "foto", "Foto",
"fotoemiss"+u"\u00E3"+"o", "Fotoemiss"+u"\u00E3"+"o", "fotofiss"+u"\u00E3"+"o", "Fotofiss"+u"\u00E3"+"o",
"fotos", "Fotos", "Fraz"+u"\u00E3"+"o", "fresquid"+u"\u00E3"+"o",
"Fresquid"+u"\u00E3"+"o", "frouxid"+u"\u00E3"+"o", "Frouxid"+u"\u00E3"+"o", "fus"+u"\u00E3"+"o",
"Fus"+u"\u00E3"+"o", "gest"+u"\u00E3"+"o", "Gest"+u"\u00E3"+"o", "gratid"+u"\u00E3"+"o",
"Gratid"+u"\u00E3"+"o", "hipertens"+u"\u00E3"+"o", "Hipertens"+u"\u00E3"+"o", "hipotens"+u"\u00E3"+"o",
"Hipotens"+u"\u00E3"+"o", "ilus"+u"\u00E3"+"o", "Ilus"+u"\u00E3"+"o", "imensid"+u"\u00E3"+"o",
"Imensid"+u"\u00E3"+"o", "imers"+u"\u00E3"+"o", "Imers"+u"\u00E3"+"o", "imiss"+u"\u00E3"+"o",
"Imiss"+u"\u00E3"+"o", "implos"+u"\u00E3"+"o", "Implos"+u"\u00E3"+"o", "imprecis"+u"\u00E3"+"o",
"Imprecis"+u"\u00E3"+"o", "impress"+u"\u00E3"+"o", "Impress"+u"\u00E3"+"o", "impuls"+u"\u00E3"+"o",
"Impuls"+u"\u00E3"+"o", "imunodepress"+u"\u00E3"+"o", "Imunodepress"+u"\u00E3"+"o", "inaptid"+u"\u00E3"+"o",
"Inaptid"+u"\u00E3"+"o", "incis"+u"\u00E3"+"o", "Incis"+u"\u00E3"+"o", "inclus"+u"\u00E3"+"o",
"Inclus"+u"\u00E3"+"o", "incompreens"+u"\u00E3"+"o", "Incompreens"+u"\u00E3"+"o", "incurs"+u"\u00E3"+"o",
"Incurs"+u"\u00E3"+"o", "indecis"+u"\u00E3"+"o", "Indecis"+u"\u00E3"+"o", "indigest"+u"\u00E3"+"o",
"Indigest"+u"\u00E3"+"o", "indivis"+u"\u00E3"+"o", "Indivis"+u"\u00E3"+"o", "inexatid"+u"\u00E3"+"o",
"Inexatid"+u"\u00E3"+"o", "inexpress"+u"\u00E3"+"o", "Inexpress"+u"\u00E3"+"o", "inflex"+u"\u00E3"+"o",
"Inflex"+u"\u00E3"+"o", "infus"+u"\u00E3"+"o", "Infus"+u"\u00E3"+"o", "ingest"+u"\u00E3"+"o",
"Ingest"+u"\u00E3"+"o", "ingratid"+u"\u00E3"+"o", "Ingratid"+u"\u00E3"+"o", "insubmiss"+u"\u00E3"+"o",
"Insubmiss"+u"\u00E3"+"o", "intens"+u"\u00E3"+"o", "Intens"+u"\u00E3"+"o", "intercess"+u"\u00E3"+"o",
"Intercess"+u"\u00E3"+"o", "interconex"+u"\u00E3"+"o", "Interconex"+u"\u00E3"+"o", "intromiss"+u"\u00E3"+"o",
"Intromiss"+u"\u00E3"+"o", "introvers"+u"\u00E3"+"o", "Introvers"+u"\u00E3"+"o", "intrus"+u"\u00E3"+"o",
"Intrus"+u"\u00E3"+"o", "invas"+u"\u00E3"+"o", "Invas"+u"\u00E3"+"o", "invers"+u"\u00E3"+"o",
"Invers"+u"\u00E3"+"o", "irreflex"+u"\u00E3"+"o", "Irreflex"+u"\u00E3"+"o", "irreligi"+u"\u00E3"+"o",
"Irreligi"+u"\u00E3"+"o", "irris"+u"\u00E3"+"o", "Irris"+u"\u00E3"+"o", "lassid"+u"\u00E3"+"o",
"Lassid"+u"\u00E3"+"o", "legi"+u"\u00E3"+"o", "Legi"+u"\u00E3"+"o", "lentid"+u"\u00E3"+"o",
"Lentid"+u"\u00E3"+"o", "les"+u"\u00E3"+"o", "Les"+u"\u00E3"+"o", "libido",
"Libido", "lipo", "Lipo", "lipos", "Lipos", "macro", "Macro", "macros", "Macros",
"mans"+u"\u00E3"+"o", "Mans"+u"\u00E3"+"o", "mansid"+u"\u00E3"+"o",
"Mansid"+u"\u00E3"+"o", "m"+u"\u00E3"+"o", "M"+u"\u00E3"+"o", "m"+u"\u00E3"+"os",
"M"+u"\u00E3"+"os", "micro", "Micro", "micros", "Micros",
"miss"+u"\u00E3"+"o", "Miss"+u"\u00E3"+"o", "mornid"+u"\u00E3"+"o",
"Mornid"+u"\u00E3"+"o", "moto", "Moto", "motos", "Motos", "motos",
"multid"+u"\u00E3"+"o", "Multid"+u"\u00E3"+"o", "negrid"+u"\u00E3"+"o",
"Negrid"+u"\u00E3"+"o", "obsess"+u"\u00E3"+"o", "Obsess"+u"\u00E3"+"o", "ocasi"+u"\u00E3"+"o",
"Ocasi"+u"\u00E3"+"o", "oclus"+u"\u00E3"+"o", "Oclus"+u"\u00E3"+"o", "oito", "Oito", "omiss"+u"\u00E3"+"o",
"Omiss"+u"\u00E3"+"o", "opini"+u"\u00E3"+"o", "Opini"+u"\u00E3"+"o", "opress"+u"\u00E3"+"o",
"Opress"+u"\u00E3"+"o", "paix"+u"\u00E3"+"o", "Paix"+u"\u00E3"+"o", "pens"+u"\u00E3"+"o",
"Pens"+u"\u00E3"+"o", "percuss"+u"\u00E3"+"o", "Percuss"+u"\u00E3"+"o", "perdig"+u"\u00E3"+"o",
"Perdig"+u"\u00E3"+"o", "perfus"+u"\u00E3"+"o", "Perfus"+u"\u00E3"+"o", "permiss"+u"\u00E3"+"o",
"Permiss"+u"\u00E3"+"o", "perplex"+u"\u00E3"+"o", "Perplex"+u"\u00E3"+"o", "persuas"+u"\u00E3"+"o",
"Persuas"+u"\u00E3"+"o", "pervers"+u"\u00E3"+"o", "Pervers"+u"\u00E3"+"o", "podrid"+u"\u00E3"+"o",
"Podrid"+u"\u00E3"+"o", "possess"+u"\u00E3"+"o", "Possess"+u"\u00E3"+"o", "precess"+u"\u00E3"+"o",
"Precess"+u"\u00E3"+"o", "precis"+u"\u00E3"+"o", "Precis"+u"\u00E3"+"o", "preclus"+u"\u00E3"+"o",
"Preclus"+u"\u00E3"+"o", "preens"+u"\u00E3"+"o",
"Preens"+u"\u00E3"+"o", "press"+u"\u00E3"+"o", "Press"+u"\u00E3"+"o", "pretens"+u"\u00E3"+"o",
"Pretens"+u"\u00E3"+"o", "pretid"+u"\u00E3"+"o", "Pretid"+u"\u00E3"+"o", "previs"+u"\u00E3"+"o",
"Previs"+u"\u00E3"+"o", "pris"+u"\u00E3"+"o", "Pris"+u"\u00E3"+"o", "prociss"+u"\u00E3"+"o",
"Prociss"+u"\u00E3"+"o", "profiss"+u"\u00E3"+"o", "Profiss"+u"\u00E3"+"o", "profus"+u"\u00E3"+"o",
"Profus"+u"\u00E3"+"o", "progress"+u"\u00E3"+"o", "Progress"+u"\u00E3"+"o", "prontid"+u"\u00E3"+"o",
"Prontid"+u"\u00E3"+"o", "propens"+u"\u00E3"+"o", "Propens"+u"\u00E3"+"o", "propuls"+u"\u00E3"+"o",
"Propuls"+u"\u00E3"+"o", "provis"+u"\u00E3"+"o", "Provis"+u"\u00E3"+"o", "puls"+u"\u00E3"+"o",
"Puls"+u"\u00E3"+"o", "quatro", "Quatro", "quest"+u"\u00E3"+"o", "Quest"+u"\u00E3"+"o", 
"R"+u"\u00E1"+"dio", "r"+u"\u00E1"+"dio", "radiodifus"+u"\u00E3"+"o",
"Radiodifus"+u"\u00E3"+"o", "radiotelevis"+u"\u00E3"+"o", "Radiotelevis"+u"\u00E3"+"o", "ras"+u"\u00E3"+"o",
"Ras"+u"\u00E3"+"o", "raz"+u"\u00E3"+"o", "Raz"+u"\u00E3"+"o", "readmiss"+u"\u00E3"+"o",
"Readmiss"+u"\u00E3"+"o", "rebeli"+u"\u00E3"+"o", "Rebeli"+u"\u00E3"+"o", "recens"+u"\u00E3"+"o",
"Recens"+u"\u00E3"+"o", "recess"+u"\u00E3"+"o", "Recess"+u"\u00E3"+"o", "reclus"+u"\u00E3"+"o",
"Reclus"+u"\u00E3"+"o", "reconvers"+u"\u00E3"+"o", "Reconvers"+u"\u00E3"+"o", "reflex"+u"\u00E3"+"o",
"Reflex"+u"\u00E3"+"o", "regi"+u"\u00E3"+"o", "Regi"+u"\u00E3"+"o", "regress"+u"\u00E3"+"o",
"Regress"+u"\u00E3"+"o", "reimpress"+u"\u00E3"+"o", "Reimpress"+u"\u00E3"+"o", "reinvers"+u"\u00E3"+"o",
"Reinvers"+u"\u00E3"+"o", "religi"+u"\u00E3"+"o", "Religi"+u"\u00E3"+"o", "remiss"+u"\u00E3"+"o",
"Remiss"+u"\u00E3"+"o", "repercuss"+u"\u00E3"+"o", "Repercuss"+u"\u00E3"+"o", "repreens"+u"\u00E3"+"o",
"Repreens"+u"\u00E3"+"o", "repress"+u"\u00E3"+"o", "Repress"+u"\u00E3"+"o", "repuls"+u"\u00E3"+"o",
"Repuls"+u"\u00E3"+"o", "rescis"+u"\u00E3"+"o", "Rescis"+u"\u00E3"+"o", "retid"+u"\u00E3"+"o",
"Retid"+u"\u00E3"+"o", "retransmiss"+u"\u00E3"+"o", "Retransmiss"+u"\u00E3"+"o", "retrovers"+u"\u00E3"+"o",
"Retrovers"+u"\u00E3"+"o", "reuni"+u"\u00E3"+"o", "Reuni"+u"\u00E3"+"o", "revers"+u"\u00E3"+"o",
"Revers"+u"\u00E3"+"o", "revis"+u"\u00E3"+"o", "Revis"+u"\u00E3"+"o", "rez"+u"\u00E3"+"o",
"Rez"+u"\u00E3"+"o", "rouquid"+u"\u00E3"+"o", "Rouquid"+u"\u00E3"+"o", "saz"+u"\u00E3"+"o",
"sans"+u"\u00E3"+"o", "Saz"+u"\u00E3"+"o", "secess"+u"\u00E3"+"o", "Secess"+u"\u00E3"+"o", "sequid"+u"\u00E3"+"o",
"Sequid"+u"\u00E3"+"o", "servid"+u"\u00E3"+"o", "Servid"+u"\u00E3"+"o", "sess"+u"\u00E3"+"o",
"Sess"+u"\u00E3"+"o", "sobrefus"+u"\u00E3"+"o", "Sobrefus"+u"\u00E3"+"o", "sofreguid"+u"\u00E3"+"o",
"Sofreguid"+u"\u00E3"+"o", "soid"+u"\u00E3"+"o", "Soid"+u"\u00E3"+"o", "solid"+u"\u00E3"+"o",
"Solid"+u"\u00E3"+"o", "soliflux"+u"\u00E3"+"o", "Soliflux"+u"\u00E3"+"o", "su"+u"\u00E3"+"o",
"Su"+u"\u00E3"+"o", "suas"+u"\u00E3"+"o", "Suas"+u"\u00E3"+"o", "subcomiss"+u"\u00E3"+"o",
"Subcomiss"+u"\u00E3"+"o", "subdivis"+u"\u00E3"+"o", "Subdivis"+u"\u00E3"+"o", "submers"+u"\u00E3"+"o",
"Submers"+u"\u00E3"+"o", "submiss"+u"\u00E3"+"o", "Submiss"+u"\u00E3"+"o", "subvers"+u"\u00E3"+"o",
"Subvers"+u"\u00E3"+"o", "sucess"+u"\u00E3"+"o", "Sucess"+u"\u00E3"+"o", "sugest"+u"\u00E3"+"o",
"Sugest"+u"\u00E3"+"o", "superpress"+u"\u00E3"+"o", "supervis"+u"\u00E3"+"o",
"Supervis"+u"\u00E3"+"o", "supress"+u"\u00E3"+"o",
"Supress"+u"\u00E3"+"o", "suspens"+u"\u00E3"+"o", "Suspens"+u"\u00E3"+"o", "televis"+u"\u00E3"+"o",
"Televis"+u"\u00E3"+"o", "tens"+u"\u00E3"+"o", "Tens"+u"\u00E3"+"o", "transfus"+u"\u00E3"+"o",
"Transfus"+u"\u00E3"+"o", "transgress"+u"\u00E3"+"o", "Transgress"+u"\u00E3"+"o", "transmiss"+u"\u00E3"+"o",
"Transmiss"+u"\u00E3"+"o", "tribo", "Tribo", "tribos",
"Tribos", "uni"+u"\u00E3"+"o", "Uni"+u"\u00E3"+"o", "vaguid"+u"\u00E3"+"o",
"Vaguid"+u"\u00E3"+"o", "vastid"+u"\u00E3"+"o", "Vastid"+u"\u00E3"+"o", "vaz"+u"\u00E3"+"o",
"Vaz"+u"\u00E3"+"o", "vermelhid"+u"\u00E3"+"o", "Vermelhid"+u"\u00E3"+"o", "vers"+u"\u00E3"+"o",
"Vers"+u"\u00E3"+"o", "vis"+u"\u00E3"+"o", "Vis"+u"\u00E3"+"o"])



masAoCrase = set(["artes"+u"\u00E3"+"o", "Artes"+u"\u00E3"+"o",
"blus"+u"\u00E3"+"o", "Blus"+u"\u00E3"+"o",
"bras"+u"\u00E3"+"o", "Bras"+u"\u00E3"+"o",
"cabe"+u"\u00E7"+u"\u00E3"+"o", "Cabe"+u"\u00E7"+u"\u00E3"+"o",
"ca"+u"\u00E7"+u"\u00E3"+"o", "Ca"+u"\u00E7"+u"\u00E3"+"o",
"cal"+u"\u00E7"+u"\u00E3"+"o", "Cal"+u"\u00E7"+u"\u00E3"+"o",
"cora"+u"\u00E7"+u"\u00E3"+"o", "Cora"+u"\u00E7"+u"\u00E3"+"o",
"cortes"+u"\u00E3"+"o", "Cortes"+u"\u00E3"+"o",
"diapas"+u"\u00E3"+"o", "Diapas"+u"\u00E3"+"o",
"fais"+u"\u00E3"+"o", "Fais"+u"\u00E3"+"o",
"gar"+u"\u00E7"+u"\u00E3"+"o", "Gar"+u"\u00E7"+u"\u00E3"+"o",
"pis"+u"\u00E3"+"o", "Pis"+u"\u00E3"+"o",
"sals"+u"\u00E3"+"o", "Sals"+u"\u00E3"+"o",
"s"+u"\u00E3"+"o", "S"+u"\u00E3"+"o",
"ti"+u"\u00E7"+u"\u00E3"+"o", "Ti"+u"\u00E7"+u"\u00E3"+"o",
"travess"+u"\u00E3"+"o", "Travess"+u"\u00E3"+"o"])


malHifen = set(["acondicionada", "acondicionadas", "acondicionado",
"acondicionados", "aconselhada", "aconselhadas", "aconselhado",
"aconselhados", "acostumada", "acostumadas", "acostumado", "acostumados",
"adaptada", "adaptadas", "adaptado", "adaptados", "afamada", "afamadas",
"afamado", "afamados", "afortunada", "afortunadas", "afortunado", "afortunados",
"agourada", "agouradas", "agourado", "agourados", "agradecida", "agradecidas",
"agradecido", "agradecidos", "ajambrada", "ajambradas", "ajambrado",
"ajambrados", "ajeitada", "ajeitadas", "ajeitado", "ajeitados", "amada",
"amadas", "amado", "amados", "apessoada", "apessoadas", "apessoado",
"apessoados", "armada", "armadas", "armado", "armados", "arranjada",
"arranjadas", "arranjado", "arranjados", "assada", "assadas", "assado",
"assados", "assombrada", "assombradas", "assombrado", "assombrados",
"assombramento", "assombramentos", "assombro", "assombros", "aventurada",
"aventuradas", "aventurado", "aventurados", "avisada", "avisadas", "avisado",
"avisados", "educada", "educadas", "educado", "educados", "empregada",
"empregadas", "empregado", "empregado", "encarada", "encaradas", "encarado",
"encarados", "enganado", "ensinado", "entendido", "entendidos", "estar",
"executada", "executadas", "executado", "executados", "habituada",
"habituadas", "habituado", "habituados", "humorada", "humoradas", "humorado",
"humorados", "informada", "informadas", "informado", "informados",
"intencionada", "intencionadas", "intencionado", "intencionados", "limpo",
"limpos"])

malMau = set(["agouro", "aluno", "aspecto", "atendimento", "bocado",
"cheiro", "colesterol", "conselho", "costume" "desempenho", "elemento",
"estado", "exemplo", "gosto", "jeito", "momento", "motorista",
"negociante", "profissional", "resultado", "sinal", "tempo", "uso"])

# AosExcecao = excecao para os prefixos auto, super, inter, sub, pseudo etc 

AosExcecao = set(["a", "agora", "ante", "antes", "ao", "aos", "as", "cada",
"censura", "ciber", "citroque", "como", "comporta", "conditione", "da", "das", "de", "depois", "do", "dois",
"dos", "duas", "e", "eis", "ela", "elas", "em", "entre", "essa", "essas",
"esse", "esses", "esta", "estas", "este", "estes", "estava", "esteve",
"estive", "examine", "extra", "extras", "feita", "feitas", "feito", "feitos", "faz",
"fez", "fica", "foi", "funciona", "giga", "gosta", "gostamos", "gostaria",
"gostava", "goste", "gosto", "gostos", "hasta", "infra", 
"inter", "isso", "isto", "judice", "j"+u"\u00FA"+"dice",
"kb", "kbyte", "kbytes", "kilo", "kilobyte", "lege", "mais", "mas", "me", "mega",
"megas", "meia", "meio", "mesma", "mesmas", "mesmo", "mesmos", "minha",
"minhas", "muita", "muitas", "muito", "muitos", "na", "nas", "no", "nos",
"nossa", "nossas", "nosso", "nossos", "o", "os", "outra", "outras", "outro",
"outros", "para", "pare", "pede", "pode", "poderia", "pois", "posso",
"pouca", "poucas", "pouco", "poucos", "pra", "porque", "pros", "quaisquer",
"qual", "quando", "qualquer", "que", "quem", "quis", "quer", "saiba", "se", "seja",
"sejam", "sente", "sentem", "somente", "sua", "suas", "super", "suporta",
"supra", "tem", "ter", "tera", "teus", "teve", "tinha", "tive", "toda", "todas",
"todo", "todos", "tua", "tuas", "tudo", "uma", "umas", "usa", "usam", "uso",
"vamos", "vem", "vez", "vinha", "vossa", "vossas", "vosso",
"vossos"])


meSes = set(["janeiro", "fevereiro", "mar"+u"\u00E7"+"o", "abril", "maio", "junho",
"julho", "agosto", "setembro", "outubro", "novemvro", "dezembro"])

seMana = set(["segunda", "ter"+u"\u00E7"+"a", "quarta", "quinta", "sexta",
"s"+u"\u00E1"+"bado", "segunda-feira", "ter"+u"\u00E7"+"a-feira",
"quarta-feira" "quinta-feira", "sexta-feira"])


preAnalise = set(["a", "alguma", "breve", "com", "cuja", "da", "de", "dessa",
"desta", "em", "essa", "esta", "esta", "farei", "faria", "fariam", "faz",
"fazem", "fazemos", "fazia", "faziam", "feita", "fez", "fiz", "fizemos",
"fizeram", "minha", "muita", "na", "nessa", "nesta", "nossa", "nova", "numa",
"outra", "para", "pela", "por", "posterior", "pouca", "primeira", "profunda",
"qualquer", "segunda", "segundo", "simples", "sob", "sua", "tamanha", "tanta",
"tem", "toda", "uma"])


posAnalise = set(["da", "das", "de", "dos", "farei", "faria", "fariam", "faz",
"fazem", "fazemos", "fazia", "faziam", "fez", "fiz", "fizemos", "fizeram",
"tem"])

preEsta = set(["agora", "ainda", "como", "e", "ela", "ele", "hoje",
"j"+u"\u00E1", "mas", "me", "te", "lhe", "n"+u"\u00E3"+"o", "que", "quem",
"se", "tamb"+u"\u00E9"+"m", "voc"+u"\u00EA"])

posEsta = set(["agora", "ali", "ao", "aqui", "assim", "bem", "bom",
"cada", "certo", "cheia", "cheio", "claro", "com", "como", "dentro",
"disposta", "disposto", "dito", "doente", "ela", "ele", "em", "entre",
"escrito", "feita", "feito", "fora", "hoje", "longe", "mais", "mal", "me",
"melhor", "mesmo", "meu", "morto", "muito", "na", "nas", "no", "nos",
"num", "numa", "o", "para", "pior", "presente", "preso",
"prevista", "previsto", "pronto", "sendo", "seu",
"sujeito", "teu", "todo", "tudo", "um", "visto", "vivo"])

excpreEsta = set(["el", "tod"])


excvelEsta = set(["abomin"+u"\u00E1"
"admir"+u"\u00E1", "ador"+u"\u00E1", "agrad"+u"\u00E1",
"am"+u"\u00E1", "apetec"+u"\u00ED", "desagrad"+u"\u00E1",
"desej"+u"\u00E1", "detest"+u"\u00E1", "formid"+u"\u00E1",
"horr"+u"\u00ED", "impag"+u"\u00E1", "imposs"+u"\u00ED",
"incans"+u"\u00E1", "incompar"+u"\u00E1", "incompreens"+u"\u00ED",
"inesgot"+u"\u00E1", "inexor"+u"\u00E1", "inexplic"+u"\u00E1",
"inqualific"+u"\u00E1", "intermin"+u"\u00E1", "invis"+u"\u00ED",
"irredut"+u"\u00ED", "lament"+u"\u00E1", "memor"+u"\u00E1",
"miser"+u"\u00E1", "not"+u"\u00E1", "poss"+u"\u00ED",
"prov"+u"\u00E1", "respeit"+u"\u00E1", "respons"+u"\u00E1",
"saud"+u"\u00E1", "terr"+u"\u00ED", "vari"+u"\u00E1",
"vulner"+u"\u00E1"])


proClise = set(["abafa", "abala", "abalei", "abalou", "abana",
"abandonei", "abandonou", "abateu", "abati", "abominei", "abominou",
"aborreceu", "aborreci", "aborrecia", "abracei", "abracou", "abria",
"abroquela", "absolveu", "absolvi", "absorvia", "acabrunha", "acalma",
"acanhava", "acariciei", "acariciou", "aceita", "aceitava",
"aceitei", "aceito", "aceitou", "acendeu", "acendi", "acendia", "acerta",
"acha", "achava", "achei", "acho", "achou", "aclama", "acolheu",
"acolhi", "acolhia", "acometeu", "acometia", "acompanha",
"acompanhava", "acompanhei", "acompanhou", "aconchega", "aconchegava",
"aconselha", "aconselhava", "aconselhei", "aconselhou",
"acorda", "acordava", "acordei", "acordou", "acuda", "acusa",
"adivinha", "adivinhei", "adivinhou", "admira", "admirava", "admita",
"admitia", "adora", "adorava", "adota", "advirta", "afaga",
"afagava", "afasta", "afastava", "afastei", "afastou", "afirma",
"afirmava", "afligia", "afoga", "afugenta", "agarra", "agarrei",
"agarrou", "agasalha", "agita", "agitava", "aguardava", "aguenta", "ajeita",
"ajuda", "ajudava", "ajudei", "ajudou", "ajusta", "alcancei",
"alegrava", "alimenta", "alimentei", "alimentou", "alucina", "ama",
"amarra", "amava", "ameacei", "amei", "amo", "amou", "amparei",
"amparou", "anima", "animava", "animou", "aniquila", "anula",
"anulou", "apaga", "apagou", "apague", "apalpa", "apalpei", "apalpou",
"apanha", "apanhei", "apanhou", "aperta", "apertava", "apertei",
"apertou", "aplaudia", "apoiei", "apoiou", "apresenta",
"apresentei", "apresentou", "aproximou", "aqueceu", "arranca", "arrancou",
"arranquei", "arrasta", "arrastava", "arrastei", "arrastou",
"arrebata", "arrebatou", "arremessa", "arremessei", "arremessou", "arruma",
"arrumei", "arrumou", "assa", "assalta", "assaltou", "assombra", "assombrou",
"assou", "assusta", "assustava", "assustou", "atacou", "atendeu", "atendi",
"aterra", "atesta", "atira", "atirei", "atirou", "atormenta",
"atormentava", "autorizei", "autorizou", "avistei", "avistou",
"baixa", "bateu", "bati", "bebeu", "bebi", "beija", "beijava",
"beijei", "beijou", "bota", "botava", "botei", "botou", "buscou", "busquei",
"cansou", "caracteriza", "carrega", "carregava", "carregou", "carreguei",
"casei", "casou", "castiga", "castigava", "castigou", "castiguei", "cega",
"censurava", "cerca", "cercou", "chama", "chamava", "chamei",
"chamou", "chegou", "cheguei", "choca", "cobria", "cola",
"colheu", "colhi", "coloca", "colocou", "coloquei", "colou",
"coma", "comeu", "comi", "comia", "comoveu", "comovi", "comovia",
"completa", "comprei", "compro", "comprou", "comunica",
"concebeu", "concebi", "condenei", "condenou", "conduza",
"confirmou", "conheceu", "conheci", "conhecia", "conquista", "consagrei",
"consagrou", "conseguira", "conservei", "conservou",
"considerava", "considerei", "considero", "considerou", "consola",
"consolava", "consolei", "consolou", "constrangia", "contaminei",
"contaminou", "contava", "contempla", "contemplava",
"contemplei", "contemplou", "continha", "contorceu", "contrariava",
"convenceu", "convenci", "converteu", "converti", "convida",
"convidava", "convidei", "convidou", "copia", "correu", "corri",
"corta", "cortou", "cria", "criei", "criou", "cumprimenta",
"cumprimentei", "cumprimentou", "dava", "decorei", "decorou", "defendeu",
"defendi", "defendia", "dei", "deita", "deitei", "deitou", "deitou", "deixa",
"deixava", "deixei", "deixo", "deixou", "derrama", "derramou",
"desconhecia", "descreveu", "descrevi", "desejava", "desejei", "desejou",
"desenhou", "desequilibrou", "despejei", "despejou", "desperta", "despertei",
"despertou", "despreza", "desprezei", "desprezou", "detesta", "detestava",
"detinha", "deu", "devolveu", "devolvi", "devora", "devorava", "devorei",
"devorou", "dilacerou", "dispensou", "ditei", "doma", "domina", "dominava",
"dominei", "dominou", "dou", "edita", "editou", "educa", "educou", "eduquei",
"elegeu", "embala", "embalava", "embalou", "embrulha", "embrulhou", "empurra",
"empurrei", "empurrou", "encaminha", "encanta", "encantava",
"encarava", "encarei", "encarou", "encarrega", "encarregou",
"encarreguei", "enchei", "encheu", "enchi", "enchia", "encontra",
"encontrava", "encontrei", "encontrou", "encoraja", "encostei",
"encostou", "endireita", "enfia", "enfureceu", "enrolei", "enrolou",
"enterrei", "enterrou", "entrega", "entregava", "entregou",
"entreguei", "entretinha", "entristeceu", "entristeci", "entristecia",
"entusiasmava", "envolveu", "envolvi", "envolvia", "enxota",
"enxotou", "ergueu", "ergui", "erguia", "escolheu", "escolhi", "esconde",
"escondeu", "escondi", "escondia", "escorri", "escuta", "escutava",
"escutei", "escutou", "esfrega", "esfregou", "esmaga", "esmagava", "esmagou",
"esmurrei", "esmurrou", "espanca", "espanta", "esperava",
"esperei", "espero", "esperou", "espreita", "esqueceu", "esqueci", "estendeu",
"estendi", "estendia", "estimula", "estimulei", "estimulo", "estimulou",
"estrangula", "estuda", "estudei", "estudo", "estudou", "evita", "evitava",
"examina", "examinava", "examinei", "examinou", "excitava",
"exorta", "expulsa", "expulsou", "expunha", "fabricou", "fabriquei", "fareja",
"fascina", "fascinava", "fecha", "fechei", "fechou", "feriu",
"fita", "fitava", "fitei", "fitou", "fixei", "fixou", "folheei",
"folheou", "forcei", "fortaleceu", "fortaleci", "furei", "furou", "ganha",
"ganhei", "ganhou", "gastei", "gastou", "guarda", "guardava",
"guardei", "guardou", "humilhava", "ignora", "imaginava", "imita",
"imitei", "imitou", "imobiliza", "imortaliza", "impedia",
"impelia", "impugna", "impunha", "incomoda", "incomodava", "indenize",
"inibia", "inquietava", "instala", "instiga", "insulta", "interdizia",
"interessa", "interpela", "interpelei", "interpelou", "interpretei",
"interpretou", "interroga", "interrogava", "interrogou", "interroguei",
"interrompeu", "interrompi", "interrompia", "invadia",
"inveja", "invejava", "invejo", "invejou", "irrita", "irritava", "irritei",
"irritou", "isola", "joga", "jogou", "joguei", "julga", "julgava",
"julgou", "julguei", "junta", "junte", "jurei", "jurou", "lancei", "larga",
"largou", "larguei", "lavei", "lavou", "leu", "leva",
"levanta", "levantei", "levantou", "levava", "levei", "levo", "levou", "li",
"lia", "liberta", "liga", "limpa", "livrei", "livrou", "magoava",
"manda", "mandava", "mandei", "mandou", "maneja", "mata",
"matei", "mato", "matou", "meteu", "meti", "modifica", "modifico",
"modificou", "monta", "mordeu", "mordi", "mordia", "mostra", "mostrava",
"mostrei", "mostrou", "moveu", "movi", "movia", "movo", "nobilita", "nomeei",
"nomeia", "nomeou", "obriga", "obrigava", "obrigou", "obriguei",
"observava", "observei", "obtinha", "odiava", "odiei", "odiou",
"ofendeu", "ofendi", "olha", "olhava", "olhei", "olho", "olhou",
"oprimia", "ordena", "ouvia", "ouvira", "paga", "pagou", "paguei",
"parta", "partilha", "passa", "passava", "passei", "passou", "pega", "pegou",
"peguei", "percebeu", "percebi", "percebia", "percorreu", "percorri",
"percorria", "perdeu", "perdi", "perseguia", "perturba", "perturbava", "pica",
"pinta", "pisa", "planta", "plantei", "plantou", "ponha", "precedeu",
"precedi", "preencheu", "preenchi", "preferia", "prenda",
"prendeu", "prendi", "prendia", "preocupa", "preocupava", "pressentia",
"previa", "procura", "procurava", "procurei", "procurou",
"promoveu", "promovi", "protegeu", "protegi", "proteja", "prova", "provoca",
"publica", "publicou", "publiquei", "pungia", "punha", "puxa",
"puxava", "puxei", "puxou", "queima", "queira", "queria", "quero", "rasga",
"rasgou", "rasguei", "realiza", "receba", "recebeu", "recebi", "recebia",
"recolheu", "recolhi", "recoloca", "recomenda", "reconheceu",
"reconheci", "reconhecia", "recorda", "recordava", "reduzia", "reencontrei",
"reencontrou", "rejeita", "relega", "releu", "reli", "remeteu", "remeti",
"remoeu", "renomeei", "renomeou", "repassa", "repelia",
"repetia", "repito", "repreendeu", "repreendi", "repreendia", "representa",
"reproduzia", "resolveu", "resolvi", "respeita", "respeitei", "respeitou",
"retarda", "retenha", "retinha", "retirei", "retirou", "retrata",
"revela", "risca", "rompeu", "rompi", "roubei", "roubou",
"sacudia", "salvei", "salvou", "saudei", "saudou", "seduzia", "seguia",
"segura", "segurei", "segurou", "senta", "sentei",
"sentia", "sentou", "servia", "siga", "sinto", "solta", "soltei",
"soltou", "submeteu", "submeti", "substitua", "suga",
"sujeita", "suplanta", "suporta", "supunha", "surpreendeu", "surpreendi",
"suspendeu", "suspendi", "sustenta", "tangeu", "tangi", "tapei", "tapou",
"temia", "tenha", "tenho", "testa", "tinha", "tira", "tirava",
"tirei", "tirou", "toca", "tocava", "tolda", "tolheu", "tolhi", "toma",
"tomava", "tomba", "tomei", "tomou", "torna", "tornava", "tornei", "tornou",
"torturava", "trabalha", "traga", "trancou", "tranquei", "tranquilizei",
"tranquilizou", "transforma", "transformei", "transformou", "trata",
"tratava", "tratei", "tratou", "trazia", "trucida", "ultrapassa",
"utiliza", "veda", "veja", "vejo", "venceu", "venci", "vencia", "venda",
"vendeu", "vendi", "verteu", "vestia", "via", "vibrou", "vinga",
"vira", "virei", "virou", "visitava", "volta"]) 


aaPlural = set(["Do", "do", "Meu", "meu", "No", "no", "Nosso",
"nosso", "O", "o", "Outro", "outro", "Pelo", "pelo", "Seu", "seu", "Teu", "teu", "Todo",
"todo", "Vosso", "vosso"])

# aaSingular - Palavras no singular que possam ir para o plural com simples adicao de -s- para formacao do plural-singular com a palavra seguinte
aaSingular = set(["A", "Alta", "alta", "Alto", "alto", "Amiga", "amiga",
"Amigo", "amigo", "Antiga", "antiga", "Antigo", "antigo", "Ao", "ao",
"Aquela", "aquela", "Aquele", "aquele", "Assunto", "assunto",
"Ato", "ato", "Baixa", "baixa", "Baixo", "baixo", "Bela", "bela", "Belo",
"belo", "Bonita", "bonita", "Bonito", "bonito",
"Certa", "certa", "Certo", "certo", "Cuja", "cuja", "Cujo", "cujo",
"Da", "da", "Derradeira", "derradeira", "Derradeiro", "derradeiro", "dessa",
"Dessa", "desta", "Desta", "deste", "Deste", "determinada",
"Determinada", "determinado", "Determinado", "Diversa",
"diversa", "Diverso", "diverso", "Do", "do", "Elemento", "elemento",
"Erro", "erro", "essa", "Essa", "esse", "Esse", "esta", "Esta", "este",
"Este", "Gasto", "gasto", "Guerreira", "guerreira", "Guerreiro", "guerreiro",
"Linda", "linda", "Lindo", "lindo", "Mesma",
"Meu", "meu", "Moderna", "moderna", "Moderno", "moderno", "Minha",
"minha", "Na", "na", "nessa", "Nessa", "nesse", "Nesse", "nesta", "Nesta",
"neste", "Neste", "No", "no", "Nossa", "nossa", "Nosso", "nosso", "Nova",
"nova", "Novo", "novo", "O", "o", "Olho", "olho", "Organismo", "organismo",
"Outra", "outra", "Outro", "outro", "Pela", "pela", "Pelo", "pelo", "Pequena",
"pequena", "Pequeno", "pequeno", "Primeira", "primeira", "Primeiro", "primeiro",
"Seu", "seu", "Sua", "sua", "Tanta", "tanta", "Teu", "teu", "Tua", "tua",
"Velha", "velha", "Velho", "velho", "Vossa", "vossa", "Vosso",
"vosso"])

# abSingular - Palavras masculinas no singular  para formacao do plural-singular com adjetivos
abSingular = set(["Alto", "alto", "Amigo", "amigo", "Antigo", "antigo", "Ao",
"ao", "Aquele", "aquele",
"Ato", "ato", "Baixo", "baixo", "Belo", "belo", "Bonito", "bonito",
"Certo", "certo", "Cujo", "cujo", "Derradeiro", "derradeiro",
"Deste", "determinado", "Determinado", "Diverso", "diverso",
"Do", "do", "Elemento", "elemento","Erro", "erro", "esse", "Esse", "este", "Este","Gasto", "gasto",
"Grupo", "grupo", "Guerreiro", "guerreiro", "Lindo", "lindo", "Meu",
"meu", "Moderno", "moderno", "nesse",
"Nesse", "neste", "Neste", "No", "no", "Novo", "novo", "Nosso", "nosso",
"O", "o", "Olho", "olho", "Organismo", "organismo", "Outro", "outro",
"Pelo", "pelo", "Pequeno", "pequeno", "Primeiro", "primeiro", "Seu",
"seu", "Teu", "teu", "Velho", "velho", "Vosso", "vosso"])


# acSingular - Palavras femininas no singular  para formacao do plural-singular com a palavra seguinte
acSingular = set(["A", "Alta", "alta", "Amiga", "amiga", "Antiga", "antiga",
"Aquela", "aquela", "Baixa", "baixa", "Bela", "bela", "Bonita", "bonita", "Carga", "carga",
"Certa", "certa", "Cuja", "cuja", "Da", "da", "Derradeira", "derradeira",
"Dessa", "dessa", "Desta", "desta", "determinada", "Determinada",
"Diversa", "diversa", "essa", "Essa", "esta", "Esta", "Guerreira",
"guerreira", "Linda", "linda", "Mesma", "Minha",
"minha",  "Moderna", "moderna", "Na", "na", "nessa", "Nessa",  "nesta",
"Nesta", "Nossa", "nossa", "Nova", "nova", "Outra", "outra", "Pela",
"pela", "Pequena", "pequena", "Primeira", "primeira", "Sua", "sua", "Tanta", "tanta",
"Tua", "tua", "Velha", "velha", "Vossa", "vossa"])


# adSingular - Idem abSingular, mas sem o No - no
adSingular = set(["Alto", "alto", "Amigo", "amigo", "Antigo", "antigo", "Ao",
"ao", "Aquele", "aquele", "Assunto", "assunto", "Ato", "ato", "Baixo", "baixo",
"Belo", "belo", "Bonito", "bonito",
"Cargo", "cargo", "Certo", "certo", "Cujo", "cujo", "Derradeiro", "derradeiro",
"Deste", "deste", "Determinado", "determinado",
"Diverso", "diverso", "Do", "do", "Elemento", "elemento",
"Erro", "erro","Esse", "esse", "Este", "este", "Gasto", "gasto", "Grupo", "grupo", "Guerreiro", "guerreiro",
"Lindo", "lindo", "Meu", "meu", "Moderno", "moderno",
"nesse", "Nesse", "neste", "Neste", "Nosso", "nosso", "Novo",
"novo", "O", "o", "Olho", "olho", "Organismo", "organismo", "Outro", "outro",
"Pelo", "pelo", "Pequeno", "pequeno", "Primeiro", "primeiro",
"Seu", "seu", "Teu", "teu", "Velho", "velho", "Vosso", "vosso"])


# aeSingular - Palavras femininas no singular  para formacao do plural-singular  - terminadas em ado ido 
aeSingular = set(["Alta", "alta", "Amiga", "amiga", "Antiga", "antiga", "Aquela", "aquela", "Baixa",
"baixa", "Bela", "bela", "Bonita", "bonita", "Certa", "certa",
"Cuja", "cuja", "Da", "da", "Derradeira", "derradeira", "determinada",
"Determinada", "Diversa", "diversa", "Guerreira", "guerreira", "Linda",
"linda", "Minha", "minha",
"Moderna", "moderna", "Na", "na",  "Nossa", "nossa", "Nova", "nova", "Outra",
"outra", "Pela", "pela", "Pequena", "pequena", "pouca", "Pouca", "Primeira", "primeira",
"Sua", "sua", "Tanta", "tanta", "Toda", "toda",
"Tua", "tua", "Velha", "velha", "Vossa", "vossa"])

# agSingular - Palavras femininas no singular  para formacao do plural-singular com a palavra seguinte SEM o A
agSingular = set(["Alta", "alta", "Amiga", "amiga", "Antiga", "antiga",
"Aquela", "aquela", "Baixa", "baixa", "Bela", "bela", "Bonita", "bonita", "Carga", "carga",
"Certa", "certa", "Cuja", "cuja", "Da", "da", "Derradeira", "derradeira",
"Dessa", "dessa", "Desta", "desta", "determinada", "Determinada",
"Diversa", "diversa", "essa", "Essa", "esta", "Esta", "Guerreira",
"guerreira", "Linda", "linda", "Longa", "longa", "Mesma", "Minha",
"minha",  "Moderna", "moderna", "Na", "na", "nessa", "Nessa",  "nesta",
"Nesta", "Nossa", "nossa", "Nova", "nova", "Outra", "outra", "Pela",
"pela", "Pequena", "pequena", "Primeira", "primeira", "Sua", "sua", "Tanta", "tanta",
"Tua", "tua", "Velha", "velha", "Vossa", "vossa"])


# ahSingular - Palavras masculinas no singular iniciadas com maiusculas
ahSingular = set(["Alto", "Amigo", "Antigo",
"Aquele", "Baixo", "Belo", "Bonito",
"Certo", "Determinado", "Diverso",
"Esse", "Este", "Grande", "Grupo", "Guerreiro", "Lindo", "Meu",
"Moderno", "Novo", "Nosso",
"O", "Outro", "Pequeno", "Primeiro", "Seu",
"Teu", "Velho", "Vosso"])


# ajSingular - Palavras femininas no singular iniciadas com maiusculas
ajSingular = set(["A", "Amiga", "Antiga",
"Aquela", "Baixa", "Bela", "Bonita",
"Certa", "Diversa", "Essa", "Esta", "Guerreira",
"Linda", "Minha",
"Moderna", "Nossa", "Nova", "Outra", "Pela",
"Pequena", "Primeira", "Sua", "Tanta",
"Tua", "Velha", "Vossa"])









# varGenero - Variacao de genero -  feminino - masculino - - - - - masculino - masculino
varGenero = {"a": "o", "A": "O", u"\u00C0": "Ao", u"\u00E0": "ao",
"alta": "alto", "Alta": "Alto", "alto": "alto", "Alto": "Alto",
"amiga": "amigo", "Amiga": "Amigo", "amigo": "amigo", "Amigo": "Amigo",
"antiga": "antigo", "Antiga": "Antigo", "antigo": "antigo", "Antigo": "Antigo",
"ao": "ao", "Ao": "Ao",
"aquela": "aquele", "Aquela": "Aquele",
"aquele": "aquele", "Aquele": "Aquele",
"artigo": "artigo", "Artigo": "Artigo",
"assunto": "assunto", "Assunto": "Assunto",
"ato": "ato", "Ato": "Ato",
"baixa": "baixo", "Baixa": "Baixo", "baixo": "baixo", "Baixo": "Baixo",
"bela": "belo", "Bela": "Belo", "belo": "belo", "Belo": "Belo",
"bonita": "bonito", "Bonita": "Bonito", "bonito": "bonito", "Bonito": "Bonito",
"carga": "cargo", "Carga": "Cargo", "cargo": "cargo", "Cargo": "Cargo",
"certa": "certo", "Certa": "Certo", "certo": "certo", "Certo": "Certo",
"cuja": "cujo", "Cuja": "Cujo",
"cujo": "cujo", "Cujo": "Cujo", "da": "do", "Da": "Do",
"Derradeira": "Derradeiro", "derradeira": "derradeiro",
"Derradeiro": "Derradeiro", "derradeiro": "derradeiro",
"dessa": "desse",
"Dessa": "Desse", "desse": "desse",  "Desse": "Desse", "desta": "deste",
"Desta": "Deste", "deste": "deste",  "Deste": "Deste",
"determinada": "determinado", "Determinada": "Determinado",
"determinado": "determinado",  "Determinado": "Determinado",
"Direito": "Direito", "direito": "direito",
"Diversa": "Diverso", "diversa": "diverso",
"Diverso": "Diverso", "diverso": "diverso",
"Do": "Do", "do": "do", "doi": "doi", "Doi": "Doi", "dua": "doi",
"Dua": "Doi", 
"Elemento": "Elemento", "elemento": "elemento",
"Erro": "Erro", "erro": "erro",
"essa": "esse", "Essa": "Esse", "esse": "esse",
"Esse": "Esse", "esta": "este", "Esta": "Este", "este": "este",
"Este": "Este",
"Gasto": "Gasto", "gasto": "gasto",
"Grupo": "Grupo", "grupo": "grupo",
"Guerreira": "Guerreiro", "guerreira": "guerreiro", 
"Guerreiro": "Guerreiro", "guerreiro": "guerreiro",
"Linda": "Lindo", "linda": "lindo",
"Lindo": "Lindo", "lindo": "lindo",
"Longa": "Longo", "longa": "longo",
"Longo": "Longo", "longo": "longo",
"mesma": "mesmo", "Mesma": "Mesmo",
"mesmo": "mesmo", "Mesmo": "Mesmo", "meu": "meu",
"Meu": "Meu", "minha": "meu", "Minha": "Meu",
"moderna": "moderno", "Moderna": "Moderno",
"moderno": "moderno", "Moderno": "Moderno", "muita": "muito",
"Muita": "Muito", "muito": "muito", "Muito": "Muito", "na": "no",
"Na": "No", "nessa": "nesse", "Nessa": "Nesse", "nesse": "nesse",
"Nesse": "Nesse", "nesta": "neste", "Nesta": "Neste", "neste": "neste",
"Neste": "Neste", "no": "no", "No": "No", "nossa": "nosso",
"Nossa": "Nosso", "nosso": "nosso", "Nosso": "Nosso",
"o": "o", "O": "O",
"Olho": "Olho", "olho": "olho",
"Organismo": "Organismo", "organismo": "organismo",
 "outra": "outro", "Outra": "Outro",
"outro": "outro", "Outro": "Outro",
"pela": "pelo", "Pela": "Pelo", "pelo": "pelo", "Pelo": "Pelo",
"pequena": "pequeno", "Pequena": "Pequeno", "pequeno": "pequeno", "Pequeno": "Pequeno",
"pouca": "pouco", "Pouca": "Pouco", "pouco": "pouco", "Pouco": "Pouco",
"primeira": "primeiro", "Primeira": "Primeiro", "primeiro": "primeiro",
"Primeiro": "Primeiro",
"seu": "seu", "Seu": "Seu", "sua": "seu", "Sua": "Seu", 
"Terceira": "Terceiro", "terceira": "terceiro",
"Terceiro": "Terceiro", "terceiro": "terceiro",
"teu": "teu", "Teu": "Teu", "toda": "todo", "Toda": "Todo", "todo": "todo",
"Todo": "Todo", "tua": "teu", "Tua": "Teu", "un": "um", "Un": "Um",
"um": "un", "Um": "Un", "uma": "um", "Uma": "Um", "vossa": "vosso",
"Vossa": "Vosso", "vosso": "vosso", "Vosso": "Vosso"}


# varGenera - Variacao de genero -  masculino - feminino - - - - feminino - feminino
varGenera = {"a": "a", "A": "A", u"\u00C0": u"\u00C0", u"\u00E0": u"\u00E0",
"alta": "alta", "Alta": "Alta", "alto": "alta", "Alto": "Alta",
"amiga": "amiga", "Amiga": "Amiga", "amigo": "amiga", "Amigo": "Amiga",
"antiga": "antiga", "Antiga": "Antiga", "antigo": "antiga", "Antigo": "Antiga",
"ao": u"\u00E0", "Ao": u"\u00C0",
"aquela": "aquela", "Aquela": "Aquela",
"aquele": "aquela", "Aquele": "Aquela",
"baixa": "baixa", "Baixa": "Baixa", "baixo": "baixa", "Baixo": "Baixa",
"bela": "bela", "Bela": "Bela", "belo": "bela", "Belo": "Bela",
"bonita": "bonita", "Bonita": "Bonita", "bonito": "bonita", "Bonito": "Bonita",
"carga": "carga", "Carga": "Carga", "cargo": "carga", "Cargo": "Carga",
"certa": "certa", "Certa": "Certa", "certo": "certa", "Certo": "Certa",
"cuja": "cuja", "Cuja": "Cuja",
"cujo": "cuja", "Cujo": "Cuja", "da": "da", "Da": "Da",
"Derradeira": "Derradeira", "derradeira": "derradeira",
"Derradeiro": "Derradeira", "derradeiro": "derradeira",
"dessa": "dessa",
"Dessa": "Dessa", "desse": "dessa",  "Desse": "Dessa", "desta": "desta",
"Desta": "Desta", "deste": "desta",  "Deste": "Desta",
"determinada": "determinada", "Determinada": "Determinada",
"determinado": "determinada",  "Determinado": "Determinada",
"Diversa": "Diversa", "diversa": "diversa",
"Diverso": "Diversa", "diverso": "diversa",
"do": "da", "Do": "Da", "doi": "dua", "Doi": "Dua", "dua": "dua",
"Dua": "Dua", "essa": "essa", "Essa": "Essa", "esse": "essa",
"Esse": "Essa", "esta": "esta", "Esta": "Esta", "este": "esta",
"Este": "Esta", 
"Guerreira": "Guerreira", "guerreira": "guerreira", 
"Guerreiro": "Guerreira", "guerreiro": "guerreira",
"Linda": "Linda", "linda": "linda",
"Lindo": "Linda", "lindo": "linda",
"Longa": "Longa", "longa": "longa",
"Longo": "Longa", "longo": "longa",
"mesma": "mesma", "Mesma": "Mesma",
"mesmo": "mesma", "Mesmo": "Mesma", "meu": "minha",
"Meu": "Minha", "minha": "minha", "Minha": "Minha",
"moderna": "moderna", "Moderna": "Moderna", "moderno": "moderna",
"Moderno": "Moderna", "muita": "muita",
"Muita": "Muita", "muito": "muita", "Muito": "Muita", "na": "na",
"Na": "Na", "nessa": "nessa", "Nessa": "Nessa", "nesse": "nessa",
"Nesse": "Nessa", "nesta": "nesta", "Nesta": "Nesta", "neste": "nesta",
"Neste": "Nesta", "no": "na", "No": "Na", "nossa": "nossa",
"Nossa": "Nossa", "nosso": "nossa", "Nosso": "Nossa",
"o": "a", "O": "A", "outra": "outra", "Outra": "Outra",
"outro": "outra", "Outra": "Outra", "pela": "pela",
"Pela": "Pela", "pelo": "pela", "Pelo": "Pela", 
"pequena": "pequena", "Pequena": "Pequena", "pequeno": "pequena", "Pequeno": "Pequena",
"pouca": "pouca", "Pouca": "Pouca", "pouco": "pouca", "Pouco": "Pouca",
"primeira": "primeira", "Primeira": "Primeira", "primeiro": "primeira",
"Primeiro": "Primeira",
"Pr"+u"\u00F3"+"pria": "Pr"+u"\u00F3"+"pria",
"pr"+u"\u00F3"+"pria": "pr"+u"\u00F3"+"pria",
"Pr"+u"\u00F3"+"prio": "Pr"+u"\u00F3"+"pria",
"pr"+u"\u00F3"+"prio": "pr"+u"\u00F3"+"pria",
"Pr"+u"\u00F3"+"xima": "Pr"+u"\u00F3"+"xima",
"pr"+u"\u00F3"+"xima": "pr"+u"\u00F3"+"xima",
"Pr"+u"\u00F3"+"ximo": "Pr"+u"\u00F3"+"xima",
"pr"+u"\u00F3"+"ximo": "pr"+u"\u00F3"+"xima",
"seu": "sua", "Seu": "Sua", "sua": "sua", "Sua": "Sua",
"tanta": "tanta", "Tanta": "Tanta",
"Terceira": "Terceira", "terceira": "terceira",
"Terceiro": "Terceira", "terceiro": "terceira",
"teu": "tua", "Teu": "Tua", "toda": "toda", "Toda": "Toda", "todo": "toda",
"Todo": "Toda", "tua": "tua", "Tua": "Tua", "un": "uma", "Un": "Uma",
"uma": "uma", "Uma": "Uma", "vossa": "vossa", "Vossa": "Vossa",
"vosso": "vossa", "Vosso": "Vossa"}


excExiste = set(["alfere", "Ante", "ante", "Ao", "ao", "apena", "Cai", "cai", "Cao", "cao", "Da", "da", "Dela",
"dela", "Dele", "dele", "Do", "do", "mai", "meno", "Na", "na", "naquela", "naquele", "No", "no", "noutra", "noutro", "simple"])

excDa = set(["carta", "curta", "descoberta", "entrada",
"oferta", "porta"])

excEmFrente = set(["com", "de", "em", "minha", "nossa", "pela", "sua", "vossa"])

corPais = set(["algum", "antigo", "ao", "belo", "bom", "cada",
"cent"+u"\u00E9"+"simo", "certo", "d"+u"\u00E9"+"cimo", "daquele",
"desse", "deste", "distante", "do", "doce", "doutro", "esse", "este",
"extenso", "grande", "heptag"+u"\u00E9"+"simo", "hexag"+u"\u00E9"+"simo",
"igual", "imenso", "jovem", "maior",  "melhor", "mesmo", "meu", "naquele",
"nenhum", "nesse", "neste", "no", "nonag"+u"\u00E9"+"simo", "nono", "nosso",
"noutro", "num", "o", "octag"+u"\u00E9"+"simo", "oitavo", "outro",
"pelo", "pequeno", "p"+u"\u00E9"+"ssimo", "pior", "poderoso",
"pr"+u"\u00F3"+"ximo", "primeiro",
"quadrag"+u"\u00E9"+"simo", "quarto", "quinquag"+u"\u00E9"+"simo",
"quinto", "respectivo", "s"+u"\u00E9"+"timo", "segundo", "semelhante",
"seu", "sexto", "tanto", "terceiro", "teu", "todo", "trig"+u"\u00E9"+"simo",
"um", u"\u00FA"+"nico", "vasto", "velho", "vig"+u"\u00E9"+"simo",
"vosso"])


excRenun = set(["a", "ao", "aos", u"\u00E0",u"\u00E0"+"s"])

excDenun = set(["a",  u"\u00E0", "alguma", "antiga", "ap"+u"\u00F3"+"s",
"apresenta",  "apresentar", "apresentou", "apura", "apurando", "apurar",
"aquela", "as", u"\u00E0"+"s", "bela", "boa", "cada", "certa", "clara",
"conforme", "confirma", "cuja", "da", "daquela", "de", "desmente",
"desmentem", "dessa", "desta", "doutra", "em", "encaminha", "encaminhar",
"encaminhou", "essa", "esta", "existe", "existia", "existindo", "falsa",
"formaliza", "formalizada", "formula", "formulada", "grande", "grave",
"h"+u"\u00E1", "houve", "houver", "igual", "investiga", "investigar",
"investigou", "julga", "julgar", "maior", "mais", "mediante", "meia",
"melhor", "mesma", "minha", "naquela", "nenhuma", "nessa", "nesta", "na",
"nossa", "noutra", "nova", "numa", "ofereceu", "outra", "pela", "pequena",
"pior", "por", "poss"+u"\u00ED"+"vel", "preciosa", "presente", "primeira",
"pr"+u"\u00F3"+"xima", "qualquer", "recebe", "recebem", "receber",
"receberam", "receberem", "recebeu", "recebia", "recebiam", "respectiva",
"segunda", "segundo", "semelhante", "sobre", "sexta", "simples", "sua",
"tanta", "tem", "terceira", "tua", "toda", "tremenda", "uma",
u"\u00FA"+"nica", "vossa"])


mauMal = set(["cabid", "casad", "cheiros", "comid",
"comportad", "conceit", "conceituad", "conduzid", "conservad",
"cozinhad", "criad", "cuidad", "dormid", "dotad", "falad",
"feit", "figurad", "furad", "grad", "jeitos", "mandad",
"parecid", "parid", "passad", "pecad", "pesad", "planejad",
"post", "procedid", "pront", "pronunciad",
"prop"+u"\u00ED"+"ci", "proporcionad", "pr"+u"\u00F3"+"pri",
"satisfeit", "sentid", "servid", "sortead", "sucedid",
"trajad", "trapilh", "vestid", "vist"])

excVerbo = set(["estiv", "tiv"])

excGerundio = set(["deve", "devem", "pode", "podem", "vai", "Vamos", "vamos", "v"+u"\u00E3"+"o", "Vou", "vou"])

excGquando = set(["qua", "quando"])

excPosso = set(["quaisqu", "qualqu", "qu", "regul", "retangul"])

posMas = set(["afinal", "agora",
"a"+u"\u00ED", "amanh"+u"\u00E3", "antes", "assim",
"claro", "coitada", "coitado", "confesso", "depois", "enfim",
"entretanto", "hoje", "sim", "sobretudo", "talvez"])


excMas = set(["a", "as", "com", "da", "das", "de", "do", "dos",
"em", "n"+u"\u00E3"+"o", "o", "os", "que", "sem", "um",
"uma"])

# excUgeral  - excecao Plural U Geral - para palavras terminadas em u 
excUgeral = set(["adeu", "campu", "corpu", "ilheu", "juventu", "manau",
"marcu", "mateu", "onibu", "onu", "plu", "pu", "statu", "venu", "viniciu",
"viru", "zeu"])


# excAos = Execao para palavras que nao variam em determinadas situacoes - regra do plural para o singular
excAos = set(["a", "ao", "agora", "amba", "ambo", "ante", "ao", "as", "cada",
"catorze", "cerca", "cinco", "cinquenta", "como", "da", "daquela", "daquele",
"daquilo", "de", "dela", "dele", "dessa", "desse", "desta", "deste",
"dezanove", "dezassete", "dezenove", "dezessete", "dezoito", "do", "doze",
"e", "ela", "entre", "exclusivamente", "extra", "giga", "gravemente",
"infelizmente", "infinitamente", "inter", "injustamente", "me", "mega", "meia",
"meio", "meno", "mini", "muito", "na", "nada", "nessa", "nesse", "nesta",
"neste", "no", "nove", "noventa", "numa", "nunca", "oitenta", "oito", "onze",
"outrora", "perpetuamente", "para", "pouca", "pouco", "pura", "quanta",
"quarenta", "quase", "quatorze", "quatro", "que", "quinze", "sessenta",
"sete", "setenta", "sua", "suas", "treze", "todo", "trinta", "vinte", "zero"])

# aacCao = Concordancia Feminino - masculino - substantivos - adjetivos  terminados em -a-  e  -o-
aacCao = {"a": "o", "A": "O"}

# aacCrase = Concordancia Feminino - masculino - substantivos - adjetivos  terminados em -a-  e  -o-
aacCrase = {u"\u00E0": "a", u"\u00C0": "A"}



# aacCfm = Concordancia Feminino - masculino - substantivos - adjetivos  terminados em or
aacCfm = {"alguma": "algum",
"Amiga": "Amigo",
"amiga": "amigo",
"antiga": "antigo",
"aquela": "aquele",
"boa": "bom",
"Cara": "Caro",
"cara": "caro",
"Da": "Do",
"da": "do",
"digna": "digno",
"Distinta": "Distinto",
"distinta": "distinto",
"Essa": "Esse",
"essa": "esse",
"Esta": "Este",
"esta": "este",
"futura": "futuro",
"leg"+u"\u00ED"+"tima": "leg"+u"\u00ED"+"timo",
"m"+u"\u00E0": "mau",
"mesma": "mesmo",
"minha": "meu",
"muita": "muito",
"Na": "No",
"na": "no",
"nenhuma": "nenhum",
"nona": "nono",
"nossa": "nosso",
"nova": "novo",
"oitava": "oitavo",
"outra": "outro",
"pela": "pelo",
"pouca": "pouco",
"Prezada": "Prezado",
"prezada": "prezado",
"primeira": "primeiro",
"pr"+u"\u00F3"+"pria": "pr"+u"\u00F3"+"prio",
"pr"+u"\u00F3"+"xima": "pr"+u"\u00F3"+"ximo",
"quarta": "quarto",
"quinta": "quinto",
"segunda": "segundo",
"Senhora": "Senhor",
"senhora": "senhor",
"s"+u"\u00E9"+"tima": "s"+u"\u00E9"+"timo",
"sexta": "sexto",
"sua": "seu",
"terceira": "terceiro",
"Uma": "Um",
"uma": "um",
"velha": "velho",
"vossa": "vosso"}

# aacPfas = Concordancia Plural Feminino -  substantivos - adjetivos  terminados em or
aacPfas = {"alguma": "algumas",
"Amiga": "Amigas",
"amiga": "amigas",
"antiga": "antigas",
"aquela": "aquelas",
"boa": "boas",
"Cara": "Caras",
"cara": "caras",
"Da": "Das",
"da": "das",
"digna": "dignas",
"Distinta": "Distintas",
"distinta": "distintas",
"Essa": "Essas",
"essa": "essas",
"Esta": "Estas",
"esta": "estas",
"futura": "futuras",
"leg"+u"\u00ED"+"tima": "leg"+u"\u00ED"+"timas",
"m"+u"\u00E0": "m"+u"\u00E0"+"s",
"mesma": "mesmas",
"minha": "minhas",
"muita": "muitas",
"Na": "Nas",
"na": "nas",
"nenhuma": "nenhumas",
"nona": "nonas",
"nossa": "nossas",
"nova": "novas",
"oitava": "oitavas",
"outra": "outras",
"pela": "pelas",
"pouca": "poucas",
"Prezada": "Prezadas",
"prezada": "prezadas",
"primeira": "primeiras",
"pr"+u"\u00F3"+"pria": "pr"+u"\u00F3"+"prias",
"pr"+u"\u00F3"+"xima": "pr"+u"\u00F3"+"ximas",
"quarta": "quartas",
"quinta": "quintas",
"segunda": "segundas",
"Senhora": "Senhoras",
"senhora": "senhoras",
"s"+u"\u00E9"+"tima": "s"+u"\u00E9"+"timas",
"sexta": "sextas",
"sua": "suas",
"terceira": "terceiras",
"Uma": "Umas",
"uma": "umas",
"velha": "velhas",
"vossa": "vossas"}

# aacPfsa = Concordancia Plural Feminino -  substantivos - adjetivos  terminados em or
aacPfsa = {"alguma": "algumas",
"Amigas": "Amiga",
"amigas": "amiga",
"antigas": "antiga",
"aquelas": "aquela",
"boas": "boa",
"Caras": "Cara",
"caras": "cara",
"Das": "Da",
"das": "da",
"dignas": "digna",
"Distintas": "Distinta",
"distintas": "distinta",
"Essas": "Essa",
"essas": "essa",
"Estas": "Esta",
"estas": "esta",
"futuras": "futura",
"leg"+u"\u00ED"+"timas": "leg"+u"\u00ED"+"tima",
"m"+u"\u00E0"+"s": "m"+u"\u00E0",
"mesmas": "mesma",
"minhas": "minha",
"muitas": "muita",
"Nas": "Na",
"nas": "na",
"nenhumas": "nenhuma",
"nonas": "nona",
"nossas": "nossa",
"novas": "nova",
"oitavas": "oitava",
"outras": "outra",
"pelas": "pela",
"poucas": "pouca",
"Prezadas": "Prezada",
"prezadas": "prezada",
"primeiras": "primeira",
"pr"+u"\u00F3"+"prias": "pr"+u"\u00F3"+"pria",
"pr"+u"\u00F3"+"ximas": "pr"+u"\u00F3"+"xima",
"quartas": "quarta",
"quintas": "quinta",
"segundas": "segunda",
"Senhoras": "Senhora",
"senhoras": "senhora",
"s"+u"\u00E9"+"timas": "s"+u"\u00E9"+"tima",
"sextas": "sexta",
"suas": "sua",
"terceiras": "terceira",
"Umas": "Uma",
"umas": "uma",
"velhas": "velha",
"vossas": "vossa"}



# conCfm = Concordancia Feminino - masculino - substantivos - adjetivos
conCfm = set(["alguma", "Amiga",  "amiga", "antiga", "aquela", "boa", "Cara",
"cara", "Da", "da", "digna", "Essa", "essa", "Esta", "esta", "futura",
"leg"+u"\u00ED"+"tima", "m"+u"\u00E0", "mesma", "minha", "muita", "Na", "na",
"nenhuma", "nona", "nossa", "nova", "oitava", "outra", "pela", "pouca",
"Prezada", "prezada", "primeira", "pr"+u"\u00F3"+"pria", "pr"+u"\u00F3"+"xima",
"quarta", "quinta", "segunda", "s"+u"\u00E9"+"tima", "sexta", "sua",
"terceira", "Uma", "uma", "velha", "vossa"])

# conPf = Concordancia Plural Feminino - substantivos - adjetivos
conPf = set(["algumas", "Amigas",  "amigas", "antigas", "aquelas", "boas", "Caras",
"caras", "Das", "das", "dignas", "Essas", "essas", "Estas", "estas", "futuras",
"leg"+u"\u00ED"+"timas", "m"+u"\u00E0"+"s", "mesmas", "minhas", "muitas", "Nas",
"nas", "nenhumas", "nonas", "nossas", "novas", "oitavas", "outras", "pelas",
"poucas", "Prezadas", "prezadas", "primeiras", "pr"+u"\u00F3"+"prias",
"pr"+u"\u00F3"+"ximas", "quartas", "quintas", "segundas",
"s"+u"\u00E9"+"timas", "sextas", "suas",
"terceiras", "Umas", "umas", "velhas", "vossas"])


# aacCmf = Concordancia  Masculino - feminino substantivos - adjetivos
aacCmf = {"algum": "alguma",
"Amigo": "Amiga",
"amigo": "amiga",
"antigo": "antiga",
"aquele": "aquela",
"bom": "boa",
"Caro": "Cara",
"caro": "cara",
"Do": "Da",
"do": "da",
"digno": "digna",
"Distinto": "Distinta",
"distinto": "distinta",
"Esse": "Essa",
"esse": "essa",
"Este": "Esta",
"este": "esta",
"futuro": "futura",
"leg"+u"\u00ED"+"timo": "leg"+u"\u00ED"+"tima",
"mau": "m"+u"\u00E0",
"meu": "minha",
"nenhum": "nenhuma",
"No": "Na",
"no": "na",
"nono": "nona",
"nosso": "nossa",
"novo": "nova",
"oitavo": "oitava",
"outro": "outra",
"pelo": "pela",
"Prezado": "Prezada",
"prezado": "prezada",
"pr"+u"\u00F3"+"prio": "pr"+u"\u00F3"+"pria",
"pr"+u"\u00F3"+"ximo": "pr"+u"\u00F3"+"xima",
"quarto": "quarta",
"quinto": "quinta",
"Senhor": "Senhora",
"senhor": "senhora",
"s"+u"\u00E9"+"timo": "s"+u"\u00E9"+"tima",
"seu": "sua",
"sexto": "sexta",
"terceiro": "terceira",
"Um": "Uma",
"um": "uma",
"velho": "velha",
"vosso": "vossa"}

# aacPmos = Concordancia  Plural Masculino - substantivos - adjetivos
aacPmos = {"algum": "alguns",
"Amigo": "Amigos",
"amigo": "amigos",
"antigo": "antigos",
"aquele": "aqueles",
"bom": "bons",
"Caro": "Caros",
"caro": "caros",
"Do": "Dos",
"do": "dos",
"digno": "dignos",
"Distinto": "Distintos",
"distinto": "distintos",
"Esse": "Esses",
"esse": "esses",
"Este": "Estes",
"este": "estes",
"futuro": "futuros",
"leg"+u"\u00ED"+"timo": "leg"+u"\u00ED"+"timos",
"mau": "maus",
"meu": "meus",
"nenhum": "nenhuns",
"No": "Nos",
"no": "nos",
"nono": "nonos",
"nosso": "nossos",
"novo": "novos",
"oitavo": "oitavos",
"outro": "outros",
"pelo": "pelos",
"Prezado": "Prezados",
"prezado": "prezados",
"pr"+u"\u00F3"+"prio": "pr"+u"\u00F3"+"prios",
"pr"+u"\u00F3"+"ximo": "pr"+u"\u00F3"+"ximos",
"quarto": "quartos",
"quinto": "quintos",
"Senhor": "Senhores",
"senhor": "senhores",
"s"+u"\u00E9"+"timo": "s"+u"\u00E9"+"timos",
"seu": "seus",
"sexto": "sextos",
"terceiro": "terceiros",
"Um": "Uns",
"um": "uns",
"velho": "velhos",
"vosso": "vossos"}

# aacPmso = Concordancia  Plural Masculino - substantivos - adjetivos
aacPmso = {"algum": "alguns",
"Amigos": "Amigo",
"amigos": "amigo",
"antigos": "antigo",
"aqueles": "aquele",
"bons": "bom",
"Caros": "Caro",
"caros": "caro",
"Dos": "Do",
"dos": "do",
"dignos": "digno",
"Distintos": "Distinto",
"distintos": "distinto",
"Esses": "Esse",
"esses": "esse",
"Estes": "Este",
"estes": "este",
"futuros": "futuro",
"leg"+u"\u00ED"+"timos": "leg"+u"\u00ED"+"timo",
"maus": "mau",
"meus": "meu",
"nenhuns": "nenhum",
"Nos": "No",
"nos": "no",
"nonos": "nono",
"nossos": "nosso",
"novos": "novo",
"oitavos": "oitavo",
"outros": "outro",
"pelos": "pelo",
"Prezados": "Prezado",
"prezados": "prezado",
"pr"+u"\u00F3"+"prios": "pr"+u"\u00F3"+"prio",
"pr"+u"\u00F3"+"ximos": "pr"+u"\u00F3"+"ximo",
"quartos": "quarto",
"quintos": "quinto",
"Senhores": "Senhor",
"senhores": "senhor",
"s"+u"\u00E9"+"timos": "s"+u"\u00E9"+"timo",
"seus": "seu",
"sextos": "sexto",
"terceiros": "terceiro",
"Uns": "Um",
"uns": "um",
"velhos": "velho",
"vossos": "vosso"}


# conCmf = Concordancia Masculino - Feminino - substantivos - adjetivos
conCmf = set(["algum", "Amigo", "amigo", "antigo", "aquele", "bom", "Caro",
"caro", "Do", "do", "digno", "Distinto", "distinto", "Esse", "esse", "Este", "este",
"futuro", "leg"+u"\u00ED"+"timo", "mau", "meu", "nenhum", "No", "no", "nono",
"nosso", "novo", "oitavo", "outro", "pelo", "Prezado", "prezado",
"pr"+u"\u00F3"+"prio", "pr"+u"\u00F3"+"ximo", "quarto", "quinto", "Senhor",
"senhor", "s"+u"\u00E9"+"timo", "seu", "sexto", "terceiro", "Um", "um",
"velho", "vosso"])

# conPm = Concordancia Plural Masculino - substantivos - adjetivos
conPm = set(["alguns", "Amigos", "amigos", "antigos", "aqueles", "bons",
"Caros", "caros", "Dos", "dos", "dignos", "Distintos", "distintos", "Esses",
"esses", "Estes", "estes", "futuros", "leg"+u"\u00ED"+"timos", "maus",
"meus", "nenhuns", "Nos", "nos", "nonos", "nossos", "novos", "oitavos",
"outros", "pelos", "Prezados", "prezados", "pr"+u"\u00F3"+"prios",
"pr"+u"\u00F3"+"ximos", "quartos", "quintos", "Senhores", "senhor",
"s"+u"\u00E9"+"timo", "seu", "sexto", "terceiro", "Uns", "uns",
"velhos", "vossos"])


# excMentes = Execao para palavras com - mente - que admitem plural -s-
excMentes = set(["acampamentense", "alimenteira", "alimenteiro",
"casamenteira", "casamenteiro", "cimenteira", "cimenteiro", "clemente",
"demente", "deprimente", "dormente", "entremente", "fermentelense",
"fremente", "gemente", "inclemente", "juramenteira", "juramenteiro",
"livramentense", "mente", "mentecapta", "mentecapto", "moimentense",
"monumentense", "movimentense", "pimenteiralense", "pimenteira",
"pimenteirense", "pimenteiro", "pimentense",  "premente", "pseudodemente",
"semente", "sementeira", "sementeiro", "temente", "testamenteira",
"testamenteiro", "tremente", "veemente", "vestimenteira", "vestimenteiro"])


# excMento palavras terminadas em mento - excecao a regra
excMento = set(["a", "aci", "ador", "aju", "ajura", "ali", "ama", "apara",
"api", "ator", "autola", "bisseg", "ce", "co", "coag", "comparti",
"comple", "conjura", "cumpri", "de", "decre", "des", "desci",
"desma", "desor"+u"\u00E7"+"a", "despara", "despavi", "despig",
"desseg", "dor", "e", "empara", "ense", "escar", "escra",
"esto", "experi", "ferra", "fo", "frag", "funda", "hemossedi",
"imple", "incre", "la", "leni", "lini", "macrosseg", "metassedi",
"movi", "orna", "para", "pedi", "predica", "reali", "reimple",
"repavi", "retroali", "sacra", "salpi", "se", "seg", "suple",
"transcomple"])


# excIsta palavras terminadas em ista que nao admitem variacao de genero
excIsta = set(["Avista", "avista", "Boavista", "Consista", "consista",
"Desista", "desista", "Exista", "exista", "Entrevista", "entrevista",
"Insista", "insista", "Invista", "invista", "Lista", "lista", "Mista", "mista",
"Persista", "persista", "Pista", "pista", "Prevista", "prevista",
"Resista", "resista", "Revista", "revista", "Vista", "vista"])


# excAdo palavras terminadas em ado e ido  que nao admitem variacao de genero ou com variacao rara
excAdo = set(["almoxarif", "arcebisp", "atent", "cade", "c"+u"\u00E1"+"g",
"caj", "calif", "cond", "consul", "c"+u"\u00F4"+"vado", "cup", "duc",
"eleitor", "emir", "episcop", "euromerc", "f"+u"\u00ED"+"g", "g",
"hipermerc", "hortomerc", "juiz", "mar", "merc", "microd",
"miniatent", "minigram", "minimerc", "minitecl",
"multialmoxarif", "multimerc", "operari", "pr", "presbiter",
"princip", "proletari", "protetor",
"reitori", "s"+u"\u00E1"+"b", "sample", "sen", "servoacion",
"superfadig", "supermerc", "supersold", "tecl", "xax"])


# excplMasco - exc. preposicao A - palavras masculinas terminadas em -o- que nao possui correspondente feminino- Ex.: A principio
excplMasco = set(["crit"+u"\u00E9"+"rio", "exemplo", "pedido", "pretexto", "princ"+u"\u00ED"+"pio", "prop"+u"\u00F3"+"sito", "tempo", "t"+u"\u00ED"+"tulo"])

# plvrMasxo - ESPECIAL - palavras masculinas terminadas em -o- que nao possui correspondente feminino- ano e auto
plvrMasxo = set(["ano", "auto"])

# plvrMasco - palavras masculinas terminadas em -o- que nao possui correspondente feminino-
plvrMasco = set(["abuso", "acento", "ac"+u"\u00F3"+"rd"+u"\u00E3"+"o", "acr"+u"\u00E9"+"scimo",
"actinomiceto", "adult"+u"\u00E9"+"rio", "adv"+u"\u00E9"+"rbio", "af"+u"\u00ED"+"dio",
"alabardeiro", "alde"+u"\u00E3"+"o", "alelo", "alfarr"+u"\u00E1"+"bio",
"algoritmo", "al"+u"\u00ED"+"vio", "almoxarifado", "amentilho", "amuleto",
"anci"+u"\u00E3"+"o", "anda"+u"\u00E7"+"o", "andanho", "andrajo",
"anfiteatro", u"\u00E2"+"ngulo", "anho", "animalejo",
"anteprojeto", "anticorpo", "antig"+u"\u00EA"+"nio",
"antro", "an"+u"\u00FA"+"ncio", "anuro", "aparelho",
"apedido", "aplauso", "aposento", "aqueduto",
"arauto", "arb"+u"\u00ED"+"trio", "arbusto", "arcebispado",
"arcebispo", "arenito", "arilho", "arm"+u"\u00E1"+"rio",
"arqueg"+u"\u00F4"+"nio", "arqu"+u"\u00E9"+"tipo", "arquip"+u"\u00E9"+"lago",
"arquivo", "arranjo", "arreio", "artelho", "artes"+u"\u00E3"+"o",
"artif"+u"\u00ED"+"cio", "artigo", "arvoredo", "aspecto",
"assassinato", "assass"+u"\u00ED"+"nio", "assunto", "atentado", u"\u00E1"+"tomo",
"atributo", "ausp"+u"\u00ED"+"cio", "autocarro",
"autoconsumo", "autodom"+u"\u00ED"+"nio", "autogiro", "autogoverno",
"autorretrato", "aux"+u"\u00ED"+"lio", "ax"+u"\u00F4"+"nio", "babu"+u"\u00ED"+"no",
"bacano", "bairro", "balcedo", "b"+u"\u00E1"+"lsamo", "balsero",
"bamb"+u"\u00FA"+"rrio", "b"+u"\u00E1"+"ratro", "barbit"+u"\u00FA"+"rico", "bardino",
"baronato", "basalto", "bas"+u"\u00ED"+"dio", "basidiocarpo",
"basidiomiceto", "bat"+u"\u00F3"+"lito", "bebedouro", "benef"+u"\u00ED"+"cio",
"benzopireno", "ber"+u"\u00E7"+u"\u00E1"+"rio", "bicampeonato", "bicoco",
"bin"+u"\u00F3"+"culo", "bin"+u"\u00F4"+"mio", "biombo", "biorritmo",
"bi"+u"\u00F3"+"topo", "bi"+u"\u00F3"+"xido", "b"+u"\u00F3"+"cio", "bolbilho",
"bolbo", "bolero", "borato", "bororo",
"borrelho", "braco", "brinquedo", "brometo",
"br"+u"\u00F4"+"nquio", "bruxedo", "bu"+u"\u00E7"+"o", "bulbo",
"busto", "buxo", "b"+u"\u00FA"+"zio", "cabelo",
"cabrestilho", "cabresto", "cacareco", "cacto",
"cadeado", "caderno", "c"+u"\u00E1"+"gado", "caixilho", "cajado",
"calabou"+u"\u00E7"+"o", "calangro", "c"+u"\u00E1"+"lculo", "calend"+u"\u00E1"+"rio",
"calhama"+u"\u00E7"+"o", "califado", "c"+u"\u00E2"+"mbio", "campeonato", "campo",
"camundongo", "candelabro", "c"+u"\u00E2"+"ntico", "carboidrato",
"carb"+u"\u00FA"+"nculo", "carro", "cartap"+u"\u00E1"+"cio", "cartucho",
"cat"+u"\u00E1"+"logo", "cedro", "celacanto", "cemit"+u"\u00E9"+"rio",
"cen"+u"\u00E1"+"rio", "censo", "cent"+u"\u00ED"+"metro", "c"+u"\u00E9"+"rebro",
"cet"+u"\u00E1"+"ceo", "chaparro", "cesto", "cestinho",
"chimango", "chuvisqueiro", "ciclamato", "cidad"+u"\u00E3"+"o",
"c"+u"\u00ED"+"lio", "cincerro", "circo", "c"+u"\u00ED"+"rculo",
"c"+u"\u00ED"+"rio", "cirro", "clad"+u"\u00F3"+"dio", "clorofluorcarboneto",
"co"+u"\u00E1"+"gulo", "c"+u"\u00F3"+"digo", "coentro", "col"+u"\u00E9"+"gio",
"colmilho", "colonato", "colono", "col"+u"\u00F3"+"quio",
"colosso", "comando", "com"+u"\u00ED"+"cio", "c"+u"\u00F4"+"moro",
"comp"+u"\u00EA"+"ndio",
"concelho", "condado", "condom"+u"\u00ED"+"nio", "congresso",
"consenso", "cons"+u"\u00F3"+"cio", "cons"+u"\u00F3"+"rcio",
"consulado", "conte"+u"\u00FA"+"do", "contexto", "contrabando", 
"contrassenso", "contratempo",
"contributo", "conv"+u"\u00EA"+"nio", "convento", "coradouro",
"corimbo", "cormo", "corniso", "corpo", "corp"+u"\u00FA"+"sculo",
"cors"+u"\u00E1"+"rio", "cotovelo", "c"+u"\u00F4"+"vado", "cr"+u"\u00E9"+"dito",
"crep"+u"\u00FA"+"sculo", "crist"+u"\u00E3"+"o", "crit"+u"\u00E9"+"rio",
"crocodilo", "cron"+u"\u00F4"+"metro", "cruzeiro",
"cub"+u"\u00ED"+"culo", "c"+u"\u00FA"+"mulo", "cupido", "curr"+u"\u00ED"+"culo", "damasco",
"dec"+u"\u00EA"+"nviro", "decl"+u"\u00ED"+"nio", "defeito", "defluxo",
"defumadoiro", "del"+u"\u00ED"+"rio", "delito", "dem"+u"\u00E9"+"rito",
"demiurgo", "dem"+u"\u00F4"+"nio", "dent"+u"\u00ED"+"culo", "dep"+u"\u00F3"+"sito",
"des"+u"\u00E1"+"gio", "des"+u"\u00E2"+"nimo", "desequil"+u"\u00ED"+"brio", "desiderato",
"des"+u"\u00ED"+"gnio", "desmando", "despenhadeiro", "desperd"+u"\u00ED"+"cio", "desporto",
"desprop"+u"\u00F3"+"sito", "desv"+u"\u00E3"+"o", "detrito", "diacho",
"di"+u"\u00E1"+"cono", "di"+u"\u00E1"+"logo", "di"+u"\u00E2"+"metro", "dieno",
"diestro", "diferendo", "d"+u"\u00ED"+"gito", "dil"+u"\u00FA"+"vio",
"d"+u"\u00ED"+"namo", "dinheiro", "dinoss"+u"\u00E1"+"urio", "dinossauro",
"diodo", "dipept"+u"\u00ED"+"dio", "dipolo", "d"+u"\u00ED"+"scolo",
"disp"+u"\u00EA"+"ndio", "dispens"+u"\u00E1"+"rio", "diss"+u"\u00ED"+"dio", "distrito",
"dist"+u"\u00FA"+"rbio", "dit"+u"\u00E9"+"rio", "ditirambo", "dividendo",
"div"+u"\u00F3"+"rcio", "dolo", "domic"+u"\u00ED"+"lio", "dom"+u"\u00ED"+"nio",
"dorso", "ducado", "ducto", "duto",
"edif"+u"\u00ED"+"cio", "educando", "efeito", "efl"+u"\u00FA"+"vio",
"eleitorado","eletrodo", "eletr"+u"\u00F3"+"lito", "elmo", "eloendro",
"emirado", "emirato", "empr"+u"\u00E9"+"stimo", "enc"+u"\u00F4"+"mio",
"end"+u"\u00F3"+"sporo", "enteropneusto", "entretanto", "e"+u"\u00F3"+"lito",
"epicentro", "episcopado", "epis"+u"\u00F3"+"dio", "epit"+u"\u00E1"+"fio", "epital"+u"\u00E2"+"mio",
"ep"+u"\u00ED"+"teto", "ep"+u"\u00F3"+"xido", "equil"+u"\u00ED"+"brio", "equin"+u"\u00F3"+"cio",
"er"+u"\u00E1"+"rio", "eri"+u"\u00F3"+"foro", "eritr"+u"\u00F3"+"cito", "esc"+u"\u00E2"+"ndalo",
"escapul"+u"\u00E1"+"rio", "esc"+u"\u00E1"+"rnio", "esconderijo", "escopro",
"escr"+u"\u00FA"+"pulo", "espantalho", "espargo", "espectro",
"espectr"+u"\u00F4"+"metro", "espet"+u"\u00E1"+"culo", "espir"+u"\u00E1"+"culo", "espirilo",
"esp"+u"\u00F3"+"lio", "espor"+u"\u00E2"+"ngio", "espor"+u"\u00F3"+"filo", "ess"+u"\u00EA"+"nio",
"est"+u"\u00E1"+"bulo", "est"+u"\u00E1"+"gio", "estatuto", "estearato",
"estegossauro", "estendedoiro", "estere"+u"\u00F3"+"tipo", "est"+u"\u00ED"+"mulo",
"est"+u"\u00F4"+"mago", "est"+u"\u00F4"+"mato", "estrato", "estr"+u"\u00E9"+"pito",
"estr"+u"\u00F3"+"bilo", "estrog"+u"\u00EA"+"nio", "estrondo", "est"+u"\u00FA"+"dio", "eucalipto",
"eunuco", "euro", "euromercado", "evangelho", "evento",
"excesso","exemplo", "exerc"+u"\u00ED"+"cio", "ex"+u"\u00E9"+"rcito", "externato",
"fag"+u"\u00F3"+"cito", "falhan"+u"\u00E7"+"o", "falucho", "faniquito",
"fanisco", "fasc"+u"\u00ED"+"culo", "fast"+u"\u00ED"+"gio", "fastio",
"feiti"+u"\u00E7"+"o", "feldspato", "fen"+u"\u00F4"+"meno", "fen"+u"\u00F3"+"tipo",
"ferom"+u"\u00F4"+"nio", "feto", "fiapo", "fiasco",
"ficomiceto", "fideicomisso", "f"+u"\u00ED"+"gado", "f"+u"\u00ED"+"garo",
"figueiredo", "filantropo", "fil"+u"\u00F3"+"sofo", "filtro", "flag"+u"\u00ED"+"cio",
"flamingo", "flanco", "fl"+u"\u00F3"+"culo", "fl"+u"\u00F3"+"sculo",
"fluorcarboneto", "fluoreto", "fluxo", "fogo",
"folguedo", "folhelho", "fol"+u"\u00ED"+"culo", "f"+u"\u00F3"+"lio",
"fol"+u"\u00ED"+"olo", "formul"+u"\u00E1"+"rio", "f"+u"\u00F3"+"sforo", "fotod"+u"\u00ED"+"odo",
"fot"+u"\u00F3"+"foro", "fraguedo", "freixo", "frontisp"+u"\u00ED"+"cio",
"fulcro", "fulereno", "funcho", "fundo", "fur"+u"\u00FA"+"nculo",
"fuzo", "gabro", "gado", "gafanhoto",
"g"+u"\u00E1"+"lago", "galfarro", "gambozino", "gamet"+u"\u00F3"+"foro",
"gancho", "g"+u"\u00E2"+"nglio", "garbo", "garraio", "g"+u"\u00E1"+"rrulo",
"gastr"+u"\u00F3"+"lito", "gat"+u"\u00E1"+"zio", "gatilho", "geco",
"g"+u"\u00EA"+"nero", "g"+u"\u00EA"+"nio", "genoc"+u"\u00ED"+"dio", "gen"+u"\u00F3"+"tipo",
"genro", "gentio", "ger"+u"\u00E2"+"nio", "gerbo",
"german"+u"\u00F3"+"fono", "gibelino", "gin"+u"\u00E1"+"sio", "girino",
"girondino", "girosc"+u"\u00F3"+"pio", "gl"+u"\u00E1"+"dio", "glasto",
"gliceralde"+u"\u00ED"+"do", "gl"+u"\u00ED"+"cido", "glicos"+u"\u00ED"+"deo", "gl"+u"\u00F3"+"bulo",
"goliardo", "gonzo", "granda"+u"\u00E7"+"o", "gr"+u"\u00E3"+"o",
"gravan"+u"\u00E7"+"o", "gravitino", "gr"+u"\u00EA"+"mio", "gringo",
"grupo", "guardamo", "guardanapo", "guardaremo",
"guelfo", "haleto", "h"+u"\u00E1"+"lito", "halogeneto",
"hausto", "hecatonquiro", "hem"+u"\u00ED"+"ptero", "hemisf"+u"\u00E9"+"rio",
"hemist"+u"\u00ED"+"quio", "hepat"+u"\u00F3"+"cito", "heterocisto", "hex"+u"\u00E2"+"metro",
"hiato", "hibisco", "hid"+u"\u00E1"+"todo", "hidreto",
"hidrocarboneto", "hidrof"+u"\u00F3"+"lio", "hidroper"+u"\u00F3"+"xido", "hidr"+u"\u00F3"+"xido",
"hieroglifo", "hier"+u"\u00F3"+"glifo", "hipermercado", "hipn"+u"\u00F3"+"sporo", "hip"+u"\u00F3"+"dromo",
"hipogrifo", "hirsuto", "hol"+u"\u00F3"+"tipo", "homic"+u"\u00ED"+"dio",
"homopol"+u"\u00ED"+"mero", "hor"+u"\u00F3"+"scopo", "hosp"+u"\u00ED"+"cio",
"hortomercado", "hussardo",
"ictiossauro", "idioblasto", "imbr"+u"\u00F3"+"glio", "imp"+u"\u00E9"+"rio",
"improp"+u"\u00E9"+"rio", "imunocomplexo", "inc"+u"\u00EA"+"ndio", "incens"+u"\u00E1"+"rio",
"incha"+u"\u00E7"+"o", "ind"+u"\u00ED"+"geno", "inferno", "influxo", "inf"+u"\u00F3"+"lio",
"infort"+u"\u00FA"+"nio", "in"+u"\u00ED"+"cio", "inqu"+u"\u00E9"+"rito", "instinto", "insucesso",
"insurrecto", "intelecto", "interc"+u"\u00E2"+"mbio", "interl"+u"\u00FA"+"dio",
"internato", "intuito", "invent"+u"\u00E1"+"rio", "inv"+u"\u00F3"+"lucro",
"is"+u"\u00F3"+"topo", "jambo", "jan"+u"\u00ED"+"zaro", "jato",
"jazigo", "jeito", "joelho", "jornaleco",
"j"+u"\u00FA"+"bilo", "juizado", "lac"+u"\u00F3"+"lito", "lactic"+u"\u00ED"+"nio",
"lajedo", "lanif"+u"\u00ED"+"cio", "lari"+u"\u00E7"+"o", "l"+u"\u00E1"+"tego",
"latic"+u"\u00ED"+"nio", "latif"+u"\u00FA"+"ndio", "latroc"+u"\u00ED"+"nio", "lavabo",
"len"+u"\u00E7"+"o", "lepidodendro", "letargo", "leuc"+u"\u00F3"+"cito",
"leucotrieno", "libreto", "licop"+u"\u00F3"+"dio", "l"+u"\u00ED"+"dio",
"linf"+u"\u00F3"+"cito", "linfonodo", "lip"+u"\u00ED"+"dio", "l"+u"\u00ED"+"pido",
"lit"+u"\u00ED"+"gio", "litro", "livreco", "lobato",
"l"+u"\u00F3"+"bulo", "l"+u"\u00F3"+"culo", "lodo", "logaritmo",
"logogrifo", "logotipo", "logradouro", "lombelo", "longo",
"longu"+u"\u00ED"+"ssimo", "loranto", "lugarejo", "lusito",
"mabeco", "malacopter"+u"\u00ED"+"gio", "malef"+u"\u00ED"+"cio", "manaixo",
"mandato", "maremoto", "maribondo", "marimbondo", "marido", "mart"+u"\u00ED"+"rio",
"mastro", "mastru"+u"\u00E7"+"o", "matadouro", "matrim"+u"\u00F4"+"nio",
"medo", "medinho",
"meg"+u"\u00E1"+"lito", "megaquir"+u"\u00F3"+"ptero", "melan"+u"\u00F3"+"cito", "membro",
"memorando", "meningococo", "mentastro", "mercado", "merc"+u"\u00FA"+"rio",
"m"+u"\u00E9"+"rito", "metab"+u"\u00F3"+"lito", "meteorito", "meteoro",
"m"+u"\u00E9"+"todo", "micr"+u"\u00F3"+"bio", "microcircuito", "microdado", "microprotalo",
"microsc"+u"\u00F3"+"pio", "microt"+u"\u00FA"+"bulo", "mijolo", "m"+u"\u00ED"+"ldio",
"miniatentado",
"min"+u"\u00E9"+"rio", "miniconcurso", "minigramado", "minimercado", "minist"+u"\u00E9"+"rio",
"miniteclado",
"miradouro", "mist"+u"\u00E9"+"rio", "m"+u"\u00F3"+"dulo", "molosso",
"molusco", "mon"+u"\u00F3"+"lito", "monop"+u"\u00F3"+"lio", "monturo",
"mortic"+u"\u00ED"+"nio", "moscardo", "mostru"+u"\u00E1"+"rio", "motociclo",
"motreco", "mult"+u"\u00ED"+"metro", "mundo", "munic"+u"\u00ED"+"pio", "munirmo",
"murm"+u"\u00FA"+"rio", "musaranho", "m"+u"\u00FA"+"sculo", 
"multialmoxarifado", "multimercado", "nababo",
"nabuco", "nasuto", "naufr"+u"\u00E1"+"gio", "n"+u"\u00E1"+"utilo",
"navio", "necrol"+u"\u00F3"+"gio", "nefro", "neg"+u"\u00F3"+"cio",
"nematocisto", "nervo", "neur"+u"\u00F4"+"nio", "neutrino",
"nexo", "n"+u"\u00ED"+"scaro", "nitrito", "n"+u"\u00F3"+"dulo",
"notici"+u"\u00E1"+"rio", "n"+u"\u00FA"+"cleo", "nucle"+u"\u00F3"+"tido", "n"+u"\u00FA"+"mero",
"nuno", "obelisco", "obst"+u"\u00E1"+"culo", "ocaso",
"ocean"+u"\u00E1"+"rio", "oceano", "ocelo", "of"+u"\u00ED"+"cio",
"oleoduto", "oligop"+u"\u00F3"+"lio", "olivedo",
"olmedo", "olmo", "ombro", "op"+u"\u00E9"+"rculo", "operariado",
"opiato", "op"+u"\u00FA"+"sculo", "or"+u"\u00E1"+"culo", "orago",
"orango", "orangotango", "orfanato", "organito",
"orif"+u"\u00ED"+"cio", "ornato", "oss"+u"\u00ED"+"culo", "ot"+u"\u00F3"+"lito",
"ouvido","ov"+u"\u00E1"+"rio", "ov"+u"\u00F3"+"cito", "ovo", "oxi"+u"\u00FA"+"ro",
"pa"+u"\u00E7"+"o", "pag"+u"\u00E3"+"o", "pal"+u"\u00E1"+"cio", "palco", "p"+u"\u00E1"+"lio",
"palmito", "pandeiro", "paneg"+u"\u00ED"+"rico", "p"+u"\u00E2"+"ntano",
"par"+u"\u00E1"+"grafo", "para"+u"\u00ED"+"so", "paralelep"+u"\u00ED"+"pedo", "par"+u"\u00E2"+"metro",
"par"+u"\u00E1"+"tipo", "p"+u"\u00E1"+"roco", "passatempo", "pat"+u"\u00ED"+"bulo",
"p"+u"\u00E1"+"tio", "patrim"+u"\u00F4"+"nio", "pecadilho", "pec"+u"\u00ED"+"olo",
"peda"+u"\u00E7"+"o", "pedregulho", "pedroi"+u"\u00E7"+"o", "pedrou"+u"\u00E7"+"o",
"ped"+u"\u00FA"+"nculo", "peitilho", "penduricalho", "penhasco",
"pequerrucho", "percurso", "per"+u"\u00ED"+"metro", "per"+u"\u00ED"+"odo",
"pesadelo", "pesco"+u"\u00E7"+"o", "petr"+u"\u00F3"+"leo", "p"+u"\u00ED"+"ncaro",
"pinheiro", "planalto", "plantio", "pol"+u"\u00ED"+"gono", "pol"+u"\u00ED"+"mero",
"polin"+u"\u00F4"+"mio", "p"+u"\u00F3"+"lipo", "p"+u"\u00F4"+"mulo", "ponto", "poro",
"portento", "postigo", "povo", "prado",
"prazo", "pre"+u"\u00E2"+"mbulo", "presbiterado", "precip"+u"\u00ED"+"cio", "pre"+u"\u00E7"+"o",
"preconceito", "pr"+u"\u00E9"+"dio", "pref"+u"\u00E1"+"cio", "preju"+u"\u00ED"+"zo",
"prelo", "prel"+u"\u00FA"+"dio", "pr"+u"\u00EA"+"mio", "pren"+u"\u00FA"+"ncio",
"pres"+u"\u00E9"+"pio", "pres"+u"\u00ED"+"dio", "press"+u"\u00E1"+"gio", "prest"+u"\u00ED"+"gio",
"pr"+u"\u00E9"+"stimo", "presunto", "prim"+u"\u00F3"+"rdio", "principado",
"princ"+u"\u00ED"+"pio", "privil"+u"\u00E9"+"gio", "processo", "prod"+u"\u00ED"+"gio", "pr"+u"\u00F3"+"dromo",
"produto", "proleg"+u"\u00F4"+"meno", "proletariado", "prop"+u"\u00F3"+"sito", "pros"+u"\u00E9"+"lito",
"prost"+u"\u00ED"+"bulo", "protalo", "protetorado", "prot"+u"\u00F3"+"tipo",
"proveito", "provento", "prov"+u"\u00E9"+"rbio", "pseud"+u"\u00F3"+"podo",
"p"+u"\u00FA"+"lpito", "quartzo", "quelato", "quesito",
"question"+u"\u00E1"+"rio", "quiabo", "qu"+u"\u00ED"+"cio", "quilo",
"quilombo", "quil"+u"\u00F4"+"metro", "quinteto", "quisto",
"rabino", "rabo", "racimo", "r"+u"\u00E1"+"cio",
"racioc"+u"\u00ED"+"nio", "radiano", "radiois"+u"\u00F3"+"topo",
"radiotelesc"+u"\u00F3"+"pio", "rapazito", "rebotalho", "recept"+u"\u00E1"+"culo",
"recesso", "redenho", "r"+u"\u00E9"+"dito", "reduto",
"refrig"+u"\u00E9"+"rio", "ref"+u"\u00FA"+"gio", "r"+u"\u00E9"+"gulo", "reixelo",
"reitoriado",
"rel"+u"\u00E2"+"mpago", "relic"+u"\u00E1"+"rio", "rel"+u"\u00F3"+"gio", "re"+u"\u00F3"+"stato",
"resfolgadouro", "res"+u"\u00ED"+"duo", "respiradouro", "resqu"+u"\u00ED"+"cio",
"ret"+u"\u00E1"+"bulo", "rev"+u"\u00E9"+"rbero", "riacho", "ricto",
"robalo", "rochedo", "rododendro", "rofo",
"romancero", "rombo", "ros"+u"\u00E1"+"rio", "rostro",
"rubicundo", "rublo", "s"+u"\u00E1"+"bado", "sacerd"+u"\u00F3"+"cio",
"sacr"+u"\u00E1"+"rio", "sacrif"+u"\u00ED"+"cio", "safio", "salmo",
"saltimbanco", "santiago", "saquito", "sarcodino",
"sarrabulho", "seculo", "s"+u"\u00E9"+"culo",
"seio", "sem"+u"\u00E1"+"foro", "seman"+u"\u00E1"+"rio", "semin"+u"\u00E1"+"rio",
"senado", "senso", "sequilho", "seresteiro",
"servi"+u"\u00E7"+"o", "shampoo", "sic"+u"\u00E1"+"rio", "sic"+u"\u00F4"+"moro",
"sil"+u"\u00EA"+"ncio", "silfo", "siltito", "silvedo",
"s"+u"\u00ED"+"mbolo", "simulacro", "Sindicato", "sindicato", "sin"+u"\u00E9"+"drio", "s"+u"\u00ED"+"ntipo",
"sipaio", "s"+u"\u00ED"+"tio", "sobretudo", "sobrolho",
"solil"+u"\u00F3"+"quio", "sorriso", "sortil"+u"\u00E9"+"gio", "sovaco",
"sublan"+u"\u00E7"+"o", "subm"+u"\u00F3"+"dulo", "subproduto", "subs"+u"\u00ED"+"dio",
"substrato", "subterf"+u"\u00FA"+"gio", "sub"+u"\u00FA"+"rbio", "sucesso",
"sud"+u"\u00E1"+"rio", "sufr"+u"\u00E1"+"gio", "suic"+u"\u00ED"+"dio", "sultanato",
"sumi"+u"\u00E7"+"o", "sumidouro", "superc"+u"\u00ED"+"lio", "supermercado",
"supersoldado", "supertalento", "supl"+u"\u00ED"+"cio", "suspiradouro",
"sustent"+u"\u00E1"+"culo", "tabuleiro", "tajasno", "talento", "tamarindo",
"tamoio", "taqu"+u"\u00ED"+"grafo", "tarro", "tartufo",
"taumaturgo", "teatro", "teclado", "t"+u"\u00E9"+"dio", "teixo",
"telecentro", "teledisco", "tel"+u"\u00E9"+"grafo", "tele"+u"\u00F3"+"steo",
"telesc"+u"\u00F3"+"pio", "teli"+u"\u00F3"+"sporo", "tel"+u"\u00F4"+"mero", "tempo",
"tent"+u"\u00E1"+"culo", "teodolito", "term"+u"\u00F4"+"metro", "term"+u"\u00F3"+"stato",
"terra"+u"\u00E7"+"o", "terremoto", "test"+u"\u00ED"+"culo", "texto",
"texugo", "tilito", "t"+u"\u00ED"+"mpano", "tirso",
"t"+u"\u00ED"+"tulo", "tojo", "tom"+u"\u00F3"+"grafo", "top"+u"\u00E1"+"zio",
"t"+u"\u00F3"+"rculo", "tornozelo", "torresmo", "tortulho",
"touti"+u"\u00E7"+"o", "trajeto", "trambelho", "trecho",
"tri"+u"\u00E2"+"ngulo", "triciclo", "tr"+u"\u00ED"+"duo", "triglicer"+u"\u00ED"+"dio",
"triglic"+u"\u00E9"+"rido", "tri"+u"\u00FA"+"nviro", "tub"+u"\u00E9"+"rculo", "t"+u"\u00FA"+"bulo",
"tug"+u"\u00FA"+"rio", "t"+u"\u00FA"+"mulo", "tumulto", "tur"+u"\u00ED"+"bulo",
"ultimato", "universo", "utens"+u"\u00ED"+"lio", "v"+u"\u00E1"+"cuo",
"vac"+u"\u00FA"+"olo", "vatic"+u"\u00ED"+"nio", "vedonho", "ve"+u"\u00ED"+"culo",
"veloc"+u"\u00ED"+"metro", "veludilho", "vencilho", "ventr"+u"\u00ED"+"culo",
"ventr"+u"\u00ED"+"loquo", "verbo", "verdugo", "veredicto", "vesti"+u"\u00E1"+"rio",
"vest"+u"\u00ED"+"gio", "vestu"+u"\u00E1"+"rio", "ves"+u"\u00FA"+"vio", "viaduto",
"v"+u"\u00ED"+"deo", "videojogo", "vidrilho", "vilarejo",
"v"+u"\u00ED"+"nculo", "vinhedo", "violino", "vitup"+u"\u00E9"+"rio",
"viveiro", "voc"+u"\u00E1"+"bulo", "v"+u"\u00F4"+"mito",
"xaxado", "xerimbabo",
"xifosuro", "xisto", "zagalejo", "z"+u"\u00E9"+"firo",
"zoilo", "zuavo"])

# plvrMasca - palavras masculinas terminadas em -a- que nao possui correspondente feminino-
plvrMasca = set(["aneurisma", "angioma", "aroma", "ateroma", "axioma",
"bioma", "bumba", "carcinoma", "carisma", "celoma",
"cinema", "clima", "c"+u"\u00F3"+"lera", "cometa",
"condroma", "cromossoma", "cronograma", "delta", "dia",
"diadema", "diafragma", "dilema", "diploma",
"DNA", "dogma", "drama", "ecossistema",
"ectoplasma", "eczema", "edema", "endosperma",
"enfisema", "enigma", "epigrama", "escriba",
"esperma", "esquema", "estigma", "estoma", "estratagema",
"estroma", "exoplaneta", "fac"+u"\u00ED"+"nora", "feocromocitoma", "fliperama", "floema",
"fluxograma", "fusca", "gameta", "g"+u"\u00E2"+"meta",
"genoma", "glaucoma", "harmonograma", "hematoma", "hemograma",
"histograma", "histoplasma", "holograma", "ideograma",
"idioma",  "inseticida", "jota",
"lema", "lexema", "linfogranuloma", "linfoma",
"macrocromossoma", "magma", "mantra", "mapa",
"mega", "melanoma", "melodrama", "mesotelioma", "microclima",
"microgameta", "mieloma", "mimodrama", "moma",
"monograma", "neuroblastoma", "neuroplasma", "organograma",
"pampa", "panorama", "papa", "paradigma",
"parricida", "patriarca", "periodograma", "perisperma",
"pesticida", "pijama", "planeta", "plasma",
"poema", "poeminha", "prisma", "problema",
"programa", "protoplasma", "psicodrama", "psicoplasma",
"puma", "quediva", "quilograma", "rapazola", "ritidoma", "rizoma", "samba",
"sarcoma", "sintoma", "sismograma", "sistema",
"sofisma", "subprograma", "subsistema", "superatleta",
"sutra", "tapa", "teledrama", "telefonema",
"telegrama", "teorema", "tracoma", "trauma", "trema",
"videograma", "zabumba"])

# plvrFemo - palavras femininas terminadas em -o- 
plvrFemo = set(["fonofoto", "Fonofoto", "foto", "Foto", "libido", "Libido", "lipo", "Lipo", "moto", "Moto", "tribo", "Tribo"])

# excplFema - palavras femininas/masculinas que podem ser usadas com os artigos a / o 
excplFema = set(["camisa"])

# plvrFema - palavras femininas terminadas em -a- que nao possuem correspondente masculino- a-o
plvrFema = set(["abertura", "ab"+u"\u00F3"+"bada", "ab"+u"\u00F3"+"bora", "abreviatura",
"absolutinha", "ac"+u"\u00E1"+"cia", "academia",
"academiazinha", "a"+u"\u00E7"+"ailandesa", "acetilcolina", "achinesa",
"actina", "a"+u"\u00E7"+"ucena", "acupunctura", "acur"+u"\u00E1"+"cia",
"adaga", "adega", "adrenalina", "advocacia",
"aeromo"+u"\u00E7"+"a", "africanesa", "afronta", "agenda",
"agendinha", "agonia", "agorafobia", "agricultura",
"agroind"+u"\u00FA"+"stria", u"\u00E1"+"gua", "agulha", "agulhinha",
"alambresa", "alameda", "alavanca", "albanesa",
"al"+u"\u00E7"+"ada", "alcova", "alcunha", "aldeia",
"aldeiazinha", "alegada", "alegoria", "alegria",
"alegriazinha", "aleluia", "alf"+u"\u00E2"+"ndega", "alfazema",
"alforreca", "alforria", "alga", "algazarra",
u"\u00E1"+"lgebra", "algibeira", "alguma", "alhambresa",
"alheira", "al"+u"\u00ED"+"quota", "alma", "alm"+u"\u00E1"+"dena",
"almalaguesa", "almanjarra", "alminha", "almofada",
"almofadinha", "alopatia", "alpaca", "alpendrada",
"alpercata", "alpercatinha", "alquimia", "als"+u"\u00E1"+"cia",
"altiva", "altura", "alva", "alvenaria",
"alverca", "alvinha", "alvorada", "alvoradinha",
"alvura", "amarantesa", "amargura", "amazona", "ambrosia",
"ambrosina", "ameba", "am"+u"\u00EA"+"ijoa", "ameixa", "ameixinha",
"am"+u"\u00EA"+"ndoa", "amendoeira", "amesterdanesa", "ametista",
"amilopectina", "amnistia", "amora",
"amoreira", "amostra", "amostrinha", "ampola",
"ampulheta", "amsterdamesa", "anacleta", "anaconda",
"analandesa", "analgesia", "anamesa", "anarquia",
"anatomia", "anaurilandesa", "anca", "andadura",
"andamanesa", "anedota", "anemia", "anestesia",
"angina", "angolesa", "ang"+u"\u00FA"+"stia", "anilina",
"anistia", "anita", "anomalia", "anorexia",
"anquinha", "anta", "antec"+u"\u00E2"+"mara", "antena",
"antera", "antessala", "antipart"+u"\u00ED"+"cula", "antipatia",
"antiportuguesa", "antropofagia", "aorta", "apatia",
"apoplexia", "aposentadoria", "aposta", "apostinha",
"aquarela", "ar"+u"\u00E1"+"bia", "aragoesa", "aragonesa",
"aramesa", "aranha", "aranhinha", "arara",
"ararinha", "arca", "arcada", "arc"+u"\u00E1"+"dia",
"arcadinha", "ardenesa", u"\u00E1"+"rea", "areia",
"areiinha", "arena", "arenga", "aresta",
"argamassa", "argila", "argola", "argolinha",
"arg"+u"\u00FA"+"cia", "arguesa", "aristocracia", "arma",
"armadilha", "armadura", "arminha", "arnesa",
"aroeira", "aroeirinha", "arouquesa", "arquibancada",
"arquitetura", "arr"+u"\u00E1"+"bida", "arraia", "arraiada",
"arrelia", "arroba", "arruda", "arr"+u"\u00FA"+"dia",
"arrumadeira", "artilharia", "asa", "ascuma",
"asfixia", "asinha", "asma", "asneira",
"aspereza", "aspirina", "assamesa", "assembleia",
"assessoria", "assimetria", "assinatura", "assinaturazinha",
"assuanesa", "astronomia", "ast"+u"\u00FA"+"cia", "ata",
"atmosfera", "atonia", "atrofia", "aud"+u"\u00E1"+"cia",
"auditoria", "aula", "aulinha", "aura",
"aur"+u"\u00E9"+"ola", "aur"+u"\u00ED"+"cula", "aurilandesa", "aurora",
"austera", "australesa", "autarquia", "autobiografiazinha",
"autoestima", "autoestrada", "autofagia", "autonomia",
"autoria", "avaitanhesa", "avareza", "avaria",
"aveia", "avenida", "avenidinha", "aventura",
"aventurazinha", "avicultura", "avilesa", "avioneta",
"aza", "az"+u"\u00E1"+"fama", "azambuja", "azambujinha",
"azeitona", "azeitoninha", "azenha", "azerbaidjanesa",
"azia", "baamesa", "baba", "baba"+u"\u00E7"+"ulandesa",
"babosa", "baboseira", "babosinha", "ba"+u"\u00E7"+"a",
"bacalhoa", "bacharela", "bacia", "baciazinha",
"badalada", "badaladinha", "baderna", "baeta",
"baforada", "baforadinha", "baga", "bagaceira",
"bagatela", "bagresa", "baia", "ba"+u"\u00ED"+"a",
"baianinha", "baiona", "baionesa", "baioneta",
"baixela", "baixeza", "baixinha", "bala", "balada",
"baladinha", "balaiada", "balaustrada",
"balaustradinha", "balb"+u"\u00FA"+"rdia", "baleia", "balela",
"balinesa", "balinha", "baliza", "balsa",
"balsinha", "baltistanesa", "bambina", "banana",
"bananeirinha", "bananinha", "bancarrota", "bandanesa",
"bandeira", "bandeirada", "bandeiradinha",
"bandeirola", "bandeja", "bandejinha", "bandurra",
"bandurrinha", "bangladesa", "banqueta",
"banquetinha", "baqueta", "barafunda", "barateza",
"barba", "barbaria", "barbatana", "barbearia",
"barbela", "barbicha", "barbinha", "barca"+u"\u00E7"+"a",
"barcarola", "barcelonesa", "bardana", "baremesa",
"barganha", "baronesa", "baronesinha",
"barreira", "barrela",
"barretina", "barrica", "barricada", "barricadinha",
"barriga", "barrigueira", "barrinha", "barroca",
"barulhada", "barulheira", "basca", "bas"+u"\u00ED"+"lica",
"bastilha", "batalhinha", "batata", "batatinha",
"b"+u"\u00E1"+"tega", "bateia", "bateria", "batina",
"batota", "batotinha", "batuta", "baunilha",
"bauxita", "baz"+u"\u00F3"+"fia", "bazuca", "bearnesa",
"beataria", "bebedeira", "bei"+u"\u00E7"+"ola", "beijoca",
"beira", "beirada", "beiradinha", "beirinha",
"beladona", "belbutina", "belda", "beleza",
"belezinha", "belida", "bemposta",
"benem"+u"\u00E9"+"rita", "bengala", "bengalesa", "bengalinha",
"benguela", "benguelinha",
"benzedura", "beripoconesa", "berlinda", "berlinesa",
"berma", "bermudesa", "bernarda", "bernardina",
"bernardinha", "bernesa", "besteira", "besteirinha",
"bestiaga", "beterraba", "b"+u"\u00E9"+"tula",
"bexiga", "bia", "b"+u"\u00ED"+"blia", "biblioteca",
"bicharada", "bicicleta", "bifada",
"bifadinha", "bigamia", "bigorna", "bilha",
"bilharda", "bilheteria", "bilirrubina", "binda",
"bintanesa", "biof"+u"\u00E1"+"brica", "biomassa", "biosfera",
"biqueira", "birmanesa", "birosca", "birra",
"birrinha", "bisarma", "bisca",
"biscaia", "bitola", "blasf"+u"\u00EA"+"mia",
"blusa", "blusinha", "boa", "boataria",
"bobina", "boca",
"bocada", "bocadinha", "bocarra", "boceta",
"bochecha", "bochechinha", "boda", "bodega",
"boemia", "bofetada", "bofetadinha", "boga",
"boiada", "boiadinha", "boicininga",
"boina", "boinha", "boininha", "bojuda",
"bojudinha", "bolacha", "bolachinha",
"bolada", "boladinha", "boleia", "boleta",
"bolha", "bolhinha", "bolonhesa",
"bolota", "bolsinha", "bomba",
"bombarda", "bombinha", "bonecra", "bonina",
"boniteza", "bonomia", "boquilha", "boquinha",
"borboleta", "borboletinha", "borda", "bordadura",
"bordalesa", "bordelesa", "bordinha", "bordoada",
"borga", "borgonha", "borgonhesa", "borla",
"borneira", "boroa", "borra", "borracha",
"borracheira", "borrachinha", "borralheira", "borrasca",
"borrega", "bossa", "bosta", "bota",
"botelha", "botica", "botija", "botina",
"botinha", "botoeira", "botulina", "brabantesa",
"bra"+u"\u00E7"+"ada", "bragancesa", "braguesa", "braguilha",
"brama", "brancura",
"brandeburguesa", "brandemburguesa", "brandura",
"braquesa", "brasa", "brasilandesa", "brasilesa",
"brasinha", "bravata", "braveza",
"bravura", "breca", "brecha", "breda",
"brejeira", "brenha", "bretanha", "brida",
"briga", "brigada", "briguinha", "brilhantina",
"brincadeira", "brincadeirinha", "brisa", "brita",
"broa", "broca", "brocha", "brochura",
"broinha", "bromocriptina", "bronca", "broncopneumonia",
"broxa", "bruaca", "bruaquinha", "bruma",
"brusca", "bruteza", "bruxaria", "bruxariazinha",
"bruxelesa", "bruzundanga", "bucha", "buchada",
"buchinha", "buda", "buenolandesa",
"bula", "bulgaresa", "bulha", "bunda",
"bundinha", "buraca", "burbonesa", "burga",
"burgalesa", "burguesa", "burguesia",
"burla", "burocracia", "burocratesa",
"burundinesa", "b"+u"\u00FA"+"ssola",
"butanesa", "buzina", "caatinga", "caba"+u"\u00E7"+"a",
"cabacinha", "cabana", "cabe"+u"\u00E7"+"ada",
"cabeceira", "cabecinha", "cabeleira",
"cabrinha", "ca"+u"\u00E7"+"a", "ca"+u"\u00E7"+"ada",
"ca"+u"\u00E7"+"adinha", "cacha"+u"\u00E7"+"a", "cachacinha", "cachoeira",
"cachoeirinha", "cachopa", "cacimba", "cacimbinha",
"cadeia", "cadeiazinha", "cadeira", "cadeirinha",
"cadela", "cadelinha", "caderneta", "cadernetinha",
"cafelandesa", "caimanesa", "calabresa", "cal"+u"\u00E7"+"a",
"calcesa", "calcinha",
"caldeira", "caldeirinha", "calha", "cali"+u"\u00E7"+"a",
"cama", "camada", "camadinha", "c"+u"\u00E2"+"mara",
"camaronesa", "cambada", "cambadinha", "cambraia",
"c"+u"\u00E2"+"mera", "camila", "camioneta", "camisa",
"camiseta", "camisetinha", "camisinha", "camisola",
"camisolinha", "camoesa", "campanha",
"campidanesa", "campina", "camponesa",
"canaresa", "canastra", "canavesa", "cancela",
"cancelinha", "candidatura", "candonga", "candura",
"canela", "canelinha", "caneta", "canetinha",
"canga", "cangalha", "cangalhinha", "canoa",
"canoinha", "canseira", "cantata", "cantatinha",
"cantiga", "cantilena", "cantonesa", "cantoria",
"canzoada", "capa", "capela", "capelinha",
"capinha", "capitania", "capivara", "capoeira",
"capoeirinha", "capota", "c"+u"\u00E1"+"psula", "captura",
"carabina", "caranguejola", "carapa"+u"\u00E7"+"a", "carapu"+u"\u00E7"+"a",
"carapucinha", "caravana", "caravela", "carca"+u"\u00E7"+"a",
"carcacinha", "carestia", "carga", "caricatura",
"car"+u"\u00ED"+"cia", "carlota", "carlotinha", "carni"+u"\u00E7"+"a",
"carnificina", "carranca", "carranquinha", "carreira",
"carreirinha", "carreta", "carri"+u"\u00E7"+"a", "carricinha",
"carro"+u"\u00E7"+"a", "carroceria", "carrocinha", "carta",
"cartada", "cartadinha", "cartaginesa", "cartilha",
"cartinha", "cartola", "cartucheira", "carvoeira",
"cascata", "cascatinha", "casinha", "caspa",
"cassilandesa", "casta", "castanha", "castanhinha",
"castinha", "catalanesa", "catanguesa", "catarina",
"categoria", "catinga", "catita", "cauda",
"caudinha", "cautela", "cavalaria", "cavalgada",
"cavalgadinha", "cavalgadura", "cavalgata", "cavalhada",
"caveira", "caveirinha", "caverna", "caverninha",
"cazaquistanesa", "cebola", "cebolinha", "c"+u"\u00E9"+"dula",
"cegueira", "ceia", "ceifa", "cela",
"celeuma", "c"+u"\u00E9"+"lula", "c"+u"\u00E9"+"lulazinha", "cena",
"censura", "centelha", "centelhinha", "cepa",
"cera", "cereja", "cerejinha", "certeza",
"certezinha", "cerveja", "cervejaria", "cervejinha",
"cesta", "cevada", "ch"+u"\u00E1"+"cara", "chacina",
"chacota", "chaga", "chala"+u"\u00E7"+"a", "chaleira",
"chamadinha", "champanhesa", "chancela",
"chancelaria", "chapa", "chapada", "chapinha",
"charada", "charadinha", "charolesa", "charrua",
"charuteira", "ch"+u"\u00E1"+"vena", "chefia", "chegada",
"chegadinha", "chibata", "chibatinha", "chinesa",
"cho"+u"\u00E7"+"a", "choradeira", "choupana", "chula",
"chusma", "chuva", "chuvinha", "cidadania",
"cidadela", "cifra", "cigarreira", "cilada",
"cilha", "cimeira", "cinemateca", "cingalesa",
"cintura", "cinturinha", "cisma",
"cisterna", "clareira", "clareza", "cl"+u"\u00E1"+"usula",
"clausura", "clava", "clavina", "clavinha",
"clientela", "clorofila", "coalhada", "coalhadinha",
"cobardia", "cobertura", "cobi"+u"\u00E7"+"a", "cobrinha",
"coca"+u"\u00ED"+"na", "cocheira", "cochinchinesa", "cochinesa",
"coimbresa", "coisa", "coisinha", "colcha",
"coleta", "colet"+u"\u00E2"+"nea", "coletinha", "colheita",
"c"+u"\u00F3"+"lica", "colina", "colinha", "colmeia",
"coluna", "comarca", "com"+u"\u00E9"+"dia", "comenda",
"comitiva", "companhia",
"compostura", "compra", "compreendia", "comprinha",
"comuna", "concha", "conchinha", "conc"+u"\u00F3"+"rdia",
"condessa", "condessinha", "conduta", "confraria",
"congolesa", "conjuntura", "consulta", "consultoria",
"contenda", "contrapartida", "contraproposta", "conversa", "conversinha",
"cooperativa", "coordenadoria", "c"+u"\u00F3"+"pia",
"c"+u"\u00F3"+"piazinha", "c"+u"\u00F3"+"pula", "cor"+u"\u00E7"+"a", "corda",
"cordilheira", "cordinha", "cordisburguesa", "cordislandesa",
"cordofanesa", "cordovesa", "corina", "corneta",
"cornetinha", "corninglesa", "cornualhesa", "cornuc"+u"\u00F3"+"pia",
"coroa", "cor"+u"\u00F4"+"a", "coroinha", "corola",
"coronha", "corregedoria", "correia", "correnteza",
"correntezinha", "correria", "correta", "corrida",
"corridinha", "cortesia", "cortina", "cortininha",
"coruja", "corujinha", "corunhesa", "corveta",
"cosipa", "costa", "costinha", "costura",
"costureirinha", "cota", "cotanesa", "cotia",
"cotinha", "cotovia", "coura"+u"\u00E7"+"a", "courela",
"cousa", "cova", "covardia", "covinha",
"coxa", "coxinha", "cozedura", "cracovesa",
"craonesa", "cratera", "crian"+u"\u00E7"+"a", "crian"+u"\u00E7"+"ada",
"criancinha", "criatura", "criaturinha", "crina", "cristalandesa",
"crixalandesa", "crosta", "crucilandesa", "crueza",
"cruzadinha", "cruzelandesa", "cueca",
"cuequinha", "culatra", "culpadinha", "cultura",
"cupromanganesa", "c"+u"\u00FA"+"pula", "curadoria",
"curva", "curvatura", "curvinha", "cust"+u"\u00F3"+"dia",
"d"+u"\u00E1"+"diva", "dama", "danceteria", "dancinha",
"danesa", "daroesa", "data", "debandada",
"debulhadeira", "d"+u"\u00E9"+"cada", "defensoria", "defesa",
"delegacia", "delfina", "delfinesa", "delicadeza",
"del"+u"\u00ED"+"cia", "demanda", "demandinha", "demasia",
"democracia", "demora", "dentada",
"dentadinha", "dentadura", "derrocada", "derrota",
"derrotinha", "desarmonia", "descarga", "descida",
"descidinha", "descompostura", "descortesia", "desculpa",
"desdita", "desembocadura", "desenvoltura", "desfeita",
"desfilada", "desfiladinha", "desforra", "desgra"+u"\u00E7"+"a",
"desgraceira", "desonra",
"desova", "despensa", "despenseira", "despesa",
"despesinha", "despreza", "destilaria", "destreza",
"desvalia", "desventura", "deusa", "devassa",
"dezena", "diabrura", "di"+u"\u00E1"+"fana", "diarreia",
"di"+u"\u00E1"+"spora", "dica", "dicotomia", "diepesa",
"dieta", "difteria", "dinamarquesa", "dinamarquesinha",
"dinastia", "dinheirama", "dioptria", "diplomacia",
"diretoria", "disciplina", "disc"+u"\u00F3"+"rdia", "discoteca",
"disenteria", "dislexia", "dispensa", "dispneia",
"distrofia", "ditadura", "diva",
"d"+u"\u00ED"+"vida", "divinolandesa",
"divisa", "dobra", "doca", "do"+u"\u00E7"+"ura",
"doencinha", "dominiquesa",
"dopamina", "d"+u"\u00F3"+"rica", "dormida", "dormidinha",
"dosimetria", "douradura", "doutora", "doutrina",
"draga", "droga", "drogaria", "drupa",
"ducha", "duchinha", "duna", "duplicata",
"duquesa", "dureza", "d"+u"\u00FA"+"vida", "d"+u"\u00FA"+"zia",
"economesa", "economia", "editoria", "efic"+u"\u00E1"+"cia",
"elvira", "ema", "embaixada", "embaixadinha",
"embira", "embocadura", "emenda", "ementa",
"empada", "empadinha", "emp"+u"\u00E1"+"fia", "empatia",
"empresa", "encefalopatia", "enciclop"+u"\u00E9"+"dia", "encomenda",
"encruzilhada", "encruzilhadinha", "endoscopia", "enfermaria",
"engenharia", "engomadeira", "enguia", "enseada",
"enseadinha", "entrada", "entradinha", "entremesa",
"entressafra", "envergadura", "enxada", "enxadinha",
"enxaqueca", "enxurrada", "enxurradinha", "enzima",
"epidemia", "epifania", "ep"+u"\u00ED"+"stola", "epopeia",
"ergonomia", "eritreia", "erva", "ervilha",
"ervilheira", "ervinha", "esbelteza", "escada",
"escadaria", "escadinha", "escaleira", "escama",
"escapada", "escapadinha", "esc"+u"\u00E1"+"pula", "escarcela",
"escarlatina", "escarpa", "escherichia", "escocesa",
"esc"+u"\u00F3"+"cia", "escola", "escolinha", "escondia",
"escopeta", "escopeteira", "escorva", "escotilha",
"escova", "escovinha", "escravaria", "escravatura",
"escritura", "escuderia", "escultura", "esculturinha",
"escuma", "escumilha", "escuta", "esfera",
"esgrima", "esmeralda", "esmeraldinha", "esmola",
"esmolinha", "espada", "espadana", "espadela",
"espadilha", "espadinha", "esp"+u"\u00E1"+"dua", "espalda",
"espanhola", "esp"+u"\u00E1"+"tula", "especiaria", "espectrometria",
"espectroscopia", "espertalhona", "esperteza",
"espessura", "espiga", "espingarda", "espinhela",
"esplanada", "esplanadinha", "esponja", "esponjinha",
"espora", "esporinha", "esp"+u"\u00F3"+"rtula", "espuma",
"esquadra", "esquadria", "esquadrinha", "esquina",
"esquiva", "esquizofrenia", "estaca", "estacada",
"estacadinha", "estadia", "estafeta", "estamenha",
"estampa", "estampinha", "estanqueira", "estaquinha",
"est"+u"\u00E1"+"tua", "estatueta", "estatura", "estearina",
"esteira", "estesia", "estola", "estopa",
"estopinha", "estrada", "estradinha", "estrangeirada",
"estrangeiradinha", "estranheza", "estrasburguesa", "estrat"+u"\u00E9"+"gia",
"estratosfera", "estrebaria", "estreia",
"estreiteza", "estreitinha", "estrela", "estrelinha",
"estrema", "estremadura", "estreptomicina", "estriga",
"estrutura", "estufa", "estult"+u"\u00ED"+"cia", "etapa",
"etapinha", "etiopesa", "etiqueta", "etnia",
"eucaristia", "euforia", "eugenia", "europeia",
"eva", "evasiva", "f"+u"\u00E1"+"brica", "f"+u"\u00E1"+"bula",
"faca", "facada", "facadinha", "fa"+u"\u00E7"+"anha",
"faceta", "facha", "fachada", "fachadinha",
"fac"+u"\u00FA"+"ndia", "fada", "fadiga", "fadinha",
"fagulha", "fagulhinha", "faia", "fa"+u"\u00ED"+"sca",
"fal"+u"\u00E1"+"cia", "falcoaria", "falda",
"faltinha", "fama",
"fanfarra", "fanga", "fantasia", "fantasmagoria",
"faquinha", "farda", "fardeta", "fardetinha",
"fardinha", "farfalha", "far"+u"\u00ED"+"ngea", "farm"+u"\u00E1"+"cia",
"farofa", "farofinha", "farpa", "farpela",
"farpinha", "farra", "farroupeira", "farroupilha",
"farrusca", "farsa", "fartura", "farturinha",
"fasc"+u"\u00ED"+"ola", "fasquia", "fat"+u"\u00ED"+"dica", "fatiota",
"fatura", "fa"+u"\u00FA"+"lha", "fauna", "fausta",
"faustina", "fava", "faveira", "favela",
"favinha", "faxina", "fazenda", "fazendinha",
"fazendola", "fechadura", "f"+u"\u00E9"+"cula", "fedelha",
"fedentina", "fedra", "feijoada", "feira",
"feirinha", "feiti"+u"\u00E7"+"aria", "feitoria", "feitura",
"feiura", "feiurinha", "fel"+u"\u00ED"+"cia", "felpa",
"f"+u"\u00EA"+"mea", "fenciclidina", "fenda",
"fen"+u"\u00ED"+"cia", "fera", "fereza", "feroesa",
"ferradura", "ferradurinha", "ferramenta", "ferrenha",
"ferroada", "ferroadinha", "ferromanganesa", "ferrovia",
"f"+u"\u00E9"+"rula", "fervura", "fervurinha", "festa",
"festina", "festinha", "fiandeira", "fibra",
"fibrina", "fibromialgia", "f"+u"\u00ED"+"bula", "ficha",
"fichinha", "fida", "fidalguia", "figueira",
"figueirinha", "figura", "figurinha", "fila",
"filantropia", "filatelia", "fil"+u"\u00E1"+"ucia", "fileira",
"filharada", "filipinesa", "filogenia", "filomena",
"filosofia", "filoxera", "f"+u"\u00ED"+"mbria", "finesa",
"fineza", "finlandesa", "finlandesinha", "finta",
"finura", "firma", "firmeza", "firmezinha",
"firminha", "fisionomia", "fissura", "f"+u"\u00ED"+"stula",
"fita", "fiteira", "fitinha", "fivela",
"fivelinha", "fl"+u"\u00E2"+"mula", "flanela",
"flanelinha", "flauta", "flecha", "flechinha",
"fleuma", "flora", "fl"+u"\u00F3"+"rea", "florentina",
"floresta", "floricultura", "florinha", "flotilha",
"fobia", "focinheira", "foda", "fodinha",
"fofoca", "fofoquinha", "foga"+u"\u00E7"+"a", "fogueira",
"fogueirinha", "foi"+u"\u00E7"+"ada", "foi"+u"\u00E7"+"adinha", "folga",
"folha", "folhinha", "folia", "forca",
"forcinha", "forja", "formatura", "f"+u"\u00F3"+"rmica",
"formiga", "forminha", "formosura", "f"+u"\u00F3"+"rmula",
"fornada", "fornalha", "forneira", "forneirinha", "forquilha",
"forrinha", "fortaleza", "fortuna", "fotoc"+u"\u00F3"+"pia",
"fotografiazinha", "fotogrametria", "fotonovela", "fotosfera",
"f"+u"\u00F3"+"vea", "fraga", "fragata", "fralda", "fraldinha",
"frama", "francela", "francesa", "francesinha",
"franja", "franjinha", "franqueza",
"franquia", "fraqueza", "fratura", "frauta",
"frecha", "freguesa", "freguesia", "freguesinha",
"freiberguesa", "freima", "freira", "freirinha",
"frescura", "fresquinha", "fressura", "fresta",
"frestinha", "frieza", "fr"+u"\u00ED"+"gia", "frigideira",
"fritura", "fronha", "frontaria",
"fronteira", "frota", "frutaria", "fruteira",
"fruticultura", "fuampa", "fuba", "fu"+u"\u00E7"+"a",
"fuga", "fugidia", "fujona", "fulva",
"fuma"+u"\u00E7"+"a", "fumaceira", "fumacinha", "fundura",
"fun"+u"\u00E9"+"rea", "f"+u"\u00FA"+"rcula", "furna", "fuzilaria",
"gabardina", "gabela", "g"+u"\u00E1"+"bia", "gabonesa",
"gadelha", "gaeta", "gaforina", "gagueira",
"gaia", "gaiata", "gaiola", "gaiolinha",
"gaita", "gaitinha", "gaivota", "gaivotinha",
"gajeira", "galantaria", "galanteria", "gal"+u"\u00E1"+"xia",
"galba", "galeota", "galera", "galeria",
"galerinha", "galesa", "galhada",
"galharda", "galhardia", "galharia", "galheta",
"galhofa", "gal"+u"\u00ED"+"cia", "galileia", "galinhola",
"galinicultura", "galocha", "galopada", "galopadinha",
"galuresa", "galvanoplastia", "gamaglobulina", "gamba",
"g"+u"\u00E2"+"mbia", "gamboa", "gamela", "gamet"+u"\u00F3"+"fita",
"gana", "ganapada", "gandaia", "g"+u"\u00E2"+"ndara",
"gandaresa", "gandra", "gandresa", "ganesa",
"ganga", "gangorra", "gangorrinha", "gangrena",
"garantia", "garapa", "garatuja",
"gar"+u"\u00E7"+"a", "garcesa", "garcinha", "gardunha",
"garfada", "garfadinha", "gargalhada", "gargalhadinha",
"gargalheira", "garganta", "gargantilha", "garnacha",
"garoa", "garonesa", "garotada", "garotadinha",
"garoupa", "garoupinha", "garra", "garrafa",
"garrafeira", "garrafinha", "garraiada", "garrana",
"garrincha", "garrota", "garrucha", "garrula",
"garupa", "garvonesa", "gasista", "gasolina",
"gastromancia", "gastronomia", "gateira", "gatinesa",
"gauchada", "gaulesa", "g"+u"\u00E1"+"vea", "gavela",
"gaveta", "gavetinha", "gavota", "gaza",
"gazela", "gazeta", "gazetilha", "geada",
"geadinha", "geba", "geesa",
"geladeira", "gelara", "gelatina",
"geleia", "gelosia", "gema", "genebra",
"genebresa", "generala", "geneta", "gengiva",
"genoesa", "genovesa", "gentalha", "gentamicina",
"gentileza", "genuesa", "geometria", "geriatria",
"giba", "gibiteca", "giesta", "giganta",
"gigantinha", "gincana", "ginecomastia", "gineta",
"ginja", "gipsofila", "girafa", "gir"+u"\u00E2"+"ndola",
"gitana", "gl"+u"\u00E2"+"ndula", "glicerina", "glicina",
"gliptoteca", "glor"+u"\u00ED"+"ola", "goa", "goanesa",
"goela", "goesa", "goiabada", "gola",
"goleada", "goleadinha", "golfada", "golfadinha",
"golinha", "golpelha", "gonadotrofina", "g"+u"\u00F4"+"ndola",
"gonorreia", "gordura", "gordurinha", "g"+u"\u00F3"+"rgona",
"gorja", "gorjeta", "gorjetinha", "gostosura",
"gota", "gotinha", "governanta", "gra"+u"\u00E7"+"a",
"gracinha", "grafiazinha", "gralha",
"gralhada", "gralhadinha", "grana", "granada",
"granadinha", "grandeza", "grandura", "graninha",
"granja", "granjinha", "granulometria", "grapa",
"gra"+u"\u00FA"+"na", "gravata", "gravatinha", "gravimetria",
"gravura", "gravurinha", "graxa", "graxinha",
"greda", "greguesa", "grelha", "grenha",
"greta", "grilheta", "grimpa", "grinalda",
"griolandesa", "grisalha", "griseta", "gritaria",
"groenlandesa", "gronelandesa", "groselha", "grosseria",
"grossura", "grota", "grua", "grulhada",
"grulhadinha", "gruta", "grutinha", "guadramilesa",
"guaiaca", "guajaratuba", "guamesa", "guampa",
"guampada", "guanina", "guarita", "guarnecia",
"guasca", "guedelha", "gueira", "guerra",
"guerrilha", "guerrinha", "guiana", "guianesa",
"guida", "guiga", "guilhotina", "guinada",
"guinadinha", "guitarra", "gula", "guloseima",
"gurguesa", "guria", "gurizada", "gurupema",
"gusa", "hamburguesa", "harmonia", "harpa",
"harpia", "hasta", "h"+u"\u00E1"+"stea", "h"+u"\u00E1"+"stia",
"havanesa", "hegemonia", "h"+u"\u00E9"+"gira", "helena",
"heliosfera", "hemeralopia", "hemiplegia", "hemocultura",
"hemofilia", "hemoglobina", "hemolinfa", "hemorragia",
"hena", "heparina", "hepatoscopia", "hera",
"heran"+u"\u00E7"+"azinha", "herbicida", "hercogamia", "herculandesa",
"herc"+u"\u00FA"+"lea", "heresia", "hero"+u"\u00ED"+"na", "hetaira",
"heterodoxia", "heterogenia", "heteromorfia", "heterosfera",
"hialina", "hica", "hidra", "hidrazina",
"hidrocefalia", "hidrocortisona", "hidrofilia", "hidrofobia",
"hidrolandesa", "hidromancia", "hidropisia", "hidrosfera",
"hidrovia", "hidroxizina", "hiena", "hierarquia",
"hifa", "hinterlandesa", "hioscina", "hipercolesterolemia",
"hiperestesiada", "hipergamia", "hipermetropia", "hiperon"+u"\u00ED"+"mia",
"hipertermia", "hipertrofia", "hipervolemia", "hipocondria",
"hipocrisia", "hipoteca", "hipotenusa", "hipotermia",
"hipovalva", "hirta", "hirudina", "histamina",
"histerectomia", "histeria", "historieta", "hola",
"holanda", "holandesa", "holandesinha", "holandilha",
"homeopatia", "homeostasia", "homeotermia", "homilia",
"homofobia", "homogenia", "homon"+u"\u00ED"+"mia", "honconguesa",
"honra", "hora", "horda", "horinha",
"hormona", "horneblenda", "horrenda", "h"+u"\u00F3"+"rrida",
"horsa", "horta", "hortali"+u"\u00E7"+"a", "hortalicinha",
"hort"+u"\u00EA"+"nsia", "horticultura", "hortinha", "h"+u"\u00F3"+"speda",
"hospedaria", "h"+u"\u00F3"+"stia", "hua", "hula",
"hulha", "hum"+u"\u00ED"+"lima", "hungaresa", "iara",
"icter"+u"\u00ED"+"cia", "ideia", "ideiazinha", "idiossincrasia",
"idiotia", "idolatria", "igreja",
"igrejinha", "iguaria", "ilha", "ilhota",
"il"+u"\u00ED"+"ada", "iliteracia", "imbituba", "imod"+u"\u00E9"+"stia",
"imper"+u"\u00ED"+"cia", "impostura", "imprensa", "imprimadura",
"impureza", "imunoglobulina", "incerteza",
"incultura", "indelicadeza", "indisciplina", "indochina",
"indochinesa", "ind"+u"\u00FA"+"stria", "inefic"+u"\u00E1"+"cia", "in"+u"\u00E9"+"pcia",
"in"+u"\u00E9"+"rcia", "inf"+u"\u00E2"+"mia", "infanta", "infantaria",
"influenza", "infraestrutura", "inglesa", "inglesinha",
"inhaca", "injusti"+u"\u00E7"+"a", "in"+u"\u00F3"+"pia", "ins"+u"\u00ED"+"dia",
"inspetoria", "insulina", "inteireza", "intemerata",
"interferometria", "interl"+u"\u00ED"+"ngua", "internetesa", "in"+u"\u00FA"+"bia",
"invernada", "inverneira", "investidura", "ionosfera",
"ipueira", "ira"+u"\u00FA"+"na", "irlandesa", "irlandesinha",
"irma", "ironia", "iroquesa", "isca",
"islandesa", "is"+u"\u00F3"+"bata", "isoc"+u"\u00F3"+"rica", "isola",
"isoloma", "isomorfia", "isoniazida", "isonomia",
"isostasia", "isoterma", "isotropia", "israelita",
"itacoatiara", "ivolandesa", "jabuticaba", "jabuticabeira",
"jaca", "jacobina", "jacuba", "jacutinga",
"jaguatirica", "jagun"+u"\u00E7"+"ada", "jaleca", "jamanta",
"jamba", "jana"+u"\u00ED"+"na", "jandaia", "janela",
"janelinha", "jangada", "jangadinha", "janianesa",
"janota", "janta", "jantarola", "japa",
"japana", "japonesa", "japonesinha", "jaqueta",
"jaquetinha", "jaquiranaboia", "jararaca", "jararaquinha",
"jarreta", "jarreteira", "jaula",
"javanesa", "jazida", "jazidinha", "jeira",
"jerarquia", "jeremiada", "jeropiga", "jetica",
"jia", "jiboia", "jinguba", "joalharia",
"joalheria", "joana", "joaninha", "joanita",
"joelhada", "joelhadinha", "joga", "jogatina",
"joia", "joiazinha", "jornada", "jornalesa",
"jota", "jotinha", "juba", "ju"+u"\u00E7"+"ara",
"jucunda", "judeia", "judia",
"judicatura", "ju"+u"\u00ED"+"za", "juncinha", "jungia",
"juntura", "jura", "jurema", "jureminha",
"jurubeba", "jurujuba", "justeza", "justi"+u"\u00E7"+"a",
"juta", "jutlandesa", "labareda", "labuta",
"laca", "lacuna", "ladeira", "ladeirinha",
"ladina", "ladra", "ladrona", "lagartixa",
"lagoa", "lagoinha", "lagosta", "lagostinha",
"l"+u"\u00E1"+"grima", "laguna", "laia", "laja",
"l"+u"\u00E1"+"jea", "lajinha", "lama", "lamba",
"lambada", "lambadinha", "l"+u"\u00E2"+"mina", "lamivudina",
"lampa", "l"+u"\u00E2"+"mpada", "lamparina", "lampinha",
"lampreia", "lampreiinha", "lan"+u"\u00E7"+"adeira", "lanceta",
"landesa", "langonha", "lanolina", "lanterna",
"lanterninha", "lapa", "lapela", "lapinha",
"lapiseira", "laqueadura", "laranjeira", "laranjeirinha",
"lareira", "lareirinha", "larga", "largada",
"largadinha", "largueza", "largura", "larva",
"lasanha", "lasca", "lasc"+u"\u00ED"+"via", "lasquinha",
"lassa", "l"+u"\u00E1"+"stima", "lata", "latada",
"lataria", "latinaria", "latinha",
"latrina", "lavandaria", "lavanderia", "lavareda",
"lavoura", "lavra", "lavratura", "lazarina",
"lazeira", "leandra", "lectina", "leda",
"legenda", "legislatura", "l"+u"\u00E9"+"gua", "leitaria",
"leitura", "leiva", "lembrancinha", "lemna",
"lena", "len"+u"\u00E7"+"aria", "lenda", "lengalenga",
"lenha", "lenhina", "lentilha", "leoa",
"leonesa", "leonina", "leopoldina", "lepra",
"lesma", "letra", "letrinha", "leucemia",
"leucena", "levadi"+u"\u00E7"+"a", "levedura", "leveza",
"leviana", "lev"+u"\u00ED"+"tica", "lexia", "lhama",
"lhaneza", "lhanura", "liamba", "liana",
"libanesa", "libanesinha", "lib"+u"\u00E2"+"nia", "lib"+u"\u00E9"+"lula",
"libra", "li"+u"\u00E7"+"a",
"licenciatura", "licencinha", "lichia", "lida",
"l"+u"\u00ED"+"dima", "lidoca"+u"\u00ED"+"na", "ligadura", "ligaresa",
"ligeireza", "lignina", "l"+u"\u00ED"+"gula", "lila",
"lima", "limburguesa", "limeira", "limonada",
"limpeza", "limpezinha", "lina", "lindeza",
"linfa", "lingala", "l"+u"\u00ED"+"ngua", "lingueta",
"linguetinha", "lingui"+u"\u00E7"+"a", "linguicinha",
"linha"+u"\u00E7"+"a", "lionesa", "liornesa", "lira",
"lisboesa", "lisbonesa", "lisinha",
"lisonja", "lisozima", "lista", "listeria",
"listinha", "lisura", "liteira", "literatura",
"litosfera", "l"+u"\u00ED"+"via", "lixeirinha",
"lix"+u"\u00ED"+"via", "lobotomia", "l"+u"\u00F4"+"brega", "loca",
"locanda", "locandeira", "logomarca",
"logudoresa", "loisa", "loja", "lojeca",
"lojinha", "lombada", "lombeira", "lombriga",
"lona", "lonca", "londresa", "londrina",
"longura", "lonjura", "lontra",
"lontrinha", "loquela", "loratadina", "lorena",
"lorenesa", "loriga", "lorota", "lotar"+u"\u00ED"+"ngia",
"loteria", "lou"+u"\u00E7"+"a", "lou"+u"\u00E7"+"ania", "lou"+u"\u00E7"+"aria",
"loucura", "loucurinha", "loureira", "lousa",
"Lua", "lua", "luanda", "luandesa", "lubina",
"l"+u"\u00FA"+"brica", "lucarna", "lucena", "lucila",
"lucr"+u"\u00E9"+"cia", "l"+u"\u00FA"+"dica", "ludomania", "ludoteca",
"lufada", "luisita", "lula", "lulinha",
"luna", "lunda", "lundesa", "luneta",
"lupa", "l"+u"\u00FA"+"rida", "lusa", "lus"+u"\u00E1"+"cia",
"lus"+u"\u00ED"+"ada", "lusofonia", "lutulenta", "luva",
"luveira", "luvinha", "luxemburguesa", "luzilandesa",
"maca", "maca"+u"\u00E7"+"aresa", "ma"+u"\u00E7"+"ada",
"ma"+u"\u00E7"+"adinha", "ma"+u"\u00E7"+"aneta", "ma"+u"\u00E7"+"aroca", "macaxeira",
"macega", "machadada", "machona", "machorra",
"macia", "macieira", "macieirinha", "ma"+u"\u00E7"+"onaria",
"maconha", "macroeconomia", "macroestrutura", "m"+u"\u00E1"+"cula",
"madalena", "madama", "madeira", "madeirinha",
"madeixa", "madona", "madrasta", "madrep"+u"\u00E9"+"rola",
"madressilva", "madrilesa", "madrugada", "madrugadinha",
"maduresa", "madureza", "m"+u"\u00E1"+"fia", "mafra",
"magda", "magia", "magistratura", "magnetosfera",
"m"+u"\u00E1"+"goa",
"magreza", "maioria",
"mala", "malaguesa", "malaquesa", "maleta",
"malga", "malha", "malhinha", "mal"+u"\u00ED"+"cia",
"malinesa", "malinha", "maloca", "malta",
"maltesa", "maltesia", "maltinha", "maluqueira",
"maluquesa", "malva", "malvina", "malvinesa",
"malvinha", "mama", "mamadeira", "maminha",
"manada", "manadinha", "man"+u"\u00E1"+"pula", "mancheia",
"mandala", "mand"+u"\u00ED"+"bula", "mandinga", "mandioca",
"mandioquinha", "mandolinata", "mandr"+u"\u00E1"+"gora", "manduca",
"maneira", "maneirinha", "manesa", "manga",
"mangaba", "mangabinha", "manganesa", "mangueira",
"mangueirinha", "manha", "mania", "manica",
"manilha", "maniota", "maniotinha", "manivela",
"manjedoura", "manjerona", "manopla", "manquesa",
"mansarda", "manteiga", "mantilha", "m"+u"\u00E1"+"quina",
"maquinaria", "maragota", "maragotinha", "maranduba",
"maratona", "maravilha", "marcenaria", "marcha",
"marchinha", "maresia", "margarida", "margaridinha",
"margarina", "margarita", "marianesa", "marina",
"mariola", "marmelada", "marmeladinha", "marm"+u"\u00F3"+"rea",
"maroca", "maronesa", "marosca", "maroteira",
"marquesa", "marquesinha", "marreta", "marretinha",
"marselhesa", "marta", "maruja", "marujada",
"marujinha", "masca", "m"+u"\u00E1"+"scara", "m"+u"\u00E1"+"scula",
"masmorra", "massa", "masseira", "massinha",
"matadeira", "matilha", "mat"+u"\u00E9"+"ria", "matinada", "matinadinha",
"matraca", "matriarca", "matr"+u"\u00ED"+"cula", "matrona",
"matula", "maura", "maur"+u"\u00ED"+"cia", "mazorca",
"mazurca", "meada", "meadinha", "mealhada",
"meca", "mecha", "medalha",
"medalhinha", "m"+u"\u00E9"+"dia", "mediania", "medicina",
"medula", "medusa", "megalomania", "megera",
"melancia", "melancolia", "melanina", "melatonina",
"melena", "melhoria", "melodia", "membrana",
"menesa", "meninada", "menopausa", "mentira",
"mentirinha", "mercadoria", "mercearia", "merda",
"merdinha", "merenda", "merendinha", "mesa",
"mesada", "mescalina", "meseta", "mesinha",
"mesnada", "mesquita", "messalina", "mestra",
"mestria", "met"+u"\u00E1"+"fora", "metralha", "mica",
"microempresa", "microestrutura", "m"+u"\u00ED"+"dia", "migalha",
"migalhinha", "milanesa", "mil"+u"\u00ED"+"cia", "mina",
"minerva", "minha", "minhoca", "miniatura",
"minoria", "min"+u"\u00FA"+"cia", "mioleira", "miopia",
"miosina", "miquelina", "mira", "mirada",
"miradinha", "mirandelesa", "mirandesa", "mirra",
"misantropia", "miseric"+u"\u00F3"+"rdia", "missa",
"missinha", "mistura", "misturinha", "mitra",
"miu"+u"\u00E7"+"alha", "moca",
"mo"+u"\u00E7"+"ada", "mocetona", "mochila", "mochilinha",
"mo"+u"\u00E7"+"oila", "mod"+u"\u00E9"+"stia",
"modorra", "moeda", "moedinha",
"moega", "moela", "moelinha", "moenda",
"mofa", "mofina", "moira", "moirama",
"moita", "mola", "mold"+u"\u00E1"+"via", "moldura",
"moleca", "molecada", "molecadinha", "mol"+u"\u00E9"+"cula",
"moleira", "moleirinha", "molequinha", "mol"+u"\u00E9"+"stia",
"moleza", "molezinha", "molhadura", "molinha",
"mona", "monarquia", "monja", "monocultura",
"monotonia", "montanha", "montanhesa",
"montearia", "montesa", "montmartresa", "montra",
"montureira", "morada", "moradia", "moradinha",
"mor"+u"\u00E1"+"via", "morcela", "morda"+u"\u00E7"+"a",
"mordedura", "moreia", "moreira", "morfina",
"morquesa", "mortalha", "morvandesa",
"morvanesa", "mosca", "mosquinha", "mosquitada",
"mostarda", "mostardinha", "mota",
"motocicleta", "motocicletinha", "movia",
"mucama", "mudancinha",
"mula", "muleta", "mulherada", "mulinha",
"multim"+u"\u00ED"+"dia", "m"+u"\u00FA"+"mia", "munheca", "muralha",
"mur"+u"\u00E7"+"a", "murceira", "murta", "musa",
"musculatura", "nabi"+u"\u00E7"+"a", "naca", "nadadeira",
"n"+u"\u00E1"+"dega", "nafta", "naftalina", "naifa",
"naja", "namoradeira", "nana", "nantesa",
"napolesa", "narbonesa", "narina", "nata",
"natinha", "natura", "natureza", "naturopatia",
"n"+u"\u00E1"+"usea", "nauseabunda", "navalha", "navalhada",
"navalhinha", "navarra", "navarresa", "neandertalesa",
"neblina", "nebrina", "necromancia", "nectarina",
"n"+u"\u00E9"+"dia", "neerlandesa", "nefelina", "nefelometria",
"negociata", "neoconcreta", "neoplasia", "neozelandesa", "nepalesa",
"neptunina", "nereida", "nervura", "n"+u"\u00E9"+"scia",
"nesga", "nespereira", "netiqueta", "neuma",
"neurastenia", "neuroanatomia", "n"+u"\u00EA"+"ustria",
"nevada", "nevadinha", "nevasca",
"n"+u"\u00E9"+"voa", "nevralgia", "nevropatia", "nexina",
"nhandirova", "niacina", "nicesa", "nicobaresa",
"nicotina", "nicotinamida", "nifedipina", "nigromancia",
"n"+u"\u00ED"+"mia", "nina", "ninfa", "ninfomania",
"ninhada", "ninhadinha", "ninharia", "nitreira",
"nitroglicerina", "n"+u"\u00ED"+"vea", "nivernesa", "niza",
"nobreza", "noca", "noctiluca", "n"+u"\u00F3"+"doa",
"nogueira", "nogueirinha", "noitada", "nomenclatura",
"n"+u"\u00F4"+"mina", "nona", "nora", "noradrenalina",
"norinha", "norlandesa", "norma", "norminha",
"nortada", "norueguesa", "norueguesinha", "nostalgia",
"not"+u"\u00ED"+"cia", "notocorda", "novela", "novelinha",
"novena", "novilhada", "nua", "nuca",
"nuinha", "nunciatura", "oba",
"obra", "obrada", "obrinha", "ocarina",
"odalisca", "odisseia",
"odrada", "oferenda", "oficina",
"ogiva", "oitiva", "ojeriza", "olaia",
"olaria", "oldemburguesa", "oleogravura", "olga",
"oligarquia", "ol"+u"\u00ED"+"mpia", "olimp"+u"\u00ED"+"ada", "olindesa",
"oliva", "oliveira", "oliveirinha", "olivina",
"ombreira", "omoplata", "on"+u"\u00E7"+"a", "oncinha",
"onda", "ondina", "ondinha", "ondurmanesa",
"onomatopeia", "ontogenia", "oogamia", "oosfera",
"opala", "opereta", "oranesa", "orca",
"orchata", "orelha", "orelhinha", "or"+u"\u00E9"+"stia",
"organela", "orla", "orleanesa", "orlinha",
"orquestra", "orqu"+u"\u00ED"+"dea", "ortiga",
"ortodoxia", "orvalhada", "orvalhadinha", "orvalheira",
"osga", "ossada", "ossadura", "ossamenta",
"ossatura", "osteossarcoma", "ostra", "otomana",
"ourama", "ourela", "ousadia", "outonada",
"outorga", "ouvidoria", "ova",
"ovelha", "ovelhinha", "ovinha", "oxitocina",
"paca", "pachorra", "pa"+u"\u00E7"+"oca", "pa"+u"\u00E7"+"oquinha",
"padaria", "p"+u"\u00E1"+"gina", "pala", "palavra",
"palavrinha", "palestra", "palestrinha", "paleta",
"palha", "palheta", "palhinha", "palho"+u"\u00E7"+"a",
"palma", "palmeira", "palmeirinha", "palminha",
"palmira", "paloma", "p"+u"\u00E1"+"lpebra", "pampa",
"pamplonesa", "pancada", "pancadaria", "p"+u"\u00E2"+"ndega",
"panela", "panelinha", "pantera", "pantomima",
"papelada", "papelaria", "paquistanesa", "par"+u"\u00E1"+"bola",
"paralisia", "parcela", "parceria", "pardela",
"pardesa", "parelha", "parenta", "parentela",
"parna"+u"\u00ED"+"ba", "par"+u"\u00F3"+"dia", "par"+u"\u00F3"+"quia", "parreira",
"parreirinha", "parta", "partenesa", "part"+u"\u00ED"+"cula",
"partitura", "parvonesa", "p"+u"\u00E1"+"scoa", "pasmaceira",
"passarada", "passarela", "passeata", "pasta",
"pastilha", "pastilhinha", "pastinha", "pastor"+u"\u00ED"+"cia",
"pataca", "pataquinha", "pateada", "patifaria",
"patilha", "patroa", "patroinha", "patuscada",
"paula", "paulada", "paulina", "pavana",
"paveia", "pe"+u"\u00E7"+"a", "pecha", "pecinha",
"pe"+u"\u00E7"+"onha", "pediatria", "pedofilia", "pedra",
"pedrada", "pedrenesa", "pedresa",
"pedrinha", "pega", "pegada", "pegadinha",
"peia", "peleja", "peli"+u"\u00E7"+"a", "pel"+u"\u00ED"+"cula",
"pena", "penedia", "peneira", "peneirinha",
"penha", "penicilina", "peninha", "pen"+u"\u00ED"+"nsula",
"penteadeira", "penumbra", "peonada", "pepsina",
"pequenada", "pequinesa", "pera", "perda",
"pereira", "pereirinha", "perereca", "perf"+u"\u00ED"+"dia",
"perfumaria", "pergunta", "perguntinha", "per"+u"\u00ED"+"cia",
"periferia", "perinha", "perip"+u"\u00E9"+"cia", "permuta",
"perna", "perninha", "p"+u"\u00E9"+"rola", "perp"+u"\u00E9"+"tua",
"perspectiva", "perspic"+u"\u00E1"+"cia", "pertin"+u"\u00E1"+"cia",
"perturba", "perua", "peruana", "peruca",
"pesca", "pescada", "pescadinha", "pescaria",
"peseta", "pesquisa", "pessoa", "pessoinha",
"peta", "p"+u"\u00E9"+"tala", "peteca", "petersburguesa",
"petinga", "petinha", "petiza", "petizinha",
"pia", "piada", "piadinha", "picardia",
"piemontesa", "pilha", "p"+u"\u00ED"+"lula", "pimenta",
"pimentinha", "pinacoteca", "pinga", "pintura",
"piovesa", "pipa", "piranha",
"pirataria", "piroga", "piscina", "piscininha",
"pista", "pistola", "pita", "pitada",
"pitadinha", "piteira", "pitinha", "pitombeira",
"pizza", "placa", "placenta", "placentinha",
"planilha", "planta", "plantinha", "planura",
"plaquinha", "plataforma", "plateia", "platina",
"pleura", "pluma", "pneumonia", "pobreza",
"pocema", "pocilga", "poda",
"poeira", "poeirada", "poeirinha", "poesia",
"poetisa", "poia", "poiaresa", "polca",
"poliandria", "pol"+u"\u00ED"+"cia", "policromia", "polifonia",
"poligamia", "polissemia", "polonesa", "polpa",
"poltrona", "p"+u"\u00F3"+"lvora", "pomada", "pompa",
"pontada", "pontadinha", "pontaria", "popa",
"popinha", "popula"+u"\u00E7"+"a", "porcaria", "porcelana",
"pororoca", "porta", "portaria", "portela",
"portinha", "portinhola", "portugalesa", "portuguesa",
"portuguesinha", "postura", "pota", "poupancinha",
"pousada", "pousadinha", "p"+u"\u00F3"+"voa", "pra"+u"\u00E7"+"a",
"pracinha", "praga", "praia", "prancha",
"pranchada", "pranchinha", "prataria", "prateleira",
"prateleirinha", "pr"+u"\u00E9"+"dica", "prefeitura", "pregui"+u"\u00E7"+"a",
"preguicinha", "preia", "preitesa", "prelada",
"premissa", "prenda", "prendinha",
"presilha", "pressa", "pressinha", "presteza",
"pretensa", "pret"+u"\u00F3"+"ria", "primavera", "primazia",
"princesa", "princesinha", "prioresa", "proa",
"procuradoria", "proeza", "profecia", "profilaxia",
"profundeza", "promessa", "promotoria", "propaganda",
"propagandinha", "propina", "prosa", "pros"+u"\u00E1"+"pia",
"prosinha", "pros"+u"\u00F3"+"dia", "pr"+u"\u00F3"+"stata", "prote"+u"\u00ED"+"na",
"prova", "provedoria", "psicologuesa", "publicitesa",
"pudic"+u"\u00ED"+"cia", "pulga", "pulseira", "pulseirinha",
"pureza", "quadraria", "quadratura", "quadrela",
"quadricomia", "quadriga", "quadrilha", "quadrilhinha",
"quadrissemana", "quantia", "quaresma", "quebra",
"quebradura", "quebrantada", "quebreira", "queda",
"quedinha", "queimadura", "queiroga", "quelha",
"quenga", "quentura", "queratina", "queratomal"+u"\u00E1"+"cia",
"quercinesa", "querela", "quietura", "quilha",
"quimbanda", "quimera", "quimioprofilaxia", "quimiotaxia",
"quimiotripsina", "quina", "quincha", "quinina",
"quininha", "quinquilharia", "quintana", "quinzena",
"quirguistanesa", "quiromancia", "quitanda", "quitina",
"quixaba", "quota", "quotinha", "rabeca",
"rabeira", "rabona", "r"+u"\u00E1"+"bula", "ra"+u"\u00E7"+"a",
"racinha", "radioastronomia", "r"+u"\u00E1"+"dula", "rainha",
"rainhazinha", "raiva", "raivinha", "rajada",
"rajadinha", "ramaria", "rampa", "rampinha",
"ranchada", "rapa", "rapadura", "rapariga",
"rapaziada", "rapina", "rapinha", "raposa",
"raposinha", "rasoira", "raspadura", "ratazana",
"ratoeira", "ravina", "reabertura", "realeza",
"rebarba", "rebarbativa", "rebeca", "rebeldia",
"reboleira", "rebu"+u"\u00E7"+"ada", "reca", "recandidatura",
"recarga", "recebedoria", "receita", "receitinha",
"recoveira", "r"+u"\u00E9"+"dea", "redoma", "redondeza",
"reengenharia", "reentrada", "reentradinha", "refinaria",
"regalia", "regata", "regatinha", "r"+u"\u00E9"+"gia",
"regina", "regra", "regrinha", "r"+u"\u00E9"+"gua",
"reitoria", "rel"+u"\u00ED"+"quia", "relojoaria", "remessa",
"rena", "renda", "rendinha",
"reninha", "rentesa", "r"+u"\u00E9"+"plica", "reposta",
"represa", "reprimenda", "rep"+u"\u00FA"+"blica", "resenha",
"resina", "resinga", "resposta",
"respostinha", "ressaca", "ressalva", "r"+u"\u00E9"+"stia",
"restinga", "retaguarda", "retina", "retinopatia",
"retorta", "retranca", "revelia", "revenda",
"reviravolta", "revista", "revistinha", "revoada",
"revoadinha", "reza", "rezina", "rezinha",
"riba", "ribaldeira", "ribalta", "ricina",
"rifa", "rifampicina", "rijeza", "riodonoresa",
"rionoresa", "riqueza", "risada", "risadinha",
"risota", "rixa", "rizicultura", "rocha",
"rochinha", "rociada", "rodela", "rodelinha",
"rodilha", "rodopsina", "rodovia", "rolada",
"roladinha", "rolda", "roldana", "roleta",
"rolha", "romanhesa", "romaria",
"romena", "ronca", "ronceira", "ronda",
"rondinha", "ronha", "roreja", "rosa",
"ros"+u"\u00E1"+"cea", "rosalina", "rosaria", "rosca",
"r"+u"\u00F3"+"sea", "roseira", "roseta", "rosilha",
"rosina", "rosinha", "rosita", "rosquilha",
"rosquinha", "rota", "roterdamesa", "rotina",
"rotunda", "rotura", "roubalheira", "roupa",
"rouparia", "roupeta", "roupinha", "rua",
"ruana", "ruanda", "ruandesa", "rub"+u"\u00E9"+"ola",
"r"+u"\u00FA"+"bida", "rubra", "rubrica", "ru"+u"\u00E7"+"a",
"rudeza", "ruela", "ruelinha", "ruga",
"ru"+u"\u00ED"+"na", "ruinha", "ruma", "rumba",
"rupia", "ruptura", "rusga", "r"+u"\u00FA"+"tila",
"sabatina", "sabedoria", "sabinada", "saboneteira",
"sacha", "sacola", "sacolinha", "sacrista",
"sacristia", "sadia", "safadeza", "safena",
"safira", "safra", "saga", "sageza",
"saia", "saiaguesa", "saiinha", "sala",
"salada", "saladinha", "salamandra", "salema",
"saleta", "salinha", "saliva", "salmoira",
"salmonela", "salmoura", "saloia", "salsa",
"salsicha", "salsinha", "samarinesa", "samarra",
"sanfona", "sanfoninha", "sanga", "sangalhesa",
"sangesa", "sangria", "sangueira", "sanhuda",
"sanita", "santana", "santiaguesa", "sanzala",
"sapataria", "sapucaia", "sarabanda", "saracura",
"saraiva", "sar"+u"\u00E7"+"a", "sarda", "sardenha",
"sardinha", "sarjeta", "sarna", "sassaresa",
"s"+u"\u00E1"+"tira", "sauna", "sa"+u"\u00FA"+"va", "savana",
"savelha", "seara", "sebastiana", "sebenta",
"secretaria", "secretina", "secura", "seda",
"segadeira",  "segunda", "segundona", "segurelha",
"seira", "seita", "seiva", "selaginela", "selha",
"selva", "selvageria", "selvajaria", "semana",
"s"+u"\u00EA"+"mea", "semeadura", "sena", "senda",
"sendinesa", "senegalesa", "senha",
"senhorita", "seninha", "senzala", "s"+u"\u00E9"+"pia",
"septicemia", "sepultura", "sequanesa", "sequeira",
"sequela", "serapilheira", "sereia", "serenata",
"seringa", "serotonina", "serpa", "serpentina",
"serra", "serrania", "serrinha",
"serventia", "s"+u"\u00E9"+"rvia", "sesmaria", "sesta",
"seta", "setemesa", "siamesa", "sibila",
"sienesa", "sigla", "signa", "silhueta",
"silicomanganesa", "silva", "silveira", "s"+u"\u00ED"+"lvia",
"silvicultura", "silvinha", "simetria", "simonia",
"simpatia", "simpleza", "sina", "sinagoga",
"sincronia", "s"+u"\u00ED"+"ndroma", "sineta", "sinfonia",
"singalesa", "singeleza", "sintonia", "sirena",
"sirigaita", "siroquesa", "sirventesa", "soalheira",
"soberania", "soberba", "sobrancelha",
"sobrancelhinha", "sobranceria", "sobrecarga", "sobrecarta",
"sobrecasaca", "sobremesa", "sobrevida", "sobrevidinha",
"sobrinha", "sociologuesa", "sodomia", "sofia",
"soja", "soldadura", "soleira",
"sol"+u"\u00E9"+"rcia", "solha", "solteirona", "soltura",
"sombra", "sombria", "sombrinha", "sonata",
"sonda", "soneca", "sonoplastia", "sopa",
"sopinha", "sorveteria", "sotaina", "sousa",
"souza", "sova", "sovela", "sovina",
"sovinha", "sua", "suazilandesa", "subcl"+u"\u00E1"+"via",
"subdelegacia", "subida", "subidinha", "subtileza",
"sucata", "s"+u"\u00FA"+"cia", "su"+u"\u00E7"+"uarana", "sudanesa",
"sudanesinha", "sujeira", "sujeirinha", "sultana",
"s"+u"\u00FA"+"mula", "sundanesa", "supercopa", "superempresa",
"superfina", "superfininha", "superliga", "superta"+u"\u00E7"+"a",
"s"+u"\u00FA"+"plica", "supremacia", "surdina", "surinamesa",
"surra", "surriada", "surriadinha", "susana",
"suserania", "sutileza", "taba", "tabacaria",
"tabaqueira", "tabatinga", "tabela", "tabelinha",
"taberna", "taberninha", "tabica", "tablatura",
"tablita", "t"+u"\u00E1"+"bua", "tabuada", "tabuleta",
"tabuletinha", "ta"+u"\u00E7"+"a", "tacada", "tacadinha",
"tacha", "tachinha", "tacinha", "taenia",
"tailandesa", "taiwanesa", "tajiquistanesa", "talagar"+u"\u00E7"+"a",
"taleira", "talidomida", "talisca", "tamareira",
"tamaresa", "tangapema", "tangerina", "tape"+u"\u00E7"+"aria",
"tapera", "taperinha", "tapioca", "taprobana",
"tapuia", "taquara", "taquarinha", "tarambola",
"taramela", "tarantela", "tarasconesa", "tarbesa",
"tardeza", "tarefa", "tarentesa", "tarifa",
"tarja", "tarjinha", "tarola", "tarouquesa",
"tarrafa", "tarragonesa", "tartaruga", "tasca",
"tascinha", "tasquinha", "tauromaquia", "tavanesa",
"taverna", "t"+u"\u00E1"+"vola", "taxa", "taxonomia",
"teada", "tebaida", "teca", "tecla",
"tecnocracia", "teia", "teima", "teimosia",
"tela", "telefonia", "telemetria", "telenovela",
"telepatia", "televisa", "telha", "telhinha",
"telinha", "telitoquia", "t"+u"\u00EA"+"mpera",
"temperatura", "t"+u"\u00EA"+"mpora",
"tempura", "tenda", "tendinha", "tenra",
"teocracia", "teoria", "teoriazinha",
"tequila", "terapiazinha", "terebintina", "teresa",
"teresinha", "terna", "ternatesa", "ternura",
"ternurinha", "Terra", "terra", "terracota", "terrantesa",
"terrina", "terrinha", "tesouraria", "tessitura",
"testosterona", "teutoburguesa", "textura", "tiara",
"tiazinha", "t"+u"\u00ED"+"bia", "tibieza", "tigela",
"tigelinha", "tigresa", "tijuca", "tintura",
"tipoia", "tirada", "tirania", "tire"+u"\u00F3"+"idea",
"tirolesa", "tiroxina", "tisana", "toada",
"toalha", "toalhinha", "tocaia", "tocha",
"toga", "togolesa", "toguesa", "toleima",
"toleteira", "tona", "tonelada", "tonganesa",
"tonina", "tonquinesa", "tonsila", "tonsura",
"tonteira", "tontura", "tora", "tordilha",
"torga", "torinha", "tornesa", "torpeza",
"torquesa", "torrada", "torradinha", "torrantesa",
"torreira", "tortura", "torva", "tosquia",
"touca", "tou"+u"\u00E7"+"a", "toupeira", "toupeirinha",
"tourada", "touradinha", "toxina", "toxinazinha",
"trabalheira", "tr"+u"\u00E1"+"cia", "trag"+u"\u00E9"+"dia", "tragicom"+u"\u00E9"+"dia",
"traineira", "tralha", "trama",
"trambolha", "tramela", "traminha", "tramoia",
"trancinha", "tranqueta", "trapa"+u"\u00E7"+"a", "trapalhada",
"traparia", "trapassaria", "trapeira", "traqueia",
"traquitana", "trava", "travadoira", "travessia",
"travessura", "travinha", "tr"+u"\u00E9"+"gua", "trela",
"treli"+u"\u00E7"+"a", "tremedeira", "tremelga", "tremesa",
"tremoceira", "tr"+u"\u00E9"+"plica", "tribuna", "trigonometria",
"trincadeira", "trincheira", "tripa", "tripe"+u"\u00E7"+"a",
"tripecinha", "tripinha", "tripla", "tripsina",
"tristeza", "tristezinha", "tristura", "trofa",
"troia", "trolha", "tromba", "trombeta",
"trompa", "tronqueira", "tropa", "tropeada",
"tropilha", "tropomiosina", "troponina", "tropopausa",
"troposfera", "trova", "trovoada", "trovoadinha",
"trufa", "truncadura", "truncatura", "truta",
"tua", "tuatara", "t"+u"\u00FA"+"bara", "tugandesa",
"tulha", "tulipa", "tulonesa", "tumba",
"tuna", "tunda", "tundra", "t"+u"\u00FA"+"nica",
"tuquianesa", "turbamulta", "turbina", "turca",
"turfa", "turinesa", "turma", "turmalina",
"turminha", "turquesa", "turquia", "turra",
"turrinha", "tutela", "tuvalesa",
"uberlandesa", "ucha", "ufania", "ugandesa",
"uiva", "ulna", "ultravioleta", "umbanda",
"umbela", "umbuzada", "unha", "unhada",
"unhagata", "unhinha", "upa", "urca",
"urdideira", "urdidura", "ureia", "uremia",
"uretra", "urgueira", "urina",
"urtiga", "urubuzada", "usbequistanesa",
"usina", "usura", "utiliza", "utopia",
"utopiazinha", "ut"+u"\u00F3"+"pica", "uva", "uvinha",
"uzbequistanesa", "vaca", "vacaria", "vacatura",
"vagareza", "vagina", "vagoneta", "vagueira",
"vaia", "vaiamontesa", "val"+u"\u00E1"+"quia", "valeira",
"valentia", "valentona", "valeta", "valetinha",
"valsa", "valsinha", "v"+u"\u00E1"+"lvula", "vanessa",
"vanguarda", "vaqueirama", "vaquejada", "vaquejadinha",
"vaquinha", "varanda", "varandinha", "vareja",
"vareta", "varga", "varicela", "variegada",
"var"+u"\u00ED"+"ola", "varredura", "v"+u"\u00E1"+"rzea", "vasectomia",
"vasilha", "vasilhinha", "vasopressina", "vassoira",
"vassoirinha", "vassoura", "vassourada", "vassourinha",
"vedeta", "veia", "veiazinha", "veiga",
"vela", "velada", "velhacaria", "velhada",
"velharia", "velhota", "velhusca", "velinha",
"veneta", "veniaga", "ventania", "ventura",
"verbena", "verbinha", "verborragia", "verdasca",
"verdelha", "verdura", "verdurinha", "vereda",
"veredinha", "verga", "vergasta", "vergonha",
"verg"+u"\u00F4"+"ntea", "veria",
"vermina", "veronesa", "verrina", "verruga",
"verruma", "versalhada", "versalhesa", "v"+u"\u00E9"+"rtebra",
"vesga", "ves"+u"\u00ED"+"cula", "vespa", "v"+u"\u00E9"+"spera",
"vespinha", "v"+u"\u00E9"+"stia", "vestidura", "vestimenta",
"vetusta", "vexada", "vexadinha", "vezeira",
"viana", "vianda", "vianesa", "vianinha",
"viatura", "v"+u"\u00ED"+"bora", "vicunha", "vida",
"videira", "videirinha", "videoendoscopia", "videoteca",
"vidinha", "vidra"+u"\u00E7"+"a", "vidra"+u"\u00E7"+"aria", "vieira",
"viela", "vienesa", "vietnamesa", "viga",
"viguesa", "vila", "vilafrancada", "vilania",
"vilari"+u"\u00E7"+"a", "vilegiatura", "vileza", "vilinha",
"vilta", "vina", "vincelha", "vindicta",
"vindima", "vindimadeira", "vindita", "v"+u"\u00ED"+"nea",
"vingancinha", "vinha", "vinha"+u"\u00E7"+"a", "vinhada",
"vinheta", "vinhetinha", "vin"+u"\u00ED"+"cola", "vintena",
"viola", "viol"+u"\u00E1"+"cea", "violinha", "virada",
"viradeira", "viradinha", "virginiana", "virgolandesa",
"v"+u"\u00ED"+"rgula", "virilha", "visada", "visadinha",
"v"+u"\u00ED"+"scera", "viscondessa", "viseira", "v"+u"\u00ED"+"spora",
"vistoria", "vita", "vitamina", "vitaminazinha",
"vitela", "vitelinha", "viticultura", "v"+u"\u00ED"+"tima",
"vitinha", "vitivinicultura", "vitrina", "vitrola",
"vivenda", "vivendinha", "viveza", "v"+u"\u00ED"+"vida",
"vizeira", "vodca", "volta",
"voltinha", "volumetria", "vol"+u"\u00FA"+"pia", "vossenhoria",
"vozearia", "vozeira", "vozeria", "vulgata",
"vulva", "vurtemberguesa", "xangaiesa", "xaputa",
"xenofobia", "x"+u"\u00ED"+"cara", "xilogravura", "zabumba",
"zabumbada", "zaga", "zaida", "zalcitabina",
"zambujeira", "zana", "zaragata", "zaragatoa",
"zarolha", "zarzuela", "zebra", "zelandesa",
"zeugma", "zibelina", "zidovudina", "zina",
"zoada", "zoadinha", "zoeira", "zoidiogamia",
"zoina", "zombaria", "zona", "zonza",
"zoolatria", "zostera", "zura", "zurrapa",
"zurzia"])


# plvrMfoa - palavras masculinas-feminina apenas com alteracao do o-a menin[o] menin[a] 
plvrMfoa = set(["abecedarian", "abelan", "abelian", "abelonian",
"abert", "abevilian", "abian", "abirian",
"abiscan", "ablut"+u"\u00F4"+"man", "abraamian", "abraquian",
"abretan", "abr"+u"\u00F3"+"tan", "absolut", "absurd",
"abul"+u"\u00F4"+"man", "abziritan", "acacian", "acadian",
"acaiacan", "acaian", "acanian", "acantodian",
"acantopterigian", "acaridian", "acarnan", "acarnanian",
"acarnean", "acarnian", "accian", "accipitrian",
"acefalian", "acerran", "acheulian", "acian",
"acidalian", u"\u00E1"+"cid", "acipitrian", "acistan", "acleidian",
"aclidian", "acolalan", "a"+u"\u00E7"+"orian", "acorizan",
"acormian", "acrian", "acridian", "acromiocoracoidian",
"act"+u"\u00E9"+"an", "adamastorian", "adamian", "addisonian",
"adept", "adi"+u"\u00E1"+"fan", "adisonian", "adlerian",
"admir"+u"\u00F4"+"man", "adonian", "adotian", "adrastian",
"adrianopolitan", "adrumetan", "ad"+u"\u00FA"+"lter", "aerian",
"aer"+u"\u00F3"+"fan", "aer"+u"\u00F4"+"man", "afifan", "african",
"agalisian", "agamian", "agarenian", "aginian",
"agonian", "agor"+u"\u00E2"+"man", "agostinian", "agrarian",
"agrian", "agripinian", "aguaian", "agud",
"aladian", "alagoan", "alaman", "alan",
"alantoidian", "alarician", "alasquian", "alban",
"albarran", "albertogalvan", "albian", "albigensian",
"albionian", "albitan", "albuquerquian", "alcan",
"alcmanian", "alcochetan", "alcool"+u"\u00F4"+"man", "aldean",
"alectorian", "alectoromancian", "alencarian", "alentejan",
"aleuromancian", "alexandrin", "alexian", "alfierian",
"algaravian", "algerian", "alg"+u"\u00F4"+"man", "algomanian",
"algonquian", "alhei", "aliban", "alifan",
"alofan", "alogian", "alpinopolitan", "alsacian",
"alt", "althusserian", "altinopolitan", "alun",
"aluvian", "alv", "alvesian", "alvinopolitan",
"alvitan", "amalfitan", "amarel", "amarelinh",
"amarg", "amastrian", "amazonian", "ambian",
"ambilian", "ambrosian", "amebian", "ameliopolitan",
"american", "american"+u"\u00F4"+"man", "amibian", "amidalian",
"amig", "amigdalian", "amonian", "amorian",
"amorinopolitan", "amorosian", "amorrean", "ampl",
"amsterdamian", "anacoret"+u"\u00F4"+"man", "anafrodisian", "analantoidian",
"anamnian", "anan", "anarian", "anastacian",
"anastasian", "anatolian", "anaxagorian", "anchietan",
"ancian", "anciran", "anconitan", "andaman",
"andorian", "andorran", "andorrian", "anedot"+u"\u00F4"+"man",
"anencefalian", "anfibian", "anficelian", "angarian",
"angatuban", "angelian", "angican", "angimacurian",
"anglican", "angl"+u"\u00F4"+"man", "angolan", "anguilan",
"angust"+u"\u00ED"+"man", "anibalian", "animiquian", "anomian",
"anopenian", "ansarian", "anselmian", "antarian",
"antelian", "antelicotragian", "antelitragian", "anterian",
"antessignan", "antic"+u"\u00F4"+"man", "ant"+u"\u00ED"+"fon", "antig",
"antigenidian", "antilian", "antilopian", "antinomian",
"antioquian", "antipodian", "antipolitan", "antirrepublican",
"antirrobertian", "antiultramontan", "antocian", "ant"+u"\u00F4"+"man",
"antonian", "antoninian", "antuerpian", "apalachian",
"apelician", "apexian", "apician", "ap"+u"\u00F3"+"fan",
"ap"+u"\u00F3"+"stol", "aprustan", "apuan", "apulean",
"apulian", "aquarian", "aquenian", "aquilean",
"aquilian", "aquilitan", "aquitan", "aquitanian",
"aracajuan", "ara"+u"\u00E7"+"atiban", "aracnean", "aracnian",
"aracnoidian", "aradian", "aragan", "araguaian",
"aralian", "aranean", "araucan", u"\u00E1"+"rbitr", "arc",
"arcadian", "arcan", "ardian", "arduinian",
"areiopolitan", "arenapolitan", "arenopolitan", "argan",
"argelian", "argentin", "argian", "argiletan",
"arguesian", "arian", "arissantorinian", "aristofanian",
"aristoxenian", "aritenoidian", "aritm"+u"\u00F4"+"man", "arlesian",
"arm"+u"\u00EA"+"ni", "armenian", "arminian", "armorican",
"arnopolitan", "arocelitan", "arquean", "arquiloquian",
"arquitet", "arsenomelan", "artejan", "artemonian",
"artesian", "arubian", "asbolan", "ascanian",
"ascaridian", "ascetan", "ascidian", "asclepitan",
"asc"+u"\u00F3"+"fan", "ascrian", "asculan", "asgardian",
"asian", "asmonian", u"\u00E1"+"sper", "aspidian", "assassin",
"assidian", "assirian", "assisian", "assotan",
"astecan", "asten"+u"\u00F4"+"man", "astian", "astigitan",
"astorgan", "astragalian", "astromancian", "asturian",
"astut", "ataman", "atelan", "atelorraquidian",
"atent", "aterian", "at"+u"\u00E9"+"rman", "atibaian",
"atilian", "atlasian", "atletican", "atloidian",
"atran", "aturian", "auburnian", "audian",
"augustan", "augustinian", "aulican", "aurelian",
"aurignacian", "ausentan", "ausetan", "ausonian",
"australasian", "australian", "austrasian", "austr"+u"\u00ED"+"ac",
"autocaricat", "autodian", "autopian", "autoplan",
"autunian", "auxitan", "avan", "avan"+u"\u00E7"+"ad",
"avelinopolitan", "averan", "avidian", "axinian",
"azerbaidjan", "azoian", "baamian", "babiruss",
"bacalaurean", "bachelardian", "bachian",
"bacian", "baconian", "bacterian", "bacteridian",
"bactrian", "bactromancian", "badarian", "baenossaurian",
"bagageir", "bahamian", "baian", "baianopolitan",
"bailadeir", "bailarin", "baix", "bajan",
"bajocian", "b"+u"\u00E1"+"lan", "balan"+u"\u00E7", "balancinh",
"balantidian", "baleeir", "balet"+u"\u00F4"+"man", "baltasarian",
"balzacian", "balzaquian", "banan", "bananeir",
"banc", "band", "banezian", "bangladechian",
"bangladeshian", "banh", "banheir", "banian",
"banqueir", "baquean", "baquian", "barat",
"barbadian", "b"+u"\u00E1"+"rbar", "barbian", "barbican",
"barbudan", "barbuzan", "barc", "barcan",
"bardesan", "bardesanian", "bardisan", "barician",
"barqueir", "barquinh", "barr", "barrac",
"barran", "barranc", "barraquinh", "barremian",
"barsanian", "bartonian", "basanomelan", "basanom"+u"\u00E9"+"lan",
"bashkirian", "basilian", "basilidian", "basoquian",
"bastard", "batan", "batesian", "baudelairian",
"baudrillardian", "bauhausian", "baxterian", "beat",
"b"+u"\u00EA"+"bad", "bebian", "bechuan", "becquerian",
"beethovenian", "bel", "belasian", "belitan",
"bellinian", "belomancian", "belost"+u"\u00EA"+"man", "beltian",
"beltran", "bempostan", "bendit", "benedit",
"beneventan", "bengasian", "benign", "benossaurian",
"bent", "benthamian", "bentopolitan", "benzedeir",
"bercian", "beregran", "bergistan", "bergitan",
"bergmanian", "bergsonian", "berkelian", "berman",
"bernardian", "bertopolitan", "bessdian", "betabacterian",
"betican", "betinian", "bezerr", "biafran",
"biamerican", "bibian", "biblioclept"+u"\u00F4"+"man", "bibli"+u"\u00F4"+"man",
"bibliomancian", "bic", "bicheir", "bicotidian",
"bigodeir", "bigua"+u"\u00E7"+"uan", "biguan", "bilaquian",
"bilheteir", "b"+u"\u00ED"+"man", "biplan", "biquotidian",
"birman", "birmanian", "bisaian", "bisbilhoteir",
"biscargitan", "bisgargitan", "bismarckian", "bismarquian",
"bisonh", "bisonheri", "bizarr", "bob",
"bobinh", "bocacian", "bocagian", "boccaccian",
"bodian", "bodonian", "bo"+u"\u00EA"+"mi", "bogomilian",
"bogotan", "boituvan", "bol", "bolan",
"bolborraquidian", "bolet", "bolinh", "bolivarian",
"bolivian", "bols", "bonec", "bonfinopolitan",
"bonifacian", "bonit", "bononian", "bonosian",
"boolean", "boolian", "bopian", "boppian",
"boran", "borbonian", "borborian", "borcan",
"bornean", "boschiman", "bosforan", "b"+u"\u00F3"+"sni",
"bosnian", "bosporan", "bosquiman", "bostonian",
"botafogan", "boticelian", "botnian", "botsuan",
"boximan", "box"+u"\u00ED"+"man", "bozonian", "bracaraugustan",
"bragan", "bragan"+u"\u00E7"+"an", "branc", "brancaran",
"brand", "brandesian", "branquinh", "braquist"+u"\u00E9"+"fan",
"braseir", "brasileir", "brasilian", "brasitan",
"brasopolitan", "braunian", "brav", "brechtian",
"brentanian", "bric"+u"\u00F4"+"man", "brigadian", "brigian",
"britan", "brixian", "brodosquian", "bromoetan",
"bromopentan", "brownian", "brucian", "brunean",
"brut", "brutian", "brux", "brux"+u"\u00F4"+"man",
"buan", "bubetan", "buenopolitan", "bugeran",
"bugi", "bulboclitoridian", "bulbopenian", "burdigalian",
"burguesinh", "burr", "burrinh", "burundian",
"busan", "butan", "byronian", "caba"+u"\u00E7"+"uan",
"caban", "cabocl", "cabriuvan", "cabu"+u"\u00E7"+"uan",
"cachoeiran", "cachorr", "cacodemon"+u"\u00F4"+"man",
"cacofat"+u"\u00F4"+"man", "cacofon"+u"\u00F4"+"man",
"cadmian", "caetan", "caetanopolitan", "cafezopolitan",
"caiaponian", "caicavian", "caietan", "cainian",
"cajan", "calaarian", "calabrian", "calan",
"cal"+u"\u00E7"+"ad", "cal"+u"\u00E7"+"adinh", "calcanean", "caledonian",
"californian", "calm", "calomelan", "calom"+u"\u00E9"+"lan",
"calovian", "calv", "camapuan", "camarajiban",
"camaran", "cambaratiban", "camberran", "cambojan",
"cambojian", "cambrian", "cameleonian", "camelian",
"cameronian", "camilian", "camonian", "campanian",
"campian", "campichan", "can", "canadian",
"canapolitan", "cancerian", "c"+u"\u00E2"+"ndid", "candidat",
"canec", "canfan", "canisian", "canossian",
"canovian", "cantaridian", "cantorian", "canzianopolitan",
"caparican", "caparitan", "capetian", "capinopolitan",
"capivarian", "capixaban", "capoeiran",
"caprelian", "caprian", "capricornian", "capsitan",
"capuan", "caralitan", "caramban", "caramulan",
"caran", "carapicuiban", "caravaggian", "carboran",
"carcaman", "cardan", "carelian", "carian",
"caribian", "cariguan", "carlopolitan", "carlostadian",
"carlovingian", "carlylian", "carman", "carmanian",
"carmelitan", "carmelopolitan", "carmopolitan", "carnacian",
"carnaiban", "carolingian", "carolstadian", "carontian",
"carotidian", "carp", "carpetan", "carpian",
"carpocracian", "carpometacarpian", "carpopterigian", "carquesian",
"carranchan", "carrapitan", "carrilan", "carroceir",
"carseolan", "carsitan", "carteir", "cartesian",
"cartimitan", "cartorian", "cartusian", "carvalhopolitan",
"cas", "casac", "casc", "caseir",
"casimirian", "caspian", "casquivan", "cassinian",
"castelhan", "castelperronian", "casterlian", "castilian",
"castrinopolitan", "cataban", "cataclismian", "catalan",
"catapan", "catarian", "catarquean", "catarrinian",
"c"+u"\u00E1"+"tedr", "catian", "catolician", "catonian",
"catruman", "caucasian", "caud"+u"\u00ED"+"man", "cauranan",
"caval", "cavalarian", "cavalerian", "cavian",
"caxemirian", "cebian", "cebuan", "cefaleditan",
"cefalodian", "cefal"+u"\u00F4"+"man", "cefalorraquian", "cefalorraquidian",
"ceg", "celestian", "celossomian", "celsian",
"celt"+u"\u00F4"+"man", "cenoman", "cenomanian", "cent"+u"\u00ED"+"man",
"cent"+u"\u00F4"+"man", "centrafrican", "cerafan", "cerban",
"cercan", "cercinitan", "cerdonian", "cerebrorraquidian",
"ceresopolitan", "cerretan", "cert", "cervicoacronian",
"cesarian", "cest"+u"\u00F3"+"plan", "cetan", "cevian",
"c"+u"\u00E9"+"zannian", "chabacan", "chabian", "chadian",
"chamusan", "chandlerian", "chaplinian", "charabilhan",
"charad"+u"\u00F4"+"man", "charian", "charmutian", "charran",
"chassidian", "chatelperronian", "chatian", "chavelhan",
"chavian", "chelean", "chic", "chilen",
"chiman", "chinel", "chiquitan", "chiraquian",
"chiriguan", "chopinian", "churchillian", "cian",
"c"+u"\u00ED"+"an", "cianocetan", "cianomelan", "cianom"+u"\u00E9"+"lan",
"cibilitan", "cicendelian", "ciceronian", "cicindelian",
"ciclan", "cicloalcan", "ciclobutan", "ciclocefalian",
"cicloexan", "ciclopentan", "ciclopian", "ciconian",
"cigan", "cigarr", "cilician", "cim"+u"\u00F3"+"fan",
"cimolian", "cingaporean", "cingapurian", "cint",
"circadian", "circameridian", "circassian", "circatejan",
"circunfaringian", "cisfretan", "cisjuran", "cisleitan",
"cispadan", "cistarian", "cistebutan", "cistidian",
"cistigritan", "cisuranian", "citopolitan", "clactonian",
"clar", "clarian", "clast"+u"\u00F4"+"man", "claudelian",
"claudiopolitan", "clazomenian", "cle"+u"\u00F3"+"fan", "clept"+u"\u00F4"+"man",
"clet"+u"\u00F4"+"man", "clidoepitroclian", "clin"+u"\u00F4"+"man", "clismian",
"clit"+u"\u00F4"+"man", "clitorian", "clitoridian", "clitr"+u"\u00F4"+"man",
"clop"+u"\u00EA"+"man", "clop"+u"\u00F4"+"man", "cloral"+u"\u00F4"+"man", "cloraluretan",
"cloretan", "clor"+u"\u00F3"+"fan", "clorofilian", "clorometan",
"clunisian", "c"+u"\u00F3"+"an", "cobert", "coblencian",
"cocain"+u"\u00F4"+"man", "coccigian", "cocigean", "c"+u"\u00F3"+"dan",
"coestaduan", "colapian", "colarian", "colcozian",
"colectan", "coler"+u"\u00F4"+"man", "colestan", "coletan",
"colheitan", "colhetan", "colombian", "colonian",
"coman", "comanchian", "comblanchian", "combonian",
"c"+u"\u00F4"+"mod", "comorian", "companheir", "comparoquian",
"complex", "compostelan", "comprovincian",
"comtean", "comtian", "c"+u"\u00F4"+"ncan", "concordian",
"condilian", "condilotroclian", "condistritan", "condrocoracoidian",
"condroepitroclian", "condrostibian", "confucian", "coniacian",
"conidian", "conquilian", "consistorian", "constantinian",
"constantinopolitan", "contan", "cont"+u"\u00ED"+"nu", "conurban",
"convertiplan", "cop", "copan", "copeir",
"copernican", "copernician", "coplan", "copr"+u"\u00F4"+"man",
"coracoidian", "coralian", "coran", "cordeiropolitan",
"cordilheiran", "cord"+u"\u00ED"+"man", "cordofanian", "corean",
"core"+u"\u00F4"+"man", "corguinhan", "corimbacterian", "corinebacterian",
"corintian", "coriorretinian", "coritan", "cormarian",
"c"+u"\u00F3"+"rne", "cornean", "cornelian", "cornian",
"coroidean", "coronarian", "coronoidean", "correggian",
"correntezan", "corrigan", "corti"+u"\u00E7", "cortonian",
"corumbaiban", "cosan", "cosmopolitan", "cossetan",
"costepitroclian", "costocoracoidian", "costopubian", "costotraquelian",
"costureir", "cotejipan", "cotian", "cotidian",
"cotiloidian", "coulombian", "couperinian", "courbetian",
"coveitian", "cozinheir", "crabilhan", "cracoian",
"cracovian", "cramulhan", "cranachian", "cranian",
"cravilhan", "cremasterian", "cremonian", "cricaritenoidian",
"cricoaritenoidian", "crioul", "criptomelan", "crisamebian",
"crisob"+u"\u00E1"+"lan", "crisopolitan", "cristalinian", "cristalofilian",
"cristianapolitan", "cristianopolitan", "cristinapolitan", "cristopolitan",
"crivellian", "croacian", "crocian", "crocodilian",
"cromagnonian", "cromerian", "cromobacterian", "crotopod"+u"\u00F4"+"man",
"cru"+u"\u00F4"+"man", "ctonian", "cuaitian", "cub",
"cuban", "cubitocarpian", "cubitofalangian", "cuboidocunian",
"cudan", "cuiaban", "cuiauan", "cuisian",
"culicidian", "culteran", "cult"+u"\u00F4"+"man", "cuman",
"cumberlandian", "cunean", "cuproasbolan", "curilian",
"curitiban", "curituban", "curvelan", "curvian",
"cutchian", "cutian", "cuuaitian", "cuvaitian",
"d'alembertian", "dacn"+u"\u00F4"+"man", "dacnomon"+u"\u00F4"+"man", "dactil"+u"\u00F3"+"graf",
"dalinian", "daltonian", "dan", "dan"+u"\u00E7"+"arin",
"dantian", "danubian", "daomean", "darwinian",
"decan", "d"+u"\u00E9"+"cim", "decr"+u"\u00E9"+"pit", "decuman",
"dec"+u"\u00FA"+"man", "defunt", "deleuzian", "dens",
"dentu"+u"\u00E7", "derradeir", "descobert", "desembara"+u"\u00E7"+"ad",
"desert", "desgra"+u"\u00E7"+"ad", "desgra"+u"\u00E7"+"adinh", "des"+u"\u00ED"+"di",
"deslandesian", "deslandian", "desmobacterian", "destr",
"destr"+u"\u00ED"+"man", "desuman", "deucalian", "deuterostomian",
"devonian", "devot", "deximontan", "dextran",
"dextr"+u"\u00ED"+"man", "dextr"+u"\u00F4"+"man", "dextrorfan", "dezembran",
"diab", "diaconian", "diacranian", "diacrian",
"di"+u"\u00E1"+"fan", "dianopolitan", "dianteir", "diat"+u"\u00E9"+"rman",
"diazometan", "dibenziletan", "dibenzilometan", "diboran",
"dibrometan", "dibromopropan", "dicerian", "dicloretan",
"diclorometan", "dicondilian", "dicran", "dicresiletan",
"dicresilofenilometan", "dicresilometan", "diderotian", "diditan",
"didonian", "difenan", "difeniletan", "difenilmetan",
"difenilobutan", "difenilodicloretan", "difenilodiclorometan", "difenilodimetiletan",
"difenilometan", "difenilopropan", "difenoletan", "diiodetan",
"diiodometan", "diluvian", "diminut", "dinaftilometan",
"dinancian", "dinarian", "dind", "dinitropropan",
"diocesan", "dionisian", "dioscorian", "diospolitan",
"dioxan", "dipirrilmetan", "diplantidian", "diplectan",
"diplobacterian", "dips"+u"\u00F4"+"man", "direit", "dirian",
"disc"+u"\u00ED"+"pul", "discret", "discurseir", "dissan",
"dissilan", "dissilican", "dissilicoetan", "distint",
"distomian", "dit"+u"\u00E9"+"rman", "divers", "divin",
"divinopolitan", "divisopolitan", "d"+u"\u00ED"+"zim", "djibutian",
"dobradi"+u"\u00E7", "doceir", "dodecan", "dodecapolitan",
"domazan", "domerian", "dominican", "don",
"donatian", "doninh", "donzel", "dordonian",
"dorisopolitan", "dormideir", "dorsiepitroclian", "dorsimedian",
"dorsoepitroclian", "dorsomedian", "dox"+u"\u00F4"+"man", "draconian",
"dragoman", "dramaturg", "drogoman", "drummondian",
"d"+u"\u00FA"+"bi", "dulcinopolitan", "dulian", "duodecan",
"duod"+u"\u00E9"+"cim", "dupl", "durkheimian",
"eberthian", "eb"+u"\u00FA"+"rne", "ebusitan", "e"+u"\u00E7"+"anian",
"ecavian", "ecetran", "ecian", "eclesian",
"eclus", "ectocranian", "ectorraquidian", "eculan",
"edetan", "edipian", "efedrin", "ef"+u"\u00EA"+"mer",
"egban", "egelestan", "egestan", "egipan",
"eg"+u"\u00ED"+"pci", "egipcian", "egitan", "egnacian",
"eicosan", "eifelian", "einstanian", "einsteinian",
"einstenian", "eisensteinian", "elafian", "elan",
"elean", "eleit", "eletrotr"+u"\u00E9"+"pan", "eleuter"+u"\u00F4"+"man",
"eleuteropolitan", "eliotian", "elisabetan", "elisabetian",
"elisian", "elizabethan", "elpinian", "elusan",
"elzevirian", "emeritan", "emilian", "emilianopolitan",
"empedoclian", "emporitan", "empreiteir", "enan",
"encefalorraquian", "encefalorraquidian", "enc"+u"\u00FA"+"man", "endaspidian",
"enderodan", "endocranian", "endorraquian", "endorraquidian",
"eneaboran", "enferm", "enfermeir", "engra"+u"\u00E7"+"ad",
"enose"+u"\u00F4"+"man", "ente"+u"\u00F4"+"man", "enterobacterian", "enterocelomian",
"entremontan", "entrepan", "entrerrian", "enxabregan",
"eocambrian", "eodevonian", "eolian", "eperlan",
"epicondilian", "epicranian", "epictonian", "epidauritan",
"ep"+u"\u00ED"+"fan", "episcopalian", "epitacian", "epitauritan",
"epitet"+u"\u00F4"+"man", "epitroclean", "epitrocleoolecranian", "epoxietan",
"equatorian", "equiculan", "equinean", "equinian",
"equinocian", "eranian", "erasmian", "erastian",
"ergasi"+u"\u00F4"+"man", "erg"+u"\u00F4"+"man", "er"+u"\u00ED"+"dan", "erot"+u"\u00F4"+"man",
"ervan", "esbelt", "escafoiducunean", "escafoldoastragalian",
"escalabitan", "escapulotroquiterian", "escardonitan", "escass",
"esclerocornean", "escorpian", "escrav", "escrib"+u"\u00F4"+"man",
"escrit", "escur", "escus", "esfenovomerian",
"esferorradian", "esfincterian", "esgui", "eslov"+u"\u00E1"+"qui",
"esmitian", "esofagian", "esopian", "esparnacian",
"espars", "espartan", "espencerian", "espenglerian",
"esperantinopolitan", "espert", "espess", "espinh",
"espintrian", "espiran", "espiritan", "espiropentan",
"espl"+u"\u00EA"+"ndid", "espoletan", "espont"+u"\u00E2"+"ne", "esquerd",
"esquilian", "esquisit", "esquizocelomian", "esquizotripan",
"estabian", "estalian", "estalinian", "estampian",
"estancarian", "estancian", "estapedian", "estatan",
"estatuderian", "est"+u"\u00E9"+"fan", "estefanian", "est"+u"\u00E9"+"gan",
"estendalian", "estenost"+u"\u00E9"+"fan", "estereorradian", "esternomasseterian",
"esternotroquinian", "esterradian", "estesicorian", "estesi"+u"\u00F4"+"man",
"estilocoplan", "estilpnocloran", "estilpnomelan", "estonian",
"estrabonian", "estramontan", "estrangeir", "estranh",
"estreit", "estricn"+u"\u00F4"+"man", "estrit", "estr"+u"\u00F4"+"man",
"estrombolian", "estrumeir", "estupend", "esubian",
"etan", "et"+u"\u00E9"+"re", "etern", "eter"+u"\u00F4"+"man",
"etiluretan", "etiopian", "etolian", "euclidean",
"euclidian", "eudoxian", "eugenopolitan", "eulerian",
"eunapolitan", "eunomian", "eurafrican", "euralian",
"euramerican", "eurasian", "euriapolitan", "euripidian",
"euritan", "euscarian", "eustaquian", "eust"+u"\u00E9"+"fan",
"eutiquian", "eutireoidian", "evangelistan", "exaspidian",
"exat", "excels", "ex"+u"\u00ED"+"gu", "exocranian",
"extens", "extern", "extint", "extrem",
"exurban", "fabian", "fabrician", "faceir",
"fagifulan", "falangean", "falangian", "falanginian",
"falansterian", "faloidian", "falopian", "fals",
"famenian", "familisterian", "faquirian", "faringian",
"farinheir", "farmac"+u"\u00F4"+"man", "fart", "fasian",
"fat"+u"\u00ED"+"can", "faunian", "faustian", "favorit",
"faxineir", "fazendeir", "feacian", "febronian",
"fedrian", "fei", "feit", "feiticeir",
"felin", "felizard", "feminin", "fenectan",
"fenian", "fenilcroman", "fenilmetan", "feniluretan",
"feretrian", "fernandopolitan", "f"+u"\u00E9"+"rre", "ferrialofan",
"ferroestilpnomelan", "ferromagnesian", "fichtian", "ficocian",
"fidalg", "fidian", "fidiculan", "fidjian",
"fieldingian", "fig", "fijian", "filadelfian",
"filantrop"+u"\u00F4"+"man", "filh", "filintian", "filipopolitan",
"filolusitan", "fil"+u"\u00F4"+"man", "filonian", "fil"+u"\u00F3"+"sof",
"fin", "financeir", "fingid", "fioropolitan",
"firman", "firminopolitan", "flacian", "flacourtian",
"flacurtian", "flagel"+u"\u00F4"+"man", "flameng", "flandrian",
"flaubertian", "flaviopolitan", "fleminguian", "flenusian",
"florestopolitan", "florian", "florianopolitan", "fluoran",
"fluormetan", "fluvian", "foc", "fof",
"fogueteir", "fon"+u"\u00F4"+"man", "fontan", "forasteir",
"for"+u"\u00E7"+"ad", "forentan", "foretan", "formand", "formian",
"formosan", "forr", "forulan", "fosc",
"fosforan", "foss", "fotinian", "fot"+u"\u00F3"+"graf",
"fotoplan", "frac", "franc", "francan",
"francian", "francinopolitan", "franciscan", "franciscopolitan",
"franc"+u"\u00F4"+"man", "franconian", "frang", "frangipan",
"franklinian", "frase"+u"\u00F4"+"man", "frasnian", "frebelian",
"fregelan", "freirean", "freirian", "frentan",
"freudian", "fri", "fris", "frisian",
"friulan", "fr"+u"\u00ED"+"vol", u"fr\u00F6belian", "froebelian",
"froux", "frut", "fuchsian", "fucsian",
"fulan", "f"+u"\u00FA"+"lgid", "fulian", "fund",
"fundan", "funest", "furan", "furfuran",
"furian", "fusan", "fusc"+u"\u00ED"+"man", "futur",
"gabalitan", "gabian", "gaditan", "gaian",
"gaj", "g"+u"\u00E1"+"lban", "galecian", "galeg",
"galen", "galerian", "galh", "galhan",
"galhoban", "galhofeir", "galican", "galician",
"galilean", "galinheir", "galizian", "galoisian",
"gal"+u"\u00F4"+"man", "galopolitan", "galvan", "gambian",
"gameleir", "gandhian", "gans", "garaban",
"garapeir", "garavan", "gardonian", "gargantuan",
"gargasian", "garot", "garran", "garrettian",
"garumnian", "gastresofagian", "gastrodi"+u"\u00E1"+"fan", "gat",
"ga"+u"\u00FA"+"ch", "gaulitan", "gaurian", "gaussian",
"gavelan", "gavieir", "gazan", "gedinian",
"gedrosian", "gefirian", "geiserian", "geladinh",
"geleir", "g"+u"\u00EA"+"me", "geminian", "genetlian",
"genian", "genovevan", "genu"+u"\u00ED"+"n", "geoplan",
"georgian", "georgin", "geran", "gercinopolitan",
"geresan", "geresian", "german", "gessnerian",
"getulian", "gidian", "ginasian", "ginec"+u"\u00F4"+"man",
"giroplan", "giscardian", "gitan", "givetian",
"givfitan", "gladstonian", "glauc"+u"\u00F3"+"fan", "gliceran",
"gliciltriptofan", "glicosaminoglican", "glossian", "glozelian",
"goan", "gobinian", "goelan", "goethian",
"goiabeir", "goian", "goianapolitan", "goianian",
"goisian", "goldonian", "gomarian", "gomesian",
"gongan", "gonzaguian", "gord", "gordian",
"gorduch", "gorgian", "gorr", "gorutuban",
"gotlandian", "gouvean", "graban", "gracianopolitan",
"graf"+u"\u00F4"+"man", "gramadeir", "grat", "gravan",
"gravetian", "graviscan", "grecian", "grec"+u"\u00F4"+"man",
"greg", "gregorian", "gril", "gritadeir",
"grocian", "groselheir", "gross", "grosseir",
"gua"+u"\u00E7"+"uan", "guaiuban", "gualterian", "guan",
"guanaban", "guarapuavan", "guaraque"+u"\u00E7"+"aban", "guaratuban",
"guaxupean", "guerreir", "guerrilheir",
"guinean", "guipuscoan", "guiratingan", "guis",
"guitan", "gunzian", "gurutuban", "gusan",
"gutembergian", "gutemberguian", "gutenbergian", "hadranitan",
"hahnemannian", "haitian", "halafian", "hallstattian",
"halomancian", "halstatian", "hamidian", "hamiltonian",
"hamletian", "handelian", "hanoverian", "hanovrian",
"hansean", "hansenian", "harm"+u"\u00F3"+"fan", "hartmanian",
"hassidian", "hastensian", "hatorian", "hauterivian",
"havaian", "havan", "haxix"+u"\u00F4"+"man", "hebraic",
"hediond", "hegelian", "heideggerian", "h"+u"\u00E9"+"li",
"heliconian", "heli"+u"\u00F3"+"fan", "heli"+u"\u00F3"+"man", "heliopolitan",
"helopian", "helvecian", "helvidian", "hemasian",
"hematot"+u"\u00ED"+"mpan", "hemimontan", "hendecan", "henrican",
"hentriacontan", "hequelian", "heraclitian", "herban",
"herbartian", "herciliopolitan", "hercinian", "herculan",
"herculanian", "herdeir", "herderian", "heredian",
"heriban", "hermesian", "hermian", "hermitian",
"hermogenian", "herodian", "herodotian", "heroic",
"heroin"+u"\u00F4"+"man", "hertzian", "hesbaian", "hesper"+u"\u00F3"+"fan",
"hessian", "heteralian", "heteroplan", "heterosoounian",
"heterost"+u"\u00E9"+"fan", "heterousian", "hexaclorocicloexan", "hexacloroetan",
"h"+u"\u00ED"+"brid", "hidraeroplan", "hidrelitan", "hidroaeroplan",
"hidrobiplan", "hidroboran", "hidrobrasileir", "hidrodips"+u"\u00F4"+"man",
"hidr"+u"\u00F3"+"fan", "hidr"+u"\u00F4"+"man", "hidromonoplan", "hidrot"+u"\u00ED"+"mpan",
"hierapolitan", "hierosolimitan", "higr"+u"\u00F3"+"fan", "hilbertian",
"hilean", "hileian", "hindustan", "hipolitan",
"hip"+u"\u00F4"+"man", "hipovarian", "hipsistarian", "hircan",
"hirt"+u"\u00ED"+"man", "hispan", "hister"+u"\u00F4"+"man", "hitlerian",
"hjelmslevian", "hobbesian", "hoffmannian", "holaspidian",
"holbachian", "holbaquian", "hollywoodian", "holocranian",
"homeousian", "hominian", "homoplan", "homousian",
"honest", "horacian", "hospedeir", "hospitaleir",
"h\u00FCbnerian", "hudsonian", "hugoan", "hullian",
"human", "humaniformian", "humanitarian", "humboldtian",
"humian", "h"+u"\u00FA"+"ngar", "hunterian", "huronian",
"husserlian", "hutchinsonian", "huterian", "huttonian",
"iacrian", "iamnotan", "iau"+u"\u00E1"+"can", "ibaran",
"ibaretan", "ibegean", "ibgean", "ibitigua"+u"\u00E7"+"uan",
"ibsenian", "icon"+u"\u00F4"+"man", "icosan", "icositan",
"idiocicl"+u"\u00F3"+"fan", "ienissean", "ienisseian", "igacian",
"igara"+u"\u00E7"+"uan", "igeditan", "igua"+u"\u00E7"+"uan", "iguan",
"iguanian", "iliberitan", "iliopretrocanterian", "iliopretrocantinian",
"iliorrotulian", "iliotrocanterian", "iliotrocantinian", "iliturgitan",
"illinoisian", "ilocan", "ilopolitan", "ilorcitan",
"imediat", "imens", "imorredour", "imperfeit",
"impolut", "imprevist", "imund", "inat",
"inaudit", "incaut", "incert", "inclus",
"inc"+u"\u00F3"+"gnit", "inc"+u"\u00F4"+"mod", "inculcadeir", "incult",
"indaiatuban", "indecis", "indian",
"indigest", "indign", "indiret", "indiscret",
"indisfar"+u"\u00E7"+"ad", "ind"+u"\u00F4"+"mit", "indon"+u"\u00E9"+"si", "indostan",
"industan", "in"+u"\u00E9"+"dit", "inequ"+u"\u00ED"+"voc", "inesian",
"infecund", "infinit", "infravermelh", "infus",
"ing"+u"\u00EA"+"nu", "ingrat", "ingrian", "inimig",
"iniodimian", "injust", "in"+u"\u00F3"+"cu", "inquiet",
"inquilin", "insan", "insensat", "ins"+u"\u00F3"+"lit",
"instant"+u"\u00E2"+"ne",
"insulindian", "inteir", "inteiri"+u"\u00E7", "intens",
"intercuspidian", "interioran", "intern", "internetian",
"interplan", "intrarraquidian", "intrus", "inuman",
"in"+u"\u00FA"+"mer", "invers", "invict", "invitatorian",
"iodometan", "iolopolitan", "ioruban", "ipojucan",
"ipresian", "ipuan", "iracemapolitan", "iranian",
"iraquian", "irian", "iridian", "iridocornean",
"irineopolitan", "irinotecan", "irituian", "irrequiet",
"irriquiet", "irvingian", "isidorian", "ismaelian",
"ismailian", "isobritan", "isobutan", "isoctan",
"is"+u"\u00F3"+"fan", "isofluoran", "isopentan", "isquiointertrocanterian",
"isquiopenian", "isquiopubian", "isquiopubitrocanterian", "isquiotrocanterian",
"israelian", "istrian", "itabiran", "itainopolitan",
"itaiopolitan", "italian", "italian"+u"\u00F4"+"man", "italician",
"ital"+u"\u00F4"+"man", "itamarandiban", "itamontan", "itan",
"itaocan", "itaparican", "itapecerican", "itapetingan",
"itapolitan", "itaquaquecetuban", "itatiban", "itibiran",
"itiquiran", "ituan", "ituiutaban", "iucatan",
"jabutian", "jacksonian", "jacobian", "jacsonian",
"jaguariban", "jaguaruan", "jalesian", "jamaican",
"jambolan", "jan", "jandaian", "jangan",
"jardineir", "jardinopolitan", "jarr", "jassidian",
"jau"+u"\u00E1"+"can", "jeceaban", "jeffersonian", "jenerian",
"jennerian", "jerezan", "jerezian", "jernean",
"jeronimian", "jerosolimitan", "jeruzan", "jesenopolitan",
"jessean", "jibutian", "joalheir", "joanesian",
"joanopolitan", "jobinian", "jordan", "jordanian",
"jovian", "jovinian", "judaic", "judean",
"julian", "juliopolitan", "jument", "jundiaian",
"junguian", "junqueir", "junqueiropolitan", "junquerian",
"junt", "jupiterian", "juquerinan", "juquerionan",
"juquirionan", "jurassian", "juruban", "jusan",
"just", "justiceir", "justinian", "justinopolitan",
"juvenalian", "kandorian", "kelvinian", "kemalian",
"keynesian", "kiribatian", "kryptonian", "kuwaitian",
"l"+u"\u00E1"+"bdan", "l"+u"\u00E1"+"bi", "labican", "labortan",
"la"+u"\u00E7"+"ad", "lacanian", "lacerdopolitan", "lacetan",
"lactametan", "lactovegetarian", "ladan", "l"+u"\u00E1"+"dan",
"ladroeir", "l"+u"\u00E1"+"gan", "lagart", "lagean",
"lagiran", "lagoan", "lagrangian", "laian",
"lajean", "lajian", "laletan", "lal"+u"\u00F4"+"man",
"lamarckian", "lamartinian", "lambran", "lamean",
"lamian", "laminitan", "lancasterian", "lancastrian",
"landenian", "landrinopolitan", "languedocian", "l"+u"\u00E2"+"nguid",
"lantan", "l"+u"\u00E2"+"ntan", "laocian", "laodicean",
"laosian", "laostan", "laotian", "lapachan",
"lapian", "laplacian", "laramian", "larendan",
"laringian", "lasciv", "lassalian", "lassallian",
"laterculian", "lat"+u"\u00ED"+"man", "latin", "laud",
"l"+u"\u00E1"+"udan", "lauran", "laurencian", "laurentian",
"lauretan", "lausanian", "lausannian", "lausian",
"lavadeir", "lavandeir", "lavaterian", "lecontian",
"legendrian", "legerian", "leg"+u"\u00ED"+"tim", "leibnitzian",
"leibnizian", "leiteir", "leman", "lemurian",
"lenan", "leninian", "lent", "leodoman",
"leoflorestan", "leonian", "leopardian", "leopolitan",
"lepidomelan", "leptitan", "leptocardian", "lesbian",
"lesotian", "let"+u"\u00F4"+"man", "leuc"+u"\u00F3"+"fan", "leucopetrian",
"levalloistan", "levaloisian", "levian", "levinopolitan",
"lhan", "liberian", "libertin", "l"+u"\u00ED"+"bi",
"libisosan", "librian", "liburnian", "licean",
"licopolitan", "liechtensteinian", "ligeir", "ligerian",
"ligitan", "ligorian", "liguorian", "ligurian",
"liliputean", "liliputian", "lilliputian", "liman",
"limian", "l"+u"\u00ED"+"mpid", "lind", "linean",
"linguedocian", "linh", "linnean", "li"+u"\u00F3"+"cran",
"liparitan", "l"+u"\u00ED"+"quid", "lis", "lisboan",
"lisonjeir", "listenstainian", "listentainian", "lisztian",
"liter"+u"\u00F4"+"man", "litrean", "littrean", "lituan",
"litu"+u"\u00E2"+"ni", "liuran", "livonian", "livramentan",
"lixeir", "lob", "lobatian", "lobeir",
"locarnian", "locustian", "loflucarban", "logaritm"+u"\u00F4"+"man",
"log"+u"\u00F4"+"man", "loir", "loira"+u"\u00E7", "lomb",
"lombocostotraquelian", "lombodorsotraquelian", "lombotrocantinian", "lombrosian",
"londinian", "longev", "long"+u"\u00ED"+"man",
"longinian", "long"+u"\u00ED"+"nqu", "longulan", "lopan",
"loretan", "lorman", "lorquian", "lotaringian",
"lotoman", "louc", "louletan", "lour",
"lourencian", "louretan", "lovelacian", "luan",
"lucaian", "lucan", "lucanian", "lucasian",
"lucianopolitan", "luc"+u"\u00ED"+"fer", "luciferian", "lucilian",
"lucopetrian", "lucrecian", "luculian", "ludian",
"ludovician", "luhmannian", "luisian", "lulian",
"lumberitan", "lupinan", "lupinopolitan", "lusacian",
"lusitan", "lusit"+u"\u00E2"+"ni", "lusitanian", "lus"+u"\u00F3"+"fon",
"lutecian", "luteran", "lutevan", "lutjan",
"ma"+u"\u00E7", "macac", "macacuan", "macatuban",
"macedonian", "machadian", "macian", "maci"+u"\u00E7",
"madeireir", "madimamuan", "maestrichtian", "mafometan",
"magan", "magdalean", "magdalenian", "magdalian",
"maglemosian", "magn", "magnesian", "magoadinh",
"magosian", "magr", "magrebian", "magrinh",
"maian", "maimonidian", "maioran", "malacitan",
"malacopterigian", "malasian", "malauian", "malavian",
"malawian", "maldan", "maldichan", "maldit",
"maldivan", "maldivian", "malian", "malign",
"mallarmean", "malpighian", "malthusian", "maluc",
"malucan", "malvernian", "mambucan", "mameluc",
"man", "manchesterian", "manchurian", "mancian",
"mancunian", "mandchurian", "mandean", "mandian",
"mandibulomentonian", "mandruan", "manducan", "mangan",
"manganomelan", "mangaratiban", "manitan", "mans",
"mansfieldian", "mant", "mantenopolitan", "mantovan",
"mantuan", "maometan", "maracajuan", "maracian",
"maragogipan", "maragojipan", "maran", "marangatuan",
"marc", "mar"+u"\u00E7"+"an", "marcelian", "marchan",
"marcian", "marcianopolitan", "marcoman", "marcosian",
"marcovian", "marculian", "mardan", "marfinian",
"mariamitan", "marian", "marianopolitan", "mariapolitan",
"marijuan", "marinettian", "marinopolitan", "mariopolitan",
"marisopolitan", "marlian", "marlowian", "marmontelian",
"marmulan", "marran", "marrec", "marsilian",
"martaban", "martinican", "martinopolitan", "marxian",
"m"+u"\u00E1"+"scul", "masculin", "massangan", "massilian",
"massilitan", "matarruan", "mateolan", "matutin",
"maupassantian", "maurian", "maurician", "mauritan",
"maurit"+u"\u00E2"+"ni", "mauritanian", "maurrasian", "m"+u"\u00E1"+"xim",
"maximilian", "maxwellian", "mazovian", "mazurian",
"mean", "meclenburgian", "m"+u"\u00E9"+"dan", "median",
"mediocarpian", "mediotarsian", "mediterran", "mediterranian",
"medonh", "megal"+u"\u00F4"+"man", "megalopolitan", "megapolitan",
"megarian", "meig", "melan",
"melanchthonian", "melanostibian", "melcoman", "melecian",
"melfalan", "melgacian", "melian", "melindan",
"melitan", "melitonian", "mel"+u"\u00F4"+"man", "melquisedecian",
"menalian", "menandrian", "mendelian", "mendelssohnian",
"mendig", "menian", "menin", "mensageir",
"mentan", "mentesan", "mentonian", "menuccian",
"meonian", "mer", "mercaptan", "mercesan",
"mercurian", "merfian", "merian", "meridian",
"meringian", "meroitan", "merovingian", "meruocan",
"mesm", "mesmerian", "mesmer"+u"\u00F4"+"man", "mesoamerican",
"mesocarpian", "mesocordilheiran", "mesodevonian", "mesometatarsian",
"mesopot"+u"\u00E2"+"mi", "mesotarsian", "mesovarian", "messalian",
"messianopolitan", "mesvinian", "metacarpian", "metacarpofalangian",
"metafisic"+u"\u00F4"+"man", "metamelfalan", "metan", "metastasian",
"metatarsian", "metatarsofalangian", "metiluretan", "metonian",
"metoxietan", "metoxifluoran", "metr"+u"\u00F4"+"man", "metropolitan",
"mexican", "micenian", "michan", "micoquian",
"micran", "microbian", "mididitan", "migecian",
"miguelopolitan", "milenarian", "milesian", "miletan",
"milevetan", "milevitan", "milhan", "milian",
"milician", "miliradian", "millian", "miltonian",
"mimosean", "mindelian", "minean", "mineir",
"minian", "m"+u"\u00ED"+"nim", "ministr", "minoan",
"minuan", "min"+u"\u00FA"+"scul", "minut", "mirab"+u"\u00E1"+"lan",
"mirab"+u"\u00F3"+"lan", "mirandolan", "mirandopolitan", "mirianopolitan",
"mirmeleonian", "mirob"+u"\u00E1"+"lan", "m"+u"\u00ED"+"ser", "missianopolitan",
"mississipian", "mississippian", "mistralian", "misulan",
"mitanian", "mitchurinian", "mit"+u"\u00F4"+"man", "mi"+u"\u00FA"+"d",
"moametan", "mo"+u"\u00E7", "mo"+u"\u00E7"+"ambican", "mocan",
"mocinh", "mococan", "mod", "modern",
"modest", "modinh", "moican", "mojian",
"molibduran", "moli\u00F8rian", "molucan", "moment"+u"\u00E2"+"ne", "monasterian",
"mongolian", "monjolan", "monocarpian", "monocefalian",
"mon"+u"\u00F4"+"man", "mononfalian", "monoplan", "monossilan",
"mon"+u"\u00F3"+"ton", "monovarian", "monozonian", "monroean",
"montan", "montemuran", "montessorian", "monteverdian",
"montevidean", "montian", "montijan", "m"+u"\u00F3"+"rbid",
"mordovian", "moren", "morfin"+u"\u00F4"+"man", "morgean",
"morgian", "moribund", "morn", "morretian",
"mort", "mosan", "mosassaurian", "motan",
"mour", "mousterian", "mozartian", "mucan",
"mu"+u"\u00E7"+"ulman", "mud", "mudinh", "mu"+u"\u00E9"+"dan",
"muensterian", "muentzerian", "muggletonian", "mugletonian",
"mulat", "multan", "multiplan", "muncausenian", "munchhausenian",
"mundan", "muratorian", "murcian", "murexan", "muribecan",
"murilhan", "muritiban", "muse"+u"\u00F4"+"man", "musican",
"music"+u"\u00F4"+"man", "mussetian", "mussolinian", "musterian",
"mutan", "m"+u"\u00FA"+"tu", "mutunopolitan", "mutustratan",
"naalian", "nabantian", "nabocovian", "nabuquian",
"naftalan", "naft"+u"\u00F4"+"man", "namibian", "namurian",
"nantuan", "napolitan", "narandiban", "narcoguerrilh",
"narc"+u"\u00F4"+"man", "nasocian", "nasofaringian", "nasogenian",
"nasserian", "nassovian", "natalin", "natercian",
"natinian", "natinopolitan", "nativan", "nativitan",
"n"+u"\u00E1"+"ufrag", "nauran", "nauruan", "naxian",
"nazarean", "neandertalian", "neapolitan", "necisian",
"necissian", "neditan", "nefand", "nefast",
"nefridian", "nefrocelian", "negr", "negreir",
"nemean", "nemertian", "nemetan", "nemian",
"neocomanian", "neocomian", "neonomian", "neopolitan",
"neorrepublican", "neorroman", "neperian", "nepotian",
"neptunian", "neronian", "neropolitan", "nerulan",
"nestorian", "net", "netunian", "neur",
"neustrian", "neutr", "newtonian", "nhangua"+u"\u00E7"+"uan",
"nhungua"+u"\u00E7"+"uan", "nicaraguan", "nicerocian", "nicerotian",
"nicomelan", "nicopolitan", "nietzschian", "nigerian",
"nilopolitan", "ninf"+u"\u00F4"+"man", "nitrofuran", "nitrometan",
"noecian", "noemian", "noetan", "noetian",
"noiv", "nojent", "nolan", "nomentan",
"nonan", "norban", "norcaran", "nordestan",
"norian", "norman", "normand", "norpinan", "nos"+u"\u00F4"+"man",
"noss", "nost"+u"\u00F4"+"man", "noterian", "not"+u"\u00ED"+"dan",
"noturn", "nov", "novacian", "novat",
"novgorodian", "novi"+u"\u00E7", "novilh", "n"+u"\u00FA"+"bi",
"nubian", "nuculan", "numestran", "numidian",
"numistran", "nunalvarian", "oasian", "obes",
"obreir", "obscen", "obscur", "obsidian",
"obsolet", "obstetrican", "obstetr"+u"\u00ED"+"ci", "oc",
"ocan", "ocau"+u"\u00E7"+"uan", "occipitatloidian", "occipitomentonian",
"ocean"+u"\u00F3"+"graf", "ocipitatloidian", "ocipitomentonian", "ocitan",
"ocran", "ocriculan", "octacosan", "octadecan",
"octavian", "octazan", "ocult", "offenbachian",
"ofidian", "ogan", "oitav", "oiticican",
"olbian", "ol"+u"\u00E9"+"cran", "olecranian", "oleir",
"ol"+u"\u00ED"+"ban", "oliculan", "olig"+u"\u00F4"+"man", "olimpian",
"olisiponian", "oliven"+u"\u00E7"+"an", "olmedan", "olulan",
"omacefalian", "omalian", "oman", "omanian",
"omocefalian", "omotraquelian", "omotroquinian", "omotroquiterian",
"one"+u"\u00F4"+"man", "oni"+u"\u00F4"+"man", "onomat"+u"\u00F4"+"man", "onquestian",
"ontarian", "onusian", "opac", "opidan",
"opimian", "opi"+u"\u00F4"+"man", "oportun", "opost",
"ops"+u"\u00F4"+"man", "opulent", "oran", "oratorian",
"orbibarian", "orcian", "orcistan", "ordovician",
"orelhan", "orelheir", "oretan", "orex"+u"\u00ED"+"man",
"organomagnesian", "or"+u"\u00ED"+"gan", "oritan", "ormuzian",
"ornitisquian", "ornit"+u"\u00F4"+"man", "orquest"+u"\u00F4"+"man", "orquidian",
"ort"+u"\u00F3"+"dan", "ortodox", "osirian", "oslerian",
"ossean", "ossenian", "ossiandrian", "ostran",
"otaian", "otavian", "otoman", "otomian",
"outr", "ovarian", "ovidian", "ovolactovegetarian",
"oxfordian", "oxican", "oxonian", "pacatuban",
"padovan", "padroeir", "paduan", "paisan",
"palacian", "palauan", "paleoamerican", "paleochadian",
"paleodevonian", "paleossiberian", "paleoterian", "palepolitan",
"palermitan", "palestin", "palestinian", "palestrinian",
"palican", "palmean", "palmeiropolitan", "palmian",
"palmifalangian", "palminopolitan", "palmiplan", "palmirian",
"palmitopolitan", "palmopolitan", "palonzan", "palouzan",
"palp"+u"\u00ED"+"man", "paludan", "paludian", "pambotan",
"p"+u"\u00E2"+"mpan", "pampangan", "pampean", "pampian",
"panamian", "pandan", "p"+u"\u00E2"+"ndan", "panfilian",
"panglossian", "panonian", "panopolitan", "panormitan",
"pantopelagian", "panurgian", "papeir", "papinian",
"papuan", "paquilemurian", "parabolan", "paracalan",
"paracefalian", "paracelsian", "paracian", "paradoxian",
"paradoxidian", "paragolcian", "paraguaian", "paraiban",
"paraisopolitan", "paralel", "paralian", "paran",
"paranaiban", "paranapolitan", "parapian", "paratireoidian",
"paratucan", "parentan", "parian", "parican",
"parisian", "parkinsonian", "parman", "parmigian",
"parnaiban", "parnasian", "paroquian", "parotian",
"parotidean", "parotidian", "parran", "parrascan",
"parsan", "parsian", "parteir", "partenian",
"partenopolitan", "parvatan", "pascalian", "pascoan",
"passagian", "past", "pasteuranian", "pasteurian",
"pastorean", "pastorian", "pastran", "pat",
"patalitan", "pataran", "paternian", "pat"+u"\u00F4"+"man",
"patrasan", "p"+u"\u00E1"+"tri", "patr"+u"\u00ED"+"ci", "patrician",
"patripassian", "patron", "pauletian", "paulian",
"paulician", "paulistan", "paulopolitan", "pausulan",
"pavan", "pavlovian", "pechuran", "peciolean",
"peciolian", "pecuran", "pedan", "pedestrian",
"ped"+u"\u00ED"+"man", "pedreir", "pedrinopolitan", "pedunculian",
"p"+u"\u00E9"+"gan", "pegasian", "peguan", "peircian",
"peixeir", "pelagian", "pelasgian", "pelean",
"pelican", "pelusian", "pelvian", "pelvitrocanterian",
"peman", "penapolitan", "pendencian", "p"+u"\u00EA"+"ndul",
"penean", "penian", "pensilvanian", "pentapolitan",
"pen"+u"\u00FA"+"ltim", "pepucian", "pepuzian", "pequen",
"pequenin", "peregrin", "perfeit", "p"+u"\u00E9"+"rfid",
"periamigdalian", "periecian", "periesofagian", "perineoclitoridian",
"perioecian", "peripenian", "periurban", "permian",
"pernambucan", "perosian", "perrian", "persepolitan",
"persian", "peruan", "peruiban", "perusian",
"peruvian", "pervers", "p"+u"\u00E9"+"ssim", "pessoan",
"pestalozzian", "pestan", "petrarquian", "petrobrucian",
"petrobrusian", "petrol"+u"\u00ED"+"fer", "petropolitan", "peuc"+u"\u00E9"+"dan",
"pfaffian", "piagetian", "pian", "picassian",
"p"+u"\u00ED"+"fan", "pilan", "pinan", "pint",
"pioneir", "pioran", "piracicaban", "piracurucan",
"piran", "pirandellian", "piratiningan", "piratuban",
"pirauban", "pirenopolitan", "piritiban", "pir"+u"\u00F3"+"fan",
"pir"+u"\u00F4"+"man", "piromelan", "pirom"+u"\u00EA"+"lan", "pironeptunian",
"pisan", "piscian", "p"+u"\u00ED"+"tan", "pitecian",
"pitopolitan", "pitulan", "placencian", "plantinian",
"pl"+u"\u00E1"+"tan", "platician", "platicran", "platonian",
"plautian", "pleban", "plen", "plesiometacarpian",
"plinian", "plumban", "plutarquian", "plutonian",
"pluvian", "pneumot"+u"\u00ED"+"mpan", "pobrezinh", "pojucan",
"polandian", "poliactinian", "polian", "poliglican",
"polinesian", "polinopolitan", "poliplan", "politic"+u"\u00F4"+"man",
"poliuretan", "polonian", "polovtsian", "pomb",
"pomeran", "pomeranian", "pompadourian", "pompean",
"pompeian", "poncian", "ponteir",
"pontian", "pontif"+u"\u00ED"+"ci", "pontocaspian", "popelican",
"poplican", "popperian", "porc", "pori"+u"\u00F4"+"man",
"port", "porteir", "portlandian", "portulan",
"potician", "potirendaban", "pot"+u"\u00F4"+"man", "potsdamian",
"pouc", "poundian", "poxorean", "pracean",
"pracian", "pradopolitan", "praian", "prasian",
"prastran", "prat", "pratapolitan", "pratian",
"praxean", "praxilian", "praxitelian", "precis",
"prefeit", "preparatorian", "pres", "presbiteran",
"presbiterian", "prestilian", "prest"+u"\u00ED"+"man", "pretan",
"pretorian", "pretucan", "pr"+u"\u00E9"+"vi", "previst",
"priabonian", "priam"+u"\u00F4"+"man", "prim", "primeir",
"principan", "priscilian", "proboscidian", "procelian",
"proclian", "proculian", "procustian", "profund",
"prolan", "pront", "propan", "propost",
"pr"+u"\u00F3"+"pri", "proran", "prossifonian", "prossimian",
"prostitut", "proteinoglican", "proteoglican", "prototerian",
"proudhonian", "proustian", "provincian", "pr"+u"\u00F3"+"xim",
"prudentopolitan", "prudhommian", "prussian", "pruss"+u"\u00F4"+"man",
"psatirian", "psilomelan", "pterigian", "pterocerian",
"pubian", "publican", "puccinian", "pudian",
"pupil", "pur", "puritan", "p"+u"\u00FA"+"rpur",
"pursarian", "put", "puteolan", "quacrian",
"quadr", "quadr"+u"\u00ED"+"man", "quadr"+u"\u00FA"+"man", "qu"+u"\u00E1"+"drupl",
"quadrurban", "quaquerian", "quarenten", "quarenton",
"quart", "quartadeciman", "quartan", "quarteir",
"quartodeciman", "quatriduan", "quatruman", "quebradeir",
"quebradi"+u"\u00E7", "quedan", "queirosian", "queix",
"quelonian", "quenian", "queplerian", "querencian",
"quer"+u"\u00F4"+"man", "querquetulan", "quevedian", "quiban",
"quiet", "quiluran", "quiman", "quimeridgian",
"quin", "quinan", "quinquegencian", "quint",
"quintadeciman", "quintan", "quintilian", "quinzopolitan",
"quiribatian", "quirinopolitan", "quir"+u"\u00F3"+"fan", "quironian",
"quitandeir", "quotidian", "r"+u"\u00E1"+"ban", "rabd"+u"\u00F3"+"fan",
"rabelaisian", "rabelian", "rabican", "rabichan",
"rabosan", "rabud", "racinian", "rad"+u"\u00ED"+"cul",
"radiocarpian", "radi"+u"\u00F4"+"man", "rafaelian", "r"+u"\u00E1"+"fan",
"rafardan", "ragusan", "rai", "raian",
"rajaputan", "rajputan", "ram", "ramalh",
"rameir", "rapan", "raquidian", "rar",
"ras", "raspadeir", "rastafarian", "rasteir",
"rat", "ratzelian", "rauracian", "ravelian",
"rechan", "rechonchud", "rec"+u"\u00ED"+"proc", "reclus",
"redavian", "redescobert", "redonian", "reduvian",
"reescrit", "referend", "regadeir", "regalian",
"regan", "regateir", "regatian", "regian",
"reginopolitan", "reichian", "religi"+u"\u00F4"+"man", "remot",
"renan", "renanian", "repentin", "repolhud",
"reptilian", "republican", "repuls", "ret",
"retilian", "retinian", "retorian", "retroestilian",
"retrogasserian", "retroiridian", "retrossifonian", "reuchlinian",
"ri", "rianapolitan", "ribacudan", "ribagorzan",
"ribanceir", "ribatejan", "ribeir", "ribeiropolitan",
"ric", "rica"+u"\u00E7", "ricardian", "ricardin",
"ricinian", "rid"+u"\u00ED"+"cul", "riemannian", "rifan",
"rij", "rinopolitan", "riojan", "riquetsian",
"risonh", "r"+u"\u00ED"+"spid", "rissian", "ritterian",
"rizoplan", "robenhausian", "robercian", "robertian",
"robervalian", "robespierrian", "robinsonian", "robust",
"ro"+u"\u00E7"+"ad", "roceir", "rod", "rodanian",
"rodarsenian", "rodelan", "rodendran", "rodesian",
"rodoarsenian", "rod"+u"\u00F3"+"fan", "rodriguian", "roemerian",
"rogacian", "rol", "rolandian", "roldonian",
"roman", "romeir", "rondonian", "rondonopolitan",
"ronsardian", "rooseveltian", "roqueir", "roquian",
"rosapolitan", "rosian", "rosman", "rosminian",
"rossinian", "rotarian", "roteiran", "rotineir",
"rotoplan", "r"+u"\u00F3"+"tul", "rotulian", "rouc",
"rousseaunian", "rox", "roxolan", "ruan",
"ruban", "rubelitan", "rubiacean", "rubican",
"rudolfian", "ruian", "ruivan", "rupelian",
"rurban", "ruselan", "rusquinian", "russ",
"russan", "russelian", "russellian", "russian",
"rustican", "rutilian", "saadian", "saarian",
"sabacian", "s"+u"\u00E1"+"ban", "sabatarian", "sabatian",
"sabean", "sabelian", "s"+u"\u00E1"+"bi", "sabin",
"sabinan", "sabinian", "sabinopolitan", "saboian",
"sac", "sacassan", "sacoman", "sacr",
"sacramentan", "sacran", "sacr"+u"\u00ED"+"leg", "sacrococcigian",
"sacroprofan", "sacrorraquian", "sacrorraquidian", "sacrossant",
"sacrotrocanterian", "saelian", "saferian", "sagitarian",
"sahelian", "saian", "salacian", "salapitan",
"salernitan", "salesian", "salesopolitan", "salgadeir",
"saliferian", "salinopolitan", "salman", "salonitan",
"salpesan", "salpingian", "salsetan", "salvaterrian",
"salvatorian", "salveterran", "samanian", "samaritan",
"sambaetiban", "sambaquian", "samian", "samoan",
"samoran", "samosatian", "sampsean", "sanchesian",
"sandemanian", "sanfranciscan", "sanoisian", "sansimonian",
"sant", "santanopolitan", "santonian", "santopolitan",
"santorian", "santorinian", "sap", "sarafan",
"sarago"+u"\u00E7"+"an", "sarapuian", "sarcocarpian", "sarcofagian",
"sardian", "sarian", "sarmacian", "sartrian",
"satanian", "saticulan", "satrican", "saturnian",
"saumurian", "saurisquian", "saussurian", "saxonian",
"schillerian", "schlegelian", "schoenbergian", "sch\u00F6nbergian",
"schopenhauerian", "schubertian", "scottian", "seabranan",
"sebastianopolitan", "sebusian", "sec", "secan",
"secret", "secundan", "secundian", "sedetan",
"segarelian", "segestan", "segovian",
"segur", "sel", "selepitan", "seleucian",
"selimbrian", "semelitan", "sementeir", "semiprofan",
"semisserran", "semisshakespearian", "semissoberan", "senegalian",
"senegambian", "senhori", "senonian", "septempedan",
"sept"+u"\u00ED"+"man", "sequacian", "s"+u"\u00E9"+"quan", "sequanian",
"sequardian", "serapidian", "seren", "sergipan",
"seringueir", "sernancelhan", "serran", "serranopolitan",
"serravalian", "sertanej", "sertanopolitan", "sertolian",
"sertorian", "serv", "servian", "sesquiplan",
"setabitan", "setempedan", "setian", "setican",
"s"+u"\u00E9"+"tim", "set"+u"\u00ED"+"man", "sevatian", "sever",
"severian", "sevilhan", "sext", "sextadeciman",
"shakespearian", "siablesian", "siagitan", "sibaritan",
"siberian", "sican", "sicelitan", "sicilian",
"sicran", "sideromelan", "siderom"+u"\u00E9"+"lan", "sideropolitan",
"sidonian", "siegenian", "sifilid"+u"\u00F4"+"man", "sifil"+u"\u00ED"+"man",
"sifil"+u"\u00F4"+"man", "sifonian", "sigl"+u"\u00F4"+"man", "siguitan",
"s"+u"\u00ED"+"lab", "silan", "silbian", "silesian",
"silican", "silicobutan", "silicoetan", "silicometan",
"silicopentan", "silicopropan", "siloxan", "silurian",
"silvan", "silvian", "silvianopolitan", "simedian",
"simelian", "simian", "simonian", "simonsenian",
"simplician", "sincefalian", "sincer", "sincranian",
"sinemurian", "s"+u"\u00ED"+"nfan", "sinfisian", "singel",
"sinian", "sinistr", "sinistr"+u"\u00F4"+"man", "sinocarotidian",
"siracusan", "sirenian", "sis", "siscian",
"sissomian", "sisud", "sitian", "siti"+u"\u00F4"+"man",
"smithian", "smithsonian", "soberan", "sobrean",
"sobredit", "sobreir", "sobressan", "s"+u"\u00F3"+"bri",
"s"+u"\u00F3"+"ci", "socinian", "socioamerican", "sofoclian",
"sof"+u"\u00F4"+"man", "sogaban", "sogdian", "sogr",
"sol", "solan", "solarian", "sold",
"soletan", "solt", "solteir", "somalian",
"son"+u"\u00E2"+"mbul", "sonor", "sons", "sopeir",
"soran", "s"+u"\u00F3"+"rdid", "sorelian", "sorian",
"sorocaban", "sorraian", "sortud", "sorveteir",
"sosan", "sosian", "soteropolitan", "soturn",
"sousean", "sousian", "sovcozian", "spencerian",
"spenglerian", "spinozian", "stahlian", "stancarian",
"stendhalian", "stenonian", "sternian", "straussian",
"strindbergian", "suan", "suarezian", "suasan",
"subacetan", "subcoracodian", "suberan", "subertan",
"s"+u"\u00FA"+"bit", "sublapsarian", "subpericranian", "subsaharian",
"subsignan", "substitut", "subuman", "suburitan",
"suculent", "sudertan", "suec", "suelitan",
"suessan", "suessetan", "suessonian", "suessulan",
"sufetulan", "sufolquian", "su"+u"\u00ED"+u"\u00E7", "suicid"+u"\u00F4"+"man",
"suj", "sujeit", "sulan", "sulcitan",
"sulfatarian", "sulfonetilmetan", "sumatriptan", "sumenian",
"sumerian", "supost", "supramundan", "surd",
"surian", "surpres", "susan", "susanopolitan",
"suseran", "susian", "suspeit", "tabac"+u"\u00F4"+"man",
"taban", "tabatingan", "tabelian", "taberneir",
"tabordian", "tacian", "t"+u"\u00E1"+"cit", "tafian",
"tagan", "tagorian", "tainian", "tairetan",
"taitian", "talaveran", "talh", "talip"+u"\u00F4"+"man",
"tamacan", "tamanh", "tamarineir", "tamian",
"tamp", "tanat"+u"\u00F4"+"man", "tanetan", "tang",
"t"+u"\u00E2"+"ngan", "tangitan", "tanzanian", "tapetorretinian",
"tapiocan", "tapiratiban", "taproban", "tapsitan",
"taquarituban", "tardenosian", "tarituban", "tarsian",
"tarsofalangian", "tarsometatarsian", "tartessian", "tarvisan",
"tasian", "tasmanian", "tatuian", "taubatean",
"taunusian", "taurian", "taurisan", "tauromiritan",
"tavastian", "taxaspidian", "taylorian", "tchacavian",
"tchadian", "tean", "teban", "tebasan",
"tege"+u"\u00F3"+"cran", "tejan", "telemetacarpian", "telian",
"telurian", "temesan", "temnospondilian", "temperadinh",
"tenitan", "teodosian", "te"+u"\u00F4"+"man", "ter"+u"\u00E7",
"terceir", "terencian", "teresian", "teresopolitan",
"tergilan", "termidorian", "termopolitan", "tern",
"tertulian", "tesour", "tessalian", "testeir",
"testemunh", "tet", "tetraboran", "tetrabromoetan",
"tetracloralcan", "tetracloretan", "tetraclorometan", "tetracontan",
"tetracosan", "tetradecan", "tetraetilplumban", "tetrafenilfuran",
"tetraidrofuran", "tetraidropiran", "tetraiodoetan", "tetrametilometan",
"tetrametilosilan", "tetranitrometan", "tetrassilan", "teut"+u"\u00F4"+"man",
"teverian", "texan", "texian", "ti",
"tian", "tibaran", "tiberian", "tibetan",
"tibiocalcanean", "tibiotarsian", "tiburcian", "tietean",
"tijucan", "tilestonian", "timboteuan", "timbrian",
"timelician", "timicitan", "tin", "tindaritan",
"tingitan", "tint", "tintureir", "tiofan",
"tiofuran", "tiran", "tiran"+u"\u00F4"+"man", "tireoidian",
"tiroidian", "tirolean", "tironian", "tirrenian",
"t"+u"\u00ED"+"san", "tisdritan", "titan", "titi",
"titil"+u"\u00F4"+"man", "titonian", "tlandrian", "toan",
"toarcian", "tobaguian", "tobian",
"tocantinopolitan", "tocarian", "tol",
"tolan", "told", "toledan", "tolentinian",
"toletan", "tolosan", "tolstoian", "tom"+u"\u00F4"+"man",
"tomosvarian", "tongan", "tongrian", "tont",
"torcularian", "torguian", "torment", "torneir",
"torrean", "torrejan", "torrian", "tort",
"tortonian", "tosc", "toscan", "totan",
"tra"+u"\u00E7", "tractarian", "traducian", "tragan",
"tragian", "traianapolitan", "trai"+u"\u00E7"+"oeir", "tralian",
"tramontan", "tramp", "tranc", "trancosan",
"tranqueir", "tranquil", "transamerican", "transarian",
"transbaicalian", "transbrasilian", "transcranian", "transerran",
"transfretan", "transiberian", "transilvan", "transilvanian",
"transjuran", "transleitan", "transmundan", "transpadan",
"transtigritan", "transuranian", "transvaalian", "transvalian",
"trapezometacarpian", "traquean", "traquelian", "traqueoesofagian",
"traseir", "trasmontan", "trasserran", "trastagan",
"travess", "travesseir", "treban", "trebian",
"trebulan", "trecorian", "tr"+u"\u00EA"+"feg", "tremend",
"tr"+u"\u00EA"+"mul", "trentonian", "trepadeir", "tr"+u"\u00E9"+"pan",
"trespan", "trev", "trevisan", "triacontan",
"trian", "tribrometan", "tribromoetan", "tribromometan",
"tribromopropan", "tribun", "trican", "tricloroetan",
"triclorometan", "tricloronitrometan", "tricloropropan", "tric"+u"\u00F4"+"man",
"tricordian", "tricosan", "tricotonan", "tricuspidian",
"tridecan", "triduan", "tr"+u"\u00ED"+"fan", "trifenilalquilidenofosforan",
"trifenilmetan", "trifluorometan", "trigueir", "trilh",
"trimarian", "trimetafan", "trimetilmetan", "trinc",
"trinitarian", "trinitrometan", "trioctubrian", "trioxan",
"triplan", "tripolitan", "triptofan", "tript"+u"\u00F3"+"fan",
"trisormian", "trissilan", "trist"+u"\u00ED"+"man", "tritetracontan",
"tritonian", "triunfan", "tro"+u"\u00E7"+"ad", "trocan",
"trocanterian", "trocantinian", "troian", "tromboxan",
"trom"+u"\u00F4"+"man", "tropan", "troquinian", "troquiscan",
"troquiterian", "trunf", "tub", "tubian",
"tucitan", "tucuman", "tufican", "tujan",
"tulian", "tunapolitan", "tunetan", "tungomelan",
"tunisian", "tupian", "turanian", "turb",
"turbulent", "turcoman", "turdetan", "turduletan",
"turingian", "turnesian", "turonian", "turquestan",
"turv", "tuscan", "tusculan", "tuvaluan",
"tzigan", "uadaian", "ubanguian", "ubatuban",
"ucecian", "ucranian", "uf"+u"\u00F4"+"man", "ugrian",
"ulan", "ulfilan", "ulissiponian", "ulsterian", u"\u00FA"+"ltim",
"ultrarrepublican", "umbilican", "umbrian", "umbr"+u"\u00ED"+"man",
"umbr"+u"\u00F4"+"man", "umerolecranian", "umeroolecranian", "unamunian",
"undecan", "und"+u"\u00E9"+"cim", "undeciman", "undecuman",
"undevicesiman", u"\u00FA"+"nic", "un"+u"\u00ED"+"man", "uquedan",
"uralian", "uran"+u"\u00F3"+"fan", "urartuan", "urban", "urs",
"ursinh", "uruguaian", "usalitan", "usidican",
"usineir", "usuan", "uter"+u"\u00F4"+"man", "uterovarian",
"uxian", "uxorian", "uzalitan", "vacan",
"vacarian", "vacean", "vadi",
"vagabund", "valangian", "valanginian", "valencian",
"valentin", "valentinian", "valerian", "valesian",
"valinginian", "valisoletan", "valombrosian", "valquirian",
"valsorcian", "valtonian", "valverdian", "vampir",
"vandean", "varan", "varangian", "varatojan",
"varcian", "varejeir", "vargian", "variscan",
"variscanian", "varronian", "varsovian", "varuan",
"varzean", "varzian", "vasc", "vassal",
"vast", "vatelian", "vaterian", "vaterladian",
"vatican", "vauclusian", "vazi", "vealdian",
"vegetalian", "vegetalin", "vegetarian", "veientan",
"vejentan", "velazquian", "velh", "velhac",
"veloc"+u"\u00ED"+"man", "vemelian", "venafran", "vendean",
"venecian", "venerand", "venetulan", "venezian",
"venezolan", "venezuelan", "venloquian", "vent",
"ventaneir", "venusian", "veragran", "veranopolitan",
"verb"+u"\u00F4"+"man", "verdadeir", "verdian", "vergestan",
"vergilian", "verlainian", "vermelh", "vermelhinh",
"vermian", "vermidian", "verneian", "verrocchian",
"verulan", "verumontan", "vesalian", "vesan",
"vescelan", "vesgueir", "vespian", "vespuccian",
"vespucian", "vestinian", "vesulian", "vesuvian",
"veteran", "vianopolitan", "vibrionian", "vicencian",
"vicentopolitan", "vidian", "vieirian", "vietnamian",
"vigilancian", "vila"+u"\u00E7"+"oran", "vilanovian", "vilpian",
"vincian", "vind", "vindian", "vindobonian",
"violeir", "violent", "virchovian", "virgilian",
"virginian", "virginopolitan", "virglorian", "virgulian",
"virsungian", "virtonian", "visaian", "visanian",
"visean", "vistulian", "vitamin"+u"\u00F4"+"man", "vitelian",
"vitian", "vitizan", "vitorian", "vitorin",
"v"+u"\u00ED"+"tre", "vi"+u"\u00FA"+"v", "viventan", "vivian",
"vizinh", "volan", "volapuc"+u"\u00F4"+"man", "volapuquian",
"volaterran", "volcentan", "volgian", "volscian",
"voltairian", "vomerian", "voprinian", "vormian",
"voss", "vulcanian", "vulfilan", "vulfilian",
"vulpian", "wagnerian", "washingtonian", "weberian",
"wildian", "xabregan", "xantitan", "xifoidian",
"xinguan", "zabian", "zairian", "zambian",
"zamoran", "z"+u"\u00E2"+"ngan", "zeugitan", "zimbabuan",
"zombeteir", "zoroastrian", "zorr", "zuinglian",
"zwinglian"])


# plvrInva - palavras de dois generos terminadas em -a- 
plvrInva = set(["acrobata", "ala", "alfa", "artista",
"atalaia", "atleta", "autarca", "bacana", "bandeirinha", "belga",
"besta", "beta", "cabe"+u"\u00E7"+"a", "cabra",
"cabrinha", "caixa", "camarada", "canalha", "capixaba",
"cara", "careca", "carioca", "cestinha", "cineasta", "cinza", "colega",
"crista", "democrata", "desavergonhada", "d"+u"\u00E9"+"spota",
"digna", "diplomata", "dourada", "extra",
"faixa", "fantasma", "ginasta", "gorila", "grama",
"grama", "guarda", "guia", "hip"+u"\u00F3"+"crita",
"idiota",  "id"+u"\u00F3"+"latra",
"impala", "ind"+u"\u00ED"+"gena", "infanticida", "japona", "java",
"jeca", "judoca", "laranja", "laranja",
"lhama", "longa", "magnata", "maia",
"maiorca", "maneta", "meia", "metropolita",
"monarca", "musicoterapeuta", "obstetra", "oligarca",
"ordenan"+u"\u00E7"+"a", "palerma", "paquera", "para"+u"\u00ED"+"ba",
"parasita", "parasita", "pateta", "patola",
"patriota", "pediatra", "penetra", "peralta",
"perneta", "picareta", "pilantra",
"pindoba", "pirata", "plutocrata", "poliglota",
"ponta", "porra", "prata", "primata",
"psicopata", "psiquiatra", "quilombola", "recruta",
"reduzida", "reserva", "sacana",
"salvaguarda", "salvaterra", "sanguessuga", "santista",
"sarcasta", "saudosa", "seguran"+u"\u00E7"+"a", "sentinela",
"sentinela", "silv"+u"\u00ED"+"cola", "s"+u"\u00F3"+"sia", "suicida",
"suicida", "suicida", "sulamita", "suprema",
"tagarela", "tanta", "taramela", "tecnocrata",
"terapeuta", "terapeuta", "triatleta", "troglodita",
"trouxa", "vigia", "violeta"])


# plvrMfr - palavras de dois generos terminadas em -r- 
plvrMfr = set(["agricolar", "agroalimentar", "algar", "anterior", "antirradar",
"antissolar", "arquissecular", "arquivulgar", "atriovascular",
"atrioventricular", "auricular", "auriculoventricular", "autossimilar",
"auxiliar", "axilar", "bacelar", "bicolor", "biliar", "bilinear",
"bipolar", "bissecular", "capilar", "capitular", "cardiopulmonar",
"cardiovascular", "cavalar", "caviar", "cedular",
"cefaloauricular", "celular", "chofer", "circular", "col"+u"\u00ED"+"der",
"complementar", "consular", "curricular", "designer", "dipolar", "disciplinar",
"discipular", "diverticular", "elementar", "epistolar", "escolar",
"espetacular", "estelar", "exemplar", "extracurricular", "exterior",
"familiar", "halter", u"\u00ED"+"mpar", "incolor", "inferior", "interior", "irregular",
"l"+u"\u00ED"+"der", "lunar", "maior", "Major", "major", "malmequer",
"m"+u"\u00E1"+"rtir", "medular", "melhor", "menor", "milhar",
"militar", "Mor", "mor", "muladar", "multicapsular",
"multicarpelar", "multidisciplinar", "multifamiliar", "multifilar",
"multilinear", "multilobular", "multilocular", "multimilenar",
"multinodular", "multinucelar", "multinuclear", "multiocular",
"muscular","nuclear", "ocular", "par", "paramilitar", "parlamentar",
"particular", "pelicular", "peninsular", "perpendicular", "pior", "polar",
"pomar", "popular", "posterior", "preliminar", "prostibular",
"protocolar","protossolar", "pulmonar", "quadrangular", "quatrissecular",
"regular", "rep"+u"\u00F3"+"rter", "retangular", "reticular",
"retilinear", "salutar", "secular", "semiangular",
"semianular", "semicircular", "semipopular", "semirregular",
"semissecular", "similar", "singular", "solar",
"subcircular", "subintervalar", "subliminar", "submandibular",
"superior", "testicular", "titular",
"transcelular", "transecular", "tricolor", "tripolar", "trissecular",
"tubercular", "tubular", "tutelar", "ultrassolar",
"uniangular", "unianular", "uniarticular", "unicapsular",
"unicarpelar", "unicelular", "uniciliar", "unifamiliar",
"unifilar", "uniglandular", "unilaminar", "unilinear",
"unilobar", "unilobular", "unimolecular", "unimuscular",
"uninuclear", "uninucleolar", "uniocular", "uniovular",
"unipolar", "unitegumentar", "univalvular", "univesicular",
"univocabular", "urceolar", "uterolombar", "utricular",
"uvular", "vaginovulvar", "valecular", "valspodar",
"valvar", "valvular", "variolar", "vascular",
"ventricular", "verticilar", "vesicular", "vestibular",
"vocabular", "volar", "vulgar", "vulvar",
"zanzibar"])

# plvrFemr - palavras femininas terminadas em -r- 
plvrFemr = set(["colher", "liminar", "mulher", "multimulher",
"supermulher"])

# plvrMascr - palavras masculinas terminadas em -r- 
plvrMascr = set(["abajur", "a"+u"\u00E7"+"or", "a"+u"\u00E7"+u"\u00FA"+"car", "afazer",
"afric"+u"\u00E2"+"nder", "alamar", "albor", "algar",
"alguidar", "altar", "alter", "alvor",
"amanhecer", "amargor", "amor", "andor",
"ar", "ardor", "ator", "avatar",
"azar", "bar", "bastidor", "bazar",
"bilhar", "biopoder", "bolor", "cad"+u"\u00E1"+"ver",
"calcanhar", "calor", "c"+u"\u00E2"+"ncer", "capacitor",
"capitular", "car"+u"\u00E1"+"cter", "car"+u"\u00E1"+"ter", "castor",
"casuar", "cateter", "celular", "c"+u"\u00E9"+"sar",
"chanceler", "clamor", "clangor", "cobertor",
"cocar", "colar", "cont"+u"\u00EA"+"iner", "contrapoder",
"cultor", "czar", "desamor",
"despertar", "desprazer", "dissabor", "d"+u"\u00F3"+"lar",
"emir", "equador", "esc"+u"\u00E2"+"ner", "esf"+u"\u00ED"+"ncter",
"esgar", "espaldar", "esplendor", "estupor",
"eurod"+u"\u00F3"+"lar", "familiar", "faquir", "fator",
"favor", "fedor", "f"+u"\u00E9"+"mur", "f"+u"\u00EA"+"mur",
"fervor", "flexor", "fl"+u"\u00FA"+"or", "folar",
"folgar", "fragor", "freezer", "fulgor",
"fulmar", "funicular", "furor", "glaciar",
"hangar", "horror", "hospitalar", "humor",
"insular", "jaguar", "jantar", "labor",
"lagar", "langor", "lar", "lavor",
"lazer", "l"+u"\u00EA"+"mur", "licor", "limiar",
"logar", "louvor", "luar", "lugar",
"luminar", "lupanar", "magiar", "malar",
"manjar", "mar", "m"+u"\u00E1"+"rtir", "maxilar",
"menir", "mester", "microcomputador", "minicomputador",
"mister", "molar", "monsenhor", "muar",
"n"+u"\u00E9"+"ctar", "nen"+u"\u00FA"+"far", "odor", "olhar",
u"\u00F4"+"mer", u"\u00F3"+"scar", "paladar", "palmar",
"palor", "par", "patamar",
"pavor", "pendor", "pendular", "penhor",
"pilar", "placar", "polegar",
"pomar", "pormenor", "p"+u"\u00F4"+"ster", "prazer",
"primor", "prior", "pudor", "pundonor",
"quasar", "quefazer", "radar", "rancor",
"reator", "rev"+u"\u00F3"+"lver", "rigor", "rubor",
"rumor", "sabor", "setor", "silhar",
"souvenir", "subpoder", "su"+u"\u00E9"+"ter", "suor",
"supercomputador", "superpoder", "superser", "suvenir",
"talher", "tambor", "tear", "teleprompter",
"televisor", "temor", "tenor", "teor",
"terror", "timor", "t"+u"\u00F4"+"ner", "torpor",
"transistor", "transp"+u"\u00F4"+"nder", "trator", "travor",
"tremor", "tu"+u"\u00ED"+"ter", "tumor", "ureter",
"valor", "vapor", "vedor",
"vetiver", "viajor", "vigor", "vizir",
"z"+u"\u00ED"+"per"])

# plvrMfal - palavras masculinas/femininas terminadas em -al- 
plvrMfal = set(["acidenta", "adiciona", "adverbia", "agricultura",
"algodoa", "amara", "ambulatoria", "amia",
"ancestra", "angelica", "anticoncepciona", "antimonia",
"anua", "area", "arena", "argumenta",
"armoria", "aroma", "arquitetura", "astra",
"atona", "atua", "audiovisua", "avena",
"azinha", "bagua", "bambua", "bamburra",
"bana", "barroca", "basa", "bestia", "bica", "Biena", "biena",
"boa", "bo"+u"\u00E7"+"a", "borea", "bra"+u"\u00E7"+"a",
"braga", "braquia", "breja", "bria",
"bruma", "bruta", "buca", "bu"+u"\u00E7"+"a",
"caba", "cadeira", "caga"+u"\u00E7"+"a", "cama",
"camaraza", "caniba", "Capita", "capita",
"carna", "carrasca", "carrega", "casua",
"catarra", "catua", "caucha", "cauda",
"celestia", "cenda", "centea", "centra",
"cerimonia", "cipoa", "circunstancia", "clora",
"colegia", "colonia", "coloquia", "colossa",
"comensa", "comercia", "comportamenta", "conceitua",
"condiciona", "confessiona", "conjuga", "continenta",
"convenciona", "cora", "cordia", "corna",
"corpora", "couva", "cova", "credencia", "crimina",
"cua", "cuba", "cultura", "descomuna",
"desigua", "deslea", "diferencia", "digita",
"distrita", "diagona", "ditatoria", "divina", "documenta",
"dominica", "dorsa", "editoria", "emociona",
"empresaria", "erva"+u"\u00E7"+"a", "esparga", "especia",
"espinha", "espiritua", "essencia", "estadua", "estata",
"estenda", "esteva", "etana", "eterna",
"eucalipta", "eventua", "excepciona", "existencia",
"experimenta", "factoria", "faia", "fana",
"fata", "fatoria", "fava", "fecha",
"federa", "fenomena", "fera", "ferragia",
"filia", "fina", "financia", "firma", "fisca",
"flora", "fluvia", "fora", "forma",
"frasca", "frecha", "freixia", "fronta",
"fruga", "funcha", "fundamenta", "gaia",
"gavia", "genia", "genita", "gera",
"gestua", "glioxa", "globa", "graa",
"gradua", "grama", "gramatica", "gutura", "habitua",
"hexagona", "hipertextua", "historia", "homossexua",
"horizonta", "idea", "igua", "ilega",
"imateria", "imora", "imorta", "imparcia",
"imperia", "impessoa", "inaugura", "incidenta", "individua",
"industria", "inferna", "informa", "inicia",
"instrumenta", "integra", "intelectua", "intempora",
"intenciona", "internaciona", "interna", "irraciona",
"irrea", "jovia", "judicia", "juvena",
"latera", "lea", "lega", "lexica", "libera",
"liga", "litera", "loca", "loda"+u"\u00E7"+"a", "logica",
"lustra", "macambira", "macega", "madriga", "magistra",
"maiora", "manua", "marcia", "margina", "marita",
"mataga", "materna", "materia", "matina", "matorra",
"matricia", "medica", "medieva", "medronha", "mensa",
"menstrua", "menta", "mercuria", "meridiona", "mesca",
"metana", "milha", "minera", "missa",
"mistra", "moira", "monumenta", "mora",
"morta", "mundia", "municipa", "mura", "multinaciona",
"musica", "naciona", "nata", "natura",
"neanderta", "neolibera", "nepa", "nomina",
"norma", "numera", "nupcia", "occipita",
"ocidenta", "oficia", "ogiva", "oliva", "orienta",
"ora", "orbita", "origina", "ova", "oven"+u"\u00E7"+"a", "pacova",
"pactua", "pasti"+u"\u00E7"+"a", "pastora", "paterna", "patriarca",
"peda", "pedesta", "peitora", "pentagona",
"pentobarbita", "pessoa", "pinha", "plura",
"poia", "policia", "pontua", "porcentua",
"porta", "posta", "potencia", "prejudicia", "presidencia",
"primordia", "principa", "profissiona", "proporciona",
"proven"+u"\u00E7"+"a", "proverbia", "providencia", "provincia",
"provisiona", "qua", "raciona", "radica",
"rea", "recita", "referencia", "rega",
"regiona", "representaciona", "residencia", "reviva",
"ria", "ritua", "riva", "rosa",
"roseira", "rotaciona", "rura", "sabuga",
"sacramenta", "sapa", "sar"+u"\u00E7"+"a", "sarrafa"+u"\u00E7"+"a",
"saturna", "semana", "semestra", "semifina", "senda",
"senega", "sensaciona", "sensua", "sentimenta",
"sercia", "seria", "seringa", "servi"+u"\u00E7"+"a",
"sexua", "sia", "sidera", "sindica",
"sisa", "situaciona", "sobrenatura", "socia",
"substancia", "sucursa", "superpolicia", "supraespinha",
"surrea", "taboca", "taipa", "ta", "teatra",
"temperamenta", "tempora", "tenda", "termina",
"terra", "tetragona", "tiopenta", "torrencia",
"tota", "tradiciona", "tramaga", "transnaciona", "transexua",
"transversa", "tremeda", "triba", "triga",
"trigona", "triunfa", "trivia", "tropica",
"tutoria", "umbra", "universa", "ura",
"urgueira", "urogenita", "usua", "va",
"vegeta", "vena", "verba", "verdea",
"vertica", "virgina", "virtua", "viscera",
"visua", "vitra", "voca", "voga",
"vorta"])

# plvrMascal - palavras masculinas terminadas em -al- 
plvrMascal = set(["anima", "arraia", "arroza", "arsena",
"ava", "aventa", "banana", "beira",
"boca", "borna", "cabeda", "cacaua",
"cadava", "cafeza", "cajueira", "camba",
"cana", "canavia", "candea", "capinza",
"carda", "cardea", "cardia", "Carnava",
"carnava", "carvalha", "casa", "castanha",
"casti"+u"\u00E7"+"a", "cerea", "cereja", "chaca",
"chaparra", "chavasca", "choupa", "colmea",
"coqueira", "crista", "curra", "deda",
"Edita", "edita", "enxova", "escoria",
"feijoa", "fenobarbita", "festiva", "feuda",
"figueira", "funera", "genera", "hospita",
"jogra", "jorna", "lama"+u"\u00E7"+"a", "laranja",
"litora", "loca", "manancia", "manantia",
"manca", "mandioca", "mangua", "mangueza",
"marecha", "materia", "memoria",
"meta", "milhara", "milheira", "palha",
"palmita", "pana", "pantana", "parda",
"parieta", "pascoa", "percentua", "pinheira",
"pomba", "ponta", "pontifica", "punha",
"quinta", "raba"+u"\u00E7"+"a", "rama", "sa",
"salgueira", "seixa", "sina", "sobra", "subtota", "telejorna",
"Tribuna", "tribuna", "vara", "vendava"])

# excPlvral - excecoes palavras terminadas em -ais- Exemplo: cais, demais, jamais, mais 
excPlvral = set(["c", "dem", "jam", "m"])


# plvrMascel - palavras masculinas terminadas em -el- 
plvrMascel = set(["broqu", "bur", "capit", "carret",
"carross", "cart", "cinz", "coquet",
"corc", "cord", "coron", "coud",
"decib", "dies", "donz",
"doss", "espinh", "farn", "f",
"fun", "furri", "g", "hidrom",
"hot", "lamb", "laur", "man",
"manu", "m", "menestr", "migu",
"moscat", "mot", "neb", "ourop",
"pain", "pap", "parc", "past",
"penafi", "pich", "pinc", "pinh",
"pix", "plant", "quart", "rap",
"rang", "redond", "rodoan", "saquit",
"sarapat", "ton", "trop", "verg",
"xair", "xar"])

# plvrAmascel - palavras Acentuadas masculinas terminadas em -el- 
plvrAmascel = set(["arr"+u"\u00E1"+"te", "b"+u"\u00E9"+"te", "corom"+u"\u00E2"+"nde",
"cromon"+u"\u00ED"+"que", "c"+u"\u00FA"+"me", "cupron"+u"\u00ED"+"que",
"d"+u"\u00ED"+"se", "esn"+u"\u00F3"+"rque", "f"+u"\u00E1"+"rte", "ferron"+u"\u00ED"+"que",
"f"+u"\u00FA"+"se", "galvanon"+u"\u00ED"+"que", "gu"+u"\u00ED"+"me", "l"+u"\u00ED"+"nge",
"misp"+u"\u00ED"+"que", "r"+u"\u00ED"+"ge", "r"+u"\u00ED"+"me", "rod"+u"\u00F4"+"me",
"sa"+u"\u00ED"+"me", "s"+u"\u00ED"+"me", "t"+u"\u00FA"+"ne"])

# excPlvrel - excecoes palavras terminadas em -eis- Exemplo: anegalheis, reis (com acenteo no e) 
excPlvrel = set(["anegalh", "el", "fart", "r"])


# plvrMfel - palavras de dois generos terminadas em -el- 
plvrMfel = set(["cru", "fi", "gosp", "infi", "isab", "isra", "nob", "no"])


# plvrFemel - palavras femininas terminadas em -el- 
plvrFemel = set(["cascav"])

# excplMascu - exc. preposicao A - palavras masculinas terminadas em -u- - Ex.: A meu, A seu, A teu
excplMascu = set(["me", "se", "te"])

# plvrMascu - palavras masculinas terminadas em -u- 
plvrMascu = set(["abane", "abante", "abapor", "ab",
"acap", "acar", "aceste", "ac",
"aganipe", "agele", "agre", "ain",
"alifire", "alise", "amade", "amate",
"amebe", "amicle", "amorre", "anamburuc",
"anance", "ance", "androce", "ang",
"Ante", "antisse", "an", "apame",
"apoge", "aque", "aquile", "aracnoide",
"arame", "ara", "ardie", "arinfe",
"Ariste", "aritenoide", "armoze", "arque",
"arvade", "asaque", "asclepiade", "ascre",
"asdode", "asmode", "asmone", "asquelone",
"asside", "atamante", "atene", "ate",
"atiade", "atiroide", "at", "aule",
"baba"+u"\u00E7", "baba", "bab", "bacalha",
"bacura", "bacurub", "bagua"+u"\u00E7", "baiac",
"bail"+u"\u00E9", "baj", "balandra", "bali",
"bamb", "bang", "barce", "barde",
"bartolome", "bar", "barzab", "Basile",
"ba", "baur", "beij", "bela",
"belel"+u"\u00E9", "belope", "berimba", "betsile",
"b"+u"\u00E9", "bij", "bisna", "biz",
"bla", "boigua"+u"\u00E7", "bol"+u"\u00E9", "Borrome",
"botar"+u"\u00E9", "botie", "brace", "braquicore",
"bresla", "bre", "briare", "briar"+u"\u00E9",
"bronte", "br", "bub", "burea",
"bur", "caagua"+u"\u00E7", "cabu"+u"\u00E7", "cacar"+u"\u00E9",
"caca", "cadme", "cadmone", "caduce",
"caduve", "cafarne", "cafu"+u"\u00E7", "cagua"+u"\u00E7",
"caitit", "caj", "cajur", "cala",
"calde", "calha", "cal", "camafe",
"camar", "camba", "canaane", "canane",
"cancle", "cande", "candir", "cangu"+u"\u00E7",
"cangur", "canoide", "canope", "carac",
"caramur", "carapa", "carapic", "carib",
"carne", "car", "caruar", "carur",
"cassiope", "catagua"+u"\u00E7", "catarre", "catata",
"catech", "cauch", "ceratoioide", "c"+u"\u00E9",
"chab"+u"\u00E9", "chande", "chap"+u"\u00E9", "char"+u"\u00E9",
"charte", "cha", "chibe", "chichisb"+u"\u00E9",
"chim", "chir", "ch", "chuch",
"churre", "ciadopitioide", "cifioide", "cilace",
"cilene", "cile", "circe", "cirene",
"Cirine", "cirsoide", "citere", "cite",
"citofaringe", "cla", "clidocoracoide", "clidomastoide",
"codie", "coip", "colipe", "colise",
"colone", "colora", "conope", "corcire",
"core", "corife", "coroide", "corone",
"coronoide", "cor", "coruch"+u"\u00E9", "cossie",
"costocoracoide", "costocoracoroide", "costoxifoide", "cotiloide",
"crajur", "cretche", "cricaritenoide", "cricoaritenoide",
"cricoide", "cricotireoide", "cricotiroide", "criptojude",
"cr", "c", "cuc", "culde",
"cunauar", "cupua"+u"\u00E7", "cura"+u"\u00E7"+"a", "cura",
"curur", "curuz", "dacriope", "degra",
"dia", "ec", "embiru"+u"\u00E7", "em",
"enx", "eorde", "epicure", "epidafne",
"epidaure", "epige", "epistrofe", "eque",
"erimante", "erine", "erite", "eritre",
"eritrofle", "er", "escarabe", "escarc"+u"\u00E9",
"esminte", "esmirne", "espale", "espele",
"espermatope", "esponde", "estilioide", "estiloide",
"estomode", "estratonice", "estubere", "ete",
"etope", "eube", "euripide", "europe",
"eve", "faetonte", "falane", "falante",
"falasarne", "fane", "fare", "farise",
"farrumpe", "febe", "feloide", "fere",
"fereze", "fich", "filace", "filiste",
"filobasile", "firmide", "flie", "floresce",
"foce", "fogar"+u"\u00E9", "frise", "frixe",
"gabir", "gaib"+u"\u00E9", "galile", "gate",
"ga", "gaze", "gebade", "gergese",
"gete", "gigante", "gimnobelide", "ginece",
"gire", "girgase", "girre", "gite",
"glenoide", "gogiare", "goiam", "gomorre",
"gordie", "grabe", "grage", "gra",
"gregue", "gris"+u"\u00E9", "gris", "guabir",
"guaicur", "guarib", "guatamb", "gua",
"guigrateute", "gur", "hamate", "hamire",
"hebre", "hecate", "hecatombe", "hemoperitone",
"heracle", "heraclite", "herie", "herme",
"hermione", "hesiode", "hete", "he",
"heve", "hialoide", "hible", "hierope",
"hilocere", "himene", "hioide", "hiotiroide",
"hipate", "hipofle", "hipoge", "hore",
"hut", "iatre", "ic", "idume",
"igara"+u"\u00E7", "igua"+u"\u00E7", "ilh"+u"\u00E9", "il",
"imb", "imp", "inamb", "incr"+u"\u00E9",
"ingranz"+u"\u00E9", "inhamb", "iola", "iridocoroide",
"isse", "ita", "itone", "iture",
"jabir", "jabur", "jacobe", "jac",
"jacurut", "jaguaru"+u"\u00E7", "jamb", "jap",
"ja", "jaze", "jebuse", "jenisse",
"jes", "jinze", "jira", "jube",
"jubile", "jude", "juj", "jungi",
"jurur", "justiniane", "lab"+u"\u00E9", "lacra",
"lala", "lamb", "landa", "landr",
"lapite", "lara", "larisse", "lebr"+u"\u00E9",
"lene", "lepidoste", "lerne", "lete",
"l"+u"\u00E9", "lice", "licre", "lige",
"liofloe", "lipare", "loc", "lorditome",
"lua", "lul", "lumar"+u"\u00E9", "macabe",
"macac", "macar"+u"\u00E9", "maca", "mac",
"madiane", "majaque", "maj", "malacoste",
"mal", "mana", "manch", "mandacar",
"mande", "mang", "manib",
"manique", "mani", "mant"+u"\u00E9", "man",
"manumiti", "marab", "maracat", "mara",
"marimba", "marip", "marone", "maruce",
"masde", "mase", "masore", "massile",
"mastar"+u"\u00E9", "mastoide", "matate", "mat",
"ma", "mausol"+u"\u00E9", "maze", "meduse",
"melibe", "melite", "menandre", "mene",
"menipe", "men", "mesenquitre", "mesofle",
"metimne", "me", "meze", "mia",
"micene", "mide", "miiofone", "milase",
"mile", "milhare", "milhere", "milhe",
"milre", "mindana", "mine", "minga",
"ming", "mitilene", "mitre", "mit",
"mize", "mocama", "molung", "monte",
"mor", "m", "muie", "mule",
"mulung", "mund"+u"\u00E9", "munduruc", "mung",
"murna", "mur", "murucutut", "murur",
"muse", "mutusce", "nabate", "nadaba",
"nal", "namb", "nape", "nasore",
"natabe", "natine", "naumaque", "nazare",
"nazire", "nazore", "nda", "nemee",
"neme", "nese", "nhamb", "nicola",
"ninfe", "nise", "n", "nube",
"odontoide", "omb", "omoioide", "omumbe",
"ore", "orse", "otrione", "ot",
"oxe", "ozue", "pacat", "paca",
"pac", "pade", "page", "palene",
"panac", "panage", "pange", "panque",
"pante", "panticape", "papanicola", "pap",
"paquifle", "parafle", "paragua"+u"\u00E7", "parale",
"parda", "parore", "partenope", "par",
"patare", "pata", "pa", "peg",
"pelebre", "pelene", "pele", "pelope",
"pelorne", "pera", "pergame", "perge",
"peridere", "perige", "perine", "peritone",
"perize", "perone", "Perse", "per",
"petre", "peucale", "peucane", "pia",
"pidne", "pige", "pigme", "pimple",
"pin"+u"\u00E9", "pint"+u"\u00E9", "piracat", "piramb",
"piraruc", "pirene", "pire", "pise",
"pissele", "pitane", "pit"+u"\u00E9", "pit",
"pitua"+u"\u00E7", "platea", "plebe", "plesiopene",
"pne", "pneumoperitone", "policne", "polidesmoide",
"Pompe", "pontine", "porne", "pot",
"povar"+u"\u00E9", "povil"+u"\u00E9", "povol"+u"\u00E9", "prestiloide",
"priape", "pritane", "procoracoide", "propile",
"proscarabe", "proteocefaloide", "prote", "pr",
"psamoscafe", "pseudofle", "psi", "pterigoide",
"pure", "purupur", "putumuj", "quante",
"quarta", "quatipur", "quene", "queneze",
"quiloperitone", "quina", "quincaj", "quinguing",
"quixexe", "radame", "ragoide", "raguse",
"rame", "ra", "reb", "recr",
"retroestiloide", "retromastoide", "r"+u"\u00E9", "rife",
"ripe", "rodope", "rome", "sabe",
"saduce", "sag", "sal", "same",
"sande", "sanha"+u"\u00E7", "sapule", "saquit"+u"\u00E9",
"sara", "sasse", "satarque", "semin",
"sens", "septogle", "serape", "sesamoide",
"se", "sicurij", "sige", "sigmoide",
"siloge", "sine", "sinimb", "sit",
"sobe", "sobrec"+u"\u00E9", "sobrec", "sola",
"solid"+u"\u00E9", "sotade", "sov"+u"\u00E9", "subatloide",
"subaxoide", "sucurij", "sucuruj", "surucuc",
"surur", "suss", "tabar"+u"\u00E9", "tab",
"tacur", "Tade", "taja"+u"\u00E7", "tange",
"tapir", "tapur", "taquara"+u"\u00E7", "taquaru"+u"\u00E7",
"tata", "tat", "ta", "taumatope",
"tcha", "tebe", "tegee", "terapne",
"tere", "terine", "tespie", "tetartope",
"tet"+u"\u00E9", "te", "tiane", "tife",
"tifoide", "timane", "timb", "timele",
"tinfe", "tip", "tireoide", "tirisse",
"tiroide", "tof", "tol", "tomabe",
"torone", "tor", "tre", "trocoide",
"trof"+u"\u00E9", "trogofle", "troque", "troussea",
"tucana"+u"\u00E7", "tur", "turu"+u"\u00E7", "tut",
"ua", "ub", "ubu"+u"\u00E7", "uira"+u"\u00E7",
"uirapur", "ulisse", "umb", "upene",
"ur", "urub", "uruc", "uruta",
"urut", "uscarde", "vace", "vade",
"vagana", "varapa", "varde", "vasse",
"va", "v"+u"\u00E9", "vile", "vit",
"xamp", "xelel"+u"\u00E9", "xere", "xex"+u"\u00E9",
"xilre", "zaibats", "zancle", "za",
"zemare"])



# plvrMfeu - palavras de dois generos terminadas em -u- 
plvrMfeu = set(["amad", "bid", "calund", "coand", "cuand", "cud", "hind",
"land", "lund", "mandch", "mand", "murund", "nand", "nhand", "urd",
"vod", "vud", "zeb", "zul"])


# plvrFemu - palavras femininas terminadas em -u- 
plvrFemu = set(["bant", "na"])




# plvrMfcao - palavras dois generos terminadas em -cao- cedilha til a o 
plvrMfcao = set(["lota"+u"\u00E7"])

# plvrMascao - palavras masculinas terminadas em -cao- cedilha til a o 
plvrMascao = set(["ar"+u"\u00E7", "bal"+u"\u00E7",
"cabe"+u"\u00E7", "cal"+u"\u00E7",
"carro"+u"\u00E7", "Cora"+u"\u00E7",  "cora"+u"\u00E7",
"crian"+u"\u00E7", "escan"+u"\u00E7",
"infan"+u"\u00E7", "ti"+u"\u00E7",
"trope"+u"\u00E7"])

# excPlvrao - palavras terminadas em -ao- cujo plural eh diferente de -oes- 
excPlvrao = set(["afeg", "alde", "alem", "amont", "anci", "bang", "bordeg",
"borguinh", "c", "capel", "capit", "catal", "ch", "charlat", "cibercidad", "cidad", "coirm", "concidad",
"concrist", "crist", "de", "engatat", "ent", "escriv", "est", "estrab", "fo"+u"\u00E7", "fugi",
"garnach", "geres", "goz", "gr", "guardi", "h",  "hortel", "ignorant",
"irm", u"\u00F3"+"rf", u"\u00F3"+"rg", u"\u00D5"+"rg", "p", "pag", "pardal", "repont",
"s", "sen", "t", "tabeli", "tintur", "zombeir", "zorn", "zorr"])

# excPlvrnasce - excecoes palavras masculinas terminadas em -E- Exemplo: convite 
excPlvrnasce = set(["convit", "toqu"])


# plvrNfemcao - palavras nao femininas terminadas em -cao- cedilha til a o 
plvrNfemcao = set(["ar"+u"\u00E7", "bal"+u"\u00E7",
"cabe"+u"\u00E7", "cal"+u"\u00E7",
"carro"+u"\u00E7", "Cora"+u"\u00E7",  "cora"+u"\u00E7",
"crian"+u"\u00E7", "escan"+u"\u00E7",
"infan"+u"\u00E7", "lota"+u"\u00E7",
"ti"+u"\u00E7",
"trope"+u"\u00E7"])

#plvrFemao  - Palavras femininas terminadas em -ao-
plvrFemao = set(["acess", "Acess",
"acupress", "Acupress", "ades", "Ades",
"admiss", "Admiss", "adust", "Adust",
"agress", "Agress", "alus", "Alus",
"amarelid", "Amarelid", "amplid", "Amplid",
"animadvers", "Animadvers", "antevis", "Antevis",
"apreens", "Apreens", "aptid", "Aptid",
"ascens", "Ascens", "aspers", "Aspers",
"autogest", "Autogest", "avers", "Avers",
"bernard", "Bernard", "bioimpress", "Bioimpress",
"branquid", "Branquid",
"brusquid", "Brusquid", "certid", "Certid",
"cess", "Cess", "circuncis", "Circuncis",
"cis", "Cis", "coaliz", "Coaliz",
"coes", "Coes", "colis", "Colis",
"combust", "Combust", "comich", "Comich",
"comiss", "Comiss", "compaix", "Compaix",
"compreens", "Compreens", "compress", "Compress",
"compuls", "Compuls", "comunh", "Comunh",
"concess", "Concess", "concis", "Concis",
"conclus", "Conclus", "concuss", "Concuss",
"conex", "Conex", "confiss", "Confiss",
"confus", "Confus", "congest", "Congest",
"contram", "contrarraz", "Contrarraz",
"contus", "Contus", "convers", "Convers",
"convex", "Convex", "convuls", "Convuls",
"corras", "Corras",
"corros", "Corros", "cosmovis", "Cosmovis",
"crucifix", "Crucifix", "Cubat", "d",
"D", "decis", "Decis", "defens",
"Defens", "deflex", "Deflex", "dem", "Dem", "demiss", "Demiss",
"densid", "Densid", "depress",
"Depress", "deris", "Deris", "descens",
"Descens", "descompress", "Descompress", "desconex",
"Desconex", "desilus", "Desilus",
"despretens", "Despretens", "desuni",
"Desuni", "devassid",
"Devassid", "difus", "Difus", "digest",
"Digest", "digress", "Digress", "dimens",
"Dimens", "discuss", "Discuss", "dispers",
"Dispers", "dissuas", "Dissuas", "distens",
"Distens", "divers", "Divers", "divis",
"Divis",
"eclos", "Eclos", "efus",
"Efus", "egest", "Egest", "electrocuss",
"Electrocuss", "eletrofus", "Eletrofus", "emers",
"Emers", "emiss", "Emiss", "emuls",
"Emuls", "entrevis", "Entrevis", "eros",
"Eros", "escans", "Escans",
"escravid", "Escravid", "escrupulid",
"Escrupulid", "escurid", "Escurid", "espessid",
"Espessid", "evas", "Evas", "Ex-pris",
"exatid", "Exatid", "exaust", "Exaust",
"excis", "Excis", "exclus", "Exclus",
"excomunh", "Excomunh", "excurs", "Excurs",
"expans", "Expans", "explos", "Explos",
"express", "Express", "expuls", "Expuls",
"extens", "Extens", "extors", "Extors",
"extrovers", "Extrovers", "extrus", "Extrus",
"f", "F", "fiss", "Fiss",
"flex", "Flex",
"fotoemiss", "Fotoemiss", "fotofiss", "Fotofiss",
"Fraz", "fresquid",
"Fresquid", "frouxid", "Frouxid", "fus",
"Fus", "gest", "Gest", "gratid",
"Gratid", "hipertens", "Hipertens", "hipotens",
"Hipotens", "ilus", "Ilus", "imensid",
"Imensid", "imers", "Imers", "imiss",
"Imiss", "implos", "Implos", "imprecis",
"Imprecis", "impress", "Impress", "impuls",
"Impuls", "imunodepress", "Imunodepress", "inaptid",
"Inaptid", "incis", "Incis", "inclus",
"Inclus", "incompreens", "Incompreens", "incurs",
"Incurs", "indecis", "Indecis", "indigest",
"Indigest", "indivis", "Indivis", "inexatid",
"Inexatid", "inexpress", "Inexpress", "inflex",
"Inflex", "infus", "Infus", "ingest",
"Ingest", "ingratid", "Ingratid", "insubmiss",
"Insubmiss", "intens", "Intens", "intercess",
"Intercess", "interconex", "Interconex", "intromiss",
"Intromiss", "introvers", "Introvers", "intrus",
"Intrus", "invas", "Invas", "invers",
"Invers", "irreflex", "Irreflex", "irreligi",
"Irreligi", "irris", "Irris", "lassid",
"Lassid", "legi", "Legi", "lentid",
"Lentid", "les", "Les",
"mans", "Mans", "mansid",
"Mansid", "m", "M", "m"+u"\u00E3"+"os",
"M"+u"\u00E3"+"os", "miss", "Miss", "mornid",
"Mornid",
"Motos", "multid", "Multid", "negrid",
"Negrid", "obsess", "Obsess", "ocasi",
"Ocasi", "oclus", "Oclus", "omiss",
"Omiss", "opini", "Opini", "opress",
"Opress", "paix", "Paix", "pens",
"Pens", "percuss", "Percuss", "perdig",
"Perdig", "perfus", "Perfus", "permiss",
"Permiss", "perplex", "Perplex", "persuas",
"Persuas", "pervers", "Pervers", "podrid",
"Podrid", "possess", "Possess", "precess",
"Precess", "precis", "Precis", 
"Preclus", "preclus", "preens",
"Preens", "press", "Press", "pretens",
"Pretens", "pretid", "Pretid", "previs",
"Previs", "pris", "Pris", "prociss",
"Prociss", "profiss", "Profiss", "profus",
"Profus", "progress", "Progress", "prontid",
"Prontid", "propens", "Propens", "propuls",
"Propuls", "provis", "Provis", "puls",
"Puls", "quest", "Quest", "radiodifus",
"Radiodifus", "radiotelevis", "Radiotelevis", "ras",
"Ras", "raz", "Raz", "readmiss",
"Readmiss", "rebeli", "Rebeli", "recens",
"Recens", "recess", "Recess", "reclus",
"Reclus", "reconvers", "Reconvers", "reflex",
"Reflex", "regi", "Regi", "regress",
"Regress", "reimpress", "Reimpress", "reinvers",
"Reinvers", "religi", "Religi", "remiss",
"Remiss", "repercuss", "Repercuss", "repreens",
"Repreens", "repress", "Repress", "repuls",
"Repuls", "rescis", "Rescis", "retid",
"Retid", "retransmiss", "Retransmiss", "retrovers",
"Retrovers", "reuni", "Reuni", "revers",
"Revers", "revis", "Revis", "rez",
"Rez", "rouquid", "Rouquid", "sans", "saz",
"Saz", "secess", "Secess", "sequid",
"Sequid", "servid", "Servid", "sess",
"Sess", "sobrefus", "Sobrefus", "sofreguid",
"Sofreguid", "soid", "Soid", "solid",
"Solid", "soliflux", "Soliflux", "su",
"Su", "suas", "Suas", "subcomiss",
"Subcomiss", "subdivis", "Subdivis", "submers",
"Submers", "submiss", "Submiss", "subvers",
"Subvers", "sucess", "Sucess", "sugest",
"Sugest",  "superpress", "supervis", "Supervis", "supress",
"Supress", "suspens", "Suspens", "televis",
"Televis", "tens", "Tens", "transfus",
"Transfus", "transgress", "Transgress", "transmiss",
"Transmiss",
"uni", "Uni", "vaguid",
"Vaguid", "vastid", "Vastid", "vaz",
"Vaz", "vermelhid", "Vermelhid", "vers",
"Vers", "vis", "Vis"])

#excPlvrio  - Excecao palavras masculinas terminadas em ario - erio - irio - orio.
excPlvrio = set(["crit"+u"\u00E9"])


#plvrFmenta  - Palavras femininas terminadas em -menta-
plvrFmenta = set(["ciu", "curti", "e", "escara",
"farda", "ferra", "frege", "frigi", "hipono",
"impedi", "ju", "multiferra", "nanoferra",
"ossa", "pala", "pi", "superferra", "superpi",
"tor", "travesti", "urdi", "varri", "vesti"])


# plvrFemal - palavras femininas terminadas em -al- 
plvrFemal = set(["bacana", "catedra", "Catedra", "espira", "vesta"])


# aacpNos = Concordancia  com o pronome Nos (com acento)  
aacpNos = {"abandona": "abandonamos",
"abandonou": "abandonamos",
"abdica": "abdicamos",
"abdicasse": "abdic"+u"\u00E1"+"ssemos",
"abdicou": "abdicamos",
"abra"+u"\u00E7"+"a": "abra"+u"\u00E7"+"amos",
"abra"+u"\u00E7"+"ou": "abra"+u"\u00E7"+"amos",
"abre": "abrimos",
"abriu": "abrimos",
"acaba": "acabamos",
"acabou": "acabamos",
"aceita": "aceitamos",
"aceitaria": "aceitar"+u"\u00ED"+"amos",
"aceitou": "aceitamos",
"acha": "achamos",
"achava": "ach"+u"\u00E1"+"vamos",
"achou": "achamos",
"acolhe": "acolhemos",
"acompanha": "acompanhamos",
"acompanhou": "acompanhamos",
"acorda": "acordamos",
"acordou": "acordamos",
"acredita": "acreditamos",
"acreditava": "acredit"+u"\u00E1"+"vamos",
"acreditou": "acreditamos",
"acrescenta": "acrescentamos",
"acrescentou": "acrescentamos",
"adivinha": "adivinhamos",
"adivinhou": "adivinhamos",
"admira": "admiramos",
"admirou": "admiramos",
"admitir": "admitirmos",
"adora": "adoramos",
"adorou": "adoramos",
"adota": "adotamos",
"adote": "adotemos",
"adotou": "adotamos",
"adquire": "adquirimos",
"adquiriu": "adquirimos",
"adulterar": "adulterarmos",
"adverte": "advertimos",
"advertiu": "advertimos",
"advoga": "advogamos",
"advogou": "advogamos",
"age": "agimos",
"agiu": "agimos",
"agradece": "agradecemos",
"agradeceu": "agradecemos",
"ajuda": "ajudamos",
"ajudar"+u"\u00E1": "ajudaremos",
"ajudou": "ajudamos",
"ajunta": "ajuntamos",
"ajuntou": "ajuntamos",
"ajusta": "ajustamos",
"ajustou": "ajustamos",
"alcan"+u"\u00E7"+"ava": "alcan"+u"\u00E7"+u"\u00E1"+"vamos",
"alimenta": "alimentamos",
"alimentou": "alimentamos",
"aliviava": "alivi"+u"\u00E1"+"vamos",
"almo"+u"\u00E7"+"a": "almo"+u"\u00E7"+"amos",
"almo"+u"\u00E7"+"ou": "almo"+u"\u00E7"+"amos",
"alugasse": "alug"+u"\u00E1"+"ssemos",
"ama": "amamos",
"amarra": "amarramos",
"amarrou": "amarramos",
"amava": "am"+u"\u00E1"+"vamos",
"amou": "amamos",
"analisa": "analisamos",
"analisasse": "analis"+u"\u00E1"+"ssemos",
"analisou": "analisamos",
"anda": "andamos",
"andava": "and"+u"\u00E1"+"vamos",
"andou": "andamos",
"anuncia": "anunciamos",
"apanha": "apanhamos",
"apanhou": "apanhamos",
"apare"+u"\u00E7"+"a": "apare"+u"\u00E7"+"amos",
"aparece": "aparecemos",
"apareceu": "aparecemos",
"aplaude": "aplaudimos",
"aplaudiu": "aplaudimos",
"aplicaria": "aplicar"+u"\u00ED"+"amos",
"apodrece": "apodrecemos",
"apodreceu": "apodrecemos",
"apoia": "apoiamos",
"apoiou": "apoiamos",
"aprende": "aprendemos",
"aprendeu": "aprendemos",
"apresenta": "apresentamos",
"apresentar"+u"\u00E1": "apresentaremos",
"apresentou": "apresentamos",
"apresta": "aprestamos",
"aprestou": "aprestamos",
"aprova": "aprovamos",
"aprovasse": "aprov"+u"\u00E1"+"ssemos",
"aproveita": "aproveitamos",
"aprovou": "aprovamos",
"arranjar"+u"\u00E1": "arranjaremos",
"arranjasse": "arranj"+u"\u00E1"+"ssemos",
"arrasta": "arrastamos",
"arrastou": "arrastamos",
"arromba": "arrombamos",
"arrombou": "arrombamos",
"aspira": "aspiramos",
"aspirou": "aspiramos",
"assina": "assinamos",
"assinar": "assinarmos",
"assinou": "assinamos",
"assiste": "assistimos",
"assistir": "assistirmos",
"assistiu": "assistimos",
"associa": "associamos",
"assume": "assumimos",
"assumiu": "assumimos",
"ataca": "atacamos",
"atacou": "atacamos",
"atiraria": "atirar"+u"\u00ED"+"amos",
"atravessa": "atravessamos",
"atravessava": "atravess"+u"\u00E1"+"vamos",
"atravessou": "atravessamos",
"aumentar": "aumentarmos",
"avalia": "avaliamos",
"avaliar": "avaliarmos",
"avaliou": "avaliamos",
"avan"+u"\u00E7"+"a": "avan"+u"\u00E7"+"amos",
"avan"+u"\u00E7"+"ou": "avan"+u"\u00E7"+"amos",
"bate": "batemos",
"bateu": "batemos",
"batia": "bat"+u"\u00ED"+"amos",
"batiza": "batizamos",
"batizou": "batizamos",
"bebe": "bebemos",
"beneficia": "beneficiamos",
"beneficiou": "beneficiamos",
"bota": "botamos",
"botou": "botamos",
"brada": "bradamos",
"bradou": "bradamos",
"briga": "brigamos",
"brigou": "brigamos",
"brinca": "brincamos",
"brincava": "brinc"+u"\u00E1"+"vamos",
"brincou": "brincamos",
"busca": "buscamos",
"buscou": "buscamos",
"cabe": "cabemos",
"ca"+u"\u00E7"+"oa": "ca"+u"\u00E7"+"oamos",
"ca"+u"\u00E7"+"oou": "ca"+u"\u00E7"+"oamos",
"cai": "ca"+u"\u00ED"+"mos",
"cairia": "cair"+u"\u00ED"+"amos",
"ca"+u"\u00ED"+"sse": "ca"+u"\u00ED"+"ssemos",
"caiu": "ca"+u"\u00ED"+"mos",
"cala": "calamos",
"calcula": "calculamos",
"calculou": "calculamos",
"calou": "calamos",
"caminha": "caminhamos",
"caminhou": "caminhamos",
"cantar"+u"\u00E1": "cantaremos",
"carece": "carecemos",
"careceu": "carecemos",
"carrega": "carregamos",
"carregava": "carreg"+u"\u00E1"+"vamos",
"carregou": "carregamos",
"casa": "casamos",
"casou": "casamos",
"cavalga": "cavalgamos",
"cavalgou": "cavalgamos",
"celebra": "celebramos",
"celebrou": "celebramos",
"chama": "chamamos",
"chamava": "cham"+u"\u00E1"+"vamos",
"chamou": "chamamos",
"checa": "checamos",
"checou": "checamos",
"chega": "chegamos",
"chegar": "chegarmos",
"chegar"+u"\u00E1": "chegaremos",
"chegava": "cheg"+u"\u00E1"+"vamos",
"chegou": "chegamos",
"chegue": "cheguemos",
"chora": "choramos",
"chorava": "chor"+u"\u00E1"+"vamos",
"chore": "choremos",
"chorou": "choramos",
"cita": "citamos",
"citou": "citamos",
"cobra": "cobramos",
"cobrou": "cobramos",
"colabora": "colaboramos",
"colaborou": "colaboramos",
"coloca": "colocamos",
"colocou": "colocamos",
"combater"+u"\u00E1": "combateremos",
"combina": "combinamos",
"combinava": "combin"+u"\u00E1"+"vamos",
"combinou": "combinamos",
"come": "comemos",
"come"+u"\u00E7"+"a": "come"+u"\u00E7"+"amos",
"come"+u"\u00E7"+"ar": "come"+u"\u00E7"+"armos",
"come"+u"\u00E7"+"ou": "come"+u"\u00E7"+"amos",
"comenta": "comentamos",
"comentou": "comentamos",
"comer": "comermos",
"comete": "cometemos",
"cometeu": "cometemos",
"comeu": "comemos",
"comia": "com"+u"\u00ED"+"amos",
"compara": "comparamos",
"comparou": "comparamos",
"completa": "completamos",
"completou": "completamos",
"compra": "compramos",
"compraria": "comprar"+u"\u00ED"+"amos",
"comprava": "compr"+u"\u00E1"+"vamos",
"compreende": "compreendemos",
"compreendeu": "compreendemos",
"comprou": "compramos",
"comunga": "comungamos",
"comungou": "comungamos",
"concorda": "concordamos",
"concordar"+u"\u00E1": "concordaremos",
"concordou": "concordamos",
"concorria": "concorr"+u"\u00ED"+"amos",
"condena": "condenamos",
"condenou": "condenamos",
"confeccionaria": "confeccionar"+u"\u00ED"+"amos",
"confiava": "confi"+u"\u00E1"+"vamos",
"confirmar": "confirmarmos",
"congelaria": "congelar"+u"\u00ED"+"amos",
"conhece": "conhecemos",
"conheceu": "conhecemos",
"conhecia": "conhec"+u"\u00ED"+"amos",
"conjetura": "conjeturamos",
"conjeturou": "conjeturamos",
"conquista": "conquistamos",
"conquistou": "conquistamos",
"consegue": "conseguimos",
"conseguia": "consegu"+u"\u00ED"+"amos",
"conseguir": "conseguirmos",
"conseguir"+u"\u00E1": "conseguiremos",
"conseguiria": "conseguir"+u"\u00ED"+"amos",
"conseguiu": "conseguimos",
"conservar": "conservarmos",
"conserve": "conservemos",
"considera": "consideramos",
"considerar": "considerarmos",
"considerava": "consider"+u"\u00E1"+"vamos",
"considerou": "consideramos",
"consiga": "consigamos",
"constata": "constatamos",
"constatou": "constatamos",
"constr"+u"\u00F3"+"i": "constru"+u"\u00ED"+"mos",
"construiu": "constru"+u"\u00ED"+"mos",
"conta": "contamos",
"continua": "continuamos",
"continuar"+u"\u00E1": "continuaremos",
"continuava": "continu"+u"\u00E1"+"vamos",
"continuou": "continuamos",
"contou": "contamos",
"contratasse": "contrat"+u"\u00E1"+"ssemos",
"contratava": "contrat"+u"\u00E1"+"vamos",
"controla": "controlamos",
"controlou": "controlamos",
"conversa": "conversamos",
"conversar"+u"\u00E1": "conversaremos",
"conversava": "convers"+u"\u00E1"+"vamos",
"conversou": "conversamos",
"convida": "convidamos",
"convidou": "convidamos",
"coopera": "cooperamos",
"cooperou": "cooperamos",
"corre": "corremos",
"correr"+u"\u00E1": "correremos",
"correu": "corremos",
"costuma": "costumamos",
"costumava": "costum"+u"\u00E1"+"vamos",
"costumou": "costumamos",
"cota": "cotamos",
"cotou": "cotamos",
"cr"+u"\u00EA": "cremos",
"credencia": "credenciamos",
"credenciou": "credenciamos",
"cresce": "crescemos",
"cresceu": "crescemos",
"crescia": "cresc"+u"\u00ED"+"amos",
"creu": "cremos",
"cria": "criamos",
"criar": "criarmos",
"criava": "cri"+u"\u00E1"+"vamos",
"criou": "criamos",
"critica": "criticamos",
"criticou": "criticamos",
"cruza": "cruzamos",
"cuida": "cuidamos",
"cuidou": "cuidamos",
"culpa": "culpamos",
"culpou": "culpamos",
"cultiva": "cultivamos",
"cultivou": "cultivamos",
"cumpre": "cumprimos",
"cumpriu": "cumprimos",
"d"+u"\u00E1": "damos",
"dan"+u"\u00E7"+"a": "dan"+u"\u00E7"+"amos",
"dan"+u"\u00E7"+"asse": "dan"+u"\u00E7"+u"\u00E1"+"ssemos",
"dan"+u"\u00E7"+"ava": "dan"+u"\u00E7"+u"\u00E1"+"vamos",
"dava": "d"+u"\u00E1"+"vamos",
"debitar"+u"\u00E1": "debitaremos",
"decide": "decidimos",
"decidir"+u"\u00E1": "decidiremos",
"decidiu": "decidimos",
"defende": "defendemos",
"defendeu": "defendemos",
"defendia": "defend"+u"\u00ED"+"amos",
"define": "definimos",
"definiu": "definimos",
"deita": "deitamos",
"deitou": "deitamos",
"deixa": "deixamos",
"deixaria": "deixar"+u"\u00ED"+"amos",
"deixou": "deixamos",
"demonstra": "demonstramos",
"demonstrou": "demonstramos",
"denomina": "denominamos",
"denominou": "denominamos",
"denuncia": "denunciamos",
"denunciou": "denunciamos",
"deposita": "depositamos",
"depositou": "depositamos",
"derrubar": "derrubarmos",
"desce": "descemos",
"descende": "descendemos",
"desceu": "descemos",
"descobre": "descobrimos",
"descobrir": "descobrirmos",
"descobrir"+u"\u00E1": "descobriremos",
"descobriu": "descobrimos",
"desconfiava": "desconfi"+u"\u00E1"+"vamos",
"descoroa": "descoroamos",
"deseja": "desejamos",
"desejaria": "desejar"+u"\u00ED"+"amos",
"desejava": "desej"+u"\u00E1"+"vamos",
"desejou": "desejamos",
"desempenha": "desempenhamos",
"desempenhou": "desempenhamos",
"desentranha": "desentranhamos",
"desentranhou": "desentranhamos",
"desenvolve": "desenvolvemos",
"desenvolveu": "desenvolvemos",
"desenvolvia": "desenvolv"+u"\u00ED"+"amos",
"designa": "designamos",
"designou": "designamos",
"despachava": "despach"+u"\u00E1"+"vamos",
"despreza": "desprezamos",
"desprezou": "desprezamos",
"desrespeitar": "desrespeitarmos",
"desviava": "desvi"+u"\u00E1"+"vamos",
"detesta": "detestamos",
"detestou": "detestamos",
"deu": "demos",
"deva": "devamos",
"deve": "devemos",
"deveria": "dever"+u"\u00ED"+"amos",
"devia": "dev"+u"\u00ED"+"amos",
"diagnosticar"+u"\u00E1": "diagnosticaremos",
"diga": "digamos",
"dir"+u"\u00E1": "diremos",
"diria": "dir"+u"\u00ED"+"amos",
"discutia": "discut"+u"\u00ED"+"amos",
"disp"+u"\u00F5"+"e": "dispomos",
"dispor": "dispormos",
"disputa": "disputamos",
"disputou": "disputamos",
"disse": "dissemos",
"dissesse": "diss"+u"\u00E9"+"ssemos",
"divaga": "divagamos",
"divagou": "divagamos",
"diverge": "divergimos",
"divergiu": "divergimos",
"divide": "dividimos",
"dividia": "divid"+u"\u00ED"+"amos",
"dividiu": "dividimos",
"diz": "dizemos",
"dizia": "diz"+u"\u00ED"+"amos",
"dobra": "dobramos",
"dobrou": "dobramos",
"dormia": "dorm"+u"\u00ED"+"amos",
"dormir"+u"\u00E1": "dormiremos",
"dormiu": "dormimos",
"dubla": "dublamos",
"duplica": "duplicamos",
"duplicou": "duplicamos",
""+u"\u00E9": "somos",
"elabora": "elaboramos",
"elaborou": "elaboramos",
"elege": "elegemos",
"elegeu": "elegemos",
"embarga": "embargamos",
"embargou": "embargamos",
"embirra": "embirramos",
"embirrou": "embirramos",
"emprega": "empregamos",
"empregou": "empregamos",
"encara": "encaramos",
"encarou": "encaramos",
"enche": "enchemos",
"encheu": "enchemos",
"encolhia": "encolh"+u"\u00ED"+"amos",
"encontra": "encontramos",
"encontrar": "encontrarmos",
"encontrar"+u"\u00E1": "encontraremos",
"encontrou": "encontramos",
"enjeita": "enjeitamos",
"entende": "entendemos",
"entendesse": "entend"+u"\u00EA"+"ssemos",
"entra": "entramos",
"entrar": "entrarmos",
"entrava": "entr"+u"\u00E1"+"vamos",
"entrega": "entregamos",
"entregou": "entregamos",
"entrevista": "entrevistamos",
"entrevistou": "entrevistamos",
"entrou": "entramos",
"envia": "enviamos",
"enviaria": "enviar"+u"\u00ED"+"amos",
"enviou": "enviamos",
"enxuga": "enxugamos",
"enxugar"+u"\u00E1": "enxugaremos",
"enxugou": "enxugamos",
"era": u"\u00E9"+"ramos",
"ergue": "erguemos",
"ergueu": "erguemos",
"erra": "erramos",
"errou": "erramos",
"escapa": "escapamos",
"escapou": "escapamos",
"escolhe": "escolhemos",
"escolheu": "escolhemos",
"escolhia": "escolh"+u"\u00ED"+"amos",
"escondia": "escond"+u"\u00ED"+"amos",
"escorrega": "escorregamos",
"escorregou": "escorregamos",
"escrevia": "escrev"+u"\u00ED"+"amos",
"escutava": "escut"+u"\u00E1"+"vamos",
"esgota": "esgotamos",
"esgotou": "esgotamos",
"especifica": "especificamos",
"especificou": "especificamos",
"espera": "esperamos",
"esperava": "esper"+u"\u00E1"+"vamos",
"esperou": "esperamos",
"esquece": "esquecemos",
"est"+u"\u00E1": "estamos",
"estabelece": "estabelecemos",
"estabelecer": "estabelecermos",
"estabeleceu": "estabelecemos",
"estar": "estarmos",
"estar"+u"\u00E1": "estaremos",
"estaria": "estar"+u"\u00ED"+"amos",
"estava": "est"+u"\u00E1"+"vamos",
"esteja": "estejamos",
"estima": "estimamos",
"estimou": "estimamos",
"estive": "estivemos",
"estiver": "estivermos",
"estivesse": "estiv"+u"\u00E9"+"ssemos",
"estornar"+u"\u00E1": "estornaremos",
"estoura": "estouramos",
"estourou": "estouramos",
"estraga": "estragamos",
"estragou": "estragamos",
"estranha": "estranhamos",
"estranhasse": "estranh"+u"\u00E1"+"ssemos",
"estranhe": "estranhemos",
"estranhou": "estranhamos",
"estudar"+u"\u00E1": "estudaremos",
"estudava": "estud"+u"\u00E1"+"vamos",
"exerce": "exercemos",
"exerceu": "exercemos",
"exige": "exigimos",
"exigiu": "exigimos",
"existe": "existimos",
"existiu": "existimos",
"exp"+u"\u00F5"+"e": "expomos",
"exportava": "export"+u"\u00E1"+"vamos",
"expressa": "expressamos",
"expressou": "expressamos",
"fa"+u"\u00E7"+"a": "fa"+u"\u00E7"+"amos",
"fala": "falamos",
"falar": "falarmos",
"falar"+u"\u00E1": "falaremos",
"falava": "fal"+u"\u00E1"+"vamos",
"falou": "falamos",
"falta": "faltamos",
"faltou": "faltamos",
"far"+u"\u00E1": "faremos",
"faria": "far"+u"\u00ED"+"amos",
"faz": "fazemos",
"fazer": "fazermos",
"fazia": "faz"+u"\u00ED"+"amos",
"fechar": "fecharmos",
"fechava": "fech"+u"\u00E1"+"vamos",
"fez": "fizemos",
"fica": "ficamos",
"ficar": "ficarmos",
"ficar"+u"\u00E1": "ficaremos",
"ficasse": "fic"+u"\u00E1"+"ssemos",
"ficava": "fic"+u"\u00E1"+"vamos",
"ficou": "ficamos",
"filma": "filmamos",
"filmou": "filmamos",
"finge": "fingimos",
"fingiu": "fingimos",
"fiscaliza": "fiscalizamos",
"fiscalizou": "fiscalizamos",
"fizer": "fizermos",
"fizesse": "fiz"+u"\u00E9"+"ssemos",
"flartava": "flart"+u"\u00E1"+"vamos",
"foge": "fugimos",
"foi": "fomos",
"folgava": "folg"+u"\u00E1"+"vamos",
"for": "formos",
"forma": "formamos",
"formava": "form"+u"\u00E1"+"vamos",
"formou": "formamos",
"fornece": "fornecemos",
"forneceu": "fornecemos",
"fosse": "fossemos",
"frequentava": "frequent"+u"\u00E1"+"vamos",
"fugisse": "fug"+u"\u00ED"+"ssemos",
"fugiu": "fugimos",
"fumava": "fum"+u"\u00E1"+"vamos",
"funda": "fundamos",
"fundava": "fund"+u"\u00E1"+"vamos",
"fundou": "fundamos",
"furta": "furtamos",
"furtou": "furtamos",
"ganha": "ganhamos",
"ganhar"+u"\u00E1": "ganharemos",
"ganhava": "ganh"+u"\u00E1"+"vamos",
"ganhou": "ganhamos",
"garantir"+u"\u00E1": "garantiremos",
"gasta": "gastamos",
"gastou": "gastamos",
"gosta": "gostamos",
"gostaria": "gostar"+u"\u00ED"+"amos",
"gostava": "gost"+u"\u00E1"+"vamos",
"gostou": "gostamos",
"goza": "gozamos",
"gozou": "gozamos",
"gradua": "graduamos",
"graduou": "graduamos",
"guarda": "guardamos",
"guardou": "guardamos",
"h"+u"\u00E1": "havemos",
"habita": "habitamos",
"habitou": "habitamos",
"havia": "hav"+u"\u00ED"+"amos",
"ia": u"\u00ED"+"amos",
"ignorava": "ignor"+u"\u00E1"+"vamos",
"imagina": "imaginamos",
"imaginava": "imagin"+u"\u00E1"+"vamos",
"imita": "imitamos",
"imitaria": "imitar"+u"\u00ED"+"amos",
"imitou": "imitamos",
"implanta": "implantamos",
"implantou": "implantamos",
"imp"+u"\u00F5"+"e": "impomos",
"importa": "importamos",
"importou": "importamos",
"imp"+u"\u00F4"+"s": "impusemos",
"impregna": "impregnamos",
"impregnou": "impregnamos",
"indica": "indicamos",
"insistia": "insist"+u"\u00ED"+"amos",
"insistiu": "insistimos",
"instala": "instalamos",
"instalou": "instalamos",
"interfere": "interferimos",
"interferiu": "interferimos",
"interpreta": "interpretamos",
"interpretou": "interpretamos",
"interviesse": "intervi"+u"\u00E9"+"ssemos",
"invejava": "invej"+u"\u00E1"+"vamos",
"inventa": "inventamos",
"inventou": "inventamos",
"investiu": "investimos",
"invocar"+u"\u00E1": "invocaremos",
"ir": "irmos",
"ir"+u"\u00E1": "iremos",
"iria": "ir"+u"\u00ED"+"amos",
"janta": "jantamos",
"jantasse": "jant"+u"\u00E1"+"ssemos",
"jantava": "jant"+u"\u00E1"+"vamos",
"jantou": "jantamos",
"joga": "jogamos",
"jogou": "jogamos",
"julga": "julgamos",
"julgou": "julgamos",
"jura": "juramos",
"jurou": "juramos",
"labora": "laboramos",
"lamenta": "lamentamos",
"lan"+u"\u00E7"+"ar": "lan"+u"\u00E7"+"armos",
"lava": "lavamos",
"lavar": "lavarmos",
"lavou": "lavamos",
"l"+u"\u00EA": "lemos",
"legalizaria": "legalizar"+u"\u00ED"+"amos",
"lembra": "lembramos",
"lembrou": "lembramos",
"lesse": "l"+u"\u00EA"+"ssemos",
"leva": "levamos",
"levanta": "levantamos",
"levantou": "levantamos",
"levar"+u"\u00E1": "levaremos",
"levava": "lev"+u"\u00E1"+"vamos",
"levou": "levamos",
"lia": "l"+u"\u00ED"+"amos",
"liberte": "libertemos",
"ligava": "lig"+u"\u00E1"+"vamos",
"localiza": "localizamos",
"localizou": "localizamos",
"lustrava": "lustr"+u"\u00E1"+"vamos",
"luta": "lutamos",
"lutou": "lutamos",
"machuca": "machucamos",
"machucou": "machucamos",
"manda": "mandamos",
"mandou": "mandamos",
"mant"+u"\u00E9"+"m": "mantemos",
"marca": "marcamos",
"marcha": "marchamos",
"marchar"+u"\u00E1": "marcharemos",
"marchou": "marchamos",
"marcou": "marcamos",
"mata": "matamos",
"matara": "mat"+u"\u00E1"+"ramos",
"matava": "mat"+u"\u00E1"+"vamos",
"matou": "matamos",
"mente": "mentimos",
"mentiu": "mentimos",
"merece": "merecemos",
"mereceu": "merecemos",
"merecia": "merec"+u"\u00ED"+"amos",
"meter": "metermos",
"mistura": "misturamos",
"misturou": "misturamos",
"molda": "moldamos",
"moldou": "moldamos",
"montaria": "montar"+u"\u00ED"+"amos",
"mora": "moramos",
"morar"+u"\u00E1": "moraremos",
"morava": "mor"+u"\u00E1"+"vamos",
"more": "moremos",
"morou": "moramos",
"morrer": "morrermos",
"morresse": "morr"+u"\u00EA"+"ssemos",
"mostra": "mostramos",
"mostrou": "mostramos",
"muda": "mudamos",
"mudou": "mudamos",
"nasce": "nascemos",
"nascer": "nascermos",
"nasceu": "nascemos",
"necessita": "necessitamos",
"necessitasse": "necessit"+u"\u00E1"+"ssemos",
"necessitou": "necessitamos",
"nota": "notamos",
"notar": "notarmos",
"notou": "notamos",
"observa": "observamos",
"observar": "observarmos",
"observou": "observamos",
"obteve": "obtivemos",
"oferece": "oferecemos",
"ofereceu": "oferecemos",
"olha": "olhamos",
"olhar": "olharmos",
"olhava": "olh"+u"\u00E1"+"vamos",
"olhou": "olhamos",
"opta": "optamos",
"optou": "optamos",
"organizava": "organiz"+u"\u00E1"+"vamos",
"ouve": "ouvimos",
"ouvia": "ouv"+u"\u00ED"+"amos",
"ouviria": "ouvir"+u"\u00ED"+"amos",
"ouviu": "ouvimos",
"paga": "pagamos",
"pagar"+u"\u00E1": "pagaremos",
"pagava": "pag"+u"\u00E1"+"vamos",
"pagou": "pagamos",
"parar": "pararmos",
"parava": "par"+u"\u00E1"+"vamos",
"parece": "parecemos",
"pareceu": "parecemos",
"parte": "partimos",
"participa": "participamos",
"participou": "participamos",
"partir"+u"\u00E1": "partiremos",
"partiu": "partimos",
"passa": "passamos",
"passar": "passarmos",
"passasse": "pass"+u"\u00E1"+"ssemos",
"passava": "pass"+u"\u00E1"+"vamos",
"passear": "passearmos",
"passou": "passamos",
"pede": "pedimos",
"pediu": "pedimos",
"pega": "pegamos",
"pegar": "pegarmos",
"pegou": "pegamos",
"pensa": "pensamos",
"pensar": "pensarmos",
"pensava": "pens"+u"\u00E1"+"vamos",
"percebe": "percebemos",
"percebeu": "percebemos",
"percebia": "perceb"+u"\u00ED"+"amos",
"percorre": "percorremos",
"percorreu": "percorremos",
"perde": "perdemos",
"perder": "perdermos",
"perder"+u"\u00E1": "perderemos",
"perdeu": "perdemos",
"perdoa": "perdoamos",
"perdoou": "perdoamos",
"pergunta": "perguntamos",
"perguntou": "perguntamos",
"pertence": "pertencemos",
"pertenceu": "pertencemos",
"pode": "podemos",
"poder"+u"\u00E1": "poderemos",
"poderia": "poder"+u"\u00ED"+"amos",
"podia": "pod"+u"\u00ED"+"amos",
"p"+u"\u00F5"+"e": "pomos",
"poria": "por"+u"\u00ED"+"amos",
"possa": "possamos",
"possui": "possu"+u"\u00ED"+"mos",
"possuiu": "possu"+u"\u00ED"+"mos",
"poupa": "poupamos",
"pouparia": "poupar"+u"\u00ED"+"amos",
"poupou": "poupamos",
"precisa": "precisamos",
"precisar": "precisarmos",
"precisaria": "precisar"+u"\u00ED"+"amos",
"precisava": "precis"+u"\u00E1"+"vamos",
"predisse": "predissemos",
"prefere": "preferimos",
"preferia": "prefer"+u"\u00ED"+"amos",
"preferiu": "preferimos",
"prende": "prendemos",
"prepara": "preparamos",
"prepararia": "preparar"+u"\u00ED"+"amos",
"preparava": "prepar"+u"\u00E1"+"vamos",
"preparou": "preparamos",
"presencia": "presenciamos",
"presenciou": "presenciamos",
"presta": "prestamos",
"prestou": "prestamos",
"pretende": "pretendemos",
"pretendeu": "pretendemos",
"pretendia": "pretend"+u"\u00ED"+"amos",
"processa": "processamos",
"processou": "processamos",
"procura": "procuramos",
"procurou": "procuramos",
"produz": "produzimos",
"produziu": "produzimos",
"profere": "proferimos",
"proferiu": "proferimos",
"projetava": "projet"+u"\u00E1"+"vamos",
"promete": "prometemos",
"prop"+u"\u00F5"+"e": "propomos",
"provar"+u"\u00E1": "provaremos",
"providencia": "providenciamos",
"publicaria": "publicar"+u"\u00ED"+"amos",
"puder": "pudermos",
"punha": "p"+u"\u00FA"+"nhamos",
"pus": "pusemos",
"puxa": "puxamos",
"puxou": "puxamos",
"quebrava": "quebr"+u"\u00E1"+"vamos",
"quer": "queremos",
"querer": "querermos",
"queria": "quer"+u"\u00ED"+"amos",
"quis": "quisemos",
"quiser": "quisermos",
"quisera": "quis"+u"\u00E9"+"ramos",
"quisesse": "quis"+u"\u00E9"+"ssemos",
"rebolava": "rebol"+u"\u00E1"+"vamos",
"receava": "rece"+u"\u00E1"+"vamos",
"recebe": "recebemos",
"receber"+u"\u00E1": "receberemos",
"recebeu": "recebemos",
"recebia": "receb"+u"\u00ED"+"amos",
"reconhece": "reconhecemos",
"reconheceu": "reconhecemos",
"recorde": "recordemos",
"reduziu": "reduzimos",
"registra": "registramos",
"registrou": "registramos",
"regrava": "regravamos",
"reivindica": "reivindicamos",
"reivindicou": "reivindicamos",
"rema": "remamos",
"reparar": "repararmos",
"repete": "repetimos",
"repetiu": "repetimos",
"representa": "representamos",
"representou": "representamos",
"requere": "requeremos",
"resiste": "resistimos",
"resistiu": "resistimos",
"resolve": "resolvemos",
"resolveria": "resolver"+u"\u00ED"+"amos",
"resolveu": "resolvemos",
"resolvia": "resolv"+u"\u00ED"+"amos",
"respeita": "respeitamos",
"respeitou": "respeitamos",
"respira": "respiramos",
"respirava": "respir"+u"\u00E1"+"vamos",
"respirou": "respiramos",
"responder": "respondermos",
"respondia": "respond"+u"\u00ED"+"amos",
"responsabiliza": "responsabilizamos",
"responsabilizou": "responsabilizamos",
"retesou": "retesamos",
"retoma": "retomamos",
"retomou": "retomamos",
"retribuiu": "retribu"+u"\u00ED"+"mos",
"retroceder": "retrocedermos",
"re"+u"\u00FA"+"ne": "reunimos",
"reuniu": "reunimos",
"revivia": "reviv"+u"\u00ED"+"amos",
"reza": "rezamos",
"reze": "rezemos",
"rezou": "rezamos",
"roeu": "roemos",
"r"+u"\u00F3"+"i": "roemos",
"rola": "rolamos",
"rolava": "rol"+u"\u00E1"+"vamos",
"rolou": "rolamos",
"rouba": "roubamos",
"roubou": "roubamos",
"sabe": "sabemos",
"saber": "sabermos",
"saber"+u"\u00E1": "saberemos",
"saberia": "saber"+u"\u00ED"+"amos",
"sabia": "sab"+u"\u00ED"+"amos",
"saboreia": "saboreamos",
"sacrifica": "sacrificamos",
"sai": "sa"+u"\u00ED"+"mos",
"sa"+u"\u00ED"+"a": "sa"+u"\u00ED"+"amos",
"saiba": "saibamos",
"sair": "sairmos",
"sair"+u"\u00E1": "sairemos",
"sa"+u"\u00ED"+"sse": "sa"+u"\u00ED"+"ssemos",
"sanciona": "sancionamos",
"saudava": "saud"+u"\u00E1"+"vamos",
"segue": "seguimos",
"seguiu": "seguimos",
"segura": "seguramos",
"segurar"+u"\u00E1": "seguraremos",
"segurou": "seguramos",
"seja": "sejamos",
"selecionar"+u"\u00E1": "selecionaremos",
"sente": "sentimos",
"sentia": "sent"+u"\u00ED"+"amos",
"sentir": "sentirmos",
"ser": "sermos",
"ser"+u"\u00E1": "seremos",
"seria": "ser"+u"\u00ED"+"amos",
"serve": "servimos",
"serviu": "servimos",
"sinta": "sintamos",
"sofre": "sofremos",
"sofreu": "sofremos",
"sonha": "sonhamos",
"sonhava": "sonh"+u"\u00E1"+"vamos",
"sonhou": "sonhamos",
"sorri": "sorrimos",
"sorria": "sorr"+u"\u00ED"+"amos",
"sorriu": "sorrimos",
"soube": "soubemos",
"souber": "soubermos",
"soubesse": "soub"+u"\u00E9"+"ssemos",
"supera": "superamos",
"superou": "superamos",
"suplica": "suplicamos",
"supomos": "supomos",
"suprir"+u"\u00E1": "supriremos",
"surpreende": "surpreendemos",
"suspeitava": "suspeit"+u"\u00E1"+"vamos",
"suspira": "suspiramos",
"suspirava": "suspir"+u"\u00E1"+"vamos",
"sustenta": "sustentamos",
"ta": "tamos",
"teimava": "teim"+u"\u00E1"+"vamos",
"tem": "temos",
"teme": "tememos",
"temia": "tem"+u"\u00ED"+"amos",
"tenciona": "tencionamos",
"tenha": "tenhamos",
"tenta": "tentamos",
"tentaria": "tentar"+u"\u00ED"+"amos",
"tentasse": "tent"+u"\u00E1"+"ssemos",
"tentou": "tentamos",
"ter": "termos",
"ter"+u"\u00E1": "teremos",
"teria": "ter"+u"\u00ED"+"amos",
"termina": "terminamos",
"terminava": "termin"+u"\u00E1"+"vamos",
"testar": "testarmos",
"tinha": "t"+u"\u00ED"+"nhamos",
"tira": "tiramos",
"tirava": "tir"+u"\u00E1"+"vamos",
"tire": "tiremos",
"tive": "tivemos",
"tiver": "tivermos",
"tivesse": "tiv"+u"\u00E9"+"ssemos",
"toca": "tocamos",
"tocou": "tocamos",
"toma": "tomamos",
"tomar": "tomarmos",
"tomar"+u"\u00E1": "tomaremos",
"tomava": "tom"+u"\u00E1"+"vamos",
"tomou": "tomamos",
"topa": "topamos",
"torna": "tornamos",
"trabalha": "trabalhamos",
"trabalhava": "trabalh"+u"\u00E1"+"vamos",
"trabalhou": "trabalhamos",
"traja": "trajamos",
"transpirava": "transpir"+u"\u00E1"+"vamos",
"trata": "tratamos",
"tratar"+u"\u00E1": "trataremos",
"tratava": "trat"+u"\u00E1"+"vamos",
"traz": "trazemos",
"trazia": "traz"+u"\u00ED"+"amos",
"treina": "treinamos",
"tremia": "trem"+u"\u00ED"+"amos",
"trepasse": "trep"+u"\u00E1"+"ssemos",
"trilha": "trilhamos",
"trouxe": "trouxemos",
"usa": "usamos",
"usava": "us"+u"\u00E1"+"vamos",
"usou": "usamos",
"usufru"+u"\u00ED"+"a": "usufru"+u"\u00ED"+"amos",
"utiliza": "utilizamos",
"utilizasse": "utiliz"+u"\u00E1"+"ssemos",
"utilizou": "utilizamos",
"vai": "vamos",
"vale": "valemos",
"valorizava": "valoriz"+u"\u00E1"+"vamos",
"vaticina": "vaticinamos",
"v"+u"\u00EA": "vemos",
"veio": "viemos",
"vem": "vimos",
"vence": "vencemos",
"venceu": "vencemos",
"vende": "vendemos",
"vendeu": "vendemos",
"ver": "vermos",
"ver"+u"\u00E1": "veremos",
"verifica": "verificamos",
"verificou": "verificamos",
"vestia": "vest"+u"\u00ED"+"amos",
"via": "v"+u"\u00ED"+"amos",
"viaja": "viajamos",
"viajou": "viajamos",
"vinha": "v"+u"\u00ED"+"nhamos",
"vir": "virmos",
"vira": "viramos",
"vire": "viremos",
"visite": "visitemos",
"visse": "v"+u"\u00ED"+"ssemos",
"visualiza": "visualizamos",
"visualizar": "visualizarmos",
"vive": "vivemos",
"vivencia": "vivenciamos",
"viver": "vivermos",
"viver"+u"\u00E1": "viveremos",
"vivesse": "viv"+u"\u00EA"+"ssemos",
"vivia": "viv"+u"\u00ED"+"amos",
"voa": "voamos",
"volta": "voltamos",
"voltar": "voltarmos",
"voltar"+u"\u00E1": "voltaremos",
"voltaria": "voltar"+u"\u00ED"+"amos",
"volte": "voltemos",
"voltou": "voltamos",
"vota": "votamos",
"votou": "votamos"}


# concNos - concordancia com o pronome Nos (com acento) - Nos vai
concNos = set(["abandona", "abandonou", "abdica", "abdicasse",
"abdicou", "abra"+u"\u00E7"+"a", "abra"+u"\u00E7"+"ou", "abre",
"abriu", "acaba", "acabou", "aceita",
"aceitaria", "aceitou", "acha", "achava",
"achou", "acolhe", "acompanha", "acompanhou",
"acorda", "acordou", "acredita", "acreditava",
"acreditou", "acrescenta", "acrescentou", "adivinha",
"adivinhou", "admira", "admirou", "admitir",
"adora", "adorou", "adota", "adote",
"adotou", "adquire", "adquiriu", "adulterar",
"adverte", "advertiu", "advoga", "advogou",
"age", "agiu", "agradece", "agradeceu",
"ajuda", "ajudar"+u"\u00E1", "ajudou", "ajunta",
"ajuntou", "ajusta", "ajustou", "alcan"+u"\u00E7"+"ava",
"alimenta", "alimentou", "aliviava", "almo"+u"\u00E7"+"a",
"almo"+u"\u00E7"+"ou", "alugasse", "ama", "amarra",
"amarrou", "amava", "amou", "analisa",
"analisasse", "analisou", "anda", "andava",
"andou", "anuncia", "apanha", "apanhou",
"apare"+u"\u00E7"+"a", "aparece", "apareceu", "aplaude",
"aplaudiu", "aplicaria", "apodrece", "apodreceu",
"apoia", "apoiou", "aprende", "aprendeu",
"apresenta", "apresentar"+u"\u00E1", "apresentou", "apresta",
"aprestou", "aprova", "aprovasse", "aproveita",
"aprovou", "arranjar"+u"\u00E1", "arranjasse", "arrasta",
"arrastou", "arromba", "arrombou", "aspira",
"aspirou", "assina", "assinar", "assinou",
"assiste", "assistir", "assistiu", "associa",
"assume", "assumiu", "ataca", "atacou",
"atiraria", "atravessa", "atravessava", "atravessou",
"aumentar", "avalia", "avaliar", "avaliou",
"avan"+u"\u00E7"+"a", "avan"+u"\u00E7"+"ou", "bate", "bateu",
"batia", "batiza", "batizou", "bebe",
"beneficia", "beneficiou", "bota", "botou",
"brada", "bradou", "briga", "brigou",
"brinca", "brincava", "brincou", "busca",
"buscou", "cabe", "ca"+u"\u00E7"+"oa", "ca"+u"\u00E7"+"oou",
"cai", "cairia", "ca"+u"\u00ED"+"sse", "caiu",
"cala", "calcula", "calculou", "calou",
"caminha", "caminhou", "cantar"+u"\u00E1", "carece",
"careceu", "carrega", "carregava", "carregou",
"casa", "casou", "cavalga", "cavalgou",
"celebra", "celebrou", "chama", "chamava",
"chamou", "checa", "checou", "chega",
"chegar", "chegar"+u"\u00E1", "chegava", "chegou",
"chegue", "chora", "chorava", "chore",
"chorou", "cita", "citou", "cobra",
"cobrou", "colabora", "colaborou", "coloca",
"colocou", "combater"+u"\u00E1", "combina", "combinava",
"combinou", "come", "come"+u"\u00E7"+"a", "come"+u"\u00E7"+"ar",
"come"+u"\u00E7"+"ou", "comenta", "comentou", "comer",
"comete", "cometeu", "comeu", "comia",
"compara", "comparou", "completa", "completou",
"compra", "compraria", "comprava", "compreende",
"compreendeu", "comprou", "comunga", "comungou",
"concorda", "concordar"+u"\u00E1", "concordou", "concorria",
"condena", "condenou", "confeccionaria", "confiava",
"confirmar", "congelaria", "conhece", "conheceu",
"conhecia", "conjetura", "conjeturou", "conquista",
"conquistou", "consegue", "conseguia", "conseguir",
"conseguir"+u"\u00E1", "conseguiria", "conseguiu", "conservar",
"conserve", "considera", "considerar", "considerava",
"considerou", "consiga", "constata", "constatou",
"constr"+u"\u00F3"+"i", "construiu", "conta", "continua",
"continuar"+u"\u00E1", "continuava", "continuou", "contou",
"contratasse", "contratava", "controla", "controlou",
"conversa", "conversar"+u"\u00E1", "conversava", "conversou",
"convida", "convidou", "coopera", "cooperou",
"corre", "correr"+u"\u00E1", "correu", "costuma",
"costumava", "costumou", "cota", "cotou",
"cr"+u"\u00EA", "credencia", "credenciou", "cresce",
"cresceu", "crescia", "creu", "cria",
"criar", "criava", "criou", "critica",
"criticou", "cruza", "cuida", "cuidou",
"culpa", "culpou", "cultiva", "cultivou",
"cumpre", "cumpriu", "d"+u"\u00E1", "dan"+u"\u00E7"+"a",
"dan"+u"\u00E7"+"asse", "dan"+u"\u00E7"+"ava", "dava", "debitar"+u"\u00E1",
"decide", "decidir"+u"\u00E1", "decidiu", "defende",
"defendeu", "defendia", "define", "definiu",
"deita", "deitou", "deixa", "deixaria",
"deixou", "demonstra", "demonstrou", "denomina",
"denominou", "denuncia", "denunciou", "deposita",
"depositou", "derrubar", "desce", "descende",
"desceu", "descobre", "descobrir", "descobrir"+u"\u00E1",
"descobriu", "desconfiava", "descoroa", "deseja",
"desejaria", "desejava", "desejou", "desempenha",
"desempenhou", "desentranha", "desentranhou", "desenvolve",
"desenvolveu", "desenvolvia", "designa", "designou",
"despachava", "despreza", "desprezou", "desrespeitar",
"desviava", "detesta", "detestou", "deu",
"deva", "deve", "deveria", "devia",
"diagnosticar"+u"\u00E1", "diga", "dir"+u"\u00E1", "diria",
"discutia", "disp"+u"\u00F5"+"e", "dispor", "disputa",
"disputou", "disse", "dissesse", "divaga",
"divagou", "diverge", "divergiu", "divide",
"dividia", "dividiu", "diz", "dizia",
"dobra", "dobrou", "dormia", "dormir"+u"\u00E1",
"dormiu", "dubla", "duplica", "duplicou",
u"\u00E9", "elabora", "elaborou", "elege",
"elegeu", "embarga", "embargou", "embirra",
"embirrou", "emprega", "empregou", "encara",
"encarou", "enche", "encheu", "encolhia",
"encontra", "encontrar", "encontrar"+u"\u00E1", "encontrou",
"enjeita", "entende", "entendesse", "entra",
"entrar", "entrava", "entrega", "entregou",
"entrevista", "entrevistou", "entrou", "envia",
"enviaria", "enviou", "enxuga", "enxugar"+u"\u00E1",
"enxugou", "era", "ergue", "ergueu",
"erra", "errou", "escapa", "escapou",
"escolhe", "escolheu", "escolhia", "escondia",
"escorrega", "escorregou", "escrevia", "escutava",
"esgota", "esgotou", "especifica", "especificou",
"espera", "esperava", "esperou", "esquece",
"est"+u"\u00E1", "estabelece", "estabelecer", "estabeleceu",
"estar", "estar"+u"\u00E1", "estaria", "estava",
"esteja", "estima", "estimou", "estive",
"estiver", "estivesse", "estornar"+u"\u00E1", "estoura",
"estourou", "estraga", "estragou", "estranha",
"estranhasse", "estranhe", "estranhou", "estudar"+u"\u00E1",
"estudava", "exerce", "exerceu", "exige",
"exigiu", "existe", "existiu", "exp"+u"\u00F5"+"e",
"exportava", "expressa", "expressou", "fa"+u"\u00E7"+"a",
"fala", "falar", "falar"+u"\u00E1", "falava",
"falou", "falta", "faltou", "far"+u"\u00E1",
"faria", "faz", "fazer", "fazia",
"fechar", "fechava", "fez", "fica",
"ficar", "ficar"+u"\u00E1", "ficasse", "ficava",
"ficou", "filma", "filmou", "finge",
"fingiu", "fiscaliza", "fiscalizou", "fizer",
"fizesse", "flartava", "foge", "foi",
"folgava", "for", "forma", "formava",
"formou", "fornece", "forneceu", "fosse",
"frequentava", "fugisse", "fugiu", "fumava",
"funda", "fundava", "fundou", "furta",
"furtou", "ganha", "ganhar"+u"\u00E1", "ganhava",
"ganhou", "garantir"+u"\u00E1", "gasta", "gastou",
"gosta", "gostaria", "gostava", "gostou",
"goza", "gozou", "gradua", "graduou",
"guarda", "guardou", "h"+u"\u00E1", "habita",
"habitou", "havia", "ia", "ignorava",
"imagina", "imaginava", "imita", "imitaria",
"imitou", "implanta", "implantou", "imp"+u"\u00F5"+"e",
"importa", "importou", "imp"+u"\u00F4"+"s", "impregna",
"impregnou", "indica", "insistia", "insistiu",
"instala", "instalou", "interfere", "interferiu",
"interpreta", "interpretou", "interviesse", "invejava",
"inventa", "inventou", "investiu", "invocar"+u"\u00E1",
"ir", "ir"+u"\u00E1", "iria", "janta",
"jantasse", "jantava", "jantou", "joga",
"jogou", "julga", "julgou", "jura",
"jurou", "labora", "lamenta", "lan"+u"\u00E7"+"ar",
"lava", "lavar", "lavou", "l"+u"\u00EA",
"legalizaria", "lembra", "lembrou", "lesse",
"leva", "levanta", "levantou", "levar"+u"\u00E1",
"levava", "levou", "lia", "liberte",
"ligava", "localiza", "localizou", "lustrava",
"luta", "lutou", "machuca", "machucou",
"manda", "mandou", "mant"+u"\u00E9"+"m", "marca",
"marcha", "marchar"+u"\u00E1", "marchou", "marcou",
"mata", "matara", "matava", "matou",
"mente", "mentiu", "merece", "mereceu",
"merecia", "meter", "mistura", "misturou",
"molda", "moldou", "montaria", "mora",
"morar"+u"\u00E1", "morava", "more", "morou",
"morrer", "morresse", "mostra", "mostrou",
"muda", "mudou", "nasce", "nascer",
"nasceu", "necessita", "necessitasse", "necessitou",
"nota", "notar", "notou", "observa",
"observar", "observou", "obteve", "oferece",
"ofereceu", "olha", "olhar", "olhava",
"olhou", "opta", "optou", "organizava",
"ouve", "ouvia", "ouviria", "ouviu",
"paga", "pagar"+u"\u00E1", "pagava", "pagou",
"parar", "parava", "parece", "pareceu",
"parte", "participa", "participou", "partir"+u"\u00E1",
"partiu", "passa", "passar", "passasse",
"passava", "passear", "passou", "pede",
"pediu", "pega", "pegar", "pegou",
"pensa", "pensar", "pensava", "percebe",
"percebeu", "percebia", "percorre", "percorreu",
"perde", "perder", "perder"+u"\u00E1", "perdeu",
"perdoa", "perdoou", "pergunta", "perguntou",
"pertence", "pertenceu", "pode", "poder"+u"\u00E1",
"poderia", "podia", "p"+u"\u00F5"+"e", "poria",
"possa", "possui", "possuiu", "poupa",
"pouparia", "poupou", "precisa", "precisar",
"precisaria", "precisava", "predisse", "prefere",
"preferia", "preferiu", "prende", "prepara",
"prepararia", "preparava", "preparou", "presencia",
"presenciou", "presta", "prestou", "pretende",
"pretendeu", "pretendia", "processa", "processou",
"procura", "procurou", "produz", "produziu",
"profere", "proferiu", "projetava", "promete",
"prop"+u"\u00F5"+"e", "provar"+u"\u00E1", "providencia", "publicaria",
"puder", "punha", "pus", "puxa",
"puxou", "quebrava", "quer", "querer",
"queria", "quis", "quiser", "quisera",
"quisesse", "rebolava", "receava", "recebe",
"receber"+u"\u00E1", "recebeu", "recebia", "reconhece",
"reconheceu", "recorde", "reduziu", "registra",
"registrou", "regrava", "reivindica", "reivindicou",
"rema", "reparar", "repete", "repetiu",
"representa", "representou", "requere", "resiste",
"resistiu", "resolve", "resolveria", "resolveu",
"resolvia", "respeita", "respeitou", "respira",
"respirava", "respirou", "responder", "respondia",
"responsabiliza", "responsabilizou", "retesou", "retoma",
"retomou", "retribuiu", "retroceder", "re"+u"\u00FA"+"ne",
"reuniu", "revivia", "reza", "reze",
"rezou", "roeu", "r"+u"\u00F3"+"i", "rola",
"rolava", "rolou", "rouba", "roubou",
"sabe", "saber", "saber"+u"\u00E1", "saberia",
"sabia", "saboreia", "sacrifica", "sai",
"sa"+u"\u00ED"+"a", "saiba", "sair", "sair"+u"\u00E1",
"sa"+u"\u00ED"+"sse", "sanciona", "saudava", "segue",
"seguiu", "segura", "segurar"+u"\u00E1", "segurou",
"seja", "selecionar"+u"\u00E1", "sente", "sentia",
"sentir", "ser", "ser"+u"\u00E1", "seria",
"serve", "serviu", "sinta", "sofre",
"sofreu", "sonha", "sonhava", "sonhou",
"sorri", "sorria", "sorriu", "soube",
"souber", "soubesse", "supera",
"superou", "suplica", "supomos", "suprir"+u"\u00E1",
"surpreende", "suspeitava", "suspira", "suspirava",
"sustenta", "ta", "teimava", "tem",
"teme", "temia", "tenciona", "tenha",
"tenta", "tentaria", "tentasse", "tentou",
"ter", "ter"+u"\u00E1", "teria", "termina",
"terminava", "testar", "tinha", "tira",
"tirava", "tire", "tive", "tiver",
"tivesse", "toca", "tocou", "toma",
"tomar", "tomar"+u"\u00E1", "tomava", "tomou",
"topa", "torna", "trabalha", "trabalhava",
"trabalhou", "traja", "transpirava", "trata",
"tratar"+u"\u00E1", "tratava", "traz", "trazia",
"treina", "tremia", "trepasse", "trilha",
"trouxe", "usa", "usava", "usou",
"usufru"+u"\u00ED"+"a", "utiliza", "utilizasse", "utilizou",
"vai", "vale", "valorizava", "vaticina",
"v"+u"\u00EA", "veio", "vem", "vence",
"venceu", "vende", "vendeu", "ver",
"ver"+u"\u00E1", "verifica", "verificou", "vestia",
"via", "viaja", "viajou", "vinha",
"vir", "vira", "vire", "visite",
"visse", "visualiza", "visualizar", "vive",
"vivencia", "viver", "viver"+u"\u00E1", "vivesse",
"vivia", "voa", "volta", "voltar",
"voltar"+u"\u00E1", "voltaria", "volte", "voltou",
"vota", "votou"])


# aacpEle = Concordancia  com o pronome Ele  
aacpEle = {"abandonamos": "abandonou",
"abdic"+u"\u00E1"+"ssemos": "abdicasse",
"abdicamos": "abdicou",
"abra"+u"\u00E7"+"amos": "abra"+u"\u00E7"+"ou",
"abrimos": "abriu",
"acabamos": "acabou",
"aceitamos": "aceitou",
"aceitar"+u"\u00ED"+"amos": "aceitaria",
"ach"+u"\u00E1"+"vamos": "achava",
"achamos": "achou",
"acolhemos": "acolhe",
"acompanhamos": "acompanhou",
"acordamos": "acordou",
"acredit"+u"\u00E1"+"vamos": "acreditava",
"acreditamos": "acreditou",
"acrescentamos": "acrescentou",
"adivinhamos": "adivinhou",
"admiramos": "admirou",
"admitirmos": "admitir",
"adoramos": "adorou",
"adotamos": "adotou",
"adotemos": "adote",
"adquirimos": "adquiriu",
"adulterarmos": "adulterar",
"advertimos": "advertiu",
"advogamos": "advogou",
"agimos": "agiu",
"agradecemos": "agradeceu",
"ajudamos": "ajudou",
"ajudaremos": "ajudar"+u"\u00E1",
"ajuntamos": "ajuntou",
"ajustamos": "ajustou",
"alcan"+u"\u00E7"+u"\u00E1"+"vamos": "alcan"+u"\u00E7"+"ava",
"alimentamos": "alimentou",
"alivi"+u"\u00E1"+"vamos": "aliviava",
"almo"+u"\u00E7"+"amos": "almo"+u"\u00E7"+"ou",
"alug"+u"\u00E1"+"ssemos": "alugasse",
"am"+u"\u00E1"+"vamos": "amava",
"amamos": "amou",
"amarramos": "amarrou",
"analis"+u"\u00E1"+"ssemos": "analisasse",
"analisamos": "analisou",
"and"+u"\u00E1"+"vamos": "andava",
"andamos": "andou",
"anunciamos": "anuncia",
"apanhamos": "apanhou",
"apare"+u"\u00E7"+"amos": "apare"+u"\u00E7"+"a",
"aparecemos": "apareceu",
"aplaudimos": "aplaudiu",
"aplicar"+u"\u00ED"+"amos": "aplicaria",
"apodrecemos": "apodreceu",
"apoiamos": "apoiou",
"aprendemos": "aprendeu",
"apresentamos": "apresentou",
"apresentaremos": "apresentar"+u"\u00E1",
"aprestamos": "aprestou",
"aprov"+u"\u00E1"+"ssemos": "aprovasse",
"aprovamos": "aprovou",
"aproveitamos": "aproveita",
"arranj"+u"\u00E1"+"ssemos": "arranjasse",
"arranjaremos": "arranjar"+u"\u00E1",
"arrastamos": "arrastou",
"arrombamos": "arrombou",
"aspiramos": "aspirou",
"assinamos": "assinou",
"assinarmos": "assinar",
"assistimos": "assistiu",
"assistirmos": "assistir",
"associamos": "associa",
"assumimos": "assumiu",
"atacamos": "atacou",
"atirar"+u"\u00ED"+"amos": "atiraria",
"atravess"+u"\u00E1"+"vamos": "atravessava",
"atravessamos": "atravessou",
"aumentarmos": "aumentar",
"avaliamos": "avaliou",
"avaliarmos": "avaliar",
"avan"+u"\u00E7"+"amos": "avan"+u"\u00E7"+"ou",
"bat"+u"\u00ED"+"amos": "batia",
"batemos": "bateu",
"batizamos": "batizou",
"bebemos": "bebe",
"beneficiamos": "beneficiou",
"botamos": "botou",
"bradamos": "bradou",
"brigamos": "brigou",
"brinc"+u"\u00E1"+"vamos": "brincava",
"brincamos": "brincou",
"buscamos": "buscou",
"ca"+u"\u00E7"+"oamos": "ca"+u"\u00E7"+"oou",
"ca"+u"\u00ED"+"mos": "caiu",
"ca"+u"\u00ED"+"ssemos": "ca"+u"\u00ED"+"sse",
"cabemos": "cabe",
"cair"+u"\u00ED"+"amos": "cairia",
"calamos": "calou",
"calculamos": "calculou",
"caminhamos": "caminhou",
"cantaremos": "cantar"+u"\u00E1",
"carecemos": "careceu",
"carreg"+u"\u00E1"+"vamos": "carregava",
"carregamos": "carregou",
"casamos": "casou",
"cavalgamos": "cavalgou",
"celebramos": "celebrou",
"cham"+u"\u00E1"+"vamos": "chamava",
"chamamos": "chamou",
"checamos": "checou",
"cheg"+u"\u00E1"+"vamos": "chegava",
"chegamos": "chegou",
"chegaremos": "chegar"+u"\u00E1",
"chegarmos": "chegar",
"cheguemos": "chegue",
"chor"+u"\u00E1"+"vamos": "chorava",
"choramos": "chorou",
"choremos": "chore",
"citamos": "citou",
"cobramos": "cobrou",
"colaboramos": "colaborou",
"colocamos": "colocou",
"com"+u"\u00ED"+"amos": "comia",
"combateremos": "combater"+u"\u00E1",
"combin"+u"\u00E1"+"vamos": "combinava",
"combinamos": "combinou",
"come"+u"\u00E7"+"amos": "come"+u"\u00E7"+"ou",
"come"+u"\u00E7"+"armos": "come"+u"\u00E7"+"ar",
"comemos": "comeu",
"comentamos": "comentou",
"comermos": "comer",
"cometemos": "cometeu",
"comparamos": "comparou",
"completamos": "completou",
"compr"+u"\u00E1"+"vamos": "comprava",
"compramos": "comprou",
"comprar"+u"\u00ED"+"amos": "compraria",
"compreendemos": "compreendeu",
"comungamos": "comungou",
"concordamos": "concordou",
"concordaremos": "concordar"+u"\u00E1",
"concorr"+u"\u00ED"+"amos": "concorria",
"condenamos": "condenou",
"confeccionar"+u"\u00ED"+"amos": "confeccionaria",
"confi"+u"\u00E1"+"vamos": "confiava",
"confirmarmos": "confirmar",
"congelar"+u"\u00ED"+"amos": "congelaria",
"conhec"+u"\u00ED"+"amos": "conhecia",
"conhecemos": "conheceu",
"conjeturamos": "conjeturou",
"conquistamos": "conquistou",
"consegu"+u"\u00ED"+"amos": "conseguia",
"conseguimos": "conseguiu",
"conseguir"+u"\u00ED"+"amos": "conseguiria",
"conseguiremos": "conseguir"+u"\u00E1",
"conseguirmos": "conseguir",
"conservarmos": "conservar",
"conservemos": "conserve",
"consider"+u"\u00E1"+"vamos": "considerava",
"consideramos": "considerou",
"considerarmos": "considerar",
"consigamos": "consiga",
"constatamos": "constatou",
"constru"+u"\u00ED"+"mos": "construiu",
"contamos": "contou",
"continu"+u"\u00E1"+"vamos": "continuava",
"continuamos": "continuou",
"continuaremos": "continuar"+u"\u00E1",
"contrat"+u"\u00E1"+"ssemos": "contratasse",
"contrat"+u"\u00E1"+"vamos": "contratava",
"controlamos": "controlou",
"convers"+u"\u00E1"+"vamos": "conversava",
"conversamos": "conversou",
"conversaremos": "conversar"+u"\u00E1",
"convidamos": "convidou",
"cooperamos": "cooperou",
"corremos": "correu",
"correremos": "correr"+u"\u00E1",
"costum"+u"\u00E1"+"vamos": "costumava",
"costumamos": "costumou",
"cotamos": "cotou",
"credenciamos": "credenciou",
"cremos": "creu",
"cresc"+u"\u00ED"+"amos": "crescia",
"crescemos": "cresceu",
"cri"+u"\u00E1"+"vamos": "criava",
"criamos": "criou",
"criarmos": "criar",
"criticamos": "criticou",
"cruzamos": "cruza",
"cuidamos": "cuidou",
"culpamos": "culpou",
"cultivamos": "cultivou",
"cumprimos": "cumpriu",
"d"+u"\u00E1"+"vamos": "dava",
"damos": "d"+u"\u00E1",
"dan"+u"\u00E7"+"amos": "dan"+u"\u00E7"+"a",
"dan"+u"\u00E7"+u"\u00E1"+"ssemos": "dan"+u"\u00E7"+"asse",
"dan"+u"\u00E7"+u"\u00E1"+"vamos": "dan"+u"\u00E7"+"ava",
"debitaremos": "debitar"+u"\u00E1",
"decidimos": "decidiu",
"decidiremos": "decidir"+u"\u00E1",
"defend"+u"\u00ED"+"amos": "defendia",
"defendemos": "defendeu",
"definimos": "definiu",
"deitamos": "deitou",
"deixamos": "deixou",
"deixar"+u"\u00ED"+"amos": "deixaria",
"demonstramos": "demonstrou",
"demos": "deu",
"denominamos": "denominou",
"denunciamos": "denunciou",
"depositamos": "depositou",
"derrubarmos": "derrubar",
"descemos": "desceu",
"descendemos": "descende",
"descobrimos": "descobriu",
"descobriremos": "descobrir"+u"\u00E1",
"descobrirmos": "descobrir",
"desconfi"+u"\u00E1"+"vamos": "desconfiava",
"descoroamos": "descoroa",
"desej"+u"\u00E1"+"vamos": "desejava",
"desejamos": "desejou",
"desejar"+u"\u00ED"+"amos": "desejaria",
"desempenhamos": "desempenhou",
"desentranhamos": "desentranhou",
"desenvolv"+u"\u00ED"+"amos": "desenvolvia",
"desenvolvemos": "desenvolveu",
"designamos": "designou",
"despach"+u"\u00E1"+"vamos": "despachava",
"desprezamos": "desprezou",
"desrespeitarmos": "desrespeitar",
"desvi"+u"\u00E1"+"vamos": "desviava",
"detestamos": "detestou",
"dev"+u"\u00ED"+"amos": "devia",
"devamos": "deva",
"devemos": "deve",
"dever"+u"\u00ED"+"amos": "deveria",
"diagnosticaremos": "diagnosticar"+u"\u00E1",
"digamos": "diga",
"dir"+u"\u00ED"+"amos": "diria",
"diremos": "dir"+u"\u00E1",
"discut"+u"\u00ED"+"amos": "discutia",
"dispomos": "disp"+u"\u00F5"+"e",
"dispormos": "dispor",
"disputamos": "disputou",
"diss"+u"\u00E9"+"ssemos": "dissesse",
"dissemos": "disse",
"divagamos": "divagou",
"divergimos": "divergiu",
"divid"+u"\u00ED"+"amos": "dividia",
"dividimos": "dividiu",
"diz"+u"\u00ED"+"amos": "dizia",
"dizemos": "diz",
"dobramos": "dobrou",
"dorm"+u"\u00ED"+"amos": "dormia",
"dormimos": "dormiu",
"dormiremos": "dormir"+u"\u00E1",
"dublamos": "dubla",
"duplicamos": "duplicou",
"elaboramos": "elaborou",
"elegemos": "elegeu",
"embargamos": "embargou",
"embirramos": "embirrou",
"empregamos": "empregou",
"encaramos": "encarou",
"enchemos": "encheu",
"encolh"+u"\u00ED"+"amos": "encolhia",
"encontramos": "encontrou",
"encontraremos": "encontrar"+u"\u00E1",
"encontrarmos": "encontrar",
"enjeitamos": "enjeita",
"entend"+u"\u00EA"+"ssemos": "entendesse",
"entendemos": "entende",
"entr"+u"\u00E1"+"vamos": "entrava",
"entramos": "entrou",
"entrarmos": "entrar",
"entregamos": "entregou",
"entrevistamos": "entrevistou",
"enviamos": "enviou",
"enviar"+u"\u00ED"+"amos": "enviaria",
"enxugamos": "enxugou",
"enxugaremos": "enxugar"+u"\u00E1",
u"\u00E9"+"ramos": "era",
"erguemos": "ergueu",
"erramos": "errou",
"escapamos": "escapou",
"escolh"+u"\u00ED"+"amos": "escolhia",
"escolhemos": "escolheu",
"escond"+u"\u00ED"+"amos": "escondia",
"escorregamos": "escorregou",
"escrev"+u"\u00ED"+"amos": "escrevia",
"escut"+u"\u00E1"+"vamos": "escutava",
"esgotamos": "esgotou",
"especificamos": "especificou",
"esper"+u"\u00E1"+"vamos": "esperava",
"esperamos": "esperou",
"esquecemos": "esquece",
"est"+u"\u00E1"+"vamos": "estava",
"estabelecemos": "estabeleceu",
"estabelecermos": "estabelecer",
"estamos": "est"+u"\u00E1",
"estar"+u"\u00ED"+"amos": "estaria",
"estaremos": "estar"+u"\u00E1",
"estarmos": "estar",
"estejamos": "esteja",
"estimamos": "estimou",
"estiv"+u"\u00E9"+"ssemos": "estivesse",
"estivemos": "estive",
"estivermos": "estiver",
"estornaremos": "estornar"+u"\u00E1",
"estouramos": "estourou",
"estragamos": "estragou",
"estranh"+u"\u00E1"+"ssemos": "estranhasse",
"estranhamos": "estranhou",
"estranhemos": "estranhe",
"estud"+u"\u00E1"+"vamos": "estudava",
"estudaremos": "estudar"+u"\u00E1",
"exercemos": "exerceu",
"exigimos": "exigiu",
"existimos": "existiu",
"expomos": "exp"+u"\u00F5"+"e",
"export"+u"\u00E1"+"vamos": "exportava",
"expressamos": "expressou",
"fa"+u"\u00E7"+"amos": "fa"+u"\u00E7"+"a",
"fal"+u"\u00E1"+"vamos": "falava",
"falamos": "falou",
"falaremos": "falar"+u"\u00E1",
"falarmos": "falar",
"faltamos": "faltou",
"far"+u"\u00ED"+"amos": "faria",
"faremos": "far"+u"\u00E1",
"faz"+u"\u00ED"+"amos": "fazia",
"fazemos": "faz",
"fazermos": "fazer",
"fech"+u"\u00E1"+"vamos": "fechava",
"fecharmos": "fechar",
"fic"+u"\u00E1"+"ssemos": "ficasse",
"fic"+u"\u00E1"+"vamos": "ficava",
"ficamos": "ficou",
"ficaremos": "ficar"+u"\u00E1",
"ficarmos": "ficar",
"filmamos": "filmou",
"fingimos": "fingiu",
"fiscalizamos": "fiscalizou",
"fiz"+u"\u00E9"+"ssemos": "fizesse",
"fizemos": "fez",
"fizermos": "fizer",
"flart"+u"\u00E1"+"vamos": "flartava",
"folg"+u"\u00E1"+"vamos": "folgava",
"fomos": "foi",
"form"+u"\u00E1"+"vamos": "formava",
"formamos": "formou",
"formos": "for",
"fornecemos": "forneceu",
"fossemos": "fosse",
"frequent"+u"\u00E1"+"vamos": "frequentava",
"fug"+u"\u00ED"+"ssemos": "fugisse",
"fugimos": "fugiu",
"fum"+u"\u00E1"+"vamos": "fumava",
"fund"+u"\u00E1"+"vamos": "fundava",
"fundamos": "fundou",
"furtamos": "furtou",
"ganh"+u"\u00E1"+"vamos": "ganhava",
"ganhamos": "ganhou",
"ganharemos": "ganhar"+u"\u00E1",
"garantiremos": "garantir"+u"\u00E1",
"gastamos": "gastou",
"gost"+u"\u00E1"+"vamos": "gostava",
"gostamos": "gostou",
"gostar"+u"\u00ED"+"amos": "gostaria",
"gozamos": "gozou",
"graduamos": "graduou",
"guardamos": "guardou",
"habitamos": "habitou",
"hav"+u"\u00ED"+"amos": "havia",
"havemos": "h"+u"\u00E1",
u"\u00ED"+"amos": "ia",
"ignor"+u"\u00E1"+"vamos": "ignorava",
"imagin"+u"\u00E1"+"vamos": "imaginava",
"imaginamos": "imagina",
"imitamos": "imitou",
"imitar"+u"\u00ED"+"amos": "imitaria",
"implantamos": "implantou",
"impomos": "imp"+u"\u00F5"+"e",
"importamos": "importou",
"impregnamos": "impregnou",
"impusemos": "imp"+u"\u00F4"+"s",
"indicamos": "indica",
"insist"+u"\u00ED"+"amos": "insistia",
"insistimos": "insistiu",
"instalamos": "instalou",
"interferimos": "interferiu",
"interpretamos": "interpretou",
"intervi"+u"\u00E9"+"ssemos": "interviesse",
"invej"+u"\u00E1"+"vamos": "invejava",
"inventamos": "inventou",
"investimos": "investiu",
"invocaremos": "invocar"+u"\u00E1",
"ir"+u"\u00ED"+"amos": "iria",
"iremos": "ir"+u"\u00E1",
"irmos": "ir",
"jant"+u"\u00E1"+"ssemos": "jantasse",
"jant"+u"\u00E1"+"vamos": "jantava",
"jantamos": "jantou",
"jogamos": "jogou",
"julgamos": "julgou",
"juramos": "jurou",
"l"+u"\u00EA"+"ssemos": "lesse",
"l"+u"\u00ED"+"amos": "lia",
"laboramos": "labora",
"lamentamos": "lamenta",
"lan"+u"\u00E7"+"armos": "lan"+u"\u00E7"+"ar",
"lavamos": "lavou",
"lavarmos": "lavar",
"legalizar"+u"\u00ED"+"amos": "legalizaria",
"lembramos": "lembrou",
"lemos": "l"+u"\u00EA",
"lev"+u"\u00E1"+"vamos": "levava",
"levamos": "levou",
"levantamos": "levantou",
"levaremos": "levar"+u"\u00E1",
"libertemos": "liberte",
"lig"+u"\u00E1"+"vamos": "ligava",
"localizamos": "localizou",
"lustr"+u"\u00E1"+"vamos": "lustrava",
"lutamos": "lutou",
"machucamos": "machucou",
"mandamos": "mandou",
"mantemos": "mant"+u"\u00E9"+"m",
"marcamos": "marcou",
"marchamos": "marchou",
"marcharemos": "marchar"+u"\u00E1",
"mat"+u"\u00E1"+"ramos": "matara",
"mat"+u"\u00E1"+"vamos": "matava",
"matamos": "matou",
"mentimos": "mentiu",
"merec"+u"\u00ED"+"amos": "merecia",
"merecemos": "mereceu",
"metermos": "meter",
"misturamos": "misturou",
"moldamos": "moldou",
"montar"+u"\u00ED"+"amos": "montaria",
"mor"+u"\u00E1"+"vamos": "morava",
"moramos": "morou",
"moraremos": "morar"+u"\u00E1",
"moremos": "more",
"morr"+u"\u00EA"+"ssemos": "morresse",
"morrermos": "morrer",
"mostramos": "mostrou",
"mudamos": "mudou",
"nascemos": "nasceu",
"nascermos": "nascer",
"necessit"+u"\u00E1"+"ssemos": "necessitasse",
"necessitamos": "necessitou",
"notamos": "notou",
"notarmos": "notar",
"observamos": "observou",
"observarmos": "observar",
"obtivemos": "obteve",
"oferecemos": "ofereceu",
"olh"+u"\u00E1"+"vamos": "olhava",
"olhamos": "olhou",
"olharmos": "olhar",
"optamos": "optou",
"organiz"+u"\u00E1"+"vamos": "organizava",
"ouv"+u"\u00ED"+"amos": "ouvia",
"ouvimos": "ouviu",
"ouvir"+u"\u00ED"+"amos": "ouviria",
"p"+u"\u00FA"+"nhamos": "punha",
"pag"+u"\u00E1"+"vamos": "pagava",
"pagamos": "pagou",
"pagaremos": "pagar"+u"\u00E1",
"par"+u"\u00E1"+"vamos": "parava",
"pararmos": "parar",
"parecemos": "pareceu",
"participamos": "participou",
"partimos": "partiu",
"partiremos": "partir"+u"\u00E1",
"pass"+u"\u00E1"+"ssemos": "passasse",
"pass"+u"\u00E1"+"vamos": "passava",
"passamos": "passou",
"passarmos": "passar",
"passearmos": "passear",
"pedimos": "pediu",
"pegamos": "pegou",
"pegarmos": "pegar",
"pens"+u"\u00E1"+"vamos": "pensava",
"pensamos": "pensa",
"pensarmos": "pensar",
"perceb"+u"\u00ED"+"amos": "percebia",
"percebemos": "percebeu",
"percorremos": "percorreu",
"perdemos": "perdeu",
"perderemos": "perder"+u"\u00E1",
"perdermos": "perder",
"perdoamos": "perdoou",
"perguntamos": "perguntou",
"pertencemos": "pertenceu",
"pod"+u"\u00ED"+"amos": "podia",
"podemos": "pode",
"poder"+u"\u00ED"+"amos": "poderia",
"poderemos": "poder"+u"\u00E1",
"pomos": "p"+u"\u00F5"+"e",
"por"+u"\u00ED"+"amos": "poria",
"possamos": "possa",
"possu"+u"\u00ED"+"mos": "possuiu",
"poupamos": "poupou",
"poupar"+u"\u00ED"+"amos": "pouparia",
"precis"+u"\u00E1"+"vamos": "precisava",
"precisamos": "precisa",
"precisar"+u"\u00ED"+"amos": "precisaria",
"precisarmos": "precisar",
"predissemos": "predisse",
"prefer"+u"\u00ED"+"amos": "preferia",
"preferimos": "preferiu",
"prendemos": "prende",
"prepar"+u"\u00E1"+"vamos": "preparava",
"preparamos": "preparou",
"preparar"+u"\u00ED"+"amos": "prepararia",
"presenciamos": "presenciou",
"prestamos": "prestou",
"pretend"+u"\u00ED"+"amos": "pretendia",
"pretendemos": "pretendeu",
"processamos": "processou",
"procuramos": "procurou",
"produzimos": "produziu",
"proferimos": "proferiu",
"projet"+u"\u00E1"+"vamos": "projetava",
"prometemos": "promete",
"propomos": "prop"+u"\u00F5"+"e",
"provaremos": "provar"+u"\u00E1",
"providenciamos": "providencia",
"publicar"+u"\u00ED"+"amos": "publicaria",
"pudermos": "puder",
"pusemos": "pus",
"puxamos": "puxou",
"quebr"+u"\u00E1"+"vamos": "quebrava",
"quer"+u"\u00ED"+"amos": "queria",
"queremos": "quer",
"querermos": "querer",
"quis"+u"\u00E9"+"ramos": "quisera",
"quis"+u"\u00E9"+"ssemos": "quisesse",
"quisemos": "quis",
"quisermos": "quiser",
"rebol"+u"\u00E1"+"vamos": "rebolava",
"rece"+u"\u00E1"+"vamos": "receava",
"receb"+u"\u00ED"+"amos": "recebia",
"recebemos": "recebeu",
"receberemos": "receber"+u"\u00E1",
"reconhecemos": "reconheceu",
"recordemos": "recorde",
"reduzimos": "reduziu",
"registramos": "registrou",
"regravamos": "regrava",
"reivindicamos": "reivindicou",
"remamos": "rema",
"repararmos": "reparar",
"repetimos": "repetiu",
"representamos": "representou",
"requeremos": "requere",
"resistimos": "resistiu",
"resolv"+u"\u00ED"+"amos": "resolvia",
"resolvemos": "resolveu",
"resolver"+u"\u00ED"+"amos": "resolveria",
"respeitamos": "respeitou",
"respir"+u"\u00E1"+"vamos": "respirava",
"respiramos": "respirou",
"respond"+u"\u00ED"+"amos": "respondia",
"respondermos": "responder",
"responsabilizamos": "responsabilizou",
"retesamos": "retesou",
"retomamos": "retomou",
"retribu"+u"\u00ED"+"mos": "retribuiu",
"retrocedermos": "retroceder",
"reunimos": "reuniu",
"reviv"+u"\u00ED"+"amos": "revivia",
"rezamos": "rezou",
"rezemos": "reze",
"roemos": "r"+u"\u00F3"+"i",
"rol"+u"\u00E1"+"vamos": "rolava",
"rolamos": "rolou",
"roubamos": "roubou",
"sa"+u"\u00ED"+"amos": "sa"+u"\u00ED"+"a",
"sa"+u"\u00ED"+"mos": "saiu",
"sa"+u"\u00ED"+"ssemos": "sa"+u"\u00ED"+"sse",
"sab"+u"\u00ED"+"amos": "sabia",
"sabemos": "sabe",
"saber"+u"\u00ED"+"amos": "saberia",
"saberemos": "saber"+u"\u00E1",
"sabermos": "saber",
"saboreamos": "saboreia",
"sacrificamos": "sacrifica",
"saibamos": "saiba",
"sairemos": "sair"+u"\u00E1",
"sairmos": "sair",
"sancionamos": "sanciona",
"saud"+u"\u00E1"+"vamos": "saudava",
"seguimos": "seguiu",
"seguramos": "segurou",
"seguraremos": "segurar"+u"\u00E1",
"sejamos": "seja",
"selecionaremos": "selecionar"+u"\u00E1",
"sent"+u"\u00ED"+"amos": "sentia",
"sentimos": "sentiu",
"sentirmos": "sentir",
"ser"+u"\u00ED"+"amos": "seria",
"seremos": "ser"+u"\u00E1",
"sermos": "ser",
"servimos": "serviu",
"sintamos": "sinta",
"sofremos": "sofreu",
"somos": ""+u"\u00E9",
"sonh"+u"\u00E1"+"vamos": "sonhava",
"sonhamos": "sonhou",
"sorr"+u"\u00ED"+"amos": "sorria",
"sorrimos": "sorriu",
"soub"+u"\u00E9"+"ssemos": "soubesse",
"soubemos": "soube",
"soubermos": "souber",
"superamos": "superou",
"suplicamos": "suplica",
"supomos": "supomos",
"supriremos": "suprir"+u"\u00E1",
"surpreendemos": "surpreendeu",
"suspeit"+u"\u00E1"+"vamos": "suspeitava",
"suspir"+u"\u00E1"+"vamos": "suspirava",
"suspiramos": "suspira",
"sustentamos": "sustenta",
"t"+u"\u00ED"+"nhamos": "tinha",
"tamos": "ta",
"teim"+u"\u00E1"+"vamos": "teimava",
"tem"+u"\u00ED"+"amos": "temia",
"tememos": "teme",
"temos": "tem",
"tencionamos": "tenciona",
"tenhamos": "tenha",
"tent"+u"\u00E1"+"ssemos": "tentasse",
"tentamos": "tentou",
"tentar"+u"\u00ED"+"amos": "tentaria",
"ter"+u"\u00ED"+"amos": "teria",
"teremos": "ter"+u"\u00E1",
"termin"+u"\u00E1"+"vamos": "terminava",
"terminamos": "termina",
"termos": "ter",
"testarmos": "testar",
"tir"+u"\u00E1"+"vamos": "tirava",
"tiramos": "tira",
"tiremos": "tire",
"tiv"+u"\u00E9"+"ssemos": "tivesse",
"tivemos": "tive",
"tivermos": "tiver",
"tocamos": "tocou",
"tom"+u"\u00E1"+"vamos": "tomava",
"tomamos": "tomou",
"tomaremos": "tomar"+u"\u00E1",
"tomarmos": "tomar",
"topamos": "topa",
"tornamos": "torna",
"trabalh"+u"\u00E1"+"vamos": "trabalhava",
"trabalhamos": "trabalhou",
"trajamos": "traja",
"transpir"+u"\u00E1"+"vamos": "transpirava",
"trat"+u"\u00E1"+"vamos": "tratava",
"tratamos": "trata",
"trataremos": "tratar"+u"\u00E1",
"traz"+u"\u00ED"+"amos": "trazia",
"trazemos": "traz",
"treinamos": "treina",
"trem"+u"\u00ED"+"amos": "tremia",
"trep"+u"\u00E1"+"ssemos": "trepasse",
"trilhamos": "trilha",
"trouxemos": "trouxe",
"us"+u"\u00E1"+"vamos": "usava",
"usamos": "usou",
"usufru"+u"\u00ED"+"amos": "usufru"+u"\u00ED"+"a",
"utiliz"+u"\u00E1"+"ssemos": "utilizasse",
"utilizamos": "utilizou",
"v"+u"\u00ED"+"amos": "via",
"v"+u"\u00ED"+"nhamos": "vinha",
"v"+u"\u00ED"+"ssemos": "visse",
"valemos": "vale",
"valoriz"+u"\u00E1"+"vamos": "valorizava",
"vamos": "vai",
"vaticinamos": "vaticina",
"vemos": "v"+u"\u00EA",
"vencemos": "venceu",
"vendemos": "vendeu",
"veremos": "ver"+u"\u00E1",
"verificamos": "verificou",
"vermos": "ver",
"vest"+u"\u00ED"+"amos": "vestia",
"viajamos": "viajou",
"viemos": "veio",
"vimos": "vem",
"viramos": "vira",
"viremos": "vire",
"virmos": "vir",
"visitemos": "visite",
"visualizamos": "visualiza",
"visualizarmos": "visualizar",
"viv"+u"\u00EA"+"ssemos": "vivesse",
"viv"+u"\u00ED"+"amos": "vivia",
"vivemos": "vive",
"vivenciamos": "vivencia",
"viveremos": "viver"+u"\u00E1",
"vivermos": "viver",
"voamos": "voou",
"voltamos": "voltou",
"voltar"+u"\u00ED"+"amos": "voltaria",
"voltaremos": "voltar"+u"\u00E1",
"voltarmos": "voltar",
"voltemos": "volte",
"votamos": "votou"}

# concEle - concordancia com o pronome Ele  - Nenhum de nos vamos
concEle = set(["abandonamos", "abdic"+u"\u00E1"+"ssemos", "abdicamos", "abra"+u"\u00E7"+"amos", "abrimos",
"acabamos", "aceitamos", "aceitar"+u"\u00ED"+"amos", "ach"+u"\u00E1"+"vamos", "achamos",
"acolhemos", "acompanhamos", "acordamos", "acredit"+u"\u00E1"+"vamos", "acreditamos",
"acrescentamos", "adivinhamos", "admiramos", "admitirmos", "adoramos",
"adotamos", "adotemos", "adquirimos", "adulterarmos", "advertimos",
"advogamos", "agimos", "agradecemos", "ajudamos", "ajudaremos",
"ajuntamos", "ajustamos", "alcan"+u"\u00E7"+u"\u00E1"+"vamos", "alimentamos", "alivi"+u"\u00E1"+"vamos",
"almo"+u"\u00E7"+"amos", "alug"+u"\u00E1"+"ssemos", "am"+u"\u00E1"+"vamos", "amamos", "amarramos",
"analis"+u"\u00E1"+"ssemos", "analisamos", "and"+u"\u00E1"+"vamos", "andamos", "anunciamos",
"apanhamos", "apare"+u"\u00E7"+"amos", "aparecemos", "aplaudimos", "aplicar"+u"\u00ED"+"amos",
"apodrecemos", "apoiamos", "aprendemos", "apresentamos", "apresentaremos",
"aprestamos", "aprov"+u"\u00E1"+"ssemos", "aprovamos", "aproveitamos", "arranj"+u"\u00E1"+"ssemos",
"arranjaremos", "arrastamos", "arrombamos", "aspiramos", "assinamos",
"assinarmos", "assistimos", "assistirmos", "associamos", "assumimos",
"atacamos", "atirar"+u"\u00ED"+"amos", "atravess"+u"\u00E1"+"vamos", "atravessamos", "aumentarmos",
"avaliamos", "avaliarmos", "avan"+u"\u00E7"+"amos", "bat"+u"\u00ED"+"amos", "batemos",
"batizamos", "bebemos", "beneficiamos", "botamos", "bradamos",
"brigamos", "brinc"+u"\u00E1"+"vamos", "brincamos", "buscamos", "ca"+u"\u00E7"+"oamos",
"ca"+u"\u00ED"+"mos", "ca"+u"\u00ED"+"ssemos", "cabemos", "cair"+u"\u00ED"+"amos", "calamos",
"calculamos", "caminhamos", "cantaremos", "carecemos", "carreg"+u"\u00E1"+"vamos",
"carregamos", "casamos", "cavalgamos", "celebramos", "cham"+u"\u00E1"+"vamos",
"chamamos", "checamos", "cheg"+u"\u00E1"+"vamos", "chegamos", "chegaremos",
"chegarmos", "cheguemos", "chor"+u"\u00E1"+"vamos", "choramos", "choremos",
"citamos", "cobramos", "colaboramos", "colocamos", "com"+u"\u00ED"+"amos",
"combateremos", "combin"+u"\u00E1"+"vamos", "combinamos", "come"+u"\u00E7"+"amos", "come"+u"\u00E7"+"armos",
"comemos", "comentamos", "comermos", "cometemos", "comparamos",
"completamos", "compr"+u"\u00E1"+"vamos", "compramos", "comprar"+u"\u00ED"+"amos", "compreendemos",
"comungamos", "concordamos", "concordaremos", "concorr"+u"\u00ED"+"amos", "condenamos",
"confeccionar"+u"\u00ED"+"amos", "confi"+u"\u00E1"+"vamos", "confirmarmos", "congelar"+u"\u00ED"+"amos", "conhec"+u"\u00ED"+"amos",
"conhecemos", "conjeturamos", "conquistamos", "consegu"+u"\u00ED"+"amos", "conseguimos",
"conseguir"+u"\u00ED"+"amos", "conseguiremos", "conseguirmos", "conservarmos", "conservemos",
"consider"+u"\u00E1"+"vamos", "consideramos", "considerarmos", "consigamos", "constatamos",
"constru"+u"\u00ED"+"mos", "contamos", "continu"+u"\u00E1"+"vamos", "continuamos", "continuaremos",
"contrat"+u"\u00E1"+"ssemos", "contrat"+u"\u00E1"+"vamos", "controlamos", "convers"+u"\u00E1"+"vamos", "conversamos",
"conversaremos", "convidamos", "cooperamos", "corremos", "correremos",
"costum"+u"\u00E1"+"vamos", "costumamos", "cotamos", "credenciamos", "cremos",
"cresc"+u"\u00ED"+"amos", "crescemos", "cri"+u"\u00E1"+"vamos", "criamos", "criarmos",
"criticamos", "cruzamos", "cuidamos", "culpamos", "cultivamos",
"cumprimos", "d"+u"\u00E1"+"vamos", "damos", "dan"+u"\u00E7"+"amos", "dan"+u"\u00E7"+u"\u00E1"+"ssemos",
"dan"+u"\u00E7"+u"\u00E1"+"vamos", "debitaremos", "decidimos", "decidiremos", "defend"+u"\u00ED"+"amos",
"defendemos", "definimos", "deitamos", "deixamos", "deixar"+u"\u00ED"+"amos",
"demonstramos", "demos", "denominamos", "denunciamos", "depositamos",
"derrubarmos", "descemos", "descendemos", "descobrimos", "descobriremos",
"descobrirmos", "desconfi"+u"\u00E1"+"vamos", "descoroamos", "desej"+u"\u00E1"+"vamos", "desejamos",
"desejar"+u"\u00ED"+"amos", "desempenhamos", "desentranhamos", "desenvolv"+u"\u00ED"+"amos", "desenvolvemos",
"designamos", "despach"+u"\u00E1"+"vamos", "desprezamos", "desrespeitarmos", "desvi"+u"\u00E1"+"vamos",
"detestamos", "dev"+u"\u00ED"+"amos", "devamos", "devemos", "dever"+u"\u00ED"+"amos",
"diagnosticaremos", "digamos", "dir"+u"\u00ED"+"amos", "diremos", "discut"+u"\u00ED"+"amos",
"dispomos", "dispormos", "disputamos", "diss"+u"\u00E9"+"ssemos", "dissemos",
"divagamos", "divergimos", "divid"+u"\u00ED"+"amos", "dividimos", "diz"+u"\u00ED"+"amos",
"dizemos", "dobramos", "dorm"+u"\u00ED"+"amos", "dormimos", "dormiremos",
"dublamos", "duplicamos", "elaboramos", "elegemos", "embargamos",
"embirramos", "empregamos", "encaramos", "enchemos", "encolh"+u"\u00ED"+"amos",
"encontramos", "encontraremos", "encontrarmos", "enjeitamos", "entend"+u"\u00EA"+"ssemos",
"entendemos", "entr"+u"\u00E1"+"vamos", "entramos", "entrarmos", "entregamos",
"entrevistamos", "enviamos", "enviar"+u"\u00ED"+"amos", "enxugamos", "enxugaremos",
u"\u00E9"+"ramos", "erguemos", "erramos", "escapamos", "escolh"+u"\u00ED"+"amos",
"escolhemos", "escond"+u"\u00ED"+"amos", "escorregamos", "escrev"+u"\u00ED"+"amos", "escut"+u"\u00E1"+"vamos",
"esgotamos", "especificamos", "esper"+u"\u00E1"+"vamos", "esperamos", "esquecemos",
"est"+u"\u00E1"+"vamos", "estabelecemos", "estabelecermos", "estamos", "estar"+u"\u00ED"+"amos",
"estaremos", "estarmos", "estejamos", "estimamos", "estiv"+u"\u00E9"+"ssemos",
"estivemos", "estivermos", "estornaremos", "estouramos", "estragamos",
"estranh"+u"\u00E1"+"ssemos", "estranhamos", "estranhemos", "estud"+u"\u00E1"+"vamos", "estudaremos",
"exercemos", "exigimos", "existimos", "expomos", "export"+u"\u00E1"+"vamos",
"expressamos", "fa"+u"\u00E7"+"amos", "fal"+u"\u00E1"+"vamos", "falamos", "falaremos",
"falarmos", "faltamos", "far"+u"\u00ED"+"amos", "faremos", "faz"+u"\u00ED"+"amos",
"fazemos", "fazermos", "fech"+u"\u00E1"+"vamos", "fecharmos", "fic"+u"\u00E1"+"ssemos",
"fic"+u"\u00E1"+"vamos", "ficamos", "ficaremos", "ficarmos", "filmamos",
"fingimos", "fiscalizamos", "fiz"+u"\u00E9"+"ssemos", "fizemos", "fizermos",
"flart"+u"\u00E1"+"vamos", "folg"+u"\u00E1"+"vamos", "fomos", "form"+u"\u00E1"+"vamos", "formamos",
"formos", "fornecemos", "fossemos", "frequent"+u"\u00E1"+"vamos", "fug"+u"\u00ED"+"ssemos",
"fugimos", "fum"+u"\u00E1"+"vamos", "fund"+u"\u00E1"+"vamos", "fundamos", "furtamos",
"ganh"+u"\u00E1"+"vamos", "ganhamos", "ganharemos", "garantiremos", "gastamos",
"gost"+u"\u00E1"+"vamos", "gostamos", "gostar"+u"\u00ED"+"amos", "gozamos", "graduamos",
"guardamos", "habitamos", "hav"+u"\u00ED"+"amos", "havemos", u"\u00ED"+"amos",
"ignor"+u"\u00E1"+"vamos", "imagin"+u"\u00E1"+"vamos", "imaginamos", "imitamos", "imitar"+u"\u00ED"+"amos",
"implantamos", "impomos", "importamos", "impregnamos", "impusemos",
"indicamos", "insist"+u"\u00ED"+"amos", "insistimos", "instalamos", "interferimos",
"interpretamos", "intervi"+u"\u00E9"+"ssemos", "invej"+u"\u00E1"+"vamos", "inventamos", "investimos",
"invocaremos", "ir"+u"\u00ED"+"amos", "iremos", "irmos", "jant"+u"\u00E1"+"ssemos",
"jant"+u"\u00E1"+"vamos", "jantamos", "jogamos", "julgamos", "juramos",
"l"+u"\u00EA"+"ssemos", "l"+u"\u00ED"+"amos", "laboramos", "lamentamos", "lan"+u"\u00E7"+"armos",
"lavamos", "lavarmos", "legalizar"+u"\u00ED"+"amos", "lembramos", "lemos",
"lev"+u"\u00E1"+"vamos", "levamos", "levantamos", "levaremos", "libertemos",
"lig"+u"\u00E1"+"vamos", "localizamos", "lustr"+u"\u00E1"+"vamos", "lutamos", "machucamos",
"mandamos", "mantemos", "marcamos", "marchamos", "marcharemos",
"mat"+u"\u00E1"+"ramos", "mat"+u"\u00E1"+"vamos", "matamos", "mentimos", "merec"+u"\u00ED"+"amos",
"merecemos", "metermos", "misturamos", "moldamos", "montar"+u"\u00ED"+"amos",
"mor"+u"\u00E1"+"vamos", "moramos", "moraremos", "moremos", "morr"+u"\u00EA"+"ssemos",
"morrermos", "mostramos", "mudamos", "nascemos", "nascermos",
"necessit"+u"\u00E1"+"ssemos", "necessitamos", "notamos", "notarmos", "observamos",
"observarmos", "obtivemos", "oferecemos", "olh"+u"\u00E1"+"vamos", "olhamos",
"olharmos", "optamos", "organiz"+u"\u00E1"+"vamos", "ouv"+u"\u00ED"+"amos", "ouvimos",
"ouvir"+u"\u00ED"+"amos", "p"+u"\u00FA"+"nhamos", "pag"+u"\u00E1"+"vamos", "pagamos", "pagaremos",
"par"+u"\u00E1"+"vamos", "pararmos", "parecemos", "participamos", "partimos",
"partiremos", "pass"+u"\u00E1"+"ssemos", "pass"+u"\u00E1"+"vamos", "passamos", "passarmos",
"passearmos", "pedimos", "pegamos", "pegarmos", "pens"+u"\u00E1"+"vamos",
"pensamos", "pensarmos", "perceb"+u"\u00ED"+"amos", "percebemos", "percorremos",
"perdemos", "perderemos", "perdermos", "perdoamos", "perguntamos",
"pertencemos", "pod"+u"\u00ED"+"amos", "podemos", "poder"+u"\u00ED"+"amos", "poderemos",
"pomos", "por"+u"\u00ED"+"amos", "possamos", "possu"+u"\u00ED"+"mos", "poupamos",
"poupar"+u"\u00ED"+"amos", "precis"+u"\u00E1"+"vamos", "precisamos", "precisar"+u"\u00ED"+"amos", "precisarmos",
"predissemos", "prefer"+u"\u00ED"+"amos", "preferimos", "prendemos", "prepar"+u"\u00E1"+"vamos",
"preparamos", "preparar"+u"\u00ED"+"amos", "presenciamos", "prestamos", "pretend"+u"\u00ED"+"amos",
"pretendemos", "processamos", "procuramos", "produzimos", "proferimos",
"projet"+u"\u00E1"+"vamos", "prometemos", "propomos", "provaremos", "providenciamos",
"publicar"+u"\u00ED"+"amos", "pudermos", "pusemos", "puxamos", "quebr"+u"\u00E1"+"vamos",
"quer"+u"\u00ED"+"amos", "queremos", "querermos", "quis"+u"\u00E9"+"ramos", "quis"+u"\u00E9"+"ssemos",
"quisemos", "quisermos", "rebol"+u"\u00E1"+"vamos", "rece"+u"\u00E1"+"vamos", "receb"+u"\u00ED"+"amos",
"recebemos", "receberemos", "reconhecemos", "recordemos", "reduzimos",
"registramos", "regravamos", "reivindicamos", "remamos", "repararmos",
"repetimos", "representamos", "requeremos", "resistimos", "resolv"+u"\u00ED"+"amos",
"resolvemos", "resolver"+u"\u00ED"+"amos", "respeitamos", "respir"+u"\u00E1"+"vamos", "respiramos",
"respond"+u"\u00ED"+"amos", "respondermos", "responsabilizamos", "retesamos", "retomamos",
"retribu"+u"\u00ED"+"mos", "retrocedermos", "reunimos", "reviv"+u"\u00ED"+"amos", "rezamos",
"rezemos", "roemos", "rol"+u"\u00E1"+"vamos", "rolamos", "roubamos",
"sa"+u"\u00ED"+"amos", "sa"+u"\u00ED"+"mos", "sa"+u"\u00ED"+"ssemos", "sab"+u"\u00ED"+"amos", "sabemos",
"saber"+u"\u00ED"+"amos", "saberemos", "sabermos", "saboreamos", "sacrificamos",
"saibamos", "sairemos", "sairmos", "sancionamos", "saud"+u"\u00E1"+"vamos",
"seguimos", "seguramos", "seguraremos", "sejamos", "selecionaremos",
"sent"+u"\u00ED"+"amos", "sentimos", "sentirmos", "ser"+u"\u00ED"+"amos", "seremos",
"sermos", "servimos", "sintamos", "sofremos", "somos",
"sonh"+u"\u00E1"+"vamos", "sonhamos", "sorr"+u"\u00ED"+"amos", "sorrimos", "soub"+u"\u00E9"+"ssemos",
"soubemos", "soubermos", "superamos", "suplicamos", "supomos",
"supriremos", "surpreendemos", "suspeit"+u"\u00E1"+"vamos", "suspir"+u"\u00E1"+"vamos", "suspiramos",
"sustentamos", "t"+u"\u00ED"+"nhamos", "tamos", "teim"+u"\u00E1"+"vamos", "tem"+u"\u00ED"+"amos",
"tememos", "temos", "tencionamos", "tenhamos", "tent"+u"\u00E1"+"ssemos",
"tentamos", "tentar"+u"\u00ED"+"amos", "ter"+u"\u00ED"+"amos", "teremos", "termin"+u"\u00E1"+"vamos",
"terminamos", "termos", "testarmos", "tir"+u"\u00E1"+"vamos", "tiramos",
"tiremos", "tiv"+u"\u00E9"+"ssemos", "tivemos", "tivermos", "tocamos",
"tom"+u"\u00E1"+"vamos", "tomamos", "tomaremos", "tomarmos", "topamos",
"tornamos", "trabalh"+u"\u00E1"+"vamos", "trabalhamos", "trajamos", "transpir"+u"\u00E1"+"vamos",
"trat"+u"\u00E1"+"vamos", "tratamos", "trataremos", "traz"+u"\u00ED"+"amos", "trazemos",
"treinamos", "trem"+u"\u00ED"+"amos", "trep"+u"\u00E1"+"ssemos", "trilhamos", "trouxemos",
"us"+u"\u00E1"+"vamos", "usamos", "usufru"+u"\u00ED"+"amos", "utiliz"+u"\u00E1"+"ssemos", "utilizamos",
"v"+u"\u00ED"+"amos", "v"+u"\u00ED"+"nhamos", "v"+u"\u00ED"+"ssemos", "valemos", "valoriz"+u"\u00E1"+"vamos",
"vamos", "vaticinamos", "vemos", "vencemos", "vendemos",
"veremos", "verificamos", "vermos", "vest"+u"\u00ED"+"amos", "viajamos",
"viemos", "vimos", "viramos", "viremos", "virmos",
"visitemos", "visualizamos", "visualizarmos", "viv"+u"\u00EA"+"ssemos", "viv"+u"\u00ED"+"amos",
"vivemos", "vivenciamos", "viveremos", "vivermos", "voamos",
"voltamos", "voltar"+u"\u00ED"+"amos", "voltaremos", "voltarmos", "voltemos",
"votamos"])


# NUMERAIS MASCULINO PARA COMPLEMENTAR COM == E UM ===
numEmasc  = set(["Bilh"+u"\u00E3"+"o", "bilh"+u"\u00E3"+"o",
"Bilh"+u"\u00F5"+"es", "bilh"+u"\u00F5"+"es", "Cento", "cento", "Cinquenta", "cinquenta",
"Duzentos", "duzentos", "Mil", "mil", "Milh"+u"\u00E3"+"o", "milh"+u"\u00E3"+"o",
"Milh"+u"\u00F5"+"es", "milh"+u"\u00F5"+"es", "Novecentos", "novecentos",
"Noventa", "noventa", "Oitenta", "oitenta", "Oitocentos", "oitocentos", "Quarenta", "quarenta",
"Quatrilh"+u"\u00E3"+"o", "quatrilh"+u"\u00E3"+"o", "Quatrilh"+u"\u00F5"+"es", "quatrilh"+u"\u00F5"+"es",
"Quatrocentos", "quatrocentos", "Quinhentos", "quinhentos", "Quintilh"+u"\u00E3"+"o",
"quintilh"+u"\u00E3"+"o", "Quintilh"+u"\u00F5"+"es", "quintilh"+u"\u00F5"+"es", "Seiscentos", "seiscentos",
"Sessenta", "sessenta", "Setecentos", "setecentos", "Setenta", "setenta", "Trezentos", "trezentos",
"Trilh"+u"\u00E3"+"o", "trilh"+u"\u00E3"+"o", "Trilh"+u"\u00F5"+"es", "trilh"+u"\u00F5"+"es",
"Trinta", "trinta", "Vinte", "vinte"])



# NUMERAIS FEMININO PARA COMPLEMENTAR COM == E UMA ====
numEfem  = set(["Bilh"+u"\u00E3"+"o", "bilh"+u"\u00E3"+"o",
"Bilh"+u"\u00F5"+"es", "bilh"+u"\u00F5"+"es", "Cento", "cento", "Cinquenta", "cinquenta",
"Duzentas", "duzentas", "Mil", "mil", "Milh"+u"\u00E3"+"o", "milh"+u"\u00E3"+"o",
"Milh"+u"\u00F5"+"es", "milh"+u"\u00F5"+"es", "Novecentas", "novecentas",
"Noventa", "noventa", "Oitenta", "oitenta", "Oitocentas", "oitocentas", "Quarenta", "quarenta",
"Quatrilh"+u"\u00E3"+"o", "quatrilh"+u"\u00E3"+"o", "Quatrilh"+u"\u00F5"+"es", "quatrilh"+u"\u00F5"+"es",
"Quatrocentas", "quatrocentas", "Quinhentas", "quinhentas", "Quintilh"+u"\u00E3"+"o",
"quintilh"+u"\u00E3"+"o", "Quintilh"+u"\u00F5"+"es", "quintilh"+u"\u00F5"+"es", "Seiscentas", "seiscentas",
"Sessenta", "sessenta", "Setecentas", "setecentas", "Setenta", "setenta", "Trezentas", "trezentas",
"Trilh"+u"\u00E3"+"o", "trilh"+u"\u00E3"+"o", "Trilh"+u"\u00F5"+"es", "trilh"+u"\u00F5"+"es",
"Trinta", "trinta", "Vinte", "vinte"])


# NUMERAIS FEMININO == duzentas canetas ==============
numFem  = set(["Catorze", "catorze", "Cem", "cem", "Cinco", "cinco",
"Cinquenta", "cinquenta", "Dez", "dez", "Dezenove", "dezenove", "Dezesseis", "dezesseis",
"Dezessete", "dezessete", "Dezoito", "dezoito", "Doze", "doze",  "Duas", "duas",
"Duzentas","duzentas", "Mil", "mil", "Nove", "nove", "Novecentas", "novecentas",
"Noventa", "noventa", "Oitenta", "oitenta", "Oito", "oito", "Oitocentas", "oitocentas",
"Onze", "onze", "Quarenta", "quarenta", "Quatorze", "quatorze", "Quatro", "quatro",
"Quatrocentas", "quatrocentas", "Quinhentas", "quinhentas", "Quinze", "quinze",
"Seis", "seis", "Seiscentas", "seiscentas", "Sessenta",  "sessenta",
"Sete", "sete", "Setecentas", "setecentas", "Setenta", "setenta",
"Tr"+u"\u00EA"+"s", "tr"+u"\u00EA"+"s", "Treze", "treze",
"Trezentas", "trezentas", "Trinta", "trinta",
"Vinte", "vinte"])



# NUMERAIS MASCULINO == duzentas canetas ==============
numMasc  = set(["Catorze", "catorze", "Cem", "cem", "Cinco", "cinco",
"Cinquenta", "cinquenta", "Dez", "dez", "Dezenove", "dezenove", "Dezesseis", "dezesseis",
"Dezessete", "dezessete", "Dezoito", "dezoito", "Dois", "dois", "Doze", "doze",
"Duzentos","duzentos", "Mil", "mil", "Nove", "nove", "Novecentos", "novecentos",
"Noventa", "noventa", "Oitenta", "oitenta", "Oito", "oito", "Oitocentos", "oitocentos",
"Onze", "onze", "Quarenta", "quarenta", "Quatorze", "quatorze", "Quatro", "quatro",
"Quatrocentos", "quatrocentos", "Quinhentos", "quinhentos", "Quinze", "quinze",
"Seis", "seis", "Seiscentos", "seiscentos", "Sessenta",  "sessenta",
"Sete", "sete", "Setecentos", "setecentos", "Setenta", "setenta",
"Tr"+u"\u00EA"+"s", "tr"+u"\u00EA"+"s", "Treze", "treze",
"Trezentos", "trezentos", "Trinta", "trinta",
"Vinte", "vinte"])


# NUMERAIS FEMININO-MASCULINO == duzentos canetas == trezentas carros =======
nuMfmoas  = set(["Duzent","duzent", "Novecent", "novecent",
"Oitocentas", "oitocent", "Quatrocent", "quatrocent", "Quinhent",
"quinhentas", "Seiscent", "seiscent", "Setecent",
"setecent", "Trezent", "trezent"])

# PALAVRAS MASCULINAS == terminadas em [aeiou]rao =til ao= mulherao, caldeirao, = calorao ==
plvrMrao  = set(["abeir", "ailer", "air", "albacor",
"albardeir", "alcatr", "alcor","alfarrobeir",
"algir", "almeir", "alpalheir", "alqueir", "alvador",
"alvadur", "amir", "angular", "ar", "arur", "asneir", "azar","bafor",
"bair", "baleeir", "bandeir", "bar", "beir", "beltr", "besour", "betatr",
"biar", "biber", "biqueir", "bonacheir", "boqueir", "botir", "brancar",
"Brasileir", "brasileir", "brejeir", "bur", "butir", "cadeir", "Caldeir",
"caldeir", "calmeir", "calor", "camar", "cambur", "canastr", "cangueir",
"canjir", "cantimar", "capeir", "capoeir", "caracanir", "car", "carreir",
"casar", "cavaleir", "caveir", "cavir", "centr", "chamborreir",
"champorreir", "chapeir", "chapeleir", "chaporreir", "char", "charqueir",
"chaveir", "cheir", "chor", "choror", "chuveir", "ciclotr", "ciger",
"cintur", "cir", "cisir", "citeropter", "cizir", "clar", "coentr", "coer",
"coir", "companheir", "congueir", "contempor", "convier", "cor", "cormor",
"coscor", "costur", "cour", "cur", "deuter", "diamor", "diatessar", "dinheir",
"doutor", "dur", "ebur", "electr", "eler", "encontr", "eriger", "escadeir",
"escaleir", "escor", "escur", "esgueir", "espadeir", "espor", "esteir",
"estir", "estupor", "estur", "faceir", "fachar", "falastr", "faveir", "feir",
"fever", "figur", "flaqueir", "flor", "fortalheir", "fraqueir", "frieir",
"fur", "galeir", "galer", "gar", "gaspar", "gir", "gizir", "gor", "gorgor",
"goteir", "grossalheir", "grosseir", "har", "hiper", "hor", "interfer",
"jacatir", "jaguar", "jaguatir", "jangueir", "jeir", "jur", "labor", "ladeir",
"lafar", "lagueir", "lamar", "lambeir", "lameir", "lampar", "langueir",
"largueir", "lazar", "leir", "lezir", "ligueir", "linguar", "lingueir",
"linhar", "lir", "liseir", "listr", "longeir", "longer", "longueir", "lumar",
"maceir", "macor", "madraceir", "madur", "magreir", "malheir", "manguar",
"mangueir", "manjer", "mar", "martir", "masseir", "matir", "mazorreir",
"megafeir", "milhar", "milheir", "milher", "Mineir", "mir", "moir",
"molangueir", "molanqueir", "moleir", "mor", "mour", "mulher", "muquir",
"mutir", "muxir", "napeir", "naper", "neutr", "olheir", "palheir", "palmeir",
"pandeir", "par", "parreir", "parvoeir", "patr", "pecador", "peir", "peneir",
"pequir", "percher", "perchur", "piapar", "piquir", "pir", "plasteir",
"plastr", "plastreir", "poceir", "poeir", "polar", "poleir", "poltr", "por",
"positr", "potir", "punxir", "putir", "puxir", "quarteir", "quarter",
"quinteir", "rafeir", "regueir", "ribeir", "salmoir", "salmour", "sanhar",
"sapir", "sar", "satir", "sear", "seir", "sendeir", "sensabor", "simplacheir",
"simplalheir", "simpleir", "sincrotr", "solteir", "sopeir", "superver",
"taleir", "tambolar", "tambur", "taper", "tar", "tempor", "tesoir", "tesour",
"tinhor", "tintur", "tir", "toir", "toleir", "tor", "toupeir", "tour",
"trigueir", "tronqueir", "tubar", "tucur", "vair", "valdur", "valeir",
"vaner", "var", "vaseir", "vassoir", "vazeir", "ver", "verber", "vibor",
"votur", "vozeir", "zar", "zombeir"])


# PALAVRAS MASCULINAS == terminadas em IDA = inseticida, pesticida
excplvrIda  = set(["alme", "almor"+u"\u00E1"+"v", "Ant"+u"\u00E1"+"rt", "ap"+u"\u00E1"+"tr",
"aquem"+u"\u00EA"+"n", u"\u00E1"+"r", "ars"+u"\u00E1"+"c", "Atl"+u"\u00E2"+"nt",
"aven", u"\u00E1"+"v", "bacteric", "bel", "bioc", "br", "br"+u"\u00ED"+"g",
"cant"+u"\u00E1"+"r", "car"+u"\u00F3"+"t", "cefe", "cianam", "ciclofosfam",
"coinc", "col", "consol", "contrapart", "conv", "cris"+u"\u00E1"+"l", "cu",
"dec", "div", "d"+u"\u00ED"+"v", "duv", "d"+u"\u00FA"+"v", "el", "eluc",
"endiv", "endo", "engrav", "enegrec", "env", "erm", "espermic",
"estup", "filic", "f"+u"\u00F3"+"c", "formam", "formic", "fratric", "fungic",
"genoc", "germic", "grav", "gu", "herbic", "homic", "inc", "infantic",
"insectic", "insetic", "intim", "inval", "isoniaz", "lap", "l"+u"\u00E1"+"p",
"liqu", "liv", "luc", "maldorm", "margar", "m", "nere", "n", "ox",
"par"+u"\u00F3"+"t", "parric", "pestic", "plac", "poliacrilam", "pres",
"progr", "quir", "ratic", "regic", "reinc", "res", "sass"+u"\u00E2"+"n",
"sel"+u"\u00EA"+"uc", "selj"+u"\u00FA"+"c", "sol", "subdiv", "suic",
"sulfam", "talidom", "t"+u"\u00E1"+"ur", "teba", "tiranic", "transgr",
"transluc", "trep", "tropicam", "truc", "uxoric", "ven", "v", "za"])


# PALAVRAS REPETIDAS == EXCECAO
excpLvrep  = set(["PARA", "Para", "para", "PELA", "Pela", "pela", "PELAS", "Pelas", "pelas",
"PELO", "Pelo", "pelo", "PELOS", "Pelos", "pelos"])


# PALAVRAS MASCULINAS TERMINADAS EM  IVA == EXCECAO
excpLvriva  = set(["adject", "adjet", "alque", "arqu", "at", "av", "cat",
"conv", "cr", "cult", "der", "desarqu", "desat", "desmot", "efet", "e",
"enra", "esqu", "incent", "invect", "mot", "Ne", "no", "object", "objet",
"pa", "pr", "qued", "reat", "reav", "rev", "sara", "Sara", "u", "v"])

# PALAVRAS TERMINADAS EM  ADA == EXCECAO
excpLvrada  = set(["Andr", "c", "Camar", "camar", "Lus"+u"\u00ED",
"lus"+u"\u00ED", "n", "n"+u"\u00F4"+"m"])

# PALAVRAS FEMININAS TERMINADAS EM  NC,A == EXCECAO = venc,a - presenc,a - ananc,a
excpLvrnca  = set(["acarr", "acri", "ader", "afi", "agrav", "alc", "am",
"ape", "aquer", "arrom", "av", "av", "bag", "bal", "bon", "b", "ca",
"c", "conv", "d", "desconv", "destr", "det", "detr", "el", "embal",
"eng", "enj", "enr", "entr", "entr", "esbal", "esbr", "esc", "escarv",
"escarv", "esfr", "esfreg", "esmi", "esper", "esp", "estr", "estr",
"estr", "ev", "fem", "furd", "g", "gar", "inj", "l", "nu", "parr",
"pert", "perv", "pi", "p", "p", "p", "r", "rasp", "reconv", "resp",
"r", "rip", "segur", "sobr", "tr", "v"])

# PALAVRAS DE DOIS GENEROS TERMINADAS EM  E == paranaense - elegante - galante
plvrMfe = set(["aaleniens", "aalens", "aaquenens", "abacatens",
"abacaxiens", "abacens", "abadens", "abadianens",
"abadiens", "abadinens", "abaeteens", "abaetetubens",
"abaetezinhens", "abaiarens", "abaibens", "abairens",
"abaitinguens", "abambresens", "abapanens", "abareens",
"abatiaens", "abderens", "abdonens", "abecedens",
"abelardens", "abelens", "abelhens", "aberrant",
"abertanens", "abevilens", "abicens", "abismens",
"abitinens", "abitureirens", "abiulens", "ablitens",
"aboborens", "abor"+u"\u00ED"+"gen", "abor"+u"\u00ED"+"gin", "abraamens",
"abra"+u"\u00E3"+"ozens", "abranchens", "abrangent", "abrantens",
"abr"+u"\u00E3"+"oens", "abrasant", "abreuens", "absorvent",
"abstinent", "abufariens", "abulens", "abun"+u"\u00E3"+"ens",
"abunanens", "abundanciens", "abundant", "abuniens",
"aca"+u"\u00E3"+"ens", "acaciens", "acadiens", "acaiacens",
"acaiaquens", "a"+u"\u00E7"+"aiens", "a"+u"\u00E7"+"ailandens", "a"+u"\u00E7"+"ailandiens",
"a"+u"\u00E7"+"aitubens", "a"+u"\u00E7"+"aizalens", "acajutibens", "acampamentens",
"acan"+u"\u00E3"+"ens", "acananens", "acaraens", "acarapens",
"acarauens", "a"+u"\u00E7"+"areens", "acariens", "aceguaens",
"acesitens", "acheulens", "aciolens", "acmonens",
"acompanhant", "aconchegant", "a"+u"\u00E7"+"orens", "a"+u"\u00E7"+"oriens",
"acrens", "acreunens", "a"+u"\u00E7"+"ucenens", "a"+u"\u00E7"+"uens",
"adamantinens", "adelaidens", "adelandens", "aderent",
"adjacent", "adolescent", "adotant", "adrianopolens",
"adstringent", "adustinens", "afifens", "afogadens",
"afraniens", "afrodescendent", "afrodisiens", "afuaens",
"agaunens", "agent", "aginens", "agirinens",
"agitant", "agonens", "agonizant", "agravant",
"agrest", "agrestinens", "agricolandens", "agridoc",
"agripens", "agripinens", "agrolandens", "aguanilens",
"agudens", "aguiarens", "aimoreens", "aiquarens",
"air"+u"\u00E3"+"oens", "aiuabens", "aiuruoquens", "ajudant",
"ajuricabens", "alabandens", "alabanens", "alanceant",
"alandroalens", "alantens", "alarmant", "alasquens",
"alavens", "albacetens", "albanens", "albens",
"albergariens", "albertinens", "albicastrens", "albicens",
"albigens", "albingaunens", "albinitens", "albintimiliens",
"albufeirens", "albuquerquens", "alcacerens", "alcalarens",
"alcanenens", "alcantarens", "alcantilens", "alceciens",
"alcetiens", "alcobacens", "alcoutenens", "alcoutinens",
"aldeens", "aldegalens", "aldematens", "alecrinens",
"alegrens", "alegret", "alegretens", "alegriens",
"alenquerens", "aletriniens", "alexandriens", "alexanens",
"alfamens", "alfandeguens", "alfenens", "alfredens",
"algarbiens", "algarviens", "algidens", "algonquiens",
"algonquinens", "alhadens", "alhandrens", "aliancens",
"aliciant", "alijoens", "aliterant", "aljezurens",
"aljubarrotens", "aljustrelens", "almadens", "almeidens",
"almeiriens", "almeirinens", "almenarens", "almens",
"almescarens", "almesquinhens", "almocens", "almodovarens",
"almofadens", "aloandiens", "alpalhoens", "alpens",
"alpercatens", "alpestrens", "alpiarcens", "alpinopolens",
"alsiens", "altairens", "altamirens", "altaneirens",
"alteadens", "altens", "alterens", "alterosens",
"altinens", "altinopolens", "altissonant", "altoens",
"altoniens", "altuens", "alucinant", "alunens",
"alurens", "alutrens", "alutriens", "alva"+u"\u00E7"+u"\u00E3"+"ozens",
"alvaiazerens", "alvaranens", "alvarelhens", "alvarelitens",
"alvarenguens", "alvarens", "alvarinens", "alvejant",
"alvens", "alverquens", "alvesens", "alvilandiens",
"alvinitent", "alvinlandens", "alvinlandiens", "alvinopolens",
"alvinopolitens", "alvitens", "alvoradens", "amacianens",
"amadorens", "amananens", "amandabens", "amandens",
"amanhecens", "amaniuens", "amant", "amanuens",
"amapaens", "amaporanens", "amaporens", "amarajiens",
"amaralens", "amarantens", "amarantinens", "amarelens",
"amarens", "amargosens", "amatariens", "amatauraens",
"amaturaens", "amazonens", "amazoniens", "ambianens",
"ambivalent", "ambl"+u"\u00ED"+"op", "ambraciens", "ambrosiens",
"ambulant", "amedarens", "ameixens", "ameliens",
"ameliopolens", "americanens", "amienens", "amiens",
"amiternens", "amitinens", "amizadens", "amocens",
"amolarens", "amoniens", "amontadens", "amorescens",
"amorinopolens", "amorosens", "amparens", "amperens",
"amuamuaipens", "anabolizant", "anacletens", "anadiens",
"anajaens", "anajasens", "anajatenens", "anajatubens",
"anajens", "analandens", "analandiens", "anamanens",
"anamariens", "ananasens", "ananatubens", "ananidens",
"ananindeuens", "anapuens", "anapuruens", "anarquizant",
"anastaciens", "anatiens", "anauaens", "anaurilandens",
"ancarens", "anchietens", "ancianens", "ancoriens",
"andant", "andaraiens", "andaricens", "andesens",
"andiraens", "andirens", "andirobalens", "andirobens",
"andologens", "andorrens", "andradens", "andradinens",
"andreens", "andrelandens", "anelant", "aneliens",
"anemuriens", "anemutuens", "angaiarens", "angaiens",
"angatubens", "angaturamens", "angejens", "angelens",
"angelicens", "angelinens", "angicalens", "angicanens",
"angic"+u"\u00E3"+"ozens", "angicens", "angiquens", "angolens",
"angrens", "anguerataens", "anguerens", "angusturens",
"anhanduiens", "anhanduizinhens", "anhangaens", "anhangaiens",
"anhanguens", "anhanguerens", "anhembiens", "anhumens",
"anicunens", "aninhens", "anisiens", "anitapolens",
"aniversariant", "anoria"+u"\u00E7"+"uens", "anoriens", "anseriform",
"ansianens", "antecedent", "antens", "anticariens",
"anticirens", "anticoagulant", "anticongelant", "antigen",
"antigonens", "antilhens", "antinhens", "antioquens",
"antioquiens", "antioxidant", "antofagastens", "antoninens",
"antuerpiens", "anunciant", "anutibens", "apach",
"apaixonant", "apajatubens", "aparecidens", "apavorant",
"aperibens", "apeuens", "apiaiens", "apinajeens",
"apodiens", "apoiant", "apolinariens", "apolonidens",
"apolonidiens", "aporaens", "aporeens", "aporemens",
"aprazibilens", "aprazivelens", "aproagens", "aproaguens",
"apuaens", "apuaremens", "apucaranens", "apucaraninhens",
"apuiens", "apuliens", "apurimaens", "aquabonens",
"aquaplatens", "aquens", "aquicaldens", "aquidabanens",
"aquidauanens", "aquileens", "aquinens", "aquirabens",
"aquirasens", "aquitanens", "arabelens", "arabricens",
"arabrigens", "arabutanens", "ara"+u"\u00E7"+"a"+u"\u00E7"+"uens", "ara"+u"\u00E7"+"aens",
"ara"+u"\u00E7"+"aibens", "ara"+u"\u00E7"+"aiens", "ara"+u"\u00E7"+"ajiens", "aracajuens",
"ara"+u"\u00E7"+"ariguamens", "aracatia"+u"\u00E7"+"uens", "aracatiarens", "aracatiens",
"ara"+u"\u00E7"+"atubens", "aracatuens", "araciens", "aracitabens",
"aracoiabens", "aracrucens", "aracruzens", "aragarcens",
"aragoianens", "aragonens", "araguacemens", "araguaciens",
"aragua"+u"\u00E7"+"uens", "araguaiarens", "araguaiens", "araguainens",
"araguainhens", "araguananens", "araguapacens", "araguariens",
"araguatinens", "arainens", "araiosens", "arajarens",
"aramariens", "arambareens", "araminens", "arananens",
"aranauens", "aranduens", "aranhens", "arant",
"arantinens", "arapaens", "arapariens", "arapeiens",
"arapiracens", "arapiraquens", "arapirens", "arapoemens",
"araponguens", "araporanens", "arapotiens", "arapuaens",
"araquaens", "araquariens", "araquenens", "araraiens",
"araranguaens", "ararapirens", "araraquarens", "ararendaens",
"ararens", "araricaens", "arariens", "araripens",
"araripinens", "arariusens", "araruamens", "araruens",
"ararunaquarens", "ararunens", "araruvens", "aratacens",
"aratamens", "aratibens", "araticuens", "aratingaubens",
"aratubens", "aratuipens", "arauaens", "araucariens",
"arauens", "araujens", "araunens", "araxaens",
"arborescent", "arcadens", "arceburguens", "arcelinens",
"arcens", "arcobotant", "arcobrigens", "arcoens",
"arcoverdens", "arcozelens", "arcuens", "ardenens",
"ardent", "areadens", "arealvens", "areanens",
"are"+u"\u00E3"+"ozens", "areens", "areinhens", "areiopolens",
"arembepens", "arenalens", "arenapolens", "arequembauens",
"arerens", "aresens", "arfant", "arganilens",
"argenitens", "argentens", "argiritens", "argoimens",
"argoinens", "argolens", "argoncilhens", "ariauens",
"aricanduvens", "aricanguens", "ariciens", "aricorens",
"arimanens", "ariminens", "arinens", "aripibriens",
"aripuanens", "ariquemens", "ariranhens", "aririens",
"aristeuzens", "aristidens", "arisuelens", "aritagnaens",
"arizonens", "arma"+u"\u00E7"+u"\u00E3"+"ozens", "armamarens", "armazenens",
"arneirosens", "arnoniens", "aroasens", "aroeirens",
"aromatizant", "arouquens", "arpoarens", "arquejant",
"arquens", "arraialens", "arraiolens", "arreens",
"arrematant", "arrependidens", "arrepiant", "arrogant",
"arrojadens", "arronchens", "arrozalens", "arrozeirens",
"arrozens", "arrudens", "arrulhant", "artemisens",
"art"+u"\u00ED"+"fic", "aruajaens", "aruananens", "aruaruens",
"aruasens", "aruauens", "aruens", "arujaens",
"arumandubens", "arumanens", "aruminens", "aruntens",
"arutiens", "arvens", "arvernens", "arvoredens",
"arvorezinhens", "ascendent", "asclep"+u"\u00ED"+"ad", "asconens",
"ascurrens", "asdrubalens", "asfixiant", "asidonens",
"aspasiens", "aspirant", "assaltant", "assiens",
"assinant", "assisens", "assistenciens", "assistent",
"assistuens", "assoant", "assuncionens", "astapens",
"astartiens", "astens", "astiens", "astipalens",
"astolfens", "astorguens", "astrapeens", "asturicens",
"atacant", "atafonens", "atalaiens", "atalantens",
"ataleens", "atalens", "atalhens", "ataquens",
"ataubens", "atendent", "ateniens", "atenuant",
"aternens", "aterradens", "aterradinhens", "atia"+u"\u00E7"+"uens",
"atibaiens", "atiliens", "atimirinens", "atimirinzens",
"atinent", "atiradens", "atlant", "atoladeirens",
"atoleirens", "atorment", "atraent", "atriens",
"atuant", "atubens", "aturiaiens", "augustobrigens",
"augustodunens", "aurelianens", "aurens", "auriens",
"auriflamens", "aurignacens", "aurilandens", "aurilandiens",
"aurinhacens", "auriverdens", "aurizonens", "auroens",
"aurorens", "ausent", "ausentens", "ausonens",
"autasasens", "autazens", "autocolant", "aut"+u"\u00F3"+"cton",
"autoimun", "automedont", "autossuficient", "avaiens",
"avanhandavens", "avareens", "avaricens", "avatinquarens",
"avelarens", "avencens", "avens", "aviacionens",
"aviltant", "avisens", "avoant", "axininens",
"axininzens", "axixaens", "azambujens", "azauriens",
"azeitonens", "azevedens", "azilens", "azuritens",
"baamens", "baba"+u"\u00E7"+"uens", "baba"+u"\u00E7"+"ulandiens", "baba"+u"\u00E7"+"uzens",
"babanens", "babiens", "babiloniens", "bacabalens",
"bacabens", "bacadens", "bacaetavens", "bacainens",
"bacatubens", "bacatuens", "bacaubens", "bacaxaens",
"bacelaens", "bacelarens", "bacuriens", "bacuriteuens",
"bacuritiens", "bacuritubens", "bacurizinhens", "badajozens",
"badenens", "baependiens", "baetens", "bageens",
"bagrens", "bagua"+u"\u00E7"+"uens", "baguariens", "bahamens",
"baianens", "bai"+u"\u00E3"+"ozens", "baiens", "baiepanens",
"bailens", "bailiquens", "baionens", "bajeens",
"bajulant", "balaiadens", "balaieirens", "balan"+u"\u00E7"+"ant",
"balbuciant", "baleens", "baleiens", "balens",
"baliniens", "balizens", "baliziens", "balneariens",
"baloi"+u"\u00E7"+"ant", "balou"+u"\u00E7"+"ant", "balsamens", "balsens",
"baltasarens", "baluartens", "bambeant", "bamboant",
"bamboleant", "bambuiens", "banabuinens", "bananalens",
"bananeirens", "bananens", "bandarrens", "bandeirant",
"bandeirantens", "bandeirens", "bangladens", "banguens",
"bannaquens", "banquetens", "banzaens", "baracajaens",
"baraunens", "barbacenens", "barbadens", "barbalhens",
"barbarens", "barbosens", "barcadens", "barcarenens",
"barcelens", "barcelonens", "barcinonens", "bardariens",
"baremens", "bargant", "baririens", "baritens",
"baronens", "barquens", "barquinhens", "barrac"+u"\u00E3"+"ozens",
"barracens", "barraconens", "barradens", "barragenens",
"barranceirens", "barranquens", "barraquens", "barraquinhens",
"barreir"+u"\u00E3"+"ozens", "barreirens", "barreirinhens", "barrens",
"barrentens", "barrestivens", "barretens", "barretinens",
"barrigadens", "barrigudens", "barrinhens", "barro"+u"\u00E7"+u"\u00E3"+"ozens",
"barrocassalens", "barrolandens", "barroquens", "barroquinhens",
"barrosens", "baruens", "barueriens", "baruriens",
"basculant", "basiliens", "bassanens", "bastens",
"bastianens", "basti"+u"\u00E3"+"ozens", "bastosens", "batagua"+u"\u00E7"+"uens",
"bataiporanens", "batalhens", "batataens", "batataiens",
"batatalens", "batateirens", "batatubens", "bateens",
"bateiens", "batel"+u"\u00E3"+"ozens", "bateriens", "batinguens",
"batoquens", "batoviens", "bauens", "bauinens",
"baunilhens", "bauruens", "bautens", "bauxiens",
"bearnens", "beatificant", "bebedourens", "beberibens",
"beculonens", "beirens", "bejens", "bejuens",
"belchiorens", "belemens", "belenens", "belezens",
"beligerant", "belisariens", "belizens", "belmontens",
"belterrens", "belunens", "belvederens", "bempostens",
"benacens", "benaventens", "bendizent", "beneditens",
"beneditinens", "beneficent", "beneleitens", "benemerent",
"benevenutens", "benevolent", "benfazent", "benfiquens",
"bengalens", "benguelens", "beninens", "bentabreuens",
"bentens", "bentinhens", "bequimoens", "bequimonens",
"berilens", "beriquiens", "beritens", "berlinens",
"bermudens", "bernardens", "bernardinens", "bernens",
"bertinhens", "bertioguens", "bertolinens", "beruriens",
"besourens", "bestificant", "betabloqueant", "betaniens",
"beterrens", "betimens", "betinens", "betuliens",
"betulonens", "beverniens", "bexiguens", "bezerrens",
"biafrens", "bibarrens", "biboquens", "bicanquens",
"bicaquens", "bicasens", "bicudens", "bicuibens",
"bidens", "bien", "biform", "bifront",
"bifurqu", "bigua"+u"\u00E7"+"uens", "biguazinhens", "bilacens",
"bilaquens", "bilhostr", "bilidens", "biliens",
"bil"+u"\u00ED"+"ngu", "biltr", "binguens", "biocenos",
"bioluminescent", "bipen", "biquens", "biquinhens",
"birbant", "biribeirens", "biribiriens", "biriguiens",
"biririquens", "biritibens", "biritinguens", "birmanens",
"biscaiens", "bissanens", "bissauens", "bitupitaens",
"bituranens", "bituriens", "biturunens", "bivalent",
"bixopaens", "bizarrens", "bloqueant", "boaciquens",
"boa"+u"\u00E7"+"uens", "boapabens", "boavianens", "bocainens",
"bocaiuvens", "bocejant", "bo"+u"\u00E7"+"oroquens", "bodincomagens",
"bodocoens", "bodocongoens", "bodoens", "bodoquenens",
"bodotens", "boegua"+u"\u00E7"+"uens", "bofetens", "bogua"+u"\u00E7"+"uens",
"boia"+u"\u00E7"+"uens", "boiant", "boiarucens", "boiaruquens",
"boinens", "boiteuxburguens", "boituvens", "bojuruens",
"bolachens", "bolamens", "bolcheviqu", "bolichens",
"bombarralens", "bonaerens", "bonens", "bonfinens",
"bonhuens", "bonifaciens", "boninalens", "bonitens",
"bonjesuens", "bononiens", "bonretirens", "bonsucessens",
"boqueir"+u"\u00E3"+"ozens", "boqueirens", "boquiens", "boquinens",
"boquirens", "boraceens", "boraens", "borbens",
"borboletens", "borboremens", "borbulhant", "bordalens",
"bordelens", "born", "bosens", "bosnens",
"bosniens", "bosquens", "bosqu"+u"\u00ED"+"man", "bostonens",
"botafoguens", "botelhens", "botens", "botiquens",
"botocudens", "botucatuens", "bovilens", "boximan",
"bracarens", "bracejant", "bracens", "bragancens",
"bragantinens", "braguens", "br"+u"\u00E2"+"man", "bram",
"branacens", "brancens", "brandonens", "brandurens",
"branquens", "branquinhens", "brasabrantens", "brasens",
"brasilandens", "brasilandiens", "brasileens", "brasileiens",
"brasileirens", "brasilens", "brasiliens", "brasinens",
"brasiolens", "brasitanens", "brasopolens", "braunens",
"bravens", "brej"+u"\u00E3"+"ozens", "brejatubens", "brejaubens",
"brejauvens", "brejens", "brejetubens", "brejinhens",
"brejoens", "brejonens", "bresciens", "bressanens",
"bretens", "brev", "brevens", "brigadens",
"brigant", "brigidens", "briguens", "brilhant",
"brilhantens", "brilhantinens", "brincant", "britanens",
"britaniens", "britens", "brixiens", "brococoens",
"brodosquiens", "brotens", "brumadens", "brumadiens",
"brumadinhens", "brumalens", "brundueens", "brusquens",
"brutamont", "bruxelens", "bruxuleant", "buarens",
"bucarestens", "bucuitubens", "budapestens", "buenairens",
"buenens", "buenolandens", "buenopolens", "bueraremens",
"bufadeirens", "bufant", "bugrens", "buiquens",
"bujaruens", "bulhonens", "bulidens", "bupevens",
"buquinens", "buracicens", "buraciquens", "buranhenens",
"buraramens", "burdegalens", "burdigalens", "burichens",
"buriens", "burietaens", "buritamens", "buritiens",
"buritiramens", "buritiranens", "buritizalens", "buritizeirens",
"buritizinhens", "burizeirens", "burquinens", "bursaonens",
"burundiens", "butanens", "butiaens", "caagua"+u"\u00E7"+"uens",
"caapiranguens", "caaporanens", "caatibens", "caatinguens",
"cabaceirens", "cabacens", "cabalianens", "cabanens",
"cabe"+u"\u00E7"+"anens", "cabeceirens", "cabecens", "cabe"+u"\u00E7"+"udens",
"cabedelens", "cabeleirens", "cabens", "cabilonens",
"cabiunens", "cabixiens", "caboclens", "cabralens",
"cabraliens", "cabreuvens", "cabriuvens", "cabroboens",
"cabu"+u"\u00E7"+"uens", "caburanens", "cabureens", "cacaiens",
"cacatuens", "cacebuens", "cacequiens", "cacerens",
"cacetens", "cachimbens", "cachoeirens", "cachoeirinhens",
"cachorrens", "cacimbens", "cacimbinhens", "cacoalens",
"cacondens", "caculeens", "cadavalens", "cadeens",
"cadent", "cadet", "cadoriens", "cadozens",
"cadurcens", "caenens", "caenzens", "caetanens",
"caeteens", "caetiteens", "cafarnaunens", "cafearens",
"cafeeirens", "cafeens", "cafelandens", "cafezalens",
"cafezalzinhens", "cafezens", "cafufaens", "cafundoens",
"cagadens", "cagua"+u"\u00E7"+"uens", "caiabuens", "caiacuens",
"caiadens", "caianens", "caiaponiens", "caibateens",
"caibiens", "cai"+u"\u00E7"+"arens", "caicoens", "caieirens",
"caienens", "caiens", "caiguataens", "caipirens",
"caiporens", "caipuens", "cairariens", "caireens",
"cairiens", "cairuens", "caitaens", "caitauens",
"caitauzens", "caiteens", "caititeens", "caitituens",
"caiuaens", "caiubens", "caiubiens", "caiuens",
"cajaens", "cajaibens", "cajamarens", "cajapioens",
"cajariens", "cajatiens", "cajazeirens", "cajobiens",
"cajubiens", "cajubinens", "cajueirens", "cajuiens",
"cajuriens", "cajuruens", "cajuzeirens", "calabacens",
"calaboucens", "calambauens", "calamens", "calanaquens",
"calatravens", "calcedonens", "calcutaens", "caldens",
"calens", "calheirens", "calhetens", "caliciform",
"californiens", "calipolens", "calmant", "calmonens",
"calogerens", "calojiens", "calpens", "calubrigens",
"calumbiens", "cama"+u"\u00E7"+"ajiens", "cama"+u"\u00E7"+"andiens", "camacanens",
"camacaocens", "cama"+u"\u00E7"+"ariens", "camac"+u"\u00E3"+"zens", "camachens",
"camacupens", "camaipiens", "camalauens", "camaldulens",
"camaleonens", "camamuens", "camanducaiens", "camapuanens",
"camapuens", "camaquanens", "camaquens", "camaradens",
"camaraens", "camarajibens", "camaratubens", "camarens",
"camarguens", "camaronens", "cambaiens", "cambaleant",
"cambaquarens", "cambaraens", "cambaratibens", "cambarazinhens",
"cambaubens", "cambeens", "camberrens", "cambiant",
"cambiascens", "cambiasquens", "cambirens", "cambiuens",
"camboapinens", "camboataens", "camboinens", "camboriuens",
"cambrens", "cambucaens", "cambuciens", "cambu"+u"\u00E7"+"uens",
"cambuiens", "cambuizens", "cambuquirens", "camburiens",
"camelens", "cametaens", "caminhant", "caminhens",
"camobiens", "camocinens", "camolenguens", "camopiens",
"campanariens", "campanens", "campanhens", "campaniens",
"campechens", "campeirens", "campelens", "campens",
"campestr", "campestrens", "campina"+u"\u00E7"+"uens", "campinens",
"campomaiorens", "camuciens", "camucinens", "camurujiens",
"camurupinens", "camuscadens", "camutanguens", "camutiens",
"canaanens", "canabarrens", "canabravens", "cana"+u"\u00E7"+"uens",
"canacuriens", "canadens", "canadiens", "canaens",
"canafistulens", "canalens", "canamanens", "canamariens",
"cananeens", "canaranens", "canarianens", "canariens",
"canastr"+u"\u00E3"+"ozens", "canastrens", "canatibens", "canauaniens",
"canavialens", "canavieirens", "cancanens", "cancel"+u"\u00E3"+"ozens",
"cancelens", "cancheens", "candealens", "candeens",
"candelariens", "candent", "candialens", "candibens",
"candoiens", "candombens", "candumbens", "caneapens",
"canecens", "canedens", "caneleirens", "canelens",
"canelinhens", "canens", "cangalhens", "cangatens",
"cangatiens", "cangotens", "canguaretamens", "cangu"+u"\u00E7"+"uens",
"canguens", "canguerens", "canhemborens", "canhobens",
"canhotinhens", "canhumanens", "canindeens", "canindezinhens",
"canitarens", "canivetens", "canjaranens", "canjiquens",
"canoananens", "canoanens", "canoeirens", "canoens",
"canoinhens", "cansan"+u"\u00E7"+u"\u00E3"+"ozens", "cantagalens", "cantanhedens",
"cantant", "cant"+u"\u00E3"+"ozens", "cantens", "cantinhens",
"cantonens", "cantoriens", "cantuariens", "canudens",
"canumanens", "canutamens", "canzens", "capacetens",
"capadens", "capaneens", "capanemens", "capanens",
"cap"+u"\u00E3"+"ozens", "caparaozens", "capatariens", "cape"+u"\u00E7"+"uens",
"capelens", "capelinens", "capelinhens", "caperens",
"capetinguens", "capiaens", "capibaribens", "capinalens",
"capinens", "capintubens", "capinzalens", "capistranens",
"capitanens", "capitariens", "capitoliens", "capitulens",
"capituvens", "capivarens", "capivariens", "capivaritens",
"capixabens", "capoeiranens", "capoeirens", "caponguens",
"caporanguens", "caprariens", "caprichens", "capsens",
"capuavens", "capubinens", "capuens", "capumbens",
"caputirens", "caquemonens", "caquendens", "caraaens",
"carabu"+u"\u00E7"+"uens", "caracaraens", "caracaraiens", "caracolens",
"caraguataiens", "caraguatatubens", "caraibalens", "caraibeirens",
"caraibens", "caraibunens", "caraiens", "caraipens",
"caraivens", "carajaens", "carajazinhens", "carajuvens",
"carambeiens", "caramuruens", "caranaibens", "caranandinens",
"caranandubens", "carananens", "caranapatubens", "carandaiens",
"carandazalens", "caranens", "carangolens", "carapajoens",
"caraparuens", "carapebens", "carapebuens", "carapevens",
"carapicuibens", "carapiens", "carapinens", "carapoens",
"carapotoens", "caratacens", "caratatenens", "caratatinens",
"caratinguens", "caratuvens", "caravelens", "carazinhens",
"carbonitens", "carburant", "carcanhens", "carcassonens",
"carcavelens", "cardealens", "cardosens", "carea"+u"\u00E7"+"uens",
"careirens", "carent", "cariacens", "cariacicens",
"cariaciquens", "carib", "cariceens", "caridadens",
"carimanens", "carimataens", "carinhanhens", "carmens",
"carmesiens", "carmonens", "carmopoliens", "carnaibens",
"carnaubaens", "carnaubalens", "carnaubeirens", "carnaubens",
"carnaubinhens", "carneirens", "carneirinhens", "carnicens",
"carnobiens", "carnoioens", "caroarens", "carolinens",
"caroviens", "carpinens", "carquejens", "carranquens",
"carrapateirens", "carrapatens", "carrapatinhens", "carrapichelens",
"carrarens", "carrazedens", "carregadorens", "carregalens",
"carreirens", "cartaginens", "cartaxens", "cartomant",
"caruaens", "carualinens", "caruarens", "caruaruens",
"caruataiens", "carubinhens", "carumbeens", "carutaperens",
"caruzuens", "carvalhens", "carvoalens", "carvoalzinhens",
"carvoeirens", "casablanquens", "cascalens", "cascalheirens",
"cascalhens", "cascantens", "cascatens", "cascatinhens",
"cascavelens", "cascudens", "caseirens", "casevelens",
"casimirens", "casinhens", "casquens", "cassens",
"cassianens", "cassiens", "cassilandens", "cassilandiens",
"cassinens", "cassiteritens", "cassuens", "castanhalens",
"castanheirens", "castanhens", "castelandens", "castelandiens",
"castelens", "castelhanens", "castelinhens", "castilhens",
"castraleucens", "castrens", "casualidadens", "catabanens",
"cataguarinens", "cataguasens", "catalanens", "catanduvens",
"catanhedens", "catarens", "catarinens", "catauarens",
"catendens", "catequesens", "cateriangonguens", "catetens",
"catiarens", "catiboabens", "catiguaens", "catiguens",
"catimbauens", "catingalens", "catingueirens", "catinguens",
"catitens", "cativant", "catolandens", "catolandiens",
"catoleens", "catolezinhens", "catrimaniens", "catuanens",
"catu"+u"\u00E7"+"abens", "catuens", "catuipens", "catuleens",
"catulezinhens", "catumbiens", "catundens", "catuneens",
"catuniens", "catupiens", "caturaiens", "caturambens",
"caturamens", "caturiaens", "caturiteens", "caubiens",
"caucaiens", "cauipens", "caurens", "cavalcantens",
"cavaleirens", "cavalheirens", "cavalinens", "cavalinhens",
"cavaquens", "caveirens", "cavens", "caxambuens",
"caxanguens", "caxeens", "caxemirens", "caxiens",
"caxitoreens", "cazaqu", "cearens", "cebedens",
"ceboleirens", "ceciens", "cedralens", "cedreirens",
"cedrens", "cedrolandens", "cedrolandiens", "cefal"+u"\u00F3"+"pod",
"ceilonens", "celebrant", "c"+u"\u00E9"+"lebr", "celejens",
"celens", "celest", "celinens", "celoricens",
"celsens", "cemoabens", "centenariens", "centralens",
"centralinens", "centroavant", "ceraimens", "ceramicens",
"cercadens", "cercalens", "cerens", "cernachens",
"cerqueirens", "cerquilhens", "cerquinhens", "cerradinhens",
"cerrens", "cerritens", "cerveirens", "cervens",
"cervinhens", "cesariens", "cessant", "ceutens",
"chacarens", "chadiens", "chafarizens", "chaguens",
"chaleens", "chamejant", "champanhens", "chamusquens",
"chancens", "chapad"+u"\u00E3"+"ozens", "chapadens", "chapadinhens",
"chapecoens", "charqueadens", "charruens", "chavalens",
"chavens", "chaviens", "chef", "chegant",
"chelens", "chiadorens", "chiapetens", "chibant",
"chibens", "chilens", "chimarronens", "chiqu",
"chiqueirens", "chiquitens", "chocant", "choninens",
"chopinzinhens", "chor"+u"\u00E3"+"ozens", "choroens", "chorozinhens",
"chorrochoens", "chumbens", "chunens", "cianortens",
"cibaumens", "cicatrizant", "cicladens", "cidreirens",
"cient", "ciganens", "ciliciens", "cilopolandiens",
"cimbrens", "cinaniens", "cinfanens", "cintilant",
"cipoalens", "cipoens", "cipolandens", "circeens",
"circeiens", "circens", "circulant", "circundant",
"circunjacent", "circunstant", "cirtens", "cisneirens",
"cisteciens", "cisterciens", "citaniens", "cizicens",
"claraibens", "clarananens", "claravalens", "clarens",
"clariniens", "classicizant", "claudicant", "claudiens",
"clazomenens", "clementinens", "clevelandens", "client",
"cluniacens", "cluniens", "coadjuvant", "coariens",
"cobard", "cobrens", "cocaens", "cocalinhens",
"cocauens", "cochinchinens", "cochinens", "cocociens",
"codajacens", "codajaens", "codoens", "coeficient",
"coerent", "coexistent", "coimbrens", "coincident",
"coirens", "coiteens", "coivarens", "colant",
"colarens", "colatinens", "coleant", "colegiens",
"colident", "colimbriens", "colinens", "coliponens",
"colombens", "colombiens", "coloniens", "coloradens",
"colorant", "colossens", "colunens", "comanch",
"comandant", "comandatubens", "comanens", "combatent",
"combinadens", "comediant", "comendens", "comerciant",
"comerciens", "comercinhens", "cometens", "comodorens",
"comorens", "comovent", "comparecent", "competent",
"complacent", "complutens", "component", "comprovant",
"comunens", "comunicant", "comurens", "concei"+u"\u00E7"+u"\u00E3"+"onens",
"concei"+u"\u00E7"+u"\u00E3"+"ozens", "conceicionens", "concei"+u"\u00E7"+"oens", "concepcionens",
"concernent", "concertant", "conchalens", "conchens",
"concludent", "concomitant", "concordant", "concordiens",
"concorrent", "concupiscent", "condadens", "condeixens",
"condescendent", "condeubens", "condicionant", "condizent",
"condorens", "conducent", "conduruens", "conferent",
"confiant", "confident", "confinant", "confinens",
"conflitant", "confluent", "cong"+u"\u00EA"+"ner", "congolens",
"congonhalens", "congonhens", "congonhinhens", "congruent",
"conguens", "conimbricens", "conimbrigens", "conivent",
"conquistens", "conradens", "conscient", "consent",
"consequent", "conservant", "consistent", "consoant",
"consort", "constanciens", "constant", "constituint",
"contagens", "contagiant", "contaminant", "contendens",
"content", "contestadens", "continent", "contingent",
"contraent", "contrafort", "contramestr", "contraproducent",
"contrastant", "contratant", "contribuint", "contundent",
"convalescent", "convenient", "conventens", "convergent",
"convincent", "convivent", "cooperant", "copacabanens",
"copatanens", "copenhaguens", "copinalens", "coqueiralens",
"coqueirens", "coquens", "coquinhens", "corant",
"corcirens", "corcovadens", "cordeirens", "cordeiropolens",
"cordens", "cordiform", "cordijesuens", "cordilheirens",
"cordimariens", "cordisburguens", "cordislandens", "cordobens",
"cordofon", "coreauens", "coreens", "coremens",
"corfinens", "corfiniens", "corguinhens", "coribens",
"corinens", "coriniens", "corintens", "coriposens",
"coririens", "coriuvens", "corixens", "corjesuens",
"cormariens", "coroaciens", "coroadens", "coroataens",
"coroens", "coromandelens", "coronelens", "corredeirens",
"corregourens", "correguens", "corrent", "correntens",
"correntinens", "correntinhens", "correspondent", "cortant", "Cort", "cort",
"cortesens", "cortonens", "coruchens", "corumbaens",
"corumbaibens", "corumbataiens", "corupaens", "coruripens",
"coruscant", "corvens", "cosmopolens", "cosmoramens",
"cossurens", "costarricens", "costarriquens", "costens",
"cotaxeens", "cotejipens", "cotiens", "cotiporanens",
"courens", "coutens", "coutinhens", "covalent",
"covard", "covilhanens", "covilhens", "covoens",
"coxilhens", "coxinens", "craibens", "craolandens",
"craqu", "crastumiens", "cratens", "crateusens",
"cravinhens", "cravolandens", "cremens", "cremerens",
"cremonens", "crent", "crepitant", "crescent",
"cretens", "criciumalens", "criciumens", "crioulens",
"crisoliens", "crisolitens", "crisopolens", "cristalandens",
"cristalandiens", "cristalens", "cristaliens", "cristalinens",
"cristianopolens", "cristinens", "cristinopolens", "cristovens",
"critens", "criuvens", "crixaens", "crixalandens",
"crixalandiens", "crixasens", "croataens", "crocant",
"croiatens", "crominiens", "crotoniens", "cruanjiens",
"crubixaens", "cru"+u"\u00E7"+"aiens", "cru"+u"\u00E7"+"almens", "cruciant",
"crucilandens", "crucilandiens", "crustuminens", "cruxatiens",
"cruzaliens", "cruzaltens", "cruzeirens", "cruzeirinhens",
"cruzelandens", "cruzelandiens", "cruzens", "cruzetens",
"cruziliens", "cubatanens", "cubat"+u"\u00E3"+"ozens", "cubatens",
"cubatiens", "cubens", "cucuiens", "cuiabaens",
"cuiabens", "cuiambuquens", "cuiariens", "cuieirens",
"cuipiranguens", "cuiranens", "cuiteens", "cuitejiens",
"cuitezeirens", "culminant", "cumariens", "cumaruens",
"cumbens", "c"+u"\u00FA"+"mplic", "cumuruxatibens", "cunaniens",
"cundinamarquens", "cunduruens", "cunhambebens", "cunhanjiens",
"cunhens", "cunquens", "cuparaquens", "cupinens",
"cupirens", "cura"+u"\u00E7"+"aens", "curariaens", "curariens",
"curarizinhens", "curatiens", "curens", "curet",
"curicuriariens", "curimataens", "curimataiens", "curionopolens",
"curitibanens", "curitibens", "curitubens", "curiuvens",
"curralens", "curralinhens", "cursant", "curuaens",
"curuaiens", "curuataiens", "curucaens", "curu"+u"\u00E7"+"aens",
"curu"+u"\u00E7"+"ambabens", "curuens", "curumuens", "curupaens",
"curupaitiens", "curupirens", "cururipens", "cururupuens",
"curuzuens", "curvelens", "custodiens", "cutiens",
"cutilant", "cutiporanens", "dacarens", "damanens",
"dan"+u"\u00E7"+"ant", "debilitant", "debutant", "decadent",
"dec"+u"\u00E1"+"pod", "decent", "decepcionant", "declarant",
"declinant", "decliv", "decorrent", "decrescent",
"deferent", "deficient", "degradant", "deliberant",
"delinquent", "deliquescent", "delirant", "deltoid",
"demandant", "dement", "democratizant", "denunciant",
"dependent", "depoent", "depositant", "depriment",
"derribadinhens", "derrubadens", "desamparadens", "desarmant",
"desbordant", "descalvadens", "descampadens", "descansens",
"descarretens", "descendent", "descobertens", "desconcertant",
"descongestionant", "descontent", "descrent", "deselegant",
"desemboquens", "desencorajant", "desequilibrant", "desertens",
"desfigurant", "desfilant", "desfolhant", "desgastant",
"desideriens", "desidratant", "designiens", "desinfectant",
"desinfetant", "desinteressant", "deslizant", "deslumbrant",
"desobedient", "desodorant", "desolant", "desoxidant",
"despachant", "desparasitant", "despenhadens", "dessemelhant",
"destoant", "desviant", "desviens", "detectiv",
"detergent", "determinant", "detetiv", "detonant",
"devassant", "devorant", "diabet", "diademens",
"dialogant", "diamantens", "diamantinens", "dianens",
"diasens", "dibionens", "diceriens", "diens",
"diferent", "dignificant", "dilacerant", "diletant",
"diligent", "diluent", "dimanant", "diniens",
"dinisiens", "diocleciens", "diodens", "diodorens",
"dioens", "diolandens", "dionisiens", "dioramens",
"diplost"+u"\u00EA"+"mon", "dirceuens", "dirigent", "diriment",
"discent", "discernent", "discordant", "discrepant",
"discriminant", "discursant", "disform", "displicent",
"disputant", "dissemelhant", "dissident", "dissolvent",
"dissonant", "distant", "diuens", "divagant",
"divalent", "divergent", "divinens", "divinesiens",
"divinolandens", "divionens", "divisens", "divisopolens",
"divitens", "djibutiens", "doberiens", "dobradens",
"dobradinhens", "docent", "doent", "doidejant",
"doiradens", "dolcinopolens", "dolearinens", "dolent",
"domeliens", "dominant", "domingadens", "dominguens",
"dominiquens", "dopant", "dorandiens", "dorens",
"dorilens", "dorizonens", "dorment", "dormidens",
"doudejant", "douradinhens", "douradoquarens", "dresdens",
"duartens", "duartinens", "dublinens", "duereens",
"dumiens", "dumontens", "dumontinens", "duobarrens",
"duocorreguens", "duplavilens", "d"+u"\u00FA"+"plic", "durandeens",
"duriens", "dutrens", "eandens", "ebaubens",
"eboracens", "eborens", "eburodunens", "echaporens",
"ecoporanguens", "edealinens", "edeens", "edeiens",
"edenens", "edificant", "edulcorant", "eferent",
"efervescent", "eficient", "efluent", "egesteniens",
"egidiens", "egiens", "eginens", "eginiens",
"egiptanens", "egiptens", "egitanens", "egitaniens",
"eirens", "eirunepeens", "elatens", "elbens",
"eldoradens", "eleens", "elegant", "elesbanens",
"elesbonens", "eletrens", "eleuteriens", "elidens",
"eliens", "elisiariens", "eloiens", "eloquent",
"elusens", "elvens", "ematubens", "emazens",
"embaraiens", "embiralens", "embiru"+u"\u00E7"+"uens", "emboabens",
"embriagant", "embuens", "embugua"+u"\u00E7"+"uens", "emburradens",
"emens", "emergent", "emeritens", "emidiens",
"emigrant", "emiliens", "eminent", "emitent",
"emocionant", "emolient", "emparedadens", "empestant",
"empo"+u"\u00E7"+"adens", "empolgant", "emulsionant", "encaixant",
"encantadens", "encantens", "encarnacionens", "enchentens",
"encruzilhadens", "endossant", "eneidens", "enemutuens",
"enens", "enervant", "enganens", "engenhens",
"engraxat", "eniens", "enorm", "enseadens",
"ensiform", "entebens", "entorpecent", "entradens",
"entredent", "entrefolhens", "entrement", "entrepeladens",
"entrerriens", "entroncamentens", "entusiasmant", "envirens",
"envolvent", "enxuens", "eordens", "epidamniens",
"epifanens", "epirens", "epitaciens", "eporens",
"equadorens", "equ"+u"\u00E2"+"nim", "equestr", "equetliens",
"equidistant", "equivalent", "erbessens", "ercavicens",
"erebanguens", "eremitens", "erereens", "eretriens",
"erexinens", "ergavicens", "ericeirens", "ermens",
"ermensidens", "ermesindens", "ermidens", "ernaginens",
"ernestinens", "errant", "ervalens", "ervaliens",
"erveirens", "esbravejant", "escadens", "escaldant",
"escalvadens", "escamiform", "escapant", "escapciens",
"escodrens", "escondidens", "escorchant", "escouralens",
"escrevent", "escurialens", "esfuziant", "esgotant",
"esmeraldens", "esmerilens", "esminens", "esnob",
"espectant", "espelhant", "esperancens", "esperantinens",
"esperens", "espessant", "espinescent", "espinhens",
"espinilhens", "espinosens", "espinozens", "esplanadens",
"esplendent", "espongitiens", "esponjitiens", "esposendens",
"espumant", "espumosens", "estabilizant", "estacadens",
"estacielens", "estadunidens", "estafant", "estalant",
"estanciens", "estandartens", "estarrejens", "estatizant",
"esteens", "estefaniens", "esteiens", "esteirens",
"esterilizant", "estilha"+u"\u00E7"+"ant", "estimulant", "estivens",
"estobens", "estonteant", "estorilens", "estradens",
"estratonicens", "estreant", "estrebuchant", "estreitens",
"estrelens", "estrem", "estremens", "estremocens",
"estremozens", "estrepitant", "estressant", "estrident",
"estrombiens", "estruturant", "estuant", "estudant",
"estupefacient", "esvoa"+u"\u00E7"+"ant", "et"+u"\u00ED"+"op", "eucariont",
"euclidens", "euromens", "eusebiens", "euxenitens",
"evanescent", "evangelistens", "evident", "exalant",
"exaltant", "exaporanens", "exasperant", "excedent",
"excelent", "excitant", "excludent", "excrescent",
"excruciant", "executant", "exercent", "exigent",
"existent", "exorbitant", "expectorant", "expedient",
"expeditens", "experient", "experiment", "expirant",
"expoent", "exponent", "extasiant", "extenuant",
"extracionens", "extraterrestr", "extravagant", "extremens",
"exuberant", "exuens", "exultant", "ezequielens",
"fabiens", "fabricant", "fabricens", "fabricianens",
"facheant", "fadistens", "fafens", "fagulhant",
"fagundens", "faialens", "fainens", "faiscant",
"falant", "falciform", "falconiform", "faleriens",
"falerionens", "faltant", "famalicens", "famens",
"f"+u"\u00E3"+"ozens", "farejant", "farens", "farfalhant",
"fariens", "farinhens", "fariscant", "farnient",
"faroens", "farolens", "farpant", "farrapens",
"farroupilhens", "farsant", "farturens", "fascinant",
"fatigant", "fatimens", "faustianens", "faustiniens",
"favalens", "favaniens", "faveirens", "favoniens",
"faxinalens", "fazendens", "febricitant", "fecundant",
"feijoens", "feirant", "feirens", "feitoriens",
"feitosens", "felgarens", "felgueirens", "felicens",
"felipens", "felisburguens", "felixlandens", "felizens",
"feltriens", "felup", "fenaens", "fenicens",
"ferent", "ferentinens", "fermedens", "fermentelens",
"fernandopolens", "feroc", "ferrazens", "ferreirens",
"ferrens", "ferruginhens", "fertilizant", "fervedourens",
"fervent", "fervilhant", "ficant", "ficobiont",
"ficulens", "fidelandens", "fidelens", "figueirens",
"figueiroens", "figurant", "filadelfens", "filadelfiens",
"filant", "filipens", "filomeliens", "filomenens",
"filtrant", "finciens", "finens", "firmens",
"firminens", "fiss"+u"\u00ED"+"ped", "fixant", "flagelant",
"flageliform", "flagrant", "flamant", "flamejant",
"flamonens", "flamoniens", "flandrens", "flaviens",
"flechalens", "flecheirens", "flexeirens", "floraiens",
"floraliens", "floranens", "floraniens", "florarriquens",
"florealens", "florejant", "florencens", "florens",
"florent", "florentinens", "florescent", "florestalens",
"florestens", "florestinens", "florianens", "floridens",
"florineens", "florinens", "floriniens", "fluent",
"fluminens", "fluorescent", "flutuant", "foceens",
"focens", "folquens", "fondu", "font",
"fontourens", "foralens", "forens", "formigant",
"formigueirens", "formiguens", "formosens", "fornens",
"forquilhens", "forquilinhens", "fortalexiens", "fortalezens",
"fort", "fortens", "fortificant", "fortunens",
"foscoens", "fosfolipas", "fosfol"+u"\u00ED"+"pas", "fosfol"+u"\u00ED"+"pid",
"fosforescent", "fotossintetizant", "fradens", "fragosens",
"fragrant", "fraiburguens", "francens", "francisquens",
"franquens", "fraquejant", "fravasonens", "frecheirens",
"frecheirinhens", "freitens", "freixiens", "frement",
"frequent", "friburguens", "frisant", "frondent",
"fronteant", "fronteirens", "frustrant", "frutalens",
"fucens", "fugent", "fuldens", "fulgent",
"fulgurant", "fulminant", "fumant", "fumegant",
"funchalens", "funcionant", "fundanens", "fund"+u"\u00E3"+"ozens",
"fundens", "fundent", "fundonens", "f"+u"\u00FA"+"nebr",
"fungiform", "furant", "furent", "furfonens",
"furnens", "fusiform", "fustigant", "fuzilant",
"fuzilens", "gabiarrens", "gabiens", "gabirobens",
"gabonens", "gabrielens", "gadens", "gaguejant",
"gaiabalens", "gaiens", "gaivotens", "galant",
"galantens", "galdinens", "galeanens", "galenens",
"galens", "galheirens", "galiens", "galileens",
"galinens", "galinhens", "galopant", "galvanens",
"gamaleirens", "gambiens", "gamboens", "gameleirens",
"gamelens", "gamens", "ganchens", "gandariens",
"gandhiens", "gandrens", "ganduens", "ganens",
"gangorrens", "ganguens", "gantens", "garajubens",
"garanhuens", "garanhunens", "garapuavens",
"garapuens", "gararuens", "garcens", "garcezens",
"garciens", "gardeniens", "gardoniens", "gargantens",
"gargasiens", "gargauens", "garibaldens", "garibaldinens",
"garimpens", "garopabens", "garrafaens", "garrotens",
"garruchens", "garuvens", "gasparens", "gaspariens",
"gastr"+u"\u00F3"+"pod", "gatens", "gauchens", "gauerens",
"gaupoens", "gauramens", "gavianens", "gavionens",
"gazeens", "gazens", "gefirens", "gelens",
"gemelens", "gement", "genabens", "genebrens",
"generalizant", "generosens", "genovens", "genuens",
"genuinens", "geolandens", "geometrizant", "georginens",
"gerent", "germaniens", "geront", "gertrudens",
"gestant", "getulandens", "getuliens", "getulinens",
"gezehens", "gigant", "gigantens", "gilbueens",
"gilens", "girant", "girvasens", "glaucilandens",
"glaucilandiens", "glaurens", "glicerens", "gliceriens",
"globiform", "gloriens", "glorinhens", "gluciniens",
"godinens", "godoens", "godoiens", "goens",
"goiabalens", "goiabeirens", "goianaens", "goianasens",
"goiandirens", "goianens", "goianesiens", "goianiens",
"goianinhens", "goianirens", "goianortens", "goiasens",
"goiatinens", "goiatubens", "goiens", "goipabens",
"goitacasens", "goitaens", "goleganens", "golpeant",
"gon"+u"\u00E7"+"alens", "gon"+u"\u00E7"+"alvens", "gondidens", "gondirens",
"gondomarens", "gonfens", "gongojiens", "gonzaguens",
"goroboens", "gorutubens", "gotardens", "gotejant",
"goticizant", "goulartens", "gouveens", "governant",
"gracens", "graciens", "graciosens", "gra"+u"\u00E7"+"uiens",
"gradauens", "gradient", "grajauens", "gramachens",
"gramadens", "gramadinhens", "gramalens", "gramamens",
"gramens", "graminens", "graminhens", "granadens",
"granatens", "grand", "grandens", "grandiloquent",
"grandolens", "granitens", "granjeirens", "granjens",
"granzatens", "gratificant", "gravataens", "gravataiens",
"gravatalens", "grav", "gregoriens", "gremiens",
"grijoens", "grilant", "grilens", "grinincens",
"gritant", "groairens", "grogu", "grossens",
"grotanens", "grotens", "gruiform", "guabijuens",
"guabirabens", "guabirubens", "guachens", "guaciens",
"guacoceens", "gua"+u"\u00E7"+"uboiens", "gua"+u"\u00E7"+"uens", "gua"+u"\u00E7"+"uiens",
"guadalupens", "guaiacens", "guaianaens", "guaianasens",
"guaibens", "guaibinens", "guai"+u"\u00E7"+"arens", "guaicuiens",
"guaiens", "guaimbeens", "guaipavens", "guaiquicens",
"guairacaens", "guaira"+u"\u00E7"+"aens", "guairens", "guaiubens",
"guajaaens", "guajaraens", "guajaratubens", "guajaratubinhens",
"guajeruens", "guajuvirens", "guamaens", "guamaranguens",
"guamareens", "guampadens", "guanabarens", "guanaceens",
"guanambiens", "guananasens", "guanandiens", "guanduens",
"guanhanens", "guanumbiens", "guapeens", "guapia"+u"\u00E7"+"uens",
"guapiarens", "guapimiriens", "guapiramens", "guapiranguens",
"guapirens", "guapoens", "guaporanguens", "guaporeens",
"guaporemens", "guaporens", "guapuanens", "guarabirens",
"guara"+u"\u00E7"+"aiens", "guaraciabens", "guaraciamens", "guaraciens",
"guaraens", "guaraiens", "guaraipens", "guaraitubens",
"guarajiens", "guaramiranguens", "guaramiriens", "guaramirinens",
"guaranesiens", "guarania"+u"\u00E7"+"uens", "guaraniens", "guarant"+u"\u00E3"+"ens",
"guarantanens", "guarapariens", "guarapiranguens", "guarapuanens",
"guarapuavens", "guarapuavinhens", "guaraque"+u"\u00E7"+"abens", "guararaens",
"guararapens", "guararemens", "guararens", "guararuens",
"guarataiens", "guaratinguens", "guaratinguetaens", "guaratubens",
"guarauens", "guaraunens", "guaraverens", "guaraxainens",
"guardens", "guardinhens", "guareiens", "guarens",
"guaribens", "guaricanguens", "guariguariens", "guarinens",
"guariramanens", "guarirobens", "guaritens", "guarizinhens",
"guartelens", "guaruens", "guarujaens", "guarulhens",
"guataens", "guatambuens", "guataparaens", "guatemalens",
"guavirens", "guaxens", "guaxiens", "guaximens",
"guaxindibens", "guaxupeens", "guedens", "guianens",
"guidens", "guidovalens", "guiens", "guimaranens",
"guimaraniens", "guindant", "guineens", "guiomaens",
"guirapaens", "guiratinguens", "guiricemens", "guizalhant",
"gulosens", "gundens", "gundianens", "guraitubens",
"gurinhat"+u"\u00E3"+"ens", "gurinhatanens", "gurinheens", "gurinhenens",
"guriuens", "gurjanens", "gurj"+u"\u00E3"+"oens", "gurupaens",
"gurupiens", "guzolandens", "habitant", "halicarnassens",
"haliciens", "halidiens", "hamaniens", "hamburguens",
"hamondens", "haplost"+u"\u00EA"+"mon", "harmoniens", "havaiens",
"hebridens", "heimatens", "heitoraiens", "helenens",
"heliodorens", "heliopoliens", "helmodens", "helveciens",
"hematitens", "hemonens", "heracleens", "heracliens",
"herbitens", "herbulens", "herciniens", "herculandens",
"herculandiens", "herculanens", "herdoniens", "hereg",
"herviens", "hesitant", "hesp"+u"\u00E9"+"rid", "hexacord",
"hiant", "hibernant", "hiblens", "hidissens",
"hidratant", "hidrolandens", "hidrolinens", "hienipens",
"hierapolens", "hierolofiens", "hilariant", "hinterlandiens",
"hipoglicemiant", "hiponens", "hippi", "hipsodont",
"hispalens", "hispaliens", "hispaniens", "histoniens",
"holandens", "hollywoodens", "holmiens", "holofotens",
"homenageant", "homogeneizant", "homonadens", "homotransplant",
"hondurens", "horaciens", "horens", "horizontens",
"horizontinens", "hormens", "horripilant", "horrorens",
"horrorizant", "hortens", "hortolandens", "hortolandiens",
"h"+u"\u00F3"+"sped", "hospiciens", "hostiens", "hostiliens",
"hotentot", "hugonapolens", "huguenot", "huitanaanens",
"humaitaens", "humbertuens", "humild", "humildens",
"iacaiaens", "iacanguens", "iaciarens", "iacriens",
"ia"+u"\u00E7"+"uens", "ianatamens", "iandareens", "ianqu",
"iapiens", "iapuens", "iaqu", "iaquiranens",
"iaramontens", "iarens", "iaripens", "iassens",
"iatecaens", "iatiens", "iauacanaens", "iauareteens",
"iaunariens", "iavaritens", "ibaitiens", "ibaramens",
"ibareens", "ibaretamens", "ibateens", "ibateguarens",
"ibatibens", "ibatuiens", "ibemens", "ibertioguens",
"ibia"+u"\u00E7"+"aens", "ibiaciens", "ibia"+u"\u00E7"+"uceens", "ibia"+u"\u00E7"+"uens",
"ibiaens", "ibiaguiens", "ibiaiens", "ibiajarens",
"ibianens", "ibiapabens", "ibiapinens", "ibiaporanens",
"ibiarens", "ibicabens", "ibicaraiens", "ibicareens",
"ibicatuens", "ibicuanens", "ibicuiens", "ibicuitabens",
"ibicuitinguens", "ibiens", "ibimirinens", "ibipebens",
"ibipetubens", "ibipetunens", "ibipirens", "ibipitanguens",
"ibiporanens", "ibiporanguens", "ibiquarens", "ibiquerens",
"ibirabens", "ibiracatuens", "ibiraciens", "ibira"+u"\u00E7"+"uens",
"ibiraens", "ibiraiarens", "ibirajaens", "ibirajubens",
"ibiramens", "ibiranguens", "ibiranhenens", "ibirapaens",
"ibirapitanguens", "ibirapu"+u"\u00E3"+"ens", "ibirapuitanens", "ibiraremens",
"ibirataiens", "ibiratinguens", "ibiriteens", "ibirocaiens",
"ibirubaens", "ibirubens", "ibitiarens", "ibitigua"+u"\u00E7"+"uens",
"ibitiguaiens", "ibitiguirens", "ibitinguens", "ibitiocens",
"ibitioquens", "ibitipocens", "ibitipoquens", "ibitiramens",
"ibitiranguens", "ibitirens", "ibitirunens", "ibititaens",
"ibitiurens", "ibitiuvens", "ibitubens", "ibituens",
"ibitunanens", "ibitupanens", "ibituporanguens", "ibiturunens",
"ibiunens", "iboens", "iborepiens", "ibotiramens",
"ibua"+u"\u00E7"+"uens", "ibugua"+u"\u00E7"+"uens", "icai"+u"\u00E7"+"arens", "icai"+u"\u00E7"+"arinhens",
"icapoaens", "icapuiens", "icaraiens", "icaraimens",
"icariens", "icatuens", "icemens", "icenens",
"icens", "icenzens", "icoens", "iconhens",
"iconiens", "icoraciens", "icozeirens", "icozinhens",
"idanhens", "idealens", "ideologizant", "idiolandens",
"idomenens", "iemenens", "ienisseens", "ienisseiens",
"iepeens", "iepens", "iersonens", "ietens",
"iga"+u"\u00E7"+"abens", "igaciens", "igapoens", "igaporanens",
"igara"+u"\u00E7"+"uens", "igaraiens", "igarapavens", "igarapebens",
"igarapeens", "igarassuens", "igarataens", "igaratens",
"igaratinguens", "igarauens", "igarens", "igariteens",
"igaroiens", "igatiens", "igatiquirens", "igatuens",
"ignorant", "igrapiunens", "igrejens", "igrejinhens",
"iguabens", "igua"+u"\u00E7"+"uens", "iguaens", "iguaibens",
"iguaiens", "igualdadens", "iguapens", "iguaraciens",
"iguara"+u"\u00E7"+"uens", "iguatamens", "iguatemiens", "iguatiens",
"iguatuens", "iguirens", "iguituens", "ijaciens",
"ijucapiramens", "ijuiens", "ildefonsens", "ilercaonens",
"ilerdens", "ilergavonens", "ilhabelens", "ilhavens",
"ilheens", "ilhenens", "ilheuens", "ilhotens",
"iliberitens", "ilicens", "iliciniens", "iliens",
"ilipens", "ilipulens", "iliturgens", "ilurconens",
"ilurgavonens", "iluronens", "ilustr", "imacarens",
"imaculadens", "imanent", "imaruiens", "imbarieens",
"imbaubens", "imbauvalens", "imbauzinhens", "imbeens",
"imbel", "imberb", "imbiru"+u"\u00E7"+"uens", "imbitubens",
"imbituvens", "imboiarens", "imbuens", "imbugua"+u"\u00E7"+"uens",
"imbuiens", "imbuirens", "imburanens", "imbuzalens",
"imigrant", "iminent", "imitant", "impacient",
"impactant", "impant", "impenitent", "imperant",
"imperatricens", "imperatrizens", "imperiens", "impermanent",
"impermeabilizant", "impersistent", "impertinent", "impetrant",
"implant", "implicant", "implorant", "imponent",
"important", "impotent", "impressionant", "imprevident",
"improcedent", "imprudent", "imp"+u"\u00FA"+"ber", "impudent",
"impun", "imun", "imunizant", "imunocompetent",
"imunodeficient", "inaciens", "inadimplent", "inajaens",
"inalant", "inambuens", "in"+u"\u00E2"+"nim", "inaparent",
"incandescent", "incapacitant", "incessant", "incident",
"incipient", "inclement", "incoerent", "inc"+u"\u00F3"+"lum",
"incombent", "incompetent", "inconcludent", "inconfidenciens",
"inconfident", "inconfidentens", "incongruent", "inconscient",
"inconsequent", "inconsistent", "inconstant", "incontinent",
"inconvenient", "indagant", "indaiabirens", "indaiaens",
"indaialens", "indaiatubens", "indaiens", "indecent",
"indeiscent", "independenciens", "independent", "indianopolens",
"indiaporanens", "indiarobens", "indiferent", "indigent",
"indignificant", "individualizant", "indolent", "indulgent",
"industrialens", "inebriant", "ineficient", "inerent",
"inerm", "inert", "inesens", "inexistent",
"inexperient", "infamant", "infam", "infant",
"infectant", "infestant", "influent", "informant",
"infren", "infringent", "ingaens", "ingazeirens",
"ingent", "ingratens", "ingredient", "ingressant",
"inhacoraens", "inhaiens", "inhambupens", "inhamunens",
"inhanduiens", "inhangapiens", "inhapiens", "inhapinens",
"inhapinhens", "inhataens", "inhatiunens", "inhaumens",
"inhobinens", "inhomirinens", "inhu"+u"\u00E7"+"uens", "inhumens",
"inhuporanguens", "iniciant", "inimatubens", "inimutabens",
"inoanens", "inobservant", "inocenciens", "inocent",
"inoens", "inoperant", "inquietant", "insalubr",
"inscient", "insign", "insignificant", "insinuant",
"insipient", "insistent", "insolent", "insolvent",
"inson", "instant", "instigant", "insubsistent",
"insuficient", "insultant", "insurgent", "intanens",
"integrant", "inteligent", "intemperant", "intendenciens",
"intendent", "interagent", "interamnens", "intercaciens",
"intercadent", "intercedent", "intercomunicant", "intercruzant",
"interdependent", "interessant", "interferent", "interfoliens",
"interlandens", "interlandiens", "intermitent", "int"+u"\u00E9"+"rpret",
"intervenient", "intolerant", "intoxicant", "intransigent",
"intrigant", "intumescent", "inubiens", "invaginant",
"invariant", "invernadens", "iogu", "iomereens",
"ionizant", "ipabens", "ipagua"+u"\u00E7"+"uens", "ipameriens",
"ipamorotiens", "ipandreens", "ipaneens", "ipanemens",
"ipangua"+u"\u00E7"+"uens", "ipaporanguens", "ipargens", "ipatinguens",
"ipau"+u"\u00E7"+"uens", "ipaumirinens", "ipaunens", "ipecaetaens",
"ipe"+u"\u00E7"+"uens", "ipeens", "ipeguens", "iperabens",
"iperoens", "ipeuatens", "ipeunens", "ipiabens",
"ipiauens", "ipiauiens", "ipiguaens", "ipiibens",
"ipiracens", "ipiraens", "ipiranguens", "ipiraquens",
"ipitanguens", "ipiunens", "ipixunens", "ipoemens",
"ipoiens", "ipojucens", "ipojuquens", "ipomeens",
"ipomeiens", "iporaens", "iporanens", "iporanguens",
"iporcens", "iprens", "ipua"+u"\u00E7"+"uens", "ipuanens",
"ipubiens", "ipu"+u"\u00E7"+"abens", "ipucens", "ipueirens",
"ipuens", "ipuiunens", "ipumirinens", "ipupiarens",
"ipuquens", "iracemapolens", "iracemens", "iraceminhens",
"iraciens", "iragua"+u"\u00E7"+"uens", "iraiens", "iraitiens",
"irajaens", "irajaiens", "irajubens", "iramaiens",
"iraniens", "irapeens", "iraporanguens", "irapuaens",
"irapuanens", "irapuens", "irapuiens", "irapuruens",
"iraput"+u"\u00E3"+"zens", "iraquarens", "iraquitanens", "iraraens",
"irarens", "iratamens", "iratiens", "iratinguens",
"irau"+u"\u00E7"+"ubens", "ireceens", "irereens", "iretamens",
"iriant", "iridescent", "iriens", "iriranens",
"iririens", "iririteuens", "iriritibens", "iritreens",
"iritreiens", "irituiens", "iroiens", "irradiant",
"irrelevant", "irreverent", "irritant", "irundiarens",
"irundiens", "irupiens", "isabelens", "isaiens",
"isidorens", "isindens", "isiodens", "isiondens",
"islandens", "isolant", "is"+u"\u00F3"+"pod", "is"+u"\u00F3"+"scel",
"isost"+u"\u00EA"+"mon", "isotransplant", "israelandens", "israelens",
"issens", "itaarens", "itabaianens", "itabaianinhens",
"itabapoanens", "itaberabens", "itaberaens", "itaberaiens",
"itaberens", "itabiens", "itabirens", "itabirinhens",
"itabiritens", "itabocens", "itaboquens", "itaboraiens",
"itabunens", "itacajaens", "itacambirens", "itacarambiens",
"itacareens", "itacatuens", "itacavens", "itacens",
"itaciens", "itaciramens", "itacoatiarens", "itacolomiens",
"ita"+u"\u00E7"+"uceens", "ita"+u"\u00E7"+"uens", "itacurubens", "itacuru"+u"\u00E7"+"aens",
"itaens", "itaeteens", "itagua"+u"\u00E7"+"uens", "itaguaens",
"itaguaiens", "itaguajeens", "itaguandibens", "itaguarens",
"itaguaruens", "itaguatinens", "itaguirens", "itaiacoquens",
"itaiaens", "itaianopolens", "itaibens", "itaiboens",
"itai"+u"\u00E7"+"abens", "itaiciens", "itaiens", "itaimbeens",
"itaimbezinhens", "itainens", "itainopolens", "itaiococens",
"itaioens", "itaiopolens", "itaipabens", "itaipavens",
"itaipeens", "itaipuens", "itaitubens", "itaituens",
"itaiubens", "itaiuens", "itaja"+u"\u00E7"+"uens", "itajaens",
"itajaiens", "itajarens", "itajibens", "itajiens",
"itajimirinens", "itajobiens", "itajubaens", "itajubaquarens",
"itajubatibens", "itajubens", "itajuens", "itajuipens",
"itajuruens", "itajutibens", "italegrens", "italicens",
"italvens", "itamanariens", "itamaracaens", "itamarajuens",
"itamarandibens", "itamarateens", "itamaratiens", "itamariens",
"itamataiens", "itambacuriens", "itambaracaens", "itambeens",
"itambiens", "itamirens", "itamirinens", "itamojiens",
"itamontens", "itamotiaguens", "itamuriens", "itanagrens",
"itanajeens", "itanens", "itanguaens", "itanhaenens",
"itanhaens", "itanhanduens", "itanheens", "itanhenens",
"itanhiens", "itanhomiens", "itaobinens", "itaocarens",
"itaocens", "itaoquens", "itapa"+u"\u00E7"+"arocens", "itapa"+u"\u00E7"+"aroquens",
"itapaciens", "itapacuraens", "itapajeens", "itapajipens",
"itapanhoacanguens", "itaparaens", "itaparicens", "itapariquens",
"itapebiens", "itapebu"+u"\u00E7"+"uens", "itapecericens", "itapeceriquens",
"itapecuruens", "itapecurumirinens", "itapeens", "itapeinens",
"itapeipuens", "itapejarens", "itapemens", "itapemirinens",
"itapequaens", "itaperens", "itaperiuens", "itaperunens",
"itapeteiuens", "itapetinens", "itapetinguens", "itapetinguiens",
"itapetininguens", "itapeuaens", "itapeunens", "itapevens",
"itapeviens", "itapi"+u"\u00E7"+"umens", "itapicuruens", "itapicurumirinens",
"itapinens", "itapinimens", "itapipocens", "itapipoquens",
"itapiranguens", "itapirapuanens", "itapiraremens", "itapirens",
"itapiru"+u"\u00E7"+"uens", "itapiruens", "itapirunens", "itapitocaiens",
"itapiunens", "itapixeens", "itapixunens", "itapoaens",
"itapoens", "itapolens", "itaporanens", "itaporanguens",
"itaporeens", "itapororocens", "itapororoquens", "itapuanens",
"itapucens", "itapucuens", "itapulens", "itapuquens",
"itapuranguens", "itapurens", "itaquaens", "itaquaquecetubens",
"itaquaraiens", "itaquarens", "itaquariens", "itaquatiarens",
"itaquerens", "itaquiens", "itaquirens", "itaquitinguens",
"itaranens", "itarantinens", "itarareens", "itaremens",
"itaririens", "itaroqueens", "itaroquenens", "itarumanens",
"itarumens", "itassuceens", "itataens", "itatiaiens",
"itatiaiu"+u"\u00E7"+"uens", "itatibens", "itatiens", "itatinens",
"itatinguens", "itatinguiens", "itatirens", "itatubens",
"itatupanens", "itaubelens", "itaubens", "itau"+u"\u00E7"+"uens",
"itaueirens", "itauens", "itaunens", "itauninhens",
"itauzens", "itaveraens", "itaveravens", "itaxamens",
"iterereens", "itibirens", "itinerant", "itinguens",
"itiquirens", "itirapinens", "itirapuanens", "itirens",
"itiru"+u"\u00E7"+"uens", "itiubens", "itiucens", "itiuquens",
"itobiens", "itororoens", "itoupavens", "itua"+u"\u00E7"+"uens",
"ituberaens", "ituens", "ituetens", "itugua"+u"\u00E7"+"uens",
"ituiens", "ituinens", "ituiutabens", "itumbiarens",
"itumirinens", "itupararanguens", "itupavens", "itupevens",
"itupiranguens", "ituporanguens", "ituquiens", "ituramens",
"itutinguens", "ituveravens", "iuaens", "iubatinguens",
"iucaiens", "iucatanens", "iuiraens", "iuiteporanens",
"iuiuens", "iunens", "iuniens", "iunzens",
"iupebens", "iuruens", "iuticens", "iutiquens",
"ivaiens", "ivaiporanens", "ivaitinguens", "ivatens",
"ivatubens", "ivinheimens", "ivitiens", "ivoiriens",
"ivolandens", "ivoraens", "ivotiens", "ixuens",
"ixuzinhens", "izarens", "jabaeteens", "jabaquarens",
"jabitacaens", "jaboatanens", "jaboat"+u"\u00E3"+"ozens", "jaboraens",
"jaborandiens", "jaboticabalens", "jabotiens", "jaburizinhens",
"jaburuens", "jaburunens", "jabuticabalens", "jabuticabeirens",
"jabuticabens", "jabuticatubens", "jabutiens", "jacamaens",
"jacampariens", "ja"+u"\u00E7"+"anens", "jacaquaens", "jacaraciens",
"jacaraipens", "jacarandaens", "jacarandirens", "jacarauens",
"jacareacanguens", "jacarecicens", "jacareciens", "jacareciquens",
"jacareens", "jacareiens", "jacarequarens", "jacaretinguens",
"jacarezinhens", "jacatanens", "jacaunens", "jaceabens",
"jacent", "jacerendiens", "jacerubens", "jaciabens",
"jaciarens", "jaciens", "jaciguaens", "jacintens",
"jaciobens", "jaciparanaens", "jaciporanens", "jaciremens",
"jacitarens", "jacksoniens", "jacobinens", "jaconeens",
"jacruaruens", "jacsoniens", "jacubens", "jacuecanguens",
"jacuens", "jacuiens", "jacuipens", "jacuizinhens",
"jacumanens", "jacundaens", "jacundazinhens", "jacupembens",
"jacupiranguens", "jacuriciens", "jacuriens", "jacurunens",
"jacururuens", "jacurutuens", "jacutinguens", "jadibaruens",
"jaeteuens", "jafens", "jagod", "jaguapitanens",
"jaguaquarens", "jaguaraciens", "jaguara"+u"\u00E7"+"uens", "jaguaraens",
"jaguaranens", "jaguarariens", "jaguareens", "jaguarembeens",
"jaguarens", "jaguaretamens", "jaguariaivens", "jaguaribarens",
"jaguaribens", "jaguaricatuens", "jaguariens", "jaguarinhens",
"jaguaripens", "jaguaritens", "jaguaritirens", "jaguariunens",
"jaguaronens", "jaguaruanens", "jaguarunens", "jaguatiricens",
"jaguatiriquens", "jaibarens", "jaibens", "jaicoens",
"jald", "jalens", "jalesens", "jamacaruens",
"jamaicens", "jamaparaens", "jamaparens", "jamariens",
"jamaxinzinhens", "jambeirens", "jambruquens", "jambua"+u"\u00E7"+"uens",
"jamprucens", "jampruquens", "janaubens", "jandaiens",
"jandairens", "jandirens", "jandiritiuens", "janduiens",
"janduissens", "jangadens", "janguens", "jansenens",
"jantant", "januarens", "januariens", "japaraibens",
"japaratinguens", "japaratubens", "japecanguens", "japeinens",
"japericens", "japeriens", "japeriquens", "japiens",
"japiinens", "japiinzens", "japirens", "japiubens",
"japoatanens", "japonens", "japuatanens", "japueens",
"japuens", "japuibens", "japumirinens", "japuraens",
"jaqueirens", "jaquiranens", "jaracatiaens", "jaraguaens",
"jaraguariens", "jaraguens", "jaramataiens", "jaraquiens",
"jararacens", "jararaquens", "jardimirinens", "jardineirens",
"jardinens", "jardinesiens", "jardinopolens", "jariens",
"jarinuens", "jarmelens", "jarudorens", "jaruens",
"jassens", "jataiens", "jataizinhens", "jataubens",
"jatiens", "jatimanens", "jatiucens", "jatiuquens",
"jatobaens", "jatobalens", "jatobazinhens", "jatuaranens",
"jauarauens", "jauariens", "jauatoens", "jaudilandens",
"jaudilandiens", "jauens", "jaupaciens", "jauquarens",
"jauruens", "jauzinhens", "javariens", "javens",
"jazent", "jeceabens", "jeitosens", "jejuiens",
"jenipabuens", "jenipapeirens", "jenipapens", "jenipaubens",
"jequeriens", "jequieens", "jequiens", "jequiezinhens",
"jequiri"+u"\u00E7"+"aens", "jequiricens", "jequitaiens", "jequitibaens",
"jequitinhonhens", "jeremiens", "jeremoabens", "jeremunens",
"jeremunzens", "jererauens", "jeria"+u"\u00E7"+"uens", "jeribaens",
"jeribatubens", "jericoaquarens", "jericoens", "jerimatalhens",
"jeriquaquarens", "jeriquarens", "jeritacoens", "jerivaens",
"jeroaquarens", "jeromenhens", "jeronimens", "jersonens",
"jeruens", "jerumenhens", "jerumirinens", "jesuanens",
"jesuaniens", "jesuens", "jesuitens", "jetens",
"jetibaens", "jiboiens", "jijuens", "jiloens",
"jinimbuens", "jiparanaens", "jipubens", "jiquiri"+u"\u00E7"+"aens",
"jiquitaiens", "jirauens", "jiribatubens", "jiruaens",
"jiruens", "jitaunens", "jitiranens", "joa"+u"\u00E7"+"abens",
"joa"+u"\u00E7"+"ubens", "joaimens", "joan", "joanens",
"joanesens", "joanesiens", "joanetens", "joanicens",
"joaninens", "joanopolens", "joaquimens", "joaquinens",
"joarizinhens", "joatubens", "joeiranens", "joiens",
"joinvilens", "jojocens", "jojoquens", "joniens",
"jordanens", "jordaniens", "jord"+u"\u00E3"+"ozinhens", "jorgens",
"joseens", "josefens", "joselandens", "joselandiens",
"jovianiens", "juabens", "jua"+u"\u00E7"+"abens", "juacemens",
"juaciens", "juaens", "juatamens", "juatubens",
"juazeirens", "juazeirinhens", "juazens", "jubaiens",
"jubilant", "jubirens", "jucaens", "ju"+u"\u00E7"+"aralens",
"ju"+u"\u00E7"+"arens", "ju"+u"\u00E7"+"ariens", "jucatiens", "juciapens",
"jucuens", "jucumanens", "jucumpemens", "jucuru"+u"\u00E7"+"uens",
"jucururiens", "jucururuens", "jucurutuens", "judaizant",
"jueranens", "juira"+u"\u00E7"+"uens", "julianens", "julianiens",
"juliens", "juliobrigens", "jumens", "jumirinens",
"juncens", "juncionens", "jundiaens", "jundiaiens",
"jundiapebens", "junens", "junqueirens", "junqueiropolens",
"junquens", "jupaguaens", "juparens", "juparitenens",
"jupiaens", "jupiens", "jupiranguens", "juquiaens",
"juquiquerens", "juquiratibens", "juquitibens", "juraciens",
"juramentens", "jurandens", "jurapaitens", "juraraens",
"jureens", "juremalens", "juremens", "juriciens",
"juripiranguens", "juritianhens", "juritiens", "juruaens",
"juruaiens", "jurubebens", "juruceens", "juruenens",
"juruens", "jurumiriens", "jurupariens", "jurupebens",
"jurupemens", "jurutiens", "juruvalhens", "juscelandens",
"juscelandiens", "juscelinens", "jutaiens", "juticens",
"jutiens", "jutiquens", "jutlandens", "jutlandiens",
"juturnaibens", "juvenciens", "juventinens", "labreens",
"lacens", "lacerant", "lacerdens", "laciens",
"laciniens", "lacobricens", "lacobrigens", "lacrimejant",
"lactant", "lactent", "lacustr", "lacustrens",
"ladainhens", "ladarens", "ladariens", "ladeirens",
"laetiens", "lafaietens", "lafonens", "lagamarens",
"lagartens", "lagoadens", "lagoanens", "lagoens",
"lagoinhens", "lagonens", "lag"+u"\u00F3"+"pod", "laguens",
"laguinhens", "lagunens", "lagunitens", "lajeadens",
"lajeadinhens", "lajed"+u"\u00E3"+"oens", "lajedens", "lajedinhens",
"lajedonens", "lajeirens", "lajens", "lajiens",
"lajinhens", "lalimens", "lamaronens", "lambariens",
"lambasens", "lambedorens", "lambedourens", "lambesens",
"lamecens", "lameirens", "laminens", "lamounierens",
"lampejant", "l"+u"\u00E2"+"mur", "lan"+u"\u00E7"+"ant", "lanciens",
"lancinant", "landrinopolens", "langu", "languipnens",
"lanhadens", "lanhelens", "lanhonens", "lanhosens",
"lantejoulant", "lantunens", "lanuens", "laodicens",
"laosens", "laozens", "lapelens", "lapens",
"lapinhens", "laplatens", "laponenens", "lapurdens",
"laranjaiens", "laranjalens", "laranjeirens", "larens",
"larguens", "larissens", "larnens", "lassancens",
"lastrens", "latejant", "latent", "lateranens",
"latoratens", "laurbanens", "laurentinens", "lausanens",
"lausperen", "lavagant", "lavinens", "laviniens",
"lavourens", "lavrant", "lavrens", "lavrinhens",
"laxant", "leandrens", "leblonens", "legent",
"legorn", "leiriens", "leitens", "leixonens",
"lemens", "lemniens", "lemovicens", "lenciens",
"len"+u"\u00E7"+"oens", "len"+u"\u00E7"+"oiens", "len"+u"\u00E7"+"olens", "lentiens",
"leodiens", "leonicens", "leopoldens", "leopoldinens",
"leopolens", "leptimagnens", "lesotens", "levalloisens",
"levaloisens", "levergerens", "levinopolens", "libarnens",
"liberalizant", "liberatens", "liberdanens", "libertens",
"libriform", "licitant", "licuriens", "lidianopolens",
"lidiens", "liechtensteinens", "liechtensteiniens", "ligant",
"l"+u"\u00ED"+"gur", "limarens", "limeirens", "limiens",
"limitant", "lim"+u"\u00ED"+"trof", "limoeirens", "lind",
"lindoestens", "lindoiens", "linens", "linharens",
"liparens", "lipsiens", "liquidant", "lirinens",
"lisbonens", "lisimaquiens", "litigant", "litisconsort",
"lituaniens", "livramentens", "lixiviant", "lizardens",
"loandens", "lobatens", "lobelens", "lobens",
"locrens", "loirejant", "lombardiens", "londinens",
"londoniens", "londrinens", "longaens", "lontrens",
"lordelens", "lorenens", "loretens", "loronens",
"lourdens", "lourencens", "lourens", "lourinhanens",
"lourinhanhens", "lousadens", "lousanens", "lousonens",
"louveirens", "lovaniens", "luandens", "lubiens",
"lubrificant", "lucaiens", "lucelburgens", "luceliens",
"lucenens", "lucens", "lucerens", "lucianopolens",
"lucilant", "lucoferonens", "lucrinens", "ludovicens",
"lufens", "lugdunens", "l"+u"\u00FA"+"gubr", "luisburguens",
"luisens", "luisianens", "luisianiens", "luizens",
"lumbricoid", "lumiarens", "luminariens", "luminescent",
"luminosens", "lundens", "lupercens", "luperciens",
"lupiens", "lupinopolens", "luquens", "lursens",
"lusitaniens", "lustosens", "luteciens", "luxemburguens",
"luxoviens", "luxuriant", "luzens", "luzent",
"luzernens", "luzianens", "luzianiens", "luziens",
"luzilandens", "luzilandiens", "mabens", "mabideriens",
"macabuens", "macabuzinhens", "ma"+u"\u00E7"+"acaraens", "macaeens",
"macaens", "ma"+u"\u00E7"+"aens", "macaibens", "macaiens",
"macajatubens", "macajubens", "macambirens", "macanareens",
"ma"+u"\u00E7"+"ant", "macaocens", "macaoquens", "macapaens",
"macaparanens", "macapuanens", "macaquens", "macaraniens",
"macarauens", "macarens", "macataens", "macatubens",
"macauariens", "macaubalens", "macaubens", "macauens",
"macedens", "macedinens", "macedoniens", "maceioens",
"machacaliens", "machacalizens", "machadens", "machadiens",
"machadinhens", "machiquens", "machorrens", "macieirens",
"macielens", "macond", "macronutrient", "macucens",
"macuens", "macuinens", "macujeens", "macupinens",
"macuquens", "macuripariens", "macurureens", "macuxixiriens",
"madagascarens", "madailens", "madalenens", "madeirens",
"madiens", "madragoens", "madrilens", "madureirens",
"maetinguens", "mafrens", "mafrensens", "magalhens",
"magdalenens", "magdens", "mageens", "magnat",
"magnet", "magnificent", "maguariens", "maiauataens",
"maiens", "maiobens", "maiquiniquens", "mairiens",
"mairiporanens", "mairipotabens", "majeens", "majorens",
"majorlandens", "majorlandiens", "majurianens", "malabarens",
"malacachetens", "malaquiens", "malaripens", "malauiens",
"maldivens", "maldizent", "maledicent", "maletens",
"malgax", "malhadens", "malhadorens", "maliciens",
"maliens", "malinens", "malletens", "malocens",
"maloquens", "maltens", "maluens", "malvinens",
"mamanacaraiens", "mamanguapens", "mambaiens", "mambemb",
"mamboreens", "mambucabens", "mambuquens", "mamedens",
"mamens", "mamiaens", "mamoeirens", "mamonens",
"mamoriaens", "mamparrens", "manacaens", "manacapuruens",
"managuens", "manairens", "manaquiriens", "manariens",
"manauens", "manda"+u"\u00E7"+"aiens", "mandacaruens", "mandagua"+u"\u00E7"+"uens",
"mandaguariens", "mandant", "mandembens", "mandiquiens",
"mandiritubens", "mandirobens", "mandiuens", "mandivens",
"manducens", "manduquens", "manduriens", "mangabeirens",
"mangabens", "mangalens", "mangaraiens", "mangarataiens",
"mangaratibens", "mangaratuens", "mangualdens", "manguariens",
"mangueirens", "mangueirinhens", "manguens", "manguinhens",
"manguncens", "manhanens", "manhua"+u"\u00E7"+"uens", "manhumirinens",
"mania"+u"\u00E7"+"uens", "manicens", "mani"+u"\u00E7"+"obens", "manicoreens",
"manifestant", "manimbuens", "manitubens", "manivens",
"mansens", "mansid"+u"\u00E3"+"oens", "mansinhens", "mansuetudinens",
"manteiguens", "mantenens", "manteniens", "mantenopolens",
"mapanens", "mapangapens", "mapiaens", "mapirenens",
"mapixiens", "mapuaens", "maquineens", "mara"+u"\u00E3"+"ens",
"maraanens", "mara"+u"\u00E3"+"zens", "marabaens", "maraca"+u"\u00E7"+"umeens",
"maracaens", "maracaiens", "maracajaens", "maracajauens",
"maracajuens", "maracananens", "maracanauens", "maracanens",
"maracasens", "maracatubens", "maracoeirens", "maracujaens",
"maragogiens", "maragojiens", "maragojipens", "maragojipinhens",
"maraguaens", "maraialens", "marajaens", "marajoarens",
"marajoens", "marambainhens", "marangatuens", "maranguapens",
"maranhens", "marapaniens", "marapaninens", "marapeens",
"marapuamens", "marariens", "mararupaens", "marataens",
"marataizens", "marauaens", "marauens", "marav",
"maravilhens", "marcant", "marcelandens", "marcelens",
"marcelinens", "marcens", "marcescent", "marchant",
"marcoens", "marcolinens", "marcondens", "marcondesiens",
"marcosiens", "marcuens", "mareant", "maremens",
"maresiens", "marfinens", "margaridens", "marialvens",
"marianens", "mariantens", "mariapolens", "maribondens",
"maricaens", "maricoabens", "mariens", "marientalens",
"marilacens", "marilandens", "marilandiens", "marilenens",
"marilens", "mariliens", "mariluzens", "marimbens",
"marimbondens", "marinens", "maringaens", "marinheirens",
"marinhens", "marinopolens", "maripaens", "mariquitens",
"mariricuens", "maristelens", "maritubens", "marizens",
"marlieriens", "marmelandens", "marmelandiens", "marmelarens",
"marmeleirens", "marmelens", "marojalens", "marombens",
"marotens", "marovens", "marquens", "marquinhens",
"marrequens", "marroquens", "marruaens", "martinens",
"martinesiens", "martinhens", "martinopolens", "martiriens",
"maruanunens", "marudaens", "maruinens", "marumbiens",
"marupiarens", "marvanens", "marzaganens", "marzaganiens",
"marzagonens", "mascotens", "massapeens", "massarelens",
"massiliens", "massilitanens", "mastaurens", "matabel",
"matanens", "matapajipens", "matapiens", "matapiquarens",
"mataracens", "matarandibens", "mataraquens", "mataripens",
"mataripuanens", "mateens", "mateirens", "matens",
"materlandens", "matiens", "matildens", "matinens",
"matinhens", "matintinens", "matipoens", "matoens",
"matonens", "matoqueimadens", "matosinhens", "matotuiens",
"matozinhens", "matrinxaens", "matritens", "matuens",
"matupaens", "maturucens", "maturuquens", "matutinens",
"mauaens", "maueens", "mauens", "mauesens",
"mauriens", "mauritiens", "maustrarens", "maxacalens",
"maxacaliens", "maxaranguapens", "maximens", "mazaganens",
"mazagonens", "mealhadens", "mearinens", "mecejanens",
"medeirens", "medens", "medianeirens", "medicilandens",
"medievalizant", "medinens", "med"+u"\u00ED"+"ocr", "mediolanens",
"mediquens", "meditacionens", "medubrigens", "meens",
"megarens", "meladens", "meladinhens", "melanciens",
"melariens", "meldens", "meleirens", "melens",
"melgacens", "meliant", "melibeens", "melissens",
"melitens", "melosens", "melquiadens", "memoriens",
"mencheviqu", "mendac", "mendanhens", "mendens",
"mendesens", "mendicant", "mendoncens", "mercant",
"mercedens", "mercesens", "mergulhant", "meridens",
"meridianens", "merirumanens", "meritiens", "merobrigens",
"mertolens", "meruocens", "meruoquens", "mesquitens",
"messanens", "messiens", "messinens", "metalinens",
"metaurens", "metelandens", "metens", "metuapirens",
"mevanionens", "mianmarens", "micaelens", "micenens",
"micuiens", "migrant", "miguelens", "miguelopens",
"milagrens", "milanens", "milasens", "milfontens",
"milharens", "mil"+u"\u00ED"+"ped", "militant", "milliarens",
"mimosens", "mindens", "minduriens", "mineirens",
"mineirolandens", "minens", "minervens", "minodunens",
"minoens", "minorens", "minturnens", "m"+u"\u00ED"+"op",
"mipibuens", "mirabelens", "mirabolant", "miracatuens",
"miracemens", "miracicens", "miraciquens", "miradorens",
"miradourens", "miragaiens", "miraginens", "miraguaiens",
"miraiens", "miraimens", "miraltens", "miralucens",
"miraluzens", "miramantens", "mirambeens", "mirandelens",
"mirandens", "mirandibens", "mirandopolens", "mirangabens",
"miranguens", "mirantens", "mirantonens", "mirapinimens",
"miraporanguens", "mirariens", "mirasselvens", "mirassolandens",
"mirassolens", "mirens", "miri"+u"\u00E1"+"pod", "mirinens",
"mirinzalens", "miritiens", "miritizalens", "miritubens",
"mirizalens", "mirobrigens", "miroroens", "misenens",
"misericordiens", "missionariens", "missionens", "mit",
"mitilenens", "mitilens", "miudezens", "moacirens",
"m"+u"\u00F3"+"bil", "mobralens", "mocajubens", "mocambeirens",
"mocambens", "mocambinhens", "mo"+u"\u00E7"+"ambiquens", "mo"+u"\u00E7"+u"\u00E2"+"med",
"mo"+u"\u00E7"+"amedens", "mo"+u"\u00E7"+u"\u00E1"+"rab", "mococens", "mocoens",
"mocoquens", "mocotoens", "modelens", "modenens",
"modernens", "modernizant", "modestens", "moedens",
"moemens", "moendens", "mogadourens", "mogeirens",
"mogonciacens", "moibandens", "moimentens", "moinant",
"moinhens", "moiporaens", "moirabens", "moitens",
"moitinhens", "mojeirens", "mojigua"+u"\u00E7"+"uens", "mojigui"+u"\u00E7"+"abens",
"mojimirinens", "mojuens", "mojutaperens", "mol",
"molequens", "molheirens", "molhens", "moloquens",
"molumbens", "momanaens", "momatens", "mombacens",
"mombejens", "mombucens", "mombuquens", "mon"+u"\u00E7"+"anens",
"monchiquens", "moncionens", "mon"+u"\u00E7"+"onens", "moncorvens",
"mondaiens", "mond", "mondeguens", "mondinens",
"mondubinens", "moneraens", "monerens", "monetens",
"monfortens", "mongaguaens", "mongubens", "monicens",
"monjolandens", "monjolens", "monjoliens", "monsanens",
"montalegrens", "montalvanens", "montalvaniens", "montanhens",
"montant", "montargilens", "montauriens", "monteirens",
"montemorens", "montenegrens", "montens", "montesens",
"montevideuens", "montezumens", "montijens", "montmartrens",
"montoitens", "montrealens", "monumentens", "moquenzens",
"moradens", "moralens", "morangabens", "moraujens",
"morceguens", "mordent", "moreirens", "morenens",
"morens", "morgantens", "moriruens", "mormacens",
"mororoens", "morparaens", "mortaguens", "mortugabens",
"morumbiens", "morungabens", "morungavens", "mosqueirens",
"mossoroens", "mostardens", "mosteirens", "motens",
"motiens", "motivant", "mouranens", "mourens",
"mouronens", "mousteriens", "movent", "movimentens",
"moxotoens", "mozarlandens", "mozarlandiens", "mozondoens",
"muanaens", "muanens", "mucajaens", "mucajaiens",
"mucambens", "mucuens", "mucujeens", "mucuratens",
"mucuriciens", "mucuriens", "mucuripens", "mudancens",
"mudanciens", "muiraneuens", "muladens", "mulatens",
"muliternens", "multiform", "multiplex", "mult"+u"\u00ED"+"plic",
"multirresistent", "mulunguens", "mumbabens", "mumbucens",
"mumbuquens", "mumificant", "mundauens", "munduriens",
"mundurucuens", "mungubens", "munhozens", "municipalens",
"mun"+u"\u00ED"+"cip", "muniens", "muniquens", "muqui"+u"\u00E7"+"abens",
"muquiens", "murajaens", "murcens", "murens",
"muriaeens", "muribecens", "muribequens", "muriciens",
"muricizalens", "muri"+u"\u00E7"+"oquens", "muriens", "muritibens",
"muritiens", "muriuens", "murmurant", "murmurejant",
"murtinhens", "murtosens", "murucupiens", "murunduens",
"murupeens", "mururuens", "murutinguens", "musticens",
"mustierens", "mutaens", "mutambeirens", "mutambens",
"mutanens", "mutant", "mut"+u"\u00E3"+"zens", "muticens",
"mutinens", "mutucens", "mutucumens", "mutuens",
"mutuipens", "mutumparanaens", "mutunens", "mutunzens",
"mutuquens", "muzambinhens", "myanmens", "nabuquens",
"nacamurens", "nacipens", "nadant", "nadens",
"nagadiens", "najeens", "nantens", "nanuquens",
"napocens", "naquens", "narandibens", "naraniuens",
"narbonens", "narcotraficant", "narniens", "narrant",
"naruens", "nascenciens", "nascent", "nascentens",
"nassadens", "natalens", "natant", "naterciens",
"natinguiens", "nativens", "natividadens", "natividens",
"nativitatens", "natubens", "nauariens", "nauruens",
"nauseant", "navarrens", "navegant", "navegantens",
"naviens", "naviraiens", "nazareens", "nazarenens",
"nazarezinhens", "nazariens", "neandertalens", "neblinant",
"necromant", "neerlandens", "negerens", "negligent",
"negociant", "negr"+u"\u00E3"+"oens", "negrejant", "negrens",
"negromant", "nelens", "nemausens", "nemausiens",
"nemetacens", "nemetens", "nemorens", "neocomiens",
"neocrucens", "neoerens", "neolimens", "neorresendens",
"neorrussens", "neossantarenens", "neossourens", "neozelandens",
"neozelandiens", "nepalens", "nepanens", "nepomucenens",
"neriens", "neropolens", "nertobrigens", "nerulonens",
"nescaniens", "netinens", "neustriens", "nevens",
"nhambiquarens", "nhamundaens", "nhandearens", "nhandutibens",
"nhecolandens", "nhecolandiens", "nhonhepens", "nhungua"+u"\u00E7"+"uens",
"nicaraguens", "niceens", "nicens", "niciens",
"nicolandens", "nicolandiens", "nicolauens", "nicomedens",
"nidificant", "nigerens", "nigromant", "nilens",
"nioaquens", "nipoanens", "nipoens", "niponens",
"niquelandens", "niquelandiens", "nisens", "nitent",
"niteroiens", "nitrificant", "nivernens", "nizens",
"nobilitant", "nobr", "nogueirens", "nolens",
"n"+u"\u00F4"+"mad", "nonatens", "nonoaiens", "nonsens",
"norbanens", "norbens", "nordestens", "nordestinens",
"norens", "noronhens", "nortelandens", "nortelandiens",
"nortens", "northamptonens", "norueguens", "novalens",
"novariens", "novens", "novilhonens", "novocomens",
"nubent", "nubiens", "nugua"+u"\u00E7"+"uens", "numiniens",
"nupebens", "nuporanguens", "nutrient", "oacariens",
"oasiens", "obdiplost"+u"\u00EA"+"mon", "obedient", "obidens",
"objetivant", "obrant", "obsedant", "obsediant",
"observant", "obstant", "obulconens", "ocarens",
"ocau"+u"\u00E7"+"uens", "oceaniens", "ocelens", "ocidentalizant",
"ocidentens", "ocorrent", "ocr", "octodurens",
"ocultens", "ocunens", "ocupant", "odemirens",
"odivelens", "odorant", "oduciens", "oeirens",
"oestens", "ofegant", "oferent", "ofertant",
"oficiant", "oficinens", "ofuscant", "oiapoquens",
"oiteirens", "oitiuens", "oitizeirens", "olariens",
"olarionens", "olbiens", "olbonens", "oleastrens",
"oleens", "olegariens", "oleirens", "olent",
"olhanens", "olifant", "oligost"+u"\u00EA"+"mon", "olimpens",
"olimpiens", "olindens", "olindinens", "olisiponens",
"olissiponens", "olivalens", "olivedens", "oliveirens",
"olivencens", "olivenciens", "olivens", "oliviens",
"omanens", "omelet", "omnipotent", "omniscient",
"oncens", "ondeant", "ondinens", "ondulant",
"onens", "onhens", "onipotent", "onipresent",
"oniscient", "onofrens", "onubens", "operant",
"opinant", "oponent", "optant", "oradens",
"oratoriens", "or"+u"\u00E7"+"ant", "ordinarot", "ordoviciens",
"oricoens", "orientalens", "orient", "orientens",
"orindiuvens", "oriolens", "orionens", "oriximinaens",
"orizaniens", "orizonens", "orlandiens", "orleanens",
"oroandens", "oroboens", "orocoens", "orosens",
"ortigueirens", "ortonens", "ortroniens", "osasquens",
"oscens", "oscilant", "osoriens", "ossonobens",
"ostiens", "ostrens", "otroniens", "ouguelens",
"ouraniens", "oureens", "ouremens", "ourenens",
"ourens", "ourenzens", "ouricanens", "ouricuriens",
"ouriens", "ourinhens", "ouriquens", "ourivens",
"ourizonens", "outeirens", "outorgant", "ouvidorens",
"ouvint", "ovant", "ovarens", "ovetens",
"ovitens", "oxidant", "pacaembuens", "pacajaens",
"pacajuens", "pacatubens", "pacatuens", "pacens",
"pachequens", "pacienciens", "paciens", "pacient",
"pa"+u"\u00E7"+"oquens", "pacotiens", "pacotubens", "pacovalens",
"pacovalzinhens", "pacuens", "pacuiens", "pacujaens",
"pacuriens", "pacutiens", "pacutubens", "padecent",
"padilhens", "paduens", "pagant", "paiaguaens",
"paiaiaens", "paialens", "paineirens", "painelens",
"painens", "paiolens", "paiolinhens", "paiquereens",
"paivens", "pajauens", "pajeens", "pajeuens",
"paju"+u"\u00E7"+"arens", "palamens", "palapoutens", "palauens",
"palenens", "palens", "paleomanens", "palermens",
"palestinens", "palestrant", "palhalens", "palhanenens",
"palhens", "palhetens", "palhocens", "palhocinhens",
"palionens", "palmarens", "palmatoriens", "palmeiralens",
"palmeirandens", "palmeirens", "palmeirinens", "palmeirinhens",
"palmelens", "palmens", "palm"+u"\u00ED"+"ped", "palmirens",
"palmitalens", "palmitens", "palmitinhens", "palmopolens",
"palomens", "palotinens", "palpitant", "paltonens",
"paluens", "pamonhens", "pampanens", "pampeirens",
"pampilhosens", "panacuiens", "panamaens", "panambiens",
"panamens", "panaripens", "pancens", "panelens",
"panfiliens", "pangareens", "panoiens", "panoramens",
"panquens", "pantanalens", "pantanalzinhens", "pantanens",
"panteens", "panticapens", "papagaiens", "papanduvens",
"papariens", "papoulens", "papuriens", "paquens",
"paquequerens", "paquetaens", "paquet", "paquevirens",
"paquicamens", "paquiderm", "paquistanens", "paracambiens",
"paracambuquens", "paracatuens", "paracuaens", "paracuruens",
"paradens", "paradisens", "paraens", "parafusens",
"paragua"+u"\u00E7"+"uens", "paraguaiens", "paraguariens", "paraibanens",
"paraibens", "paraibunens", "paraiens", "paraipabens",
"paraisens", "paraisolandens", "paraisolandiens", "paraisopolens",
"paraitinguens", "paraiuvens", "parajuens", "parajuruens",
"paralisant", "paralissens", "paramanens", "parambuens",
"paramirinens", "paramirinhens", "paramotiens", "paranabiens",
"paranacitens", "paranacityens", "paranaens", "paran"+u"\u00E3"+"ens",
"paranagua"+u"\u00E7"+"uens", "paranaguaens", "paranaibens", "paranaiens",
"paranajiens", "parananens", "paranapanemens", "paranapoemens",
"paranapuanens", "paranariens", "paranatamens", "paranavaiens",
"paran"+u"\u00E3"+"zens", "parangabens", "paranhens", "paraopebens",
"parapeumens", "parapuanens", "parapuens", "paraquemens",
"parariens", "paratariens", "parateiens", "paratibens",
"paratiens", "paratijiens", "paratinguens", "paratudalens",
"parauaraens", "parauens", "paraunens", "parazinhens",
"parceirens", "pardalens", "pardilhoens", "pardinhens",
"pareciens", "paredens", "paredonens", "parelheirens",
"parelhens", "parent", "paricaens", "paricatubens",
"pariconhens", "parijoens", "parintinens", "paripiens",
"paripiranguens", "paripueirens", "pariquerens", "paririens",
"parisiens", "parmens", "parnaguaens", "parnaguens",
"parnaibens", "parnaiuicavens", "parnamirinens", "parnaramens",
"parnasens", "parobeens", "parodiant", "pars",
"participant", "part"+u"\u00ED"+"cip", "parturient", "paruruens",
"pascoalens", "pascoens", "passabenens", "passabenzens",
"passagenens", "passagens", "passagenzens", "passant",
"passarinhens", "passaunens", "passavant", "passeant",
"passeens", "passens", "passeriform", "passirens",
"pastens", "pastilhens", "pastinhens", "pastorens",
"pataibens", "patamuteens", "patavicens", "pateant",
"pateironens", "patens", "patiens", "patif",
"patizalens", "patracens", "patrens", "patriarquens",
"patriciens", "patrimoniens", "patrocinens", "patrociniens",
"patrulhens", "patuens", "pauapixunens", "pauiniens",
"paulens", "pauliceens", "pauliceiens", "paulinens",
"pauliniens", "paulistanens", "paulistaniens", "paulistens",
"paulivens", "pausulens", "pauxiens", "paveramens",
"paviens", "pavoens", "pavonens", "pavu"+u"\u00E7"+"uens",
"pavunens", "paxiuens", "paxuens", "pazens",
"peabiruens", "pebens", "pe"+u"\u00E7"+"anhens", "pecenens",
"pecenzens", "pedant", "pederneirens", "pedestr",
"pedint", "pedralvens", "pedrariens", "pedregulhens",
"pedreirens", "pedrens", "pedrinhens", "pedritens",
"pedroguens", "pedronens", "pedrosens", "peixens",
"peixinhens", "peixotens", "peju"+u"\u00E7"+"arens", "peladens",
"pelenens", "pelens", "peloponens", "pelotens",
"penacovens", "penafidelens", "penafortens", "penaguiens",
"penalvens", "penamacorens", "penamarens", "penant",
"penapolens", "pendanguens", "pendenciens", "pendent",
"penedens", "penediens", "penedonens", "penelens",
"penetrant", "penichens", "penitent", "pensant",
"pepitalens", "pequeriens", "pequiaens", "pequiens",
"pequiriens", "pequizeirens", "perapeunens", "percevejens",
"perciform", "percucient", "perdicens", "perdicionens",
"perdigonens", "perdiguens", "perdizens", "perdonens",
"peregrinens", "pereirens", "pereirinhens", "peren",
"perequeens", "perfurant", "pergens", "pergunt",
"periclitant", "pericoens", "perigarens", "perigosens",
"perimboens", "perimiriens", "perimirinens", "periquarens",
"periquitens", "peritoroens", "permanent", "pernambuquens",
"pernambuquinhens", "perobalens", "perobens", "perolens",
"perpetuens", "perrengu", "perrexilens", "perscrutant",
"perseveranciens", "perseverant", "persistent", "pertencent",
"pertilens", "pertinent", "perturbant", "peruanens",
"peruens", "peruibens", "pervagant", "pescadorens",
"pesqueirens", "pesqueirinhens", "pessanhens", "pessegueirens",
"pessoens", "pest", "pestilent", "petavionens",
"petavoniens", "petinens", "petragoricens", "petrens",
"petrogradens", "petrolandens", "petrolandiens", "petrolinens",
"petrovinens", "petulant", "petuniens", "piabanhens",
"piabens", "piacabu"+u"\u00E7"+"uens", "pia"+u"\u00E7"+"abu"+u"\u00E7"+"uens", "piacaens",
"piacatubens", "piacatuens", "pia"+u"\u00E7"+"uens", "piancoens",
"piassabu"+u"\u00E7"+"uens", "piat"+u"\u00E3"+"ens", "piatanens", "piat"+u"\u00E3"+"zens",
"piauens", "piauiens", "piaunens", "piavorens",
"picadens", "picadinhens", "picant", "pichens",
"piciform", "picinguabens", "picoens", "pictavens",
"picuens", "picuiens", "piedadens", "piedens",
"pienens", "piens", "pietatens", "pilarens",
"pilens", "piliens", "pilonens", "pimenteiralens",
"pimenteirens", "pimentelens", "pimentens", "pinareens",
"pincens", "pindaibens", "pindaiens", "pindaivens",
"pindamonhangabens", "pindareens", "pindens", "pindoba"+u"\u00E7"+"uens",
"pindobalens", "pindobens", "pindoguabens", "pindoramens",
"pindoretamens", "pindotibens", "pindovalens", "pinens",
"pingadorens", "pingant", "pingent", "pinhalens",
"pinhalitens", "pinhalonens", "pinhalzinhens", "pinh"+u"\u00E3"+"oens",
"pinh"+u"\u00E3"+"ozens", "pinh"+u"\u00E3"+"ozinhens", "pinheiralens", "pinheirens",
"pinheirinhens", "pinhelens", "pinhonens", "pinhotibens",
"pin"+u"\u00ED"+"ped", "pintadens", "pintens", "pinturilant",
"pioneirens", "pionens", "pipeirens", "pipens",
"pipilant", "pipinuquens", "piquens", "piquerobiens",
"piquetens", "piquiriens", "pirabeirabens", "pirabens",
"pirabibuens", "piracaibens", "piracaiens", "piracanjubens",
"piracauaens", "piracemens", "piracicabens", "pira"+u"\u00E7"+"ununguens",
"piracurucens", "piracuruquens", "piraguarens", "piraiens",
"pirainhens", "piraiuarens", "pirajaens", "pirajibens",
"pirajubens", "pirajuens", "pirajuiens", "piramboiens",
"pirambuens", "pirangaiens", "pirangu"+u"\u00E7"+"uens", "piranguens",
"piranguinhens", "piranguitens", "piranhens", "piranjiens",
"piranjinens", "pirapanemens", "pirapemens", "pirapetinguens",
"pirapitinguiens", "pirapoens", "piraporens", "pirapozens",
"pirapozinhens", "piraputanguens", "piraquarens", "piratabens",
"piratiniens", "piratininguens", "piratubens", "pirauaens",
"piraubens", "pirens", "pirgens", "piriaens",
"piriens", "piriform", "piripaens", "piripiriens",
"piritibens", "piritubens", "pirizalens", "pirizeirens",
"pirpiritubens", "pisaurens", "pisauriens", "piscambens",
"pistoriens", "pitangueirens", "pitanguens", "pitanguiens",
"pitimbuens", "pititinguens", "pitombeirens", "pitubens",
"pitunens", "piuens", "piuiens", "piumens",
"piunens", "piunzens", "piuvens", "pixainens",
"pixainzens", "planaltens", "planaltinens", "planetens",
"plangens", "plangent", "planinens", "planurens",
"plastificant", "plateens", "platelmint", "platens",
"platinens", "pleninens", "pleuronectiform", "poaens",
"poaiens", "pobr", "po"+u"\u00E7"+"anens", "po"+u"\u00E7"+u"\u00E3"+"oens",
"po"+u"\u00E7"+u"\u00E3"+"ozens", "pocens", "pochiquens", "pocinhens",
"pocionens", "po"+u"\u00E7"+"oens", "poconeens", "po"+u"\u00E7"+"oneens",
"pocranens", "podimirinens", "podr", "poeirens",
"poent", "poiarens", "poiriens", "pojuquens",
"polens", "policarpens", "politizant", "polivalent",
"poloniens", "poluent", "polvorens", "pombalens",
"pombens", "pombinhens", "pomeranens", "pomerodens",
"pometinens", "pompeens", "pompelonens", "pompeuens",
"poncianens", "pongaiens", "pontalens", "pontaletens",
"pontalindens", "pontalinens", "pontalzinhens", "pontens",
"pontevilens", "pontezinhens", "pontiens", "pontinens",
"pontinhens", "pontonens", "populinens", "populoniens",
"porangabens", "porangatuens", "poranguens", "porciunculens",
"porecatuens", "porfiriens", "poronguens", "pororoquens",
"portaginens", "portalegrens", "porteirens", "porteirinhens",
"portelens", "portens", "portimonens", "portinhens",
"portlandiens", "portonens", "portucalens", "portuens",
"portugalens", "porvirens", "possant", "possens",
"possident", "postulant", "poteens", "potenjiens",
"potent", "potiens", "potiraguaens", "potiraguarens",
"potiraguens", "potirendabens", "potirens", "potiretamens",
"potreirens", "potreiritens", "potunduvens", "pousens",
"povens", "povoacens", "povoens", "poxorens",
"poxoreuens", "pracinhens", "pradens", "pradinhens",
"pradosens", "praguens", "praiens", "prainhens",
"pranchadens", "pranchitens", "prataniens", "pratapolens",
"pratens", "praticant", "pratinhens", "prazerens",
"precedent", "prec"+u"\u00ED"+"pit", "precoc", "predisponent",
"predominant", "preeminent", "preexcelent", "preexistent",
"preguens", "preguicens", "prement", "premonstratens",
"prensens", "preocupant", "preponderant", "preponent",
"prepotent", "present", "president", "presidiens",
"presidutrens", "pressionant", "prestant", "prestens",
"pretendent", "pretoriens", "prevalecent", "prevalent",
"prevenidens", "prevident", "prienens", "primaiens",
"primaverens", "primorens", "princesens", "principiant",
"privatizant", "probidadens", "procariont", "procedent",
"procopiens", "proeminent", "proencens", "profetens",
"profissionalizant", "progressens", "promissens", "propagant",
"propelent", "proponent", "propriaens", "prosperidadens",
"protecionens", "proteiform", "protestant", "provenient",
"providenciens", "provident", "provocant", "prudent",
"prudentens", "prusens", "prusiens", "puanens",
"pu"+u"\u00E3"+"zens", "pubent", "pubescent", "pu"+u"\u00E7"+"aens",
"pucalpens", "puguens", "puira"+u"\u00E7"+"uens", "pujant",
"puladorens", "pulsant", "puluens", "punens",
"pungent", "puntiform", "pupiniens", "purezens",
"purgant", "purilandens", "purilandiens", "pusil"+u"\u00E2"+"nim",
"putinguens", "putiriens", "puxavant", "puxinan"+u"\u00E3"+"ens",
"puxinan"+u"\u00E3"+"zens", "quadrant", "quadrens", "quadriburgens",
"quadr"+u"\u00FA"+"ped", "quajaraens", "quara"+u"\u00E7"+"uens", "quaraiens",
"quarentens", "quariens", "quartelens", "quataens",
"quati"+u"\u00E7"+"abens", "quatiens", "quatiguabens", "quatiguaens",
"quatiguens", "quatipuruens", "quatiquanens", "quatitubens",
"quebequens", "queboens", "quebrachens", "quebrangulens",
"quebrant", "que"+u"\u00E7"+"abens", "quedens", "queimadens",
"queirosens", "queixadens", "queluzens", "queluzitens",
"quent", "queremens", "querenciens", "querent",
"querronens", "quersonens", "quevedens", "qui"+u"\u00E7"+"amaens",
"qui"+u"\u00E7"+"am"+u"\u00E3"+"ens", "qui"+u"\u00E7"+"amanens", "quiceens", "quiescent",
"quietens", "quijinguens", "quilombens", "quil"+u"\u00F3"+"pod",
"quimamiens", "quimiossintetizant", "quincoeens", "quincuncaens",
"quintanens", "quintens", "quintinens", "quipaens",
"quipapaens", "quiririens", "quiririnens", "quiriritiuens",
"quiromant", "quitandinhens", "quiterianopolens", "quiteriens",
"quitimbuens", "quitundens", "quitunguens", "quixabeirens",
"quixabens", "quixadaens", "quixariuens", "quixeloens",
"quixeramobiens", "quixeramobinens", "quixereens", "quixoaens",
"quizotens", "rabdomant", "rabecens", "rabequens",
"rabichens", "rabugic", "rabulic", "raciarens",
"raciocinant", "radiadorens", "radiant", "radiouvint",
"rafaelens", "rafardens", "raimundens", "raizalens",
"raizamens", "raizens", "rajadens", "ramadens",
"ramalhens", "ramalhetens", "ramens", "ramirens",
"ramnens", "ramonens", "ranchariens", "ranchens",
"ranchinhens", "rangent", "rapac", "rapant",
"rapinant", "raposens", "rareant", "rarescent",
"rasant", "rascant", "rasens", "raspadorens",
"rastejant", "rateens", "ratens", "ratic",
"ratit", "ratonens", "ravenatens", "ravenens",
"reafirmant", "reagent", "realens", "realezens",
"rebeld", "rebent", "reboant", "rebojens",
"rebordosens", "reboucens", "rebrilhant", "recalcitrant",
"recendent", "recent", "reciens", "recifens",
"recipient", "reclamant", "reconfortant", "recorrent",
"recreens", "recuperadens", "recursens", "recusant",
"recusens", "redencens", "redencionens", "reden"+u"\u00E7"+"onens",
"redent", "redentorens", "redinhens", "redolent",
"redondens", "redundant", "redutens", "referent",
"refervent", "reformens", "refrigerant", "refugiadens",
"refulgent", "regant", "regenciens", "regeneracionens",
"regenera"+u"\u00E7"+"onens", "regenerens", "regent", "regentens",
"regiens", "reginens", "reginopolens", "registrens",
"regrant", "reguenguens", "reguens", "regurgitant",
"reinant", "reincident", "relampejant", "relaxant",
"relevant", "reliquiens", "relumbrant", "relutant",
"reluzent", "relvadens", "remanescent", "remansonens",
"remediens", "remens", "r"+u"\u00EA"+"mig", "remigens",
"remigiens", "reminiscent", "remont", "renascencens",
"renascent", "renitent", "renqu", "rent",
"repartimenens", "repelent", "repent", "repetent",
"replicant", "repont", "repousant", "representant",
"reptant", "repugnant", "requerent", "requint",
"reriutabens", "rescendent", "resendens", "reservens",
"resfolegant", "residenciens", "resident", "resistent",
"respeitant", "resplandecent", "resplandens", "resplandescent",
"resplendent", "resplendorens", "respondent", "ressaltant",
"ressaquens", "ressoant", "ressonant", "restant",
"restauracionens", "restaurant", "restinguens", "restringent",
"resultant", "resvalant", "retardant", "reticent",
"retiens", "retirant", "retirolandens", "retirolandiens",
"retumbant", "reumatoid", "reverberant", "reverent",
"revoltant", "riach"+u"\u00E3"+"oens", "riachens", "riachinhens",
"riachonens", "riachuelens", "rialmens", "rialtens",
"riazulens", "ribacudens", "ribaduriens", "ribamarens",
"ribatejens", "ribeir"+u"\u00E3"+"oens", "ribeirens", "ribeirinhens",
"ribeironens", "ribeiropolens", "ricinens", "rident",
"riens", "rifainens", "rinareens", "rinconens",
"rinoceront", "rinopolens", "rioclarens", "riograndens",
"riolandens", "riozinhens", "riquezens", "ritapolens",
"ritens", "rivens", "riversulens", "robenhausiens",
"robertens", "ro"+u"\u00E7"+"adens", "ro"+u"\u00E7"+"agant", "rochedens",
"rochedinhens", "rocinant", "rodaniens", "rodant",
"rodeens", "rodeiens", "rodeirens", "rodens",
"rodiens", "rodopiant", "roladorens", "rolandens",
"rolandiens", "rolant", "rolantens", "rolimorens",
"rom"+u"\u00E3"+"oens", "romarens", "romariens", "romelandens",
"romelandiens", "rompant", "rompent", "roncadens",
"roncadorens", "rondant", "rondinhens", "rondonens",
"rondoniens", "ronflant", "roquens", "roraimens",
"rorejant", "rosalandens", "rosalens", "rosaliens",
"rosalindens", "rosanens", "rosarens", "rosariens",
"roseiralens", "roseirens", "roteirens", "rotomagens",
"rotundens", "ruandens", "rubelitens", "rubescent",
"rubiaceens", "rubiacens", "rubiatabens", "rubiens",
"rubineens", "rubinens", "rud", "rugent",
"ruilandens", "ruinens", "ruminant", "rumorejant",
"rupestr", "russens", "rutenens", "rutilant",
"saarens", "sabaens", "sabaraens", "sabarens",
"sabaudiens", "sabaunens", "sabiaens", "sabiaguabens",
"sabinens", "sabinopolens", "saboeirens", "saboguens",
"saboiens", "sabonetens", "saborens", "sabratens",
"sabrosens", "sabugalens", "sabugueirens", "sabujiens",
"sacadens", "sacambuens", "sacavenens", "sacerdot",
"sacoiens", "sacramentens", "sacripant", "sacrocordiens",
"sadens", "saens", "safarens", "safirens",
"sagradens", "sagrens", "saibrens", "saic"+u"\u00E3"+"zens",
"saiens", "sainaumens", "saireens", "sajens",
"salaciens", "salamanquens", "salamanticens", "salariens",
"salavessens", "saldanhens", "salens", "salesens",
"salesopolens", "saletens", "salgadaliens", "salgadens",
"salgadinhens", "salgueirens", "salient", "salinadens",
"salinens", "saliniens", "salininhens", "salitrens",
"salmanticens", "salmouranens", "saloaens", "salobinhens",
"salobrens", "salomeens", "salonens", "salonicens",
"salsens", "saltant", "saltens", "saltinhens",
"saltitant", "salubr", "salutariens", "salvacionens",
"salvadorens", "salvaterrens", "salvens", "salzedens",
"samambaiens", "samaumens", "sambaetibens", "sambaibatubens",
"sambaibens", "sambaquiens", "sambarnabens", "sambeens",
"sambeneditens", "sambernardens", "sambertens", "sambonaventurens",
"samoens", "samorens", "samosatens", "samuelens",
"sanagens", "sananduvens", "sancandidens", "sanderens",
"sandesideriens", "sandominicens", "sandovalens", "sandovalinens",
"sandumonens", "sanens", "sanestefanens", "sanfelicens",
"sanfirminens", "sanfranciscens", "sangalhens", "sanganens",
"sanganhanens", "sangonens", "sanguessuguens", "sanharoens",
"sanjoanens", "sanjosefens", "sanludovicens", "sansimoniens",
"santanderiens", "santanens", "santanesiens", "santareens",
"santarenens", "santarens", "santelmens", "santens",
"santiaguens", "santomens", "santuens", "s"+u"\u00E3"+"omarinhens",
"sapatariens", "sapea"+u"\u00E7"+"uens", "sapecadens", "sapeens",
"sapens", "sapezalens", "sapient", "sapiranguens",
"sapopemens", "sapucaiens", "sapucaranens", "sapuparens",
"saquaremens", "saquinhens", "saracurens", "saracuruens",
"saraivens", "sarandiens", "sarandirens", "sarapuiens",
"sararaens", "sardiens", "sardinhens", "sardoaens",
"sardoalens", "sarens", "sargentens", "sarutaiaens",
"sarzedens", "sat"+u"\u00E9"+"lit", "satanens", "satens", "satubens",
"satubinens", "satubinzens", "saturnens", "saubarens",
"saudadens", "saudavelens", "saudens", "saueens",
"sauiaens", "sauraurens", "saur"+u"\u00F3"+"pod", "sauvens",
"seabrens", "searens", "sebalens", "sebastianens",
"sebastopolens", "seberiens", "secadorens", "secanens",
"secant", "sedens", "segestens", "segiens",
"segisamajuliens", "segisamonens", "segobrigens", "segontinens",
"segredens", "seguint", "segundens", "segurens",
"segusiens", "seichelens", "seivalens", "seixalens",
"selbachens", "sel", "seleucens", "seleuciens",
"selmens", "selvens", "semelhant", "semovent",
"senadorens", "senegalens", "senegambiens", "senens",
"senhoritens", "sentinelens", "separacionens", "sepetibens",
"sepotubens", "sepulturens", "sequens", "sequent",
"serenens", "sergiens", "sergipens", "seribinhens",
"sericitens", "seridoens", "seriens", "seritinguens",
"serjiens", "sernachens", "sernancelhens", "serpeant",
"serpens", "serpenteant", "serralitrens", "serranens",
"serraniens", "serrariens", "serrazens", "serrens",
"serrinhens", "serritens", "serrolandens", "serrolandiens",
"serrotens", "sertaginens", "sertanejens", "sertanens",
"sertaniens", "sertanopolens", "sert"+u"\u00E3"+"ozinhens", "servens",
"servent", "sesimbrens", "setubalens",
"setubinhens", "severens", "severinens", "severiniens",
"seychellens", "siamens", "siariens", "sibilant",
"siboens", "sicens", "sicilens", "siciliens",
"sidens", "sidrolandens", "sidrolandiens", "sienens",
"significant", "silenciens", "silens", "silent",
"silvadens", "silvaldens", "silvanectens", "silvanens",
"silvaniens", "silvant", "silveiraniens", "silveirens",
"silvens", "silveriens", "silvestr", "silvestrens",
"silvianopolens", "silviens", "simbiont", "s"+u"\u00ED"+"mil",
"simonens", "simonesiens", "simonsenens", "simpatiens",
"simpatizant", "simpliciens", "sinadens", "sincoraens",
"sindens", "sineens", "sinfanens", "singapurens",
"singidunens", "singlet", "singrant", "siniens",
"sinimbuens", "sinopens", "sintrens", "sipicueens",
"siqu", "siqueirens", "siquimens", "siriens",
"sirigadens", "sirijiens", "sirinhaenens", "siririens",
"sirmiens", "sisaponens", "sisolens", "sitiaens",
"sitiant", "sitiens", "sitionovens", "siupeens",
"soajens", "soant", "soarens", "sobejant",
"soberanens", "sobradens", "sobradinhens", "sobralens",
"sobraliens", "sobrant", "sobreirens", "sobrejacent",
"sobrelevant", "sobressalent", "sobrevivent", "socamirinens",
"socavanens", "sociedadens", "socorrens", "sodrelandens",
"sodrelandiens", "sodreliens", "sofalens", "solaneens",
"solanens", "soledadens", "solemarens", "solen",
"solert", "solicitant", "solimariacens", "solu"+u"\u00E7"+"ant",
"solvens", "solvent", "sonant", "sopens",
"soquens", "sorocabanens", "sorocabens", "sororoens",
"sorrident", "sosseguens", "soteiens", "sourens",
"sousaniens", "souselens", "sousens", "soutens",
"stevensonens", "sua"+u"\u00E7"+"uiens", "sua"+u"\u00E7"+"uranens", "suanens",
"suav", "suazilandens", "suazilandiens", "subaiens",
"subaumens", "subchef", "subcomponent", "subconscient",
"subidens", "subjacent", "sublim", "sublimizant",
"subsequent", "subservient", "subsistent", "substituint",
"subtenent", "suburanens", "sucatinguens", "sucessens",
"suciant", "sucubens", "sucupirens", "sucureens",
"sucuriens", "sucurijuens", "sucuriuens", "sucurizinhens",
"sucuruens", "sucurujuens", "sudanens", "sudoest",
"suessiens", "suezens", "sufetulens", "suficient",
"sufocant", "sugant", "suicens", "sulcens",
"sulinens", "sulmonens", "sumareens", "sumaumens",
"sumeens", "sumidourens", "sumuriens", "superaguiens",
"supergigant", "superimportant", "superintendent", "superpotent",
"supervenient", "suplent", "suplicant", "s"+u"\u00FA"+"plic",
"supurant", "surfactant", "surinamens", "surpreendent",
"surpresens", "surrealizant", "surrinens", "surubinens",
"suruiens", "susanens", "suspens", "suspirant",
"suspirens", "sussurrant", "sustinens", "tabacens",
"tabaiens", "tabainhens", "tabajarens", "tabapuanens",
"tabaquens", "tabatinguens", "tabaunens", "tabenens",
"taberoeens", "tabirens", "taboadens", "tabocalens",
"taboquens", "taboquinhens", "tabosens", "tabracens",
"tabuacens", "tabuadens", "tabualens", "tabuanens",
"tabudens", "tabuens", "tabuinhens", "tabuleirens",
"tabuleironens", "tacaimboens", "tacaleens", "tacaratuens",
"tachens", "taciateuens", "tacibens", "tacimens",
"tacimirinens", "taciuanens", "tacteant", "tacuiquenens",
"tacuruens", "tadeuens", "tagant", "tagastens",
"taguaiens", "taguatinguens", "taia"+u"\u00E7"+"uens", "taia"+u"\u00E7"+"upebens",
"taiamanens", "taibens", "tainens", "tainhens",
"taiobeirens", "taipens", "taipinhens", "taipuens",
"tairetaens", "taiuvens", "tajatubens", "tajiens",
"tajuabens", "talabricens", "talabrigens", "talarens",
"talhadens", "talhant", "talimanens", "tamandareens",
"tamanduaens", "tamaniquaens", "tamanqueirens", "tamaquariens",
"tamaranens", "tamarineirens", "tambaquiens", "tambauens",
"tambeens", "tamboarens", "tamboataens", "tamborilens",
"tamborilzinhens", "tamburiens", "tameguens", "tamequens",
"tamoiens", "tamulens", "tanabiens", "tanariens",
"tancredinhens", "tanetiens", "tangaraens", "tangent",
"tangilens", "tanguaens", "tanha"+u"\u00E7"+"uens", "tanquens",
"tanquinhens", "tapaiunens", "taparaens", "taparubens",
"tapauaens", "tapejarens", "tapens", "taperens",
"taperiens", "taperinhens", "taperoaens", "taperuabens",
"tapinens", "tapiracanguens", "tapiraiens", "tapirainens",
"tapiraipens", "tapiramens", "tapiramutaens", "tapiranguens",
"tapirapuanens", "tapiratibens", "tapirens", "taporens",
"tapuiens", "tapuiramens", "tapuruens", "tapuruquarens",
"taquapiriens", "taquara"+u"\u00E7"+"uens", "taquaralens", "taquarandiens",
"taquaremboens", "taquarens", "taquariens", "taquarilens",
"taquarinhens", "taquaritinguens", "taquaritubens", "taquarivaiens",
"taquarixinens", "taquaru"+u"\u00E7"+"uens", "tarabaiens", "taraens",
"taraguaens", "taramens", "taraquaens", "tarauacaens",
"tardenens", "tardenoisens", "tardenoisiens", "targinens",
"tarianens", "taritubens", "tarouquens", "tarquiniens",
"tarracens", "tarracinens", "tarraconens", "tarrafens",
"tarragens", "tarsens", "tartamudeant", "tartaruguens",
"tarua"+u"\u00E7"+"uens", "tarumanens", "tarumirinens", "tarumirinhens",
"tarusconiens", "tasmaniens", "tatairens", "tatajubens",
"tate/ment", "tateant", "tatuamunhens", "tatuens",
"tatuiens", "tatuquarens", "tauaens", "tauape"+u"\u00E7"+"a"+u"\u00E7"+"uens",
"tauapens", "tauapiranguens", "tauarianens", "tauari"+u"\u00E3"+"zinhens",
"tauariens", "tauaruanens", "taubateens", "taumataens",
"taumaturgens", "taumaturguens", "taunayens", "taunens",
"tauoaens", "tauriniens", "taurunens", "tavarens",
"tavirens", "tavorens", "teanens", "tebaidens",
"tebens", "te"+u"\u00E7"+"aindaens", "tefeens", "tegianens",
"teixeirens", "teju"+u"\u00E7"+"uoquens", "tejucupapens", "tejupaens",
"telheirens", "telhens", "telmessens", "telonens",
"tement", "temerantens", "temperens", "tendent",
"tenent", "tens", "tentugalens", "t"+u"\u00EA"+"nu",
"tenuipens", "teodorens", "teotonens", "teotoniens",
"terceirens", "terebrant", "terenens", "terereens",
"teresens", "teresinens", "teresinhens", "termens",
"termessens", "terminant", "ter"+u"\u00F3"+"pod", "terracinens",
"terratenent", "terreirens", "terrens", "terrestr",
"terrinhens", "tesourens", "tesourinhens", "tespiens",
"tessalonicens", "tetaniform", "tetens", "tetr"+u"\u00E1"+"pod",
"tetravalent", "teudalens", "teuiniens", "teutoburgiens",
"teutoniens", "tiaguens", "tianguaens", "tiarajuens",
"tibajiens", "tibauens", "tibiens", "tibiri"+u"\u00E7"+"aens",
"tibiriens", "tibunens", "ticiens", "ticinens",
"tieteens", "tigr", "tigrens", "tigrinhens",
"tijipioens", "tijocens", "tijoquens", "tijua"+u"\u00E7"+"uens",
"tijucalens", "tijucens", "tijucu"+u"\u00E7"+"uens", "tijuquens",
"tijuquinhens", "tilintant", "timacens", "timbaubens",
"timbeens", "timbirens", "timboens", "timboraens",
"timboreens", "timboteuens", "timbozalens", "timbozinhens",
"timbuens", "timbuiens", "timburiens", "timiadens",
"timorens", "tinguaens", "tinguatibens", "tipiens",
"tipitiens", "tipl", "tiquaru"+u"\u00E7"+"uens", "tiraciens",
"tiracinens", "tiradentens", "tirant", "tirens",
"tiriens", "tiriricalens", "tiririquens", "tiritant",
"tirocinant", "tirsens", "tissanens", "tissens",
"titeens", "titiaens", "titiarrens", "titilant",
"titubeant", "tiuiens", "toant", "tobatiens",
"tobiarens", "tobiens", "tocaiens", "tocandirens",
"tocanquirens", "tocant", "tocantinens", "tocantiniens",
"togiens", "togolens", "toguens", "toldens",
"toledens", "tolerant", "toleriens", "toloiriens",
"tolosens", "tomarens", "tomarutemens", "tomasiens",
"tomasinens", "tomazinens", "tombadorens", "tombant",
"tombens", "tomens", "tonant", "tonantinens",
"tondelens", "tondens", "tonificant", "toniolens",
"tonitroant", "tonitruant", "tonteant", "tontens",
"topaziens", "topens", "torcidens", "toriparuens",
"toritamens", "torixoreuens", "tormentinhens", "tornacens",
"torneirens", "tornejant", "tornens", "toropiens",
"toroquaens", "torranens", "torreant", "torreens",
"torrens", "torrent", "torreonens", "torriens",
"torrinhens", "torronens", "tortens", "torturant",
"totalizant", "tourens", "tourinhens", "toxicodependent",
"trabijuens", "tracens", "tracuaens", "tracuateuens",
"tracunhaenens", "tracupaens", "traficant", "traianens",
"traipuens", "trairens", "trairiens", "traironens",
"traituens", "trajanens", "tral", "tralens",
"tramandaiens", "trancosens", "tranqueirens", "tranquilizant",
"transbordant", "transcendent", "transeunt", "transformant",
"transient", "transigent", "transparent", "transplant",
"transudant", "trapiaens", "trapichens", "trapipens",
"trasbordant", "trast", "tratant", "traumatizant",
"travesseirens", "travessens", "travessiens", "travessonens",
"treiens", "trejens", "trelens", "tremant",
"tremecens", "tremedalens", "tremeluzent", "tremembeens",
"trement", "tremulant", "trepidant", "trescalant",
"tressuant", "trevisens", "triacens", "triangulens",
"triarroiens", "tribarrens", "tribiquens", "tribiuens",
"triboboens", "triboquens", "tribracens", "tributant",
"tricachoeirens", "tricasens", "triciens", "tricoronens",
"tricorreguens", "tricrucens", "trident", "tridivisens",
"trifigueirens", "triforquilhens", "trifraternens", "trifronteirens",
"triladeirens", "trilagoens", "trilagunens", "trimaiens",
"trimariens", "trimontens", "trimorrens", "trinchant",
"trindadens", "tripassens", "tripetrens", "tripinens",
"tripinheirens", "tr"+u"\u00ED"+"plic", "tripocens", "tripontens",
"tripudiant", "tripulant", "trirranchens", "trirriens",
"trirrivens", "trist", "tristradens", "tritiens",
"triturant", "triunfant", "triunfens", "trivalent",
"trivendens", "troant", "trofens", "troiens",
"trombens", "trombudens", "trombuquens", "tronquens",
"trovejant", "tru"+u"\u00E7"+"uens", "tuaregu", "tubaronens",
"tubendubens", "tubuliform", "tubunens", "tucanens",
"tuciens", "tucumadeuens", "tucumaens", "tucunareens",
"tucundubens", "tucunduvens", "tucunens", "tucunzens",
"tucupareens", "tucuribens", "tucuruiens", "tucuruviens",
"tudens", "tuguriens", "tuinens", "tuiueens",
"tuiuiuens", "tuiutiens", "tuiutinguens", "tularens",
"tumbens", "tumiritinguens", "tunant", "tuneirens",
"tunens", "tungus", "tuntuens", "tuntumens",
"tuntunens", "tupaciguarens", "tup"+u"\u00E3"+"ens", "tupanaciens",
"tupanatinguens", "tupanciretanens", "tupandiens", "tupanens",
"tupantubens", "tupaocens", "tupaoquens", "tuparaceens",
"tuparaiens", "tuparandiens", "tuparendiens", "tuparetamens",
"tup"+u"\u00E3"+"zens", "tupeens", "tupiens", "tupinambaens",
"tupinens", "tupira"+u"\u00E7"+"abens", "tupiramens", "tupiratinens",
"tupitinguens", "turbant", "turbilhonant", "turia"+u"\u00E7"+"uens",
"turiassonens", "turielens", "turinens", "turiubens",
"turmalinens", "turonens", "turonicens", "turtuveant",
"turumanens", "tururuens", "turvaniens", "turvelandens",
"turvens", "turviniens", "turvolandens", "tusculandens",
"tutiens", "tutoiens", "uacananens", "uacariens",
"uacociens", "uacurizalens", "uanaracens", "uapeens",
"uaraens", "uariens", "uarinens", "uarirambaens",
"uarumens", "uatum"+u"\u00E3"+"ens", "uauaens", "uaupeens",
"ubaens", "ubaiens", "ubairens", "ubaitabens",
"ubajarens", "ubaporanguens", "ubaranens", "ubariens",
"ubatanens", "ubatens", "ubatibens", "ubatubens",
"ubaunens", "ubazinhens", "uberabens", "uberlandens",
"uberlandiens", "ubira"+u"\u00E7"+"abens", "ubira"+u"\u00E7"+"uens", "ubiraitaens",
"ubirajarens", "ubirataiens", "ubiratanens", "ubiretamens",
"ubiretens", "uburanens", "uchoens", "ucraniens",
"ugandens", "uibaiens", "uiraponguens", "uiraunens",
"uivant", "ulbiens", "ulianopolens", "uliens",
"ulissiponens", "ulsteriens", "ultrajant", "ultralev",
"ultravident", "ulubrens", "ululant", "umanens",
"umariens", "umaritubens", "umarizalens", "umbaraens",
"umbaubens", "umbeliform", "umbelinens", "umbuens",
"umburanens", "umburaninhens", "umburatibens", "umburetamens",
"umbuzeirens", "umiriens", "umirinens", "umuaramens",
"unaiens", "un"+u"\u00E2"+"nim", "unciform", "undant",
"unens", "unidens", "uniflorens", "uniform",
"unionens", "univalent", "univalv", "universalizant",
"universens", "uomaloliens", "upacaraiens", "upamirinens",
"upanemens", "upatininguens", "upgrad", "uraiens",
"urandiens", "uraniens", "urariquerens", "urent",
"urgent", "urticant", "usbequ", "uscanens",
"uscens", "usinens", "uspens", "utariens",
"utent", "utiaritiens", "uticens", "utinguens",
"uvaens", "uvaiens", "uvalens", "uxamens",
"uxitubens", "uzalens", "uzbequ", "uzitens",
"vacacaiens", "vacaricens", "vacariens", "vacens",
"vacilant", "vacinant", "vadens", "vagant",
"vagens", "vaguens", "vaiamontens", "valadarens",
"valanens", "valanginens", "valboens", "valcovens",
"valdelandens", "valdens", "valdigens", "valencens",
"valenciens", "valens", "valent", "valentens",
"valentinens", "valerianens", "valeriens", "valinhens",
"valonguens", "valpacens", "valparaisens", "valsant",
"valverdens", "vandeens", "vanderleiens", "vanens",
"vangloriens", "vanuatens", "vanuatuens", "vapicens",
"vapincens", "vaquejadorens", "vardagatens", "varganens",
"vargeanens", "vargedens", "varginhens", "vargitens",
"varguens", "variant", "varjanens", "varjotens",
"varpens", "varsoviens", "vartens", "varzeanens",
"varzeariens", "varzedens", "varzeens", "varzelandens",
"varzelandiens", "varzinhens", "vasconcelens", "vascuens",
"vasiens", "vasquesens", "vassourens", "vastens",
"vaudens", "vauens", "vauthierens", "vauzinhens",
"vazamorens", "vazant", "vazantens", "vazlandens",
"vazlandiens", "veadeirens", "veadens", "veadinhens",
"vedant", "veement", "veiguens", "veimarens",
"veirens", "velabrens", "velanens", "velaniens",
"velens", "veliens", "venanciens", "venceslauens",
"venceslauzens", "vendeens", "vendens", "venenens",
"venens", "venetens", "venezens", "ventaniens",
"ventripotent", "veraneant", "veranens", "veravens",
"vercelens", "verd", "verdejant", "verdejantens",
"verdens", "verdunens", "veredens", "veredinhens",
"vereens", "verens", "vergiliens", "verissimens",
"vermelhens", "vermiform", "vermoilens", "verodunens",
"veronens", "verruguens", "vertent", "vertentens",
"vesicant", "vesonciens", "vespasianens", "vestiens",
"veterocretens", "vetonens", "vetuloniens", "viaciens",
"viadutens", "viajant", "viamanens", "viamonens",
"viandant", "vianens", "vianopolens", "viatens",
"viboniens", "vibrant", "vic", "vicejant", "vicelens",
"vicenciens", "vicentadens", "vicent", "vicentens",
"vicentinens", "viciant", "vidalens", "videirens",
"vident", "vidigalens", "vidigueirens", "vidinhens",
"vieguens", "vieirens", "vienens", "vietcongu",
"vietnamens", "vigariens", "vigent", "vigiens",
"vigilant", "vilacondens", "vilafranquens", "vilanovens",
"vilhenens", "vimaranens", "vimeirens", "vimieirens",
"vimiosens", "vinciens", "vinculant", "vindobonens",
"vindonissens", "vinhaens", "vinhalens", "vinhaticens",
"vinhedens", "violetens", "viracionens", "viradourens",
"virent", "virgiliens", "virginens", "virginiens",
"virgolandens", "virmondens", "virodunens", "virunens",
"viscens", "viseens", "viseuens", "visiens",
"visitant", "vistant", "vitelens", "vitodurens",
"vitorianens", "vitoriens", "vitorinens", "vitudurens",
"vivac", "vivariens", "vivent", "vivificant",
"vizelens", "voant", "vociferant", "volant",
"volateant", "volcens", "volitant", "volsiniens",
"voltens", "voltiniens", "vorac", "voraens",
"vordens", "votant", "votorantinens", "votourens",
"votuporanguens", "voturunens", "votuveravens", "vouzelens",
"vozeant", "vulgiens", "vurtemberguens", "wagneriens",
"xambioaens", "xambioazens", "xambioens", "xambreens",
"xambrens", "xangaiens", "xanxerens", "xapuriens",
"xavant", "xavantens", "xavantinens", "xavierens",
"xaxiens", "xaxinens", "xinguarens", "xinguens",
"xiquexiquens", "zacariens", "zairens", "zambiens",
"zamens", "zanatens", "zandavalens", "zargunchant",
"zelot", "zeolit", "zigurat", "zimbabuens",
"zot", "zuriquens"])


# PALAVRAS MASCULINAS TERMINADAS EM  E == abacate - chocolate
plvrMasce = set(["abacat", "abat", "acident", "a"+u"\u00E7"+"ougu",
"a"+u"\u00E7"+"ud", "afluent", "alambiqu", "alard",
"alarm", "albergu", "alfaiat", "alfinet",
"alicat", "alicerc", "almanaqu", "almirant",
"alp", "altifalant", "ambient", "andaim",
"anticiclon", "ant"+u"\u00ED"+"lop", "apart", "ap"+u"\u00EA"+"ndic",
"apetit", "aram", "arrabald", "arrebit",
"arrecif", "arremat", "artr"+u"\u00F3"+"pod", "atabaqu",
"ata"+u"\u00FA"+"d", "aug", "azedum", "azeit",
"azimut", "bacamart", "badulaqu", "bagarot",
"bagr", "bail", "balancet", "bala"+u"\u00FA"+"str",
"bald", "baldram", "baleot", "balot",
"baluart", "banquet", "baqu", "barbant",
"barit", "baronet", "barret", "barrot",
"bastonet", "bataqu", "batent", "batuqu",
"bazulaqu", "belich", "bequ", "berloqu",
"berrant", "besant", "betum", "bib",
"bicam", "bich", "bicorn", "bident",
"bif", "bigod", "bilhet", "b"+u"\u00ED"+"ped",
"biscat", "bisont", "bispot", "bitat",
"bit", "bivaqu", "blagu", "blecaut",
"bobech", "bocet", "bod", "bof",
"bofet", "boicot", "boiot", "bo\u00EEt",
"bond", "bonifrat", "bosqu", "bosquet",
"bot", "botoqu", "box", "bracelet",
"brand", "breloqu", "brequ", "bricabraqu",
"brilharet", "brioch", "bris", "broch",
"bronz", "brulot", "bucr", "bufet",
"bugr", "bul", "buqu", "byt",
"cabid", "cacet", "cachalot", "caciqu",
"cacoet", "cad", "cafajest", "caid",
"caingangu", "caixot", "calafat", "calhambequ",
"c"+u"\u00E1"+"lic", "calot", "camalot", "camarot",
"candomb", "canivet", "c"+u"\u00E2"+"non", "capacet",
"capot", "caracter", "carbon", "cardum",
"carm", "carpet", "cart", "casebr",
"cassetet", "caul", "cavalet", "certam",
"chal", "chantr", "chequ", "chern",
"cheviot", "chicot", "chifr", "chisp",
"chist", "chocolat", "chop", "choqu",
"chut", "ciceron", "ciclon", "ciclop",
"cineclub", "cisn", "ci"+u"\u00FA"+"m", "clarinet",
"clavinot", "clip", "cliqu", "clon",
"club", "coch", "cocot", "c"+u"\u00F3"+"dic",
"cofr", "cognom", "coic", "coiot",
"colchet", "coldr", "colet", "coliform",
"combat", "compadr", "conclav", "cond",
"con", "confet", "confrad", "conhaqu",
"c"+u"\u00F4"+"njug", "contrachequ", "contrast", "control",
"convit", "coqu", "corpet",
"corticoesteroid", "corticosteroid", "cortum", "costum",
"cotil"+u"\u00E9"+"don", "couc", "couv", "covilhet",
"coxot", "crem", "crep", "creton",
"crim", "croqu", "croquet", "cual",
"cum", "cuneiform", "cunhet", "curtum",
"debat", "debiqu", "deboch", "decalqu",
"decot", "d"+u"\u00E9"+"fic", "deleit", "dent",
"dervix", "desair", "desastr", "deuteromicet",
"diabret", "diamant", "dichot", "dicinodont",
"dinamit", "diplont", "diqu", "disparat",
"disquet", "ditam", "dog", "dogu",
"dolomit", "drinqu", "driv", "duend",
"duqu", "eclips", "ef"+u"\u00ED"+"gi", "eland",
"elefant", "eletrochoqu", "embat", "enclav",
"enfart", "enfeit", "enlac", "entalh",
"ent", "envelop", "enxam", "escadot",
"escalop", "escalp", "escaparat", "escarlat",
"escroqu", "esmalt", "espadart", "esp"+u"\u00E9"+"cim",
"espermatozoid", "espinafr", "espongiform", "esport",
"esquif", "estam", "estandart", "estand",
"estep", "esteroid", "estilet", "estip",
"est"+u"\u00ED"+"pit", "estress", "estrum", "estuqu",
"eunect", "exam", "factoid", "fad",
"fagot", "falang", "falset", "fantoch",
"fanzin", "faroest", "farolet", "fat",
"fedelhot", "feix", "ferret", "fetich",
"fiacr", "fiambr", "fidalgot", "filet",
"filhot", "filiform", "filip", "film",
"filmet", "fiord", "fland", "flandr",
"flet", "floret", "fogon", "foguet",
"fol", "fon", "frad", "franchinot",
"frand", "frangot", "franquisqu", "fraqu",
"freg", "freir", "frem", "frescalhot",
"fret", "front", "fust", "fut",
"gabinet", "galeot", "galhardet", "galop",
"garrot", "gendarm", "gen", "geod",
"gerifalt", "germ", "gibanet", "gigabyt",
"ginet", "glid", "gneiss", "god",
"godem", "gol", "golp", "grafit",
"gramofon", "gravam", "grud", "grumet",
"guach", "guant", "gud", "gued",
"gum", "halter", "haplobiont", "hectar",
"heteroterm", "heterotransplant", "hidrant", "hierofant",
"holofot", "horizont", "host", "iat",
"icebergu", "ien", "ilhot", "infart",
"ingr", "inguch", "inham", "interclub",
"interfon", "invit", "iogurt", "ipr",
"jab", "jardinet", "jarret", "jegu",
"jip", "joanet", "joguet", "joul",
"jul", "labrost", "la"+u"\u00E7"+"arot", "lacet",
"lacr", "lanch", "larot", "laud",
"lazaron", "legum", "leit", "lembret",
"l"+u"\u00EA"+"mur", "lequ", "liam", "licorn",
"liern", "limonet", "linc", "linet",
"lingot", "lipot", "livret", "lord",
"lot", "lovelac", "lugr", "lum",
"lustr", "macadam", "mac", "macet",
"machet", "magaref", "magazin", "magot",
"mainfram", "malamut", "malot", "mamut",
"mangot", "mantelet", "mapuch", "margrav",
"mascat", "massacr", "massam",
"mastodont", "matalot", "mavort", "maxil"+u"\u00ED"+"ped",
"maxix", "megabyt", "meiot", "melcatref",
"membranofon", "mendengu", "meninot", "mentastr",
"mequetref", "mestr", "microfilm", "microfon",
"micronutrient", "milagr", "mil", "milhafr",
"minaret", "minuet", "mirant", "miron",
"mixomicet", "molecot", "molequ", "monandengu",
"mong", "monovolum", "moringu", "morrot",
"mosquet", "mot", "motet", "mouch",
"mous", "mouss", "mucequ", "mujiqu",
"mussequ", "naip", "nair", "negrum",
"nepent", "nom", "nort", "novilhot",
"num", "nun", "odr", "ogr",
"optimat", "orat", "origon", "pacot",
"padr", "pagod", "palacet", "palanqu",
"palet", "palpit", "palud", "palustr",
"papelot", "parch", "par"+u"\u00EA"+"ntes", "parqu",
"passaman", "passaport", "pechisbequ", "peix",
"pelam", "pelot", "pen", "pent",
"pep", "pequenot", "perendengu", "perfum",
"pern", "pernoit", "pesadum", "p"+u"\u00EA"+"sam",
"petimetr", "petit", "pexot", "picl",
"pilot", "pinot", "pintalegr", "pintalegret",
"piparot", "pipelin", "pipot", "piqueniqu",
"piquet", "pistolet", "pivet", "ponch",
"pont"+u"\u00ED"+"fic", "porr", "porret", "post",
"pot", "potich", "prenom", "pr"+u"\u00ED"+"ncip",
"pronom", "pseud"+u"\u00F3"+"pod", "puf", "quacr",
"queixum", "quep", "quequ", "quib",
"quilat", "quilobit", "quiosqu", "quir",
"qu"+u"\u00ED"+"ri", "quirit", "quitut", "quixot",
"quocient", "rabanet", "radiotelefon", "raid",
"ralhet", "ramalhet", "ramilhet", "ran",
"rapazot", "rebenqu", "rebit", "reboqu",
"rebot", "recalqu", "recif", "reclam",
"record", "recort", "redingot", "regim",
"remoqu", "repeniqu", "repiquet", "reproch",
"requif", "requifif", "retoqu", "retrofoguet",
"reviret", "ricochet", "rifl", "ringu",
"riz"+u"\u00F3"+"pod", "rob", "robl", "rodet",
"rolet", "romanc", "romancet", "roqu",
"roquet", "rosbif", "roug", "sabonet",
"sabr", "sainet", "saiot", "salamalequ",
"salmonet", "salmoniform", "sandu"+u"\u00ED"+"ch", "sangu",
"saqu", "sax", "saxofon",
"semblant", "semestr", "serrot", "sil",
"silicon", "sinet", "sit", "skat",
"slid", "sobrenom", "soquet", "sorvet",
"sotaqu", "sub"+u"\u00ED"+"ndic", "tabef", "tablet",
"tacap", "tal", "talh", "talip",
"talvegu", "tamboret", "tanqu", "tapet",
"tapum", "tarambot", "tatam", "teatrofon",
"telefilm", "telefon", "tentam", "test",
"timbal", "timbr", "tim", "tiqu",
"t"+u"\u00ED"+"quet", "tiranet", "toalet", "toalhet",
"toilet", "told", "tolet", "tomat",
"topet", "torniquet", "t"+u"\u00F3"+"tem", "tr"+u"\u00E2"+"mit",
"transport", "traqu", "traquet", "trimestr",
"trombon", "trompet", "truqu", "trust",
"tul", "ubr", "u"+u"\u00ED"+"squ", "upanixad",
"vagonet", "valet", "vasilham", "velam",
"velhot", "veloc"+u"\u00ED"+"ped", "ventr", "verbet",
"verdet", "verm", "vernissag", "versalet",
"vexam", "videocasset", "videoclip", "videofon",
"videogam", "vilancet", "vim", "vinagr",
"vindim", "vinhet", "virot", "viscond",
"voltaret", "volum", "vombat", "v"+u"\u00F3"+"rtic",
"xail", "xal", "xarop", "xenotransplant",
"xerif", "xilofon", "xiquexiqu", "zagalot",
"ziguezagu"])



# PALAVRAS FEMININAS TERMINADAS EM  E == alface - comadre
plvrFeme = set(["aeronav", "aguardent", "alfac", "altitud",
"amiloidos", "amizad", "amplitud", "an"+u"\u00E1"+"lis",
"ant"+u"\u00ED"+"tes", "ap"+u"\u00F3"+"lic", "ap"+u"\u00F3"+"strof", "apoteos",
"art", "artrit", "artros", u"\u00E1"+"rvor",
"atitud", "autoclav", "avalanch", "av",
"bacant", "bacilos", "bacinet", "bacoric",
"bacterios", "barb"+u"\u00E1"+"ri", "bas", "basidiomicet",
"beatic", "bebedic", "b"+u"\u00E9"+"rber", "bernardic",
"bipatent", "bipir"+u"\u00E2"+"mid", "bisbilhotic", "bizarric",
"boat", "b"+u"\u00F3"+"lid", "borboletic", "boutiqu",
"brejeiric", "bronquit", "bront", "burric",
"bursit", "butiqu", "cabin", "calech",
"caminhonet", "camionet", "cari"+u"\u00E1"+"tid", "c"+u"\u00E1"+"ri",
"carn", "carolic", "casset", "cat"+u"\u00E1"+"strof",
"caturric", "celulas", "celulos", "chanc",
"charg", "charret", "chatic", "chav",
"chican", "chochic", "c"+u"\u00ED"+"clad", "cl"+u"\u00E2"+"mid",
"claqu", "class", "clav", "comadr",
"commoditi", "coort", "coqueluch", "c"+u"\u00F4"+"rt",
"cortesanic", "crech", "crendic", "cretinic",
"criancic", "cris", "c"+u"\u00FA"+"spid", "danaid",
"deb"+u"\u00EA"+"ntur", "d"+u"\u00E9"+"march", "dendrit", "dengu",
"denguic", "dermatit", "dermatos", "diaclas",
"di"+u"\u00E1"+"lis", "di"+u"\u00E1"+"stol", "di"+u"\u00E1"+"tes", "diatrib",
"dies", "dioces", "doidic", "dos",
"doutoric", "dr"+u"\u00ED"+"ad", "echarp", "efem"+u"\u00E9"+"rid",
"elips", "elit", "enchent", "enfiteus",
"enquet", "enqu"+u"\u00EA"+"t", "epiderm", "ep"+u"\u00ED"+"fis",
"ep"+u"\u00ED"+"graf", "epistax", "equimos", "equinoderm",
"equip", "esfing", "esofagit", "espa"+u"\u00E7"+"onav",
"esp"+u"\u00E1"+"dic", "esp"+u"\u00E9"+"ci", "esquisitic", "estalactit",
"estalagmit", "estant", "estenos", "estirp",
"estouvic", "estrangeiric", "estrig", "estrof",
"estroinic", "estultic", "eum"+u"\u00EA"+"nid", "exeges",
"exopeptidas", "fac", "faceiric", "fanfarronic",
"fanfreluch", "farfalhic", "fas", "fauc",
"febr", "fidalguet", "filel", "finitud",
"foic", "fom", "fosfatas", "fouc",
"francesic", "fras", "fraud", "frent",
"frol", "frond", "furunculos", "gabardin",
"gabolic", "gad", "gaf", "gaiatic",
"galanic", "galeguic", "gangu", "garag",
"gar"+u"\u00E7"+"onet", "garotic", "garridic", "garrulic",
"gastrit", "gastroenterit", "gatunic", "g"+u"\u00EA"+"nes",
"gengivit", "gent", "gland", "glicogenos",
"glomerulonefrit", "glot", "gnaiss", "g"+u"\u00F3"+"rgon",
"grad", "grev", "grif", "grip",
"gulodic", "gulosic", "habitud", "hast",
"hecatomb", "h"+u"\u00E9"+"lic", "hemap"+u"\u00F3"+"fis", "hematos",
"hemodi"+u"\u00E1"+"lis", "hemoptis", "hepatit", "hexos",
"hidrocel", "hidrolas", "hidr"+u"\u00F3"+"lis", "hip"+u"\u00E9"+"rbol",
"hip"+u"\u00F3"+"tes", "hipotipos", "idiotic", "imposturic",
"imundic", "imund"+u"\u00ED"+"ci", "incompletud", "infinitud",
"inimizad", "inquietud", "intemp"+u"\u00E9"+"ri", "interfac",
"interfas", "intrujic", "invencionic", "jambacoss",
"janotic", "jogralic", "jubart", "judit",
"juventud", "ladro"+u"\u00ED"+"c", "lag", "laj",
"lambaric", "lamechic", "lanchonet", "land",
"l"+u"\u00E1"+"pid", "laring", "lassitud", "latitud",
"lebr", "leishmanios", "lent", "leptomital",
"limousin", "limusin", "linfatit", "lingeri",
"lipas", "literatic", "lit"+u"\u00F3"+"clas", "longitud",
"louvaminhic", "macaquic", "madam", "madracic",
"madr", "magnitud", "maiones", "majestad",
"majoret", "malandret", "malandric", "malaquit",
"malcriadic", "maluquet", "maluquic", "manchet",
"mandalet", "mandaret", "manhosic", "mapl",
"maquet", "maquiavelic", "maquillag", "marinet",
"marionet", "mariquic", "maritorn", "marotic",
"marquis", "martinet", "mascot", "mastit", "matrafic",
"megal"+u"\u00F3"+"pol", "meiguic", "m"+u"\u00E9"+"nad", "m"+u"\u00EA"+"nad",
"mening", "meningit", "meninic", "merc",
"mesmic", "mes"+u"\u00F3"+"clis", "metad", "metamorfos",
"met"+u"\u00E1"+"stas", "metempsicos", "m"+u"\u00E9"+"top", "metr"+u"\u00F3"+"pol",
"mexeriquic", "micos", "miniss"+u"\u00E9"+"ri", "mir"+u"\u00ED"+"ad",
"miten", "modernic", "molenguic", "momic",
"mort", "multitud", "muss", "n"+u"\u00E1"+"iad",
"nav", "necr"+u"\u00F3"+"pol", "necros", "nefelibatic",
"nereid", "neuros", "nev", "nevros",
"niquic", "noit", "nout", "nuanc",
"nucleas", "nucleoss"+u"\u00ED"+"ntes", "oceanit", "olimp"+u"\u00ED"+"ad",
"onzenic", "ordinaric", "or"+u"\u00E9"+"ad", "osteoartros",
"overdos", "palermic", "pan", "pantomimic",
"par"+u"\u00E1"+"fras", "pared", "parlapatic", "part",
"parvo"+u"\u00ED"+"c", "patent", "patetic", "penc",
"pen"+u"\u00E9"+"lop", "pentos", "peptidas", "peraltic",
"performanc", "per"+u"\u00ED"+"fras", "perliquitet", "perric",
"pevid", "picap", "pieguic", "pios",
"pir"+u"\u00E2"+"mid", "pirit", "plan"+u"\u00ED"+"ci", "pleb",
"pl"+u"\u00EA"+"iad", "ple"+u"\u00F3"+"pod", "pol", "politiquic",
"pont", "popelin", "potestad", "prax",
"prec", "progl"+u"\u00F3"+"tid", "prol", "proteas",
"pr"+u"\u00F3"+"tes", "psicos", "quermess", "quietud",
"quimas", "quitinet", "raqu", "raquet",
"red", "repris", "retret", "revanch",
"rosal", "sabujic", "sandic", "sa"+u"\u00FA"+"d",
"sed", "seg", "sement", "s"+u"\u00E9"+"ri",
"serp", "serpent", "s"+u"\u00ED"+"lfid", "simbios",
"similitud", "s"+u"\u00ED"+"mplic", "sinaps", "s"+u"\u00ED"+"ncop",
"s"+u"\u00ED"+"ndrom", "sinops", "sintax", "s"+u"\u00ED"+"ntes",
"siren", "sirt", "sobrevest", "socialit",
"solicitud", "sort", "soviet", "sovinic",
"subclass", "subenfiteus", "subesp"+u"\u00E9"+"ci", "subpart",
"su"+u"\u00ED"+"t", "superclass", "superf"+u"\u00ED"+"ci", "sutach",
"tab", "tagarelic", "talmud", "talud",
"tard", "tart", "tempestad", "tendinit",
"tes", "teslinic", "t"+u"\u00E9"+"trad", "tirit",
"toilett", "tolic", "tontic", "torpitud",
"toss", "trafulhic", "trampolinic", "transaminas",
"trapalhic", "traquinic", "tratantic", "tremeliqu",
"trios", "tripartit", "trombos", "truanic",
"trud", "trup", "tub", "tuberal",
"tuberculos", "turbo"+u"\u00E9"+"lic", "turpitud", "urb",
"valis", "vedet", "velhic", "v"+u"\u00E9"+"rtic",
"vicissitud", "vigaric", "viros",
"virtud", "vitrin", "voltair", "vontad",
"zoonos"])

# PALAVRAS MASCULINAS TERMINADAS EM  Z == CARTAZ
plvrMascz = set(["albatro", "alborno", "alcatra", "alcatru", "alfo",
"algero", "algo", "almofari", "andalu", "antra",
"arcabu", "arca", "arroa", "bissetri", "caba",
"capata", "capu", "carta", "chafari", "chamari",
"entreme", "gilva", "gora", "jae", "ja",
"machaca", "mati", "ma", "nari", "pratarra",
"rapa", "reve", "ri", "verni", "xadre"])



# PALAVRAS FEMININAS TERMINADAS EM  Z == cruz - gravidez
plvrFemz = set(["atri", "avestru", "cicatri", "codorni", "cru",
"diretri", "eletromotri", "gravide", "imperatri", "languide",
"lu", "matri", "meretri", "motri", "no",
"nude", "nutri", "operatri", "pa", "pequene",
"perdi", "polide", "rectri", "sordide", "tamari",
"tectri", "timide", "ve", "vo"])



# PALAVRAS DE DOIS GENEROS TERMINADAS EM  Z == CAPAZ
plvrInvaz = set(["alfara", "aprendi", "atro", "auda", "bre",
"camba", "capa", "contuma", "efica", "fala",
"famana", "feli", "fero", "fuga", "incapa",
"inefica", "infeli", "lamba", "loqua", "menda",
"mina", "morda", "ori", "perspica", "pertina",
"peti", "prima", "pugna", "roa", "saga",
"sequa", "soe", "suspica", "tena", "torqua",
"velo", "vera", "viva", "vora"])





# EXCECAO PARA PALAVRAS TERMINADAS EM  inho == Carlinhos - Marquinhos
excInho = set(["Carl", "carl", "Marqu", "marqu"])

# EXCECAO PARA PALAVRAS MASCULINAS TERMINADAS  o == MESMO
excMist = set(["Mesm", "mesm"])

# EXCECAO AO - PARA- REPETIDO
excPara= set(["lhe", "lhes", "me", "nos", "se", "si", "te", "vos"])

# PALAVRAS QUE ATRAEM A PROCLISE
aProclise = set(["abaixo", "acaso", "acima", "acinte", "acol"+u"\u00E1", "adentro", "adiante",
"afinal", "afora", "agora", "a"+u"\u00ED", "al"+u"\u00E9"+"m", "algo",
"algu"+u"\u00E9"+"m", "algum", "ali", "amanh"+u"\u00E3", "ami"+u"\u00FA"+"de",
"antes", "aquela", "aquelas", "aquele", "aqueles", "aqu"+u"\u00E9"+"m", "aqui",
"aquilo", "assaz", "assim", "atr"+u"\u00E1"+"s", "bastante", "bem", "breve",
"c"+u"\u00E1", "cedo", "certa", "certo", "como", "conforme", "debalde", "decerto",
"demais", "demasiado", "dentro", "depois", "depressa", "devagar",
"diversas", "diversos", "embaixo", "enfim", "essa", "essas", "esse",
"esses", "esta", "estas", "este", "estes", "hoje", "imenso", "isso",
"isto", "l"+u"\u00E1", "longe", "mais", "mal", "melhor", "menos", "muita",
"muitas", "muito", "muitos", "ningu"+u"\u00E9"+"m", "ontem", "ora", "ou", "outra", "outras",
"outro", "outrora", "outros",
"perto", "pior", "porque", "porventura", "pouca", "poucas", "pouco", "pouco",
"poucos", "quando", "quanto", "qu"+u"\u00E3"+"o", "quase", "quer", "qui"+u"\u00E7"+u"\u00E1", "sempre",
"sim", "s"+u"\u00F3", "talvez", "tampouco", "tanto", "tarde", "tem", "t"+u"\u00EA"+"m", "temos",
"tenha", "tenham",  "tenho", "teria", "teremos", "teriam", "teve", "tinha", "tinham", "tiveram", "toda",
"todas", "todo", "todos", "tudo", "v"+u"\u00E1"+"rias", "v"+u"\u00E1"+"rios"])

# ATRAI PROCLISE
mProclise = {"abaixo": "Abaixo",
"acaso": "Acaso",
"acima": "Acima",
"acinte": "Acinte",
"acol"+u"\u00E1": "Acol"+u"\u00E1",
"adentro": "Adentro",
"adiante":"Adiante",
"afinal": "Afinal",
"afora": "Afora",
"agora": "Agora",
"a"+u"\u00ED": "A"+u"\u00ED",
"al"+u"\u00E9"+"m": "Al"+u"\u00E9"+"m",
"algo": "Algo",
"algu"+u"\u00E9"+"m": "Algu"+u"\u00E9"+"m",
"algum": "Algum",
"ali": "Ali",
"amanh"+u"\u00E3": "Amanh"+u"\u00E3",
"ami"+u"\u00FA"+"de": "Ami"+u"\u00FA"+"de",
"antes": "Antes",
"aquela": "Aquela",
"aquelas": "Aquelas",
"aquele": "Aquele",
"aqueles": "Aqueles",
"aqu"+u"\u00E9"+"m": "Aqu"+u"\u00E9"+"m",
"aqui": "Aqui",
"aquilo": "Aquilo",
"assaz": "Assaz",
"assim": "Assim",
"atr"+u"\u00E1"+"s": "Atr"+u"\u00E1"+"s",
"bastante": "Bastante",
"bem": "Bem",
"breve": "Breve",
"c"+u"\u00E1": "C"+u"\u00E1",
"cedo": "Cedo",
"certa": "Certa",
"certo": "Certo",
"como": "Como",
"conforme": "Conforme",
"debalde": "Debalde",
"decerto": "Decerto",
"demais": "Demais",
"demasiado": "Demasiado",
"dentro": "Dentro",
"depois": "Depois",
"depressa": "Depressa",
"devagar": "Devagar",
"diversas": "Diversas",
"diversos": "Diversos",
"embaixo": "Embaixo",
"enfim": "Enfim",
"essa": "Essa",
"essas": "Essas",
"esse": "Esse",
"esses": "Esses",
"esta": "Esta",
"estas": "Estas",
"este": "Este",
"estes": "Estes",
"hoje": "Hoje",
"imenso": "Imenso",
"isso": "Isso",
"isto": "Isto",
"l"+u"\u00E1": "L"+u"\u00E1",
"longe": "Longe",
"mais": "Mais",
"mal": "Mal",
"melhor": "Melhor",
"menos": "Menos",
"muita": "Muita",
"muitas": "Muitas",
"muito": "Muito",
"muitos": "Muitos",
"ningu"+u"\u00E9"+"m": "Ningu"+u"\u00E9"+"m",
"ontem": "Ontem",
"ora": "Ora",
"ou": "Ou",
"outra": "Outra",
"outras": "Outras",
"outro": "Outro",
"outrora": "Outrora",
"outros": "Outros",
"perto": "Perto",
"pior": "Pior",
"porque": "Porque",
"porventura": "Porventura",
"pouca": "Pouca",
"poucas": "Poucas",
"pouco": "Pouco",
"pouco": "Pouco",
"poucos": "Poucos",
"quando": "Quando",
"quanto": "Quanto",
"qu"+u"\u00E3"+"o": "Qu"+u"\u00E3"+"o",
"quase": "Quase",
"quer": "Quer",
"qui"+u"\u00E7"+u"\u00E1": "Qui"+u"\u00E7"+u"\u00E1",
"sempre": "Sempre",
"sim": "Sim",
"s"+u"\u00F3": "S"+u"\u00F3",
"talvez": "Talvez",
"tampouco": "Tampouco",
"tanto": "Tanto",
"tarde": "Tarde",
"tem": "Tem",
"t"+u"\u00EA"+"m": "T"+u"\u00EA"+"m",
"temos": "Temos",
"tenha": "Tenha",
"tenham": "Tenham",
"tenho": "Tenho",
"teria": "Teria",
"teremos": "Teremos",
"teriam": "Teriam",
"teve": "Teve",
"tinha": "Tinha",
"tinham":"Tinham",
"tiveram": "Tiveram",
"toda": "Toda",
"todas": "Todas",
"todo": "Todo",
"todos": "Todos",
"tudo": "Tudo",
"v"+u"\u00E1"+"rias": "V"+u"\u00E1"+"rias",
"v"+u"\u00E1"+"rios": "V"+u"\u00E1"+"rios"}


# VERBOS REGULARES TERMINADOS EM AR - RADICAL
plvrVbrar = set(["abaf", "abaix", "abal", "abaliz", "abalo", "aban",
"abarc", "abarrot", "abasm", "abast", "abelg", "abelh",
"abelhud", "aben"+u"\u00E7"+"o", "abest", "abgreg", "abilol", "abjudic",
"abjug", "abjurg", "ableg", "abneg", "aboc", "abodeg",
"abomin", "abon", "aborbot", "abort", "aboug", "abra"+u"\u00E7",
"abrand", "abrasileir", "abrevi", "abrig", "abrilhant", "abund",
"abundanci", "abus", "acab", "acalent", "acalm", "acam",
"acamp", "acanel", "acarici", "acarinh", "acarret", "acartol",
"acasal", "acastel", "acat", "acaval", "acebol", "aceit",
"acen", "acentu", "acerc", "acert", "acess", "acetin",
"ach", "achancel", "achat", "acheg", "achocalh", "achocolat",
"acident", "acidific", "aciment", "acinzent", "acion", "acirolog",
"acirr", "aclam", "aclar", "aclimat", "aclimatiz", "aco",
"acobert", "acocor", "acoit", "a"+u"\u00E7"+"oit", "acomod", "acompanh",
"aconch", "aconchav", "aconcheg", "acondicion", "acondiment", "aconselh",
"aconsonant", "acopl", "acorcov", "acorcund", "acord", "acorrent",
"acortin", "acoss", "acost", "acostum", "acotovel", "acovard",
"acredit", "acrescent", "acrian"+u"\u00E7", "acristan", "acu", "a"+u"\u00E7"+"ucar",
"acumul", "acunh", "acupuntur", "acur", "acus", "acutel",
"adapt", "adarg", "adeg", "adelg", "adeng", "adens",
"adentr", "aderg", "adi", "adiant", "adicion", "adinheir",
"adit", "adivinh", "adjetiv", "adjetiviz", "adjudic", "adjunt",
"adjur", "adjutor", "adjuv", "adlig", "administr", "admir",
"admoest", "admonest", "adnomin", "adnot", "adnumer", "ado"+u"\u00E7",
"adocic", "adoent", "ador", "adot", "adreg", "adub",
"adul", "adulter", "adun", "adverbi", "advers", "advincul",
"advoc", "advog", "advult", "aerific", "aeriz", "aerofotograf",
"aeromodel", "aeroplan", "aeroport", "aerossoliz", "aerossond", "aerotransport",
"afag", "afam", "afan", "afarinh", "afast", "afei"+u"\u00E7"+"o",
"afeijo", "afemin", "afet", "afezo", "afi", "afian"+u"\u00E7",
"afidalg", "afili", "afin", "afinc", "afirm", "afivel",
"afix", "aflameng", "afob", "afof", "afog", "afrescalh",
"afront", "afroux", "afugent", "afund", "afunil", "ag",
"agach", "agaleg", "agarr", "agasalh", "agenci", "agend",
"agigant", "agiliz", "agit", "aglomer", "aglutin", "agoniz",
"agrav", "agreg", "agrup", "aguard", "agu"+u"\u00E7", "aguent",
"agulh", "ajeit", "ajindung", "ajoelh", "ajud", "ajudeng",
"ajunt", "alag", "alambr", "alaranj", "alarg", "alastr",
"alaud", "alavanc", "alberg", "al"+u"\u00E7", "alcan"+u"\u00E7", "alcooliz",
"alcovit", "alcunh", "aldeag", "alegr", "aleij", "aleit",
"alent", "alert", "aletarg", "alevant", "alfabetiz", "alfandeg",
"algem", "alguerg", "ali", "alian"+u"\u00E7", "alicer"+u"\u00E7", "alici",
"alien", "alij", "aliment", "alinh", "alinhav", "alis",
"alist", "alivi", "almeceg", "almej", "almo"+u"\u00E7", "almofad",
"almondeg", "aloc", "aloj", "along", "alter", "altern",
"altiv", "alu", "alucin", "alug", "alum", "alumbr",
"alumi", "aluvi", "alvej", "alverg", "alvor", "alvorat",
"alvorej", "alvori"+u"\u00E7", "alvoro"+u"\u00E7", "alvorot", "alvot", "am",
"amaci", "amag", "amaldi"+u"\u00E7"+"o", "amans", "amanteig", "amarel",
"amarr", "amarrot", "amartel", "amasi", "amass", "amatung",
"amatut", "ambicion", "ambient", "amea"+u"\u00E7", "amealh", "amedront",
"amenin", "amig", "amim", "amineir", "amofin", "amoit",
"amol", "amold", "amoldur", "amolec", "amoleg", "amolent",
"amolest", "amolg", "amonto", "amoquec", "amoraliz", "amorda"+u"\u00E7",
"amoren", "amorfanh", "amorfiz", "amorisc", "amorn", "amortiz",
"amoss", "amostr", "amotin", "amour", "ampar", "ampli",
"amplific", "amput", "amu", "amulat", "amulher", "amulhereng",
"amumi", "anaboliz", "anafrag", "analgiz", "analis", "anarcis",
"anarquiz", "ancor", "and", "andaim", "andarilh", "andej",
"anedotiz", "anegr", "anegralh", "anel", "anex", "angari",
"angeliz", "angliz", "angul", "angusti", "anilh", "anim",
"animaliz", "aniversari", "anodiz", "anoit", "anoj", "anot",
"anov", "anovel", "anoxi", "ant", "antecip", "antedat",
"antepar", "antepass", "antepost", "anterioriz", "anualiz", "anul",
"anumer", "anunci", "anuvi", "apacific", "apadrinh", "apadro",
"apag", "apaixon", "apalavr", "apalm", "apalp", "apanh",
"apar", "aparament", "aparcel", "aparelh", "aparent", "aparentel",
"aparoqui", "aparreir", "apart", "aparv", "aparvalh", "aparvej",
"aparvo", "apascent", "apasco", "apassiv", "apastor", "apatron",
"apatrulh", "apaulist", "apaut", "apavor", "apavoriz", "apedr",
"apedrej", "apeg", "apej", "apel", "apelid",
"apenh", "apenhor", "apens", "apension", "apequen", "aperfei"+u"\u00E7"+"o",
"aperr", "apert", "aperu", "apesent", "apespont", "apesso",
"apincel", "apinh", "apit", "aplac", "aplain", "aplan",
"aplic", "aplum", "apoder", "apont", "apoquent", "aport",
"aportinh", "aportugues", "aposent", "aposs", "apost", "apostil",
"apostilh", "apostol", "apostrof", "apostur", "apotec", "apotent",
"apraz", "apreci", "apresent", "apresilh", "apress", "aprimor",
"aprocheg", "aprofund", "apront", "apropri", "aproveit", "aproxim",
"aprum", "aquartel", "aquebrant", "aqued", "aquiet", "ar",
"aram", "arbitr", "arboriz", "arc", "ardig", "arej",
"aren", "areng", "argol", "argument", "arisc", "arm",
"armazen", "armeng", "arquitet", "arquiv", "arraig", "arranc",
"arran"+u"\u00E7", "arranch", "arranh", "arranj", "arras", "arrast",
"arrebanh", "arrebat", "arrebent", "arrebic", "arrebit", "arrecad",
"arredr", "arrega"+u"\u00E7", "arregal", "arreganh", "arregiment", "arregr",
"arregueir", "arreig", "arremang", "arremat", "arreneg", "arreng",
"arrepi", "arrepic", "arrig", "arrisc", "arroch", "arrog",
"arrol", "arromb", "arrot", "arrox", "arru", "arrua"+u"\u00E7",
"arruf", "arrug", "arrum", "arrupi", "articul", "arvor",
"arvorej", "asfalt", "asfixi", "asneir", "aspir", "ass",
"assalari", "assalt", "assanh", "assedi", "assegur", "asseit",
"assent", "assert", "assexualiz", "assimil", "assin", "assinal",
"asso", "assoalh", "assobi", "associ", "assol", "assombr",
"assopr", "assosseg", "assunt", "assust", "atabalho", "atabern",
"atabu", "atac", "atalh", "atanaz", "ataref", "atartarug",
"atenaz", "atent", "atenu", "aterr", "aterriss", "aterriz",
"aterroriz", "atest", "ati"+u"\u00E7", "atin", "atir", "ativ",
"ato", "atoalh", "atocai", "atoch", "atoi"+u"\u00E7", "atol",
"atomiz", "aton", "atordo", "atorment", "atrac", "atrai"+u"\u00E7"+"o",
"atras", "atrel", "atribul", "atricul", "atrit", "atrofi",
"atropel", "atu", "atualiz", "atur", "aturg", "audit",
"auditor", "aug", "augur", "aument", "ausent", "auspici",
"autentic", "autentific", "autodeslig", "autograf", "autolig", "automedic",
"automotiv", "autoproclam", "autoprogram", "autopropag", "autopsi", "autoriz",
"autorrevog", "auxili", "avacalh", "avali", "avaliz", "avan"+u"\u00E7",
"avancarg", "avarg", "avari", "avassal", "aventur", "averb",
"averdug", "averdung", "avergonh", "avermelh", "avex", "avi",
"avis", "avist", "aviv", "avo", "avult", "azar",
"azed", "azeit", "azoug", "bab", "bafor", "bag",
"bagun"+u"\u00E7", "bail", "bainh", "baix", "bajog", "bajul",
"balan"+u"\u00E7", "balang", "baliz", "banc", "banh", "bard",
"barganh", "barr", "barreg", "barreir", "barric", "bast",
"batalh", "batiz", "batuc", "beberic", "beij", "beir",
"benefici", "berr", "berreg", "bestific", "betum", "bexig",
"bibliograf", "bifurc", "bimb", "bin", "bing", "biodegrad",
"biodiversific", "bioensai", "biofilosof", "biofortific", "biograf", "bisbilhot",
"bisnag", "bitol", "blef", "blog", "bocej", "bocel",
"boch", "bochacr", "bochech", "bochinch", "bod", "bodej",
"bof", "bofej", "bofet", "bog", "boi"+u"\u00E7", "boicot",
"boir", "boj", "bol", "bolach", "bolc", "bol"+u"\u00E7",
"bolcheviz", "bolet", "bolh", "bolich", "bolin", "bols",
"bomb", "bombard", "bombej", "bombi", "bombic", "bonan"+u"\u00E7",
"bon"+u"\u00E7", "bond", "bonderiz", "bonec", "bong", "bonh",
"bonific", "boquej", "boquiz", "borbolet", "borborej", "borborinh",
"borbot", "borboto", "borboton", "borbulh", "borc", "borcel",
"bord", "bordej", "borr", "borra"+u"\u00E7", "borrat", "borreg",
"borri"+u"\u00E7", "borrif", "borrisc", "bosquej", "bosquerej", "boss",
"bost", "bostej", "bot", "botec", "boto", "botoc",
"bou"+u"\u00E7", "bouch", "bour", "bovariz", "box", "brab",
"bracej", "bracteol", "brad", "bradej", "bralh", "bram",
"bramaniz", "brand", "branquej", "braquicefaliz", "bras", "brasilianiz",
"brasilific", "brasiliz", "braso", "brason", "brav", "bravej",
"brec", "brech", "breganh", "brejeir", "bretanhiz", "bretech",
"bretes", "bretess", "brevet", "brevi", "brid", "brig",
"brilh", "brilhant", "brinc", "brincalh", "brincalhot", "brind",
"brioc", "briquet", "briquit", "bris", "brit", "britaniz",
"broc", "broch", "brol", "brolh", "brom", "broncograf",
"broquel", "brosl", "bross", "brot", "brotoej", "brox",
"brum", "brumaliz", "brutaliz", "brutific", "brux", "bubui",
"bubuluc", "bu"+u"\u00E7"+"al", "bucoliz", "budo", "buf", "bufarinh",
"bugi", "bugigang", "buibil", "bul", "bulat", "bulev",
"bulh", "bumb", "bund", "bung", "bungul", "bunh",
"buquin", "burac", "burburej", "burburij", "burburinh", "burg",
"burgues", "buril", "burl", "burocratiz", "burr", "burric",
"burrific", "busc", "buscul", "bussol", "but", "butuc",
"buzeg", "buzi", "buzin", "buzul", "cabal", "cabed",
"cabl", "cabograf", "caboroc", "cabort", "cabot", "cabotin",
"cabouc", "cabraz", "cabrej", "cabriol", "cabrit", "cabroc",
"cabruc", "cabuet", "cabul", "ca"+u"\u00E7", "ca"+u"\u00E7"+"amb", "cacarej",
"cacet", "cach", "cachet", "cachimb", "cachin", "cacho",
"cachoeir", "cachuch", "cacic", "cacif", "cacimb", "ca"+u"\u00E7"+"o",
"cacofoni", "cacofoniz", "cacograf", "cacuminaliz", "cadastr", "cadaveriz",
"cadel", "cadenci", "cadinh", "cadmi", "cadmium", "caduc",
"cafang", "cafein", "cafel", "cafif", "caftin", "caftiniz",
"cafulet", "cafumb", "cafun", "cafung", "cag", "cagaliz",
"caguet", "cai", "caibr", "cain"+u"\u00E7", "cairel", "caititu",
"caixeir", "Cajam", "cal", "calabariz", "calabrot", "calaceir",
"calafet", "calafri", "calamistr", "calamoc", "calandr", "calapult",
"calc", "cal"+u"\u00E7", "calcet", "calcific", "calcin", "calcitr",
"calcograf", "calcul", "calefri", "calej", "calembur", "calemburg",
"calemburiz", "calendariz", "calh", "calhanc", "calibr", "calicul",
"caligin", "caligraf", "calm", "calorific", "caloriz", "caluni",
"calv", "calvej", "camarad", "camarinh", "camb", "cambai",
"cambalach", "cambalhot", "cambi", "cambit", "cambo", "cambr",
"cambul", "camburil", "caminh", "camoniz", "camp", "campainh",
"campan", "campanul", "campean", "campeir", "campin", "camufl",
"canaliz", "cancaniz", "cancel", "cancer", "canceriz", "candi",
"candidat", "candidiz", "candil", "candol", "candong", "canel",
"canet", "canfor", "cang", "canguerej", "canguinh", "canhon",
"canibaliz", "canoniz", "canoriz", "cans", "cant", "cantacant",
"cantarej", "cantaric", "cantarid", "cantarol", "cantorinh", "canud",
"canutilh", "caotiz", "cap", "capacit", "capang", "caparazon",
"capataz", "capeng", "capiang", "capin", "capisc", "capit",
"capitaliz", "capitul", "capoeir", "capot", "caprich", "caprific",
"capsul", "capt", "captur", "capuch", "caquerej", "caquin",
"caracol", "caracolej", "caracteriz", "carambol", "caramel", "carameliz",
"caraminhol", "caramunh", "carang", "caranguej", "caranguejol", "carantonh",
"carapet", "carateriz", "carbon", "carbonat", "carbonet", "carbonific",
"carbonil", "carbonitrogen", "carboniz", "carboret", "carboxil", "carbur",
"carburiz", "carcach", "carcav", "carcer", "card", "cardanh",
"cardiovasculariz", "cardiovasculiz", "cardu"+u"\u00E7", "caren", "carep", "carg",
"carguej", "cari", "caric", "caricatur", "carici", "carij",
"carimb", "carmesin", "carmin", "carn", "carnaliz", "carnavaliz",
"carneir", "carnific", "caroc", "carop", "carp", "carpent",
"carpet", "carpinteir", "carpintej", "carr", "carrameliz", "carrapat",
"carreg", "carrej", "carret", "carril", "carrilh", "carrilhon",
"carrinh", "carruaj", "cart", "cartabux", "carteliz", "cartilaginific",
"cartograf", "carton", "cartorializ", "caruj", "carunch", "carvej",
"carvoej", "cas", "casal", "casamat", "casament", "casbi",
"casc", "cascabulh", "cascalh", "cascat", "cascavel", "cascavilh",
"cascunh", "caseific", "casmurr", "caspach", "casquej", "casquilh",
"casquin", "cass", "cassumbul", "castanhol", "castelhaniz", "casti"+u"\u00E7",
"castific", "castig", "castr", "castramet", "castrin", "cat",
"cataboliz", "catadupej", "catalis", "cataliz", "catalog", "catalogiz",
"catan", "cataplasm", "catapult", "catecumeniz", "catedratiz", "categoriz",
"catequiz", "cateteriz", "catimb", "cating", "catit", "cativ",
"catoliciz", "catoliz", "catoniz", "catraceg", "catrafi", "catrafil",
"catrai", "catrapisc", "catuc", "caturr", "caturrit", "cauchut",
"caucion", "caudilh", "caulific", "caulin", "cauliniz", "caurin",
"caus", "causaliz", "caustic", "caustific", "cautel", "cauteri",
"cauteriz", "cav", "cavac", "cavalg", "cavalo", "cavatin",
"cavern", "cavid", "cavil", "cavilh", "cavit", "cavoc",
"cavouc", "cavuc", "cavumbul", "caxing", "cedilh", "cefaliz",
"ceg", "ceif", "ceiv", "celebr", "celebriz", "celeum",
"celific", "celticiz", "celtiz", "celulariz", "celv", "cement",
"cendr", "censoriz", "censur", "censuriz", "centelh", "centissecul",
"centoniz", "centr", "centraliz", "centrific", "centrifug", "centupl",
"centuplic", "ceot", "cepilh", "cer", "cerabel", "cerabol",
"ceramoplasm", "ceratiniz", "cerc", "cercilh", "cerebel", "cerebraliz",
"ceresit", "cerific", "cerimoni", "cerimonializ", "ceringonh", "cern",
"cerr", "cert", "certific", "cervej", "cesariz", "cespit",
"cess", "cesur", "cetoderiv", "cetoniz", "cev", "chaba"+u"\u00E7",
"chabouc", "cha"+u"\u00E7", "chacin", "chacoalh", "chacolh", "chafalh",
"chafund", "chafurd", "chag", "chair", "chal", "chala"+u"\u00E7",
"chaldr", "chaleir", "chalr", "cham", "chamalot", "chamboc",
"chambreg", "chameg", "chamej", "champ", "champanhiz", "chamusc",
"chanasc", "chanc", "chancel", "chanel", "chanfalh", "chanfr",
"chang", "changueir", "chanquelh", "chant", "chantaj", "chanto",
"chap", "chapej", "chapi"+u"\u00E7", "chapilh", "chapin", "chapinh",
"chapisc", "chapod", "chapolet", "chapot", "chapu"+u"\u00E7", "chapuj",
"chapulh", "chapuz", "chaqui"+u"\u00E7", "charabald", "charamel", "charang",
"charavisc", "charing", "charivariz", "charl", "charo", "charol",
"charond", "charrisc", "charru", "charut", "chasc", "chasquin",
"chatin", "chauviniz", "chavanisc", "chavasc", "chavet", "chec",
"chefi", "cheg", "cheir", "cheiret", "cheiric", "cheirin",
"cheld", "chelindr", "cheravasc", "chi", "chiang", "chiat",
"chib", "chibat", "chi"+u"\u00E7", "chicalh", "chican", "chich",
"chichel", "chichi", "chichil", "chichorrobi", "chicot", "chifr",
"chileniz", "chilid", "chilr", "chimbi", "chimp", "chin",
"chinc", "chincalh", "chinch", "chinel", "chinfr", "chinfrin",
"chiniz", "chiqueir", "chisc", "chisn", "chisp", "choc",
"chocalej", "chocalh", "choch", "chocolat", "chofer", "chofr",
"choin", "choit", "chon", "chor", "choramig", "choraming",
"chorinc", "chorococ", "chorr", "chorrilh", "chorrisc", "chot",
"choup", "chous", "chouss", "chout", "chu"+u"\u00E7", "chuch",
"chuchu", "chuet", "chuf", "chuma"+u"\u00E7", "chumb", "chumbreg",
"chumbut", "chup", "chupinh", "chupist", "chupit", "churd",
"churin", "churri", "chusm", "chut", "chuvenisc", "chuverisc",
"chuvilh", "chuvinh", "chuvisc", "ci", "cianet", "cianiz",
"cianogen", "cianos", "cianur", "cib", "cibernetiz", "cicatriz",
"cicer", "ciceronianiz", "ciceroniz", "cici", "cicliz", "cientific",
"cif", "cifr", "cigan", "cigarr", "cilh", "cilici",
"cilindr", "cimbr", "ciment", "cinc", "cinch", "cinchoniz",
"cincunval", "cineangiocardiograf", "cineangiocoronariograf", "cinecoronariograf", "cinefic", "cinegraf",
"cinematiz", "cinematograf", "cinemiz", "ciner", "cingul", "cint",
"cintil", "cintur", "cinz", "cinzel", "cipo", "cirand",
"circi", "circin", "circuit", "circunaveg", "circuncentr", "circuncid",
"circuncircul", "circund", "circundut", "circungir", "circunrod", "circunson",
"circunspeccion", "circunspecion", "circunst", "circunstanci", "circunstancializ", "circunstancion",
"circuntorn", "circunvag", "circunval", "circunvizinh", "circunvo", "ciriliz",
"cis", "cisalh", "cisc", "cisel", "cism", "cisp",
"cistorradiograf", "cit", "citariz", "citograf", "citol", "citolis",
"cium", "civiliz", "cizan", "cizani", "cizel", "clam",
"clangor", "clangorej", "claraboi", "clarej", "clarific", "clarimostr",
"clarin", "classiciz", "classific", "claudic", "claustr", "clausul",
"clausur", "clav", "claveliz", "clemenci", "cleopatriz", "clericaliz",
"clic", "climatiz", "clinic", "clinqueriz", "clip", "clis",
"clisteriz", "cliv", "cloc", "clon", "clor", "cloraliz",
"cloretiliz", "cloritiz", "cloriz", "cloroderiv", "clorofil", "cloroform",
"cloroformiz", "clorossulfon", "clorovaporiz", "co", "coabit", "coacerv",
"coact", "coadapt", "coadat", "coadjuv", "coadministr", "coadun",
"coagit", "coaglutin", "coagment", "coagul", "coalh", "coaliz",
"coangust", "coanh", "coapostol", "coapt", "coarct", "coarrend",
"coart", "coassin", "coassoci", "coat", "coax", "cobalt",
"cobaltiz", "cobi"+u"\u00E7", "cobr", "cobrej", "coc", "co"+u"\u00E7",
"cocainiz", "coch", "cochich", "cochil", "cochin", "cocient",
"cocontrat", "cocor", "cocori", "cocoric", "cocri", "cocul",
"codej", "codemand", "codific", "codilh", "codinh", "codiz",
"codomin", "coedit", "coeduc", "coelabor", "coerd", "coestrel",
"cofi", "cofinanci", "cofinh", "cofund", "cogi", "cogit",
"cognomin", "cogul", "cogumel", "coicinh", "coif", "coigual",
"coim", "coimper", "coin", "coinch", "coinciner", "coindic",
"coinfect", "coinici", "coinquin", "cointegr", "cointeress", "coiquinh",
"coira"+u"\u00E7", "coirman", "cois", "coisific", "coit", "coivar",
"col", "colabor", "colacion", "colapis", "colaps", "colchet",
"colcho", "colecion", "colecistograf", "colect", "coleg", "colet",
"coletiviz", "colg", "colider", "colig", "colim", "colimit",
"colitig", "colm", "colma"+u"\u00E7", "colmat", "coloc", "colodion",
"coloniz", "colopliz", "coloqui", "color", "coloriz", "colostomiz",
"colquiciniz", "colt", "coltariz", "colubrej", "colui", "colun",
"comand", "comandit", "comarc", "combin", "combor"+u"\u00E7", "combust",
"come"+u"\u00E7", "comedi", "comemor", "comend", "comensur", "coment",
"comerci", "comercializ", "comich", "comil", "comin", "coministr",
"comir", "comisc", "comiser", "comissari", "comission", "comistur",
"comocion", "comod", "comonomeriz", "comor", "compact", "compactu",
"compadr", "compagin", "companh", "compar", "comparticip", "compartilh",
"compartiment", "compartimentaliz", "compass", "compatibiliz", "compe"+u"\u00E7", "compeg",
"compendi", "compenetr", "compens", "compermut", "compil", "complain",
"complan", "complei"+u"\u00E7"+"o", "complement", "complet", "complex", "complexific",
"complic", "complici", "compliz", "complot", "comport", "compr",
"compromiss", "comprov", "compugn", "compuls", "compurg", "comput",
"computadoriz", "computatoriz", "computoriz", "comud", "comung", "comunic",
"comunitariz", "comuniz", "comut", "conap", "conaturaliz", "con"+u"\u00E7",
"concaniz", "concaten", "concav", "conceit", "conceitu", "conceitualiz",
"concelebr", "concentr", "concentraliz", "conceptualiz", "concert", "concession",
"conch", "conchav", "concheg", "concili", "concion", "concit",
"conclam", "concomit", "concord", "concrecion", "concret", "concretiz",
"concri", "concubin", "conculc", "concultur", "concurs", "concurv",
"condecor", "conden", "condens", "condess", "condici", "condicion",
"condign", "condiment", "condo", "condrific", "conduplic", "condut",
"conect", "conectoriz", "conet", "conexion", "confabul", "confeccion",
"confeder", "confei"+u"\u00E7"+"o", "confeit", "conferenci", "confess", "confi",
"conficion", "confidenci", "configur", "confin", "confirm", "confisc",
"conflagr", "conflit", "conflitu", "conform", "conformaliz", "confort",
"confratern", "confraterniz", "confric", "confront", "confusion", "confut",
"congalardo", "congel", "congemin", "congener", "congestion", "conglob",
"conglomer", "conglutin", "congonh", "congra"+u"\u00E7", "congraci", "congratul",
"congreg", "conhaquiz", "conjectur", "conjeitur", "conjetur", "conjug",
"conjumin", "conjunt", "conjur", "conjurament", "conlui", "conomin",
"conot", "conquist", "consagr", "consarcin", "consciencializ", "conscienciz",
"conscientiz", "consecr", "conselh", "conselheir", "consensualiz", "consert",
"conserv", "consider", "consign", "consignific", "conso", "consoant",
"consoci", "consogr", "consol", "consolid", "conson", "consonantiz",
"consorci", "conspir", "conspurc", "const", "constat", "constel",
"constern", "constip", "constitucionaliz", "consubstanci", "consult", "consum",
"consumeriz", "cont", "contabiliz", "contact", "contagi", "contamin",
"contat", "conteir", "contempl", "contemporiz", "content", "contentoriz",
"contest", "contextu", "contextualiz", "contingenci", "continu", "contorcion",
"contorn", "contorv", "contrabalan"+u"\u00E7", "contraband", "contracambi", "contracen",
"contracol", "contracunh", "contradan"+u"\u00E7", "contradeclar", "contradit", "contraestimul",
"contrafort", "contraindic", "contrainform", "contramalh", "contramand", "contramanifest",
"contramanobr", "contramarc", "contramarch", "contramin", "contraminut", "contramold",
"contramur", "contranivel", "contraorden", "contrapedal", "contrapes", "contrapont",
"contraprotest", "contraprov", "contrapun"+u"\u00E7"+"o", "contrari", "contrarraciocin", "contrarreform",
"contrarregr", "contrarreplic", "contrarrept", "contrarrevolt", "contrarrevolucion", "contrassalv",
"contrassegur", "contrassel", "contrast", "contrat", "contratel", "contratestemunh",
"contratir", "contratualiz", "contraval", "contravent", "contrist", "control",
"contubern", "conturb", "conumer", "conurb", "convalid", "convencion",
"convencionaliz", "conveni", "conventicul", "convers", "conversalh", "convibr",
"convid", "convizinh", "convoc", "convol", "convuls", "convulsion",
"coob", "coobrig", "coocup", "coofici", "coonest", "cooper",
"coopt", "coorden", "coorient", "coossific", "cop", "coparticip",
"copatrocin", "copeir", "copej", "copel", "copi", "copidesc",
"copil", "copilot", "copiograf", "coplan", "copod", "copolimeriz",
"coprocess", "coptiz", "copul", "coqueific", "cor", "coraliz",
"corandel", "corc", "corcov", "cordo", "coreograf", "corisc",
"corn", "cornach", "cornej", "cornific", "cornitromb", "cornud",
"coro", "corograf", "coronariograf", "corporaliz", "corporific", "corporiz",
"corpusculiz", "correlacion", "correlat", "corresponsabiliz", "corret", "correx",
"corric", "corrich", "corrip", "corrobor", "corroteiriz", "corruchi",
"corrug", "corrupi", "cort", "cortej", "cortilh", "cortin",
"coruj", "corunh", "corusc", "corvej", "coscor", "coscorej",
"coscuvilh", "cosic", "cosip", "cosmin", "cosmopolitiz", "cosmopoliz",
"cossegur", "costel", "costum", "costur", "cot", "cotanilh",
"cotej", "coti", "cotidianiz", "cotiz", "coton", "cotovel",
"cotul", "cotutel", "cou"+u"\u00E7", "coucinh", "coudel", "coura"+u"\u00E7",
"courel", "cous", "cousific", "cout", "couv", "covach",
"covend", "cozinh", "craion", "crani", "craquej", "craticul",
"crav", "cravej", "cravin", "credenci", "credibiliz", "credit",
"crel", "crem", "cremoriz", "cren", "creolin", "creosot",
"crepit", "crepusculej", "crepusculiz", "cresp", "crest", "cretific",
"cretiniz", "cri", "cricril", "crimin", "criminaliz", "crimp",
"criocauteriz", "crioconcentr", "criodessec", "criptanalis", "criptoanalis", "criptograf",
"cris", "crisalid", "crism", "crisol", "crisp", "cristaliz",
"cristianiz", "cristologiz", "critic", "critiquiz", "criv", "crocidiz",
"crocit", "crom", "cromatiz", "cromatolis", "cromiz", "cromofotograv",
"cromofotolit", "cromofototipograv", "cromofotozincograv", "cromograv", "cromolitograv", "cromotipi",
"cromotipograv", "cromoxilograf", "cromoxilograv", "cromozincograv", "cromozincolitograv", "cromozincotipograv",
"cronic", "croniquiz", "croniz", "cronogoniometr", "cronometr", "cronoprojet",
"cronotaquimetr", "cronotelemetr", "crotoniz", "cruci", "crucific", "crucifix",
"crucigi", "cruent", "crustific", "cruz", "cuanh", "cuarl",
"cub", "cubaniz", "cubic", "cubific", "cuc", "cucul",
"cucurbit", "cucuric", "cucurit", "cuf", "cugi", "cuid",
"cuil", "cuinch", "cuinh", "cuji", "culap", "culatr",
"culc", "culim", "culmin", "culp", "culpabiliz", "cultiv",
"cultu", "culturaliz", "cumplici", "cumpriment", "cumul", "cungugut",
"cunh", "cuoz", "cur", "curariz", "curatel", "curet",
"curi", "curiband", "curibot", "curios", "curr", "curs",
"cursometr", "curuc", "curv", "curvej", "curvet", "curvigraf",
"cuspilh", "cuspinh", "cust", "custodi", "customiz", "cutil",
"cutiniz", "cutis", "cutivacin", "cutuc", "dactilograf", "dadiv",
"daguerreotip", "daltoniz", "damasquin", "damej", "dan", "dan"+u"\u00E7",
"dan"+u"\u00E7"+"aric", "dan"+u"\u00E7"+"arin", "dandin", "danific", "daninh", "dard",
"dardej", "dat", "datilograf", "dealb", "dealv", "deambul",
"dearticul", "deaur", "deb", "debac", "debag", "deband",
"debang", "debel", "debentur", "debic", "debilit", "debit",
"deblater", "deboch", "deboi"+u"\u00E7", "deborc", "debord", "debot",
"debou"+u"\u00E7", "debras", "debrid", "debru", "debru"+u"\u00E7", "debulh",
"debut", "debux", "decalc", "decalv", "decamp", "decant",
"decap", "decapit", "decein", "decemplic", "decentraliz", "decep",
"decepcion", "decert", "decifr", "decilitr", "decim", "decimaliz",
"declam", "declar", "declin", "decliv", "deco", "decodific",
"decol", "decor", "decortic", "decot", "decrement", "decrepit",
"decret", "decript", "decru", "decup", "decupl", "decuplic",
"decuri", "decurs", "decuss", "ded", "dedecor", "dedetiz",
"dedic", "dedign", "dedilh", "dedur", "defas", "defec",
"defenestr", "defens", "defibril", "definh", "deflacion", "deflag",
"deflagr", "deflegm", "deflexion", "deflor", "deform", "defraud",
"defront", "defum", "defunt", "degas", "degast", "degel",
"degener", "deglobuliz", "degol", "degot", "degrad", "degran",
"degred", "degringol", "degust", "deific", "deioniz", "deit",
"deix", "dejarret", "dejeju", "dejet", "delagrimej", "delapid",
"delat", "deleg", "deleit", "delet", "delga"+u"\u00E7", "delib",
"deliber", "delici", "delig", "delimit", "deling", "delir",
"delivr", "delong", "demand", "demarc", "demasi", "dement",
"demitiz", "democratiz", "demodul", "demograf", "demolh", "demonetiz",
"demoniz", "demonstr", "demor", "demostr", "demud", "deneg",
"denerv", "denod", "denomin", "denot", "dens", "densific",
"dent", "dentel", "denticul", "denud", "denunci", "depar",
"departamentaliz", "depauper", "depen", "dependur", "depenic", "depil",
"deplor", "deplum", "depolariz", "depolm", "depopul", "deport",
"deposit", "deprav", "deprec", "depreci", "depred", "depress",
"depur", "deput", "dequit", "deriv", "derog", "derrab",
"derraig", "derram", "derranc", "derrap", "derratiz", "derreg",
"derreig", "derrelh", "derreng", "derrepast", "derrib", "derri"+u"\u00E7",
"derrisc", "derroc", "derro"+u"\u00E7", "derroch", "derrog", "derronch",
"derrot", "derrub", "dervirg", "des", "desab", "desabaf",
"desabal", "desabaliz", "desabalro", "desabanc", "desabast", "desabeir",
"desabelh", "desaben"+u"\u00E7"+"o", "desabez", "desabilit", "desabit", "desabitu",
"desaboc", "desabo"+u"\u00E7", "desabon", "desaborc", "desabord", "desabotin",
"desaboto", "desabra"+u"\u00E7", "desabragal", "desabrasileir", "desabrig", "desabroch",
"desabrolh", "desabus", "desacadel", "desa"+u"\u00E7"+"aim", "desacam", "desa"+u"\u00E7"+"am",
"desacamarad", "desacamp", "desacanalh", "desacanh", "desacaravelh", "desacasal",
"desacat", "desacaudel", "desacaudilh", "desacautel", "desacaval", "desaceit",
"desaceler", "desacentu", "desacer", "desacerb", "desacert", "desacheg",
"desachorum", "desacident", "desacidific", "desacidul", "desacism", "desaclim",
"desaclimat", "desacobard", "desacoch", "desacoim", "desacoit", "desacolchet",
"desacolcho", "desacolher", "desacomod", "desacompanh", "desacompass", "desaconcheg",
"desaconselh", "desacopl", "desacora"+u"\u00E7"+"o", "desacor"+u"\u00E7"+"o", "desacord", "desacoro"+u"\u00E7"+"o",
"desacorrent", "desacost", "desacostum", "desacoto", "desacout", "desacovard",
"desacrav", "desacravelh", "desacredit", "desacu", "desacumul", "desacunh",
"desacur", "desadapt", "desadens", "desadere"+u"\u00E7", "desadestr", "desadjetiv",
"desadministr", "desadmoest", "desador", "desadorment", "desadorn", "desaduch",
"desadul", "desadun", "desadvog", "desaer", "desafag", "desafaim",
"desafam", "desafast", "desafei"+u"\u00E7"+"o", "desafeit", "desaferr", "desaferro",
"desaferrolh", "desaferrolho", "desafervor", "desafet", "desafi", "desafigur",
"desafilh", "desafin", "desafiuz", "desafivel", "desafix", "desafog",
"desafolh", "desafor", "desaforr", "desafortun", "desafrances", "desafregues",
"desafreim", "desafront", "desafu", "desafueir", "desafum", "desafund",
"desafusc", "desagalardo", "desagalinh", "desagalo", "desagalon", "desagarr",
"desagasalh", "desagast", "desagend", "desaglomer", "desagoir", "desagoni",
"desagour", "desagrad", "desagrav", "desagreg", "desagrilho", "desagrup",
"desaguach", "desaguax", "desagu"+u"\u00E7", "desaguis", "desain", "desair",
"desajeit", "desajouj", "desajud", "desajunt", "desajust",
"desalag", "desalap", "desalarg", "desalarm", "desalastr", "desalbard",
"desalcaliz", "desalcan"+u"\u00E7", "desalcatro", "desalcoil", "desaleald", "desalegr",
"desaleit", "desalent", "desalergiz", "desalfai", "desalfandeg", "desalforj",
"desalgem", "desali", "desalicer"+u"\u00E7", "desalij", "desalinh", "desalinhav",
"desalist", "desalivi", "desalm", "desalmej", "desaloc", "desalogen",
"desaloj", "desalquil", "desalter", "desalug", "desalumbr", "desalvor",
"desam", "desamachuc", "desamain", "desamalgam", "desamament", "desamanceb",
"desamaneir", "desamanh", "desamant", "desamantilh", "desamarinh", "desamarinheir",
"desamarr", "desamarrot", "desamartel", "desamass", "desambicion", "desambient",
"desambigu", "desambiguiz", "desamealh", "desamedront", "desamid", "desamig",
"desamim", "desamin", "desaminiz", "desamist", "desamnisti", "desamodorr",
"desamoed", "desamoest", "desamofin", "desamolg", "desamont", "desamonto",
"desamortalh", "desamortiz", "desamotin", "desamouc", "desampar", "desamu",
"desamur", "desamuriz", "desanacroniz", "desanalfabetiz", "desanc", "desancor",
"desand", "desanel", "desanex", "desangeliz", "desangusti", "desanich",
"desanil", "desanim", "desanimaliz", "desanimiz", "desaninh", "desanisti",
"desanoj", "desanul", "desanuvi", "desapacient", "desapadrinh", "desapadro",
"desapag", "desapaixon", "desaparafus", "desaparelh", "desaparent", "desapart",
"desapassion", "desapavor", "desape"+u"\u00E7"+"onh", "desape"+u"\u00E7"+"onhent", "desapeg", "desapeir",
"desapen", "desapendo", "desapens", "desaperr", "desapert", "desapiad",
"desapied", "desaplic", "desapoder", "desapolvilh", "desapont", "desapoquent",
"desaportugues", "desaposent", "desaposs", "desapre"+u"\u00E7", "desapreci", "desapresilh",
"desapress", "desaprest", "desaprimor", "desaproposit", "desapropri", "desaprov",
"desaproveit", "desaprovision", "desaproxim", "desaprum", "desapur", "desaquartel",
"desaqueren"+u"\u00E7", "desaquerenci", "desaquinho", "desar", "desaranh", "desarboriz",
"desarc", "desarej", "desarest", "desargent", "desargentiz", "desarist",
"desarm", "desarmadilh", "desarmoniz", "desarom", "desaromaliz", "desaromatiz",
"desarquitet", "desarquiv", "desarraig", "desarranc", "desarranch", "desarranh",
"desarranj", "desarrazo", "desarreat", "desarrebit", "desarred", "desarredond",
"desarrega"+u"\u00E7", "desarreig", "desarremedi", "desarreneg", "desarrest", "desarreves",
"desarrig", "desarrim", "desarrinc", "desarrisc", "desarroch", "desarrolh",
"desarrot", "desarroup", "desarru", "desarruf", "desarrug", "desarrulh",
"desarrum", "desarterializ", "desarticul", "desartilh", "desarvor", "desas",
"desasn", "desaspir", "desassanh", "desassazo", "desassazon", "desassel",
"desasselvaj", "desassemelh", "desassent", "desassest", "desassimil", "desassimilh",
"desassinal", "desassis", "desassoci", "desassolv", "desassom", "desassombr",
"desassosseg", "desassunt", "desassust", "desastr", "desat", "desatabaf",
"desatac", "desatafulh", "desatapet", "desatapulh", "desatarrax", "desatasc",
"desataud", "desatavi", "desatedi", "desatemor", "desatemoriz", "desatent",
"desaterr", "desatest", "desatilh", "desatin", "desativ", "desatoch",
"desatol", "desatordo", "desatrac", "desatranc", "desatravanc", "desatravess",
"desatrel", "desatrem", "desatribul", "desatualiz", "desaug", "desautor",
"desautoriz", "desauxili", "desavacalh", "desavag", "desavantaj", "desaventur",
"desaverb", "desavergonh", "desavex", "desavez", "desavi", "desavigor",
"desavilt", "desavinc", "desavinh", "desavis", "desavist", "desaviv",
"desavolum", "desavult", "desaz", "desazinhavr", "desazot", "desbab",
"desbabeliz", "desbadal", "desbag", "desbago", "desbagulh", "desbalan"+u"\u00E7",
"desbal"+u"\u00E7", "desbalis", "desbaliz", "desbals", "desbamb", "desbanaliz",
"desbanc", "desband", "desbandalh", "desbandeir", "desbar", "desbarat",
"desbarb", "desbarbariz", "desbarr", "desbarranc", "desbarret", "desbarrig",
"desbast", "desbastard", "desbatiz", "desbatoc", "desbei"+u"\u00E7", "desbenzoil",
"desbenzol", "desbestializ", "desbic", "desbitol", "desbobin", "desboc",
"desbolin", "desbolot", "desboqueir", "desborc", "desborcel", "desborcin",
"desbord", "desboro", "desborr", "desbot", "desboto", "desbrag",
"desbrasileir", "desbrav", "desbravej", "desbrec", "desbri", "desbrid",
"desbrinc", "desbritaniz", "desbro", "desbroch", "desbrot", "desbrum",
"desbrutaliz", "desbuch", "desbulh", "desbund", "desburocratiz", "desburric",
"desburrific", "desbutaniz", "descaba"+u"\u00E7", "descabel", "desca"+u"\u00E7", "descacha"+u"\u00E7",
"descachel", "descachol", "descadastr", "descadeir", "descafein", "descafeiniz",
"descafel", "descal", "descalabr", "descalavr", "descal"+u"\u00E7", "descalcet",
"descalcific", "descalcin", "descalho", "descalibr", "descali"+u"\u00E7", "descalv",
"descam", "descamb", "descambimb", "descambulh", "descaminh", "descamis",
"descamp", "descan", "descancar", "descancel", "descanch", "descandaliz",
"descanel", "descang", "descangalh", "descangot", "descanhot", "descanjic",
"descans", "descant", "descap", "descapacit", "descapel", "descapitaliz",
"descapot", "descapsul", "descar", "descaracteriz", "descarapu"+u"\u00E7", "descarateriz",
"descarbon", "descarbonat", "descarbonet", "descarboniz", "descarboxil", "descarbur",
"descarburiz", "descarg", "descarn", "descarnaliz", "descarnardesenterr", "descarniz",
"descaro"+u"\u00E7", "descarol", "descarr", "descarreg", "descarreir", "descarret",
"descarri", "descarril", "descarrilh", "descart", "descarteliz", "descartorializ",
"descas", "descasal", "descasc", "descasp", "descasquej", "descasul",
"descatequiz", "descativ", "descatoliz", "descaud", "descautel", "descav",
"descavalg", "descaveir", "descaxel", "descefaliz", "desceg", "descelulariz",
"descement", "descentr", "descentraliz", "descep", "descerc", "descerebel",
"descerebr", "descerebriz", "descerr", "deschancel", "deschapel", "deschumb",
"descicliz", "descifr", "descimbr", "desciment", "desciminaliz", "descint",
"descintr", "desciviliz", "desclassific", "desclor", "descloret", "descloroformiz",
"descoagul", "descoalh", "descoc", "descoch", "descodific", "descofr",
"descogot", "descoif", "descoim", "descoit", "descoivar", "descol",
"descolg", "descolm", "descoloc", "descoloniz", "descolor", "descoloriz",
"descomand", "descombin", "descomission", "descompact", "descompadr", "descompagin",
"descompanh", "descomparticip", "descompartiment", "descompartimentaliz", "descompass", "descompatibiliz",
"descompens", "descompil", "descomplic", "descompromiss", "descomung", "desconceitu",
"desconcentr", "desconcert", "desconchav", "desconcheg", "desconcili", "desconcord",
"descondens", "descondicion", "desconect", "desconfeit", "desconfess", "desconfi",
"desconfigur", "desconform", "desconfort", "desconfront", "descongel", "descongestion",
"desconjug", "desconjunt", "desconjur", "desconsagr", "desconsert", "desconsider",
"desconsol", "desconstitucionaliz", "descont", "descontagi", "descontamin", "descontent",
"descontentoriz", "descontextualiz", "descontingenci", "descontinu", "descontrat", "descontrol",
"desconturb", "desconvers", "desconvid", "desconvoc", "descoorden", "descop",
"descor", "descorch", "descor"+u"\u00E7"+"o", "descord", "descorn", "descoro",
"descoro"+u"\u00E7"+"o", "descorporiz", "descorrelacion", "descorrent", "descortej", "descortic",
"descorti"+u"\u00E7", "descortin", "descoruch", "descost", "descostum", "descostur",
"descoto", "descout", "descrav", "descraveir", "descravej", "descraviz",
"descredenci", "descredibiliz", "descredit", "descrem", "descrimin", "descriminaliz",
"descriptograf", "descrist", "descristianiz", "descrom", "descruz", "descuid",
"desculatr", "desculp", "desculpabiliz", "descultiv", "descultu", "descultur",
"descunh", "descupiniz", "descur", "descurv", "descuscut", "desdeix",
"desdemoniz", "desdenh", "desdenomin", "desdent", "desdiagram", "desdiferenci",
"desdob", "desdobr", "desdoir", "desdour", "desdoutrin", "desdramatiz",
"deseclips", "deseconomiz", "desedific", "deseduc", "desefemin", "deseiv",
"deseix", "desej", "deseletriz", "deselimin", "deselitiz", "desema"+u"\u00E7",
"desemadeir", "desemal", "desemalh", "desemalhet", "desemaranh", "desematilh",
"desemba"+u"\u00E7", "desembaci", "desembainh", "desembal", "desembals", "desembalsam",
"desembandeir", "desembara"+u"\u00E7", "desembaralh", "desembarc", "desembarg", "desembarr",
"desembarranc", "desembarrig", "desembarril", "desembati", "desembaul", "desembebed",
"desembei"+u"\u00E7", "desembest", "desembezerr", "desembirr", "desemblinh", "desemboc",
"desembol", "desembols", "desemborc", "desemborr", "desemborrach", "desemborrasc",
"desembosc", "desembot", "desembra"+u"\u00E7", "desembrai", "desembram", "desembrech",
"desembrenh", "desembriag", "desembrid", "desembrion", "desembrom", "desembrulh",
"desembrum", "desembrusc", "desembrux", "desembu"+u"\u00E7", "desembuch", "desemburilh",
"desemburr", "desemburric", "desemed", "desemend", "desemetiniz", "desemo"+u"\u00E7",
"desemoglobiniz", "desemoinh", "desemoldur", "desemp", "desempac", "desempacavir",
"desempach", "desempacot", "desempadralh", "desempal", "desempalh", "desempalm",
"desempan", "desempandeir", "desempanturr", "desempap", "desempapel", "desempar",
"desemparceir", "desempared", "desemparelh", "desempast", "desempastel", "desempat",
"desempaves", "desempe"+u"\u00E7", "desempecilh", "desempe"+u"\u00E7"+"onh", "desempedr", "desempeflig",
"desempeg", "desempen", "desempenc", "desempenh", "desempern", "desemperr",
"desempertig", "desempest", "desempilh", "desempin", "desemplast", "desemplastr",
"desemplum", "desempo", "desempo"+u"\u00E7", "desempoeir", "desempol", "desempoleir",
"desempolg", "desempolh", "desemposs", "desempreg", "desemprenh", "desempro",
"desempunh", "desemulsific", "desemulsion", "desenamor", "desenastr", "desenat",
"desencab", "desencabe"+u"\u00E7", "desencabel", "desencabrest", "desencabrit", "desencabul",
"desencach", "desencacha"+u"\u00E7", "desencadern", "desencadilh", "desencafif", "desencafu",
"desencaipor", "desencaix", "desencaixilh", "desencaixot", "desencalacr", "desencalh",
"desencalm", "desencamarad", "desencamb", "desencaminh", "desencamis", "desencamp",
"desencan", "desencanalh", "desencanastr", "desencang", "desencangalh", "desencant",
"desencanto", "desencap", "desencapar", "desencapel", "desencapoeir", "desencapot",
"desencapsul", "desencaracol", "desencaralh", "desencarang", "desencarapel", "desencarapinh",
"desencarapu"+u"\u00E7", "desencarcer", "desencarg", "desencarn", "desencarquilh", "desencarr",
"desencarranc", "desencarrapit", "desencarreg", "desencarreir", "desencarret", "desencarril",
"desencarrilh", "desencart", "desencas", "desencasac", "desencasc", "desencasquet",
"desencastel", "desencasto", "desencatarr", "desencatarro", "desencav", "desencavac",
"desencavalg", "desencavern", "desencavilh", "desencep", "desencer", "desencerr",
"desencharc", "desencilh", "desenclaustr", "desenclavinh", "desencobr", "desencoc",
"desencofr", "desencoif", "desencoir", "desencoira"+u"\u00E7", "desencoiv", "desencoivar",
"desencol", "desencoleriz", "desencomend", "desenconch", "desencontr", "desencor",
"desencoraj", "desencordo", "desencorp", "desencorpor", "desencorti"+u"\u00E7", "desencoscor",
"desencost", "desencostal", "desencour", "desencoura"+u"\u00E7", "desencout", "desencov",
"desencovil", "desencovilh", "desencrav", "desencravelh", "desencravilh", "desencrenc",
"desencresp", "desencrost", "desencru", "desencruz", "desencub", "desencuc",
"desenculatr", "desencurral", "desencurv", "desendemoninh", "desendeus", "desendireit",
"desendivid", "desenergiz", "desenerv", "desenevo", "desenfad", "desenfaix",
"desenfard", "desenfardel", "desenfarpel", "desenfarrusc", "desenfart", "desenfasti",
"desenfeit", "desenfeiti"+u"\u00E7", "desenfeix", "desenferm", "desenferruj", "desenfest",
"desenfeud", "desenfez", "desenfi", "desenfileir", "desenflech", "desenflor",
"desenfoc", "desenforc", "desenforj", "desenform", "desenforn", "desenfrasc",
"desenfrech", "desenfren", "desenfronh", "desenfueir", "desenfun", "desenfurn",
"desenfusc", "desenga"+u"\u00E7", "desengaiol", "desengaj", "desengalap", "desengalfinh",
"desengami", "desengan", "desenganch", "desengaranh", "desengarraf", "desengarranch",
"desengasg", "desengast", "desengastalh", "desengat", "desengatilh", "desengavet",
"desengelh", "desenglob", "desengod", "desengolf", "desengom", "desengon"+u"\u00E7",
"desengord", "desengordur", "desengorgit", "desengra"+u"\u00E7", "desengrad", "desengrainh",
"desengralh", "desengranz", "desengravat", "desengravit", "desengrax", "desengraz",
"desengren", "desengrenh", "desengril", "desengrimp", "desengrinald", "desengrip",
"desengross", "desengrum", "desengrun", "desenguaranch", "desengui"+u"\u00E7", "desengulh",
"desengurgit", "desengurunh", "desenh", "desenjaul", "desenjo", "desenjorc",
"desenla"+u"\u00E7", "desenlambuz", "desenlap", "desenlev", "desenli"+u"\u00E7", "desenlod",
"desenloda"+u"\u00E7", "desenlut", "desenluv", "desenodo", "desenoj", "desenovel",
"desenquadr", "desenqueix", "desenrai", "desenraiv", "desenram", "desenrasc",
"desenred", "desenregel", "desenri"+u"\u00E7", "desenrij", "desenrist", "desenriz",
"desenrodilh", "desenrol", "desenrolh", "desenrosc", "desenroup", "desenrug",
"desensabo", "desensaburr", "desensac", "desensanguent", "desensarilh", "desensart",
"desenseb", "desensil", "desensin", "desensoc", "desensolv", "desensombr",
"desensop", "desensurr", "desentabu", "desentabul", "desentaip", "desental",
"desentalisc", "desentalo", "desentaramel", "desentarrax", "desentedi", "desenterr",
"desenterro", "desentes", "desentesoir", "desentesour", "desentibi", "desento",
"desentoc", "desenton", "desentorro", "desentort", "desentralh", "desentramel",
"desentran"+u"\u00E7", "desentranh", "desentrap", "desentrav", "desentrela"+u"\u00E7", "desentrelinh",
"desentressolh", "desentrev", "desentrever", "desentrincheir", "desentroix", "desentronc",
"desentroniz", "desentropilh", "desentros", "desentroux", "desentub", "desentulh",
"desenturm", "desenturv", "desentusiasm", "desenubl", "desenvas", "desenvasilh",
"desenvencilh", "desenvenen", "desenvered", "desenverg", "desenvergonh", "desenverniz",
"desenvies", "desenvincilh", "desenviol", "desenvisc", "desenxarci", "desenxerg",
"desenxofr", "desenxov", "desenxovalh", "desequilibr", "desequip", "desequivoc",
"deserd", "deserotiz", "desert", "desertific", "desescal", "desescolariz",
"desescud", "desespant", "desespartilh", "desesper", "desesperan"+u"\u00E7", "desespinh",
"desespiraliz", "desespiritualiz", "desesquip", "desessenci", "desestabiliz", "desestagn",
"desestaliniz", "desestanh", "desestatiz", "desesteir", "desestim", "desestimul",
"desestiv", "desestorv", "desestr", "desestrelej", "desestrib", "desestrutur",
"desestud", "desetaniz", "desevangeliz", "desexcit", "desexcomung", "desfabric",
"desfabul", "desfa"+u"\u00E7", "desfadig", "desfai", "desfalc", "desfalca"+u"\u00E7",
"desfalco", "desfanatiz", "desfantasi", "desfard", "desfarel", "desfarinh",
"desfas", "desfascistiz", "desfataliz", "desfatan", "desfavel", "desfaz",
"desfebr", "desfech", "desfeminiz", "desfenestr", "desferr", "desferrolh",
"desferruj", "desfertiliz", "desfi", "desfibr", "desfibril", "desfibrilh",
"desfibrin", "desfigur", "desfil", "desfileir", "desfilh", "desfili",
"desfilipiz", "desfit", "desfivel", "desflegm", "desflor", "desflorest",
"desfoc", "desfocaliz", "desfoleg", "desfolg", "desfolh", "desfoli",
"desfonologiz", "desfor"+u"\u00E7", "desform", "desformat", "desforn", "desforr",
"desfortific", "desfosfor", "desfosforil", "desfrad", "desfragment", "desfrald",
"desfranj", "desfraterniz", "desfrech", "desfrequent", "desfris", "desfroux",
"desfrunch", "desfrut", "desfunch", "desfuncion", "desfund", "desfurfur",
"desgab", "desgadelh", "desgaf", "desgaiv", "desgalg", "desgalh",
"desgalvaniz", "desgargal", "desgarr", "desgarron", "desgas", "desgasconiz",
"desgaseific", "desgasific", "desgast", "desgavel", "desgel", "desgenitaliz",
"desgentiliz", "desgeometriz", "desgermaniz", "desgermin", "desglabr", "desglamouriz",
"desgliceriniz", "desglobuliz", "desglos", "desgoel", "desgordur", "desgorgomil",
"desgorj", "desgost", "desgovern", "desgovernamentaliz", "desgra"+u"\u00E7", "desgraci",
"desgradu", "desgram", "desgramaticaliz", "desgran", "desgrav", "desgravid",
"desgravit", "desgraviz", "desgrax", "desgreg", "desgrenh", "desgret",
"desgrilho", "desgrinald", "desgrud", "desgrum", "desguamp", "desguard",
"desguarit", "desguedelh", "desgui", "desidealiz", "desidentific", "desideologiz",
"desidrat", "desidrogen", "desidrogeniz", "design", "desigual", "desilumin",
"desilusion", "desilustr", "desimagin", "desiman", "desimaniz", "desimant",
"desimobiliz", "desimplant", "desimplic", "desimpossibilit", "desimpossibiliz", "desimpregn",
"desimprens", "desimpression", "desin"+u"\u00E7", "desincentiv", "desinch", "desinclin",
"desincompatibiliz", "desincorpor", "desincrimin", "desincrust", "desincub", "desindex",
"desindic", "desindici", "desindividualiz", "desindustrializ", "desinfam", "desinfeccion",
"desinfecion", "desinfect", "desinfern", "desinfest", "desinfet", "desinficion",
"desinfl", "desinflacion", "desinflam", "desinfluenci", "desinform", "desingles",
"desingurgit", "desinjuri", "desinquiet", "desinset", "desinsetiz", "desinstal",
"desinstitucionaliz", "desinsufl", "desintegr", "desinteir", "desinteiri"+u"\u00E7", "desinterdit",
"desinteress", "desinterlig", "desintern", "desinternacionaliz", "desinterpret", "desintestin",
"desintimid", "desintoxic", "desintric", "desintrinc", "desinvagin", "desinvern",
"desinviol", "desioniz", "desipnotiz", "desipotec", "desirman", "desisc",
"desislamiz", "desjarret", "desjeju", "desjunt", "deslabializ", "desla"+u"\u00E7",
"deslacr", "desladrilh", "deslan", "deslanch", "deslap", "deslapid",
"deslarg", "deslarv", "deslass", "deslastr", "deslat", "deslaud",
"deslav", "deslavr", "desleald", "desleg", "deslegitim", "desleig",
"desleit", "desleix", "deslembr", "desli", "deslig", "deslignific",
"deslim", "deslind", "deslinhific", "deslint", "deslis", "deslivr",
"desliz", "desloc", "deslocaliz", "deslod", "deslogr", "deslomb",
"deslouc", "deslouv", "deslumbr", "deslustr", "desmacul", "desmadeir",
"desmagin", "desmagnetiz", "desmai", "desmain", "desmalencoliz", "desmalh",
"desmalhet", "desmalici", "desmam", "desmament", "desman", "desmanch",
"desmand", "desmang", "desmangalh", "desmanganes", "desmanilh", "desmaninh",
"desmaniv", "desmantel", "desmantilh", "desmaquil", "desmaquilh", "desmaranh",
"desmarc", "desmarchet", "desmarcializ", "desmargariniz", "desmargin", "desmarisc",
"desmascar", "desmassaj", "desmastr", "desmat", "desmaterializ", "desmatricul",
"desmazel", "desmazorr", "desmedr", "desmedul", "desmelancoliz", "desmelen",
"desmelhor", "desmelindr", "desmembr", "desmemori", "desmensur", "desment",
"desmergulh", "desmescl", "desmesmeriz", "desmesur", "desmetaliz", "desmetodiz",
"desmielin", "desmieliniz", "desmilitariz", "desmin", "desmineraliz", "desmiol",
"desmistific", "desmitific", "desmiud", "desmobil", "desmobilh", "desmobiliz",
"desmo"+u"\u00E7", "desmoch", "desmoder", "desmodul", "desmoed", "desmoir",
"desmoit", "desmold", "desmond", "desmonetiz", "desmonopoliz", "desmonotoniz",
"desmont", "desmonto", "desmoraliz", "desmorfiniz", "desmoron", "desmortalh",
"desmortific", "desmotiv", "desmour", "desmout", "desmud", "desmultiplic",
"desmunhec", "desmunici", "desmunicion", "desmur", "desmurch", "desmuscul",
"desmusg", "desnacionaliz", "desnalg", "desnarig", "desnasal", "desnasaliz",
"desnastr", "desnat", "desnatur", "desnaturaliz", "desnazific", "desnecessit",
"desneg", "desnegoci", "desnerv", "desnev", "desninh", "desniquel",
"desnivel", "desnoc", "desnodo", "desnoit", "desnoiv", "desnot",
"desnovel", "desnu", "desnubl", "desnuc", "desnucleariz", "desnud",
"desobrig", "desobumbr", "desocult", "desocup", "desodor", "desodoriz",
"desoficializ", "desofusc", "desol", "desoleific", "desolh", "desoner",
"desonest", "desonr", "desopil", "desopress", "desorbit", "desor"+u"\u00E7"+"ament",
"desorden", "desorelh", "desorganiz", "desorient", "desorn", "desoss",
"desoug", "desov", "desoxid", "desoxigen", "despach", "despacient",
"despadr", "despaganiz", "despagin", "despais", "despalataliz", "despalatiz",
"despalet", "despalh", "despalhet", "despalm", "despalmilh", "despamp",
"despampan", "despancreatiz", "despap", "desparafin", "desparafus", "desparament",
"desparasit", "desparceir", "desparcel", "desparelh", "desparr", "desparram",
"despartidariz", "desparv", "despass", "despast", "despatri", "despatrocin",
"despaulific", "despavil", "despaviment", "despe"+u"\u00E7", "despecuni", "despeda"+u"\u00E7",
"despedr", "despeg", "despeit", "despeitor", "despej", "despel",
"despelh", "despen", "despenaliz", "despenc", "despendur", "despenh",
"despenitenci", "despentaniz", "desperdi"+u"\u00E7", "desperfil", "despersonaliz", "despersonific",
"despert", "despes", "despesc", "despestan", "despetal", "despetrech",
"despic", "despied", "despigment", "despilch", "despin"+u"\u00E7", "desping",
"despinhaliz", "despinic", "despint", "despiolh", "despiroc", "despist",
"desplant", "desplum", "despoeir", "despoetiz", "despoj", "despolariz",
"despolet", "despoli", "despolimeriz", "despoliniz", "despolitiz", "despoloniz",
"despolp", "despolvilh", "desponder", "despong", "despont", "despontu",
"despopul", "despopulariz", "desportilh", "desportugues", "despos", "desposs",
"desposti"+u"\u00E7", "despostig", "despotiz", "despovo", "despraguej", "desprecat",
"despreci", "despreg", "despregui"+u"\u00E7", "despremi", "desprenh", "despreocup",
"desprepar", "despresilh", "despressuriz", "desprestigi", "desprez", "desprimor",
"desprior", "despriv", "desprivatiz", "desprivilegi", "desprocess", "desprofan",
"desprogram", "desproletariz", "despronunci", "despropaniz", "desproporcion", "desproposit",
"despropri", "desproteiniz", "desprotoniz", "desprum", "desprussianiz", "despucel",
"despudor", "despueriliz", "despum", "despurific", "desquadr", "desquadrilh",
"desquadrinh", "desqualific", "desquam", "desquart", "desquartin", "desquebr",
"desqueix", "desqueixol", "desqui", "desquici", "desquiet", "desquit",
"desrab", "desraig", "desram", "desrat", "desratiz", "desrecalc",
"desrefolh", "desregionaliz", "desregr", "desregulament", "desrelv", "desremedi",
"desrend", "desrepublicaniz", "desresin", "desrespeit", "desresponsabiliz", "desretrat",
"desri"+u"\u00E7", "desrisc", "desrisonh", "desriz", "desroch", "desrolh",
"desromaniz", "desromantiz", "desrosc", "desroup", "desrug", "desrum",
"desrussific", "dess", "dessabor", "dessaburr", "dessacr", "dessacraliz",
"dessagr", "dessaibr", "dessain", "dessalg", "dessalific", "dessalinific",
"dessaliniz", "dessalitr", "dessamouc", "dessangr", "dessarr", "dessarro",
"dessazon", "dessazonaliz", "dessec", "dessedent", "dessegment", "dessegred",
"dessegreg", "dessegur", "dessein", "desseiv", "dessel", "dessemelh",
"dessensibiliz", "dessepult", "dessesm", "dessexu", "dessexualiz", "dessilic",
"dessilicat", "dessilicific", "dessimpatiz", "dessinal", "dessincroniz", "dessintoniz",
"dessiti", "desso", "dessoalh", "dessobra"+u"\u00E7", "dessoc", "desso"+u"\u00E7"+"obr",
"dessol", "dessold", "dessolh", "dessolidariz", "dessolvat", "dessor",
"dessosseg", "dessoterr", "dessu", "dessubjug", "dessubstanci", "dessud",
"dessuj", "dessulfon", "dessulfur", "dessulfuriz", "dessurr", "destabiliz",
"destaboc", "destac", "destal", "destalh", "destaling", "destamp",
"destanch", "destaniniz", "destaniz", "destap", "destar", "destaramel",
"destarrach", "destel", "destelh", "destemper", "desterneir", "desterr",
"desterraplen", "desterritorializ", "desterro", "destet", "destil", "destin",
"destipific", "destiraniz", "destisn", "desto", "destoc", "destoiti"+u"\u00E7",
"destold", "destomb", "deston", "destor", "destornilh", "destorro",
"destorv", "destouc", "destouti"+u"\u00E7", "destoutu"+u"\u00E7", "destoxific", "destra"+u"\u00E7",
"destrag", "destraj", "destram", "destrambelh", "destramel", "destranc",
"destran"+u"\u00E7", "destrat", "destrav", "destravanc", "destravess", "destrein",
"destrel", "destrep", "destrib", "destribaliz", "destri"+u"\u00E7", "destrig",
"destrinc", "destrin"+u"\u00E7", "destrinch", "destrip", "destripul", "destroc",
"destro"+u"\u00E7", "destron", "destronc", "destroniz", "destropilh", "destrunf",
"desturv", "destutel", "desug", "desultraj", "desuman", "desumaniz",
"desumect", "desumidific", "desunh", "desunific", "desuniform", "desurbaniz",
"desus", "desvac", "desvair", "desvali", "desvalid", "desvalij",
"desvalis", "desvalor", "desvaloriz", "desvalvul", "desvar", "desvari",
"desvasculariz", "desvassal", "desvel", "desvelej", "desvencelh", "desvencilh",
"desvend", "desvener", "desventr", "desventur", "desvergonh", "desvermin",
"desvertebr", "desverticaliz", "desvez", "desvi", "desvidr", "desvidra"+u"\u00E7",
"desvig", "desvigi", "desvigor", "desvigoriz", "desvinc", "desvincilh",
"desvincul", "desvir", "desvirg", "desvirgin", "desvirginiz", "desvirgul",
"desviriliz", "desvirtu", "desviscer", "desvisg", "desvitaliz", "desvitamin",
"desvitaminiz", "desvitrific", "desvizinh", "desvocaliz", "desvolum", "desvoluntari",
"desvulcaniz", "desvulgariz", "desxadrez", "desxaret", "deszel", "deszinc",
"deszip", "detalh", "detard", "detect", "deten"+u"\u00E7", "deterior",
"determin", "detest", "detet", "deton", "detor", "detram",
"detrat", "detrect", "detrin"+u"\u00E7", "detron", "detrunc", "deturb",
"deturp", "devass", "devast", "devel", "deventr", "devis",
"devitrific", "devor", "devot", "devulcaniz", "devulg", "dezinc",
"dezincific", "diaboliz", "diaclas", "diadem", "diafaniz", "diafragm",
"diagnostic", "diagonaliz", "diagram", "dialect", "dialet", "dialis",
"dialog", "diamantiz", "diariz", "diatermocoagul", "diazot", "diazotiz",
"dic", "dicionari", "dicionariz", "dicotomiz", "dies", "dieseliz",
"difam", "diferen"+u"\u00E7", "diferenci", "dificult", "difrat", "digit",
"digitaliniz", "digitaliz", "digladi", "dign", "dignific", "digress",
"digression", "dilacer", "dilani", "dilapid", "dilarg", "dilat",
"diligenci", "dilucid", "diluvi", "diman", "dimension", "dimeriz",
"dimidi", "dinamit", "dinamitiz", "dinamiz", "dind", "dinumer",
"diploidiz", "diplom", "diplomaci", "diplomatiz", "direcion", "dirradic",
"disc", "discant", "disciplin", "discord", "discrep", "discretiz",
"discrimin", "discurs", "discus", "discutinh", "diseliz", "disfar"+u"\u00E7",
"disform", "disgram", "disgreg", "disjunt", "dispar", "disparat",
"dispens", "dispers", "disponibiliz", "disput", "dissec", "dissemelh",
"dissemin", "dissert", "dissidi", "dissimetriz", "dissimil", "dissimul",
"dissip", "dissoci", "disson", "dist", "distanci", "distension",
"distil", "distrat", "distrofic", "disturb", "dit", "ditong",
"diuturniz", "divag", "divaric", "diversific", "diviniz", "divis",
"divorci", "divulg", "dizim", "do", "dob", "dobr",
"do"+u"\u00E7", "dociliz", "document", "doest", "dogmatiz", "doidej",
"doidivan", "doir", "doirej", "dolariz", "dolomitiz", "dolomiz",
"doloriz", "dom", "domestic", "domicili", "domific", "domin",
"doming", "dop", "dorment", "dorminh", "dorminhoc", "dormit",
"dorsaliz", "dos", "dosific", "dossel", "dot", "dotaliz",
"doudej", "doudivan", "dour", "dourej", "douti", "doutor",
"doutrin", "drag", "drain", "dramatiz", "drapej", "dren",
"dribl", "drog", "dualiz", "dubl", "duch", "ductiliz",
"duel", "dueliz", "duet", "dulcific", "dul"+u"\u00E7"+"or", "dupl",
"duplic", "dur", "durb", "duvid", "ealm", "earc",
"ebaniz", "eburn", "eburnific", "ec", "echacorv", "eclips",
"eco", "ecocardiograf", "ecoencefalograf", "ecograf", "ecoloc", "ecolocaliz",
"economiz", "ecossistematiz", "ectomorfiz", "ecumeniz", "edafostratigraf", "edafostratograf",
"edemaci", "edematiz", "edeniz", "edent", "edicion", "edicul",
"edific", "edit", "editor", "editori", "educ", "educomunic",
"edul"+u"\u00E7", "edulcor", "edul"+u"\u00E7"+"or", "eequacion", "efabul", "efeitu",
"efemeriz", "efemin", "efeminiz", "efervent", "efetiv", "efetu",
"eficaci", "efigi", "egipcianiz", "egocentr", "egocentraliz", "egoidealiz",
"egoism", "ei"+u"\u00E7", "eim", "eiv", "eix", "ejacul",
"eject", "ejet", "el", "elabor", "elan"+u"\u00E7", "elastific",
"elastiz", "eleboriz", "electrific", "electriz", "electrocortigraf", "electrocut",
"electrodeposit", "electrodiagnostic", "electrolis", "electromedic", "electron", "electrotip",
"elegantiz", "elenc", "eletrific", "eletriz", "eletrocortigraf", "eletrocut",
"eletrodeposit", "eletrolis", "eletromedic", "eletron", "eletroniz", "eletrotip",
"elev", "elici", "elicit", "elimin", "elips", "elitiz",
"elogi", "elong", "elucid", "elucubr", "elutri", "ema"+u"\u00E7",
"ema"+u"\u00E7"+"aroc", "emaci", "emacul", "emadeir", "emadeix", "emagazin",
"emagot", "emagr", "emagrent", "emal", "emalh", "emalhet",
"eman", "emanch", "emancip", "emanding", "emangueir", "emanhu"+u"\u00E7",
"emanilh", "emanjeric", "emanoc", "emant", "emantilh", "emar",
"emaranh", "emarj", "emarlot", "emartilh", "emascar", "emascul",
"emass", "emassilh", "emastr", "ematilh", "embabac", "embabad",
"emba"+u"\u00E7", "embacel", "embaci", "embag", "embaga"+u"\u00E7", "embainh",
"embal", "embalan"+u"\u00E7", "embal"+u"\u00E7", "embaloi"+u"\u00E7", "embalou"+u"\u00E7", "embals",
"embalsam", "emban", "embanan", "emband", "embandeir", "embara"+u"\u00E7",
"embarafust", "embaralh", "embarb", "embarbasc", "embarbel", "embarbilh",
"embarc", "embard", "embarg", "embarr", "embarrac", "embarranc",
"embarreir", "embarrel", "embarret", "embarric", "embarrig", "embarril",
"embarulh", "embas", "embasbac", "embast", "embasti", "embastilh",
"embatin", "embatoc", "embatuc", "embatum", "embauc", "embaul",
"embebec", "embebed", "embeber", "embebor", "embei"+u"\u00E7", "embelec",
"embelen", "embelez", "embelg", "embelinh", "embeloir", "embelour",
"embelout", "embern", "emberrinch", "embesilh", "embesoir", "embesour",
"embespinh", "embest", "embetesg", "embetum", "embezerr", "embiboc",
"embic", "embicheir", "embigot", "embilh", "embiloc", "embing",
"embioc", "embiot", "embir", "embirich", "embiroc", "embirr",
"embirrinch", "embirut", "embisc", "embitesg", "embizug", "emblem",
"emblematiz", "embob", "embobin", "emboc", "embo"+u"\u00E7", "embo"+u"\u00E7"+"al",
"embocet", "embochech", "embodalh", "embodeg", "embodoc", "emboit",
"emboj", "embol", "embolat", "emboldreg", "emboldri", "embolet",
"embolig", "embolinh", "emboliz", "emboloir", "embolor", "embolot",
"embolour", "embols", "embon", "embonec", "embonecr", "embonic",
"emboquilh", "emborbet", "emborc", "embornal", "emborquilh", "emborr",
"emborrach", "emborralh", "emborrasc", "embosc", "embosn", "embost",
"embostel", "embot", "embotelh", "embotij", "embotilh", "emboug",
"embout", "embrab", "embra"+u"\u00E7", "embraceir", "embrag", "embrai",
"embram", "embranc", "embrech", "embrenh", "embret", "embriag",
"embric", "embrid", "embrigad", "embrinc", "embrion", "embroc",
"embroch", "embrom", "embruac", "embrulh", "embrum", "embrusc",
"embrut", "embrux", "embu"+u"\u00E7", "embu"+u"\u00E7"+"al", "embuch", "embud",
"emburac", "emburel", "emburgues", "emburilh", "emburr", "emburric",
"emburrisc", "emburulh", "embuzi", "embuzin", "emech", "emed",
"emedoi"+u"\u00E7", "emedoich", "emedou"+u"\u00E7", "emedouch", "emel", "emend",
"emendic", "emenin", "ement", "emeroi"+u"\u00E7", "emerou"+u"\u00E7", "emeruj",
"emetiz", "emezi", "emigr", "eminenci", "eminhoc", "emocion",
"emoir", "emoit", "emold", "emoldur", "emoleir", "emolh",
"emon", "emorda"+u"\u00E7", "emoroi"+u"\u00E7", "emorou"+u"\u00E7", "emorrinh", "emosc",
"emosqueir", "emost", "emour", "emp", "empac", "empacavir",
"empach", "empa"+u"\u00E7"+"oc", "empacot", "empades", "empadilh", "empadro",
"empaiol", "empal", "empalam", "empalet", "empalh", "empalhasc",
"empalheir", "empali", "empalit", "empalm", "empalud", "empampan",
"empan", "empanad", "empanc", "empandeir", "empandilh", "empandin",
"empando", "empaneir", "empanem", "empang", "empangin", "empanh",
"empantan", "empantuf", "empanturr", "empanzin", "empap", "empapel",
"empapoil", "empapoul", "empapu"+u"\u00E7", "empaquet", "empar", "emparament",
"emparc", "emparceir", "emparcel", "empard", "empared", "emparelh",
"emparent", "emparr", "emparreir", "emparv", "emparvat", "emparvo",
"empasm", "empast", "empastel", "empat", "empaturr", "empaupic",
"empaves", "empavo", "empavon", "empe"+u"\u00E7", "empecad", "empecilh",
"empe"+u"\u00E7"+"onh", "empe"+u"\u00E7"+"onhent", "empedern", "empedr", "empeg", "empeir",
"empeireir", "empeitic", "empej", "empel", "empelam", "empelic",
"empelicul", "empelot", "empen", "empenach", "empenc", "empendo",
"empenetr", "empenh", "empenhor", "empenuj", "empeol", "empepin",
"empequenit", "empereir", "emperiquit", "emperl", "empern", "empernic",
"emperol", "emperr", "empertig", "empes", "empesg", "empesso",
"empest", "empetec", "empetelec", "empez", "empezinh", "empic",
"empicot", "empilch", "empilec", "empilh", "empin", "empinhoc",
"empinoc", "empiol", "empiolh", "empior", "empip", "empipoc",
"empisc", "empistol", "empiteir", "emplac", "emplanch", "emplasm",
"emplast", "emplastr", "emplum", "empo", "empobr", "empo"+u"\u00E7",
"empocilg", "empoder", "empoeir", "empoit", "empol", "empolam",
"empolasm", "empoleir", "empolg", "empolh", "empolm", "empolv",
"empolvilh", "empomad", "empomb", "emponch", "emponder", "empont",
"empop", "emporc", "emporcalh", "emposs", "empost", "empostem",
"empostig", "emprateleir", "empraz", "empreg", "empregui"+u"\u00E7", "empreit",
"emprenh", "empres", "empresari", "emprest", "emprestad", "emprision",
"empro", "emprum", "empulg", "empulh", "empunh", "empup",
"empurpur", "empurr", "empux", "emul", "emulsific", "emulsion",
"emund", "emuralh", "emurch", "enagen", "enaif", "enaip",
"enalt", "enamor", "enarr", "enastr", "enat", "enateir",
"encab", "encabe"+u"\u00E7", "encabeir", "encabel", "encabeliz", "encabreim",
"encabrest", "encabrit", "encabrun", "encabul", "encabulh", "enca"+u"\u00E7"+"ap",
"encach", "encacha"+u"\u00E7", "encachapu"+u"\u00E7", "encachi", "encachoeir", "encachol",
"encadarro", "encadeir", "encadern", "encafif", "encafu", "encafurn",
"encaga"+u"\u00E7", "encaibr", "encaieir", "encaipor", "encairel", "encaix",
"encaixilh", "encaixot", "encal", "encalacr", "encalamech", "encalamistr",
"encalamouc", "encalburd", "encalc", "encal"+u"\u00E7", "encald", "encaldeir",
"encaleir", "encalg", "encalh", "encalic", "encali"+u"\u00E7", "encalist",
"encalistr", "encalm", "encalmurr", "encalomb", "encalor", "encalung",
"encam", "encama"+u"\u00E7", "encamarad", "encamarinh", "encamarot", "encamb",
"encambeir", "encambich", "encambit", "encambo", "encambon", "encambuc",
"encambulh", "encamel", "encamer", "encaminh", "encamis", "encamoroi"+u"\u00E7",
"encamorou"+u"\u00E7", "encamp", "en"+u"\u00E7"+"amp", "encampan", "encampesin", "encamur"+u"\u00E7",
"encan", "encanalh", "encanastr", "encancel", "encancer", "encandil",
"encandol", "encanel", "encanfinf", "encang", "encangalh", "encangot",
"encanh", "encanic", "encani"+u"\u00E7", "encanit", "encaniz", "encano",
"encant", "encanteir", "encantin", "encantinh", "encanto", "encanton",
"encanut", "encanzin", "encanzo", "encap", "encapach", "encapar",
"encapel", "encapet", "encapoeir", "encapot", "encaprich", "encapsul",
"encapuch", "encapuz", "encar", "encaracol", "encaramanch", "encaramel",
"encaramon", "encaramuj", "encarang", "encaranguej", "encarant", "encarantonh",
"encarapel", "encarapinh", "encarapit", "encarapu"+u"\u00E7", "encaravelh", "encarboniz",
"encarcad", "encarcer", "encarch", "encardum", "encarent", "encaret",
"encarg", "encarij", "encarioc", "encarn", "encarneir", "encarni"+u"\u00E7",
"encaro"+u"\u00E7", "encaroch", "encarol", "encarquilh", "encarr", "encarramon",
"encarranc", "encarrapi"+u"\u00E7", "encarrapit", "encarrasc", "encarraspan", "encarreg",
"encarreir", "encarret", "encarril", "encarrilh", "encarrisc", "encart",
"encartalh", "encartaz", "encarteir", "encartol", "encartuch", "encarvo",
"encarvoej", "encarvoi"+u"\u00E7", "encas", "encasac", "encasal", "encasc",
"encascalh", "encascor", "encasmurr", "encasquet", "encasquilh", "encastalh",
"encastel", "encasto", "encastr", "encasul", "encataplasm", "encatarr",
"encatarro", "encatrafi", "encatramon", "encatravilh", "encatrin", "encauch",
"encaud", "encaustic", "encav", "encavac", "encaval", "encavaleir",
"encavalg", "encavalit", "encavern", "encavilh", "encazuc", "encebol",
"encedr", "encegueir", "enceir", "encel", "enceleir", "encen",
"encendr", "encensori", "encentr", "encep", "encer", "encerebr",
"enceroil", "enceroul", "encerr", "encert", "encervej", "encest",
"encet", "enceved", "enchacot", "enchafurd", "enchamarr", "enchambo",
"enchambr", "enchamej", "enchampanh", "enchapel", "enchapot", "enchapour",
"enchapu"+u"\u00E7", "encharc", "encharol", "enchavet", "enchi"+u"\u00E7"+"ar", "enchicharr",
"enchimarr", "enchiment", "enchiqueir", "encho"+u"\u00E7", "enchocalh", "enchoiri"+u"\u00E7",
"enchombr", "enchouri"+u"\u00E7", "enchous", "enchuma"+u"\u00E7", "enchumarr", "enchumb",
"enchumbeir", "enchusm", "encieir", "encilh", "encim", "encimalh",
"encinch", "encinh", "encint", "encinz", "encinzeir", "encirr",
"encisc", "encist", "encium", "enclaustr", "enclausur", "enclavinh",
"encloac", "encobard", "encobert", "encobilh", "encobr", "encoch",
"encocurut", "encod", "encodo", "encof", "encofr", "encoif",
"encoim", "encoir", "encoira"+u"\u00E7", "encoit", "encoivar", "encol",
"encoleir", "encoleriz", "encoloc", "encomend", "encomi", "encomiss",
"encomoro"+u"\u00E7", "encomoroi"+u"\u00E7", "encomorou"+u"\u00E7", "encomprid", "encomunh", "encon",
"enconap", "enconc", "enconch", "encond", "encondiz", "enconformiz",
"enconhac", "enconic", "enconteir", "enconto", "encontr", "encontro",
"encop", "encoquin", "encoquinh", "encor", "encoraj", "encorc",
"encorcund", "encordel", "encordo", "encorn", "encornet", "encornich",
"encoronh", "encorp", "encorpor", "encorquilh", "encorrelh", "encorrent",
"encorri", "encorric", "encorrilh", "encorrovilh", "encortelh", "encorti"+u"\u00E7",
"encortin", "encoruj", "encoscor", "encospi", "encost", "encostal",
"encostel", "encostr", "encotinh", "encouch", "encour", "encoura"+u"\u00E7",
"encout", "encov", "encovard", "encovil", "encovilh", "encovit",
"encramoio"+u"\u00E7", "encramou"+u"\u00E7", "encrass", "encrav", "encraveir", "encravelh",
"encravilh", "encrenc", "encresp", "encript", "encris", "encrist",
"encristin", "encroc", "encrost", "encru", "encruent", "encrust",
"encruz", "encruzilh", "encub", "encuc", "encucharr", "encueir",
"enculp", "encultur", "encunh", "encurral", "encurrelh", "encurric",
"encurrich", "encurrilh", "encurt", "encurv", "encutinh", "endech",
"endeflux", "endej", "endemiz", "endemoninh", "endent", "endere"+u"\u00E7",
"endeus", "endiabr", "endilg", "endinh", "endinheir", "endireit",
"endivid", "endocardiograf", "endocit", "endocruz", "endoid", "endoming",
"endors", "endoss", "endoud", "endoutrin", "endrag", "endromin",
"endur", "endurent", "enebri", "energiz", "enerv", "enesg",
"enev", "enevo", "enfaceir", "enfachoc", "enfad", "enfaix",
"enfang", "enfanic", "enfar", "enfard", "enfardel", "enfardo",
"enfardon", "enfarel", "enfarfalh", "enfarinh", "enfarn", "enfaro",
"enfarp", "enfarpel", "enfarrap", "enfarrusc", "enfart", "enfas",
"enfascal", "enfasti", "enfatiot", "enfatiz", "enfatu", "enfebr",
"enfechel", "enfedilh", "enfei", "enfeir", "enfeit", "enfeiti"+u"\u00E7",
"enfeix", "enfelp", "enfeltr", "enfeluj", "enfen", "enferm",
"enferr", "enferret", "enferrolh", "enferruj", "enfest", "enfesto",
"enfeston", "enfeud", "enfez", "enfezin", "enfi", "enfieir",
"enfileir", "enfinc", "enfirm", "enfistul", "enfit", "enfiteutic",
"enfivel", "enfix", "enflait", "enflanel", "enfleim", "enflor",
"enfobi", "enfoc", "enfog", "enfoi"+u"\u00E7", "enfol", "enfolech",
"enfolh", "enfolip", "enforc", "enforj", "enform", "enformos",
"enforn", "enforquilh", "enforr", "enfort", "enfou"+u"\u00E7", "enfrac",
"enfrag", "enfraldic", "enfranj", "enfraquent", "enfrasc", "enfrech",
"enfren", "enfrenesi", "enfrenis", "enfrent", "enfrest", "enfri",
"enfrond", "enfronh", "enfront", "enfrut", "enfu", "enfueir",
"enfuleim", "enfulij", "enfum", "enfuma"+u"\u00E7", "enfumar", "enfun",
"enfuner", "enfunil", "enfunisc", "enfurd", "enfuri", "enfurn",
"enfus", "enfusc", "enfust", "enfuzil", "eng", "engabel",
"engabrich", "enga"+u"\u00E7", "engadanh", "engadelh", "engai", "engaifon",
"engaiol", "engaj", "engajat", "engal", "engalan", "engalap",
"engaldrap", "engaldrip", "engalfinh", "engalg", "engalh", "engalhardet",
"engalhop", "engalhot", "engali", "engalic", "engalicism", "engalinh",
"engalisp", "engalop", "engalri"+u"\u00E7", "engambel", "engambit", "engamel",
"engan", "enganch", "engangorr", "engangren", "engaramp", "engaranh",
"engarant", "engarap", "engaravit", "engarbon", "engarf", "engargant",
"engarguel", "engarnach", "engarr", "engarraf", "engarrafon", "engarup",
"engasg", "engasgalh", "engasgat", "engast", "engastalh", "engasto",
"engat", "engatalh", "engatanh", "engati"+u"\u00E7", "engatilh", "engatinh",
"engavel", "engavet", "engavinh", "engazofil", "engazop", "engazup",
"engelh", "engemin", "engendr", "engenh", "engeroc", "engess",
"engig", "englob", "englobul", "engob", "engod", "engodilh",
"engolf", "engolip", "engom", "engon"+u"\u00E7", "engonh", "engoni",
"engord", "engordur", "engorgit", "engorr", "engra"+u"\u00E7", "engrad",
"engrambel", "engramp", "engrampon", "engranz", "engravat", "engravid",
"engravit", "engrax", "engraz", "engrazul", "engrel", "engren",
"engrenh", "engrif", "engril", "engrim", "engrimp", "engrimpin",
"engrimpon", "engrinald", "engrip", "engriquilh", "engrol", "engross",
"engrum", "engrunh", "enguedelh", "engui"+u"\u00E7", "enguirland", "enguizalh",
"engulh", "engulip", "engulosin", "engunh", "enigm", "enigmatiz",
"enjaez", "enjalg", "enjambr", "enjang", "enjangr", "enjaul",
"enjeir", "enjeit", "enjeriz", "enjic", "enjo", "enjoi",
"enjoin", "enjon", "enjorc", "enjorg", "enjouj", "enjuan",
"enjug", "enjun"+u"\u00E7", "enlabared", "enlabi", "enlabirint", "enlabrusc",
"enlabuz", "enla"+u"\u00E7", "enla"+u"\u00E7"+"arot", "enlacr", "enladeir", "enlag",
"enlaiv", "enlam", "enlambuj", "enlambuz", "enlamin", "enlanz",
"enlap", "enlaru"+u"\u00E7", "enlat", "enleiv", "enlerd", "enlev",
"enli"+u"\u00E7", "enloc", "enlod", "enloda"+u"\u00E7", "enloi"+u"\u00E7", "enloir",
"enlois", "enloj", "enlomb", "enlouc", "enlou"+u"\u00E7", "enlour",
"enlous", "enluar", "enludr", "enlur", "enlut", "enluv",
"enod", "enodo", "enog", "enoiri"+u"\u00E7", "enoit", "enoj",
"enoliz", "enosilh", "enouri"+u"\u00E7", "enout", "enov", "enovel",
"enquadr", "enquadrilh", "enquart", "enqueij", "enqueix", "enquezil",
"enquilh", "enquist", "enquit", "enquizil", "enrab", "enrabeir",
"enrabich", "enrabit", "enra"+u"\u00E7", "enrai", "enraiv", "enram",
"enramalh", "enramalhet", "enramel", "enramilhet", "enran"+u"\u00E7", "enranch",
"enrapos", "enrasc", "enrastilh", "enred", "enredoi"+u"\u00E7", "enredom",
"enredou"+u"\u00E7", "enreg", "enregel", "enregist", "enregistr", "enregueir",
"enreix", "enrelh", "enrelheir", "enrelv", "enremel", "enremin",
"enremiss", "enrend", "enrepolh", "enresin", "enresm", "enrest",
"enresti", "enreves", "enri", "enric", "enri"+u"\u00E7", "enricon",
"enrif", "enrij", "enrilh", "enrilheir", "enrim", "enrincon",
"enrip", "enriquent", "enrist", "enristr", "enrix", "enriz",
"enroc", "enroch", "enrod", "enrodel", "enrodilh", "enrodrig",
"enrol", "enrolh", "enrosc", "enroup", "enrox", "enru"+u"\u00E7",
"enruf", "enrug", "enrumin", "ensabo", "ensaburr", "ensac",
"ensacol", "ensai", "ensaibr", "ensainh", "ensal"+u"\u00E7", "ensaliv",
"ensalm", "ensalmoir", "ensalmour", "ensalsich", "ensamarr", "ensambenit",
"ensambl", "ensamp", "ensanch", "ensandal", "ensandol", "ensanduich",
"ensanef", "ensangu", "ensanguent", "ensanguinh", "ensanh", "ensanzor",
"ensapez", "ensarapulh", "ensardinh", "ensarilh", "ensarjet", "ensarraf",
"ensarranh", "ensart", "ensartilh", "enseb", "ensec", "ensed",
"ensei", "enseir", "enseiv", "ensej", "enselv", "ensement",
"enserralh", "ensesg", "ensev", "ensil", "ensilv", "ensilveir",
"ensimesm", "ensin", "enso", "ensoalh", "ensoalheir", "ensoberb",
"ensobrad", "ensoc", "ensod", "ensofreg", "ensog", "ensoiss",
"ensoleir", "ensolv", "ensombr", "enson", "ensop", "ensopi",
"ensorn", "ensoss", "ensovac", "ensuj", "ensujent", "ensumagr",
"ensurr", "ensurraip", "ensurro", "entabic", "entaboc", "entabu",
"entabul", "entaburr", "entach", "entafulh", "entaip", "ental",
"entaleig", "entaleir", "entalh", "entalic", "entaling", "entalisc",
"entalo", "entam", "entanc", "entang", "entani"+u"\u00E7", "entap",
"entapet", "entapi"+u"\u00E7", "entapig", "entapiz", "entaramel", "entarrax",
"entartalh", "entavol", "entec", "entedi", "enteix", "entej",
"entejol", "entejuc", "entel", "entelh", "entelheir", "entendi",
"entenebr", "entenx", "enterr", "enterreir", "enterro", "enterturb",
"entes", "enteso", "entesoir", "entesoiris", "entesour", "entesouriz",
"entest", "entesteferr", "entibi", "entic", "entijol", "entijuc",
"entint", "entirit", "entisic", "entisn", "entiv", "ento",
"entoalh", "entoc", "entocai", "entog", "entoi"+u"\u00E7", "entoiceir",
"entoir", "entoj", "entol", "entold", "entolh", "enton",
"entont", "entorn", "entorp", "entorro", "entort", "entouc",
"entou"+u"\u00E7", "entouceir", "entoupeir", "entour", "entourinh", "entr",
"entraj", "entralh", "entram", "entrambelic", "entrambic", "entran"+u"\u00E7",
"entranh", "entranqueir", "entrap", "entrast", "entravess",
"entravisc", "entreadivinh", "entreajud", "entream", "entrearremess", "entrebeij",
"entrecerr", "entrech", "entrechoc", "entrecort", "entrecruz", "entredevor",
"entredilacer", "entredisput", "entrefal", "entrefech", "entrefestej", "entrefi",
"entrefigur", "entrefolh", "entreg", "entrela"+u"\u00E7", "entrelembr", "entrelh",
"entrelig", "entrelinh", "entreloi"+u"\u00E7", "entrelou"+u"\u00E7", "entremescl", "entremistur",
"entremostr", "entremurmur", "entrenubl", "entreocult", "entreolh", "entrepar",
"entrepass", "entrepelej", "entrepens", "entrepern", "entrepic", "entrepigait",
"entrepil", "entreplic", "entrerreclam", "entrerreg", "entrescut", "entresilh",
"entressac", "entressach", "entressolh", "entressonh", "entressufoc", "entretalh",
"entretel", "entretoc", "entretrucid", "entreturb", "entrev", "entrevaz",
"entrevel", "entrever", "entrevist", "entrez", "entrezilh", "entrilh",
"entrin"+u"\u00E7", "entrincheir", "entriz", "entroc", "entro"+u"\u00E7", "entroix",
"entromb", "entron", "entronc", "entronch", "entroniz", "entrope"+u"\u00E7",
"entropic", "entropigait", "entropilh", "entros", "entroux", "entrovisc",
"entrud", "entruj", "entrunf", "entrunfinh", "entrupic", "entrupigait",
"entub", "entuch", "entuf", "entujuc", "entulh", "entupigait",
"enturb", "enturm", "enturv", "enturvisc", "entusiasm", "enubl",
"enumer", "enunci", "enuvi", "envagin", "envaid", "enval",
"envalenton", "envar", "envaret", "envarot", "envas", "envasilh",
"envaz", "envelhac", "envelhent", "envelop", "envencelh", "envencilh",
"envenen", "enventan", "enver", "enverdej", "envered", "enverg",
"envergonh", "envermelh", "enverniz", "enverrug", "enves", "envesg",
"envess", "envi", "envi"+u"\u00E7", "envid", "envidilh", "envidra"+u"\u00E7",
"envieir", "envies", "envig", "envigor", "envigot", "envinagr",
"envincilh", "envinh", "envinhadalh", "enviper", "envisc", "envisg",
"enviveir", "envol", "envolucr", "envolut", "envult", "enxabelh",
"enxad", "enxadrez", "enxalavi", "enxal"+u"\u00E7", "enxalm", "enxambl",
"enxambr", "enxamelh", "enxampo", "enxampu", "enxaquec", "enxaquet",
"enxarci", "enxarop", "enxaug", "enxec", "enxelh", "enxequet",
"enxerc", "enxerg", "enxert", "enxi", "enxilh", "enxofr",
"enxog", "enxogalh", "enxombr", "enxor", "enxot", "enxov",
"enxovalh", "enxudr", "enxug", "enxumbr", "enxundi", "enxurd",
"enxurr", "enxurri", "enzamp", "enzamparin", "enzampon", "enzin",
"enzon", "enzonz", "eoliz", "epicuriz", "epidemi", "epigeniz",
"epigraf", "epigram", "epigramatic", "epigramatiz", "epil", "epilepsi",
"epileptiz", "epilog", "episodi", "epist", "epistoliz", "epitet",
"epitom", "equacion", "equaliz", "equidimension", "equidist", "equifinaliz",
"equilibr", "equimos", "equip", "equipar", "equiponder", "equivoc",
"ergastul", "ergotiz", "eri"+u"\u00E7", "erisipel", "erm",
"eros", "erotiz", "err", "erradic", "erri"+u"\u00E7", "eruct",
"erudit", "erv", "ervilh", "ervo", "esamp", "esbabac",
"esbaga"+u"\u00E7", "esbagach", "esbaganh", "esbago", "esbagulh", "esbalan"+u"\u00E7",
"esbald", "esbalh", "esbaliz", "esbalouc", "esbamb", "esbambo",
"esbandalh", "esbandeir", "esbandulh", "esbang", "esbanj", "esbanzalh",
"esbar", "esbarafust", "esbaralh", "esbarb", "esbarbot", "esbard",
"esbardalh", "esbarr", "esbarranc", "esbarreg", "esbarrel", "esbarrig",
"esbarro", "esbarroc", "esbarrond", "esbarrot", "esbarru", "esbarrunt",
"esbeat", "esbei"+u"\u00E7", "esbeij", "esbeir", "esbelt", "esbeltific",
"esbeltiz", "esbenair", "esbenic", "esberra"+u"\u00E7", "esberri", "esbic",
"esbich", "esbij", "esbilhot", "esbilit", "esbirr", "esbo"+u"\u00E7",
"esbocel", "esbodeg", "esbodel", "esbof", "esbofet", "esboi"+u"\u00E7",
"esboicel", "esbol", "esbombard", "esborcel", "esborcin", "esbord",
"esbordo", "esborni", "esboro", "esborol", "esborr", "esborra"+u"\u00E7",
"esborrach", "esborralh", "esborrat", "esborreg", "esborret", "esborrif",
"esborrot", "esboten", "esbou"+u"\u00E7", "esboucel", "esbrabej", "esbra"+u"\u00E7",
"esbracej", "esbram", "esbranqui"+u"\u00E7", "esbravej", "esbrech", "esbrin"+u"\u00E7",
"esbrit", "esbriz", "esbro", "esbronc", "esbrucin", "esbrug",
"esbrum", "esbugalh", "esbulh", "esburac", "esburg", "esbux",
"escabe"+u"\u00E7", "escabech", "escabel", "escabeleir", "escabeliz", "escabich",
"escabr", "escabuf", "escabuj", "escabulh", "escac", "escach",
"escacho", "escachol", "escachou"+u"\u00E7", "escacilh", "esca"+u"\u00E7"+"o", "escadabulh",
"escadeir", "escadra"+u"\u00E7", "escadrilh", "escafandr", "escafel", "escafon",
"escaful", "escafun", "escaga"+u"\u00E7", "escaganit", "escagarrinh", "escaiol",
"escal", "escalad", "escalamanc", "escalambr", "escalamouc", "escalavr",
"escald", "escaldeir", "escalf", "escali"+u"\u00E7", "escalmorr", "escalon",
"escalp", "escalpel", "escalpeliz", "escalpirr", "escalrach", "escalv",
"escalvej", "escam", "escamalh", "escamalho", "escamb", "escambi",
"escambich", "escambr", "escamech", "escamel", "escamis", "escamond",
"escamos", "escamot", "escamp", "escamuj", "escamur"+u"\u00E7", "escamurreng",
"escan", "escan"+u"\u00E7", "escancar", "escancel", "escanch", "escand",
"escandaliz", "escanec", "escaneriz", "escang", "escangalh", "escanganh",
"escangot", "escanho", "escanhot", "escanifr", "escaninh", "escantilh",
"escanto", "escanzel", "escanzi", "escanzurr", "escap", "escapel",
"escapolitiz", "escaqueir", "escaraban", "escarafol", "escarafunch", "escaramb",
"escaramel", "escaramu"+u"\u00E7", "escarapel", "escarapu"+u"\u00E7", "escaravelh", "escar"+u"\u00E7",
"escarcalh", "escarcavel", "escarch", "escarchol", "escard", "escarda"+u"\u00E7",
"escardi"+u"\u00E7", "escardich", "escardilh", "escardu"+u"\u00E7", "escarfunch", "escarg",
"escari", "escarific", "escarlat", "escarment", "escarmou"+u"\u00E7", "escarn",
"escarnefuch", "escarni", "escarnic", "escarnific", "escarninh", "escarnitiz",
"escaro"+u"\u00E7", "escarol", "escarou"+u"\u00E7", "escarp", "escarpel", "escarpi",
"escarpilh", "escarquej", "escarr", "escarrach", "escarraman", "escarranch",
"escarrapach", "escarrapanch", "escarrapat", "escarrapi"+u"\u00E7", "escarrapich", "escarrinh",
"escarum", "escarv", "escarvan"+u"\u00E7", "escarvo", "escarvun"+u"\u00E7", "escasal",
"escasc", "escassilh", "escasul", "escatel", "escatim", "escatin",
"escatoliz", "escav", "escavac", "escava"+u"\u00E7", "escavach", "escavalo",
"escaveir", "escavin", "escavinh", "escaxel", "eschamej", "esclamouc",
"esclerific", "escleros", "esco", "escobilh", "escoc", "escoch",
"escochin", "escod", "escogi", "escoi"+u"\u00E7", "escoicinh", "escoim",
"escolariz", "escoldrinh", "escoli", "escolm", "escolt", "escombr",
"escongemin", "esconjunt", "esconjur", "escons", "escopet", "escor",
"escor"+u"\u00E7", "escorcemel", "escorch", "escor"+u"\u00E7"+"omel", "escord", "escori",
"escorific", "escorj", "escorn", "escornich", "escoro", "escorra"+u"\u00E7",
"escorrach", "escorreg", "escorrich", "escorrima"+u"\u00E7", "escorripich", "escorropich",
"escortic", "escortin", "escortinh", "escoruj", "escorv", "escosip",
"escoti", "escotomiz", "escou"+u"\u00E7", "escoucemel", "escoucepinh", "escoucinh",
"escov", "escovalh", "escovilh", "escovinh", "escozic", "escrach",
"escrachet", "escrachin", "escrafunch", "escramalh", "escramel", "escrament",
"escramouc", "escramu"+u"\u00E7", "escravelh", "escraviz", "escresp", "escrevinh",
"escritur", "escriv", "escrofulariz", "escrofuliz", "escruch", "escruci",
"escrunch", "escrupuliz", "escrut", "escrutin", "escud", "escudeir",
"escudel", "escudrinh", "esculach", "esculc", "esculhamb", "escult",
"escultur", "escum", "escumilh", "escur", "escurej", "escurent",
"escus", "escut", "esdruxul", "esdruxuliz", "esfacel", "esfagunt",
"esfaim", "esfalc", "esfalca"+u"\u00E7", "esfalf", "esfalrip", "esfandeg",
"esfango", "esfanic", "esfard", "esfarel", "esfarfalh", "esfarinh",
"esfarp", "esfarpel", "esfarrap", "esfarrip", "esfarrunc", "esfatach",
"esfatan", "esfatech", "esfateix", "esfati", "esfavel", "esfelg",
"esfergulh", "esfervilh", "esfi", "esfiamp", "esfiap", "esfibr",
"esfloc", "esflor", "esfocinh", "esfoeir", "esfogueir", "esfoi"+u"\u00E7",
"esfoir", "esfol", "esfoleg", "esfolh", "esfolha"+u"\u00E7", "esfoli",
"esfor"+u"\u00E7", "esforfalh", "esforgulh", "esformig", "esforneir", "esfornic",
"esfornig", "esforquilh", "esforric", "esfoss", "esfossiliz", "esfossinh",
"esfou"+u"\u00E7", "esfrald", "esfran"+u"\u00E7", "esfrandelh", "esfrangalh", "esfreg",
"esfregon"+u"\u00E7", "esfregulh", "esfri", "esfrol", "esfrunh", "esfueir",
"esfugant", "esfugent", "esfulinh", "esfum", "esfuma"+u"\u00E7", "esfumar",
"esfumilh", "esfuminh", "esfundilh", "esfurac", "esfuranc", "esfurgalh",
"esfuri", "esfurinh", "esfuro", "esfutric", "esfuzi", "esfuzil",
"esg", "esga"+u"\u00E7", "esgadanh", "esgadelh", "esgadunh", "esgaiv",
"esgaldrip", "esgalg", "esgalh", "esgalinh", "esgalrich", "esgamo",
"esgamp", "esgan", "esganch", "esgani"+u"\u00E7", "esganif", "esganifr",
"esganit", "esgar", "esgarabanh", "esgarabulh", "esgarafunch", "esgarafunh",
"esgaratuj", "esgaravan", "esgaravat", "esgaravunch", "esgaravunh", "esgar"+u"\u00E7",
"esgardunh", "esgarg", "esgargal", "esgargalh", "esgarn", "esgarnach",
"esgarr", "esgarrabunh", "esgarrafunch", "esgarranch", "esgarranh", "esgasgalh",
"esgatanh", "esgati"+u"\u00E7", "esgod", "esgoel", "esgoldrej", "esgorj",
"esgot", "esgrabulh", "esgrafi", "esgrafit", "esgraminh", "esgravan",
"esgravat", "esgravelh", "esgravulh", "esgrelh", "esgril", "esgrim",
"esgrouvi", "esgrouvinh", "esgrovinh", "esguard", "esguarit", "esguedelh",
"esgueir", "esguelh", "esgui", "esguich", "esguilh", "esguit",
"esgurej", "esladro", "eslag", "eslagart", "eslaviz", "eslavoir",
"eslavour", "esling", "esm", "esmadrig", "esmag", "esmagach",
"esmagalh", "esmagunch", "esmai", "esmaleit", "esmalh", "esmalm",
"esmalt", "esmamon", "esmani", "esmanj", "esmantel", "esmaranh",
"esmaravalh", "esmarc", "esmarmoir", "esmarmour", "esmarnec", "esmarnoc",
"esmarro", "esmarrot", "esmartu"+u"\u00E7", "esmatru"+u"\u00E7", "esmech", "esmelm",
"esmelodr", "esmen", "esmer", "esmeril", "esmerilh", "esmerm",
"esmeti", "esmich", "esmifr", "esmigalh", "esmija"+u"\u00E7", "esmilh",
"esmiol", "esmirr", "esmiud", "esmiun"+u"\u00E7", "esmoc", "esmoch",
"esmoic", "esmoit", "esmol", "esmolamb", "esmon", "esmonc",
"esmond", "esmor"+u"\u00E7", "esmorda"+u"\u00E7", "esmordel", "esmordic", "esmordi"+u"\u00E7",
"esmorm", "esmoron", "esmorr", "esmorra"+u"\u00E7", "esmorrat", "esmorz",
"esmoss", "esmouc", "esmouquel", "esmout", "esmur", "esmurr",
"esmurra"+u"\u00E7", "esmurreg", "esmurreng", "esnac", "esnob", "esnoc",
"esnu", "espa"+u"\u00E7", "espacej", "espaci", "espacializ", "espad",
"espadachin", "espadag", "espadan", "espadeir", "espadel", "espadi",
"espadilh", "espadu", "espajeg", "espald", "espaldeir", "espalh",
"espalhafat", "espalhag", "espalit", "espalm", "espalomb", "espampan",
"espampar", "espan", "espanasc", "espanc", "espandong", "espanej",
"espanhol", "espanholiz", "espant", "espantarol", "espap", "espapa"+u"\u00E7",
"espaparr", "esparcel", "espardalh", "espargel", "esparr", "esparrach",
"esparralh", "esparram", "esparranh", "esparreg", "esparrel", "esparrim",
"esparrinh", "esparrod", "espars", "espartej", "espartilh", "esparv",
"esparvo", "espasm", "espassaric", "espassarot", "espatarr", "espatif",
"espatilh", "espatul", "espavent", "espavil", "espavit", "espavoriz",
"espec", "espe"+u"\u00E7", "especi", "especializ", "especific", "espect",
"espectraliz", "espectrific", "especul", "espeda"+u"\u00E7", "espedar"+u"\u00E7", "espedreg",
"espeit", "espeitor", "espelh", "espendur", "espenej", "espenic",
"espenifr", "espenuj", "esper", "esperan"+u"\u00E7", "esperanci", "esperdi"+u"\u00E7",
"esperdigot", "esperjur", "esperlux", "espermatiz", "espermeg", "esperneg",
"esperrinch", "espert", "espertin", "espesco"+u"\u00E7", "espess", "espet",
"espetaculiz", "espetal", "espetraliz", "espetrific", "espevit", "espezinh",
"espi", "espiant", "espica"+u"\u00E7", "espica"+u"\u00E7"+"o", "espich", "espicoi"+u"\u00E7",
"espicou"+u"\u00E7", "espicul", "espif", "espig", "espigait", "espiguilh",
"espildr", "espilr", "espinafr", "espin"+u"\u00E7", "espinch", "espingalh",
"espingard", "espinh", "espinic", "espinot", "espiolh", "espion",
"espip", "espipoc", "espir", "espiral", "espiraliz", "espirantiz",
"espirit", "espiritiz", "espiritualiz", "espirr", "espirrich", "esplain",
"esplendor", "esplenectomiz", "espo", "espoc", "espoj", "espojinh",
"espoldr", "espoldrinh", "espolet", "espoli", "espolin", "espolinh",
"espolp", "esponj", "espont", "esponteir", "esponto", "espor",
"esporific", "esporo", "esporr", "esporteir", "esportul", "esporul",
"espos", "espostej", "espot", "espotrej", "espotric", "espouc",
"espragat", "esprai", "espregui"+u"\u00E7", "espreit", "espremeg", "esprit",
"espulg", "espum", "espuma"+u"\u00E7", "espumarej", "espumej", "espupin",
"esput", "esquadr", "esquadra"+u"\u00E7", "esquadrej", "esquadri", "esquadrilh",
"esquadrinh", "esquadron", "esquartej", "esquartel", "esquartilh", "esquematiz",
"esquent", "esquerd", "esquerdiz", "esqui", "esqui"+u"\u00E7", "esquif",
"esquil", "esquin", "esquip", "esquirol", "esquiss", "esquistific",
"esquit", "esquiv", "estaban", "estabilit", "estabiliz", "estaboug",
"estabul", "estac", "esta"+u"\u00E7", "estacion", "estadi", "estadiz",
"estadualiz", "estaf", "estafeg", "estaferm", "estafet", "estafon",
"estagflacion", "estagi", "estagn", "estai", "estal", "estaleir",
"estalej", "estalic", "estalid", "estalon", "estambr", "estamin",
"estamp", "estampilh", "estanc", "estanci", "estandardiz",
"estanh", "estard", "estardalh", "estardalha"+u"\u00E7", "estarel", "estarrific",
"estarrinc", "estarro", "estartal", "estasi", "estassalh", "estatel",
"estatistific", "estatiz", "estatu", "estaz", "esteanaz", "esteatiz",
"esteir", "estelic", "estenaz", "estenodactilograf", "estenodatilograf", "estenofotolitograf",
"estenograf", "estenos", "estenotip", "ester", "esterc", "ester"+u"\u00E7",
"estereofotograf", "estereograf", "estereotip", "estereotomiz", "esterific", "esteriliz",
"esterlic", "esternic", "esternud", "esternut", "esterotomiz", "esterr",
"esterro", "estertor", "estertoriz", "estesi", "estesilh", "estesoir",
"estesour", "esteticiz", "estetiz", "estev", "esti", "estic",
"estigm", "estigmatiz", "estil", "estilet", "estiletiz", "estilh",
"estilha"+u"\u00E7", "estilicid", "estiliz", "estim", "estimul", "esting",
"estinh", "estiol", "estiomen", "estipendi", "estipul", "estir",
"estira"+u"\u00E7", "estiv", "estoc", "esto"+u"\u00E7", "estof", "estoir",
"estoirinh", "estoj", "estol", "estomag", "estomeg", "estoment",
"eston", "estonh", "estont", "estop", "estopet", "estoqueir",
"estorceg", "estorcic", "estordeg", "estorn", "estorneg", "estornic",
"estorrej", "estorric", "estorro", "estorteg", "estortur", "estorv",
"estorved", "estour", "estourinh", "estoutin", "estouv", "estov",
"estr", "estrab", "estrabouc", "estrabouch", "estra"+u"\u00E7"+"alh", "estracinh",
"estra"+u"\u00E7"+"o", "estrad", "estradul", "estrafeg", "estrag", "estral",
"estralej", "estramalh", "estrambalh", "estrambelh", "estramont", "estramp",
"estrampalh", "estramunh", "estran"+u"\u00E7", "estrancilh", "estrancinh", "estranfeni",
"estranfoli", "estranfolinh", "estranfoni", "estrangalh", "estrangeir", "estrangul",
"estranh", "estrapa"+u"\u00E7", "estrapilh", "estraquej", "estrategi", "estratific",
"estrav", "estravir", "estrebang", "estrebord", "estrebuch", "estrefeg",
"estreg", "estreit", "estrel", "estrelej", "estrelic", "estrelou"+u"\u00E7",
"estrem", "estremadur", "estremalh", "estremud", "estremulh", "estremunh",
"estrenoit", "estrenout", "estrep", "estrepass", "estrepit", "estress",
"estressalh", "estretal", "estri", "estrib", "estribilh", "estri"+u"\u00E7",
"estrich", "estricniniz", "estricniz", "estridul", "estrig", "estril",
"estrim", "estrinc", "estrin"+u"\u00E7", "estrinch", "estrip", "estrobiliz",
"estro"+u"\u00E7", "estrocic", "estroin", "estroix", "estromb", "estromp",
"estronc", "estron"+u"\u00E7", "estrond", "estrondej", "estronf", "estrongalh",
"estropeli", "estropi", "estrotej", "estrouvinh", "estroux", "estrovinh",
"estrufeg", "estrum", "estrumbic", "estrupid", "estrutur", "estruturaliz",
"estu", "estuc", "estuch", "estud", "estuf", "estug",
"estuliz", "estultific", "estum", "estupefic", "estupid", "estupidific",
"estupigait", "estupor", "estupr", "esturdi", "esturr", "esturric",
"esturrinh", "esvali", "esvalij", "esvar", "esvaz", "esvazi",
"esvent", "esventr", "esver"+u"\u00E7", "esverdinh", "esvid", "esvidig",
"esvinh", "esviscer", "esvoa"+u"\u00E7", "esvoej", "esvurm", "eterific",
"eteriz", "etern", "eterniz", "eticiz", "etil", "etiliz",
"etimologiz", "etiquet", "eufemiz", "eufoniz", "euforiz", "eugeniz",
"eutrofiz", "ev", "evacu", "evagin", "evangeliz", "evapor",
"evaporiz", "evapotranspir", "evaz", "eventr", "evermin", "evidenci",
"evir", "eviscer", "evit", "evoc", "evol", "evolucion",
"exabund", "exacerb", "exager", "exagit", "exal", "exal"+u"\u00E7",
"exalt", "examin", "exar", "exarticul", "exasper", "exatific",
"exaust", "exautor", "excarcer", "excardin", "excarnific", "excecion",
"excels", "excepcion", "exceptu", "excetu", "excis", "excit",
"exclam", "exclaustr", "exclausur", "excogit", "excomung", "excomunh",
"excorpor", "excrement", "excret", "excruci", "exculp", "excurs",
"excursion", "excurv", "execr", "execut", "exempl", "exemplariz",
"exemplific", "exempt", "exercit", "exerd", "exgreg", "exiguific",
"exil", "exit", "exoner", "exor", "exorbit", "exorcism",
"exorciz", "exordi", "exorn", "exort", "expatri", "expect",
"expector", "expedr", "expeitor", "experienci", "experiment", "expet",
"expetor", "expi", "expil", "expir", "explan", "explant",
"explic", "explicit", "explor", "explot", "exponenci", "export",
"expostul", "express", "exprob", "exprobr", "expropri", "expugn",
"expuls", "expurg", "exsic", "exst", "exsu", "exsud",
"extasi", "extenu", "exterioriz", "extermin", "extern", "externaliz",
"externiz", "extirp", "extrabord", "extract", "extradicion", "extradit",
"extrafoli", "extrapass", "extrapol", "extrat", "extravaganci", "extravas",
"extravi", "extrem", "extric", "extrofi", "extub", "exuber",
"exul", "exulcer", "exult", "exum", "exund", "exurbaniz",
"exventr", "fabric", "fabul", "fabuliz", "faccion", "faceci",
"faceir", "facej", "facet", "fach", "fachol", "facilit",
"facion", "factor", "facult", "facundi", "fad", "fadej",
"fadig", "fadist", "fadistocratiz", "fag", "fagulh", "fai",
"fain", "fair", "faix", "fajut", "fal", "fala"+u"\u00E7",
"falaci", "falad", "faladr", "falaz", "falc", "falca"+u"\u00E7",
"falcatru", "falco", "falej", "falh", "falhoc", "faloc",
"falporri", "falquej", "fals", "falsific", "falt", "faluc",
"fambiss", "familiariz", "familot", "famul", "fan", "fanatiz",
"fandang", "fanding", "fanf", "fanfarr", "fanfarron", "fanfurri",
"fangan", "fangarel", "fanic", "fanich", "fantasi", "fantasm",
"fantasmagoriz", "fantasmiz", "fantoch", "fantochin", "fanuc", "faquiriz",
"far", "faradiz", "farandol", "fard", "farej", "farel",
"faretr", "farfalh", "farfalhej", "farfalin", "farin", "faringaliz",
"farinh", "farisc", "farofi", "farol", "faroliz", "farp",
"farpo", "farr", "farrap", "farrasc", "farronc", "farsol",
"fart", "farung", "fascin", "fascistiz", "fasqui", "fast",
"fastigi", "fataliz", "fatan", "fatej", "fati", "fatig",
"fator", "fatoriz", "fatuej", "fatur", "faul", "faulh",
"fautoriz", "fav", "faveliz", "favoni", "favorit", "favoriz",
"faxin", "febricit", "febriliz", "fech", "fecund", "fecundiz",
"fedanh", "fedelh", "feder", "federaliz", "fedinch", "fedonh",
"fefet", "feir", "feit", "feiti", "feitor", "feitori",
"feitoriz", "fel", "feldspatiz", "felicit", "feloni", "felp",
"feltr", "femb", "femen"+u"\u00E7", "feminiliz", "feminiz", "fen",
"fendilh", "fendrelh", "fendrilh", "fenestr", "fenoliz", "fenomenaliz",
"feri", "ferment", "ferr", "ferrament", "ferrej", "ferret",
"ferreto", "ferro", "ferrob", "ferrolh", "fertiliz", "fervelh",
"fervent", "fervilh", "fervor", "fescoliz", "fest", "festej",
"festin", "festo", "feston", "feudaliz", "fi", "fiamp",
"fibril", "fic", "fich", "fideliz", "fig", "figur",
"fil", "filagran", "filet", "filh", "filhar", "filhastr",
"fili", "filigran", "filip", "filipiz", "filistri", "film",
"filmograf", "filoidin", "filosof", "filtr", "fimbri", "fin",
"finaliz", "financi", "finc", "find", "findaliz", "finf",
"fint", "firm", "firr", "fiscaliz", "fiscel", "fisg",
"fisiologiz", "fissiliz", "fission", "fissur", "fistul", "fistuliz",
"fit", "fituc", "fix", "flabel", "flafl", "flagel",
"flagici", "flagr", "flain", "flam", "flamb", "flamej",
"flamispir", "flamul", "flan", "flart", "flaut", "flavortiz",
"flebotomiz", "flech", "flegmasi", "flert", "flexibiliz", "flexion",
"flexu", "flim", "flip", "flirt", "flit", "floc",
"flocon", "flocul", "flogistic", "flogos", "flor", "florej",
"florest", "flostri", "flot", "fluid", "fluidific", "fluidiz",
"fluor", "flutu", "fob", "fobit", "foc", "fo"+u"\u00E7",
"focaliz", "focinh", "fof", "fofoc", "fofuenh", "fogach",
"foi"+u"\u00E7", "foicinh", "folcloriz", "folec", "foleg", "folerp",
"folestri", "folg", "folgaz", "folh", "folhec", "folhetiniz",
"foli", "foment", "fonetiz", "fonfon", "fonocardiograf", "fonocinematograf",
"fonofilm", "fonograf", "fonologiz", "forc", "for"+u"\u00E7", "forcej",
"forj", "forjic", "form", "formaliz", "format", "formic",
"formicid", "formig", "formiguej", "formilh", "formol", "formoliz",
"formosent", "formul", "formulariz", "fornej", "fornic", "fornig",
"fornizi", "forquilh", "forr", "forraj", "forrej", "forric",
"forta"+u"\u00E7", "fortaleg", "fortalez", "fortific", "fortun", "fosc",
"fosfat", "fosfor", "fosforej", "fosforil", "fosforit", "fosforiz",
"foss", "fossang", "fossilific", "fossiliz", "fotoativ", "fotocopi",
"fotodissoci", "fotofosforiliz", "fotograf", "fotograv", "fotointerpret", "fotolit",
"fotolitograf", "fotometr", "fotominiaturiz", "fotomont", "fotomultiplic", "fotosforil",
"fotossensibiliz", "fotossintetiz", "fototip", "fototipi", "fou"+u"\u00E7", "foucinh",
"frac", "fracass", "fraccion", "fracion", "fractur", "frad",
"fradej", "fragiliz", "fragment", "fragor", "frald", "fraldej",
"fraldic", "frama"+u"\u00E7"+"oniz", "frances", "francesi", "franciuz", "francofoniz",
"frangalh", "franj", "franqui", "fraquej", "fraquent", "frasquej",
"frasquinh", "fraterniz", "fratur", "fraud", "fraut", "frech",
"freg", "freir", "frejuc", "fren", "frenesi", "frequent",
"fres", "fresc", "fresquej", "fret", "fretej", "frev",
"fricatiz", "friccion", "frigorific", "frinch", "fris", "frit",
"froc", "froix", "frol", "frond", "frondej", "front",
"fronteir", "froux", "frufrulh", "frufrut", "frulh", "frustr",
"frut", "fruteg", "frutific", "fubec", "fu"+u"\u00E7", "fueir",
"fug", "fujic", "fulaniz", "fulec", "fulgent", "fulgur",
"fulguraresfuzi", "fulmin", "fulo", "fum", "fuma"+u"\u00E7", "fumar",
"fumeg", "fumig", "fun", "funambul", "funcion", "funcionaliz",
"fund", "fundalh", "fundament", "fundilh", "funest", "funf",
"funfi", "fung", "fur", "furac", "furdun"+u"\u00E7", "furfunh",
"furifun", "furo", "furri", "furt", "furtic", "furunf",
"furung", "fus", "fusc", "fusg", "fusion", "fustig",
"futic", "futiliz", "futret", "futric", "futuc", "futur",
"futuriz", "fuxic", "fuzil", "gab", "gabarit", "gabarol",
"gabi", "gabion", "gadanh", "gadi"+u"\u00E7", "gadunh", "gaf",
"gafeir", "gafej", "gaguej", "gai", "gaiat", "gaifon",
"gaimo", "gait", "gaiv", "gaj", "gal", "galan",
"galantiz", "galardi", "galardo", "galderi", "galdrop", "galeguiz",
"galg", "galhof", "galic", "galiciparl", "galicism", "galiciz",
"galim", "galimatiz", "galinh", "galiv", "galo", "galomaniz",
"galon", "galop", "galopin", "galr", "galrej", "galrich",
"galroch", "galvaniz", "galvanotip", "gam", "gambel", "gambi",
"gamenh", "gananci", "gan"+u"\u00E7", "ganch", "gandai", "gandul",
"ganf", "gangorr", "gangren", "ganguiss", "ganh", "gani"+u"\u00E7",
"ganiz", "ganz", "gapin", "gapui", "garabulh", "garalh",
"garan"+u"\u00E7", "garatuj", "garavet", "garf", "garfej", "garfi",
"garg", "gargada"+u"\u00E7", "gargala"+u"\u00E7", "gargalej", "gargalh", "gargantu",
"gargarej", "gargolej", "gargorej", "garguel", "garimb", "garimp",
"garin", "garn", "garo", "garot", "garr", "garram",
"garranch", "garrid", "garroch", "garrot", "garrotearestrangul", "garruch",
"garrul", "garrulej", "garu", "gasalh", "gasconiz", "gaseific",
"gasguit", "gasific", "gasm", "gast", "gastrofotograf", "gastrul",
"gat", "gatafunh", "gatanh", "gatinh", "gatun", "gatunh",
"gatur", "gaturr", "gaud", "gauder", "gauderi", "gaudi",
"gaudin", "gav", "gavion", "gazil", "gazilh", "gazin",
"gazofil", "gazol", "gazu", "gazul", "gazup", "geb",
"gebr", "gegadelh", "gel", "gelatin", "gelatinific", "gelatiniz",
"geleific", "gelific", "gem", "gemelg", "gemelhic", "gemi",
"gemic", "gemin", "gemul", "generaliz", "gentiliz", "geocentr",
"geograf", "geometriz", "georreferenci", "ger", "gerenci", "german",
"germaniz", "germin", "gess", "gest", "gesticul", "gestion",
"gestualiz", "gigantiz", "ginastic", "ging", "gingr", "gir",
"giraldin", "girandol", "girassol", "girogir", "girovag", "giz",
"gla"+u"\u00E7", "gladi", "glamoriz", "glanduliz", "glicerin", "glicuroniz",
"gliss", "globaliz", "globuliz", "glomer", "glori", "glorific",
"glos", "gloss", "gloter", "glotiz", "glotor", "glugluej",
"gluglurej", "gluglut", "glutin", "godemiz", "goder", "godi",
"goel", "gofr", "goi", "goit", "goiv", "gol",
"golej", "golelh", "golf", "golfej", "golip", "golp",
"gom", "gombel", "gomific", "gomit", "gomo", "gon",
"gondol", "gong", "gongor", "gongoriz", "gor", "gordur",
"gorgol", "gorgolej", "gorgolh", "gorgorej", "gorr", "gosm",
"gosmilh", "gost", "gotej", "goticiz", "goug", "govern",
"governamentaliz", "governich", "goz", "gracej", "gracit", "gra"+u"\u00E7"+"ol",
"grad", "gradearesterro", "gradej", "gradin", "gradu", "graf",
"grafil", "grafit", "grafitiz", "grai", "gralh", "gram",
"gramatic", "gramaticaliz", "gramaticiz", "graminh", "gramofon", "gramp",
"gran", "granid", "granit", "granitific", "graniz", "granul",
"granzin", "grasn", "grass", "grassit", "gratific", "gratin",
"gratul", "grav", "gravet", "gravid", "gravit", "gravunh",
"gravur", "grazin", "greciz", "greguej", "grel", "grelh",
"gret", "grif", "gril", "grilhet", "grimp", "grinald",
"grinf", "grip", "gris", "grisalh", "grit", "griv",
"grogolej", "grol", "grom", "grong", "gros", "grosm",
"grossul", "grud", "grugrulej", "grugrulh", "grugulej", "grugunz",
"grulh", "grum", "grup", "gruzi", "guaguax", "guai",
"gualdrip", "guard", "guarit", "guasc", "guaz", "guedelh",
"gueng", "guerram", "guerrilh", "gueviss", "gugluej", "gui",
"guid", "guilhoch", "guilhotin", "guiloch", "guin", "guinch",
"guind", "guinh", "guirland", "guis", "guiz", "guizalh",
"gulaim", "gulap", "gulaz", "guleim", "gulos", "guloseim",
"gulosin", "gunf", "gungun", "gurgit", "gust", "gut",
"guturaliz", "habilit", "habit", "habitu", "hachur", "halit",
"halogen", "halogenoderiv", "haragan", "hariol", "harmoniz", "harp",
"harpej", "harpeji", "hastilh", "hausmaniz", "haussemaniz", "haussmanniz",
"hebet", "hebetiz", "hectometr", "hegemoniz", "heleboriz", "heleniz",
"heliport", "helitransport", "hemaglutin", "hematiz", "hematos", "hemoaglutin",
"hemoconcentr", "hemodialis", "hemolis", "hemossediment", "hepatectomiz", "hepatiz",
"herboriz", "herd", "heresi", "hermetific", "herni", "heroiciz",
"heroific", "heroiz", "hesit", "het", "heteroaglutin", "heterocarreg",
"heterocromatiz", "heterogeneiz", "heteroimuniz", "heteropolimeriz", "heterossugestion", "heterotransplant",
"hialiniz", "hiatiz", "hibern", "hibrid", "hibridiz", "hidrat",
"hidrobor", "hidrocarbon", "hidrocut", "hidrodessulfuriz", "hidroextrat", "hidrofobiz",
"hidrogen", "hidrolis", "hidrometr", "hidroport", "hidrotomiz", "hidroxideriv",
"hiemaliz", "hierarquiz", "hieratiz", "hifeniz", "higieniz", "hilari",
"hilariz", "hiperadapt", "hiperafirm", "hiperajust", "hiperaliment", "hiperaplic",
"hiperaprimor", "hiperativ", "hiperbenefici", "hiperboliz", "hipercapacit", "hipercaricatur",
"hipercomplex", "hiperdimension", "hiperdos", "hiperemi", "hiperequip", "hiperestesi",
"hiperestrutur", "hiperexager", "hiperexplor", "hiperglicemi", "hiperimpregn", "hiperimuniz",
"hiperindustri", "hiperindustrializ", "hiperinfeccion", "hiperinflacion", "hiperirrit", "hiperliterariz",
"hiperocup", "hiperofert", "hiperorden", "hiperorganiz", "hiperotimiz", "hiperoxid",
"hiperoxigen", "hiperpigment", "hiperpolariz", "hipersemantiz", "hipersensibiliz", "hipertrofi",
"hiperurbaniz", "hipervaloriz", "hipervasculariz", "hiperventil", "hipnotiz", "hipocratiz",
"hipofisectomiz", "hipoglicemi", "hipossensibiliz", "hipostasi", "hiposteniz", "hipotec",
"hipotrofi", "hipoxi", "hirt", "hirudiniz", "hisp", "hispanific",
"hispaniz", "hispid", "hissop", "histeriz", "histocompatibiliz", "histodiferenci",
"histori", "historializ", "historiciz", "historiz", "histrion", "holandiz",
"holocaust", "holofotiz", "holograf", "homeotransplant", "homicid", "homili",
"hominiz", "homizi", "homogeniz", "homolog", "homopolimeriz", "homostimul",
"homotransplant", "honest", "honestiz", "honor", "honorific", "honr",
"honrific", "hor", "horej", "horizont", "horizontaliz", "hormoniz",
"horoscop", "horoscopiz", "horrific", "horripil", "horroriz", "hort",
"hortej", "horticult", "hosped", "hospitaliz", "hostiliz", "hulhific",
"human", "humaniz", "humet", "humific", "humild", "humilh",
"humoriz", "ianquiz", "iarovis", "iberiz", "i"+u"\u00E7", "ichacorv",
"iconiz", "icterici", "idealiz", "identific", "ideograf", "ideologiz",
"idiliz", "idiot", "idiotific", "idiotiz", "idolatr", "idolatriz",
"ignific", "ignifug", "igniz", "ignobiliz", "ignomini", "ignor",
"igual", "iguald", "igualh", "igualiz", "il", "ilegaliz",
"ilegitim", "ilh", "ilib", "ili"+u"\u00E7", "ilig", "ilimit",
"ilumin", "ilusion", "ilustr", "ilut", "iluvi", "imagin",
"iman", "imanentiz", "imaniz", "imant", "imantiz", "imaterializ",
"imbecilit", "imbeciliz", "imbelic", "imbelinh", "imbic",
"imbond", "imbram", "imbric", "imedi", "imediat", "imers",
"imigr", "imit", "imobiliz", "imoir", "imol", "imortaliz",
"imot", "imour", "imp", "impacient", "impact", "impalud",
"impan", "imparcializ", "impassibiliz", "impelg", "imper", "imperfei"+u"\u00E7"+"o",
"imperializ", "imperl", "impermeabiliz", "impersonaliz", "impersonific", "impertinenci",
"impessoaliz", "impet", "impetilh", "impetr", "impinim", "implant",
"implement", "implic", "implor", "impont", "impopulariz", "import",
"importun", "impossibilit", "impost", "impostur", "imprec", "imprecat",
"imprecis", "impregn", "impremedit", "imprens", "impression", "imprestabiliz",
"imprim", "improb", "impront", "impronunci", "improper", "improporcion",
"impropri", "improv", "improvis", "impugn", "impuls", "impulsion",
"impulsiv", "impurific", "imput", "imund", "imuniz", "imut",
"imutabiliz", "inabilit", "inabit", "inadapt", "inadat", "inadestr",
"inadun", "inal", "inalien", "inalter", "inambul", "inanim",
"inativ", "inaugur", "in"+u"\u00E7", "incamer", "incapacit", "incapsul",
"incaracteriz", "incardin", "incarn", "incens", "incentiv", "incert",
"incest", "incestu", "inch", "inci", "incident", "inciner",
"incis", "incit", "inclin", "inco", "incolor", "incomod",
"incompatibiliz", "incomunic", "inconform", "inconfort", "inconsolid", "inconst",
"inconstitucionaliz", "incontest", "incorpor", "incrass", "increment", "increp",
"incrimin", "incris", "incruent", "incrust", "incub", "inculc",
"inculp", "incurric", "incursion", "indag", "indegr", "indelimit",
"indemniz", "indeniz", "independentiz", "independiz", "indetect", "indetermin",
"index", "indianiz", "indic", "indici", "indicific", "indigeniz",
"indigest", "indigit", "indign", "indignific", "indilg", "indinh",
"indisciplin", "indiscrimin", "individu", "individualiz", "indociliz", "indocument",
"indolent", "indoutrin", "indromin", "indulgenci", "indult", "indument",
"industri", "industrializ", "indut", "inebri", "ineptiz", "inequacion",
"inerci", "inertiz", "inerv", "inespoli", "infam", "infantiliz",
"infart", "infeccion", "infecion", "infect", "infecund", "infelicit",
"inferioriz", "infern", "inferniz", "infertiliz", "infest", "infet",
"infibul", "inficion", "infiltr", "infinc", "infirm", "infix",
"infl", "inflacion", "inflam", "inflexion", "inflor", "influenci",
"infon", "inform", "informaliz", "informatiz", "infortun", "infrond",
"infuleim", "infundament", "infunic", "infus", "ingenu", "ingles",
"ingress", "ingund", "ingurgit", "inici", "inicializ", "inigual",
"inimist", "inimit", "inimiz", "injeriz", "injet", "injun"+u"\u00E7",
"injuri", "injusti"+u"\u00E7", "injustific", "inlernaliz", "inli"+u"\u00E7", "inobserv",
"inocent", "inocul", "inor", "inoscul", "inov", "inquart",
"inquezil", "inquiet", "inquilin", "inquin", "inquizil", "insacul",
"insaliv", "insatur", "insemin", "insensibiliz", "insert", "insidi",
"insigni", "insimul", "insinu", "insipid", "insol", "insolubiliz",
"insoni", "insonoriz", "insoss", "inspecion", "inspet", "inspir",
"inspiss", "inst", "instabiliz", "instal", "instanci", "instantaniz",
"instaur", "instig", "instil", "institucionaliz", "instrument", "instrumentaliz",
"insubordin", "insufl", "insul", "insult", "insum", "insurreccion",
"insurrecion", "integr", "integraliz", "inteig", "inteir", "inteiri"+u"\u00E7",
"inteleccion", "intelecion", "intelectualiz", "inteligenci", "intemper", "intemperiz",
"intencion", "intens", "intensific", "intensiv", "intent", "interatu",
"intercal", "intercambi", "intercept", "intercomunic", "interconect", "intercruz",
"interdigit", "interdimension", "interdit", "interespacej", "interess", "interesterific",
"interestratific", "interfoli", "interfon", "interi"+u"\u00E7", "interin", "interinfluenci",
"interioriz", "interjeccion", "interjecion", "interjectiv", "interjeicion", "interjetiv",
"interla"+u"\u00E7", "interlig", "interm", "intermigr", "intermistur", "intermostr",
"intern", "internacionaliz", "internaliz", "interoper", "interpel", "interpenetr",
"interpic", "interpol", "interpolimeriz", "interpres", "interpret", "interrog",
"intersach", "intersec", "interseccion", "intersect", "interset", "intertransform",
"intertrav", "interturb", "interval", "intervencion", "intestel", "intestin",
"intic", "intim", "intimid", "intitul", "inton", "intoxic",
"intranquiliz", "intransitiv", "intrassel", "intrech", "intric", "intrigalh",
"intrinc", "introject", "introjet", "intrors", "intruid", "intruj",
"intub", "intuicion", "inum", "inund", "inutiliz", "invagin",
"invalid", "invari", "invas", "invect", "invectiv", "invej",
"invencion", "invent", "inventari", "invern", "invespr", "investig",
"invet", "inveter", "invetiv", "inviabiliz", "inviccion", "invid",
"invidi", "inviolabiliz", "inviper", "inviscer", "invisibiliz", "invit",
"invoc", "involucr", "invulg", "inzon", "iob", "iod",
"iodet", "iodin", "ioniz", "iontiz", "iotiz", "ir",
"iri", "iridi", "iris", "iriz", "irman", "ironiz",
"irracionaliz", "irradi", "irrealiz", "irreconcili", "irresponsabiliz", "irreverenci",
"irrig", "irrit", "irrog", "irror", "isc", "isent",
"islamiz", "isoacentu", "isoaglutin", "isocroniz", "isol", "isomeriz",
"israeliz", "italianiz", "italiciz", "itemiz", "iter", "jacobiniz",
"jact", "jactit", "jacul", "jaez", "jan", "janel",
"jang", "janot", "jant", "jantaric", "jantarinh", "japones",
"japoniz", "Japonv", "jard", "jardin", "jaroviz", "jarret",
"jarund", "jat", "jatanci", "jatit", "javeliz", "javr",
"jazzific", "jeit", "jeju", "jerarquiz", "jeremi", "jeremin",
"jesuitiz", "jimbr", "job", "joeir", "joelh", "jog",
"jogalh", "jogat", "jogatilh", "jogral", "jograliz", "joguet",
"jol", "jong", "jonj", "jornalistic", "jornaliz", "jorr",
"jovializ", "jubil", "judi", "judic", "judici", "judicializ",
"jug", "jugad", "jugul", "julg", "junc", "junt",
"jur", "jurament", "jurem", "jurg", "juridiciz", "jurisdicion",
"jurisdicionaliz", "just", "justi"+u"\u00E7", "justific", "juveniliz", "labanot",
"labefact", "label", "labializ", "labiovelariz", "labirint", "labor",
"labreg", "labrest", "labut", "labuz", "lac", "la"+u"\u00E7",
"lacai", "lacer", "laconiz", "lacr", "lacram", "lacran",
"lacrim", "lacrimej", "lact", "lactoniz", "lacun", "ladeir",
"ladr", "ladrej", "ladrilh", "ladrinch", "ladrinh", "ladrip",
"ladro", "ladroeir", "lagrim", "lagrimej", "laicific", "laiciz",
"laiv", "laj", "lal", "lamb", "lambar", "lambarej",
"lambarisc", "lambaz", "lambic", "lambisc", "lambit", "lambrec",
"lambreg", "lambrequinej", "lambris", "lambriz", "lambu"+u"\u00E7", "lambuj",
"lambuz", "lamech", "lameir", "lamel", "lament", "lamin",
"lamp", "lampadej", "lamparej", "lampej", "lampi", "lamuri",
"lamurinh", "lan", "lan"+u"\u00E7", "lanceol", "lancet", "lanch",
"lancin", "lancissond", "langu"+u"\u00E7", "lanh", "lantejoil", "lantejoul",
"lantern", "lanzo", "lap", "laparotomiz", "lapias", "lapid",
"lapidific", "lapij", "lapis", "lapislazul", "larach", "larapi",
"larapin", "larg", "laringograf", "larp", "larunj", "larv",
"larvej", "lasc", "lass", "lastim", "lastr", "latanh",
"lateg", "latej", "lateraliz", "latic", "latin", "lating",
"latiniz", "latitudin", "latrocin", "laudaniz", "lav", "lavaj",
"lavarej", "lavarisc", "lavasc", "laveg", "lavoir", "lavor",
"lavour", "lavr", "lavuj", "lax", "lazar", "lazeir",
"lazer", "leald", "leccion", "lecion", "leg", "legaliz",
"legend", "legifer", "legisl", "legitim", "legr", "leiaut",
"leig", "leilo", "leir", "leit", "leix", "lematiz",
"lemb", "lembr", "leng", "lengaleng", "lenh", "lenhific",
"lenific", "leniment", "lent", "lentej", "lentejoil", "lentejoul",
"leri", "les", "lesion", "lesm", "lest", "letargi",
"letific", "letr", "letriz", "lev", "levant", "leved",
"levi", "levig", "levit", "lexicaliz", "lexicograf", "li",
"lib", "libel", "liber", "liberaliz", "libert", "libr",
"librin", "licenci", "licit", "lid", "lidair", "lider",
"lidim", "lig", "lignific", "ligul", "lilliputianiz", "lim",
"limit", "limp", "linc", "linch", "lind", "lineariz",
"linf", "linfangiograf", "linfatiz", "linfoangiograf", "linfolis", "ling",
"lingot", "linguaj", "linguar", "linguarej", "linific", "liniment",
"linotip", "liofiliz", "lipoaspir", "lipolis", "liquenific", "liquid",
"liquidific", "liric", "liriciz", "lisboetiz", "lisc", "lism",
"lisogeniz", "lisonj", "list", "listr", "lit", "literariz",
"literat", "literatej", "literatiz", "literatur", "litic", "litific",
"litig", "litigi", "litograf", "livel", "livid", "livr",
"lix", "lixivi", "liz", "lo", "lobol", "lobotomiz",
"lobreg", "lobreguej", "lobrig", "loc", "localiz", "locion",
"locucion", "locuplet", "logic", "logiciz", "logorr",
"logr", "loir", "loirej", "lois", "lombaliz", "lombariz",
"lombi", "lombilh", "lombrig", "lorc", "lord", "lorg",
"lorot", "losangol", "lot", "lotej", "lou"+u"\u00E7"+"ainh", "louquej",
"lour", "lourej", "lous", "louv", "louvaminh", "luademel",
"luarej", "luariz", "lubric", "lubrific", "lucej", "lucern",
"luch", "lucid", "lucifer", "lucil", "luciol", "lucr",
"lucubr", "ludibri", "luf", "lug", "lumi", "lumpesin",
"lungul", "lunul", "lup", "lupan", "lupuliz", "lur",
"lusc", "luscofusc", "lusific", "lusitaniz", "lustr", "lut",
"luteiniz", "lux", "luxuri", "ma"+u"\u00E7", "macac", "macadamiz",
"macambuzi", "ma"+u"\u00E7"+"anet", "ma"+u"\u00E7"+"aroc", "macarron", "macavenc", "macer",
"macet", "machad", "machi", "machic", "machin", "machuc",
"maci", "maconh", "ma"+u"\u00E7"+"oniz", "macrodimension", "macroequip", "macrograf",
"macrossegment", "macuc", "macul", "macumb", "madefic", "madeir",
"madeiriz", "madeix", "madgi", "madoniz", "madorn", "madorr",
"madra"+u"\u00E7", "madraceir", "madrigaliz", "madrinh", "madrug", "madur",
"mafi", "magi", "magic", "magin", "magnetiz", "magnific",
"mago", "magulh", "mai", "maiusculiz", "major", "mal",
"malabariz", "malacachet", "malandr", "malax", "malbarat", "malconfi",
"mald", "maldi"+u"\u00E7"+"o", "maleabiliz", "malefici", "maleiniz", "malfad",
"malform", "malgast", "malgovern", "malh", "malhet", "malici",
"malign", "maligniz", "malin", "malingu", "maloc", "malogr",
"malpar", "malquist", "malsegur", "malsin", "malt", "maltos",
"maltrabalh", "maltrat", "maluc", "malvers", "malvez", "mam",
"mamaen", "mambemb", "mamp", "mamparr", "mamuj", "man",
"manc", "manch", "mancip", "mancolit", "mancomun", "mancorn",
"mand", "manding", "mandjol", "mandri", "mandrian", "mandril",
"mandrion", "manduc", "manduqui", "maneir", "manej", "mang",
"mangab", "mangol", "mangon", "mangonh", "mangr", "manguinh",
"manheir", "manhent", "manhos", "maniat", "maniet", "manifest",
"maniganci", "manilh", "maninh", "manipul", "manisfest",
"manivel", "manj", "manjeric", "manobr", "manoc", "manquej",
"manquit", "manquitol", "manualiz", "manufact", "manufactur", "manufat",
"manufatur", "manzanz", "maometaniz", "mapi", "maqui", "maquiaveliz",
"maquil", "maquilh", "maquin", "maquiniz", "mar", "maraf",
"marambai", "marangol", "maranh", "marasm", "maratiz", "maravilh",
"marc", "marcej", "marceneir", "march", "marchet", "marcializ",
"marechaliz", "marej", "marf", "marfiniz", "marfolh", "marg",
"margin", "marginaliz", "marid", "marimb", "marin", "marinh",
"marinheir", "mariol", "maripos", "marisc", "marlot", "marm",
"marmel", "marmoriz", "maromb", "marques", "marr", "marralh",
"marranch", "marret", "marric", "marro", "marroquin", "marruc",
"marteir", "martel", "martelej", "martilh", "martiriz", "maruj",
"marulh", "masc", "mascab", "mascar", "mascarr", "mascav",
"mascot", "masculiniz", "mass", "massacr", "massaj", "massific",
"masteriz", "mastig", "mastuc", "masturb", "mat", "matabich",
"matalot", "matej", "matematiz", "materializ", "matic", "matilh",
"matin", "matiz", "matrac", "matracolej", "matraquej", "matreir",
"matri", "matricul", "matrimoni", "matriz", "matruc", "matur",
"maturrang", "matut", "maximaliz", "maximiz", "maxix", "mazanz",
"mazel", "mazurc", "mealh", "meandr", "mecaniz", "mecanograf",
"mech", "medalh", "medianiz", "mediatiz", "medic", "medicament",
"medicin", "medievaliz", "mediocriz", "medit", "medr", "mefitiz",
"meig", "meirinh", "mel", "melancoliz", "melaniz", "melhor",
"melific", "melifluent", "melindr", "melingr", "melodi", "melodiz",
"melodram", "melodramatiz", "melotipograf", "memor", "memori", "memoriz",
"men", "mencion", "mendic", "mendig", "meng", "menoriz",
"menoscab", "menospre"+u"\u00E7", "menospreci", "menosprez", "mensaliz", "menstru",
"mensur", "ment", "mentaliz", "mentir", "mentirol", "merc",
"mercadej", "mercadiz", "mercanci", "mercantiliz", "merceriz", "mercurializ",
"mercurific", "merej", "merend", "merenduc", "meretrici", "merg",
"mergulh", "merm", "meruj", "merunh", "mesc", "mescab",
"mescl", "mesmeriz", "mesquinh", "mess", "messianiz", "mesti"+u"\u00E7",
"mesticiz", "mesur", "metaboliz", "metacomunic", "metaconglomer", "metafisic",
"metaforiz", "metagraf", "metalific", "metaliz", "metameriz", "metamorfiz",
"metassediment", "metast", "metatetiz", "metatiz", "meteoriz", "metil",
"metodific", "metodiz", "metr", "metralh", "metrific", "metrocistograf",
"metronomiz", "metropoliz", "mexelh", "mexeric", "mexeruc", "mexilh",
"mezinh", "mi", "miasm", "mich", "microbi", "microcinematograf",
"microcircul", "microcoagul", "microcopi", "microdimension", "microequip", "microfilm",
"microfossiliz", "microfotograf", "micrograf", "microgranul", "micrograv", "micromanipul",
"micrometr", "micropotencializ", "microprocess", "microscop", "microsserrilh", "mict",
"midiatiz", "mieliniz", "mig", "migalh", "migmatiz", "migr",
"mij", "milici", "milimetr", "milionariz", "milit", "militariz",
"milonitiz", "mim", "mimeograf", "mimetiz", "mimic", "min",
"miner", "mineraliz", "ming", "minhoc", "mini", "miniatur",
"miniaturiz", "minific", "minimaliz", "miniminiz", "minimiz", "ministr",
"ministrific", "minor", "minoriz", "minotaur", "minotauriz", "minuci",
"minud", "minudenci", "minuscul", "minusculiz", "minut", "mir",
"miracul", "mirific", "miron", "mirr", "misc", "miscigen",
"miscr", "miser", "miserabiliz", "misgalh", "miss", "mission",
"misticiz", "mistific", "mistur", "mitific", "mitig", "mitiz",
"mitr", "mitridatiz", "mix", "mixorof", "mnemoniz", "mobil",
"mobilh", "mobiliz", "moc", "mo"+u"\u00E7", "moch", "modaliz",
"model", "modeliz", "moder", "modern", "moderniz", "modic",
"modific", "modilh", "modorn", "modorr", "modul", "modulariz",
"moest", "mof", "mofumb", "mofund", "mogang", "mogic",
"moin", "moinh", "moir", "moirej", "moiriz", "moit",
"moj", "mojic", "mold", "moldur", "molec", "moleculariz",
"molej", "moleng", "molest", "molg", "molh", "molific",
"molin", "molinh", "molism", "mom", "monaquiz", "monarqui",
"monarquiz", "monc", "mond", "monetiz", "mongoliz", "monidrat",
"monilh", "monitor", "monitoriz", "monoclor", "monoculiz", "monodi",
"monograf", "monogram", "monolocul", "monolog", "monopoliz", "monossilab",
"monotip", "monotiz", "monotong", "monotoniz", "mont", "monumentaliz",
"moquec", "mor", "moraliz", "morang", "morbidiz", "morceg",
"morchet", "morda"+u"\u00E7", "mordic", "mordisc", "mordom", "mordomiz",
"morej", "morf", "morfil", "morfin", "morfiniz", "morfodiferenci",
"morg", "moriger", "morn", "morra"+u"\u00E7", "morrinh", "morseg",
"mortalh", "mortaliz", "mortific", "mosc", "mosquit", "moss",
"mossec", "mosseg", "mostr", "mostreng", "motej", "motet",
"motin", "motiv", "motomecaniz", "motoriz", "mouc", "mouf",
"moum", "mour", "mourej", "mourisc", "mouriz", "moviment",
"mox", "moxam", "muchic", "mu"+u"\u00E7"+"ulmaniz", "mud", "muem",
"muen", "mufic", "mufumb", "mugang", "mujimb", "mujin",
"mulad", "mult", "multangul", "multiangul", "multiarticul", "multicelul",
"multicit", "multigraf", "multinund", "multiplex", "multiplic", "multiprocess",
"multivari", "multivibr", "mumific", "mumiz", "mundan", "mundaniz",
"mundi", "mundializ", "mundific", "mungang", "munhec", "munici",
"municion", "municipaliz", "mupic", "mupuc", "muquinh", "mur",
"muralh", "murch", "murjin", "murmulh", "murmur", "murmurej",
"murmuri", "murmurinh", "muruj", "musealiz", "musg", "music",
"musicaliz", "mussit", "mutarrot", "mutil", "mutr", "mutreit",
"mutret", "mutu", "muxic", "muxim", "muxox", "muxurund",
"nacar", "nacariz", "nacionaliz", "nad", "nadific", "nag",
"naip", "namor", "namoric", "namorich", "namorisc", "nan",
"narcis", "narcotiz", "narr", "nasal", "nasaliz", "nastr",
"natur", "naturaliz", "naufrag", "navalh", "naveg", "navel",
"nazific", "neblin", "nebrin", "nebuliz", "necessit", "necrologi",
"necropsi", "necros", "necrotiz", "nectariz", "nefanalis", "nefeliniz",
"neg", "nega"+u"\u00E7", "negativ", "negativiz", "neglicenci", "negligenci",
"negoci", "negraliz", "negrej", "negrit", "nenec", "neocoloniz",
"neoform", "neologism", "neologiz", "nervosiz", "nesg", "neurasteniz",
"neuros", "neurotiz", "neutraliz", "nev", "nevisc", "nevo",
"nevoa"+u"\u00E7", "nevoent", "nic", "nich", "nicotin", "nicotiz",
"nict", "nid", "nidific", "nigel", "niilific", "nili",
"nimb", "nin", "ninain", "ninh", "niponiz", "niquel",
"niqueliz", "niquil", "nirvan", "nirvaniz", "nitidific", "nitidiz",
"nitr", "nitragin", "nitrat", "nitret", "nitrific", "nitroderiv",
"nitros", "nitrur", "nivel", "nobeliz", "nobilific", "nobilit",
"nobiliz", "noc", "noctambul", "noctivag", "nod", "nodo",
"noiv", "noj", "nomadiz", "nomencl", "nomenclatur", "nomin",
"nominaliz", "normaliz", "normatiz", "nosograf", "nostalgiz", "not",
"notabiliz", "notambul", "notici", "notific", "noturniz", "nov",
"novel", "novici", "nuan"+u"\u00E7", "nubel", "nubl", "nucleariz",
"nulific", "numer", "nunci", "nupci", "nut", "nutrific",
"nuverinh", "obcec", "obdur", "ober", "obfirm", "objeccion",
"objecion", "objectific", "objectiv", "objet", "objetific", "objetiv",
"objurg", "oblat", "obliter", "obnubil", "obnunci", "obr",
"obrad", "obrej", "obrig", "obsceniz", "obscurantiz", "obsecr",
"obsed", "obsedi", "obsequi", "observ", "obsession", "obsidi",
"obsolet", "obst", "obstacul", "obstaculiz", "obstin", "obstinarporfi",
"obstip", "obtemper", "obtest", "obtur", "obumbr", "obvi",
"oc", "ocasion", "ocidentaliz", "octan", "octangul", "octuplic",
"ocult", "ocup", "ocurs", "odor", "odoriz", "ofeg",
"oferend", "ofert", "ofici", "oficializ", "ofusc", "og",
"ogiv", "oig", "oir", "oirej", "oiri"+u"\u00E7", "oitav",
"ojeriz", "olead", "oleagin", "olfact", "olfat", "olh",
"oligarquiz", "oliv", "olivel", "oloriz", "olvid", "omin",
"onaniz", "ond", "ondej", "ondo", "ondul", "oner",
"onzen", "onzeneir", "op", "opac", "opacific", "opalin",
"opaliniz", "opaliz", "oper", "operacionaliz", "opi", "opil",
"opin", "oportuniz", "opt", "optimiz", "opugn", "opulent",
"opuscul", "or", "oracul", "oraculiz", "oraliz", "orbit",
"or"+u"\u00E7", "or"+u"\u00E7"+"ament", "orden", "ordenh", "ordinh", "orelh",
"orfan", "organific", "organiz", "organoform", "organofosfor", "organsin",
"orgi", "orgulh", "orient", "orientaliz", "origin", "originaliz",
"orl", "orladur", "orn", "ornament", "ornej", "orograf",
"orquestr", "orquiectomiz", "ortig", "ortocromatiz", "ortogonaliz", "ortograf",
"ortonormaliz", "ortopantomograf", "ortorretific", "orvalh", "oscil", "oscit",
"oscul", "osleriz", "osseaniz", "ossianiz", "ossific", "ostent",
"ostraciz", "oteeniz", "oteniz", "otimiz", "oug", "our",
"ourej", "ourel", "ouri"+u"\u00E7", "ourin", "ous", "ousi",
"out", "outav", "outon", "outorg", "ov", "ovacion",
"oval", "ovaliz", "ovul", "oxid", "oxigen", "oxigeniz",
"oxitoniz", "ozon", "ozonific", "ozoniz", "pabul", "pacej",
"pachol", "pacient", "pacific", "pact", "pactu", "padej",
"padr", "padrinh", "padroniz", "paduc", "pag", "pagai",
"paganiz", "pagin", "pair", "paiv", "paladin", "palanc",
"palataliz", "palatiz", "palavr", "palavrej", "palecion", "palej",
"paleograf", "palerm", "palestr", "palet", "palh", "palhet",
"palhi"+u"\u00E7", "pali", "palidej", "palific", "palit", "palm",
"palmato", "palmatori", "palmej", "palmilh", "palom", "palomb",
"palor", "palorej", "palp", "palpabiliz", "palpebrej", "palpit",
"palr", "pan", "pana"+u"\u00E7", "pand", "pandeg", "pandemiz",
"pandicul", "pandilh", "pandulh", "panegiric", "panegiriz", "panej",
"panflet", "pang", "pangai", "panific", "panoramiz", "panri",
"pansianiz", "pantaniz", "pantograf", "pantomim", "pantomin", "panzu",
"pap", "papagai", "papagu"+u"\u00E7", "papaguej", "paparic", "paparrot",
"papej", "papelon", "papi", "papoc", "papoil", "papouc",
"papoul", "papuj", "paqueir", "paquer", "par", "parabeniz",
"paraboliz", "paradigmatiz", "paradox", "parafin", "parafiniz", "parafus",
"paragon", "paragraf", "paraleliz", "paralis", "paralitic", "paralogiz",
"parament", "parametriz", "paramilitariz", "paramuscul", "parangon", "paraninf",
"parapeit", "parasit", "parcel", "parcelariz", "parcialid", "parcializ",
"pard", "pardal", "pardalit", "pardej", "parej", "parelh",
"parent", "parentes", "parentesi", "paresi", "parific", "parisianiz",
"parl", "parlament", "parlamentariz", "parlamentiz", "parlend", "parleng",
"parodi", "parol", "paroqui", "parouvel", "paroxitoniz", "parr",
"parraf", "parran"+u"\u00E7", "parron", "partej", "particion", "particip",
"particulariz", "partidariz", "partilh", "parturej", "parvoeir", "parvoej",
"pascent", "pasco", "pasigraf", "pasm", "paspalh", "pasquin",
"pass", "passaj", "passaman", "passarinh", "passarit", "passej",
"passinh", "passiv", "past", "pastej", "pasteuriz", "pastich",
"pastilh", "pastinh", "pastor", "pastoraliz", "pastorej", "pastoriz",
"patanh", "patarat", "pateg", "patej", "patent", "patentiz",
"patet", "pati", "patin", "patinh", "patoil", "patol",
"patranh", "patriz", "patro", "patrocin", "patrol", "patroniz",
"patrulh", "patusc", "paul", "paulific", "paulistaniz", "pauperiz",
"paurometaboliz", "paus", "paut", "pavej", "paves", "paviment",
"paxaliz", "peal", "pec", "pech", "pechinch", "pechiring",
"pe"+u"\u00E7"+"onhent", "peculiariz", "pedagogiz", "pedal", "pedalej", "pedantiz",
"pedestaliz", "pedic", "pedicel", "pediculiz", "pediment", "pedinch",
"pedinh", "pedint", "pediplan", "pedr", "peg", "peganh",
"peguilh", "peguinh", "pegunt", "peid", "peidorr", "pein",
"peit", "peitalh", "peix", "pej", "pejor", "pel",
"pelech", "pelej", "peletiz", "pelich", "pelid", "pelintr",
"pelletiz", "peloriz", "pelotiz", "pember", "pembr", "pen",
"penach", "penaliz", "pendenci", "pendo", "pendor", "pendul",
"pendur", "penduric", "peneir", "penej", "peneplaniz", "penetr",
"penhor", "penic", "penific", "penisc", "penitenci", "pens",
"pensament", "pension", "pentelh", "penuj", "penumbr", "pepin",
"peptiz", "peptoniz", "pequilh", "pequinh", "peragr", "peralt",
"peralvilh", "perambul", "perc", "percal"+u"\u00E7", "percepcion", "perceptualiz",
"percevej", "perch", "percint", "percol", "percorug", "perderiv",
"perdig", "perdigot", "perdo", "perdulari", "perdur", "peregrin",
"pereniz", "pererec", "perfectibiliz", "perfeicion", "perfei"+u"\u00E7"+"o", "perfi",
"perfil", "perfilh", "perfoli", "perform", "perfum", "perfur",
"pergaminh", "pergunt", "perici", "periclit", "perig", "periodiz",
"peripatetiz", "periquit", "perit", "peritoniz", "perjur", "perl",
"perlabor", "perlav", "perlej", "perleng", "perlong", "perlustr",
"permanganiz", "permeabiliz", "permineraliz", "permud", "permut", "pernej",
"pernoit", "pernosticiz", "pernout", "perol", "peroliz", "peror",
"peroxid", "perpass", "perpetr", "perpetu", "perreng", "perscrut",
"persever", "persign", "persilh", "personaliz", "personific", "persoterr",
"perspectiv", "perspetiv", "perspir", "pertent", "pertuch", "perturb",
"peru", "pervag", "pervoeir", "pes", "pesc", "pesg",
"pespeg", "pespont", "pesquis", "pessoaliz", "pestanej", "pestifer",
"pestilenci", "pestinh", "pet", "petanisc", "petard", "petarol",
"petec", "peteg", "peticion", "petilh", "petisc", "petit",
"petrarquiz", "petrech", "petrific", "petroliz", "pexot", "pezunh",
"pfuc", "pi", "piab", "piach", "piaf", "pial",
"pialh", "pian"+u"\u00E7", "pianiz", "pianol", "pic", "picardi",
"picaret", "pi"+u"\u00E7"+"arr", "pich", "piciric", "picot", "picu",
"picuinh", "pielograf", "pif", "pig", "pigarr", "pigment",
"pil", "pilch", "pild", "pildr", "pilh", "pilheri",
"pilot", "pilul", "pimb", "pimp", "pimpl", "pimpolh",
"pimpon", "pin", "pin"+u"\u00E7", "pincel", "pinch", "pincharol",
"pindariz", "pindoc", "pindong", "pindr", "pinel", "ping",
"pingalh", "pingent", "pingol", "pinic", "pinimb", "pinoc",
"pinot", "pint", "pintainh", "pintal", "pintalg", "pintalh",
"pintarol", "pintipar", "pinturil", "piolh", "pior", "pip",
"piparot", "pipet", "pipi", "pipil", "pipin", "pipiric",
"pipit", "pipoc", "piquet", "pir", "piramb", "piramid",
"piramidiz", "pirang", "pirat", "pirilamp", "pirilampej", "piriric",
"piroc", "pirograf", "pirograv", "pirolis", "pirotecniz", "pirra"+u"\u00E7",
"pirroniz", "piruet", "pis", "pisc", "piscainh", "piscalh",
"pisg", "piso", "pissit", "pist", "pit", "piteir",
"pitosg", "piuc", "pivot", "pixer", "placid", "placit",
"plagi", "plain", "plan", "planch", "planej", "planetiz",
"planific", "planilh", "planimetr", "planiz", "planopolariz", "plant",
"plasm", "plasmolis", "plastic", "plasticiz", "plastific", "platin",
"platoniz", "plebiscit", "plebiz", "pleit", "plenific", "pletor",
"pletoriz", "plic", "plid", "pliss", "plot", "plug",
"plum", "plumbagin", "plumej", "pluraliz", "pluricelul", "pluriestratific",
"plurific", "plurissecul", "pluristratific", "plutarquiz", "plutocratiz", "pneumencefalograf",
"pneumoencefalograf", "poc", "pod", "podometr", "poeir", "poematiz",
"poent", "poet", "poetific", "poetiz", "pois", "poit",
"poj", "polariz", "polc", "poleg", "polemic", "polemiz",
"polici", "policit", "policopi", "policrom", "policromiz", "poligraf",
"poliment", "polimeriz", "poliniz", "polirramific", "politic", "politip",
"politiz", "politon", "poloniz", "polp", "polvariz", "polvilh",
"polvoriz", "pombeir", "pomej", "pomific", "pomiz", "pompoariz",
"pon"+u"\u00E7", "ponder", "pong", "ponh", "pont", "pontalet",
"ponticul", "pontific", "pontilh", "pontinh", "ponto", "pontu",
"pontualiz", "popi", "popoc", "popul", "populariz", "porej",
"porfi", "porfiriz", "pormenoriz", "pornograf", "pororoc", "port",
"portaj", "portuch", "portugaliz", "portugues", "porvisc", "pos",
"posfaci", "posicion", "positiv", "pospont", "poss", "possession",
"possibilit", "possibiliz", "post", "posteg", "postej", "postem",
"posterg", "posteriz", "posti"+u"\u00E7", "postil", "postul", "postur",
"potenci", "potencializ", "potoc", "poup", "pous", "pousent",
"pout", "povo", "pracat", "pracej", "pradej", "prag",
"pragalh", "pragmatiz", "praguej", "pranch", "prant", "prasm",
"pratej", "pratic", "prax", "preadapt", "preadivinh", "prealeg",
"pream", "preambul", "preanunci", "prebend", "precal"+u"\u00E7", "precant",
"precariz", "precat", "precaucion", "precautel", "preceit", "preceitu",
"precific", "precint", "precipit", "precis", "precogit", "preconceitu",
"precondicion", "preconiz", "predat", "predemarc", "predestin", "predetermin",
"predic", "predicament", "predilecion", "predit", "predomin", "preemin",
"preestipul", "prefabric", "prefaci", "preferenci", "prefigur", "prefiltr",
"prefix", "prefoli", "prefulgur", "preg", "pregalh", "pregn",
"prego", "pregost", "pregui"+u"\u00E7", "pregunt", "pregust", "preit",
"preiteg", "preitej", "prejudic", "prelaci", "prelazi", "prelecion",
"prelev", "preli", "prelib", "preludi", "prem", "premasteriz",
"prematur", "premedit", "premi", "premud", "prend", "prenh",
"prenomin", "prenot", "prens", "prenunci", "prenunic", "preocup",
"preopin", "preorden", "prepar", "preponder", "preposicion", "preposter",
"preprocess", "prequestion", "pres", "presagi", "prescrut", "presenci",
"present", "presentific", "preserv", "presidi", "presig", "presilh",
"pressagi", "presselecion", "pression", "pressuriz", "prest", "prestacion",
"prestidigit", "prestigi", "presum", "pretej", "preternaturaliz", "pretext",
"pretorianiz", "pretra"+u"\u00E7", "prevaric", "prez", "prim", "primaver",
"primaveriz", "primitiv", "primor", "principi", "prior", "prioriz",
"prisc", "prism", "prismatiz", "pristin", "priv", "privatiz",
"privilegi", "pro", "probabiliz", "problematiz", "procedimentaliz", "process",
"procession", "proclam", "proclin", "procrastin", "procri", "procur",
"procurv", "prodig", "prodigaliz", "proej", "proemi", "proemin",
"profa"+u"\u00E7", "profal"+u"\u00E7", "profan", "profaniz", "profess", "professor",
"profet", "profetiz", "profil", "profissionaliz", "proflig", "profund",
"progener", "prognostic", "program", "programatiz", "progress", "projeit",
"projet", "prolat", "proletariz", "prolifer", "prolific", "prolog",
"prolong", "proman", "promand", "promulg", "pronomi", "pronomin",
"pronominaliz", "pronostic", "prontific", "prontuari", "pronunci", "propag",
"propal", "proparoxitoniz", "propici", "propin", "propolis", "propoliz",
"proporcion", "proporcionaliz", "proposit", "propugn", "propuls", "propulsion",
"prorrog", "pros", "prosific", "prospect", "prosper", "prospet",
"prostern", "prostr", "protagoniz", "protecion", "protel", "protest",
"protestantiz", "protocol", "protocoliz", "proton", "prototific", "protuber",
"prov", "proven"+u"\u00E7"+"aliz", "proverbi", "proverbializ", "providenci", "provincializ",
"provincianiz", "provisc", "provision", "provoc", "proz", "prudenci",
"prum", "prussianiz", "pseudojustific", "pseudotrabalh", "psicanalis", "psicoanalis",
"psicograf", "psicolog", "psicologiz", "pu", "pub", "public",
"publicit", "publiciz", "pudl", "pueriliz", "pugil", "pugn",
"puit", "puj", "pul", "pulh", "pulic", "pulinh",
"puls", "pulsion", "pult", "pulul", "pulveriz", "pulvin",
"pun"+u"\u00E7", "puncej", "puncion", "pun"+u"\u00E7"+"o", "punt", "pup",
"pupil", "pupiniz", "purg", "purific", "purpur", "purpurej",
"purpuriz", "putrific", "pux", "quadr", "quadricul", "quadrilh",
"quadrilob", "quadriniz", "quadruped", "quadrupl", "quadruplic", "qualific",
"quantific", "quantiz", "quar", "quarenten", "quaresm", "quart",
"quartej", "quartilh", "quass", "quaterniz", "quebr", "quebrant",
"qued", "queij", "queim", "quein", "queix", "quel",
"quenh", "quent", "queratiniz", "querel", "queren", "querenci",
"querend", "querosen", "question", "questiuncul", "quevind", "quezil",
"quiband", "quic", "quiet", "quilat", "quilh", "quilific",
"quilometr", "quilot", "quilunz", "quimb", "quimeriz", "quimific",
"quimiossintetiz", "quin", "quinch", "quing", "quinho", "quint",
"quintav", "quintessenci", "quintuplic", "quirograf", "quit", "quitand",
"quitiniz", "quixotiz", "quizil", "quocient", "quot", "quotidianiz",
"quotiz", "raban", "rabeir", "rabej", "rabi", "rabid",
"rabiniz", "rabios", "rabisc", "rabon", "rabot", "rabuj",
"rabul", "rabulej", "rabun", "rabunh", "rabusc", "racemiz",
"rach", "racin", "raciocin", "racion", "racionaliz", "ra"+u"\u00E7"+"o",
"racont", "radi", "radiativ", "radic", "radicaliz", "radici",
"radioativ", "radiobaliz", "radiocromatograf", "radiodetect", "radiodiagnostic", "radiofoniz",
"radiograf", "radiolocaliz", "radiomarc", "radionaveg", "radiopreg", "radiossond",
"radiotelegraf", "radiotelevis", "radiotransistoriz", "radob", "radub", "raf",
"raff", "rafi", "rafin", "rai", "raiv", "raivej",
"raj", "ral", "ralent", "ralh", "ram", "ramalh",
"ramalhet", "ramang", "ramel", "ramific", "ramilhet", "ramoc",
"rampan", "ran"+u"\u00E7", "ranceon", "rancidific", "rancific", "rancorej",
"rancur", "randomiz", "ranf", "rang", "ranguinh", "ranh",
"ranhur", "ranhuz", "ranilh", "ranzinz", "rap", "rapel",
"rapi", "rapig", "rapilh", "rapin", "rapinh", "rapos",
"raposinh", "rapt", "raquitiz", "rar", "rarej", "ras",
"rasc", "rascanh", "rascunh", "rasg", "rasmolg", "rasoir",
"rasour", "rasp", "raspan"+u"\u00E7", "raspinh", "rastaquer", "rasteir",
"rastej", "rastel", "rasteriz", "rastilh", "rastinh", "rastolh",
"rastrej", "rasur", "rat", "ratazan", "ratific", "ratin",
"ratinh", "raton", "raus", "ravenh", "ravin", "ravinh",
"razi", "razo", "razon", "reabilit", "reabit", "reabitu",
"reabjur", "reaboto", "reabra"+u"\u00E7", "reabrasileir", "reabrolh", "reabrot",
"reaceit", "reach", "reacion", "reacomod", "reacompanh", "reacopl",
"reacord", "reacostum", "reacumul", "reacus", "readapt", "readjudic",
"readmir", "readot", "reafet", "reafirm", "reafix", "reafret",
"reagend", "reaglomer", "reagrav", "reagrup", "reajunt", "reajust",
"realarg", "real"+u"\u00E7", "realegr", "realej", "realent", "realiment",
"realinh", "realist", "realiz", "realoc", "realoj", "realug",
"realumi", "ream", "reambul", "reanalis", "reand", "reanex",
"reanim", "reanimaliz", "reanod", "reapaixon", "reaparelh", "reapert",
"reaplic", "reapoder", "reapont", "reaportugues", "reaposs", "reapreci",
"reapresent", "reaprision", "reapropri", "reaprov", "reaproveit", "reaprovision",
"reaproxim", "reaquist", "rearboriz", "rearm", "rearmazen", "rearquitet",
"rearranj", "rearrum", "rearticul", "reasfalt", "reassegur", "reassent",
"reassimil", "reassin", "reassoci", "reassoldad", "reassom", "reassopr",
"reat", "reatent", "reaterr", "reatest", "reati"+u"\u00E7", "reativ",
"reatoch", "reatoriz", "reatravess", "reatualiz", "reatulh", "reaument",
"reautentic", "reautoriz", "reautu", "reavali", "reavi", "reavigor",
"reavis", "reavist", "reaviv", "reavult", "reb", "rebaix",
"rebalan"+u"\u00E7", "rebals", "reban", "rebanh", "rebarb", "rebarbariz",
"rebat", "rebatinh", "rebatiz", "rebeij", "rebel", "rebelic",
"rebelion", "rebent", "reberr", "rebesbelh", "rebic", "rebimb",
"rebimbalh", "rebiopsi", "rebisc", "rebit", "rebitol", "rebo",
"rebobin", "reboc", "reboj", "rebol", "rebolc", "reboli"+u"\u00E7",
"rebomb", "rebor", "reborbulh", "rebord", "reborqui", "rebot",
"rebou"+u"\u00E7", "rebrad", "rebram", "rebrasil", "rebrilh", "rebrot",
"rebu"+u"\u00E7", "rebuf", "rebulh", "rebuli"+u"\u00E7", "rebumb", "rebusc",
"rebusn", "rec", "recabed", "recach", "recad", "recadastr",
"recadej", "recal", "recalc", "recal"+u"\u00E7", "recalcific", "recalcitr",
"recalcul", "recalendariz", "recalibr", "recam", "recambi", "recanaliz",
"recandidat", "recant", "recap", "recapacit", "recapitaliz", "recapitul",
"recapt", "recaptur", "recarbonat", "recarboniz", "recarbur", "recard",
"recarg", "recarimb", "recarreg", "recartilh", "recartorializ", "recas",
"recat", "recategoriz", "recativ", "recauchut", "recaucion", "recautel",
"recav", "recavalg", "receit", "recelulariz", "recen", "recentr",
"recepcion", "recept", "recerr", "recha"+u"\u00E7", "rechap", "rechec",
"recheg", "rechi", "rechin", "rechoch", "rechonch", "rechup",
"recib", "recicl", "recidiv", "recint", "reciproc", "recit",
"reciviliz", "reclacitr", "reclam", "reclamiz", "reclassific", "reclin",
"reclus", "recobr", "recoc", "recogit", "recoit", "recol",
"recolig", "recoloc", "recoloniz", "recolt", "recombin", "recome"+u"\u00E7",
"recomend", "recoment", "recomod", "recompact", "recompartiment", "recompartimentaliz",
"recompens", "recompil", "recomplet", "recompr", "recompromiss", "reconceitu",
"reconcentr", "reconcert", "reconcili", "recondicion", "reconect", "reconfess",
"reconfigur", "reconfirm", "reconfort", "recongra"+u"\u00E7", "reconquist", "reconsagr",
"reconsert", "reconsider", "reconsolid", "reconsorci", "reconstitucionaliz", "reconsult",
"recont", "recontat", "recontextualiz", "recontrat", "reconvoc", "recopi",
"recopil", "recor"+u"\u00E7", "record", "recoro", "recort", "recost",
"recout", "recov", "recrav", "recredenci", "recrest", "recri",
"recrimin", "recriptograf", "recristaliz", "recristianiz", "recrucific", "recrut",
"recruz", "recu", "recud", "recuid", "recuinh", "recuit",
"recultiv", "recunh", "recuper", "recur", "recurs", "recurv",
"recus", "redactilograf", "redam", "redambal", "redardej", "redatilograf",
"redatori", "redeclar", "redecor", "redecret", "rededetiz", "rededic",
"redemocratiz", "redemoinh", "redemonstr", "redeng", "redeposit", "redescont",
"redesenh", "redesgrenh", "redesign", "redespert", "redestin", "redigit",
"redimension", "redintegr", "redirecion", "redispar", "rediviniz", "redizim",
"redob", "redobr", "redoi"+u"\u00E7", "redoir", "redomoinh", "redopi",
"redou"+u"\u00E7", "redour", "redr", "redubl", "redund", "reduplic",
"reeco", "reedific", "reedit", "reeditor", "reeduc", "reelabor",
"reembaralh", "reembarc", "reembas", "reembols", "reembrulh", "reemend",
"reemigr", "reempacot", "reempalh", "reempolg", "reemporc", "reemposs",
"reempreg", "reencadern", "reencafu", "reencaix", "reencaminh", "reencan",
"reencap", "reencarcer", "reencarn", "reencen", "reencet", "reencontr",
"reencuc", "reendere"+u"\u00E7", "reendireit", "reendoss", "reenergiz", "reenfi",
"reenfolh", "reengaj", "reengendr", "reengolip", "reenla"+u"\u00E7", "reenovel",
"reenquadr", "reensai", "reensin", "reentel", "reenterr", "reentr",
"reentroniz", "reenvas", "reenverniz", "reenvi", "reenvid", "reenxert",
"reequilibr", "reequip", "reescal", "reescalon", "reescrach", "reescrachet",
"reescrachin", "reescrafunch", "reescramalh", "reescramel", "reescrament", "reescramouc",
"reescramu"+u"\u00E7", "reescraviz", "reescrevinh", "reescritur", "reespecific", "reesper",
"reespos", "reespum", "reestamp", "reestiliz", "reestimul", "reestipul",
"reestrutur", "reestud", "reetiquet", "reexager", "reexamin", "reexecut",
"reexit", "reexpatri", "reexpi", "reexplic", "reexplor", "reexport",
"reextradit", "refal", "refals", "refart", "refastel", "refator",
"refatur", "refech", "refecund", "refeg", "refent", "referenci",
"referend", "referment", "referr", "refert", "refestel", "refez",
"refi", "refigur", "refil", "refilh", "refili", "refilm",
"refiltr", "refin", "refinanci", "refinc", "refint", "refirm",
"refit", "refl", "reflacion", "reflexion", "reflorest", "reflutu",
"reflux", "refoc", "refocaliz", "refocil", "refog", "refoleg",
"refolg", "refolh", "refoment", "refonologiz", "refor"+u"\u00E7", "reforj",
"reform", "reformat", "reformul", "refortific", "refotograf", "refract",
"refractur", "refranj", "refrat", "refratur", "refreg", "refrequent",
"refresc", "refret", "refric", "refriger", "refris", "refrit",
"refrond", "refrulh", "refug", "refugi", "refulgur", "refum",
"refund", "refunf", "refunfumeg", "refung", "refunh", "refus",
"refust", "refut", "reg", "rega"+u"\u00E7", "regal", "regalardo",
"regalvaniz", "regambi", "reganh", "regaseific", "regat", "regatinh",
"regel", "regener", "regent", "reger", "regermin", "regiment",
"regin", "regionaliz", "regir", "regist", "registr", "rego",
"regojiz", "regolf", "regom", "regoug", "regozij", "regr",
"regraci", "regrav", "regrax", "regress", "reguard", "regueif",
"regueir", "reguinch", "reguing", "regulament", "regulariz", "regurgit",
"reidrat", "reific", "reimagin", "reimplant", "reimplement", "reimport",
"reimput", "rein", "reinaugur", "reincarn", "reincit", "reincorpor",
"reincrust", "reindex", "reindustrializ", "reinerv", "reinfecion", "reinfect",
"reinfet", "reinflam", "reingress", "reinici", "reinicializ", "reinjet",
"reinocul", "reinstal", "reinstaur", "reintegr", "reintern", "reinternacionaliz",
"reinternaliz", "reinterpret", "reinterrog", "reinum", "reinvent", "reinvid",
"reinvoc", "reioniz", "reislamiz", "reiter", "reitoriz", "reiun",
"reivindic", "reix", "rejeit", "rejubil", "rejunc", "rejunt",
"rejur", "rejustific", "rel", "relacion", "relacr", "relamp",
"relampad", "relampadej", "relampaguej", "relampej", "relampr", "relan"+u"\u00E7",
"relancet", "relarg", "relat", "relativ", "relativiz", "relav",
"relax", "releg", "relegitim", "releix", "relembr", "relent",
"relep", "relev", "relevant", "relh", "relicit", "relig",
"relim", "relinch", "reling", "reloc", "relocaliz", "reloj",
"relouc", "relum", "relumbr", "relustr", "relut", "relv",
"relvej", "rem", "remadaleniz", "remagnetiz", "remalh", "remalinh",
"remanch", "remand", "remanej", "remang", "remanipul", "remanisc",
"remans", "remanufatur", "remarc", "remarid", "remartel", "remasc",
"remasteriz", "remastig", "remat", "rematerializ", "rematricul", "rembrandtiz",
"rembraniz", "remed", "remel", "remelg", "remembr", "rememor",
"rememori", "remend", "remenic", "remergulh", "remess", "remestr",
"remigi", "remigr", "remilitariz", "remim", "remin", "remineraliz",
"reminiscenci", "remir", "remistur", "remix", "remobil", "remobiliz",
"remoc", "remo"+u"\u00E7", "remodel", "remodul", "remoel", "remoidanh",
"remoinh", "remolc", "remolg", "remolh", "remonarquiz", "remond",
"remonotoniz", "remont", "remor", "remorc", "remordic", "remordisc",
"remost", "remostr", "remouc", "remud", "remug", "remunc",
"remuner", "remunhet", "remurmur", "remurmurej", "remusg", "renacionaliz",
"renatur", "renaturaliz", "renaveg", "rend", "renderiz", "rendibiliz",
"rendilh", "renegoci", "renegrej", "renembr", "renet", "renifl",
"renivel", "renormaliz", "renov", "rent", "rentabiliz", "rentilh",
"renumer", "renunci", "renzilh", "reobrig", "reobserv", "reocup",
"reoficializ", "reolis", "reoper", "reopin", "reorden", "reorganiz",
"reorient", "reorquestr", "reoxid", "reoxigen", "reoxigeniz", "repactu",
"repag", "repaganiz", "repagin", "repair", "repanh", "repapoil",
"repapoul", "repar", "reparafus", "reparametriz", "reparcel", "reparticion",
"repartilh", "repass", "repassarit", "repast", "repatan", "repatri",
"repatrocin", "repaviment", "repe"+u"\u00E7", "repech", "reped", "repeg",
"repeit", "repel", "repenetr", "repenic", "repens", "reperfil",
"repergunt", "repertori", "repes", "repesc", "repeten", "repic",
"repilg", "repimp", "repimpin", "repimpon", "repinch", "repinhaliz",
"repinic", "repint", "repiquet", "repis", "repisc", "repivatiz",
"replain", "replanej", "replanilh", "replant", "replen", "replet",
"replic", "repois", "repolariz", "repoleg", "repolg", "repolh",
"repoliniz", "reponder", "repont", "repontu", "repopul", "repopulariz",
"report", "reposicion", "repost", "repotron", "repous", "repovo",
"repreg", "repres", "represent", "reprimend", "reprincipi", "repris",
"repristin", "repristiniz", "reprivatiz", "reprocess", "reproch", "reprofund",
"reprogram", "reprojet", "reprov", "rept", "republic", "republicaniz",
"repudi", "repugn", "repul", "repuls", "repulul", "repurg",
"repurific", "reput", "repux", "requadr", "requadruped", "requalific",
"requebr", "requeim", "requent", "request", "requestion", "requint",
"requisit", "rerradi", "rerratific", "rerrespir", "resbun", "rescald",
"rescum", "resenh", "reserv", "resfolg", "resfri", "resg",
"resgalg", "resgat", "resguard", "residenci", "resign", "resin",
"resinific", "reslumbr", "resm", "resmalh", "resmeleng", "resmolg",
"resmoninh", "resmun", "resmung", "resmunh", "resmuninh", "resmusg",
"respald", "respan"+u"\u00E7", "respeit", "respig", "resping", "respinh",
"respir", "respons", "responsabiliz", "respont", "respost", "respumej",
"resqui", "ressabi", "ressac", "ressaci", "ressaib", "ressaibi",
"ressalt", "ressaltit", "ressalv", "ressanfonin", "ressangr", "ressec",
"ressegund", "ressegur", "ressel", "resseren", "resserr", "ressic",
"ressimboliz", "ressincroniz", "ressintetiz", "resso", "ressobr", "ressocializ",
"ressold", "ressolh", "ressolt", "resson", "ressonh", "ressopr",
"ressu", "ressubtiliz", "ressud", "ressulc", "ressum", "ressumbr",
"ressupin", "ressuscit", "rest", "restamp", "restart", "restaur",
"restauric", "restel", "restil", "restir", "restit", "restiv",
"restolh", "restral", "restrel", "restrib", "restrical", "restridul",
"restrug", "restrutur", "restuc", "result", "resval", "ret",
"retabul", "retac", "retalh", "retali", "retanch", "retard",
"reteim", "retelefon", "retelegraf", "retelh", "retemper", "retent",
"reterritorializ", "retes", "retesi", "retest", "reticenci", "retific",
"retilint", "retiliz", "retintin", "retipific", "retir", "retisn",
"retoc", "retoi"+u"\u00E7", "retom", "retonh", "retoric", "retorn",
"retos", "retost", "retouc", "retou"+u"\u00E7", "retov", "retrabalh",
"retra"+u"\u00E7", "retram", "retranc", "retran"+u"\u00E7", "retranslad", "retransplant",
"retransport", "retras", "retraslad", "retrat", "retrav", "retrein",
"retrilh", "retrin", "retrinc", "retrinch", "retro", "retroaliment",
"retroativ", "retrocarreg", "retrocit", "retrocruz", "retrodat", "retrograd",
"retroilumin", "retropedal", "retrovacin", "retrovej", "retruc", "retub",
"retumb", "reumaniz", "reumatiz", "reunific", "reurbaniz", "reus",
"reutiliz", "revacin", "revalid", "revaloriz", "revasculariz", "reveget",
"revel", "revener", "reverber", "reverdej", "reverenci", "reverg",
"reverific", "revermelh", "revers", "reves", "revess", "revez",
"revibr", "revi"+u"\u00E7", "revid", "revigor", "revigorent", "revigoriz",
"revindic", "reving", "revir", "revirginiz", "revis", "revisit",
"revist", "revitaliz", "revivenci", "revivific", "revo", "revoc",
"revoej", "revolc", "revolit", "revolt", "revolucion", "revor",
"revuls", "rez", "rezing", "rezip", "rib", "ribald",
"ribeir", "ribomb", "ri"+u"\u00E7", "ricerc", "ricochet", "ridic",
"ridiculariz", "ridiculiz", "rif", "rifl", "rig", "rij",
"ril", "rilh", "rim", "rimbomb", "rin"+u"\u00E7", "rinch",
"rinchal", "rinchavelh", "rincon", "rinf", "rinh", "rip",
"ripan"+u"\u00E7", "ripost", "risc", "riscanh", "ritm", "ritmiz",
"ritornel", "ritualiz", "rivaliz", "rix", "riz", "robech",
"robor", "roboriz", "robotiz", "roc", "ro"+u"\u00E7", "ro"+u"\u00E7"+"ag",
"rocalh", "roceg", "roci", "rocin", "rocl", "rod",
"rodej", "rodilh", "rodizi", "rodomoinh", "rodopi", "rodrig",
"rog", "roj", "rol", "rold", "rolet", "rolh",
"roli"+u"\u00E7", "rom", "romanesc", "romaniz", "romantiz", "romesent",
"ronc", "ronch", "rond", "ronfl", "ronquej", "ronron",
"ror", "rorej", "ros", "rosc", "roset", "rosmaninh",
"rosn", "rosnor", "rostilh", "rot", "rotacion", "rotativ",
"roteiriz", "rotej", "rotiniz", "rotograv", "rotul", "roub",
"rou"+u"\u00E7", "roufenh", "roup", "rouquej", "rouss", "rouxinol",
"rouxinoliz", "ru", "rubej", "rubific", "ruborej", "ruboriz",
"rubric", "rubriz", "ru"+u"\u00E7", "rudimentariz", "ruf", "rufi",
"rufl", "rug", "rugin", "rugit", "ruid", "rul",
"rulh", "rum", "rumb", "rumi", "rumin", "rumor",
"rumorej", "runf", "runh", "rup", "ruraliz", "rurbaniz",
"rusg", "russ", "russianiz", "russific", "russiz", "rustic",
"rustific", "rut", "rutil", "sabadej", "sabatin", "sabatiz",
"saberec", "sabererec", "sabich", "sabot", "sabrec", "sabuc",
"sabug", "sabuj", "sabul", "saburr", "sac", "sa"+u"\u00E7"+"aric",
"sacarific", "sacarrolh", "sach", "sachol", "saci", "sacol",
"sacolej", "sacraliz", "sacrament", "sacrific", "sacul", "saf",
"safan", "safiriz", "safrej", "sagin", "sagit", "sagorr",
"sagot", "sagr", "saibr", "salamord", "salari", "sald",
"salg", "salh", "salicil", "salient", "salific", "salin",
"saliniz", "salitr", "salitriz", "saliv", "salmej", "salmiz",
"salmodej", "salmodi", "salmoeir", "salmoir", "salmour", "salpic",
"salpiment", "salpint", "salpoeir", "salpres", "sals", "salt",
"saltarel", "saltaric", "saltarilh", "saltarinh", "saltinh", "saltit",
"salubrific", "salubriz", "salu"+u"\u00E7", "salud", "salv", "salvaguard",
"samang", "samb", "sambarc", "sambenit", "sambl", "samboc",
"sambr", "samp", "san", "sanatoriz", "sancion", "sand",
"sandej", "sanef", "sanfon", "sanfonin", "sanforiz", "sangr",
"sangrentej", "sanguej", "sanguel", "sanguent", "sanguessug", "sanguific",
"sanguinh", "sanic", "sanific", "sanitariz", "sanitiz", "sanj",
"sanquit", "sanscritiz", "santific", "santig", "sap", "sapat",
"sapec", "sapej", "saperec", "sapoil", "saponific", "sapori",
"sapoul", "saprem", "saquel", "sar", "sarafusc", "saraiv",
"sarandilh", "sarangonh", "sarapant", "sarapic", "sarapint", "saraqueim",
"saraquit", "sard", "sardanapaliz", "sardoniz", "sarilh", "sarj",
"sarn", "sarp", "sarr", "sarrabisc", "sarraf", "sarrafa"+u"\u00E7",
"sarrazin", "sart", "sass", "satani", "sataniz", "satelitiz",
"sateliz", "satiriz", "satisd", "satur", "saussuritiz", "saxific",
"sazo", "sazon", "schilleriz", "sebenisc", "sec", "seccion",
"secession", "seci", "secion", "secret", "secretari", "seculariz",
"secund", "secundariz", "securitiz", "sed", "sedentariz", "sedi",
"sediment", "seg", "segment", "segred", "segreg", "segund",
"segur", "seit", "seiv", "sel", "selecion", "selet",
"selvatic", "semantiz", "semb", "sembr", "semelh", "sement",
"semiadapt", "semiafirm", "semiajust", "semianimaliz", "semiaplic", "semiaprimor",
"semiaprov", "semibenefici", "semicadastr", "semicapacit", "semicarboniz", "semicaricatur",
"semicerr", "semicol", "semiconect", "semidespert", "semidevor", "semidiscretiz",
"semidiviniz", "semidobr", "semidomestic", "semienergiz", "semiequip", "semifech",
"semiformat", "semilevant", "semimanufatur", "seminorm", "semiofert", "semiograf",
"semipatrocin", "semipint", "semipoliniz", "semitiz", "semito", "semiton",
"semivo", "semostr", "seng", "senhoriz", "seniliz", "sensaboriz",
"sensibiliz", "sensific", "sensualiz", "sent", "sentenci", "sentimentaliz",
"sentinel", "separ", "sept", "septuplic", "sepulcr", "sepult",
"sequel", "sequenci", "sequestr", "sequi", "serafiz", "seramang",
"serand", "seranz", "seren", "sereniz", "seri", "serigait",
"serigraf", "sering", "seringonh", "seriograf", "sermo", "sermon",
"sero", "serpej", "serpent", "serpentiniz", "serr", "serralh",
"serramanc", "serrazin", "serrilh", "serrin", "serrot", "sertanej",
"sertaniz", "serviliz", "sesm", "sess", "session", "sestr",
"set", "seteir", "setin", "setoriz", "setuplic", "sev",
"sevandij", "severiz", "sevici", "sex", "sextav", "sextuplic",
"sexu", "sexualiz", "sezo", "si", "sibarit", "sibil",
"sibilantiz", "sibit", "sider", "sideriz", "sifiliz", "sifl",
"sifon", "sigil", "sigl", "sigliz", "siglonimiz", "sign",
"signific", "silab", "silabific", "silenci", "silhuet", "silicat",
"silicatiz", "silicific", "silicon", "silogism", "silogistic", "silogiz",
"silv", "simbol", "simboliz", "simetriz", "simorfil", "simpatiz",
"simplific", "simul", "sin", "sinal", "sinalej", "sinaliz",
"sinapiz", "sincel", "sinceriz", "sincop", "sincopiz", "sincretiz",
"sincroniz", "sindic", "sindicaliz", "sindicat", "sinet", "sinfoniz",
"singr", "singulariz", "singult", "sinistr", "sinistriz", "siniz",
"sinonimiz", "sinoptiz", "sintec", "sinteriz", "sintetiz", "sintomatiz",
"sintoniz", "sinu", "sinuc", "sinuos", "siren", "sirg",
"sirigait", "siriric", "sis", "sism", "sistem", "sistematiz",
"sistol", "siti", "situ", "snif", "so", "soalh",
"soass", "sobal"+u"\u00E7", "sobej", "soberaniz", "soberb", "sobescav",
"sobest", "sobi", "sobneg", "soborralh", "sobr", "sobra"+u"\u00E7",
"sobrad", "sobran"+u"\u00E7", "sobras", "sobreabund", "sobreagit", "sobrealcooliz",
"sobrealcunh", "sobrealiment", "sobreavali", "sobreavis", "sobreazed", "sobrecab",
"sobrecapitaliz", "sobrecasac", "sobrecifr", "sobrecompens", "sobrecostur", "sobredoir",
"sobredour", "sobrefatur", "sobreger", "sobregiz", "sobreirrit", "sobreleit",
"sobrelev", "sobrelot", "sobrelustr", "sobremaravilh", "sobremont", "sobrenad",
"sobrenaturaliz", "sobreolh", "sobrepair", "sobrepartilh", "sobrepass", "sobrepens",
"sobrepes", "sobrepovo", "sobrepuj", "sobrerrest", "sobrerrold", "sobrerrond",
"sobrescrit", "sobresper", "sobressalt", "sobressar", "sobressatur", "sobressel",
"sobressol", "sobrestim", "sobretax", "sobretro", "sobreus", "sobrevaloriz",
"sobrevigi", "sobrevo", "soc", "socalc", "socav", "sochi",
"sociabiliz", "socializ", "socioculturaliz", "so"+u"\u00E7"+"obr", "so"+u"\u00E7"+"oc", "socorn",
"socratiz", "socrest", "sod", "sodomiz", "sofism", "sofistic",
"sofor", "sofrald", "sofreg", "sofren", "sog", "sogr",
"sojig", "sojorn", "sojug", "sol", "sola"+u"\u00E7", "solanc",
"solap", "solariz", "solavanc", "solaz", "sold", "solec",
"soleciz", "soleniz", "soletr", "solev", "solevant", "solf",
"solfej", "solh", "solicit", "solid", "solidariz", "solidific",
"soliloqui", "solinh", "solmiz", "solt", "solubiliz", "solu"+u"\u00E7",
"solucion", "som", "somatiz", "sombr", "sombrej", "somergulh",
"somitic", "somp", "son", "sonambul", "sond", "soneg",
"sonet", "sonh", "sonoriz", "sopap", "sopeg", "sopes",
"sopi", "sopit", "sopont", "soporiz", "sopr", "sopraniz",
"sopres", "soprez", "sopux", "soqueix", "sor", "sorbic",
"sorn", "soroneutraliz", "sororoc", "sorovacin", "sorrab", "sorrasc",
"sorren", "sorv", "soslai", "sospit", "sosseg", "sossobr",
"sosteniz", "sostrej", "sot", "sotalh", "sotavent", "soterr",
"sotranc", "soturn", "soturniz", "soupic", "sout", "sov",
"sovel", "sovietiz", "sovin", "sprint", "su", "suaviz",
"subadapt", "subafirm", "subaglomer", "subajust", "subaliment", "subaltern",
"subalterniz", "subaplic", "subaprimor", "subaprov", "subaproveit", "subarrend",
"subassin", "subast", "subavali", "subcandidat", "subcapacit", "subcaricatur",
"subclassific", "subconcession", "subconscientiz", "subcontrat", "subdelir", "subderiv",
"subdimension", "subelisc", "subemend", "subempraz", "subempreit", "subenfiteutic",
"subenxert", "subequip", "suberific", "suberiniz", "suberiz", "subespecific",
"subestim", "subexamin", "subfatur", "subfinanci", "subformat", "subfret",
"subgrup", "subinstal", "subintitul", "subjectiv", "subjeit", "subjetiv",
"subjetiviz", "subjug", "sublamel", "sublev", "sublevant", "sublicenci",
"sublim", "sublimiz", "sublinh", "subloc", "subminiaturiz", "subministr",
"subnivel", "subofert", "suborbit", "subordin", "suborn", "subparcel",
"subpatrocin", "subplanilh", "subpuj", "subscrit", "subsecretari", "subsequenci",
"subsidi", "subsol", "substanci", "substancializ", "substantific", "substantiv",
"subsult", "subterr", "subtiliz", "subtitul", "subtotaliz", "subtrat",
"suburbaniz", "subutiliz", "subvaloriz", "subvencion", "subverbet", "suc",
"sucat", "suci", "sud", "sudoest", "sudorific", "suest",
"sufix", "sufl", "sufoc", "sufrag", "sufumig", "sug",
"sugestion", "sugil", "suicid", "suj", "sujeit", "sujic",
"sujig", "sujug", "sul", "sulanc", "sulavent", "sulc",
"sulfat", "sulfatiz", "sulfet", "sulficarbonat", "sulfit", "sulfon",
"sulfur", "sulfuret", "sulfuriz", "sumagr", "sumari", "sumariz",
"sumbr", "sumptualiz", "sumul", "sung", "sunguil", "suntualiz",
"supedit", "supenhor", "super", "superabund", "superadapt", "superafirm",
"superaglomer", "superajunt", "superajust", "superaliment", "superaplic", "superaprimor",
"superaprov", "superativ", "superauxili", "superavali", "superbenefici", "supercalandr",
"supercalcin", "supercandidat", "supercapacit", "supercapitaliz", "supercaricatur", "superciviliz",
"supercompact", "supercompens", "supercomplex", "superconcentr", "superconfort", "superdiagnostic",
"superdimension", "superendivid", "superenergiz", "superequip", "superescal", "superestim",
"superestrutur", "superexager", "superexalt", "superexamin", "superexcit", "superexplor",
"superfatur", "superfet", "superformat", "superimpregn", "superindustri", "superindustrializ",
"superinterpret", "superioriz", "superlativ", "superlot", "superocup", "superofert",
"superorden", "superorganiz", "superotimiz", "superoxid", "superoxigen", "superpatrocin",
"superplanilh", "superpopul", "superpovo", "supersatur", "superselecion", "superst",
"supertrein", "supervaloriz", "supervis", "supervision", "suplant", "suplement",
"suplic", "suplici", "supont", "suport", "supracit", "supraexcit",
"supralimin", "supralist", "supur", "suput", "surc", "surdin",
"surf", "surmont", "surpass", "surpres", "surr", "surrealiz",
"surri", "surrib", "surripi", "surripilh", "surrupi", "surt",
"suruc", "sururuc", "susceptibiliz", "suscetibiliz", "suscit", "suspeit",
"suspir", "sussurr", "sussurrej", "sust", "susteniz", "sustent",
"sut", "sutiliz", "sutur", "sux", "tabardilh", "tabel",
"tabeli", "tabelion", "tabeliz", "tabic", "tabiz", "tabuc",
"tabul", "tabulet", "taburn", "tac", "tach", "tachon",
"tactiliz", "taful", "tafulh", "tagant", "tagarel", "tain",
"taip", "tairoc", "tal", "taler", "talh", "talim",
"taling", "talion", "talon", "talud", "tamanc", "tamborej",
"tamboril", "tamborin", "tamis", "tamp", "tampon", "tamusc",
"tan", "tanaz", "tanc", "tancam", "tanch", "tang",
"tangenci", "tanific", "tanin", "taniniz", "tano", "tantaliz",
"tap", "tape"+u"\u00E7", "tapet", "tapetiz", "tapi"+u"\u00E7", "tapiz",
"tapulh", "taquigraf", "tar", "taralh", "taramel", "tarantul",
"tard", "tardinh", "taref", "tarel", "tarif", "tarific",
"tarimb", "tarj", "taroc", "tarol", "tarouc", "tarraf",
"tarrax", "tarrinc", "tarrut", "tartamel", "tartariz", "tartufic",
"tartufiz", "tarug", "tasc", "tascanh", "tasco", "tasg",
"tasquin", "tasquinh", "tassalh", "tastavel", "tastavelh", "tatal",
"tataranh", "tatiliz", "tatu", "tauroboliz", "tauxi", "tax",
"taxi", "taxidemiz", "taxidermiz", "tayloriz", "teatin", "teatraliz",
"tec", "tecl", "tecniciz", "tecnocratiz", "tecnologiz", "teim",
"teimosi", "tel", "telecomand", "telecomunic", "telecurs", "teledetect",
"telefon", "telefoniz", "telefotograf", "telegraf", "telegui", "teleport",
"telerradiograf", "telerregul", "telescop", "telessinaliz", "telessond", "telessupervision",
"teletrabalh", "teletransport", "televis", "television", "telex", "telh",
"telint", "telomeriz", "teluriz", "tematiz", "tembl", "temon",
"temoriz", "temper", "tempestu", "templ", "temporaliz", "temporej",
"temporiz", "tenalh", "tenaz", "tencion", "ten"+u"\u00E7"+"o", "tendenci",
"tenebriz", "tenoriz", "tenrific", "tension", "tent", "tenten",
"teocratiz", "teofiliz", "teologiz", "teoriz", "tequet", "ter"+u"\u00E7",
"terceir", "terceiriz", "tercet", "terci", "terebintin", "terebr",
"tergivers", "terjur", "terlad", "terlint", "termin", "termistoriz",
"termonucleariz", "termorregul", "tern", "terr", "terraform", "terramot",
"terraplan", "terraplen", "terrej", "terrific", "terrinc", "territorializ",
"terror", "terroriz", "tes", "tesauriz", "tesmin", "tesoir",
"tesour", "tessel", "test", "testament", "testavilh", "testemunh",
"testic", "testific", "testilh", "testug", "tet", "tetaniz",
"tetoniz", "textiliz", "textualiz", "textur", "texturiz", "tib",
"tibir", "tibung", "tic", "tiet", "tijol", "til",
"tild", "tilint", "timbalej", "timbr", "timbu", "timon",
"timpaniz", "tindaliz", "tindalliz", "ting", "tinguij", "tint",
"tintalh", "tintil", "tintin", "tintinabul", "tintl", "tipic",
"tipific", "tipiz", "tipograf", "tir", "tiramol", "tiraniz",
"tirilint", "tiriric", "tirit", "tirlint", "tirocin", "tisic",
"tisn", "titic", "titil", "titiric", "titub", "titul",
"titulariz", "tlint", "to", "toc", "tocai", "toes",
"tog", "toi"+u"\u00E7", "toir", "toirej", "told", "tolej",
"toler", "tolin", "tom", "tomb", "tombol", "tomograf",
"ton", "tonaliz", "tonific", "tonitro", "tonitru", "toniz",
"tons", "tonsur", "tontej", "top", "topej", "topet",
"topi", "topicaliz", "topograf", "tor", "torceg", "torcicol",
"torcion", "torcul", "torment", "torn", "tornej", "toron",
"torped", "torpilh", "torporiz", "torr", "torrej", "torric",
"torrific", "torrisc", "tortuej", "tortur", "torv", "torvelinh",
"tos", "tosc", "toscanej", "tosquenej", "tosqui", "tosseg",
"tossic", "tossig", "tost", "tosteg", "totaliz", "totemiz",
"touc", "tou"+u"\u00E7", "tour", "tourej", "toxic", "trabalh",
"trabalhisc", "trabalhuc", "trabelh", "trabuc", "trabul", "tra"+u"\u00E7",
"tracej", "tracion", "tracolej", "trad", "tradicionaliz", "trafeg",
"trafic", "trafulh", "trag", "tragedi", "tragin", "traicion",
"train", "traj", "tralad", "tralh", "tram", "tramalh",
"trambalh", "trambec", "trambic", "trambolh", "tramel", "tramit",
"tramont", "trampej", "trampolin", "tran", "tranc", "tran"+u"\u00E7",
"trancafi", "trancafil", "tranch", "tranqueir", "tranquelej", "tranquibern",
"tranquiberni", "tranquiliz", "tranquin", "trans", "transacion", "transbord",
"transbordej", "transcendentaliz", "transco", "transcodific", "transcol", "transcomplement",
"transcultur", "transcur", "transcurs", "transdisciplin", "transeccion", "transecion",
"transexualiz", "transfect", "transfeg", "transfigur", "transfiltr", "transfinaliz",
"transfix", "transfon", "transfor", "transform", "transfret", "transfur",
"transfus", "transignific", "transistoriz", "transit", "transitiv", "translad",
"transliter", "transloc", "translouc", "translucid", "translumbr", "transmalh",
"transmetil", "transmigr", "transmont", "transmud", "transmut", "transnacionaliz",
"transnad", "transnoit", "transnomin", "transnout", "transparent", "transpass",
"transpir", "transplant", "transport", "transtorn", "transtroc", "transubstanci",
"transud", "transum", "transuman", "transvas", "transvaz", "transverber",
"transvi", "transvo", "transvor", "trapa"+u"\u00E7", "trapach", "trapass",
"trapej", "traquej", "traquin", "traquitan", "trasand", "trasbord",
"trascal", "trascur", "trasfeg", "trasfoli", "traslad", "trasmalh",
"trasmont", "trasmud", "trasnoit", "trasnout", "traspass", "trasplant",
"trasport", "trass", "trastej", "trastemp", "trastorn", "trastroc",
"trasvas", "trasvaz", "trasvi", "trat", "trator", "traulit",
"traumatiz", "traut", "trav", "travej", "traves", "travess",
"travessur", "trebelh", "trebo", "trefeg", "trefil", "trefin",
"trein", "treit", "trejeit", "trejug", "trejur", "trelad",
"treli"+u"\u00E7", "trem", "tremebrilh", "tremel", "tremelej", "tremelhic",
"tremelic", "tremo"+u"\u00E7", "tremud", "tremul", "tremulej", "tremulin",
"tren", "trep", "trepan", "trepic", "trepid", "treplic",
"tresand", "tresbord", "trescal", "tresdobr", "tresfeg", "tresfi",
"tresfigur", "tresfil", "tresfoleg", "tresfolg", "tresfoli", "tresgast",
"tresjur", "treslad", "treslouc", "tresmalh", "tresmont", "tresmud",
"tresnoit", "tresnout", "trespass", "tressalt", "tressu", "trestamp",
"trestorn", "tresvari", "tresvi", "trevisc", "tri", "triangul",
"triangulariz", "tribaliz", "tribof", "tribul", "tribut", "tricalh",
"tricot", "trifurc", "trig", "trijunt", "tril", "trilater",
"trilh", "trimbol", "trin", "trinc", "trincafi", "trincafil",
"trinch", "trincheir", "trincolej", "trinf", "trint", "trinul",
"trip", "triplic", "tripsiniz", "tripudi", "tripul", "tris",
"trisc", "triss", "trissec", "trissect", "trissulc", "tristific",
"trit", "tritin", "tritrin", "tritur", "triunf", "trivializ",
"tro", "troc", "tro"+u"\u00E7", "trocadilh", "trocafi", "troch",
"trocisc", "tromb", "trombej", "trombet", "trombic", "trombocintilograf",
"trombos", "tromp", "trompej", "trompic", "trompilh", "tron",
"tronc", "tronch", "tronej", "trop", "trope"+u"\u00E7", "tropeg",
"tropej", "tropeli", "tropic", "tropicaliz", "troquel", "troquilh",
"trosqui", "trot", "trotej", "trotin", "trou"+u"\u00E7", "trov",
"trovej", "trovisc", "trovo", "truc", "trucid", "trucil",
"truf", "trumbic", "trunc", "trunf", "trup", "trupit",
"trusqui", "trustific", "trutin", "tub", "tuberculiniz", "tuberculiz",
"tuberific", "tuberiz", "tubuliz", "tucan", "tuf", "tumb",
"tumefic", "tumidific", "tumific", "tumul", "tumuliz", "tumultu",
"tun", "tund", "tung", "tunisific", "tur", "turb",
"turbilho", "turbilhon", "turbin", "turbolej", "turbulej", "turciz",
"turibul", "turibuliz", "turifer", "turific", "turm", "turquiz",
"turr", "turturej", "turturin", "turturinh", "turtuvi", "turv",
"turvej", "tut", "tutej", "tutel", "tuti", "tutor",
"tutuc", "tutuvi", "uber", "ufan", "ug", "ugalh",
"uiv", "ulcer", "ultim", "ultraciviliz", "ultracolaps", "ultradimension",
"ultraequip", "ultraestrutur", "ultrafiltr", "ultraj", "ultram", "ultramontaniz",
"ultraofert", "ultraotimiz", "ultrapass", "ultrapasteuriz", "ultraprov", "ultrapurific",
"ultrarromantiz", "ultrassonograf", "ulul", "umect", "umidific", "un",
"unanim", "unanimific", "unanimiz", "und", "undul", "unguent",
"unh", "unific", "uniform", "uniformiz", "unilocul", "unitiz",
"universaliz", "univoc", "unt", "untit", "up", "uralitiz",
"uraliz", "uran", "uraniz", "urbaniz", "urin", "urr",
"urtic", "urtig", "urubuz", "us", "usin", "usit",
"usm", "ustul", "usufructu", "usufrutu", "usur", "usurp",
"ut", "utensili", "utiliz", "utopiz", "vac", "vacil",
"vacin", "vacuoliz", "vadi", "vag", "vagabund", "vagamund",
"vaguej", "vai", "val", "vali", "valid", "valor",
"valoriz", "vals", "valsej", "vampir", "vampiriz", "vand",
"vandaliz", "vanglori", "vanguej", "vantaj", "vapor", "vaporiz",
"vapul", "vaquean", "vaqueir", "vaquej", "var", "vardasc",
"varej", "vari", "varicel", "varieg", "varioliz", "varrenh",
"varrisc", "varrusc", "vasc", "vascolej", "vasculariz", "vasculh",
"vasculiz", "vaselin", "vasoativ", "vasodilat", "vasquej", "vassal",
"vassoir", "vassour", "vast", "vaticin", "vavuc", "vaz",
"vazi", "vecej", "ved", "veementiz", "veget", "vegetaliz",
"vei", "veicul", "vel", "velariz", "velej", "velhac",
"velic", "velud", "velut", "venaliz", "vencelh", "vend",
"venefic", "venefici", "venen", "vener", "veniag", "venializ",
"vent", "ventanej", "ventil", "ventoinh", "ventur", "verbaliz",
"verbej", "verber", "verbet", "verdasc", "verdeg", "verdej",
"verduniz", "verg", "vergalh", "vergast", "vergonh", "verific",
"vermelh", "vermelhej", "vermicul", "vermifug", "vermin", "vermit",
"vern", "vernaculiz", "vernaliz", "veroduniz", "verrin", "verrug",
"verrum", "vers", "versej", "versific", "version", "vertebraliz",
"verticaliz", "vertigin", "vertiginiz", "vesic", "vess", "vest",
"vestirenxurr", "vet", "vetoriz", "vex", "vez", "viabiliz",
"viaj", "viand", "viatic", "vibr", "vibratiliz", "vi"+u"\u00E7",
"vicej", "vici", "vid", "vidr", "vig", "vigariz",
"vigi", "vigil", "vigor", "vigoriz", "vilegiatur", "vili",
"vilic", "vilific", "vilipendi", "vim", "vinagr", "vinc",
"vincilh", "vincul", "vindic", "vindim", "ving", "vinh",
"vinific", "vintiplic", "viol", "violent", "vir", "viravir",
"viravolt", "virginaliz", "virginiz", "virgul", "viriliz", "virtu",
"virtualiz", "vis", "visc", "viscondiz", "visg", "visibiliz",
"vision", "visit", "vislumbr", "visp", "visper", "vispor",
"vist", "vistori", "vistoriz", "visualiz", "vitalici", "vitaliz",
"vitamin", "vitaminiz", "vitim", "vitimiz", "vitori", "vitraliz",
"vitrific", "vitriol", "vitrioliz", "vitualh", "vituper", "viv",
"vivalh", "vivenci", "vivent", "vivific", "vivissect", "vizinh",
"vo", "vocabulariz", "vocabuliz", "vocacion", "vocaliz", "vocifer",
"voej", "vog", "volatiliz", "volatiz", "volc", "volit",
"volt", "voltej", "voltij", "voltiz", "volubiliz", "volum",
"voluntari", "volut", "volutu", "vomit", "vortilhon", "vot",
"vozeir", "vulcaniz", "vulgariz", "vulner", "vulnerabiliz", "vult",
"vun", "vung", "vunz", "vurm", "vuvuc", "xabouc",
"xacat", "xadrez", "xant", "xaquet", "xarop", "xavec",
"xaxat", "xen", "xerardiz", "xeret", "xeroc", "xerocopi",
"xerograf", "xerox", "xetr", "xilograf", "xilograv", "ximbic",
"xin", "xing", "xinguil", "xoc", "xon", "xot",
"xumberg", "xumbreg", "zabumb", "zagai", "zagalot", "zagunch",
"zalum", "zamp", "zamparin", "zang", "zangarilh", "zangul",
"zangurr", "zanz", "zanzarilh", "zaragat", "zaragato", "zaranz",
"zarelh", "zargunch", "zarp", "zav", "zebr", "zel",
"zer", "zi", "zich", "zimbr", "zinc", "zincograf",
"zincograv", "zing", "zingr", "zinzilul", "zip", "zirr",
"zizani", "zizi", "zo", "zomb", "zon", "zonch",
"zongol", "zonz", "zonzon", "zoograf", "zorn", "zorrag",
"zort", "zuc", "zuel", "zul", "zumb", "zumbai",
"zumbeir", "zumbr", "zun", "zung", "zunzun", "zup",
"zurr"])

# VERBOS REGULARES TERMINADOS EM CAR, GAR E CCEDILHA AR - RADICAL
plvrCVbrar = set(["a", "abar", "abel", "abgre", "abju", "abjudi",
"abjur", "able", "abne", "abo", "abode", "abou",
"abra", "abri", "acer", "ache", "acidifi", "acirolo",
"aconche", "acrian", "adar", "ade", "adel", "aden",
"ader", "adjudi", "adli", "ado", "adoci", "adre",
"advo", "aerifi", "afa", "afian", "afidal", "afin",
"aflamen", "afo", "agale", "agre", "agu", "ajindun",
"ajuden", "al", "ala", "alar", "alavan", "alber",
"alcan", "aldea", "aletar", "alfande", "alguer", "alian",
"alicer", "almece", "almo", "almonde", "alo", "alon",
"alu", "alver", "alvori", "alvoro", "ama", "amantei",
"amatun", "amea", "ami", "amol", "amole", "amoque",
"amorda", "amoris", "amplifi", "amulheren", "anafra", "apa",
"apacifi", "ape", "apla", "apli", "apote", "aproche",
"ar", "ardi", "aren", "aris", "armen", "arrai",
"arran", "arrebi", "arrega", "arrei", "arreman", "arren",
"arrene", "arrepi", "arri", "arris", "arro", "arru",
"arrua", "assosse", "ata", "atartaru", "ati", "atoi",
"atra", "atur", "au", "autenti", "autentifi", "autodesli",
"autoli", "automedi", "autopropa", "autorrevo", "avan", "avancar",
"avar", "averdu", "averdun", "azou", "ba", "bagun",
"bajo", "balan", "ban", "barre", "barri", "batu",
"beberi", "berre", "bestifi", "bexi", "bifur", "bin",
"biodiversifi", "biofortifi", "bisna", "blo", "bo", "boi",
"bol", "bombi", "bon", "bonan", "bone", "bonifi",
"bor", "borra", "borre", "borri", "borris", "bote",
"boto", "bou", "brasilifi", "bre", "bri", "brin",
"brio", "bro", "brutifi", "bubulu", "bun", "bur",
"bura", "burri", "burrifi", "bus", "butu", "buze",
"ca", "caboro", "cabou", "cabro", "cabru", "caci",
"cadu", "cafan", "cafun", "cain", "cal", "calamo",
"calcifi", "calembur", "calhan", "calorifi", "can", "candon",
"cantari", "capan", "capen", "capian", "capis", "caprifi",
"car", "caran", "carbonifi", "cardu", "cari", "carnifi",
"caro", "carre", "cartilaginifi", "cas", "caseifi", "casti",
"castifi", "catalo", "catin", "catrace", "catrapis", "catu",
"caulifi", "causti", "caustifi", "cava", "caval", "cavo",
"cavou", "cavu", "caxin", "ce", "celifi", "centrifi",
"centrifu", "centupli", "cer", "cerifi", "certifi", "cha",
"chaba", "chabou", "chala", "chambo", "chambre", "chame",
"chamus", "chan", "chanas", "chapi", "chapis", "chapu",
"chaqui", "charan", "charavis", "charin", "charris", "chas",
"chavanis", "chavas", "che", "cheiri", "cheravas", "chi",
"chian", "chin", "chis", "cho", "chorami", "choramin",
"chorin", "choroco", "chorris", "chu", "chuma", "chumbre",
"chuvenis", "chuveris", "chuvis", "cientifi", "cin", "cinefi",
"circunave", "circunva", "cis", "clarifi", "classifi", "claudi",
"cli", "clini", "clo", "co", "cobi", "cocori",
"codifi", "coedu", "coindi", "coira", "coisifi", "col",
"cole", "coli", "coliti", "colma", "colo", "comar",
"combor", "come", "comis", "compe", "complexifi", "compli",
"compur", "comun", "comuni", "con", "conche", "concul",
"condrifi", "condupli", "confis", "confri", "congra", "congre",
"conju", "consignifi", "conspur", "contrabalan", "contradan",
"contraindi", "contramar", "contrarrepli", "convo", "coobri", "coossifi",
"copides", "coqueifi", "cor", "coris", "cornifi", "corporifi",
"corri", "corru", "corus", "cosi", "cou", "coura",
"cousifi", "cretifi", "criodesse", "criti", "croni", "crucifi",
"crustifi", "cu", "cubi", "cubifi", "cucuri", "cul",
"curu", "cutu", "dan", "dan"+u"\u00E7"+"ari", "danifi", "deba",
"deban", "debi", "deboi", "debor", "debou", "debru",
"decal", "decempli", "decodifi", "decorti", "decupli", "dedi",
"defe", "defla", "deifi", "dele", "delga", "deli",
"delin", "delon", "demar", "dene", "densifi", "depeni",
"depre", "dero", "derrai", "derran", "derre", "derrei",
"derren", "derri", "derris", "derro", "dervir", "desaban",
"desabo", "desabor", "desabra", "desabri", "desache", "desacidifi",
"desaconche", "desadere", "desadvo", "desafa", "desafo", "desafus",
"desagre", "desagu", "desala", "desalar", "desalcan", "desalfande",
"desalicer", "desalo", "desalu", "desamachu", "desami", "desamol",
"desamou", "desan", "desapa", "desape", "desapli", "desapre",
"desaqueren", "desar", "desarrai", "desarran", "desarrega", "desarrei",
"desarrene", "desarri", "desarrin", "desarris", "desarru", "desassosse",
"desata", "desatas", "desatra", "desatran", "desatravan", "desau",
"desava", "desavin", "desba", "desbal", "desbalan", "desban",
"desbarran", "desbarri", "desbato", "desbei", "desbi", "desbo",
"desbor", "desbra", "desbre", "desbrin", "desburri", "desburrifi",
"desca", "descaba", "descacha", "descal", "descalcifi", "descali",
"descan", "descanji", "descar", "descarapu", "descaro", "descarre",
"descas", "descaval", "desce", "descer", "desclassifi", "desco",
"descodifi", "descol", "descolo", "descompli", "descomun", "desconche",
"desconju", "desconvo", "descorti", "desedifi", "desedu", "desema",
"desemba", "desembar", "desembara", "desembarran", "desembarri", "desembei",
"desembo", "desembor", "desemborras", "desembos", "desembra", "desembria",
"desembrus", "desembu", "desemburri", "desemo", "desempa", "desempe",
"desempefli", "desempen", "desemperti", "desempo", "desempol", "desempre",
"desemulsifi", "desencabe", "desencacha", "desencan", "desencar", "desencaran",
"desencarapu", "desencarran", "desencarre", "desencas", "desencasa", "desencava",
"desencaval", "desenchar", "desenco", "desencoira", "desencorti", "desencoura",
"desencren", "desencu", "desenfarrus", "desenfeiti", "desenfo", "desenfor",
"desenfras", "desenfus", "desenga", "desengas", "desengon", "desengra",
"desengui", "desenjor", "desenla", "desenli", "desenloda", "desenras",
"desenri", "desenros", "desenru", "desensa", "desenso", "desentalis",
"desento", "desentran", "desentrela", "desentron", "desenver", "desenvis",
"desenxer", "desequivo", "desertifi", "desesperan", "desexcomun", "desfa",
"desfabri", "desfadi", "desfal", "desfalca", "desfo", "desfol",
"desfole", "desfor", "desfortifi", "desgal", "desgaseifi", "desgasifi",
"desgra", "desgre", "desidentifi", "desimpli", "desin", "desindi",
"desinteiri", "desinterli", "desintoxi", "desintri", "desintrin", "desipote",
"desis", "desla", "deslar", "desle", "deslei", "desli",
"deslignifi", "deslinhifi", "deslo", "deslou", "desman", "desmar",
"desmaris", "desmistifi", "desmitifi", "desmo", "desmortifi", "desmultipli",
"desmunhe", "desmus", "desnal", "desnari", "desnazifi", "desne",
"desno", "desnu", "desobri", "desofus", "desoleifi", "desou",
"despaulifi", "despe", "despeda", "despen", "desperdi", "despersonifi",
"despes", "despi", "despin", "despini", "despiro", "despon",
"desposti", "despre", "despregui", "despurifi", "desqualifi", "desrai",
"desrecal", "desri", "desris", "desros", "desru", "desrussifi",
"dessal", "dessalifi", "dessalinifi", "dessamou", "desse", "dessegre",
"dessili", "dessilicifi", "desso", "dessobra", "dessosse", "dessubju",
"desta", "destabo", "destalin", "destipifi", "desto", "destoiti",
"destou", "destouti", "destoutu", "destoxifi", "destra", "destran",
"destravan", "destri", "destrin", "destro", "destron", "desu",
"desumidifi", "desunifi", "desva", "desvi", "desvidra", "desvin",
"desvir", "desvis", "desvitrifi", "deszin", "deten", "detrin",
"detrun", "devitrifi", "devul", "dezin", "dezincifi", "di",
"diagnosti", "dialo", "diferen", "dignifi", "dilar", "dirradi",
"dis", "disfar", "disgre", "disse", "distrofi", "diton",
"diva", "divari", "diversifi", "divul", "do", "domesti",
"domifi", "domin", "dorminho", "dosifi", "dra", "dro",
"dulcifi", "dupli", "e", "ear", "eburnifi", "ecolo",
"edifi", "edu", "educomuni", "edul", "ei", "elan",
"elastifi", "electrifi", "electrodiagnosti", "electromedi", "elen", "eletrifi",
"eletromedi", "elon", "ema", "ema"+u"\u00E7"+"aro", "emandin", "emanhu",
"emanjeri", "emano", "emba", "embaba", "embaga", "embal",
"embalan", "embaloi", "embalou", "embar", "embara", "embarbas",
"embarra", "embarran", "embarri", "embasba", "embato", "embatu",
"embau", "embebe", "embei", "embel", "embele", "embetes",
"embi", "embibo", "embilo", "embin", "embio", "embiro",
"embis", "embites", "embizu", "embo", "embode", "embodo",
"emboldre", "emboli", "embone", "emboni", "embor", "emborras",
"embos", "embou", "embra", "embran", "embri", "embria",
"embrin", "embro", "embrua", "embrus", "embu", "embura",
"emburri", "emburris", "emedoi", "emedou", "emendi", "emeroi",
"emerou", "eminho", "emorda", "emoroi", "emorou", "emos",
"empa", "empa"+u"\u00E7"+"o", "empalhas", "empan", "empapu", "empar",
"empaupi", "empe", "empeiti", "empeli", "empen", "emperni",
"emperti", "empes", "empete", "empetele", "empi", "empile",
"empinho", "empino", "empipo", "empis", "empla", "empo",
"empocil", "empol", "empor", "emposti", "empre", "empregui",
"empul", "emulsifi", "en", "encabe", "encacha", "encachapu",
"encaga", "encal", "encalamou", "encali", "encalun", "encama",
"encambu", "encamoroi", "encamorou", "encamur", "encan", "encani",
"encar", "encaran", "encarapu", "encario", "encarni", "encaro",
"encarran", "encarrapi", "encarras", "encarre", "encarris", "encarvoi",
"encas", "encasa", "encausti", "encava", "encaval", "encazu",
"enchapu", "enchar", "encho", "enchoiri", "enchouri", "enchuma",
"encis", "encloa", "encoira", "encolo", "encomoro", "encomoroi",
"encomorou", "encon", "enconha", "enconi", "encor", "encorri",
"encorti", "encoura", "encramoio", "encramou", "encren", "encro",
"encu", "encurri", "endere", "endil", "endomin", "endra",
"enes", "enfacho", "enfan", "enfani", "enfarrus", "enfeiti",
"enfin", "enfiteuti", "enfo", "enfoi", "enfor", "enfou",
"enfra", "enfraldi", "enfras", "enfuma", "enfunis", "enfus",
"enga", "engal", "engali", "engalri", "engas", "engati",
"engero", "engi", "engon", "engra", "engui", "enjal",
"enjan", "enji", "enjor", "enju", "enjun", "enla",
"enlabrus", "enlaru", "enli", "enlo", "enloda", "enloi",
"enlou", "eno", "enoiri", "enouri", "enra", "enran",
"enras", "enre", "enredoi", "enredou", "enri", "enro",
"enrodri", "enros", "enru", "ensa", "ensal", "ense",
"enses", "enso", "ensofre", "ensova", "entabi", "entabo",
"entalei", "entali", "entalin", "entalis", "entan", "entani",
"entapi", "ente", "enteju", "enti", "entiju", "entisi",
"ento", "entoi", "entou", "entrambeli", "entrambi", "entran",
"entravis", "entre", "entrecho", "entrela", "entreli", "entreloi",
"entrelou", "entrepi", "entrepli", "entrerre", "entressa", "entressufo",
"entreto", "entrin", "entro", "entron", "entrope", "entropi",
"entrovis", "entrupi", "entuju", "enturvis", "envelha", "enver",
"enverru", "enves", "envi", "envidra", "envis", "enxal",
"enxaque", "enxau", "enxe", "enxer", "enxo", "enxu",
"epigramati", "epilo", "equivo", "eri", "erradi", "erri",
"es", "esbaba", "esbaga", "esbalan", "esbalou", "esban",
"esbarran", "esbarre", "esbarri", "esbarro", "esbei", "esbeltifi",
"esbeni", "esberra", "esbi", "esbo", "esbode", "esboi",
"esborra", "esborre", "esbou", "esbra", "esbranqui", "esbrin",
"esbron", "esbru", "esbur", "esbura", "esca", "escabe",
"escachou", "escadra", "escaga", "escalaman", "escalamou", "escali",
"escamur", "escamurren", "escan", "escane", "escar", "escaramu",
"escarapu", "escarda", "escardi", "escardu", "escarifi", "escarmou",
"escarni", "escarnifi", "escaro", "escarou", "escarrapi", "escarvan",
"escarvun", "escas", "escava", "esclamou", "esclerifi", "esco",
"escoi", "escor", "escorifi", "escorra", "escorre", "escorrima",
"escorti", "escou", "escozi", "escramou", "escramu", "escul",
"esfal", "esfalca", "esfande", "esfani", "esfarrun", "esfel",
"esflo", "esfoi", "esfole", "esfolha", "esfor", "esformi",
"esforni", "esforri", "esfou", "esfran", "esfre", "esfregon",
"esfuma", "esfura", "esfuran", "esfutri", "esga", "esgal",
"esgani", "esgar", "esgati", "esla", "eslin", "esma",
"esmadri", "esmar", "esmarne", "esmarno", "esmartu", "esmatru",
"esmija", "esmiun", "esmo", "esmoi", "esmon", "esmor",
"esmorda", "esmordi", "esmorra", "esmou", "esmurra", "esmurre",
"esmurren", "esna", "esno", "espa", "espada", "espaje",
"espalha", "espan", "espanas", "espandon", "espapa", "esparre",
"espassari", "espe", "especifi", "espectrifi", "espeda", "espedar",
"espedre", "espeni", "esperan", "esperdi", "esperme", "esperne",
"espesco", "espetrifi", "espi", "espica", "espicoi", "espicou",
"espin", "espini", "espipo", "espo", "esporifi", "espotri",
"espou", "espregui", "espreme", "espul", "espuma", "esquadra",
"esqui", "esquistifi", "esta", "estabou", "estafe", "estali",
"estan", "estardalha", "estarrifi", "estarrin", "estatistifi", "esteli",
"ester", "esterifi", "esterli", "esterni", "esti", "estilha",
"estin", "estira", "esto", "estoma", "estome", "estorce",
"estorci", "estorde", "estorne", "estorni", "estorri", "estorte",
"estra", "estrabou", "estrafe", "estran", "estrapa", "estratifi",
"estre", "estreban", "estrefe", "estreli", "estrelou", "estri",
"estrin", "estro", "estroci", "estron", "estrufe", "estrumbi",
"estu", "estultifi", "estupefi", "estupidifi", "esturri", "esver",
"esvidi", "esvoa", "eterifi", "evo", "exal", "exatifi",
"excarnifi", "excomun", "exemplifi", "exgre", "exiguifi", "expli",
"expur", "exsi", "extri", "fa", "fabri", "fadi",
"fal", "fala", "falca", "falho", "falo", "falsifi",
"falu", "fandan", "fandin", "fani", "fanu", "faris",
"farras", "farron", "farun", "fati", "femen", "fi",
"fin", "fis", "fitu", "flo", "flogisti", "fluidifi",
"fo", "fofo", "foi", "fol", "fole", "folhe",
"for", "forji", "formi", "forni", "forri", "forta",
"fortale", "fortifi", "fos", "fossan", "fossilifi", "fotomultipli",
"fou", "fra", "fraldi", "fre", "freju", "fres",
"frigorifi", "fro", "frute", "frutifi", "fu", "fube",
"fuji", "fule", "fuma", "fume", "fumi", "fun",
"fura", "furdun", "furti", "furun", "fus", "fusti",
"futi", "futri", "futu", "fuxi", "gadi", "gal",
"gali", "gan", "gani", "gar", "garan", "gargada",
"gargala", "gaseifi", "gasifi", "gelatinifi", "geleifi", "gelifi",
"gemel", "gemelhi", "gemi", "gin", "ginasti", "girova",
"gla", "glorifi", "gomifi", "gon", "gou", "gramati",
"granitifi", "gratifi", "gron", "guas", "guen", "hermetifi",
"heroifi", "heterocarre", "hiperapli", "hipote", "hispanifi", "homolo",
"honorifi", "honrifi", "horrifi", "hulhifi", "humifi", "i",
"identifi", "idiotifi", "ignifi", "ignifu", "ili", "imbeli",
"imbi", "imbri", "impel", "impersonifi", "impli", "impre",
"impurifi", "in", "incomuni", "incul", "incurri", "inda",
"indi", "indicifi", "indignifi", "indil", "infin", "infuni",
"injun", "injusti", "injustifi", "inli", "insti", "intei",
"inteiri", "intensifi", "intercomuni", "interesterifi", "interestratifi", "interi",
"interla", "interli", "interpi", "interro", "interse", "inti",
"intoxi", "intri", "intrin", "investi", "invo", "invul",
"irri", "irro", "is", "jan", "jantari", "jazzifi",
"jo", "jon", "jornalisti", "ju", "judi", "jul",
"jun", "jur", "justi", "justifi", "la", "labre",
"laicifi", "lambaris", "lambi", "lambis", "lambre", "lambu",
"lan", "langu", "lapidifi", "lar", "las", "late",
"lati", "latin", "lavaris", "lavas", "lave", "le",
"lei", "len", "lengalen", "lenhifi", "lenifi", "letifi",
"levi", "li", "lignifi", "lin", "linifi", "liquenifi",
"liquidifi", "liri", "lis", "liti", "litifi", "lo",
"lobre", "lobri", "logi", "lombri", "lor", "lu",
"lubri", "lubrifi", "lus", "luscofus", "lusifi", "ma",
"maca", "ma"+u"\u00E7"+"aro", "macaven", "machi", "machu", "macu",
"madefi", "madra", "madru", "magi", "magnifi", "malo",
"malu", "man", "mandin", "mandu", "manjeri", "mano",
"mar", "maris", "marri", "marru", "mas", "massifi",
"masti", "mastu", "mati", "matra", "matru", "maturran",
"mazur", "medi", "mei", "melifi", "men", "mendi",
"menospre", "mer", "mercurifi", "merendu", "mes", "mesti",
"metacomuni", "metafisi", "metalifi", "metodifi", "metrifi", "mexeri",
"mexeru", "mi", "mimi", "min", "minho", "minifi",
"ministrifi", "mirifi", "mis", "mistifi", "miti", "mitifi",
"mo", "modi", "modifi", "mogan", "mogi", "moji",
"mol", "mole", "molen", "molifi", "mon", "monolo",
"monoton", "moque", "mor", "moran", "morce", "morda",
"mordi", "mordis", "morra", "morse", "mortifi", "mos",
"mosse", "mostren", "mou", "mouris", "muchi", "mufi",
"mugan", "multipli", "mumifi", "mundifi", "mungan", "munhe",
"mupi", "mupu", "mus", "musi", "muxi", "na",
"nadifi", "namori", "namoris", "naufra", "nave", "nazifi",
"ne", "nega", "nene", "nes", "nevis", "nevoa",
"ni", "nidifi", "niilifi", "nitidifi", "nitrifi", "no",
"nobilifi", "noctiva", "notifi", "nuan", "nulifi", "nutrifi",
"o", "obce", "objectifi", "objetifi", "objur", "obri",
"octupli", "ofe", "ofus", "oi", "oiri", "opa",
"opacifi", "or", "organifi", "orti", "ortorretifi", "ossifi",
"ou", "ouri", "outor", "ozonifi", "pa", "pacifi",
"padu", "palan", "palhi", "palifi", "pan", "pana",
"pande", "panegiri", "panifi", "papagu", "papari", "papo",
"papou", "paraliti", "parifi", "parlen", "parran", "pate",
"patus", "paulifi", "pe", "pechirin", "pedi", "penduri",
"peni", "penifi", "penis", "per", "percal", "percoru",
"perdi", "perere", "peri", "perlen", "perlon", "perren",
"personifi", "perva", "pes", "pespe", "petanis", "pete",
"petis", "petrifi", "pfu", "pi", "pian", "piciri",
"pin", "pindo", "pindon", "pini", "pino", "pintal",
"pipiri", "pipo", "piran", "piriri", "piro", "pirra",
"pis", "pitos", "piu", "planifi", "plasti",
"plastifi", "plenifi", "pli", "plu", "pluriestratifi", "plurifi",
"pluristratifi", "po", "poetifi", "pol", "pole", "polemi",
"polirramifi", "politi", "pomifi", "pon", "pontifi", "popo",
"pororo", "porvis", "poste", "poster", "posti", "poto",
"pra", "prati", "pre", "preale", "precal", "precifi",
"predemar", "predi", "prefabri", "pregui", "preite", "prejudi",
"prenuni", "presentifi", "presi", "pretra", "prevari", "pris",
"prodi", "profa", "profal", "profli", "prognosti", "prolifi",
"prolo", "prolon", "promul", "pronosti", "prontifi", "propa",
"prorro", "prosifi", "prototifi", "provis", "provo", "pseudojustifi",
"psicolo", "publi", "puli", "pun", "pur", "purifi",
"putrifi", "quadrupli", "qualifi", "quantifi", "qui", "quilifi",
"quimifi", "quin", "quintupli", "rabis", "rabus", "radi",
"radiodiagnosti", "radiomar", "radionave", "radiopre", "raman", "ramifi",
"ramo", "ran", "rancidifi", "rancifi", "rapi", "ras",
"rasmol", "raspan", "ratifi", "re", "reabra", "readjudi",
"real", "realar", "realo", "realu", "reapli", "reati",
"reautenti", "rebalan", "rebeli", "rebi", "rebis", "rebo",
"rebol", "reboli", "rebou", "rebu", "rebuli", "rebus",
"recal", "recalcifi", "recar", "recarre", "recaval", "recha",
"reche", "recipro", "reclassifi", "reco", "recoli", "recolo",
"recome", "recongra", "reconvo", "recor", "recrucifi", "rededi",
"reden", "redoi", "redou", "redupli", "reedifi", "reedu",
"reembar", "reempol", "reempor", "reempre", "reencu", "reendere",
"reenla", "reescramou", "reescramu", "reespecifi", "reexpli", "refe",
"refin", "refo", "refol", "refole", "refor", "refortifi",
"refre", "refres", "refri", "refu", "refun", "refunfume",
"rega", "regaseifi", "regou", "reguin", "reifi", "reinterro",
"reinvo", "reivindi", "rejun", "rejustifi", "relan", "relar",
"rele", "reli", "relin", "relo", "relou", "reman",
"remanis", "remar", "remas", "remasti", "remel", "remeni",
"remo", "remol", "remor", "remordi", "remordis", "remou",
"remu", "remun", "remus", "renave", "reobri", "repa",
"repe", "repeni", "repes", "repi", "repil", "repini",
"repis", "repli", "repol", "repole", "repre", "republi",
"repur", "repurifi", "requalifi", "rerratifi", "res", "resfol",
"resgal", "resinifi", "resmelen", "resmol", "resmun", "resmus",
"respan", "respi", "respin", "ressa", "resse", "ressi",
"ressul", "restauri", "restru", "restu", "reta", "retifi",
"retipifi", "reto", "retoi", "retori", "retou", "retra",
"retran", "retrin", "retrocarre", "retru", "reunifi", "rever",
"reverifi", "revi", "revin", "revindi", "revivifi", "revo",
"revol", "rezin", "ri", "ricer", "ridi", "rin",
"ripan", "ris", "ro", "ro"+u"\u00E7"+"a", "roce", "rodri",
"roli", "romanes", "ron", "ros", "rou", "ru",
"rubifi", "rubri", "rus", "russifi", "rusti", "rustifi",
"sa", "sabere", "saberere", "sabre", "sabu", "sa"+u"\u00E7"+"ari",
"sacarifi", "sacrifi", "sal", "salifi", "salpi", "saltari",
"salu", "salubrifi", "saman", "sambar", "sambo", "sanguessu",
"sanguifi", "sani", "sanifi", "santi", "santifi", "sape",
"sapere", "saponifi", "sarafus", "sarapi", "sarrabis", "sarrafa",
"saxifi", "se", "sebenis", "segre", "selvati", "semiapli",
"semidomesti", "sen", "sensifi", "septupli", "seraman", "serin",
"serraman", "setupli", "sextupli", "signifi", "silabifi", "silicifi",
"silogisti", "simplifi", "sindi", "sinte", "sinu", "sir",
"siriri", "so", "sobal", "sobne", "sobra", "sobran",
"sobrecasa", "socal", "so"+u"\u00E7"+"o", "sofisti", "sofre",
"soji", "soju", "sola", "solan", "solavan", "sole",
"solidifi", "solu", "somiti", "sone", "sope", "sorbi",
"sororo", "sorras", "sosse", "sotran", "soupi", "su",
"subapli", "subclassifi", "subelis", "subenfiteuti", "suberifi", "subespecifi",
"subju", "sublo", "substantifi", "sudorifi", "sufo", "sufra",
"sufumi", "suji", "suju", "sul", "sulan", "sun",
"superapli", "superdiagnosti", "supli", "sur", "suru", "sururu",
"ta", "tabi", "tabu", "tairo", "talin", "taman",
"tamus", "tan", "tanifi", "tape", "tapi", "tarifi",
"taro", "tarou", "tarrin", "tartufi", "taru", "tas",
"te", "telecomuni", "tenrifi", "ter", "terrifi", "terrin",
"testi", "testifi", "testu", "ti", "tibun", "tin",
"tipi", "tipifi", "tiriri", "tisi", "titi", "titiri",
"to", "toi", "tonifi", "torce", "torri", "torrifi",
"torris", "tos", "tosse", "tossi", "toste", "tou",
"toxi", "tra", "trabalhis", "trabalhu", "trabu", "trafe",
"trafi", "trambe", "trambi", "tran", "transcodifi", "transfe",
"transignifi", "translo", "translou", "transtro", "trapa", "trasfe",
"trastro", "trefe", "treju", "treli", "tremelhi", "tremeli",
"tremo", "trepi", "trepli", "tresfe", "tresfol", "tresfole",
"treslou", "trevis", "tri", "trifur", "trin", "tripli",
"tris", "trisse", "trissul", "tristifi", "tro", "trocis",
"trombi", "trompi", "tron", "trope", "tropi", "trou",
"trovis", "tru", "trumbi", "trun", "trustifi", "tuberifi",
"tumefi", "tumidifi", "tumifi", "tun", "tunisifi", "turifi",
"tutu", "u", "ultrapurifi", "umidifi", "unanimifi", "unifi",
"univo", "urti", "va", "vardas", "varie", "varris",
"varrus", "vas", "vavu", "velha", "veli", "venefi",
"venia", "ver", "verdas", "verde", "verifi", "vermifu",
"verru", "versifi", "vesi", "vi", "viati", "vili",
"vilifi", "vin", "vindi", "vinifi", "vintipli", "vis",
"vitrifi", "vivifi", "vo", "vol", "vun", "vuvu",
"xabou", "xave", "xero", "ximbi", "xin", "xo",
"xumber", "xumbre", "zan", "zin", "zorra", "zu",
"zun"])

# Palavras compostas, plural - singular - publicos-alvo ,  arquivos-fonte
tCompostops = set(["abar"+u"\u00E9"+"s-tucura", "abatis-timba"+u"\u00ED", "abelhas-cachorro", "abelhas-crian"+u"\u00E7"+"a",
"abelhas-europa", "abelhas-flor", "abelhas-irapu"+u"\u00E1", "abelhas-irapu"+u"\u00E3",
"abelhas-lim"+u"\u00E3"+"o", "abelhas-m"+u"\u00E3"+"e", "abelhas-mosquito", "abelhas-ouro",
"abelhas-rainha", "abelhas-uru"+u"\u00E7"+"u", "abioranas-guta", "abiuranas-guta",
"ac"+u"\u00E1"+"cias-angico", "ac"+u"\u00E1"+"cias-parassol", "a"+u"\u00E7"+"afr"+u"\u00F5"+"es-palhinha", "a"+u"\u00E7"+"a"+u"\u00ED"+"s-catinga",
"a"+u"\u00E7"+"a"+u"\u00ED"+"s-chumbo", "acajus-catinga", "acampamentos-base", "acar"+u"\u00E1"+"s-ai"+u"\u00E1",
"acar"+u"\u00E1"+"s-apu"+u"\u00E3", "acar"+u"\u00E1"+"s-bandeira", "acar"+u"\u00E1"+"s-bararu"+u"\u00E1", "acar"+u"\u00E1"+"s-berer"+u"\u00EA",
"acar"+u"\u00E1"+"s-bobo", "acar"+u"\u00E1"+"s-camale"+u"\u00E3"+"o", "acar"+u"\u00E1"+"s-diadema", "acar"+u"\u00E1"+"s-disco",
"acar"+u"\u00E1"+"s-fantasma", "acar"+u"\u00E1"+"s-ferro", "acar"+u"\u00E1"+"s-fuma"+u"\u00E7"+"a", "acar"+u"\u00E1"+"s-fuso",
"acar"+u"\u00E1"+"s-mana"+u"\u00E7"+"arav"+u"\u00E9", "acar"+u"\u00E1"+"s-moc"+u"\u00F3", "acar"+u"\u00E1"+"s-papagaio", "acar"+u"\u00E1"+"s-pataquira",
"acar"+u"\u00E1"+"s-pinima", "acar"+u"\u00E1"+"s-prata", "acar"+u"\u00E1"+"s-salema", "acar"+u"\u00E1"+"s-saveiro",
"acar"+u"\u00E1"+"s-til"+u"\u00E1"+"pia", "acar"+u"\u00E1"+"s-topete", "acaris-cacimba", "acaris-chicote",
"acaris-espada", "acaris-laranja", "acaris-lima", "a"+u"\u00E7"+"os-carbono",
"a"+u"\u00E7"+"os-liga", "a"+u"\u00E7"+"ordas-mona", "a"+u"\u00E7"+u"\u00FA"+"cares-cande", "a"+u"\u00E7"+u"\u00FA"+"cares-c"+u"\u00E2"+"ndi",
"a"+u"\u00E7"+u"\u00FA"+"cares-pedra", "acutis-tapuia", u"\u00E1"+"gnus-dei", u"\u00E1"+"guas-belense",
u"\u00E1"+"guas-formosense", u"\u00E1"+"guas-mato", u"\u00E1"+"guas-mel", u"\u00E1"+"guas-morna",
u"\u00E1"+"guas-mornense", u"\u00E1"+"guas-tinta", u"\u00E1"+"guas-vermelhense", "agulhas-m"+u"\u00E3"+"e",
"agulhas-padr"+u"\u00E3"+"o", "agulh"+u"\u00F5"+"es-bandeira", "agulh"+u"\u00F5"+"es-trombeta", "aipos-r"+u"\u00E1"+"bano",
"alas-piv"+u"\u00F4", "albacoras-bandolim", "algod"+u"\u00F5"+"es-col"+u"\u00F3"+"dio", "algod"+u"\u00F5"+"es-cravo",
"algod"+u"\u00F5"+"es-macaco", "algod"+u"\u00F5"+"es-p"+u"\u00F3"+"lvora", "alhos-rocambole", "almo"+u"\u00E7"+"os-debate",
"amarelos-can"+u"\u00E1"+"rio", "amarelos-cidr"+u"\u00E3"+"o", "amarelos-cobalto", "amarelos-enxofre",
"amarelos-laranja", "amarelos-lim"+u"\u00E3"+"o", "amarelos-oca", "amarelos-ocre",
"amarelos-oiro", "amarelos-ouro", "amarelos-palha", "amarelos-top"+u"\u00E1"+"zio",
"amendoeiras-coca", "amoreiras-cip"+u"\u00F3", "amoreiras-tata"+u"\u00ED"+"ba", "amostras-tipo",
"amperes-espira", "amperes-giro", "amperes-hora", "amperes-segundo",
"amperes-volta", "amp"+u"\u00E9"+"rios-espira", "amp"+u"\u00E9"+"rios-giro", "amp"+u"\u00E9"+"rios-hora",
"amp"+u"\u00E9"+"rios-segundo", "amp"+u"\u00E9"+"rios-volta", "anamb"+u"\u00E9"+"s-piti"+u"\u00FA", "anamb"+u"\u00E9"+"s-pombo",
"anamb"+u"\u00E9"+"s-sol", "anan"+u"\u00E1"+"s-selvagem", "anan"+u"\u00E1"+"s-silvestre", "andins-palmeira",
"andirobas-jareu"+u"\u00E1", "andores-imp"+u"\u00E9"+"rio", "angelins-araroba", "angelins-coco",
"angelins-pedra", "angelins-rosa", "angicos-barbatim"+u"\u00E3"+"o", "angicos-cangalha",
"angicos-cedro", "angicos-rosa", "angicos-surucucu", "angonas-pu"+u"\u00ED"+"ta",
"anis-doce", "anis-estrelado", "anis-verde", "animais-flor",
"animais-m"+u"\u00E1"+"quina", "animais-planta", "anjos-viola", "anos-base",
"anos-calend"+u"\u00E1"+"rio", "anos-luz", "an"+u"\u00FA"+"ncios-sandu"+u"\u00ED"+"che", "ara"+u"\u00E7"+u"\u00E1"+"s-congonha",
"ara"+u"\u00E7"+u"\u00E1"+"s-cot"+u"\u00E3"+"o", "ara"+u"\u00E7"+u"\u00E1"+"s-lima",
"ara"+u"\u00E7"+u"\u00E1"+"s-pedra", "ara"+u"\u00E7"+u"\u00E1"+"s-rosa",
"ara"+u"\u00E7"+"aris-banana", "ara"+u"\u00E7"+"aris-minhoca", "arai"+u"\u00E1"+"s-garapa", "aranhas-bode",
"aranhas-caranguejeira", "aranhas-caranguejo", "araras-canind"+u"\u00E9", "araras-chauanau"+u"\u00E1",
"araras-macau", "araras-piti"+u"\u00FA", "ararutas-palmeira", "araticuns-ap"+u"\u00EA",
"araticuns-caca", "araticuns-catinga", "araticuns-corti"+u"\u00E7"+"a", "araticuns-pitai"+u"\u00E1",
u"\u00E1"+"reas-alvo", u"\u00E1"+"reas-cliente", "arecas-bambu", "arecas-bangu"+u"\u00E1",
"areias-manteiga", "aroeiras-mucuri", "aroeiras-salso", "Arquivos-fonte", "arquivos-fonte",
"arquivos-texto", "arraias-acau"+u"\u00E3", "arraias-bor"+u"\u00F3", "arraias-cocal",
"arraias-papagaio", "arraias-santo", "arrozes-agulha", "arrozes-cateto",
"arrudas-pariet"+u"\u00E1"+"ria", u"\u00E1"+"rvores-m"+u"\u00E3"+"e", u"\u00E1"+"rvores-vaca", "asas-delta",
"asilos-escola", "assis-brasiliense", "assuntos-chave", "aterros-barragem",
"atividades-fim", "atividades-meio", "atos-show", u"\u00E1"+"tomos-grama",
"atuns-bandolim", "aux"+u"\u00ED"+"lios-alimenta"+u"\u00E7"+u"\u00E3"+"o", "Aux"+u"\u00ED"+"lios-Alimenta"+u"\u00E7"+u"\u00E3"+"o", "aux"+u"\u00ED"+"lios-creche",
"aux"+u"\u00ED"+"lios-doen"+u"\u00E7"+"a", "aux"+u"\u00ED"+"lios-enfermidade", "aux"+u"\u00ED"+"lios-funeral", "aux"+u"\u00ED"+"lios-maternidade",
"aux"+u"\u00ED"+"lios-moradia", "aux"+u"\u00ED"+"lios-natalidade", "aux"+u"\u00ED"+"lios-palet"+u"\u00F3", "aux"+u"\u00ED"+"lios-reclus"+u"\u00E3"+"o",
"aux"+u"\u00ED"+"lios-sa"+u"\u00FA"+"de", "aux"+u"\u00ED"+"lios-transporte", "aves-elefante", "aves-lira",
"avi"+u"\u00F5"+"es-correio", "avi"+u"\u00F5"+"es-espi"+u"\u00E3"+"o", "avi"+u"\u00F5"+"es-foguete", "avi"+u"\u00F5"+"es-laborat"+u"\u00F3"+"rio",
"avi"+u"\u00F5"+"es-tanque", "azeitonas-rei", "azereiros-pado", "azuis-cobalto",
"azuis-faian"+u"\u00E7"+"a", "azuis-ferrete", "azuis-pav"+u"\u00E3"+"o", "azuis-piscina",
"azuis-pombinho", "azuis-seda", "azuis-turquesa", "azul"+u"\u00F5"+"es-boia",
"bacus-pedra", "bacuparis-cip"+u"\u00F3", "bacuraus-pitu"+u"\u00ED", "bacuraus-tesoura",
"bacurubus-ficheira", "badejos-ferro", "badejos-fogo", "badejos-mira",
"badejos-padre", "badejos-sab"+u"\u00E3"+"o", "badejos-salt"+u"\u00E3"+"o", "badejos-sangue",
"bagres-aria"+u"\u00E7"+"u", "bagres-bandeira", "bagres-cachola", "bagres-cacumo",
"bagres-caiacoco", "bagres-cambeja", "bagres-cangat"+u"\u00E1", "bagres-catinga",
"bagres-fita", "bagres-gaivota", "bagres-gonguito", "bagres-guri",
"bagres-leil"+u"\u00E3"+"o", "bagres-mandi", "bagres-mandim", "bagres-morcego",
"bagres-papai", "bagres-sapipoca", "bagres-sapo", "bagres-sari",
"bagres-surubim", "bagres-urutu", "bagres-veludo", "baiacus-areia",
"baiacus-caix"+u"\u00E3"+"o", "baiacus-cofre", "baiacus-dondom", "baiacus-espinho",
"baiacus-franguinho", "baiacus-gorajuba", "baiacus-graviola", "baiacus-guaiama",
"baiacus-guaiam"+u"\u00E1", "baiacus-guarajuba", "baiacus-panela", "baiacus-pinima",
"baiacus-pininga", "baiacus-pintado", "baleias-dente", "baleias-espadarte",
"baleias-minke", "baleias-pamonha", "baleias-piloto", "baleias-sei",
"bal"+u"\u00F5"+"es-piloto", "bal"+u"\u00F5"+"es-sonda", "b"+u"\u00E1"+"lsamos-caboriba", "b"+u"\u00E1"+"lsamos-coral",
"bambus-balde", "bambus-bengala", "bambus-cana", "bambus-taquara",
"ban"+u"\u00E1"+"s-muela", "bananas-figo", "bananas-gabu", "bananas-inaj"+u"\u00E1",
"bananas-mac"+u"\u00E3", "bananas-ma"+u"\u00E7"+u"\u00E3", "bananas-m"+u"\u00E3"+"e", "bananas-m"+u"\u00E2"+"nea",
"bananas-menina", "bananas-naj"+u"\u00E1", "bananas-ouro", "bananas-pacova",
"bananas-pacov"+u"\u00E1", "bananas-p"+u"\u00E3"+"o", "bananas-pedra", "bananas-prata",
"bananas-timb"+u"\u00F3", "bananeirinhas-zebra", "banhos-maria", "b"+u"\u00E1"+"rbus-sumatrano",
"b"+u"\u00E1"+"rbus-ticto", "barcas-farol", "barca"+u"\u00E7"+"as-cisterna", "barcos-armadilha",
"barcos-autom"+u"\u00F3"+"vel", "barcos-bomba", "barcos-drag"+u"\u00E3"+"o", "barcos-farol",
"barcos-luz", "barreiras-chave", "batatas-aipo", "batatas-atum",
"batatas-cear"+u"\u00E1", "batatas-cenoura", "batatas-cogumelo", "batatas-ovo",
"batatas-salsa", "ba"+u"\u00FA"+"nas-fogo", "beg"+u"\u00F4"+"nias-bengala", "beg"+u"\u00F4"+"nias-sangue",
"beijos-pir"+u"\u00E1", "beijus-moqueca", "beijus-poqueca", "berdelhos-feij"+u"\u00E3"+"o",
"berimbaus-viola", "besourinhos-ametista", "besouros-bola", "besouros-mangang"+u"\u00E1",
"besouros-rinoceronte", "bias-fortense", "bichas-alfinete", "bichos-bolo",
"bichos-cadela", "bichos-engenheiro", "bichos-folha", "bichos-lixeiro",
"bichos-palha", "bichos-pau", "bichos-pregui"+u"\u00E7"+"a", "bichos-trem",
"bichos-tutu", "bicos-pimenta", "bicos-ser"+u"\u00E1", "bicudas-mangal"+u"\u00F4",
"bicudos-maquin"+u"\u00E9", "bicudos-pimenta", "Bioc"+u"\u00F3"+"digos-fonte", "bioc"+u"\u00F3"+"digos-fonte",
"bisnagas-bengala",
"boais-bonif"+u"\u00E1"+"cio", "boais-natura", "bodi"+u"\u00F5"+"es-batata", "bodi"+u"\u00F5"+"es-lamego",
"bodi"+u"\u00F5"+"es-papagaio", "bodi"+u"\u00F5"+"es-sabonete", "bodi"+u"\u00F5"+"es-tucano", "bois-bumb"+u"\u00E1",
"bois-calemba", "bois-calumba", "bois-cavalo", "bois-culemba",
"bois-esp"+u"\u00E1"+"cio", "bois-espa"+u"\u00E7"+"o", "bombas-foguete", "bombas-granada",
"bombas-parafuso", "bombas-rel"+u"\u00F3"+"gio", "bombas-turbina", "bonecos-palito",
"bonitos-cachorro", "bonitos-can"+u"\u00E1"+"rio", "bonitos-cavala", "bor"+u"\u00E1"+"s-boi",
"bor"+u"\u00E1"+"s-cavalo", "borboletas-azuis-seda", "borboletas-carij"+u"\u00F3", "borboletas-coruja",
"borboletas-espelho", "borboletas-folha", "borboletas-imperador", "borboletas-jandaia",
"borboletas-oitenta", "borboletas-rubi", "botos-cinza", "botos-tucuxi",
"bot"+u"\u00F5"+"es-polaina", "brancos-parreira", "bredos-caruru", "bredos-malho",
"bredos-mama", "bredos-raba"+u"\u00E7"+"a", "breus-jauaricica", "breus-sucuruba",
"brigues-barca", "brigues-escuna", "bugios-labareda", "buritis-palito",
"buritis-palmito", "cabas-beiju", "cabas-camale"+u"\u00E3"+"o", "cabas-mutuca",
"caba"+u"\u00E7"+"as-purunga", "caba"+u"\u00E7"+"os-grogoj"+u"\u00F3", "caba"+u"\u00E7"+"os-marimba", "cabe"+u"\u00E7"+"as-duras-prego",
"cabe"+u"\u00E7"+"as-duras-rel"+u"\u00F3"+"gio", "cabe"+u"\u00E7"+"as-pitanga", "cabos-guia", "cabras-cabriola",
"cacaus-jacar"+u"\u00E9", "cacaus-maracuj"+u"\u00E1", "cacaueiros-laranja", "cacaueiros-suriname",
"cacaueiros-trindade", "cacaueiros-venezuela", "cach"+u"\u00F5"+"es-porango", "cachorros-rabeca",
"cactos-mel"+u"\u00E3"+"o", "cactos-ouri"+u"\u00E7"+"o", "cactos-rosa", "Cades-Barneia",
"caf"+u"\u00E9"+"s-bucha", "caf"+u"\u00E9"+"s-cabe"+u"\u00E7"+"a", "caf"+u"\u00E9"+"s-caneca", "caf"+u"\u00E9"+"s-chifre",
"caf"+u"\u00E9"+"s-concerto", "caf"+u"\u00E9"+"s-concha", "caf"+u"\u00E9"+"s-conosco", "caf"+u"\u00E9"+"s-escolha",
"caf"+u"\u00E9"+"s-moca", "caf"+u"\u00E9"+"s-teatro", "caixas-bomba", "caj"+u"\u00E1"+"s-manga",
"calorias-grama", "camas-beliche", "camadas-limite", "camale"+u"\u00F5"+"es-caba",
"camale"+u"\u00F5"+"es-ferro", "camar"+u"\u00F5"+"es-ferro", "camar"+u"\u00F5"+"es-lagosta", "camar"+u"\u00F5"+"es-lixo",
"camar"+u"\u00F5"+"es-pistola", "camar"+u"\u00F5"+"es-rosa", "camar"+u"\u00F5"+"es-sossego", "cambar"+u"\u00E1"+"s-pitanga",
"cambar"+u"\u00E1"+"s-rosa", "cambu"+u"\u00ED"+"s-bala", "caminh"+u"\u00F5"+"es-ba"+u"\u00FA", "caminh"+u"\u00F5"+"es-betoneira",
"caminh"+u"\u00F5"+"es-bomba", "caminh"+u"\u00F5"+"es-fossa", "caminh"+u"\u00F5"+"es-pipa", "caminh"+u"\u00F5"+"es-tanque",
"camomilas-catinga", "camorins-sovela", "campos-altense", "campos-geraiense",
"campos-novense", "camurins-corcunda", "camurins-flecha", "camurins-sovela",
"camurins-tapa", "camurins-ticop"+u"\u00E1", "canas-capim", "canas-manteiga",
"cananas-capeta", "can"+u"\u00E1"+"rios-assobio", "can"+u"\u00E1"+"rios-baeta", "candirus-caju",
"canelas-abacate", "canelas-anhu"+u"\u00ED"+"ba", "canelas-bagre", "canelas-bara"+u"\u00FA"+"na",
"canelas-batalha", "canelas-batata", "canelas-bezerro", "canelas-bibiru",
"canelas-bra"+u"\u00FA"+"na", "canelas-caixeta", "canelas-catigu"+u"\u00E1", "canelas-cedro",
"canelas-cravo", "canelas-ferrugem", "canelas-fogo", "canelas-funcho",
"canelas-goiaba", "canelas-goiac"+u"\u00E1", "canelas-gosma", "canelas-guac"+u"\u00E1",
"canelas-guaiac"+u"\u00E1", "canelas-guaic"+u"\u00E1", "canelas-imbuia", "canelas-inha"+u"\u00ED"+"ba",
"canelas-inha"+u"\u00FA"+"ba", "canelas-jacu", "canelas-japu", "canelas-lim"+u"\u00E3"+"o",
"canelas-louro", "canelas-marmelada", "canelas-massap"+u"\u00EA", "canelas-mate",
"canelas-merda", "canelas-mescla", "canelas-muri"+u"\u00E1", "canelas-murici",
"canelas-oiti", "canelas-pimenta", "canelas-poca", "canelas-pororoca",
"canelas-prego", "canelas-rapadura", "canelas-rosa", "canelas-samambaia",
"canelas-seiva", "canelas-tapinha", "canelas-tapinho"+u"\u00E3", "canelas-tatu",
"caneleiras-cravo", "canelinhas-ub"+u"\u00E1", "Canetas-fonte", "canetas-fonte", "canetas-tinteiro",
"cangas-rosa", "cangulos-bai"+u"\u00E9", "cangulos-rei", "cangurus-rato",
"canh"+u"\u00F5"+"es-arp"+u"\u00E3"+"o", "canh"+u"\u00F5"+"es-rev"+u"\u00F3"+"lver", "canjicas-piruruca", "canoas-rosa",
"capas-capote", "capas-gorja", "capas-palhi"+u"\u00E7"+"a", "capas-palho"+u"\u00E7"+"a",
"capas-pele", "capas-saia", "capas-zero", "capacetinhos-cinza",
"capelas-carapu"+u"\u00E7"+"a", "capins-alpiste", "capins-amonjeaba", "capins-andaca"+u"\u00E1",
"capins-andrequic"+u"\u00E9", "capins-angola", "capins-angol"+u"\u00E3"+"o", "capins-ap"+u"\u00E9",
"capins-aragua"+u"\u00ED", "capins-arroz", "capins-atana", "capins-aveia",
"capins-bal"+u"\u00E7"+"a", "capins-bambu", "capins-bambuzinho", "capins-bandeira",
"capins-barata", "capins-batatal", "capins-bengala", "capins-bob"+u"\u00F3",
"capins-bob"+u"\u00F4", "capins-bolota", "capins-burr"+u"\u00E3"+"o", "capins-cabai"+u"\u00FA",
"capins-caiana", "capins-calandr"+u"\u00ED"+"nia", "capins-camalote", "capins-camel"+u"\u00E3"+"o",
"capins-canarana", "capins-caninha", "capins-canudinho", "capins-canut"+u"\u00E3"+"o",
"capins-capivara", "capins-cati", "capins-catinga", "capins-cebola",
"capins-cevada", "capins-chuvisco", "capins-cidreira", "capins-cidrilho",
"capins-cidr"+u"\u00F3", "capins-cocorob"+u"\u00F3", "capins-colch"+u"\u00E3"+"o", "capins-colol"+u"\u00F3",
"capins-col"+u"\u00F4"+"nia", "capins-coloni"+u"\u00E3"+"o", "capins-coqueirinho", "capins-cortesia",
"capins-elefante", "capins-elimo", "capins-espartilho", "capins-esteira",
"capins-estrela", "capins-fartura", "capins-ferro", "capins-flabelo",
"capins-flecha", "capins-flor", "capins-foice", "capins-frecha",
"capins-gafanhoto", "capins-gengibre", "capins-gordura", "capins-grama",
"capins-guaiamum", "capins-guatemala", "capins-guin"+u"\u00E9", "capins-ja"+u"\u00E7"+"ap"+u"\u00E9",
"capins-jaguar"+u"\u00E9", "capins-jaragu"+u"\u00E1", "capins-jasmim", "capins-ja"+u"\u00FA",
"capins-jeriv"+u"\u00E1", "capins-junco", "capins-lanceta", "capins-leque",
"capins-lim"+u"\u00E3"+"o", "capins-lixa", "capins-ma"+u"\u00E7"+"ambar"+u"\u00E1", "capins-malota",
"capins-manga", "capins-maraj"+u"\u00F3", "capins-marmelada", "capins-marreca",
"capins-massap"+u"\u00E9", "capins-membeca", "capins-meruqui"+u"\u00E1", "capins-mi"+u"\u00E7"+"anga",
"capins-milh"+u"\u00E3", "capins-mina", "capins-mium", "capins-mori",
"capins-morot"+u"\u00F3", "capins-mour"+u"\u00E3"+"o", "capins-murim", "capins-mururu",
"capins-napier", "capins-natal", "capins-navalha", "capins-navalheira",
"capins-on"+u"\u00E7"+"a", "capins-orvalho", "capins-palma", "capins-palmeira",
"capins-panasco", "capins-pancu"+u"\u00E3", "capins-pangola", "capins-papu"+u"\u00E3",
"capins-paratur"+u"\u00E1", "capins-penacho", "capins-peripomongo", "capins-piqui",
"capins-pirimembeca", "capins-poror"+u"\u00F3", "capins-quic"+u"\u00E9", "capins-quicuio",
"capins-quicui"+u"\u00FA", "capins-uam"+u"\u00E1", "capins-vassoura", "capins-veludo",
"capins-vetiver", "capins-zaranza", "capit"+u"\u00E3"+"es-chico", "capras-capela",
"car"+u"\u00E1"+"s-coco", "car"+u"\u00E1"+"s-cova", "car"+u"\u00E1"+"s-doce", "car"+u"\u00E1"+"s-inhame",
"caraguat"+u"\u00E1"+"s-acanga", "caraguat"+u"\u00E1"+"s-piteira", "caramujos-boi", "caramujos-cascudo",
"caramujos-ma"+u"\u00E7"+u"\u00E3", "caramujos-pi"+u"\u00E3"+"o", "caramujos-tigela", "carand"+u"\u00E1"+"s-muriti",
"carand"+u"\u00E1"+"s-piranga", "caranguejos-aranha", "caranguejos-baioneta", "caranguejos-u"+u"\u00E7"+u"\u00E1",
"carapan"+u"\u00E3"+"s-ora", "carapan"+u"\u00E3"+"s-pinima", "cardeais-bispo", "cardeais-di"+u"\u00E1"+"cono",
"cardeais-violeta", "cardos-acanto", "cardos-anan"+u"\u00E1", "cardos-bosta",
"cardos-lim"+u"\u00E3"+"o", "cardos-mandacaru", "cardos-mel"+u"\u00E3"+"o", "cardos-palmat"+u"\u00F3"+"ria",
"cardos-sanguinho", "cargas-chumbo", "cargas-limite", "carlos-alvense",
"carlos-barbosense", "carlos-chaguense", "carlos-gomense", "carlos-quinto",
"carmins-fibrina", "carmins-nacar"+u"\u00E1", "carmins-nata", "carm"+u"\u00F3"+"polis-mineiro",
"carrapatos-estrela", "carrapatos-fogo", "carrapatos-p"+u"\u00F3"+"lvora", "carros-bomba",
"carros-chefe", "carros-conceito", "carros-dormit"+u"\u00F3"+"rio", "carros-guincho",
"carros-leito", "carros-madrinha", "carros-pipa", "carros-restaurante",
"carros-sal"+u"\u00E3"+"o", "carros-suporte", "carros-tanque", "carruagens-sal"+u"\u00E3"+"o",
"cartas-bilhete", "cartas-bomba", "cartas-compromisso", "cartas-contrato",
"cartas-fian"+u"\u00E7"+"a", "cartas-piloto", "cartas-telegrama", "cart"+u"\u00F5"+"es-couro",
"cart"+u"\u00F5"+"es-fibra", "cart"+u"\u00F5"+"es-ficha", "cart"+u"\u00F5"+"es-janela", "cart"+u"\u00F5"+"es-palha",
"cart"+u"\u00F5"+"es-pedra", "cart"+u"\u00F5"+"es-resposta", "carumas-cacueme", "carvalhos-an"+u"\u00E3"+"o",
"carvalhos-corti"+u"\u00E7"+"a", "carvalhos-roble", "carvalhos-rosa", "carv"+u"\u00F5"+"es-ti"+u"\u00E7"+u"\u00E3"+"o",
"casas-abrigo", "casas-piano", "casas-torre", "cascas-a"+u"\u00E7"+"ucena",
"cascas-pereira", "cascudos-espada", "cascudos-espinho", "cascudos-lima",
"cascudos-viola", "casos-limite", "castanhas-caiat"+u"\u00E9", "castanhas-jurera",
"castanhas-sapucaia", "castanhos-barril", "castanhos-pipa", "catas-altense",
"cat"+u"\u00E1"+"logos-dicion"+u"\u00E1"+"rio", "catanh"+u"\u00F5"+"es-tesoura", "cateteres-bal"+u"\u00E3"+"o", "causas-chave",
"cavalas-aipim", "cavalas-canga", "cavalas-cangu"+u"\u00E7"+"u", "cavalinhos-mosca",
"cavalos-boi", "cavalos-fantasma", "cavalos-hora", "cavalos-mosca",
"cavalos-vapor", "cebolas-barr"+u"\u00E3"+"o", "cebolas-cec"+u"\u00E9"+"m", "cedros-batata",
"cedros-canela", "cedros-canjerana", "cedros-faia", "cedros-mogno",
"cedros-rosa", "c"+u"\u00E9"+"lulas-alvo", "c"+u"\u00E9"+"lulas-filha", "c"+u"\u00E9"+"lulas-flama",
"c"+u"\u00E9"+"lulas-guarda", "c"+u"\u00E9"+"lulas-m"+u"\u00E3"+"e", "c"+u"\u00E9"+"lulas-mestre", "c"+u"\u00E9"+"lulas-ovo",
"c"+u"\u00E9"+"lulas-semente", "c"+u"\u00E9"+"lulas-tronco", "C"+u"\u00E9"+"lulas-Tronco", "cent"+u"\u00E1"+"ureas-calc"+u"\u00ED"+"trapa",
"centigramas-for"+u"\u00E7"+"a", "cerigados-sab"+u"\u00E3"+"o", "cerigados-tapo"+u"\u00E3", "ch"+u"\u00E1"+"s-biriba",
"ch"+u"\u00E1"+"s-mate", "ch"+u"\u00E1"+"s-puc"+u"\u00E1", "chapas-base", "chap"+u"\u00E9"+"us-boneca",
"chap"+u"\u00E9"+"us-chile", "chap"+u"\u00E9"+"us-coco", "chap"+u"\u00E9"+"us-cone", "chap"+u"\u00E9"+"us-panam"+u"\u00E1",
"charadas-anagrama", "charadas-metagrama", "chefes-banda", "chefes-barra",
"chefes-pala", "cheques-borracha", "cheques-cal"+u"\u00E7"+u"\u00E3"+"o", "chicharros-calabar",
"chicharros-cavala", "chicos-angu", "chimarritas-bal"+u"\u00E3"+"o", "chor"+u"\u00F5"+"es-salgueiro",
"chorr"+u"\u00F3"+"s-boi", "cicl"+u"\u00ED"+"deos-borboleta", "cidades-cogumelo", "cidades-dormit"+u"\u00F3"+"rio",
"cidades-estado", "cidades-Estado", "cidades-fantasma", "cidades-jardim",
"cidades-laborat"+u"\u00F3"+"rio", "cidades-museu", "cidades-sat"+u"\u00E9"+"lite", "cidades-sede",
"Cidades-Sede", "cigarras-cobra", "cigarras-rainha", "cigarrinhas-empoasca",
"cilindros-eixo", "cimentos-amianto", "cinamis-coca"+u"\u00ED"+"na", "cinemas-olho",
"cip"+u"\u00F3"+"s-abacate", "cip"+u"\u00F3"+"s-alho", "cip"+u"\u00F3"+"s-alm"+u"\u00E9"+"cega", "cip"+u"\u00F3"+"s-azougue",
"cip"+u"\u00F3"+"s-barroca", "cip"+u"\u00F3"+"s-brasil", "cip"+u"\u00F3"+"s-caatinga", "cip"+u"\u00F3"+"s-caboclo",
"cip"+u"\u00F3"+"s-ca"+u"\u00E7"+"a"+u"\u00FA", "cip"+u"\u00F3"+"s-caf"+u"\u00E9", "cip"+u"\u00F3"+"s-camar"+u"\u00E3"+"o", "cip"+u"\u00F3"+"s-caniana",
"cip"+u"\u00F3"+"s-caravu"+u"\u00ED", "cip"+u"\u00F3"+"s-carij"+u"\u00F3", "cip"+u"\u00F3"+"s-carneiro", "cip"+u"\u00F3"+"s-catigu"+u"\u00E1",
"cip"+u"\u00F3"+"s-catinga", "cip"+u"\u00F3"+"s-chumbo", "cip"+u"\u00F3"+"s-cobra", "cip"+u"\u00F3"+"s-cola",
"cip"+u"\u00F3"+"s-corimb"+u"\u00F3", "cip"+u"\u00F3"+"s-correlha", "cip"+u"\u00F3"+"s-cravo", "cip"+u"\u00F3"+"s-cruz",
"cip"+u"\u00F3"+"s-cumaruap"+u"\u00E9", "cip"+u"\u00F3"+"s-curap"+u"\u00E9", "cip"+u"\u00F3"+"s-curimb"+u"\u00F3", "cip"+u"\u00F3"+"s-curura",
"cip"+u"\u00F3"+"s-cururu", "cip"+u"\u00F3"+"s-escada", "cip"+u"\u00F3"+"s-escova", "cip"+u"\u00F3"+"s-flor"+u"\u00E3"+"o",
"cip"+u"\u00F3"+"s-gua"+u"\u00E7"+"u", "cip"+u"\u00F3"+"s-guib"+u"\u00E9", "cip"+u"\u00F3"+"s-guira", "cip"+u"\u00F3"+"s-icica",
"cip"+u"\u00F3"+"s-imb"+u"\u00E9", "cip"+u"\u00F3"+"s-jabut"+u"\u00E1", "cip"+u"\u00F3"+"s-jagube", "cip"+u"\u00F3"+"s-jarrinha",
"cip"+u"\u00F3"+"s-jatob"+u"\u00E1", "cip"+u"\u00F3"+"s-jiboia", "cip"+u"\u00F3"+"s-l"+u"\u00ED"+"ngua", "cip"+u"\u00F3"+"s-mata",
"cip"+u"\u00F3"+"s-morcego", "cip"+u"\u00F3"+"s-paratudo", "cip"+u"\u00F3"+"s-par"+u"\u00E9", "cip"+u"\u00F3"+"s-pau",
"cip"+u"\u00F3"+"s-prata", "cirurgi"+u"\u00E3"+"s-dentista", "cirurgi"+u"\u00F5"+"es-dentista", "claros-escuro",
"clav"+u"\u00E1"+"rias-pil"+u"\u00E3"+"o", "clientes-aluno", "clientes-alvo", "cobertas-abrigo",
"cobertas-sal"+u"\u00E3"+"o", "cobras-alcatifa", "cobras-capelo", "cobras-capim",
"cobras-cascavel", "cobras-chicote", "cobras-cip"+u"\u00F3", "cobras-coral",
"cobras-escada", "cobras-escorpi"+u"\u00E3"+"o", "cobras-espada", "cobras-fac"+u"\u00E3"+"o",
"cobras-furtacor", "cobras-jabuti", "cobras-madeira", "cobras-norato",
"cobras-papagaio", "cobras-pil"+u"\u00E3"+"o", "cobras-tigre", "cobras-topete",
"cobras-veado", "cobras-vidro", "cocadas-puxa", "cochonilhas-farinha",
"cochonilhas-pimenta", "cochonilhas-v"+u"\u00ED"+"rgula", "cocos-bingol"+u"\u00EA", "cocos-catul"+u"\u00E9",
"cocos-desafio", "cocos-feij"+u"\u00E3"+"o", "cocos-gavi"+u"\u00E3"+"o", "cocos-guariroba",
"cocos-indai"+u"\u00E1", "cocos-lanho", "cocos-maca"+u"\u00FA"+"ba", "cocos-nai"+u"\u00E1",
"cocos-peneru"+u"\u00EA", "cocos-pindoba", "C"+u"\u00F3"+"digos-fonte", "c"+u"\u00F3"+"digos-fonte",
"c"+u"\u00F3"+"digos-objeto",
"coelhos-borboleta", "colas-esmalte", "coletores-tronco", "Coletores-Tronco",
"com"+u"\u00E9"+"dias-bailado", "com"+u"\u00E9"+"dias-bal"+u"\u00E9", "com"+u"\u00E9"+"dias-drama", "com"+u"\u00E9"+"dias-farsa",
"com"+u"\u00E9"+"dias-pastel"+u"\u00E3"+"o", "conceitos-chave", "congonhas-cachimbo", "congonhas-caixeta",
"congros-rosa", "contas-sal"+u"\u00E1"+"rio", "copais-cauri", "copa"+u"\u00ED"+"bas-angelim",
"copa"+u"\u00ED"+"bas-juta"+u"\u00ED", "copa"+u"\u00ED"+"bas-marimari", "copos-bal"+u"\u00E3"+"o", "coqueiros-a"+u"\u00E7"+"a"+u"\u00ED",
"coqueiros-acun"+u"\u00E3", "coqueiros-airi", "coqueiros-anai"+u"\u00E1", "coqueiros-an"+u"\u00E3"+"o",
"coqueiros-aracuri", "coqueiros-ariri", "coqueiros-babunha", "coqueiros-bacaba",
"coqueiros-bocaiuva", "coqueiros-breja"+u"\u00FA"+"ba", "coqueiros-bu"+u"\u00E7"+"u", "coqueiros-buriti",
"coqueiros-caiau"+u"\u00E9", "coqueiros-carand"+u"\u00E1", "coqueiros-caranda"+u"\u00ED", "coqueiros-catul"+u"\u00E9",
"coqueiros-cau"+u"\u00ED", "coqueiros-dicuri", "coqueiros-guariroba", "coqueiros-guriri",
"coqueiros-iri", "coqueiros-jata"+u"\u00ED", "coqueiros-javari", "coqueiros-jerib"+u"\u00E1",
"coqueiros-macho", "coqueiros-maraja"+u"\u00ED"+"ba", "coqueiros-mumbaca", "coqueiros-nai"+u"\u00E1",
"coqueiros-pia"+u"\u00E7"+"aba", "coqueiros-pissand"+u"\u00F3", "coqueiros-tarampaba", "coqueiros-tucum",
"coqueiros-tucum"+u"\u00E3", "coquinhos-bab"+u"\u00E1", "corais-boia", "corais-c"+u"\u00E9"+"rebro",
"corcorocas-jurumirim", "corcorocas-sargo", "cornos-godinho", "coroas-cr"+u"\u00ED"+"sti",
"corvinas-linha", "costas-abaixo", "costas-acima", "costas-arriba",
"couros-cromo", "couves-cara"+u"\u00ED"+"ba", "couves-cavaleiro", "couves-coco",
"couves-cravinho", "couves-flor", "couves-manteiga", "couves-nabi"+u"\u00E7"+"a",
"couves-nabo", "couves-penca", "couves-r"+u"\u00E1"+"bano", "couves-r"+u"\u00E1"+"b"+u"\u00E3"+"o",
"couves-repolho", "couves-rosa", "couves-saboia", "crateras-lago",
"craterelos-cornuc"+u"\u00F3"+"pia", "cravos-cabecinha", "cravos-girofl"+u"\u00E9", "cravos-minhardise",
"cravos-renda", "crepes-cetim", "crepes-georgete", "crepes-marroquim",
"crian"+u"\u00E7"+"as-problema", "crici"+u"\u00FA"+"mas-cip"+u"\u00F3", "cristas-ervilha", "cristais-paulistense",
"crit"+u"\u00E9"+"rios-base", "crocorocas-jurumim", "cruzadores-escola", "cruzes-diabo",
"cuacos-malagueta", "cuias-marac"+u"\u00E1", "cumarus-ferro", "c"+u"\u00FA"+"mulos-cirro",
"c"+u"\u00FA"+"mulos-estrato", "c"+u"\u00FA"+"mulos-nimbo", "c"+u"\u00FA"+"mulos-vulc"+u"\u00E3"+"o", "curiangos-tesoura",
"curiangos-tesour"+u"\u00E3"+"o", "curicas-bacabal", "curicas-urubu", "curimat"+u"\u00E3"+"s-pacu",
"currais-novense", "curu"+u"\u00E1"+"s-iuquira", "curu"+u"\u00E1"+"s-pixuna", "Dados-fonte", "dados-fonte",
"datas-base", "datas-marco", "decretos-lei", "deus-dar"+u"\u00E1",
"Deus-Homem", "deus-mercado", "dias-coelhense", "dias-multa",
"diabos-alma", "diamantes-alen"+u"\u00E7"+"on", "diamantes-benef"+u"\u00ED"+"cio", "diamantes-bristol",
"diamantes-chapa", "diamantes-fazenda", "diamantes-matura", "diamantes-mesa",
"diamantes-rosa", "diamantes-tabla", "dianteiros-centro", "dinas-cent"+u"\u00ED"+"metro",
"dinamarqu"+u"\u00EA"+"s-espanhol", "dinheiros-papel", "discos-madre", "discos-m"+u"\u00E3"+"e",
"discos-mestre", "dispositivos-alvo", "documentos-base", "dois-abrilense",
"dois-correguense", "dois-irmanense", "dois-irm"+u"\u00E3"+"osense", "dois-lajeadense",
"dois-riachense", "dois-riense", "dois-vizinhense", "domingos-costense",
"domingos-martinense", "domingos-paisense", "domingos-pratense", "donzelas-fogo",
"dores-campense", "dores-guanh"+u"\u00E3"+"esense", "dores-indaiaense", "dores-macabuense",
"dores-marmeladense", "dores-paraibunense", "dores-turvense", "dores-vitoriense",
"doses-padr"+u"\u00E3"+"o", "dourados-carapau", u"\u00E9"+"s-nordeste", u"\u00E9"+"s-noroeste",
u"\u00E9"+"s-sudeste", u"\u00E9"+"s-sudoeste", u"\u00E9"+"s-sueste", "edif"+u"\u00ED"+"cios-garagem",
"elementos-chave", "elias-faustense", "emba"+u"\u00FA"+"bas-manga", "embiras-sebo",
"empregos-alvo", "empresas-alvo", "empresas-m"+u"\u00E3"+"e", "endossos-cau"+u"\u00E7"+u"\u00E3"+"o",
"endossos-mandato", u"\u00EA"+"nulas-campana", "epis"+u"\u00F3"+"dios-piloto", "equivalentes-grama",
"ervas-abelha", "ervas-agrim"+u"\u00F4"+"nia", "ervas-aleluia", "ervas-alheira",
"ervas-alm"+u"\u00ED"+"scar", "ervas-almiscareira", "ervas-ambr"+u"\u00F3"+"sia", "ervas-andorinha",
"ervas-aranha", "ervas-arroz", "ervas-azebre", "ervas-baleeira",
"ervas-belida", "ervas-besteira", "ervas-bezerra", "ervas-bicha",
"ervas-borboleta", "ervas-bot"+u"\u00E3"+"o", "ervas-brico", "ervas-bruxa",
"ervas-bugre", "ervas-can"+u"\u00E1"+"ria", "ervas-candeia", "ervas-capit"+u"\u00E3"+"o",
"ervas-carocha", "ervas-carpinteiro", "ervas-carvalhinha", "ervas-ca"+u"\u00FA"+"na",
"ervas-cavalinha", "ervas-chumbo", "ervas-coentrinha", "ervas-coivinha",
"ervas-col"+u"\u00E9"+"gio", "ervas-contraveneno", "ervas-couvinha", "ervas-crina",
"ervas-cruz", "ervas-dedal", "ervas-sal", "escamas-farinha",
"escamas-peixe", "escamas-prego", "escaravelhos-bosteiro", "escolas-cantina",
"escolas-modelo", "escorpi"+u"\u00F5"+"es-vinagre", "escovinhas-margarida", "espa"+u"\u00E7"+"os-imagem",
"espa"+u"\u00E7"+"os-objeto", "espa"+u"\u00E7"+"os-tempo", "espadas-sabre", "espadartas-maca",
"espadartes-meca", "esp"+u"\u00E9"+"cies-alvo", "esp"+u"\u00E9"+"cies-tipo", "espinelas-rubi",
"espin"+u"\u00E9"+"lios-almandina", "espinheiros-cachorro", "espinheiros-cambra", "espinheiros-carneiro",
"espinhos-guabiru", "espinhos-rosa", "espodum"+u"\u00EA"+"nios-esmeralda", "estacas-prancha",
"estacas-testemunha", "Estados-ilha", "Estados-na"+u"\u00E7"+u"\u00E3"+"o", "Estados-Na"+u"\u00E7"+u"\u00E3"+"o",
"estados-provid"+u"\u00EA"+"ncia", "estados-tamp"+u"\u00E3"+"o", "estrat"+u"\u00E9"+"gias-chave", "estratos-cirro",
"estratos-c"+u"\u00FA"+"mulo", "estratos-nimbo", "estrelas-guia", "estrelinhas-ametista",
"estudos-piloto", "eucaliptos-lim"+u"\u00E3"+"o", "eucaliptos-pimenta", "euclides-cunhense",
"exibi"+u"\u00E7"+u"\u00F5"+"es-teste", "experi"+u"\u00EA"+"ncias-piloto", "facheiros-cardeiro", "facheiros-jamacaru",
"fadas-madrinha", "fadigas-corros"+u"\u00E3"+"o", "falc"+u"\u00F5"+"es-cabur"+u"\u00E9", "falc"+u"\u00F5"+"es-gerifalte",
"falc"+u"\u00F5"+"es-tagarote", "falsas-ervas-mate", "falsos-paratudo", "falsos-paus-brasil",
"farias-britense", "farinhas-farelo", "fatos-macaco", "fatores-chave",
"favas-assaria", "favas-barbatim"+u"\u00E3"+"o", "favas-bel"+u"\u00E9"+"m", "favas-caf"+u"\u00E9",
"favas-caranguejo", "favas-chicote", "favas-coceira", "favas-contra",
"favas-cumand"+u"\u00E1"+"lia", "favas-oli"+u"\u00E1", "favas-or"+u"\u00F3", "favas-pixurim",
"favas-terra", "favas-tonca", "fazendas-modelo", "fechos-rel"+u"\u00E2"+"mpago",
"feij"+u"\u00F5"+"es-adzuki", "feij"+u"\u00F5"+"es-alfanje", "feij"+u"\u00F5"+"es-amendoim", "feij"+u"\u00F5"+"es-aspargo",
"feij"+u"\u00F5"+"es-azeite", "feij"+u"\u00F5"+"es-azuqui", "feij"+u"\u00F5"+"es-bacamarte", "feij"+u"\u00F5"+"es-baet"+u"\u00E3"+"o",
"feij"+u"\u00F5"+"es-baru", "feij"+u"\u00F5"+"es-batata", "feij"+u"\u00F5"+"es-besugo", "feij"+u"\u00F5"+"es-cabacinha",
"feij"+u"\u00F5"+"es-caf"+u"\u00E9", "feij"+u"\u00F5"+"es-canad"+u"\u00E1", "feij"+u"\u00F5"+"es-cap"+u"\u00E3"+"o", "feij"+u"\u00F5"+"es-careta",
"feij"+u"\u00F5"+"es-carito", "feij"+u"\u00F5"+"es-carrapatinho", "feij"+u"\u00F5"+"es-carrapato", "feij"+u"\u00F5"+"es-carumb"+u"\u00E9",
"feij"+u"\u00F5"+"es-cavalinho", "feij"+u"\u00F5"+"es-chicote", "feij"+u"\u00F5"+"es-china", "feij"+u"\u00F5"+"es-chocolate",
"feij"+u"\u00F5"+"es-chumbinho", "feij"+u"\u00F5"+"es-coco", "feij"+u"\u00F5"+"es-congo", "feij"+u"\u00F5"+"es-corda",
"feij"+u"\u00F5"+"es-cutelinho", "feij"+u"\u00F5"+"es-enxofre", "feij"+u"\u00F5"+"es-espada", "feij"+u"\u00F5"+"es-espadinho",
"feij"+u"\u00F5"+"es-farinha", "feij"+u"\u00F5"+"es-favona", "feij"+u"\u00F5"+"es-fidalgo", "feij"+u"\u00F5"+"es-figueira",
"feij"+u"\u00F5"+"es-flor", "feij"+u"\u00F5"+"es-frade", "feij"+u"\u00F5"+"es-fradinho", "feij"+u"\u00F5"+"es-gl"+u"\u00F3"+"ria",
"feij"+u"\u00F5"+"es-guando", "feij"+u"\u00F5"+"es-gurutuba", "feij"+u"\u00F5"+"es-imperador", "feij"+u"\u00F5"+"es-jiritana",
"feij"+u"\u00F5"+"es-lablabe", "feij"+u"\u00F5"+"es-lagartixa", "feij"+u"\u00F5"+"es-mac"+u"\u00E1"+u"\u00E7"+"ar", "feij"+u"\u00F5"+"es-mac"+u"\u00FA"+"ndi",
"feij"+u"\u00F5"+"es-m"+u"\u00E3"+"ezinha", "feij"+u"\u00F5"+"es-mancanha", "feij"+u"\u00F5"+"es-manteiga", "feij"+u"\u00F5"+"es-mucuna",
"feij"+u"\u00F5"+"es-mula", "feij"+u"\u00F5"+"es-mungo", "feij"+u"\u00F5"+"es-or"+u"\u00F3", "feij"+u"\u00F5"+"es-papa",
"feij"+u"\u00F5"+"es-pical", "feij"+u"\u00F5"+"es-pinheirinho", "feij"+u"\u00F5"+"es-pombinha", "feij"+u"\u00F5"+"es-sapota",
"feij"+u"\u00F5"+"es-selim", "feij"+u"\u00F5"+"es-soja", "feij"+u"\u00F5"+"es-telefone", "feij"+u"\u00F5"+"es-tepari",
"feij"+u"\u00F5"+"es-tonca", "feij"+u"\u00F5"+"es-tropeiro", "feij"+u"\u00F5"+"es-vassoura", "feij"+u"\u00F5"+"es-veludo",
"feij"+u"\u00F5"+"es-vinha", "feijoeiros-lagartixa", "feixas-fradinho", "ferrais-ma"+u"\u00E7"+u"\u00E3",
"ferrais-t"+u"\u00E2"+"mara", "ferros-ferrita", "ferros-gusa", "ferros-ilmenita",
"ferros-pau", "festas-homenagem", "fetos-cip"+u"\u00F3", "f"+u"\u00ED"+"cus-benjamim",
"figos-flor", "figos-porco", "figueiras-babosa", "figueiras-benjamim",
"figueiras-bispo", "figueirinhas-hera", "figuras-chave", "filhos-fam"+u"\u00ED"+"lia",
"filmes-cat"+u"\u00E1"+"strofe", "filmes-evento", "filmes-fam"+u"\u00ED"+"lia", "filmes-piloto",
"filodendros-imb"+u"\u00E9", "filtros-prensa", "fios-sardinha", "firmas-membro",
"Firmas-Membro", "fitas-crepe", "fitas-demo", "flos-sant"+u"\u00F3"+"rio",
"flores-camar"+u"\u00E3"+"o", "flores-cunhense", "focas-harpa", "focas-leopardo",
"focas-monge", "foguetes-alvo", "foguetes-sonda", "folhas-flor",
"folhas-formiga", "folhas-galo", "folhas-lixa", "for"+u"\u00E7"+"as-tarefa",
"formigas-brasa", "formigas-caba"+u"\u00E7"+"a", "formigas-caiap"+u"\u00F3", "formigas-cascavel",
"formigas-cobra", "formigas-conga", "formigas-correi"+u"\u00E7"+u"\u00E3"+"o", "formigas-ferro",
"formigas-le"+u"\u00E3"+"o", "formigas-mandioca", "formigas-maravilha", "formigas-oncinha",
"formigas-pixixica", "formigas-quenqu"+u"\u00E9"+"m", "formigas-rainha", "formigas-raspa",
"formigas-sa"+u"\u00FA"+"va", "formigas-vespa", "f"+u"\u00F3"+"rmulas-grama", "f"+u"\u00F3"+"sseis-guia",
"f"+u"\u00F3"+"sseis-"+u"\u00ED"+"ndice", "fossos-diamante", "frases-t"+u"\u00ED"+"tulo", "freis-bode",
"freis-jorge", "frevos-abafo", "frevos-can"+u"\u00E7"+u"\u00E3"+"o", "frevos-coqueiro",
"frevos-ventania", "fronteiras-faixa", "fronteiras-linha", "fronteiras-zona",
"frutas-barrete", "frutas-bolsa", "frutas-caurim", "frutas-gargantilha",
"frutas-gundra", "frutas-manga", "frutas-manila", "frutas-p"+u"\u00E3"+"o",
"frutas-tirilha", "frutas-trilha", "fruteiras-p"+u"\u00E3"+"o", "fulanos-dos-anz"+u"\u00F3"+"is-carapu"+u"\u00E7"+"a",
"fungadores-on"+u"\u00E7"+"a", "furri"+u"\u00E9"+"is-rosa", "g"+u"\u00E1"+"s-estufa", "g"+u"\u00E1"+"s-mil",
"gafanhotos-bandeira", "gafanhotos-cobra", "galos-bandeira", "galos-brejo",
"gansos-cisne", "gansos-patola", "gar"+u"\u00E7"+"as-cinza", "gar"+u"\u00E7"+"as-soco"+u"\u00ED",
"gargantas-rubi", "garotas-propaganda", "garotos-propaganda", "garoupas-bexiga",
"garoupas-chita", "garoupas-gato", "garrinchas-trov"+u"\u00E3"+"o", "gases-estufa",
"gatos-lag"+u"\u00E1"+"ria", "gatos-maracaj"+u"\u00E1", "gatos-paul", "gatos-sapato",
"gatos-tigre", "gavi"+u"\u00F5"+"es-caboclo", "gavi"+u"\u00F5"+"es-cabur"+u"\u00E9", "gavi"+u"\u00F5"+"es-caripira",
"gavi"+u"\u00F5"+"es-cav"+u"\u00E3", "gavi"+u"\u00F5"+"es-cov"+u"\u00E3", "gavi"+u"\u00F5"+"es-formiga", "gavi"+u"\u00F5"+"es-fuma"+u"\u00E7"+"a",
"gavi"+u"\u00F5"+"es-inaj"+u"\u00E9", "gavi"+u"\u00F5"+"es-indai"+u"\u00E9", "gavi"+u"\u00F5"+"es-lavadeira", "gavi"+u"\u00F5"+"es-mateiro",
"gavi"+u"\u00F5"+"es-padre", "gavi"+u"\u00F5"+"es-pato", "gavi"+u"\u00F5"+"es-peneira", "gavi"+u"\u00F5"+"es-pinh"+u"\u00E9",
"gavi"+u"\u00F5"+"es-pombo", "gavi"+u"\u00F5"+"es-puva", "gavi"+u"\u00F5"+"es-quiriquiri", "gavi"+u"\u00F5"+"es-rapina",
"gavi"+u"\u00F5"+"es-rel"+u"\u00F3"+"gio", "gavi"+u"\u00F5"+"es-sauveiro", "gavi"+u"\u00F5"+"es-tanat"+u"\u00F3", "gavi"+u"\u00F5"+"es-tesoira",
"gavi"+u"\u00F5"+"es-tesoura", "gavi"+u"\u00F5"+"es-tinga", "gavi"+u"\u00F5"+"es-vaqueiro", "ger"+u"\u00E2"+"nios-hera",
"ger"+u"\u00E2"+"nios-rosa", "gessos-cr"+u"\u00E9", "gessos-estuque", "gestos-chave",
"gilas-caiota", "globul"+u"\u00E1"+"rias-turbito", "goitac"+u"\u00E1"+"s-jacurito", "goitac"+u"\u00E1"+"s-mopi",
"golfinhos-piloto", "gomas-laca", "gomas-resina", "goran"+u"\u00E1"+"s-timb"+u"\u00F3",
"gotas-serena", "gralhas-canc"+u"\u00E3", "gramas-cravo", "gramas-forquilha",
"gramas-lanceta", "gramas-orvalho", "gramas-sapo", "gramas-seda",
"graminhas-seda", "granadas-foguete", "gravatas-borboleta", "gravat"+u"\u00E1"+"s-borboleta",
"gravat"+u"\u00E1"+"s-canga", "gravat"+u"\u00E1"+"s-zebra", "grilos-cantor", "grilos-toupeira",
"gro"+u"\u00E7"+"a"+u"\u00ED"+"s-azeite", "grupos-base", "guajan"+u"\u00E1"+"s-timb"+u"\u00F3", "guajar"+u"\u00E1"+"s-abio",
"guajar"+u"\u00E1"+"s-cinza", "guajar"+u"\u00E1"+"s-curumim", "guajar"+u"\u00E1"+"s-poca", "guajar"+u"\u00E1"+"s-preto",
"guapu"+u"\u00ED"+"s-cip"+u"\u00F3", "guar"+u"\u00E1"+"s-tereba", "guarabus-amarelo", "guarabus-batata",
"guarabus-cebola", "guaran"+u"\u00E1"+"s-timb"+u"\u00F3", "guardas-espelho", "guardas-marinha",
"Guarulhos-Nazar"+u"\u00E9", "guatambus-marfim", "guatambus-peroba", "guatambus-rosa",
"guerras-rel"+u"\u00E2"+"mpago", "guir"+u"\u00E1"+"s-acangatara", "habeas-data", "her"+u"\u00E1"+"s-herahum",
"herpes-vir"+u"\u00F3"+"tico", "herpes-zoster", "herpes-z"+u"\u00F3"+"ster", "homens-aranha",
"homens-bomba", "homens-chave", "homens-gol", "homens-m"+u"\u00E1"+"quina",
"Homens-M"+u"\u00E1"+"quina", "homens-massa", "homens-morcego", "homens-mosca",
"homens-orquestra", "homens-padr"+u"\u00E3"+"o", "homens-p"+u"\u00E1"+"ssaro", "homens-peixe",
"homens-placa", "homens-sandu"+u"\u00ED"+"che", "homens-tipo", "horas-atividade",
"horas-aula", "horas-homem", "horas-luz", "horas-m"+u"\u00E1"+"quina",
"hortel"+u"\u00E3"+"s-boi", "hortel"+u"\u00E3"+"s-pimenta", "hospitais-escola", u"\u00ED"+"bis-branco",
u"\u00ED"+"bis-sagrado", "icacor"+u"\u00E9"+"s-catinga", "idades-limite", "ideias-chave",
"ideias-m"+u"\u00E3"+"e", "ideias-n"+u"\u00FA"+"mero", "ilhas-Estado", "Ilhas-estado",
"Ilhas-Estado", "inajaranas-envira", "inambus-galinha", "inambus-rel"+u"\u00F3"+"gio",
"inambus-saracu"+u"\u00ED"+"ra", "inamus-galinha", "inamus-rel"+u"\u00F3"+"gio", "inamus-saracu"+u"\u00ED"+"ra",
"indiv"+u"\u00ED"+"duos-alvo", "informa"+u"\u00E7"+u"\u00F5"+"es-chave", "ing"+u"\u00E1"+"s-caix"+u"\u00E3"+"o", "ing"+u"\u00E1"+"s-cip"+u"\u00F3",
"ing"+u"\u00E1"+"s-cururu", "ing"+u"\u00E1"+"s-feij"+u"\u00E3"+"o", "ing"+u"\u00E1"+"s-ferradura", "ing"+u"\u00E1"+"s-macaco",
"ing"+u"\u00E1"+"s-veludo", "ingl"+u"\u00EA"+"s-russo", "inhambus-galinha", "inhambus-peba",
"inhambus-rel"+u"\u00F3"+"gio", "inhambus-saracu"+u"\u00ED"+"ra", "inhambus-tona", "inhames-cigarra",
"inhames-farinha", "inhames-gudu", "inhames-monstro", "inhames-nambu",
"inhames-taro", "inhames-t"+u"\u00ED"+"voli", "inhames-zambuco", "instrumentos-chave",
"interdi"+u"\u00E7"+u"\u00F5"+"es-rel"+u"\u00E2"+"mpago", u"\u00ED"+"ons-grama", "ip"+u"\u00EA"+"s-batata", "ip"+u"\u00EA"+"s-boia",
"ip"+u"\u00EA"+"s-mandioca", "ip"+u"\u00EA"+"s-tabaco", "ipecacuanhas-poaia", "itu"+u"\u00ED"+"s-cavalo",
"itu"+u"\u00ED"+"s-pinima", "j"+u"\u00E3"+"s-vaz", "jaburus-moleque", "jabutis-aperema",
"jabutis-araconha", "jabutis-carumb"+u"\u00E9", "jabutis-jurema", "jabutis-machado",
"jabutis-tucum"+u"\u00E3", "jacamins-copejuba", "jacamins-copetinga", "jacamins-cunh"+u"\u00E3",
"jacamins-cupejuba", "jacamins-cupijuba", "jacamins-cupitinga", "jacamins-una",
"jacarand"+u"\u00E1"+"s-banana", "jacarand"+u"\u00E1"+"s-cabi"+u"\u00FA"+"na", "jacarand"+u"\u00E1"+"s-caroba", "jacarand"+u"\u00E1"+"s-cavi"+u"\u00FA"+"na",
"jacarand"+u"\u00E1"+"s-ferro", "jacarand"+u"\u00E1"+"s-pitanga", "jacarand"+u"\u00E1"+"s-rosa", "jacar"+u"\u00E9"+"s-cacau",
"jacar"+u"\u00E9"+"s-catinga", "jacar"+u"\u00E9"+"s-copa"+u"\u00ED"+"ba", "jacar"+u"\u00E9"+"s-coroa", "jacar"+u"\u00E9"+"s-curua",
"jacar"+u"\u00E9"+"s-curulana", "jacar"+u"\u00E9"+"s-pagu"+u"\u00E1", "jacus-estalo", "jacus-molambo",
"jacus-porco", "jacus-queixada", "jacus-taquara", "jacund"+u"\u00E1"+"s-coroa",
"jacund"+u"\u00E1"+"s-pinima", "jacund"+u"\u00E1"+"s-piranga", "jades-calif"+u"\u00F3"+"rnia", "jagu"+u"\u00E1"+"s-arroz",
"jaguaret"+u"\u00E9"+"s-pixuna", "jaguaret"+u"\u00EA"+"s-pixuna", "jambeiros-rosa", "jambos-ch"+u"\u00E1",
"jambos-rosa", "japiins-xex"+u"\u00E9"+"u", "japins-soldado", "japus-gamela",
"jararacas-cruzeira", "jararacas-tai"+u"\u00E1", "jararacu"+u"\u00E7"+"us-tapete", "jararacu"+u"\u00E7"+"us-tipiti",
"jararacu"+u"\u00E7"+"us-topete", "jardins-escola", "jarrinhas-arraia", "jarrinhas-monstro",
"jasmins-bogari", "jasmins-cambraia", "jasmins-catavento", "jasmins-estrela",
"jasmins-junquilho", "jasmins-lacre", "jasmins-laranja", "jasmins-manac"+u"\u00E1",
"jasmins-manga", "jasmins-mangueira", "jasmins-manteiga", "jasmins-murta",
"jasmins-pipoca", "jasmins-porcelana", "jasmins-sombra", "jasmins-trombeta",
"jasmins-vapor", "jata"+u"\u00ED"+"s-mond"+u"\u00E9", "jata"+u"\u00ED"+"s-mosquito", "jata"+u"\u00ED"+"s-poror"+u"\u00F3",
"jata"+u"\u00ED"+"s-pororoca", "jatob"+u"\u00E1"+"s-cap"+u"\u00E3"+"o", "jenipapos-rosa", "jequitib"+u"\u00E1"+"s-cedro",
"jequitib"+u"\u00E1"+"s-rosa", "jesus-amparense", "jesus-galhense", "jesus-itabapoanense",
"jesus-lapense", "jimbas-muxi", "joanas-guenza", "joaninhas-guenza",
"jogos-treino", "jo"+u"\u00F5"+"es-bal"+u"\u00E3"+"o", "jo"+u"\u00F5"+"es-minhoca", "jo"+u"\u00F5"+"es-porca",
"j"+u"\u00F3"+"queis-clube", "ju"+u"\u00E1"+"s-uva", "juritis-pupu", "juta"+u"\u00ED"+"s-caf"+u"\u00E9",
"juta"+u"\u00ED"+"s-catinga", "juta"+u"\u00ED"+"s-pororoca", "lagartas-aranha", "lagartas-compasso",
"lagartas-elasmo", "lagartas-rosca", "lagos-mar", "lambaris-prata",
"lambdas-zero", "l"+u"\u00E1"+"pis-laz"+u"\u00FA"+"li", "l"+u"\u00E1"+"pis-tinta", "laranjas-aperu",
"laranjas-ba"+u"\u00ED"+"a", "laranjas-cravo", "laranjas-lima", "laranjas-melancia",
"laranjas-natal", "laranjas-pera", "laranjas-toranja", "laranjeiras-melancia",
"laranjeiras-pera", "laranjeiras-toranja", "larvas-alfinete", "larvas-arame",
"l"+u"\u00E9"+"s-nordeste", "l"+u"\u00E9"+"s-noroeste", "l"+u"\u00E9"+"s-oeste", "l"+u"\u00E9"+"s-sudeste",
"l"+u"\u00E9"+"s-sudoeste", "l"+u"\u00E9"+"s-sueste", "lega"+u"\u00E7"+"os-cozinho", "lega"+u"\u00E7"+u"\u00F5"+"es-cabecinho",
"len"+u"\u00E7"+"os-papel", "leos-metro", "letras-guia", "libras-massa",
"libras-peso", "licen"+u"\u00E7"+"as-maternidade", "licen"+u"\u00E7"+"as-paternidade", "licen"+u"\u00E7"+"as-pr"+u"\u00EA"+"mio",
"limas-faca", "limas-serra", "lim"+u"\u00F5"+"es-canudo", "lim"+u"\u00F5"+"es-cravo",
"l"+u"\u00ED"+"nguas-alvo", "L"+u"\u00ED"+"nguas-fonte", "l"+u"\u00ED"+"nguas-fonte",
"l"+u"\u00ED"+"nguas-m"+u"\u00E3"+"e", "l"+u"\u00ED"+"nguas-padr"+u"\u00E3"+"o",
"linguados-lixa", "linguagens-fonte", "linguagens-objeto", "linhas-base",
"linhas-bloco", "linhos-c"+u"\u00E2"+"nhamo", "l"+u"\u00ED"+"rios-ferro", "l"+u"\u00ED"+"rios-tigre",
"l"+u"\u00ED"+"rios-tuicara", "livros-caixa", "livros-cassete", "livros-raz"+u"\u00E3"+"o",
"livros-texto", "lobos-guar"+u"\u00E1", "lobos-tigre", "lo"+u"\u00E9"+"s-nordeste",
"lo"+u"\u00E9"+"s-noroeste", "lo"+u"\u00E9"+"s-sudoeste", "lo"+u"\u00E9"+"s-sueste", "loireiros-rosa",
"loiros-abacate", "loiros-batata", "loiros-bosta", "loiros-c"+u"\u00E2"+"nfora",
"loiros-cerejo", "loiros-cravo", "loiros-faia", "loiros-inhamu"+u"\u00ED",
"loiros-mamorim", "loiros-nhamu"+u"\u00ED", "loiros-pimenta", "loiros-puxuri",
"loiros-rosa", "loiros-tamanc"+u"\u00E3"+"o", "loiros-tamanco", "l"+u"\u00F3"+"tus-amarelo",
"l"+u"\u00F3"+"tus-americano", "l"+u"\u00F3"+"tus-azul", "l"+u"\u00F3"+"tus-"+u"\u00ED"+"ndico", "l"+u"\u00F3"+"tus-vermelho",
"loureiros-cerejo", "loureiros-rosa", "louros-abacate", "louros-aritu",
"louros-batata", "louros-bosta", "louros-c"+u"\u00E2"+"nfora", "louros-cerejo",
"louros-cravo", "louros-inhamu"+u"\u00ED", "louros-mamorim", "louros-mamu"+u"\u00ED",
"louros-nhamu"+u"\u00ED", "louros-pimenta", "louros-puxuri", "louros-rosa",
"louros-salgueiro", "louros-tamanc"+u"\u00E3"+"o", "louros-tim", "l"+u"\u00FA"+"cias-lima",
"lugres-barca", "lugres-escuna", "lugres-iate", "lugres-patacho",
"lu"+u"\u00ED"+"s-alvense", "lu"+u"\u00ED"+"s-antoniense", "lu"+u"\u00ED"+"s-cacheiro", "lu"+u"\u00ED"+"s-caixeiro",
"lu"+u"\u00ED"+"s-correense", "lu"+u"\u00ED"+"s-correiense", "lu"+u"\u00ED"+"s-dominguense", "lu"+u"\u00ED"+"s-gomense",
"lu"+u"\u00ED"+"s-gonzaguense", "lu"+u"\u00ED"+"s-quatorze", "lu"+u"\u00ED"+"s-quinze", "lu"+u"\u00ED"+"s-teixeira",
"lu"+u"\u00ED"+"s-treze", "lu"+u"\u00ED"+"s-vianense", "lumens-bilirrubina", "lumens-hora",
"l"+u"\u00FA"+"menes-bilirrubina", "l"+u"\u00FA"+"menes-hora", "luzes-cuco", "ma"+u"\u00E7"+u"\u00E3"+"s-cuca",
"macacos-aranha", "macacos-castanha", "macacos-cip"+u"\u00F3", "macacos-esquilo",
"macacos-itapo"+u"\u00E1", "macacos-le"+u"\u00E3"+"o", "macacos-patrona", "macacos-prego",
"madeiras-seringa", "madres-cravo", "m"+u"\u00E3"+"es-tia", "magalh"+u"\u00E3"+"es-almeidense",
"magalh"+u"\u00E3"+"es-baratense", "magn"+u"\u00F3"+"lias-tulipa", "maguaris-pistola", "maguaris-pistol"+u"\u00E3"+"o",
"maitacas-curica", "malas-arte", "malvas-algod"+u"\u00E3"+"o", "malvas-caju"+u"\u00E7"+"ara", "malvas-carrapicho",
"malvas-flor", "malvas-piranga", "malvas-reloginho", "malvas-rel"+u"\u00F3"+"gio",
"malvas-rosa", "malvas-sardinha", "malvas-taquari", "mamas-cachorra",
"mamas-cadela", "mam"+u"\u00E3"+"es-dengue", "mamatos-c"+u"\u00FA"+"mulo", "mamoeiros-mel"+u"\u00E3"+"o",
"manas-chica", "mandatos-tamp"+u"\u00E3"+"o", "mandis-bagre", "mandis-bandeira",
"mandis-boi", "mandis-cachorro", "mandis-cambaia", "mandis-casaca",
"mandis-fidalgo", "mandis-iraj"+u"\u00E1", "mandis-moela", "mandis-palha",
"mandis-urutu", "mandiocas-tucum"+u"\u00E3", "mandioquinhas-salsa", "man"+u"\u00E9"+"s-jac"+u"\u00E1",
"mangas-rosa", "mangabas-ovo", "mangan"+u"\u00EA"+"s-preto", "mangar"+u"\u00E1"+"s-pena",
"manguaris-pistola", "manguaris-pistol"+u"\u00E3"+"o", "mangues-canaponga", "mangues-cano"+u"\u00E9",
"mangues-ciri"+u"\u00FA"+"ba", "mangues-guapir"+u"\u00E1", "mangues-sara"+u"\u00ED"+"ba", "mangues-seri"+u"\u00FA"+"ba",
"manjares-grude", "mantos-xale", "mapas-m"+u"\u00FA"+"ndi", "m"+u"\u00E1"+"quinas-caix"+u"\u00E3"+"o",
"m"+u"\u00E1"+"quinas-ferramenta", "maracuj"+u"\u00E1"+"s-mam"+u"\u00E3"+"o", "maracuj"+u"\u00E1"+"s-mel"+u"\u00E3"+"o", "maracuj"+u"\u00E1"+"s-periquito",
"maracuj"+u"\u00E1"+"s-peroba", "maracuj"+u"\u00E1"+"s-sururuca", "maracuj"+u"\u00E1"+"s-suspiro", "maraj"+u"\u00E1"+"s-piranga",
"marcas-s"+u"\u00ED"+"mbolo", "marcelas-marga"+u"\u00E7"+"a", "marchas-rancho", "marchas-regresso",
"marcos-parentense", "marias-cadeira", "marias-candel"+u"\u00E1"+"ria", "marias-farinha",
"marias-fede", "marias-ferrugem", "marias-fia", "marias-fuma"+u"\u00E7"+"a",
"marias-irr"+u"\u00E9", "marias-isabel", "marias-lecre", "marias-leite",
"marias-mucangu"+u"\u00EA", "marias-rap"+u"\u00E9", "marias-rita", "marias-rosa",
"marias-teresa", "maribondos-cavalo", "maribondos-chap"+u"\u00E9"+"u", "maribondos-tatu",
"marimbondos-beju", "marimbondos-cachorro", "marimbondos-cavalo", "marimbondos-chap"+u"\u00E9"+"u",
"marimbondos-mangang"+u"\u00E1", "marimbondos-tatu", "mariposas-espelito", "mariposas-imperador",
"mariscos-faca", "mariscos-pedra", "mariscos-tatu", "marmeladas-cachorro",
"marques-loureiro", "marqu"+u"\u00EA"+"s-valencense", "marrecas-anana"+u"\u00ED", "marrecas-apa"+u"\u00ED",
"marrecas-arrebio", "marrecas-cabocla", "marrecas-carij"+u"\u00F3", "marrecas-cricri",
"marrecas-espelho", "marrecas-toicinho", "marrecas-toucinho", "marrecas-tururu",
"marrecos-mandarim", "martins-gil", "martins-gravata", "mat"+u"\u00E9"+"rias-espelho",
"mateus-lemense", "mateus-sulino", "matias-barbosense", "matias-cardosense",
"matias-olimpiense", "matos-capim", "matos-salema", "matrizes-gaveta",
"m"+u"\u00EA"+"s-luz", "melodias-tenor", "mel"+u"\u00F5"+"es-caboclo", "mel"+u"\u00F5"+"es-cantalupo",
"melros-cachoeiro", "mendes-pimentelense", "meninas-aranha", "meninos-aranha",
"meninos-prod"+u"\u00ED"+"gio", "menos-mal", "menos-mau", "menos-valia",
"mercados-alvo", "merendibas-bagre", "meridianos-origem", "mesas-ninho",
"meses-luz", "mestres-empada", "mestres-escama", "mestres-escola",
"mestres-terreiro", "mestres-vinagre", "metais-tipo", "metros-esteno",
"metros-hipercinese", "metros-hiperestesia", "metros-newton", "miaos-iao",
"micholes-quati", "micos-estrela", "micos-le"+u"\u00E3"+"o", "microgramas-for"+u"\u00E7"+"a",
"milandos-homem", "milh"+u"\u00E3"+"s-gar"+u"\u00E7"+"a", "milhos-alho", "milhos-alpiste",
"milhos-arroz", "milhos-dente", "milhos-pipoca", "milhos-rei",
"milhos-sorgo", "miligramas-for"+u"\u00E7"+"a", "miligramas-minuto", "miligramas-segundo",
"minas-bocainense", "minas-butiaense", "minas-entrerriense", "minas-generalense",
"minas-geralense", "minas-jardinense", "minas-madrense", "minas-mimosense",
"minas-montalegrense", "minas-novense", "minas-vicentino", "mineiros-tieteense",
"minerais-gema", "minerais-"+u"\u00ED"+"ndice", "minutas-padr"+u"\u00E3"+"o", "minutos-luz",
"mirins-pregui"+u"\u00E7"+"a", "mirindibas-bagre", "mirindibas-rosa", "miror"+u"\u00F3"+"s-caramuru",
"mirras-mofina", "mochos-carij"+u"\u00F3", "mochos-diabo", "mocoins-caca",
"modelos-base", "moedas-ouro", "moedas-papel", "moldes-caldeira",
"mol"+u"\u00E9"+"culas-grama", "molhes-molhe", "monjolos-ferro", "montes-altense",
"montes-clarense", "montes-clarino", "morcegos-buldogue", "morcegos-pescador",
"morcegos-vampa", "morcegos-vampiro", "morm"+u"\u00F3"+"lices-folha", "moror"+u"\u00F3"+"s-cip"+u"\u00F3",
"morosos-cigano", "moscas-aranha", "moscas-bicheira", "moscas-oura",
"moscas-piroga", "moscas-vareja", "mosquitos-berne", "mosquitos-palha",
"mosquitos-pernilongo", "mosquitos-p"+u"\u00F3"+"lvora", "mosquitos-prego", "mosquitos-remela",
"mostardeiras-violeta", "motores-canh"+u"\u00E3"+"o", "motores-foguete", "mu"+u"\u00E7"+"amb"+u"\u00E9"+"s-catinga",
"mu"+u"\u00E7"+"andeiras-sangue", "mucutas-veado", "mudan"+u"\u00E7"+"as-chave", "mulheres-aranha",
"mulheres-gato", "mulheres-homem", "mulheres-objeto", "munic"+u"\u00ED"+"pios-sede",
"muricis-pitanga", "murtas-"+u"\u00ED"+"ndia", "murtas-pimenta", "murur"+u"\u00E9"+"s-carrapatinho",
"murur"+u"\u00E9"+"s-paj"+u"\u00E9", "musgos-piran"+u"\u00E1", "mutuns-cavalo", "nambus-rel"+u"\u00F3"+"gio",
"nambus-saracu"+u"\u00ED"+"ra", "narcisos-ramilhete", "navios-ar"+u"\u00ED"+"ete", "navios-cisterna",
"navios-correio", "navios-escola", "navios-escolta", "navios-farol",
"navios-hospital", "navios-laborat"+u"\u00F3"+"rio", "navios-oficina", "navios-sonda",
"navios-tanque", "navios-t"+u"\u00EA"+"nder", "navios-transporte", "negras-mina",
"negros-a"+u"\u00E7"+"a", "negros-mina", "neur"+u"\u00F4"+"nios-chave", "neur"+u"\u00F4"+"nios-espelho",
"neves-paulistense", "nhambus-gu"+u"\u00E7"+"u", "nhambus-rel"+u"\u00F3"+"gio", "nhambus-saracu"+u"\u00ED"+"ra",
"nh"+u"\u00F4"+"s-chico", "nogueiras-pec"+u"\u00E3", "normas-padr"+u"\u00E3"+"o", "nozes-macad"+u"\u00E2"+"mia",
"nozes-pec"+u"\u00E3", "n"+u"\u00FA"+"meros-"+u"\u00ED"+"ndice", "obaranas-rato", "objetivos-chave",
"objetos-alvo", "objetos-part"+u"\u00ED"+"cula", "objetos-s"+u"\u00ED"+"mbolo", "o"+u"\u00E9"+"s-nordeste",
"o"+u"\u00E9"+"s-noroeste", "o"+u"\u00E9"+"s-sudoeste", "o"+u"\u00E9"+"s-sueste", "og"+u"\u00E3"+"s-ilu",
"ohms-cent"+u"\u00ED"+"metro", "oiros-fio", "oiros-pigmento", "oiros-pimenta",
"oiticicas-cica", "olhos-cia", "on"+u"\u00E7"+"as-boi", "ondas-mar"+u"\u00E9",
"ondas-piloto", u"\u00F4"+"nibus-camarote", "opalas-arlequim", "opalas-f"+u"\u00ED"+"gado",
"opalas-musgo", u"\u00F3"+"peras-bufa", u"\u00F3"+"peras-rock", "or"+u"\u00E7"+"amentos-programa",
"ostraceiros-pirupiru", "otis-xavante", "ouri"+u"\u00E7"+"os-escudo", "ouros-pigmento",
"ouros-pimenta", "pacam"+u"\u00F5"+"es-cu"+u"\u00ED"+"ca", "pacam"+u"\u00F5"+"es-niquim", "pacos-caatinga",
"pacos-catinga", "pacovas-catinga", "pacovas-sororoca", "pacus-ap"+u"\u00F3",
"pacus-chid"+u"\u00E3"+"o", "pacus-oerud"+u"\u00E1", "pacu"+u"\u00E1"+"s-sororoca", "padinas-pav"+u"\u00E3"+"o",
"p"+u"\u00E3"+"es-bengala", "p"+u"\u00E3"+"es-canoa", "paes-landinense", "pa"+u"\u00ED"+"s-alvo",
"pa"+u"\u00ED"+"s-chave", "pa"+u"\u00ED"+"s-irm"+u"\u00E3"+"o", "pais-lemense", "pa"+u"\u00ED"+"s-membro",
"pa"+u"\u00ED"+"s-parceiro", "pa"+u"\u00ED"+"s-sede", "painas-cip"+u"\u00F3", "painas-ganga",
"pa"+u"\u00ED"+"ses-alvo", "pa"+u"\u00ED"+"ses-chave", "pa"+u"\u00ED"+"ses-ilha", "Pa"+u"\u00ED"+"ses-Ilha",
"pa"+u"\u00ED"+"ses-sede", "paj"+u"\u00E9"+"s-sacaca", "palavras-chave", "palavras-fantasma",
"palavras-filtro", "palavras-testemunho", "palavras-valise", "palet"+u"\u00F3"+"s-saco",
"palhas-fede", "p"+u"\u00E1"+"lios-cirro", "p"+u"\u00E1"+"lios-c"+u"\u00FA"+"mulo", "palmas-christi",
"palmeiras-a"+u"\u00E7"+"a"+u"\u00ED", "palmeiras-andim", "palmeiras-areca", "palmeiras-bambu",
"palmeiras-buri", "palmeiras-camed"+u"\u00F3"+"ria", "palmeiras-catol"+u"\u00E9", "palmeiras-catul"+u"\u00E9",
"palmeiras-chifre", "palmeiras-cip"+u"\u00F3", "palmeiras-dend"+u"\u00EA", "palmeiras-iar"+u"\u00E1",
"palmeiras-jara"+u"\u00FA"+"va", "palmeiras-ji"+u"\u00E7"+"ara", "palmeiras-ju"+u"\u00E7"+"ara", "palmeiras-laca",
"palmeiras-leque", "palmeiras-palmira", "palmeiras-pia"+u"\u00E7"+"aba", "palmeiras-princesa",
"palmeiras-pupunha", "palmitos-ju"+u"\u00E7"+"ara", "p"+u"\u00E2"+"mpanos-pacu", "pampos-arabeb"+u"\u00E9"+"u",
"pampos-aracanguira", "pampos-tibilo", "pampos-tibiro", "panapan"+u"\u00E1"+"s-piranga",
"panapan"+u"\u00E1"+"s-tau"+u"\u00E1", "panos-baju", "panos-fam"+u"\u00ED"+"lia", "panos-piloto",
"papagaios-curau", "papagaios-goiaba", "papagaios-poaeiro", "papagaios-urubu",
"pap"+u"\u00E9"+"is-alum"+u"\u00ED"+"nio", "pap"+u"\u00E9"+"is-arroz", "pap"+u"\u00E9"+"is-b"+u"\u00ED"+"blia", "pap"+u"\u00E9"+"is-bond",
"pap"+u"\u00E9"+"is-bule", "pap"+u"\u00E9"+"is-carbono", "pap"+u"\u00E9"+"is-cart"+u"\u00E3"+"o", "pap"+u"\u00E9"+"is-carv"+u"\u00E3"+"o",
"pap"+u"\u00E9"+"is-celofane", "pap"+u"\u00E9"+"is-chup"+u"\u00E3"+"o", "pap"+u"\u00E9"+"is-filtro", "pap"+u"\u00E9"+"is-ilustra"+u"\u00E7"+u"\u00E3"+"o",
"pap"+u"\u00E9"+"is-jornal", "pap"+u"\u00E9"+"is-ladr"+u"\u00E3"+"o", "pap"+u"\u00E9"+"is-manilha", "pap"+u"\u00E9"+"is-manteiga",
"pap"+u"\u00E9"+"is-moeda", "pap"+u"\u00E9"+"is-of"+u"\u00ED"+"cio", "pap"+u"\u00E9"+"is-oxford", "pap"+u"\u00E9"+"is-ozalide",
"pap"+u"\u00E9"+"is-palha", "pap"+u"\u00E9"+"is-paquete", "pap"+u"\u00E9"+"is-pergaminho", "pap"+u"\u00E9"+"is-pigmento",
"pap"+u"\u00E9"+"is-pluma", "pap"+u"\u00E9"+"is-porcelana", "pap"+u"\u00E9"+"is-registro", "pap"+u"\u00E9"+"is-rel"+u"\u00E2"+"mpago",
"pap"+u"\u00E9"+"is-tabaco", "pap"+u"\u00E9"+"is-tela", "pap"+u"\u00E9"+"is-t"+u"\u00ED"+"tulo", "parapar"+u"\u00E1"+"s-pe"+u"\u00FA"+"va",
"pardais-boi", "pardos-mato", "pardos-mosto", "pardos-violeta",
"pargos-rosa", "pargos-tereso", "Paris-Dakar", "Paris-T"+u"\u00F3"+"quio",
"parreiras-caapeba", "part"+u"\u00ED"+"culas-espelho", "parus-frade", "parus-jandaia",
"parus-papagaio", "parus-soldado", "parus-tucano", "passageiros-quil"+u"\u00F4"+"metro",
"passarinhos-moscardo", "p"+u"\u00E1"+"ssaros-alfaiate", "p"+u"\u00E1"+"ssaros-angu", "p"+u"\u00E1"+"ssaros-boi",
"p"+u"\u00E1"+"ssaros-frade", "p"+u"\u00E1"+"ssaros-martelo", "p"+u"\u00E1"+"ssaros-ma"+u"\u00FA", "p"+u"\u00E1"+"ssaros-mosca",
"p"+u"\u00E1"+"ssaros-p"+u"\u00EA"+"ndulo", "p"+u"\u00E1"+"ssaros-pretos-soldado", "p"+u"\u00E1"+"ssaros-sino", "p"+u"\u00E1"+"ssaros-sol",
"p"+u"\u00E1"+"ssaros-soldado", "p"+u"\u00E1"+"ssaros-trombeta", "p"+u"\u00E1"+"ssaros-voa", "passes-passe",
"pastos-aranha", "pastos-borla", "patos-arminho", "patos-coral",
"patos-ferr"+u"\u00E3"+"o", "patos-ganso", "patos-lavanco", "patos-mandarim",
"patos-minense", "patos-pataca", "patos-pomer"+u"\u00E2"+"nia", "patu"+u"\u00E1"+"s-balaio",
"paus-"+u"\u00E1"+"gua", "paus-alaz"+u"\u00E3"+"o", "paus-alecrim", "paus-alho",
"paus-ama", "paus-amendoim", "paus-andrade", "paus-arara",
"paus-azeite", "paus-azeitona", "paus-azevim", "paus-bala",
"paus-b"+u"\u00E1"+"lsamo", "paus-barro", "paus-bicho", "paus-boia",
"paus-bosta", "paus-brasil", "paus-breu", "paus-cabra",
"paus-cachorro", "paus-cadeira", "paus-cadela", "paus-caix"+u"\u00E3"+"o",
"paus-caixeta", "paus-candeia", "paus-canela", "paus-caninana",
"paus-canoa", "paus-canudo", "paus-capit"+u"\u00E3"+"o", "paus-cardoso",
"paus-carga", "paus-carne", "paus-carrapato", "paus-carv"+u"\u00E3"+"o",
"paus-caseco", "paus-catinga", "paus-cavalo", "paus-caxique",
"paus-cera", "paus-cetim", "paus-cobra", "paus-colher",
"paus-concha", "paus-conta", "paus-coral", "paus-costa",
"paus-cravo", "paus-crioulo", "paus-cruz", "paus-cumaru",
"paus-danejo", "paus-dumo", "paus-duno", "paus-espeto",
"paus-espinha", "paus-esteira", "paus-facho", "paus-farinha",
"paus-fava", "paus-fede", "paus-feiti"+u"\u00E7"+"o", "paus-ferro",
"paus-f"+u"\u00ED"+"gado", "paus-formiga", "paus-forquilha", "paus-fuma"+u"\u00E7"+"a",
"paus-funcho", "paus-gamela", "paus-gasolina", "paus-gon"+u"\u00E7"+"alo",
"paus-guin"+u"\u00E9", "paus-homem", "paus-jangada", "paus-jantar",
"paus-jerimu", "paus-jerimum", "paus-jiboia", "paus-ji"+u"\u00E7"+"ara",
"paus-jos"+u"\u00E9", "paus-lacre", "paus-l"+u"\u00E1"+"grima", "paus-laranja",
"paus-leite", "paus-lepra", "paus-lim"+u"\u00E3"+"o", "paus-l"+u"\u00ED"+"rio",
"paus-lixa", "paus-ma"+u"\u00E7"+u"\u00E3", "paus-macaco", "paus-macucu",
"paus-mam"+u"\u00E3"+"o", "paus-manteiga", "paus-marfim", "paus-milho",
"paus-moc"+u"\u00F3", "paus-moeda", "paus-mond"+u"\u00E9", "paus-monjolo",
"paus-morcego", "paus-mucumbi", "paus-mungongue", "paus-ob"+u"\u00E1",
"paus-osso", "paus-ov"+u"\u00E1", "paus-pacobala", "paus-papel",
"paus-para"+u"\u00ED"+"ba", "paus-pedra", "paus-pente", "paus-pepu",
"paus-perdiz", "paus-pernambuco", "paus-p"+u"\u00E9"+"rola", "paus-p"+u"\u00EA"+"ssego",
"paus-pia"+u"\u00E7"+"aba", "paus-pia"+u"\u00E7"+"ava", "paus-pimenta", "paus-pombo",
"paus-ponte", "paus-porco", "paus-purga", "paus-qu"+u"\u00E1"+"ssia",
"paus-quiabo", "paus-quizemba", "paus-rainha", "paus-rei",
"paus-ripa", "paus-rosa", "paus-sab"+u"\u00E3"+"o", "paus-sangue",
"paus-santos-macaco", "paus-seringa", "paus-serrote", "paus-tartaruga",
"paus-tatu", "paus-tenente", "paus-terra", "paus-terras-jundia"+u"\u00ED",
"paus-trombeta", "paus-tucano", "paus-urubu", "paus-vaca",
"paus-vala", "paus-vint"+u"\u00E9"+"m", "paus-viola", "paus-violeta",
"paus-visgo", "paus-zebra", "pavios-china", "pav"+u"\u00F5"+"es-bode",
"paxi"+u"\u00FA"+"bas-manjerona", "p"+u"\u00E9"+"s-terra", "p"+u"\u00E9"+"s-vela", "pe"+u"\u00E7"+"as-chave",
"pe"+u"\u00E7"+"as-rev"+u"\u00F3"+"lver", "pedras-bazar", "pedras-bronze", "pedras-canta",
"pedras-curuba", "pedras-escr"+u"\u00F3"+"fula", "pedras-ferro", "pedras-fita",
"pedras-fogo", "pedras-foguense", "pedras-gema", "pedras-gr"+u"\u00E3"+"o",
"pedras-"+u"\u00ED"+"m"+u"\u00E3", "pedras-jade", "pedras-judeu", "pedras-lar",
"pedras-lipse", "pedras-lispe", "pedras-lou"+u"\u00E7"+"a", "pedras-m"+u"\u00E1"+"rmore",
"pedras-mendobi", "pedras-olar", "pedras-pombo", "pedras-rosa",
"pedras-sab"+u"\u00E3"+"o", "pedras-salamandra", "pedras-soda", "pedras-ume",
"pedras-umes-ca"+u"\u00E1", "pegas-boi", "pegas-flor", "peixes-agulha",
"peixes-aipim", "peixes-alecrim", "peixes-anjo", "peixes-aranha",
"peixes-azeite", "peixes-balgado", "peixes-beta", "peixes-boi",
"peixes-borboleta", "peixes-boto", "peixes-briga", "peixes-bruxa",
"peixes-cabra", "peixes-cachimbo", "peixes-cachorro", "peixes-cadela",
"peixes-cana", "peixes-canga", "peixes-c"+u"\u00E3"+"o", "peixes-capim",
"peixes-carago", "peixes-carlim", "peixes-carta", "peixes-cavaleiro",
"peixes-cavalo", "peixes-charuto", "peixes-cigarra", "peixes-cobra",
"peixes-cobrelo", "peixes-coelho", "peixes-coiro", "peixes-congo",
"peixes-corneta", "peixes-corno", "peixes-correio", "peixes-couro",
"peixes-cravo", "peixes-diabo", "peixes-disco", "peixes-doutor",
"peixes-dr"+u"\u00E1"+"cula", "peixes-elefante", "peixes-escama", "peixes-escorpi"+u"\u00E3"+"o",
"peixes-esmargal", "peixes-espada", "peixes-espinho", "peixes-fila",
"peixes-flor", "peixes-folha", "peixes-frade", "peixes-galo",
"peixes-gato", "peixes-homem", "peixes-imperador", "peixes-japoneses-corneta",
"peixes-lagarto", "peixes-le"+u"\u00E3"+"o", "peixes-lenha", "peixes-leque",
"peixes-lima", "peixes-lixa", "peixes-lobo", "peixes-lua",
"peixes-macaco", "peixes-machado", "peixes-madama", "peixes-martelo",
"peixes-moela", "peixes-morcego", "peixes-mulher", "peixes-naire",
"peixes-palmito", "peixes-papagaio", "peixes-para"+u"\u00ED"+"so", "peixes-pau",
"peixes-pedra", "peixes-pena", "peixes-p"+u"\u00E9"+"rsico", "peixes-pescador",
"peixes-pica", "peixes-pi"+u"\u00E7"+"a", "peixes-piloto", "peixes-pimenta",
"peixes-piolho", "peixes-pombo", "peixes-porco", "peixes-prata",
"peixes-prego", "peixes-rabo", "peixes-raposo", "peixes-rato",
"peixes-rei", "peixes-roda", "peixes-sab"+u"\u00E3"+"o", "peixes-sanguessuga",
"peixes-sapo", "peixes-serra", "peixes-sol", "peixes-soldado",
"peixes-sono", "peixes-tamarindo", "peixes-tigre", "peixes-tordo",
"peixes-trombeta", "peixes-vaca", "peixes-vampiro", "peixes-vela",
"peixes-ventosa", "peixes-zebra", "peixes-zorro", "penas-base",
"penas-sargo", "pepinos-castanha", "pequi"+u"\u00E1"+"s-caf"+u"\u00E9", "pequi"+u"\u00E1"+"s-cetim",
"pequi"+u"\u00E1"+"s-mam"+u"\u00E3"+"o", "pequi"+u"\u00E1"+"s-mamona", "pequi"+u"\u00E1"+"s-marfim", "pequi"+u"\u00E1"+"s-pedra",
"peras-"+u"\u00E1"+"gua", "peras-amerim", "peras-amorim", "peras-figo",
"peras-fita", "peras-manteiga", "peras-marmelo", "peras-marquesa",
"peras-noiva", "peras-p"+u"\u00E3"+"o", "peras-p"+u"\u00E9"+"rola", "peras-prata",
"peras-rosa", "peramelas-narigudo", "percevejos-pernilongo", "pereiras-bergamota",
"pereiras-vaqueta", "perfis-diagrama", "per"+u"\u00ED"+"odos-chave", "periquitos-estrela",
"periquitos-gangarra", "periquitos-rei", "periquitos-tapuia", "periquitos-testinha",
"periquitos-tuim", "periquitos-urubu", "periquitos-vassoura", "perobas-caf"+u"\u00E9",
"perobas-cetim", "perobas-poca", "perobas-rosa", "perobas-tambu",
"perobas-tigre", "perp"+u"\u00E9"+"tuas-rosa", "personagens-t"+u"\u00ED"+"tulo", "pescadas-aratanha",
"pescadas-banana", "pescadas-cachorro", "pescadas-cambuci", "pescadas-cambucu",
"pescadas-carvoa", "pescadas-corvina", "pescadas-cutinga", "pescadas-dent"+u"\u00E3"+"o",
"pescadas-enchova", "pescadas-foguete", "pescadas-goiva", "pescadas-jaguara",
"pescadas-manteiga", "pescadas-marmota", "pescadas-rosa", "pescadas-sofia",
"pescadinhas-goete", "pescadinhas-marmota", "pesos-f"+u"\u00F3"+"rmula", "pesos-galo",
"pesos-mosca", "pesos-pena", "pesos-pesado", "pesos-pluma",
"pessoas-chave", "petos-cavalinho", "petos-formigueiro", "pianos-bar",
"picais-polho", "pic"+u"\u00F5"+"es-u"+u"\u00E7"+"u", "pictogramas-signo", "pictogramas-sinal",
"picu"+u"\u00ED"+"s-caboclo", "pimentas-albarr"+u"\u00E3", "pimentas-apu"+u"\u00E3", "pimentas-cereja",
"pimentas-cumari", "pimentas-cumarim", "pimentas-lambari", "pimentas-malagueta",
"pimentas-pitanga", "pimentas-sino", "piment"+u"\u00F5"+"es-ma"+u"\u00E7"+u"\u00E3", "pind"+u"\u00E1"+"s-siririca",
"pingos-pingo", "pinguins-imperador", "pinhas-ata", "pinheiros-aiacauite",
"pinheiros-casquinha", "pinheiros-lar"+u"\u00ED"+"cio", "pinh"+u"\u00F5"+"es-paraguai", "pinos-osso",
"pintas-caldeira", "pintas-cardeira", "pintas-cardim", "pintas-cuia",
"pintas-ferreiro", "pintos-cordeiro", "pi"+u"\u00F5"+"es-madeira", "pir"+u"\u00E1"+"s-andir"+u"\u00E1",
"pir"+u"\u00E1"+"s-bandeira", "pir"+u"\u00E1"+"s-batata", "pir"+u"\u00E1"+"s-inambu", "pir"+u"\u00E1"+"s-tamandu"+u"\u00E1",
"pir"+u"\u00E1"+"s-tapioca", "pir"+u"\u00E1"+"s-tatu", "piraj"+u"\u00E1"+"s-pupunha", "pirangas-sa"+u"\u00ED"+"ra",
"piranhas-cachorro", "piranhas-caju", "piranhas-mapar"+u"\u00E1", "pires-albuquerquense",
"pires-ferreirense", "pires-riense", "p"+u"\u00ED"+"retros-part"+u"\u00EA"+"nio", "pirucaias-ubeba",
"piscos-ribeiro", "pist"+u"\u00F5"+"es-coleira", "pitangas-a"+u"\u00E7"+"u", "pitangas-pimenta",
"pitangas-tra"+u"\u00ED"+"ra", "pitangu"+u"\u00E1"+"s-a"+u"\u00E7"+"u", "piti"+u"\u00E1"+"s-caf"+u"\u00E9", "piti"+u"\u00FA"+"s-a"+u"\u00E7"+"u",
"pitos-cobra", "pitos-formigueiro", "pitombas-a"+u"\u00E7"+"u", "placas-m"+u"\u00E3"+"e",
"planos-sequ"+u"\u00EA"+"ncia", "plantas-alimento", "plantas-camar"+u"\u00E3"+"o", "plantas-m"+u"\u00E3"+"e",
"plantas-milho", "plantas-rainha", "plantas-tel"+u"\u00E9"+"grafo", "pneus-bal"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-abdome", "p"+u"\u00F3"+"s-abd"+u"\u00F4"+"men", "p"+u"\u00F3"+"s-abdominal", "p"+u"\u00F3"+"s-acelera"+u"\u00E7"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-acordo", "p"+u"\u00F3"+"s-adolesc"+u"\u00EA"+"ncia", "p"+u"\u00F3"+"s-albiano", "p"+u"\u00F3"+"s-almo"+u"\u00E7"+"o",
"p"+u"\u00F3"+"s-alveolar", "p"+u"\u00F3"+"s-anestesia", "p"+u"\u00F3"+"s-anest"+u"\u00E9"+"sico", "p"+u"\u00F3"+"s-apocal"+u"\u00ED"+"ptico",
"p"+u"\u00F3"+"s-apocal"+u"\u00ED"+"tico", "p"+u"\u00F3"+"s-apost"+u"\u00F3"+"lico", "p"+u"\u00F3"+"s-aptiano", "p"+u"\u00F3"+"s-articulado",
"p"+u"\u00F3"+"s-auricular", "p"+u"\u00F3"+"s-barba", "p"+u"\u00F3"+"s-b"+u"\u00E9"+"lico", "p"+u"\u00F3"+"s-b"+u"\u00ED"+"blico",
"p"+u"\u00F3"+"s-Big", "p"+u"\u00F3"+"s-boca", "p"+u"\u00F3"+"s-braquial", "p"+u"\u00F3"+"s-cabralino",
"p"+u"\u00F3"+"s-cartilag"+u"\u00ED"+"neo", "p"+u"\u00F3"+"s-cirurgia", "p"+u"\u00F3"+"s-cir"+u"\u00FA"+"rgico", "p"+u"\u00F3"+"s-classicismo",
"p"+u"\u00F3"+"s-cl"+u"\u00E1"+"ssico", "p"+u"\u00F3"+"s-clora"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-coito", "p"+u"\u00F3"+"s-colonial",
"p"+u"\u00F3"+"s-combust"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-comicial", "p"+u"\u00F3"+"s-complexo", "p"+u"\u00F3"+"s-comunh"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-comunismo", "p"+u"\u00F3"+"s-conciliar", "p"+u"\u00F3"+"s-concurso", "p"+u"\u00F3"+"s-condicionar",
"p"+u"\u00F3"+"s-conflito", "p"+u"\u00F3"+"s-consulado", "p"+u"\u00F3"+"s-Copenhague", "p"+u"\u00F3"+"s-cr"+u"\u00E9"+"dito",
"p"+u"\u00F3"+"s-crise", "p"+u"\u00F3"+"s-datar", "p"+u"\u00F3"+"s-decisional", "p"+u"\u00F3"+"s-dental",
"p"+u"\u00F3"+"s-dilata"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-diluviano", "p"+u"\u00F3"+"s-dorsal", "p"+u"\u00F3"+"s-dorso",
"p"+u"\u00F3"+"s-doutor", "p"+u"\u00F3"+"s-doutorado", "p"+u"\u00F3"+"s-doutoramento", "p"+u"\u00F3"+"s-dublagem",
"p"+u"\u00F3"+"s-ed"+u"\u00EA"+"mico", "p"+u"\u00F3"+"s-eleitoral", "p"+u"\u00F3"+"s-embrion"+u"\u00E1"+"rio", "p"+u"\u00F3"+"s-emiss"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-escolar", "p"+u"\u00F3"+"s-escrito", "p"+u"\u00F3"+"s-espermatog"+u"\u00EA"+"nese", "p"+u"\u00F3"+"s-espermatogen"+u"\u00E9"+"tico",
"p"+u"\u00F3"+"s-esqui", "p"+u"\u00F3"+"s-esten"+u"\u00F3"+"tico", "p"+u"\u00F3"+"s-estruturalismo", "p"+u"\u00F3"+"s-evento",
"p"+u"\u00F3"+"s-exerc"+u"\u00ED"+"cio", "p"+u"\u00F3"+"s-ex"+u"\u00ED"+"lico", "p"+u"\u00F3"+"s-ex"+u"\u00ED"+"lio", "p"+u"\u00F3"+"s-fato",
"p"+u"\u00F3"+"s-fen"+u"\u00ED"+"cio", "p"+u"\u00F3"+"s-fixado", "p"+u"\u00F3"+"s-flandriano", "p"+u"\u00F3"+"s-flora"+u"\u00E7"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-forma"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-franquista", "p"+u"\u00F3"+"s-frontal", "p"+u"\u00F3"+"s-ganglionar",
"p"+u"\u00F3"+"s-g"+u"\u00EA"+"nito", "p"+u"\u00F3"+"s-gen"+u"\u00F4"+"mico", "p"+u"\u00F3"+"s-glacial", "p"+u"\u00F3"+"s-glaci"+u"\u00E1"+"rio",
"p"+u"\u00F3"+"s-gradua"+u"\u00E7"+u"\u00E3"+"o", "P"+u"\u00F3"+"s-Gradua"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-graduado", "P"+u"\u00F3"+"s-Graduado",
"p"+u"\u00F3"+"s-graduando", "p"+u"\u00F3"+"s-graduar", "p"+u"\u00F3"+"s-guerra", "p"+u"\u00F3"+"s-Guerra",
"p"+u"\u00F3"+"s-hipn"+u"\u00F3"+"tico", "p"+u"\u00F3"+"s-hipofis"+u"\u00E1"+"rio", "p"+u"\u00F3"+"s-hip"+u"\u00F3"+"fise", "p"+u"\u00F3"+"s-hom"+u"\u00E9"+"rico",
"p"+u"\u00F3"+"s-ictal", "p"+u"\u00F3"+"s-impressionismo", "p"+u"\u00F3"+"s-impression"+u"\u00ED"+"stico", "p"+u"\u00F3"+"s-industrial",
"p"+u"\u00F3"+"s-instala"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-interna"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-jogo", "p"+u"\u00F3"+"s-kantianismo",
"p"+u"\u00F3"+"s-kantian"+u"\u00ED"+"stico", "p"+u"\u00F3"+"s-kantiano", "p"+u"\u00F3"+"s-kantismo", "p"+u"\u00F3"+"s-kant"+u"\u00ED"+"stico",
"p"+u"\u00F3"+"s-Kioto", "p"+u"\u00F3"+"s-laboral", "p"+u"\u00F3"+"s-larva", "p"+u"\u00F3"+"s-letal",
"p"+u"\u00F3"+"s-maestrichtiano", "p"+u"\u00F3"+"s-magm"+u"\u00E1"+"tico", "p"+u"\u00F3"+"s-m"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-matura"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-medieval", "p"+u"\u00F3"+"s-m"+u"\u00E9"+"dio", "p"+u"\u00F3"+"s-membranoso",
"p"+u"\u00F3"+"s-menopausa", "p"+u"\u00F3"+"s-menopausado", "p"+u"\u00F3"+"s-menop"+u"\u00E1"+"usico", "p"+u"\u00F3"+"s-mensal"+u"\u00E3"+"o",
"p"+u"\u00F3"+"s-mento", "p"+u"\u00F3"+"s-meridiano", "p"+u"\u00F3"+"s-milagre", "p"+u"\u00F3"+"s-milenarismo",
"p"+u"\u00F3"+"s-militar", "p"+u"\u00F3"+"s-minimalismo", "p"+u"\u00F3"+"s-mit"+u"\u00F3"+"tico", "p"+u"\u00F3"+"s-modernismo",
"P"+u"\u00F3"+"s-Modernismo", "p"+u"\u00F3"+"s-modern"+u"\u00ED"+"stico", "p"+u"\u00F3"+"s-moderno", "p"+u"\u00F3"+"s-mortal",
"p"+u"\u00F3"+"s-morte", "p"+u"\u00F3"+"s-natal", "p"+u"\u00F3"+"s-nominal", "p"+u"\u00F3"+"s-nuclear",
"p"+u"\u00F3"+"s-nupcial", "p"+u"\u00F3"+"s-occip"+u"\u00FA"+"cio", "p"+u"\u00F3"+"s-ocular", "p"+u"\u00F3"+"s-operat"+u"\u00F3"+"rio",
"p"+u"\u00F3"+"s-opercular", "p"+u"\u00F3"+"s-oral", "p"+u"\u00F3"+"s-orbital", "p"+u"\u00F3"+"s-pago",
"p"+u"\u00F3"+"s-palatal", "p"+u"\u00F3"+"s-palato", "p"+u"\u00F3"+"s-parametrizado", "p"+u"\u00F3"+"s-parnasiano",
"p"+u"\u00F3"+"s-parto", "p"+u"\u00F3"+"s-pasteuriano", "p"+u"\u00F3"+"s-pastoriano", "p"+u"\u00F3"+"s-peitoral",
"p"+u"\u00F3"+"s-perna", "p"+u"\u00F3"+"s-perturba"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-piloc"+u"\u00EA"+"nico", "p"+u"\u00F3"+"s-plioceno",
"p"+u"\u00F3"+"s-positivismo", "p"+u"\u00F3"+"s-prandial", "p"+u"\u00F3"+"s-premia"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-processamento",
"p"+u"\u00F3"+"s-produ"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-produtor", "p"+u"\u00F3"+"s-puberal", "p"+u"\u00F3"+"s-puerperal",
"p"+u"\u00F3"+"s-punk", "p"+u"\u00F3"+"s-qualifica"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-Quioto", "p"+u"\u00F3"+"s-radical",
"p"+u"\u00F3"+"s-reforma", "p"+u"\u00F3"+"s-revolu"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-Revolu"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-romano",
"p"+u"\u00F3"+"s-rom"+u"\u00E2"+"ntico", "p"+u"\u00F3"+"s-romantismo", "p"+u"\u00F3"+"s-seco", "p"+u"\u00F3"+"s-secund"+u"\u00E1"+"rio",
"p"+u"\u00F3"+"s-separa"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-sequelamento", "p"+u"\u00F3"+"s-siluriano", "p"+u"\u00F3"+"s-simbolismo",
"p"+u"\u00F3"+"s-simbol"+u"\u00ED"+"stico", "p"+u"\u00F3"+"s-sin"+u"\u00E1"+"ptico", "p"+u"\u00F3"+"s-sincronia", "p"+u"\u00F3"+"s-sincr"+u"\u00F4"+"nico",
"p"+u"\u00F3"+"s-sincroniza"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-sincronizar", "p"+u"\u00F3"+"s-sincroniz"+u"\u00E1"+"vel", "p"+u"\u00F3"+"s-socr"+u"\u00E1"+"tico",
"p"+u"\u00F3"+"s-subversivo", "p"+u"\u00F3"+"s-tect"+u"\u00F4"+"nico", "p"+u"\u00F3"+"s-temporada", "p"+u"\u00F3"+"s-temporal",
"p"+u"\u00F3"+"s-teodosiano", "p"+u"\u00F3"+"s-terci"+u"\u00E1"+"rio", "p"+u"\u00F3"+"s-textual", "p"+u"\u00F3"+"s-t"+u"\u00F4"+"nico",
"p"+u"\u00F3"+"s-traducional", "p"+u"\u00F3"+"s-transcricional", "p"+u"\u00F3"+"s-transfusional", "p"+u"\u00F3"+"s-translacional",
"p"+u"\u00F3"+"s-tratamento", "p"+u"\u00F3"+"s-trauma", "p"+u"\u00F3"+"s-traum"+u"\u00E1"+"tico", "P"+u"\u00F3"+"s-Traum"+u"\u00E1"+"tico",
"p"+u"\u00F3"+"s-vacina"+u"\u00E7"+u"\u00E3"+"o", "p"+u"\u00F3"+"s-vacinal", "p"+u"\u00F3"+"s-v"+u"\u00E9"+"dico", "p"+u"\u00F3"+"s-velar",
"p"+u"\u00F3"+"s-venda", "p"+u"\u00F3"+"s-verbal", "p"+u"\u00F3"+"s-vida", "p"+u"\u00F3"+"s-voc"+u"\u00E1"+"lico",
"p"+u"\u00F3"+"s-werberniano", "p"+u"\u00F3"+"s-zonatoso", "po"+u"\u00E7"+"os-caldense", "podrid"+u"\u00F5"+"es-castanha",
"poil"+u"\u00F5"+"es-forro", "polcas-mazurca", "polegadas-libra", "p"+u"\u00F3"+"lvoras-calhau",
"p"+u"\u00F3"+"lvoras-dinamite", "pomas-candil", "pombas-cabocla", "pombas-carij"+u"\u00F3",
"pombas-cascavel", "pombas-espelho", "pombas-juriti", "pombas-pararu",
"pombas-pariri", "pombas-picu"+u"\u00ED", "pombas-puca"+u"\u00E7"+"u", "pombas-rola",
"pombas-trocal", "pombinhas-cascavel", "pombos-anamb"+u"\u00E9", "pombos-andorinha",
"pombos-cambalhota", "pombos-carneau", "pombos-carrier", "pombos-cetim",
"pombos-cetineta", "pombos-correio", "pombos-drag"+u"\u00E3"+"o", "pombos-gravatinha",
"pombos-lua", "pombos-nicobar", "pombos-pega", "pombos-tambor",
"pombos-torcaz", "pombos-trocal", "pombos-trocaz", "pombos-tumbler",
"pontas-cabe"+u"\u00E7"+"a", "pontas-dentro", "pontes-aqueduto", "pontes-canal",
"pontes-gestalense", "pontes-safena", "pontos-chave", "pontos-limite",
"pontos-vogal", "popula"+u"\u00E7"+u"\u00F5"+"es-alvo", "porcos-espim", "porcos-espinho",
"porcos-pisco", "porcos-ribeiro", "porcos-veado", "portos-abrigo",
"portugu"+u"\u00EA"+"s-espanhol", "posi"+u"\u00E7"+u"\u00F5"+"es-chave", "postos-chave", "pot"+u"\u00F3"+"s-pimenta",
"poupas-boubela", "pre"+u"\u00E7"+"os-alvo", "pregos-cachorro", "pretos-a"+u"\u00E7"+"a",
"pretos-amarelo", "pretos-azeviche", "pretos-carv"+u"\u00E3"+"o", "priores-vig"+u"\u00E1"+"rio",
"pris"+u"\u00F5"+"es-albergue", "processos-chave", "produ"+u"\u00E7"+u"\u00F5"+"es-fam"+u"\u00ED"+"lia", "profissionais-padr"+u"\u00E3"+"o",
"Programas-fonte","programas-fonte", "programas-objeto", "proj"+u"\u00E9"+"teis-foguete", "projetos-piloto",
"projetos-tipo", "prote"+u"\u00ED"+"nas-chave", "p"+u"\u00FA"+"blicos-alvo", "pu"+u"\u00E7"+u"\u00E1"+"s-coca",
"pu"+u"\u00E7"+u"\u00E1"+"s-poca", "pulas-pula", "pulg"+u"\u00F5"+"es-carmim", "pulseiras-rel"+u"\u00F3"+"gio",
"punhais-baioneta", "punhos-punhete", "pupunhas-piranga", "pupunhas-pitanga",
"puracis-iara", "putumui"+u"\u00FA"+"s-iririb"+u"\u00E1", "putumujus-iririb"+u"\u00E1", "quadros-alvo",
"quartzos-fantasma", "quartzos-m"+u"\u00F3"+"rion", "qu"+u"\u00E1"+"ssias-simaruba", "queijos-cavalo",
"queixas-crime", "quenqu"+u"\u00E9"+"ns-caiap"+u"\u00F3", "quest"+u"\u00F5"+"es-chave", "quic"+u"\u00EA"+"s-acica",
"quilogramas-for"+u"\u00E7"+"a", "quilogramas-metro", "quilogramas-peso", "quil"+u"\u00F4"+"metros-passageiro",
"quilovares-hora", "quilov"+u"\u00E1"+"tios-hora", "quilovolts-ampere", "quilovolts-amp\u00E8re",
"quilowats-hora", "quinas-calisaia", "quinas-carib"+u"\u00E9", "quinas-cruzeiro",
"quinas-laranjeira", "quinas-pereira", "quissandas-muxito", "quotas-parte",
"r"+u"\u00E3"+"s-cachorro", "r"+u"\u00E3"+"s-manteiga", "rabelos-coelha", "r"+u"\u00E1"+"dios-cassete",
"r"+u"\u00E1"+"dios-gravador", "r"+u"\u00E1"+"dios-vitrola", "raias-arara", "raias-boi",
"raias-borboleta", "raias-chita", "raias-gererera", "raias-jamanta",
"raias-leopardo", "raias-lixa", "raias-ma"+u"\u00E7"+u"\u00E3", "raias-manteiga",
"raias-prego", "raias-sapo", "raias-ticonha", "raias-viola",
"rainhas-m"+u"\u00E3"+"e", "ra"+u"\u00ED"+"zes-escora", "ratos-calunga", "ratos-canguru",
"ratos-capivara", "ratos-catita", "ratos-chinchila", "ratos-cor"+u"\u00F3",
"ratos-cuandu", "ratos-espinho", "ratos-musgo", "ratos-tor"+u"\u00F3",
"redes-fole", "regi"+u"\u00F5"+"es-continente", "relacionamentos-rel"+u"\u00E2"+"mpago", "rel"+u"\u00F3"+"gios-calend"+u"\u00E1"+"rio",
"rel"+u"\u00F3"+"gios-pulseira", "rel"+u"\u00F3"+"gios-vassoura", "reservas-provis"+u"\u00E3"+"o", "reuni"+u"\u00F5"+"es-almo"+u"\u00E7"+"o",
"robalos-aratanha", "robalos-estoque", "robalos-flecha", "robalos-miraguaia",
"roceiros-planta", "rochas-reservat"+u"\u00F3"+"rio", "rodas-coroa", "rodas-pagode",
"rodas-vinho", "rolas-baga"+u"\u00E7"+"o", "rolas-capa", "rolas-cascavel",
"rolas-paje"+u"\u00FA", "rolas-pau", "rolhas-esp"+u"\u00E9"+"cie", "rolinhas-cascavel",
"romances-rel"+u"\u00E2"+"mpago", "romances-rio", "roncadores-taboca", "rosas-am"+u"\u00E9"+"lia",
"rosas-bilanca", "rosas-ch"+u"\u00E1", "rosas-madeira", "rosas-maravilha",
"rosas-mesquita", "rosas-mosqueta", "rosas-palmeir"+u"\u00E3"+"o", "roxos-terra",
"rubis-almandina", "rubis-balaio", "rubis-calif"+u"\u00F3"+"rnia", "rubis-espinela",
"rubis-estrela", "rubis-top"+u"\u00E1"+"zio", "sabi"+u"\u00E1"+"s-anu", "sabi"+u"\u00E1"+"s-barranco",
"sabi"+u"\u00E1"+"s-cachorro", "sabi"+u"\u00E1"+"s-cavalo", "sabi"+u"\u00E1"+"s-coleira", "sabi"+u"\u00E1"+"s-gong"+u"\u00E1",
"sabi"+u"\u00E1"+"s-gute", "sabi"+u"\u00E1"+"s-laranja", "sabi"+u"\u00E1"+"s-laranjeira", "sabi"+u"\u00E1"+"s-pimenta",
"sab"+u"\u00F5"+"es-dentifr"+u"\u00ED"+"cio", "sabres-baioneta", "sabres-punhal", "sachas-cebola",
"sachas-manga", "sachas-rosa", "safiras-estrela", "saganhos-oiro",
"saganhos-ouro", "saguis-carantinga", "saguis-imperador", "saguins-caratinga",
"saguins-imperador", "sa"+u"\u00ED"+"s-andorinha", "sa"+u"\u00ED"+"s-arara", "sa"+u"\u00ED"+"s-papagaio",
"sa"+u"\u00ED"+"s-sapucaia", "saias-bal"+u"\u00E3"+"o", "saias-cal"+u"\u00E7"+"a", "saias-cal"+u"\u00E7"+u"\u00E3"+"o",
"salas-cofre", "sal"+u"\u00E1"+"rios-base", "sal"+u"\u00E1"+"rios-fam"+u"\u00ED"+"lia", "sal"+u"\u00E1"+"rios-hora",
"sal"+u"\u00E1"+"rios-maternidade", "sales-oliveirense", "salm"+u"\u00F5"+"es-rei", "salm"+u"\u00F5"+"es-truta",
"salsas-lim"+u"\u00E3"+"o", "salsas-proa", "samambaias-cumaru", "samambaias-mosquito",
"sambas-can"+u"\u00E7"+u"\u00E3"+"o", "sambas-choro", "sambas-enredo", "sambas-exalta"+u"\u00E7"+u"\u00E3"+"o",
"sambas-len"+u"\u00E7"+"o", "sambas-roda", "sanha"+u"\u00E7"+"os-coqueiro", "sanha"+u"\u00E7"+"os-frade",
"sanha"+u"\u00E7"+"os-pardo", "sanha"+u"\u00E7"+"us-coqueiro", "sanha"+u"\u00E7"+"us-frade", "santos-dumontense",
"Santos-Guaruj"+u"\u00E1", "sapajus-aurora", "sapos-aru", "sapos-boi",
"sapos-cachorro", "sapos-folha", "sapos-intanha", "sapos-peixe",
"sapos-pipa", "sapos-pulga", "saquis-monge", "saracuras-muir"+u"\u00E1",
"sarap"+u"\u00F3"+"s-tuvira", "sardas-"+u"\u00E1"+"gata", "sardinhas-bandeira", "sardinhas-bandeirada",
"sardinhas-fac"+u"\u00E3"+"o", "sardinhas-gato", "sardinhas-laje", "sardinhas-maromba",
"sardinhas-prata", "sargos-veado", "sassafr"+u"\u00E1"+"s-amarelo", "sau"+u"\u00ED"+"s-caratinga",
"saui"+u"\u00E1"+"s-bandeira", "sa"+u"\u00FA"+"nas-rolha", "savanas-parque", "sebos-confeito",
"segundos-luz", "seguros-democracia", "seguros-desemprego", "seguros-garantia",
"seguros-maternidade", "seguros-sa"+u"\u00FA"+"de", "semens-contra", "s"+u"\u00EA"+"menes-contra",
"sen"+u"\u00E1"+"tus-consulto", "sequestros-rel"+u"\u00E2"+"mpago", "serigados-badejo", "serigados-cherne",
"serigados-sab"+u"\u00E3"+"o", "serigados-tapu"+u"\u00E3", "serigados-xerre", "seringueiras-chicote",
"seringueiras-ita"+u"\u00FA"+"ba", "seringueiras-tambaqui", "serpentinas-jararaca", "serras-abaixo",
"serras-acima", "servi"+u"\u00E7"+"os-chave", "setores-alvo", "setores-chave",
"sic"+u"\u00F4"+"moros-figueira", "sigmas-zero", "signos-salom"+u"\u00E3"+"o", "silenes-aurora",
"s"+u"\u00ED"+"licas-gel", "silvas-framboesa", "simarubas-copaia", "s"+u"\u00ED"+"miles-pen"+u"\u00ED"+"nsula",
"sinh"+u"\u00E1"+"s-carand"+u"\u00E1", "sintomas-chave", "siris-ba"+u"\u00FA", "siris-candeia",
"siris-chita", "siris-patola", "sirigados-badejo", "sirigados-cherne",
"sirigados-sab"+u"\u00E3"+"o", "sirigados-tapu"+u"\u00E3", "sirigados-xerre", "situa"+u"\u00E7"+u"\u00F5"+"es-limite",
"situa"+u"\u00E7"+u"\u00F5"+"es-problema", "soc"+u"\u00F3"+"s-boi", "sof"+u"\u00E1"+"s-bicama", "sof"+u"\u00E1"+"s-cama",
"solas-sapato", "sorgos-vassoura", "subst"+u"\u00E2"+"ncias-tra"+u"\u00E7"+"o", "surubins-caparari",
"surubins-chicote", "surubins-lima", "surubins-lira", "surubins-mena",
"surubins-pirambubu", "surucucus-patioba", "surucucus-pindoba", "surucucus-tapete",
"tai"+u"\u00E1"+"s-jararaca", "tai"+u"\u00E1"+"s-jiboia", "tai"+u"\u00E1"+"s-mangarito", "taia"+u"\u00E7"+"us-caiaguira",
"taia"+u"\u00E7"+"us-carapi"+u"\u00E1", "taj"+u"\u00E1"+"s-jiboia", "taj"+u"\u00E1"+"s-mangarito", "taja"+u"\u00E7"+"us-caiaguira",
"taja"+u"\u00E7"+"us-carapi"+u"\u00E1", "tamandu"+u"\u00E1"+"s-bandeira", "tamandu"+u"\u00E1"+"s-cavalo", "tamandu"+u"\u00E1"+"s-colete",
"tamandu"+u"\u00E1"+"s-jaleco", "tamandu"+u"\u00E1"+"s-macambira", "tambores-on"+u"\u00E7"+"a", "tangerinas-cravo",
"tanques-rede", "tapereb"+u"\u00E1"+"s-cedro", "taperib"+u"\u00E1"+"s-cedro", "tapiras-caapora",
"taquaras-bambu", "tarifas-pr"+u"\u00EA"+"mio", "tartaranh"+u"\u00F5"+"es-pigargo", "tartarugas-aligator",
"tartarugas-caix"+u"\u00E3"+"o", "tartarugas-lira", "tatus-bola", "tatus-canastra",
"tatus-cascudo", "tatus-folha", "tatus-galinha", "tatus-mulita",
"tatus-veado", "tatus-vespa", "teatros-revista", "telegramas-vale",
"telhas-canal", "temas-chave", "tendas-abrigo", "tentos-carolina",
"teodolitos-b"+u"\u00FA"+"ssola", "terras-m"+u"\u00E3"+"e", "testes-diagn"+u"\u00F3"+"stico", "textos-base",
"textos-foguete", "Textos-fonte", "textos-fonte", "textos-legenda", "textos-piloto",
"tias-rosa", "ti"+u"\u00E9"+"s-cabur"+u"\u00E9", "ti"+u"\u00E9"+"s-fogo", "ti"+u"\u00E9"+"s-galo",
"ti"+u"\u00E9"+"s-gua"+u"\u00E7"+"us-paroara", "ti"+u"\u00E9"+"s-piranga", "ti"+u"\u00E9"+"s-sangue", "ti"+u"\u00EA"+"s-sangue",
"ti"+u"\u00E9"+"s-veludo", "timb"+u"\u00F3"+"s-botic"+u"\u00E1"+"rio", "timb"+u"\u00F3"+"s-ca"+u"\u00E1", "timb"+u"\u00F3"+"s-carajuru",
"timb"+u"\u00F3"+"s-catinga", "timb"+u"\u00F3"+"s-cip"+u"\u00F3", "timb"+u"\u00F3"+"s-macaquinho", "timb"+u"\u00F3"+"s-pau",
"timb"+u"\u00F3"+"s-taturuaia", "timb"+u"\u00F3"+"s-titica", "timb"+u"\u00F3"+"s-urucu", "tinguis-bot"+u"\u00F3",
"tinguis-capeta", "tintas-c"+u"\u00E3"+"o", "tintas-francisca", "tipos-altura",
"t"+u"\u00ED"+"quetes-alimenta"+u"\u00E7"+u"\u00E3"+"o", "t"+u"\u00ED"+"quetes-refei"+u"\u00E7"+u"\u00E3"+"o", "t"+u"\u00ED"+"quetes-restaurante", "t"+u"\u00ED"+"tulos-chave",
"tobias-barretense", "todos-santense", "tojos-chamusco", "tojos-gat"+u"\u00E3"+"o",
"tojos-gatum", "tojos-gatunha", "tom"+u"\u00E1"+"s-aquinense", "tom"+u"\u00E1"+"s-gonzaguense",
"tomadas-banana", "tomates-cabacinha", "tomates-caj"+u"\u00E1", "tomates-cereja",
"tomates-chimango", "tomates-groselha", "tomates-p"+u"\u00EA"+"ra", "toneladas-for"+u"\u00E7"+"a",
"toneladas-milha", "toneladas-quil"+u"\u00F4"+"metro", "top"+u"\u00E1"+"zios-ba"+u"\u00ED"+"a", "top"+u"\u00E1"+"zios-ouro",
"top"+u"\u00E1"+"zios-palmira", "tordos-pisco", "tornos-rev"+u"\u00F3"+"lver", "tr"+u"\u00E1"+"s-anteontem",
"tra"+u"\u00ED"+"ras-pedra", "tra"+u"\u00ED"+"ras-pixuna", "tra"+u"\u00ED"+"ras-pix"+u"\u00FA"+"ria", "travessas-cal"+u"\u00E7"+"o",
"tr"+u"\u00EA"+"s-agostense", "tr"+u"\u00EA"+"s-arroiense", "tr"+u"\u00EA"+"s-barrense", "tr"+u"\u00EA"+"s-biquense",
"tr"+u"\u00EA"+"s-boquense", "tr"+u"\u00EA"+"s-bracense", "tr"+u"\u00EA"+"s-cachoeirense", "tr"+u"\u00EA"+"s-casense",
"tr"+u"\u00EA"+"s-coroense", "tr"+u"\u00EA"+"s-coronense", "tr"+u"\u00EA"+"s-correguense", "tr"+u"\u00EA"+"s-cruzense",
"tr"+u"\u00EA"+"s-divisense", "tr"+u"\u00EA"+"s-estradense", "tr"+u"\u00EA"+"s-figueirense", "tr"+u"\u00EA"+"s-forquilhense",
"tr"+u"\u00EA"+"s-fronteirense", "tr"+u"\u00EA"+"s-ilhense", "tr"+u"\u00EA"+"s-irmanense", "tr"+u"\u00EA"+"s-ladeirense",
"tr"+u"\u00EA"+"s-lagoano", "tr"+u"\u00EA"+"s-lagoense", "tr"+u"\u00EA"+"s-maiense", "tr"+u"\u00EA"+"s-mariense",
"tr"+u"\u00EA"+"s-morrense", "tr"+u"\u00EA"+"s-outubrense", "tr"+u"\u00EA"+"s-passense", "tr"+u"\u00EA"+"s-pedrense",
"tr"+u"\u00EA"+"s-pinheirense", "tr"+u"\u00EA"+"s-pocense", "tr"+u"\u00EA"+"s-pontano", "tr"+u"\u00EA"+"s-pontense",
"tr"+u"\u00EA"+"s-ranchense", "tr"+u"\u00EA"+"s-riense", "tr"+u"\u00EA"+"s-vendense", "trem"+u"\u00EA"+"s-arroxeado",
"trem"+u"\u00EA"+"s-branco", "trem"+u"\u00EA"+"s-preto", "trem"+u"\u00EA"+"s-ribeirinho", "trem"+u"\u00EA"+"s-rijo",
"trem"+u"\u00EA"+"s-ruivo", "trens-bala", "trens-hospital", "trens-tipo",
"trevos-ma"+u"\u00E7"+"aroca", "trigos-arroz", "trigos-barbela", "trincheiras-abrigo",
"tubar"+u"\u00F5"+"es-anequim", "tubar"+u"\u00F5"+"es-baleia", "tubar"+u"\u00F5"+"es-elefante", "tubar"+u"\u00F5"+"es-grelha",
"tubar"+u"\u00F5"+"es-jaguara", "tubar"+u"\u00F5"+"es-lim"+u"\u00E3"+"o", "tubar"+u"\u00F5"+"es-martelo", "tubar"+u"\u00F5"+"es-raposa",
"tubar"+u"\u00F5"+"es-tigre", "tubar"+u"\u00F5"+"es-tintureira", "tucanos-boi", "tucanos-cachorrinho",
"tucanos-pacova", "tucum"+u"\u00E3"+"s-arara", "tucum"+u"\u00E3"+"s-piranga", "tucum"+u"\u00E3"+"s-piririca",
"tucum"+u"\u00E3"+"s-purupuru", "tucuns-maraj"+u"\u00E1", "tucunar"+u"\u00E9"+"s-cip"+u"\u00F3", "tucunar"+u"\u00E9"+"s-mere"+u"\u00E7"+u"\u00E1",
"tuiparas-estrelinha", "tuiu"+u"\u00E9"+"s-tipi", "tumbas-sat"+u"\u00E9"+"lite", "turbilhos-escadinha",
"turbilhos-roseta", "turistas-padr"+u"\u00E3"+"o", "turmas-piloto", "ua"+u"\u00E7"+"a"+u"\u00ED"+"s-chumbo",
"ubaias-muxama", "ubaranas-camorim", "ubaranas-jiruma", "ubaranas-rato",
"ubins-rana", "uir"+u"\u00E1"+"s-angu", "umiris-b"+u"\u00E1"+"lsamo", "unguentos-basilic"+u"\u00E3"+"o",
"unguentos-popule"+u"\u00E3"+"o", "unhas-gata", "unidades-base", "universos-bolha",
"universos-ilha", "unu"+u"\u00E9"+"s-bolina", "upas-antiar", "upas-tieut"+u"\u00E9",
"usinas-piloto", "usinas-prot"+u"\u00F3"+"tipo", "usu"+u"\u00E1"+"rios-chave", "uvas-c"+u"\u00E3"+"o",
"uvas-espim", "uvas-espinha", "uvas-isabel", "uvas-ma"+u"\u00E7"+u"\u00E3",
"uvas-rei", "uvas-tinta", "vag"+u"\u00F5"+"es-cavalari"+u"\u00E7"+"a", "vag"+u"\u00F5"+"es-cisterna",
"vag"+u"\u00F5"+"es-leito", "vag"+u"\u00F5"+"es-pipa", "vag"+u"\u00F5"+"es-restaurante", "vag"+u"\u00F5"+"es-sal"+u"\u00E3"+"o",
"vag"+u"\u00F5"+"es-tanque", "vag"+u"\u00F5"+"es-tremonha", "vales-alimenta"+u"\u00E7"+u"\u00E3"+"o", "vales-brinde",
"vales-cultura", "Vales-Cultura", "vales-refei"+u"\u00E7"+u"\u00E3"+"o", "vales-transporte",
"valores-oiro", "valores-ouro", "valores-papel", "vamos-embora",
"vamos-peneirar", "vars-hora", "vars-hor"+u"\u00ED"+"metro", "vari"+u"\u00E1"+"veis-chave",
"varicelas-z"+u"\u00F3"+"ster", "v"+u"\u00E1"+"tios-hora", "veados-boror"+u"\u00F3", "veados-camocica",
"veados-canela", "veados-caracu", "veados-cariacu", "veados-foboca",
"veados-garapu", "veados-guatapar"+u"\u00E1", "veados-sua"+u"\u00E7"+"utinga", "veados-vir"+u"\u00E1",
"velas-luz", "velames-metro", "verbos-suporte", "verdes-abacate",
"verdes-"+u"\u00E1"+"gua", "verdes-alface", "verdes-alga", "verdes-alvo",
"verdes-azeitona", "verdes-bandeira", "verdes-bexiga", "verdes-bronze",
"verdes-cinza", "verdes-claro", "verdes-cr"+u"\u00E9", "verdes-escuro",
"verdes-esmeralda", "verdes-flavo", "verdes-gai", "verdes-gaio",
"verdes-garrafa", "verdes-imperial", "verdes-jade", "verdes-limo",
"verdes-limoso", "verdes-malva", "verdes-mar", "verdes-mineral",
"verdes-montanha", "verdes-musgo", "verdes-negro", "verdes-oliva",
"verdes-pav"+u"\u00E3"+"o", "verdes-piscina", "verdes-salsa", "verdes-verdete",
"verdes-viena", "verdes-virgo", "verdeais-branco", "verdeais-cobran"+u"\u00E7"+"osa",
"verdeais-quebran"+u"\u00E7"+"osa", "verdeais-rija", "verdeais-tinta", "verdelhos-feij"+u"\u00E3"+"o",
"vermes-arame", "vermes-cabelo", "vermes-chicote", "vermes-le"+u"\u00E3"+"o",
"vermelhos-aric"+u"\u00F3", "vermelhos-cachorro", "vermelhos-cereja", "vermelhos-cioba",
"vermelhos-henrique", "vermelhos-paramirim", "vermelhos-p"+u"\u00FA"+"rpura", "vermelhos-salm"+u"\u00E3"+"o",
"vermelhos-siri"+u"\u00FA"+"ba", "vernizes-emuls"+u"\u00E3"+"o", "vespas-tatu", "viaturas-muni"+u"\u00E7"+u"\u00E3"+"o",
"v"+u"\u00ED"+"boras-alcatifa", "vidoeiros-cerejeira", "vilas-boense", "vimeiros-br"+u"\u00F3"+"zio",
"vinh"+u"\u00E1"+"ticos-cabeleira", "v"+u"\u00ED"+"rus-base", "vogas-avante", "volumes-minuto",
"vor"+u"\u00E1"+"s-boi", "votos-vista", "watts-hora", "watts-minuto",
"watts-segundo", "z"+u"\u00E9"+"s-ningu"+u"\u00E9"+"m", "zonas-alvo"])


# Termos compostos com duas palavras iguais
igualTc = set(["agar", u"\u00E1"+"gar", "agora", "assim",
"baco", "baga", "bang", "bango",
"bangu", "bangue", "bate", "bau",
"b"+u"\u00E9"+"u", "bilo", "bique", "bira",
"biri", "bite", "bole", "boles",
"buia", "bula", "bule", "bules",
"bum", "c"+u"\u00EA", "ch"+u"\u00E3", "chape",
"ch"+u"\u00E1"+"s", "chau", "cheira", "chia",
"chica", "chico", "chora", "chuca",
"chupa", "chupas", "cina", "cloque",
"coi"+u"\u00F3", "coi"+u"\u00F3"+"s", "c"+u"\u00F4"+"ncavo", "convexo",
"cor"+u"\u00F3", "corre", "corres", "cospe",
"cospes", "cot"+u"\u00F3", "cr"+u"\u00E1"+"s", "cubo",
"cui"+u"\u00FA", "cur"+u"\u00E1", "curo", "curre",
"curro", "cuspe", "d"+u"\u00E3"+"o", "d"+u"\u00F3"+"i",
"dois", "dorme", "dormes", "duplo",
u"\u00EA"+"h", "empurra", "empurras", "esconde",
"escondes", "fecha", "fechas", "ferve",
"fiau", "fila", "fim", "foge",
"foges", "fuco", "fula", "fusca",
"fusco", "ganha", "gris", "guaju",
"gue", "guira", "gumba", "hula",
"ido", "jaga", "jile", "jiles",
"jorro", "lambe", "lemba", "lenga",
"lengue", "lero", "liga", "lio",
"lufa", "luze", "luzes", "mal",
"marche", "mari", "mas", "mata",
"mato", "mela", "menga", "mexe",
"mexes", "mi"+u"\u00E1", "mija", "mijas",
"mio", "mocha", "mochas", "mole",
"molhe", "moxa", "muana", "mut"+u"\u00E1",
"ningres", "oba", "og"+u"\u00E1", "paco",
"pala", "panga", "passa", "passe",
"pega", "pegas", "pia", "pica",
"pico", "pilha", "pinga", "pingas",
"pingo", "pio", "pire", "puco",
"pula", "puxa", "puxas", "quebra",
"queima", "quente", "quer"+u"\u00EA", "quero",
"quete", "qui"+u"\u00EA", "quinti", "quiri",
"range", "ranges", "raspa", "reco",
"rema", "remas", "reque", "r"+u"\u00E9"+"u",
"r"+u"\u00F3", "r"+u"\u00F3"+"i", "rola", "rolas",
"ronca", "roque", "rufe", "rufes",
"ruge", "ruges", "safa", "safas",
"sape", "sede", "sel"+u"\u00E1", "serra",
"serras", "sina", "sinas", "siro",
"soque", "sui", "tange", "tanges",
"tape", "teco", "tefe", "tem",
"tepe", "teque", "ter"+u"\u00E9"+"m", "tero",
"terra", "t"+u"\u00E9"+"u", "ti"+u"\u00E3", "tico",
"Tico", "tio", "tique", "toca",
"tole", "toque", "torna", "treme",
"tremes", "troca", "trocas", "truz",
"tuco", "vai", "vais", "vem",
"verga", "vergas", "vim", "vira",
"viras", "xaque", "Xique", "zirra",
"zum"])




# Termos compostos com duas palavras diferentes
tcmpsto = set(["abacaxi-branco", "abacaxi-silvestre", "abacaxis-brancos",
"abacaxis-silvestres", "abaixo-assinado", "abaixo-assinados",
"abaj"+u"\u00E1"+"-mirim", "abelha-africana", "abelha-alem"+u"\u00E3",
"abelha-amarela", "abelha-brava", "abelha-cachorro",
"abelha-campeadora", "abelha-caucasiana", "abelha-comum",
"abelha-crian"+u"\u00E7"+"a", "abelha-dom"+u"\u00E9"+"stica", "abelha-escura",
"abelha-europa", "abelha-europeia", "abelha-flor",
"abelha-ind"+u"\u00ED"+"gena", "abelha-italiana", "abelha-lim"+u"\u00E3"+"o",
"abelha-macho", "abelha-m"+u"\u00E3"+"e", "abelha-mestra",
"abelha-mirim", "abelha-mosquito", "abelha-mosquitos",
"abelha-mulata", "abelha-neutra", "abelha-obreira",
"abelha-oper"+u"\u00E1"+"ria", "abelha-ouro", "abelha-preta",
"abelha-sanhar"+u"\u00F3", "abelha-uru"+u"\u00E7"+"u", "abelhas-africanas",
"abelhas-alem"+u"\u00E3"+"s", "abelhas-amarelas", "abelhas-bravas",
"abelhas-campeadoras", "abelhas-caucasianas", "abelhas-comuns",
"abelhas-dom"+u"\u00E9"+"sticas", "abelhas-escuras", "abelhas-europeias",
"abelhas-ind"+u"\u00ED"+"genas", "abelhas-italianas", "abelhas-machas",
"abelhas-machos", "abelhas-mestras", "abelhas-mirins",
"abelhas-mulatas", "abelhas-neutras", "abelhas-obreiras",
"abelhas-oper"+u"\u00E1"+"rias", "abelhas-pretas", "abelhas-sanhar"+u"\u00F3"+"s",
"abelhas-uru"+u"\u00E7"+"u", "ab"+u"\u00F3"+"bora-caba"+u"\u00E7"+"a", "ab"+u"\u00F3"+"bora-carneira",
"ab"+u"\u00F3"+"bora-chila", "ab"+u"\u00F3"+"bora-gila", "ab"+u"\u00F3"+"bora-menina",
"ab"+u"\u00F3"+"bora-porqueira", "abutre-barbudo", "abutre-cin"+u"\u00E9"+"reo",
"abutre-comum", "abutre-fouveiro", "abutre-fusco",
"abutres-barbudos", "abutres-cin"+u"\u00E9"+"reos", "abutres-comuns",
"abutres-fouveiros", "abutres-fuscos", "ac"+u"\u00E1"+"cia-amarela",
"ac"+u"\u00E1"+"cia-angico", "ac"+u"\u00E1"+"cia-ar"+u"\u00E1"+"bica", "ac"+u"\u00E1"+"cia-argentina",
"ac"+u"\u00E1"+"cia-asi"+u"\u00E1"+"tica", "ac"+u"\u00E1"+"cia-bastarda", "ac"+u"\u00E1"+"cia-branca",
"ac"+u"\u00E1"+"cia-dealbada", "ac"+u"\u00E1"+"cia-falsa", "ac"+u"\u00E1"+"cia-meleira",
"ac"+u"\u00E1"+"cia-mimosa", "ac"+u"\u00E1"+"cia-negra", "ac"+u"\u00E1"+"cia-verdadeira",
"ac"+u"\u00E1"+"cia-vermelha", "ac"+u"\u00E1"+"cias-amarelas", "ac"+u"\u00E1"+"cias-ar"+u"\u00E1"+"bicas",
"ac"+u"\u00E1"+"cias-argentinas", "ac"+u"\u00E1"+"cias-asi"+u"\u00E1"+"ticas", "ac"+u"\u00E1"+"cias-bastardas",
"ac"+u"\u00E1"+"cias-brancas", "ac"+u"\u00E1"+"cias-dealbadas", "ac"+u"\u00E1"+"cias-falsas",
"ac"+u"\u00E1"+"cias-meleiras", "ac"+u"\u00E1"+"cias-mimosas", "ac"+u"\u00E1"+"cias-negras",
"ac"+u"\u00E1"+"cias-verdadeiras", "ac"+u"\u00E1"+"cias-vermelhas", "acad"+u"\u00EA"+"mico-cient"+u"\u00ED"+"fico",
"acad"+u"\u00EA"+"mico-profissional", "a"+u"\u00E7"+"afr"+u"\u00E3"+"o-agreste", "a"+u"\u00E7"+"afr"+u"\u00E3"+"o-bastardo",
"a"+u"\u00E7"+"afr"+u"\u00E3"+"o-bravo", "a"+u"\u00E7"+"afr"+u"\u00E3"+"o-palhinha", "a"+u"\u00E7"+"afr"+u"\u00E3"+"o-vermelho",
"a"+u"\u00E7"+"afroeira-ind"+u"\u00ED"+"gena", "a"+u"\u00E7"+"afroeiras-ind"+u"\u00ED"+"genas", "a"+u"\u00E7"+"afr"+u"\u00F5"+"es-agrestes",
"a"+u"\u00E7"+"afr"+u"\u00F5"+"es-bastardos", "a"+u"\u00E7"+"afr"+u"\u00F5"+"es-bravos", "a"+u"\u00E7"+"afr"+u"\u00F5"+"es-vermelhos",
"a"+u"\u00E7"+"a"+u"\u00ED"+"-branco", "a"+u"\u00E7"+"a"+u"\u00ED"+"-catinga", "a"+u"\u00E7"+"a"+u"\u00ED"+"-chumbo",
"a"+u"\u00E7"+"a"+u"\u00ED"+"-mirim", "a"+u"\u00E7"+"a"+u"\u00ED"+"s-brancos", "a"+u"\u00E7"+"a"+u"\u00ED"+"s-mirins",
"acaju-catinga", "acalenta-menino", "acampamento-base",
"acanto-bastardo", "acapurana-vermelha", "acapuranas-vermelhas",
"a"+u"\u00E7"+"o-carbono", "a"+u"\u00E7"+"o-liga", "a"+u"\u00E7"+"oita-cavalo",
"acre-doce", "a"+u"\u00E7"+u"\u00FA"+"car-pedra", "adiposo-genital",
"adiposos-genitais", "administrador-delegado", "administrador-deposit"+u"\u00E1"+"rio",
"administrador-geral", "administrador-judicial", "administradora-delegada",
"administradora-judicial", "administradores-gerais", "administrativo-financeiro",
"Administrativo-Financeiro", "Advocacia-Geral", "advogada-geral",
"Advogada-Geral", "advogado-geral", "Advogado-Geral",
"afeg"+u"\u00E3"+"o-americano", "afonso-arinense", "afonso-bezerrense",
"afonso-claudiense", "afonso-cunhense", "afro-americana",
"afro-americanas", "afro-americano", "afro-americanos",
"afro-amer"+u"\u00ED"+"ndio", "afro-"+u"\u00E1"+"rabe", "afro-asi"+u"\u00E1"+"tico",
"afro-baiana", "afro-baianas", "afro-baiano", "afro-baianos", "afro-brasileirismo", "afro-brasileir"+u"\u00ED"+"stico",
"afro-brasileira", "afro-brasileiras", "afro-brasileiro", "Afro-Brasileiro", "afro-brasileiros",
"afro-brit"+u"\u00E2"+"nica", "afro-brit"+u"\u00E2"+"nicas", "afro-brit"+u"\u00E2"+"nico",
"afro-brit"+u"\u00E2"+"nicos", "afro-caribenho", "afro-cubano",
"afro-franc"+u"\u00EA"+"s", "afro-ind"+u"\u00ED"+"gena", "afro-ingl"+u"\u00EA"+"s",
"afro-lusitanismo", "afro-lusitan"+u"\u00ED"+"stico", "afro-lusitana",
 "afro-lusitanas", "afro-lusitano", "afro-lustanos",
"afro-negrismo", "afro-negr"+u"\u00ED"+"stico", "afro-negro",
"agente-motorista", "agentes-motoristas", "agri"+u"\u00E3"+"o-bravo",
"agri"+u"\u00E3"+"o-falso", "agri"+u"\u00E3"+"o-menor", u"\u00E1"+"grico-ind"+u"\u00FA"+"stria",
u"\u00E1"+"grico-industrial", u"\u00E1"+"grico-pecus"+u"\u00E1"+"rio", "agrim"+u"\u00F4"+"nia-bastarda",
"agrim"+u"\u00F4"+"nias-bastardas", "agri"+u"\u00F5"+"es-bravos", "agri"+u"\u00F5"+"es-falsos",
"agri"+u"\u00F5"+"es-menores", "agro-doce", "agro-hidrologia",
"agro-h"+u"\u00F3"+"rtico", u"\u00E1"+"gua-viva", u"\u00E1"+"guas-vivas",
u"\u00E1"+"guia-belicosa", u"\u00E1"+"guia-branquense", u"\u00E1"+"guia-brasileira",
u"\u00E1"+"guia-calva", u"\u00E1"+"guia-chilena", u"\u00E1"+"guia-cinzenta",
u"\u00E1"+"guia-coroada", u"\u00E1"+"guia-destruidora", u"\u00E1"+"guia-equatorial",
u"\u00E1"+"guia-gritadeira", u"\u00E1"+"guia-pescadora", u"\u00E1"+"guia-pesqueira",
u"\u00E1"+"guia-rabalva", u"\u00E1"+"guia-real", u"\u00E1"+"guias-belicosas",
u"\u00E1"+"guias-brasileiras", u"\u00E1"+"guias-calvas", u"\u00E1"+"guias-chilenas",
u"\u00E1"+"guias-cinzentas", u"\u00E1"+"guias-coroadas", u"\u00E1"+"guias-destruidoras",
u"\u00E1"+"guias-equatoriais", u"\u00E1"+"guias-gritadeiras", u"\u00E1"+"guias-pescadoras",
u"\u00E1"+"guias-pesqueiras", u"\u00E1"+"guias-rabalvas", u"\u00E1"+"guias-reais",
"agulha-branca", "agulha-crioula", "agulha-ferrugenta",
"agulha-m"+u"\u00E3"+"e", "agulha-padr"+u"\u00E3"+"o", "agulha-preta",
"agulh"+u"\u00E3"+"o-azul", "agulh"+u"\u00E3"+"o-bandeira", "agulh"+u"\u00E3"+"o-branco",
"agulh"+u"\u00E3"+"o-lambaio", "agulh"+u"\u00E3"+"o-negro", "agulh"+u"\u00E3"+"o-roli"+u"\u00E7"+"o",
"agulh"+u"\u00E3"+"o-trombeta", "agulhas-brancas", "agulhas-crioulas",
"agulhas-ferrugentas", "agulhas-pretas", "agulheira-menor",
"agulheira-moscada", "agulheiras-menores", "agulheiras-moscadas",
"agulh"+u"\u00F5"+"es-azuis", "agulh"+u"\u00F5"+"es-brancos", "agulh"+u"\u00F5"+"es-lambaios",
"agulh"+u"\u00F5"+"es-negros", "agulh"+u"\u00F5"+"es-roli"+u"\u00E7"+"os", "aipo-bravo",
"aipo-chimarr"+u"\u00E3"+"o", "aipo-falso", "aipo-hortense",
"aipo-r"+u"\u00E1"+"bano", "aipo-sals"+u"\u00E3"+"o", "aipos-bravos",
"aipos-chimarr"+u"\u00F5"+"es", "aipos-falsos", "aipos-hortenses",
"aipos-sals"+u"\u00F5"+"es", "air-bag", "ajudante-general",
"ajunta-pedra", "Al-Man"+u"\u00E1"+"mah", "Al-Qaeda",
"ala-armador", "ala-piv"+u"\u00F4", "alabastro-calc"+u"\u00E1"+"rio",
"alabastro-oriental", "alabastros-calc"+u"\u00E1"+"rios", "alabastros-orientais",
"alamanda-amarela", "alamanda-cheirosa", "alamandas-amarelas",
"alamandas-cheirosas", u"\u00E1"+"lamo-branco", u"\u00E1"+"lamo-negro",
u"\u00E1"+"lamo-preto", u"\u00E1"+"lamos-brancos", u"\u00E1"+"lamos-negros",
u"\u00E1"+"lamos-pretos", "alas-armadoras", "alas-armadores",
"albacora-azul", "albacora-bandolim", "albacora-branca",
"albacora-cachorra", "albacora-lajeira", "albacora-verdadeira",
"albacoras-azuis", "albacoras-brancas", "albacoras-cachorras",
"albacoras-lajeiras", "albacoras-verdadeiras", "albatroz-errante",
"albatroz-gigante", "albatroz-real", "albatroz-viageiro",
"albatrozes-errantes", "albatrozes-gigantes", "albatrozes-reais",
"albatrozes-viageiros", "al"+u"\u00E7"+"a-perna", "alcachofra-brava",
"alcachofra-hortense", "alcachofras-bravas", "alcachofras-hortenses",
"alca"+u"\u00E7"+"uz-bravo", "alca"+u"\u00E7"+"uz-falso", "alca"+u"\u00E7"+"uz-indiano",
"alca"+u"\u00E7"+"uz-silvestre", "alca"+u"\u00E7"+"uzes-bravos", "alca"+u"\u00E7"+"uzes-falsos",
"alca"+u"\u00E7"+"uzes-indianos", "alca"+u"\u00E7"+"uzes-silvestres", "alcaidaria-mor",
"alcaidarias-mores", "alcaide-menor", "alcaide-mor",
"alcaide-pequeno", "alcaides-menores", "alcaides-mores",
"alcaides-pequenos", "alcalino-terroso", "alcaparreira-cheirosa",
"alcaparreiras-cheirosas", "alecrim-bravo", "alecrins-bravos",
"alegra-campo", "al"+u"\u00E9"+"m-Atl"+u"\u00E2"+"ntico", "al"+u"\u00E9"+"m-Brasil",
"al"+u"\u00E9"+"m-fronteira", "al"+u"\u00E9"+"m-mar", "al"+u"\u00E9"+"m-mundo",
"al"+u"\u00E9"+"m-Piren"+u"\u00E9"+"us", "al"+u"\u00E9"+"m-Reno", "al"+u"\u00E9"+"m-t"+u"\u00FA"+"mulo",
"alem"+u"\u00E3"+"es-ocidentais", "alem"+u"\u00E3"+"es-orientais", "alem"+u"\u00E3"+"o-ocidental",
"alem"+u"\u00E3"+"o-oriental", "alfa-amilase", "alfa-h"+u"\u00E9"+"lice",
"alfa-sinucle"+u"\u00ED"+"na", "alfa-Sinucle"+u"\u00ED"+"na", "alface-americana",
"alface-brava", "alface-crespa", "alface-frisada",
"alface-paulista", "alface-repolhuda", "alface-romana",
"alfaces-americanas", "alfaces-bravas", "alfaces-crespas",
"alfaces-frisadas", "alfaces-paulistas", "alfaces-repolhudas",
"alfaces-romanas", "alfafa-arb"+u"\u00F3"+"rea", "alfafa-brasileira",
"alfafa-espinhosa", "alfafa-gigante", "alfafa-lupulina",
"alfafa-mineira", "alfafa-nacional", "alfafa-r"+u"\u00FA"+"stica",
"alfafa-verdadeira", "alfafas-arb"+u"\u00F3"+"reas", "alfafas-brasileiras",
"alfafas-espinhosas", "alfafas-gigantes", "alfafas-lupulinas",
"alfafas-mineiras", "alfafas-nacionais", "alfafas-r"+u"\u00FA"+"sticas",
"alfafas-verdadeiras", "alfavaca-carpunha", "alfavaca-cheirosa",
"alfavaca-silvestre", "alfavacas-carpunhas", "alfavacas-cheirosas",
"alfavacas-silvestres", "alfazema-brava", "alfazemas-bravas",
"alforva-brava", "alforvas-bravas", "alga-perlada",
"algarobeira-preta", "algarobeiras-pretas", "algas-perladas",
"algod"+u"\u00E3"+"o-bravo", "algod"+u"\u00E3"+"o-col"+u"\u00F3"+"dio", "algod"+u"\u00E3"+"o-cravo",
"algod"+u"\u00E3"+"o-doce", "algod"+u"\u00E3"+"o-macaco", "algod"+u"\u00E3"+"o-p"+u"\u00F3"+"lvora",
"algodoeiro-americano", "algodoeiro-bravo", "algodoeiro-crioulo",
"algodoeiro-gigante", "algodoeiros-americanos", "algodoeiros-bravos",
"algodoeiros-crioulos", "algodoeiros-gigantes", "algod"+u"\u00F5"+"es-bravos",
"algod"+u"\u00F5"+"es-doces", "alho-espanhol", "alho-franc"+u"\u00EA"+"s",
"alho-grosso", "alho-"+u"\u00ED"+"ngreme", "alho-macho",
"alho-m"+u"\u00E1"+"gico", "alho-mourisco", "alho-ordin"+u"\u00E1"+"rio",
"alho-por"+u"\u00F3", "alho-porro", "alho-porr"+u"\u00F3",
"alho-porr"+u"\u00F4", "alho-rocambole", "alho-silvestre",
"alhos-espanh"+u"\u00F3"+"is", "alhos-franceses", "alhos-grossos",
"alhos-"+u"\u00ED"+"ngremes", "alhos-machos", "alhos-m"+u"\u00E1"+"gicos",
"alhos-mouriscos", "alhos-ordin"+u"\u00E1"+"rios", "alhos-por"+u"\u00F3"+"s",
"alhos-porros", "alhos-porr"+u"\u00F3"+"s", "alhos-porr"+u"\u00F4"+"s",
"alhos-silvestres", "aliba-ca"+u"\u00E7"+u"\u00F4", "alisso-cheiroso",
"alissos-cheirosos", "alma-aflita", "alma-danada",
"alma-perdida", "almas-aflitas", "almas-danadas",
"almas-perdidas", "alm"+u"\u00E9"+"cega-verdadeira", "alm"+u"\u00E9"+"cegas-verdadeiras",
"almecegueira-cheirosa", "almecegueira-mansa", "almecegueira-verdadeira",
"almecegueira-vermelha", "almecegueiras-cheirosas", "almecegueiras-mansas",
"almecegueiras-verdadeiras", "almecegueiras-vermelhas", "almecegueiro-bravo",
"almecegueiros-bravos", "almino-afonsense", "almirante-mor",
"almirante-vermelho", "almirantes-mores", "almirantes-vermelhos",
"almo"+u"\u00E7"+"o-ajantarado", "almo"+u"\u00E7"+"o-ceia", "almo"+u"\u00E7"+"o-debate",
"almo"+u"\u00E7"+"os-ajantarados", "almotac"+u"\u00E9"+"-mor", "almotac"+u"\u00E9"+"s-mores",
"alpiste-mi"+u"\u00FA"+"do", "alpistes-mi"+u"\u00FA"+"dos", "alqueire-mineiro",
"alqueire-paulista", "alqueires-mineiros", "alqueires-paulistas",
"alquequenje-amarelo", "alquequenje-bastardo", "alquequenjes-amarelos",
"alquequenjes-bastardos", "als"+u"\u00E1"+"cio-loreno", "alta-costura",
"alta-fidelidade", "alta-florense", "alta-luz",
"alta-roda", "alta-tens"+u"\u00E3"+"o", "altar-mor",
"altares-mores", "altas-costuras", "altas-fidelidades",
"altas-luzes", "altas-rodas", "altas-tens"+u"\u00F5"+"es",
"alto-alecrinense", "alto-alegrense", "alto-alem"+u"\u00E3"+"o",
"alto-alentejano", "alto-araguaiense", "alto-astral",
"alto-brilho", "alto-comando", "alto-comissariado",
"alto-comissariato", "alto-comiss"+u"\u00E1"+"rio", "alto-contraste",
"alto-c"+u"\u00FA"+"mulo", "alto-duriense", "alto-estrato",
"alto-explosivo", "alto-falante", "alto-forno",
"alto-fuste", "alto-garcense", "alto-grandense",
"alto-hor"+u"\u00E1"+"rio", "alto-itaunense", "alto-lajeadense",
"alto-longaense", "alto-madeirense", "alto-mar",
"alto-maranhense", "alto-minhoto", "alto-navarro",
"alto-paraguaiense", "alto-paraguaio", "alto-paraisense",
"alto-paranaense", "alto-paranaibense", "alto-parnaibano",
"alto-piquirinense", "alto-relevo", "alto-rodriguense",
"alto-santense", "alto-sax"+u"\u00E3"+"o", "alto-serrano",
"alto-v"+u"\u00E1"+"cuo", "alto-voltaico", "alto-voltense",
"altos-alem"+u"\u00E3"+"es", "altos-astrais", "altos-comandos",
"altos-comissariados", "altos-comiss"+u"\u00E1"+"rios", "altos-contrastes",
"altos-c"+u"\u00FA"+"mulos", "altos-estratos", "altos-explosivos",
"altos-fornos", "altos-fundos", "altos-fustes",
"altos-hor"+u"\u00E1"+"rios", "altos-lissos", "altos-mares",
"altos-relevos", "altos-v"+u"\u00E1"+"cuos", "aluno-aprendiz",
"aluno-mestre", "aluno-professor", "alunos-mestres",
"alv"+u"\u00E9"+"loa-amarela", "alv"+u"\u00E9"+"ola-branca", "alv"+u"\u00E9"+"olas-brancas",
"alv"+u"\u00E9"+"olo-condiliano", "alv"+u"\u00E9"+"olo-dental", "alv"+u"\u00E9"+"olo-dent"+u"\u00E1"+"rio",
"alv"+u"\u00E9"+"olo-labial", "alv"+u"\u00E9"+"olo-nasal", "alv"+u"\u00E9"+"olo-palatal",
"alv"+u"\u00E9"+"olo-subnasal", "ama-seca", "amansa-besta",
"amansa-senhor", "amaranto-verde", "amarantos-verdes",
"amarelo-can"+u"\u00E1"+"rio", "amarelo-cidr"+u"\u00E3"+"o", "amarelo-cinzento",
"amarelo-claro", "amarelo-cobalto", "amarelo-enxofre",
"amarelo-escuro", "amarelo-esverdeado", "amarelo-fosco",
"amarelo-gualdo", "amarelo-laranja", "amarelo-lim"+u"\u00E3"+"o",
"amarelo-oca", "amarelo-ocre", "amarelo-oiro",
"amarelo-ouro", "amarelo-palha", "amarelo-top"+u"\u00E1"+"zio",
"amarelo-torrado", "amarelos-cinzentos", "amarelos-claros",
"amarelos-escuros", "amarelos-esverdeados", "amarelos-foscos",
"amarelos-torrados", "amaro-leitense", "amarra-pinto",
"amas-secas", "amassa-barro", u"\u00E2"+"mbar-amarelo",
u"\u00E2"+"mbar-cinzento", u"\u00E2"+"mbares-amarelos", u"\u00E2"+"mbares-cinzentos",
"amba"+u"\u00FA"+"ba-mansa", "amba"+u"\u00FA"+"bas-mansas", "amba"+u"\u00FA"+"va-mansa",
"amba"+u"\u00FA"+"vas-mansas", "ambr"+u"\u00F3"+"sia-americana", "ambr"+u"\u00F3"+"sias-americanas",
"am"+u"\u00EA"+"ijoa-branca", "am"+u"\u00EA"+"ijoa-redonda", "am"+u"\u00EA"+"ijoas-brancas",
"am"+u"\u00EA"+"ijoas-redondas", "ameixa-amarela", "ameixa-americana",
"ameixa-japonesa", "ameixa-preta", "ameixas-amarelas",
"ameixas-americanas", "ameixas-japonesas", "ameixas-pretas",
"ameixeira-brava", "ameixeiras-bravas", "am"+u"\u00EA"+"ndoa-brava",
"am"+u"\u00EA"+"ndoa-dur"+u"\u00E1"+"zia", "am"+u"\u00EA"+"ndoa-molar", "am"+u"\u00EA"+"ndoas-bravas",
"am"+u"\u00EA"+"ndoas-dur"+u"\u00E1"+"zias", "am"+u"\u00EA"+"ndoas-molares", "amendoeira-coca",
"amendoeira-dur"+u"\u00E1"+"zia", "amendoeira-molar", "amendoeiras-dur"+u"\u00E1"+"zias",
"amendoeiras-molares", "amendoim-bravo", "amendoim-falso",
"amendoim-rajado", "amendoim-rasteiro", "amendoim-roxo",
"amendoins-bravos", "amendoins-falsos", "amendoins-rajados",
"amendoins-rasteiros", "amendoins-roxos", "americana-brasileira",
"americana-canadense", "americana-venezuelana", "americanas-canadenses",
"americano-brasileiro", "americano-canadense", "americano-sovi"+u"\u00E9"+"tico",
"americano-venezuelano", "americanos-canadenses", "ametista-oriental",
"ametistas-orientais", "amido-mosqueado", "amido-negro",
"amido-preto", "amidos-mosqueados", "amidos-negros",
"amidos-pretos", "amieiro-negro", "amieiros-negros",
"amigo-oculto", "amigo-secreto", "amigo-urso",
"amigos-ocultos", "amigos-secretos", "amigos-ursos",
u"\u00E2"+"mio-maior", u"\u00E2"+"mio-vulgar", "amizade-colorida",
"amizades-coloridas", "amor-agarradinho", "amor-agarrado",
"amor-crescido", "amor-entrela"+u"\u00E7"+"ado", "amor-perfeito",
"amor-pr"+u"\u00F3"+"prio", "amor-seco", "amora-branca",
"amora-brava", "amora-preta", "amora-verde",
"amora-vermelha", "amoras-brancas", "amoras-bravas",
"amoras-pretas", "amoras-verdes", "amoras-vermelhas",
"amor"+u"\u00E9"+"-preto", "amoreira-branca", "amoreira-cip"+u"\u00F3",
"amoreira-negra", "amoreira-preta", "amoreira-tata"+u"\u00ED"+"ba",
"amoreiras-brancas", "amoreiras-negras", "amoreiras-pretas",
"amores-agarradinhos", "amores-agarrados", "amores-crescidos",
"amores-entrela"+u"\u00E7"+"ados", "amores-perfeitos", "amores-pr"+u"\u00F3"+"prios",
"amores-secos", "amor"+u"\u00E9"+"s-pretos", "amostra-tipo",
"amovo-inamov"+u"\u00ED"+"vel", "ampere-espira", "ampere-giro",
"ampere-hora", "ampere-segundo", "ampere-volta",
"amp"+u"\u00E9"+"rio-espira", "amp"+u"\u00E9"+"rio-giro", "amp"+u"\u00E9"+"rio-hora",
"amp"+u"\u00E9"+"rio-segundo", "amp"+u"\u00E9"+"rio-volta", "anac"+u"\u00E3"+"-mirim",
"anac"+u"\u00E3"+"s-mirins", "anad"+u"\u00E9"+"is-mores", "anadel-mor",
"anafa-cheirosa", "anafa-maior", "anafas-cheirosas",
"anafas-maiores", "anafe-menor", "anafe-ordin"+u"\u00E1"+"rio",
"anafes-menores", "anafes-ordin"+u"\u00E1"+"rios", "anaj"+u"\u00E1"+"-brava",
"anaj"+u"\u00E1"+"-mirim", "anaj"+u"\u00E1"+"s-bravas", "anaj"+u"\u00E1"+"s-mirins",
"analista-chefe", "analista-programador", "analistas-chefes",
"anal"+u"\u00ED"+"tico-expositivo", "anamb"+u"\u00E9"+"-azul", "anamb"+u"\u00E9"+"-branco",
"anamb"+u"\u00E9"+"-grande", "anamb"+u"\u00E9"+"-militar", "anamb"+u"\u00E9"+"-piti"+u"\u00FA",
"anamb"+u"\u00E9"+"-pombo", "anamb"+u"\u00E9"+"-preto", "anamb"+u"\u00E9"+"-roxo",
"anamb"+u"\u00E9"+"-sol", "anamb"+u"\u00E9"+"-vermelho", "anamb"+u"\u00E9"+"s-azuis",
"anamb"+u"\u00E9"+"s-brancos", "anamb"+u"\u00E9"+"s-grandes", "anamb"+u"\u00E9"+"s-militares",
"anamb"+u"\u00E9"+"s-piti"+u"\u00FA", "anamb"+u"\u00E9"+"s-pretos", "anamb"+u"\u00E9"+"s-roxos",
"anamb"+u"\u00E9"+"s-vermelhos", "ananases-selvagens", "ananases-silvestres",
"anarco-comunismo", "anarco-comunista", "anarco-sindicalismo",
"anarco-sindicalista", "anchova-preta", "anchovas-pretas",
"andiroba-branca", "andiroba-jareu"+u"\u00E1", "andiroba-suruba",
"andiroba-vermelha", "andirobas-brancas", "andirobas-surubas",
"andirobas-vermelhas", "ando-boliviano", "ando-chileno",
"ando-colombiano", "ando-equatoriano", "ando-peruano",
"andor-imp"+u"\u00E9"+"rio", "andorinha-azul", "andorinha-cavadeira",
"andorinha-coleira", "andorinha-grande", "andorinhas-azuis",
"andorinhas-cavadeiras", "andorinhas-coleiras", "ang"+u"\u00E9"+"lica-branca",
"ang"+u"\u00E9"+"lica-mansa", "ang"+u"\u00E9"+"lica-silvestre", "ang"+u"\u00E9"+"licas-brancas",
"ang"+u"\u00E9"+"licas-mansas", "angelim-amarelo", "angelim-amargo",
"angelim-amargoso", "angelim-araroba", "angelim-coco",
"angelim-doce", "angelim-pedra", "angelim-pintado",
"angelim-preto", "angelim-rajado", "angelim-rosa",
"angelins-amarelos", "angelins-amargos", "angelins-amargosos",
"angelins-doces", "angelins-pintados", "angelins-pretos",
"angelins-rajados", "angico-amarelo", "angico-barbatim"+u"\u00E3"+"o",
"angico-branco", "angico-cangalha", "angico-cedro",
"angico-preto", "angico-rajado", "angico-rosa",
"angico-roxo", "angico-surucucu", "angico-verdadeiro",
"angico-vermelho", "angicos-amarelos", "angicos-brancos",
"angicos-pretos", "angicos-rajados", "angicos-roxos",
"angicos-surucucu", "angicos-verdadeiros", "angicos-vermelhos",
"anglo-africano", "anglo-alem"+u"\u00E3"+"o", "anglo-americano",
"anglo-"+u"\u00E1"+"rabe", "anglo-asi"+u"\u00E1"+"tico", "anglo-australiano",
"anglo-brasileiro", "anglo-bret"+u"\u00E3"+"o", "anglo-canadense",
"anglo-canadiano", "anglo-catolicismo", "anglo-cat"+u"\u00F3"+"lico",
"anglo-franc"+u"\u00EA"+"s", "anglo-galo", "anglo-gaul"+u"\u00EA"+"s",
"anglo-germ"+u"\u00E2"+"nico", "anglo-holand"+u"\u00EA"+"s", "anglo-indiano",
"anglo-irland"+u"\u00EA"+"s", "anglo-israelismo", "anglo-italiano",
"anglo-luso", "anglo-normando", "anglo-nubiano",
"anglo-portugu"+u"\u00EA"+"s", "anglo-russo", "anglo-sax"+u"\u00E3",
"anglo-sax"+u"\u00E3"+"o", "anglo-sax"+u"\u00F4"+"nico", "anglo-sax"+u"\u00F4"+"nio",
"anglo-saxonismo", "angona-pu"+u"\u00ED"+"ta", "anil-bastardo",
"anil-bravo", "anil-mi"+u"\u00FA"+"do", "anil-trepador",
"anileira-verdadeira", "anileiras-verdadeiras", "anima-membeca",
"anis-bastardos", "anis-bravos", "anis-mi"+u"\u00FA"+"dos",
"anis-trepadores", "anises-doces", "anises-estrelados",
"anises-verdes", "aniso-hiperleucocitose", "aniso-hipocitose",
"aniso-hipoleucocitose", "anjo-bento", "anjo-mau",
"anjo-papudo", "anjo-viola", "anjos-bentos",
"anjos-maus", "anjos-papudos", "ano-base",
"ano-bom", "ano-calend"+u"\u00E1"+"rio", "ano-exerc"+u"\u00ED"+"cio",
"ano-luz", "Ano-novo", "Ano-Novo",
"ano-opsia", "ano-"+u"\u00F3"+"ptico", "anos-bons",
"anos-exerc"+u"\u00ED"+"cios", "anos-luz", "anos-novos",
"Anos-Novos", "ansarinha-branca", "ansarinha-malhada",
"ansarinha-verm"+u"\u00ED"+"fuga", "ansarinhas-brancas", "ansarinhas-malhadas",
"ansarinhas-verm"+u"\u00ED"+"fugas", "anta-gameleira", "anta-sapateira",
"antas-gameleiras", "antas-sapateiras", "ante-ep"+u"\u00ED"+"grafe",
"ante-h"+u"\u00E9"+"lice", "ante-hip"+u"\u00F3"+"fise", "ante-hist"+u"\u00F3"+"rico",
u"\u00E2"+"ntero-dorsal", u"\u00E2"+"ntero-inferior", u"\u00E2"+"ntero-lateral",
u"\u00E2"+"ntero-posterior", u"\u00E2"+"ntero-superior", "anti-Bush",
"anti-China", "anti-halo", "anti-h"+u"\u00E9"+"lio",
"anti-HIV", "anti-inato", "anti-"+u"\u00ED"+"neo",
"anti-Marx", "anti-Rho", "anti-Saddam",
"anti-spam", "ant"+u"\u00ED"+"lope-real", "ant"+u"\u00ED"+"lopes-reais",
"anti"+u"\u00F4"+"mega-mais", "ant"+u"\u00F4"+"nio-almeidense", "ant"+u"\u00F4"+"nio-carlense",
"ant"+u"\u00F4"+"nio-diense", "ant"+u"\u00F4"+"nio-joanense", "ant"+u"\u00F4"+"nio-martinense",
"antro-nasal", "antropo-histografia", "antropo-histogr"+u"\u00E1"+"fico",
"anu-branco", "anu-chorado", "anu-coroca",
"anu-coroia", "anu-corrido", "anu-dourado",
"anu-galego", "anu-grande", "anu-guaicuru",
"anu-peixe", "anu-pequeno", "anu-preto",
"anu-velho", "anum-branco", "anum-chorado",
"anum-coroca", "anum-coroia", "anum-corrido",
"anum-dourado", "anum-galego", "anum-grande",
"anum-guaicuru", "anum-peixe", "anum-pequeno",
"anum-preto", "anum-velho", "anuns-brancos",
"anuns-chorados", "anuns-corocas", "anuns-coroias",
"anuns-corridos", "anuns-dourados", "anuns-galegos",
"anuns-grandes", "anuns-guaicurus", "anuns-peixes",
"anuns-pequenos", "anuns-pretos", "anuns-velhos",
"anus-brancos", "anus-chorados", "anus-corocas",
"anus-coroias", "anus-corridos", "anus-dourados",
"anus-galegos", "anus-grandes", "anus-guaicurus",
"anus-peixes", "anus-pequenos", "anus-pretos",
"anus-velhos", "apa-bico", "apacanim-branco",
"apacanins-brancos", "apaga-fan"+u"\u00F3"+"is", "apaga-g"+u"\u00E1"+"s",
"apaga-gases", "apaga-penol", "apaga-tristeza",
"apanha-bolas", "apanha-migalhas", "apanha-moscas",
"apanha-saia", "apap"+u"\u00E1"+"-verdadeiro", "apap"+u"\u00E1"+"s-verdadeiros",
"apara-l"+u"\u00E1"+"pis", "apara-livros", "apara-mangaba",
"apart-hotel", "aperta-chico", "aperta-galha",
"aperta-goela", "aperta-livros", "aperta-luvas",
"aperta-nervos", "aperta-pap"+u"\u00E9"+"is", "aperta-ru"+u"\u00E3"+"o",
"Apicum-A"+u"\u00E7"+"u", "apoia-bra"+u"\u00E7"+"os", "apoia-m"+u"\u00E3"+"os",
"apoia-nuca", "aposentador-mor", "aposentadores-mores",
"apost"+u"\u00F3"+"lico-crist"+u"\u00E3"+"o", "aprendiz-marinheiro", "aprendizes-marinheiros",
"aqu"+u"\u00E9"+"m-fronteiras", "aqu"+u"\u00E9"+"m-mar", "aqu"+u"\u00E9"+"m-oceano",
"aqu"+u"\u00E9"+"m-pireneus", "ar-condicionado", "ar-condicionados",
"ar-refrigerado", u"\u00E1"+"rabe-brasileiro", u"\u00E1"+"rabe-Brasileiro",
u"\u00E1"+"rabe-israelense", u"\u00E1"+"rabe-saudita", "ar"+u"\u00E1"+"bico-brit"+u"\u00E2"+"nico",
"ar"+u"\u00E1"+"bico-espanhol", "ar"+u"\u00E1"+"bico-portugu"+u"\u00EA"+"s", "ar"+u"\u00E1"+"bigo-brit"+u"\u00E2"+"nico",
"ar"+u"\u00E1"+"bigo-espanhol", "ar"+u"\u00E1"+"bigo-portugu"+u"\u00EA"+"s", "ara"+u"\u00E7"+u"\u00E1"+"-branco",
"ara"+u"\u00E7"+u"\u00E1"+"-bravo", "ara"+u"\u00E7"+u"\u00E1"+"-cag"+u"\u00E3"+"o", "ara"+u"\u00E7"+u"\u00E1"+"-cinzento",
"ara"+u"\u00E7"+u"\u00E1"+"-congonha", "ara"+u"\u00E7"+u"\u00E1"+"-cot"+u"\u00E3"+"o", "ara"+u"\u00E7"+u"\u00E1"+"-felpudo",
"ara"+u"\u00E7"+u"\u00E1"+"-lima", "ara"+u"\u00E7"+u"\u00E1"+"-mirim", "ara"+u"\u00E7"+u"\u00E1"+"-pedra",
"ara"+u"\u00E7"+u"\u00E1"+"-rosa", "ara"+u"\u00E7"+u"\u00E1"+"-roxo", "ara"+u"\u00E7"+u"\u00E1"+"-verde",
"ara"+u"\u00E7"+u"\u00E1"+"-vermelho", "Aracaju-Barra", "ara"+u"\u00E7"+"arana-mirim",
"ara"+u"\u00E7"+"aranas-mirins", "aranha-bode", "aranha-caranguejeira",
"aranha-caranguejo", "arara-amarela", "arara-azul",
"arara-canind"+u"\u00E9", "arara-caro", "arara-celeste",
"arara-chauanau"+u"\u00E1", "arara-cip"+u"\u00F3", "arara-hiacinta",
"arara-jacinto", "arara-macau", "arara-pequena",
"arara-piti"+u"\u00FA", "arara-preta", "arara-verde",
"arara-vermelha", "araras-amarelas", "araras-azuis",
"araras-caros", "araras-celestes", "araras-cip"+u"\u00F3"+"s",
"araras-hiacintas", "araras-jacintos", "araras-macau",
"araras-pequenas", "araras-piti"+u"\u00FA", "araras-pretas",
"araras-verdes", "araras-vermelhas", "araticum-abareno",
"araticum-alvadio", "araticum-ap"+u"\u00EA", "araticum-bravo",
"araticum-caca", "araticum-cag"+u"\u00E3"+"o", "araticum-catinga",
"araticum-corti"+u"\u00E7"+"a", "araticum-fedorento", "araticum-grande",
"araticum-mirim", "araticum-peludo", "araticum-pitai"+u"\u00E1",
"araticum-ponh"+u"\u00EA", "araticuns-abarenos", "araticuns-alvadios",
"araticuns-bravos", "araticuns-cag"+u"\u00F5"+"es", "araticuns-fedorentos",
"araticuns-grandes", "araticuns-mirins", "araticuns-peludos",
"araticuns-ponh"+u"\u00EA"+"s", "aratu-marinheiro", "aratu-vermelho",
"aratus-marinheiros", "aratus-vermelhos", "arbusto-milagroso",
"arbustos-milagrosos", "arco-celeste", "arco-"+u"\u00ED"+"ris",
"Arco-"+u"\u00ED"+"ris", "arco-irisado", "arco-irisar",
"arco-irisense", "arco-verdense", "arcos-celestes",
"arcos-"+u"\u00ED"+"ris", "arcos-irisados", u"\u00E1"+"rea-alvo",
u"\u00E1"+"rea-chave", u"\u00E1"+"rea-cliente", u"\u00E1"+"rea-fim",
u"\u00E1"+"reas-fins", "areia-branquense", "areia-engolideira",
"areia-gulosa", "areia-manteiga", "areias-engolideiras",
"areias-gordas", "areias-gulosas", "areno-argiloso",
"areno-humoso", "areno-siltoso", "ares-condicionados",
"ares-refrigerados", "argentino-brasileiro", "argento-vivo",
"argentos-vivos", "argilo-arenoso", "armado-comum",
"armador-gerente", "armador-locat"+u"\u00E1"+"rio", "armador-propriet"+u"\u00E1"+"rio",
"armadores-gerentes", "armadores-locat"+u"\u00E1"+"rios", "armadores-propriet"+u"\u00E1"+"rios",
"armados-comuns", "armaz"+u"\u00E9"+"m-geral", "armaz"+u"\u00E9"+"ns-gerais",
"armeiro-mor", "armeiros-mores", "arquip"+u"\u00E9"+"lago-estado",
"arquivo-fonte", "Arquivo-Fonte", "arquivo-texto",
"arraia-acau"+u"\u00E3", "arraia-bor"+u"\u00F3", "arraia-branca",
"arraia-listrada", "arraia-mijona", "arraia-mi"+u"\u00FA"+"da",
"arraia-papagaio", "arraia-preta", "arraia-santo",
"arraia-tremedeira", "arraias-brancas", "arraias-listradas",
"arraias-mijonas", "arraias-mi"+u"\u00FA"+"das", "arraias-pintadinhas",
"arraias-pretas", "arraias-tremedeiras", "arranha-c"+u"\u00E9"+"u",
"arroio-grandense", "arroio-meense", "arromba-peito",
"arroz-agulha", "arroz-amarel"+u"\u00E3"+"o", "arroz-bravo",
"arroz-cateto", "arroz-crioulo", "arroz-doce",
"arroz-pequeno", "arroz-silvestre", "arrozes-amarel"+u"\u00F5"+"es",
"arrozes-bravos", "arrozes-crioulos", "arrozes-doces",
"arrozes-pequenos", "arrozes-silvestres", "arruda-amarela",
"arruda-brava", "arruda-comum", "arruda-fedorenta",
"arruda-pariet"+u"\u00E1"+"ria", "arruda-preta", "arruda-rajada",
"arruda-silvestre", "arruda-vermelha", "arrudas-amarelas",
"arrudas-bravas", "arrudas-comuns", "arrudas-fedorentas",
"arrudas-pretas", "arrudas-rajadas", "arrudas-silvestres",
"arrudas-vermelhas", "arte-finalizar", "artem"+u"\u00ED"+"sia-brava",
"artem"+u"\u00ED"+"sia-verdadeira", "artem"+u"\u00ED"+"sias-bravas", "artem"+u"\u00ED"+"sias-verdadeiras",
"art"+u"\u00ED"+"stico-cultural", "art"+u"\u00ED"+"stico-liter"+u"\u00E1"+"rio", u"\u00E1"+"rvore-branca",
u"\u00E1"+"rvore-copal", u"\u00E1"+"rvore-m"+u"\u00E3"+"e", u"\u00E1"+"rvore-santa",
u"\u00E1"+"rvore-triste", u"\u00E1"+"rvores-brancas", u"\u00E1"+"rvores-copais",
u"\u00E1"+"rvores-santas", u"\u00E1"+"rvores-tristes", "asfalto-vanad"+u"\u00ED"+"fero",
"asfaltos-vanad"+u"\u00ED"+"feros", "asilo-col"+u"\u00F4"+"nia", "asilo-escola",
"asilos-col"+u"\u00F4"+"nias", u"\u00E1"+"sio-americano", u"\u00E1"+"sio-australiano",
u"\u00E1"+"sio-brasileiro", u"\u00E1"+"sio-franc"+u"\u00EA"+"s", u"\u00E1"+"sio-ingl"+u"\u00EA"+"s",
u"\u00E1"+"sio-lusitanismo", u"\u00E1"+"sio-lusitan"+u"\u00ED"+"stico", u"\u00E1"+"sio-lusitano",
u"\u00E1"+"sio-portugu"+u"\u00EA"+"s", u"\u00E1"+"sio-sax"+u"\u00E3"+"o", "assessor-chefe", "assessora-chefe",
"assessoras-chefes", "assessores-chefes", "astro-rei",
"astros-reis", "atividade-fim", "atividade-meio",
"ativo-assistido", "ato-show", u"\u00E1"+"tomo-grama",
"ator-empres"+u"\u00E1"+"rio", "atores-empres"+u"\u00E1"+"rios", "atos-show",
"atriz-empres"+u"\u00E1"+"ria", "atrizes-empres"+u"\u00E1"+"rias", "atum-azul",
"atum-bandolim", "atum-branco", "atum-preto",
"atum-rabilho", "atum-verdadeiro", "atuns-azuis",
"atuns-brancos", "atuns-pretos", "atuns-verdadeiros",
"audio-olfat"+u"\u00F3"+"rio", "audio-oral", "auditor-fiscal",
"Auditor-Fiscal", "auditores-fiscais", "Auditores-Fiscais",
"augusto-correense", "augusto-limense", "augusto-pestanense",
"augusto-severense", "australo-africano", "austro-africano",
"austro-"+u"\u00E1"+"frico", "austro-alem"+u"\u00E3"+"o", "austro-asi"+u"\u00E1"+"tico",
"austro-b"+u"\u00E1"+"varo", "austro-brasileiro", "austro-central",
"austro-flamengo", "austro-holand"+u"\u00EA"+"s", "austro-h"+u"\u00FA"+"ngaro",
"austro-siberiano", "autor-cientista", "autora-cientista",
"autoras-cientistas", "autores-cientistas",
"aux"+u"\u00ED"+"lio-acident"+u"\u00E1"+"rio", "aux"+u"\u00ED"+"lio-acidente",
"Aux"+u"\u00ED"+"lio-Acidente", "aux"+u"\u00ED"+"lio-alimenta"+u"\u00E7"+u"\u00E3"+"o",
"Aux"+u"\u00ED"+"lio-Alimenta"+u"\u00E7"+u"\u00E3"+"o", "aux"+u"\u00ED"+"lio-creche",
"aux"+u"\u00ED"+"lio-desemprego", "Aux"+u"\u00ED"+"lio-Desemprego",
"aux"+u"\u00ED"+"lio-doen"+u"\u00E7"+"a", "aux"+u"\u00ED"+"lio-educa"+u"\u00E7"+u"\u00E3"+"o",
"aux"+u"\u00ED"+"lio-enfermidade", "aux"+u"\u00ED"+"lio-funeral",
"aux"+u"\u00ED"+"lio-maternidade", "aux"+u"\u00ED"+"lio-moradia",
"aux"+u"\u00ED"+"lio-natalidade", "aux"+u"\u00ED"+"lio-palet"+u"\u00F3",
"aux"+u"\u00ED"+"lio-reclus"+u"\u00E3"+"o", "aux"+u"\u00ED"+"lio-refei"+u"\u00E7"+u"\u00E3"+"o",
"Aux"+u"\u00ED"+"lio-Refei"+u"\u00E7"+u"\u00E3"+"o", "aux"+u"\u00ED"+"lio-sa"+u"\u00FA"+"de",
"aux"+u"\u00ED"+"lio-transporte",
"avant-garde", "avant-premi\u00E8re", "avant-sc\u00E8ne", "ave-maria",
"Ave-Maria", "ave-marias", "Ave-Marias",
"aveia-amarela", "aveia-comum", "aveia-est"+u"\u00E9"+"ril",
"aveias-amarelas", "aveias-comuns", "aveias-est"+u"\u00E9"+"reis",
"avenca-branca", "avenca-brasileira", "avenca-cuneiforme",
"avenca-estrelada", "avenca-grande", "avenca-mi"+u"\u00FA"+"da",
"avenca-negra", "avenca-paulista", "avenca-trapeziforme",
"avencas-brasileiras", "avencas-cuneiformes", "avencas-estreladas",
"avencas-grandes", "avencas-mi"+u"\u00FA"+"das", "avencas-negras",
"avencas-paulistas", "avencas-trapeziformes", "avi"+u"\u00E3"+"o-cegonha",
"avi"+u"\u00E3"+"o-correio", "avi"+u"\u00E3"+"o-espi"+u"\u00E3"+"o", "avi"+u"\u00E3"+"o-foguete",
"avi"+u"\u00E3"+"o-laborat"+u"\u00F3"+"rio", "avi"+u"\u00E3"+"o-suicida", "avi"+u"\u00E3"+"o-tanque",
"avi"+u"\u00F5"+"es-suicidas", "aviso-pr"+u"\u00E9"+"vio", "avisos-pr"+u"\u00E9"+"vios",
"av"+u"\u00F3"+"-rainha", "azeitona-rei", "azuis-celestes",
"azuis-claras", "azuis-claros", "azuis-esbranqui"+u"\u00E7"+"adas",
"azuis-esbranqui"+u"\u00E7"+"ados", "azuis-escuras", "azuis-escuros",
"azuis-marinhos", "azul-celeste", "azul-cinzento",
"azul-claro", "azul-cobalto", "azul-esbranqui"+u"\u00E7"+"ado",
"azul-escuro", "azul-esverdeado", "azul-faian"+u"\u00E7"+"a",
"azul-ferrete", "azul-marinho", "azul-piscina",
"azul-seda", "azul-turquesa", "azul"+u"\u00E3"+"o-bicudo",
"azul"+u"\u00F5"+"es-bicudos", "baba-ovo", "bailarino-escarlate",
"bailarinos-escarlates", "baixa-grandense", "baixa-mar",
"baixa-verdense", "baixo-alem"+u"\u00E3"+"o", "baixo-astral",
"baixo-bret"+u"\u00E3"+"o", "baixo-espiritismo", "baixo-imp"+u"\u00E9"+"rio",
"baixo-latim", "baixo-relevo", "baixo-ventre",
"baixos-relevos", "baleia-an"+u"\u00E3", "baleia-assassina",
"baleia-azul", "baleia-bicuda", "baleia-branca",
"baleia-corcunda", "baleia-dente", "baleia-espadarte",
"baleia-franca", "baleia-lisa", "baleia-mirim",
"baleia-pamonha", "baleia-piloto", "baleia-pintada",
"baleia-preta", "baleia-verdadeira", "baleias-an"+u"\u00E3"+"s",
"baleias-assassinas", "baleias-azuis", "baleias-bicudas",
"baleias-brancas", "baleias-corcundas", "baleias-francas",
"baleias-lisas", "baleias-mirins", "baleias-pintadas",
"baleias-pretas", "baleias-verdadeiras", "banana-an"+u"\u00E3",
"banana-branca", "banana-caturra", "banana-comprida",
"banana-figo", "banana-francesa", "banana-grande",
"banana-inaj"+u"\u00E1", "banana-ma"+u"\u00E7"+u"\u00E3", "banana-m"+u"\u00E3"+"e",
"banana-menina", "banana-mole", "banana-naj"+u"\u00E1",
"banana-nanica", "banana-ouro", "banana-pacova",
"banana-pacov"+u"\u00E1", "banana-p"+u"\u00E3"+"o", "banana-passa",
"banana-pedra", "banana-prata", "banana-rajada",
"banana-real", "banana-roxa", "banana-timb"+u"\u00F3",
"banana-verde", "bananas-an"+u"\u00E3"+"s", "bananas-brancas",
"bananas-caturras", "bananas-compridas", "bananas-francesas",
"bananas-grandes", "bananas-moles", "bananas-nanicas",
"bananas-passas", "bananas-rajadas", "bananas-reais",
"bananas-roxas", "bananas-split", "bananas-verdes",
"bananeira-an"+u"\u00E3", "bananeira-branca", "bananeira-brava",
"bananeira-chinesa", "bananeira-nanica", "bananeira-peti"+u"\u00E7"+"a",
"bananeira-rainha", "bananeira-vermelha", "bananeiras-an"+u"\u00E3"+"s",
"bananeiras-brancas", "bananeiras-bravas", "bananeiras-chinesas",
"bananeiras-nanicas", "bananeiras-peti"+u"\u00E7"+"as", "bananeiras-rainhas",
"bananeiras-vermelhas", "batata-baroa", "batata-baronesa",
"batata-branca", "batata-brava", "batata-cear"+u"\u00E1",
"batata-chinesa", "batata-cogumelo", "batata-doce",
"batata-inglesa", "batata-mi"+u"\u00FA"+"da", "batata-portuguesa",
"batata-roxa", "batata-salsa", "batata-semente",
"batata-silvestre", "batata-su"+u"\u00ED"+u"\u00E7"+"a", "batatas-baroas",
"batatas-baronesas", "batatas-brancas", "batatas-bravas",
"batatas-chinesas", "batatas-doces", "batatas-inglesas",
"batatas-mi"+u"\u00FA"+"das", "batatas-portuguesas", "batatas-roxas",
"batatas-silvestres", "batatas-su"+u"\u00ED"+u"\u00E7"+"as", "bate-estaca",
"bate-estacas", "beb"+u"\u00EA"+"-proveta", "beb"+u"\u00EA"+"s-provetas",
"belo-horizontina", "belo-horizontinas", "belo-horizontino", "belo-horizontinos",
"bem-acabado", "bem-acabada", "bem-acabadas", "bem-acabados",
"bem-aceita", "bem-aceitas", "bem-aceito", "bem-aceitos",
"bem-acondicionada", "bem-acondicionadas", "bem-acondicionado", "bem-acondicionados",
"bem-acondi"+u"\u00E7"+"oada", "bem-acondi"+u"\u00E7"+"oadas", "bem-acondi"+u"\u00E7"+"oado", "bem-acondi"+u"\u00E7"+"oados",
"bem-aconselhada", "bem-aconselhadas", "bem-aconselhado", "bem-aconselhados",
"bem-acostumada", "bem-acostumadas", "bem-acostumado", "bem-acostumados",
"bem-adaptada", "bem-adaptadas", "bem-adaptado", "bem-adaptados",
"bem-afamada", "bem-afamadas", "bem-afamado", "bem-afamados",
"bem-afei"+u"\u00E7"+"oada", "bem-afei"+u"\u00E7"+"oadas", "bem-afei"+u"\u00E7"+"oado", "bem-afei"+u"\u00E7"+"oados",
"bem-afortunar", "bem-agradecida", "bem-agradecidas", "bem-agradecido",
"bem-agradecidos", "bem-ajambrada", "bem-ajambradas", "bem-ajambrado",
"bem-ajambrados", "bem-amada", "bem-amadas", "bem-amado",
"bem-amados", "bem-andan"+u"\u00E7"+"a", "bem-andan"+u"\u00E7"+"as", "bem-andante",
"bem-andantes", "bem-apanhada", "bem-apanhadas", "bem-apanhado",
"bem-apanhados", "bem-apessoada", "bem-apessoadas", "bem-apessoado",
"bem-apessoados", "bem-apresentada", "bem-apresentadas", "bem-apresentado",
"bem-apresentados", "bem-arranjada", "bem-arranjadas", "bem-arranjado",
"bem-arranjados", "bem-arrumada", "bem-arrumadas", "bem-arrumado",
"bem-arrumados", "Bem-aventurada", "bem-aventurada", "Bem-aventuradas",
"bem-aventuradas", "Bem-aventurado", "Bem-aventurados", "bem-aventurados",
"bem-aventuran"+u"\u00E7"+"a", "bem-aventuran"+u"\u00E7"+"as", "bem-aventurar", "bem-avinda",
"bem-avindas", "bem-avindo", "bem-avindos", "bem-avisada",
"bem-avisadas", "bem-avisado", "bem-avisados", "bem-bom",
"bem-bons", "bem-ca"+u"\u00E7"+"ada", "bem-ca"+u"\u00E7"+"adas", "bem-ca"+u"\u00E7"+"ado",
"bem-ca"+u"\u00E7"+"ados", "bem-casada", "bem-casadas", "bem-casadinho",
"bem-casadinhos", "bem-casado", "bem-casados", "bem-comportada",
"bem-comportadas", "bem-comportado", "bem-comportados", "bem-composta",
"bem-compostas", "bem-composto", "bem-compostos", "bem-conceituada",
"bem-conceituadas", "bem-conceituado", "bem-conceituados", "bem-conformada",
"bem-conformadas", "bem-conformado", "bem-conformados", "bem-convidada",
"bem-convidadas", "bem-convidado", "bem-convidados", "bem-criada",
"bem-criadas", "bem-criado", "bem-criados", "bem-curada",
"bem-curadas", "bem-curado", "bem-curados", "bem-dada",
"bem-dadas", "bem-dado", "bem-dados", "bem-disposta",
"bem-dispostas", "bem-disposto", "bem-dispostos", "bem-ditosa",
"bem-ditosas", "bem-ditoso", "bem-ditosos", "bem-dizente",
"bem-dizentes", "bem-dizer", "bem-dormida", "bem-dormidas",
"bem-dormido", "bem-dormidos", "bem-dotada", "bem-dotadas",
"bem-dotado", "bem-dotados", "bem-educada", "bem-educadas",
"bem-educado", "bem-educados", "bem-encarada", "bem-encaradas",
"bem-encarado", "bem-encarados", "bem-ensinada", "bem-ensinadas",
"bem-ensinado", "bem-ensinados", "bem-estar", "bem-fadada",
"bem-fadadas", "bem-fadado", "bem-fadados", "bem-fadar",
"bem-falante", "bem-falantes", "bem-fazente", "bem-fazentes",
"bem-fazer", "bem-feita", "bem-feitas", "bem-feito",
"bem-feitos", "bem-haja", "bem-humorada", "bem-humoradas",
"bem-humorado", "bem-humorados", "bem-ida", "bem-idas",
"bem-ido", "bem-idos", "bem-intencionada", "bem-intencionadas",
"bem-intencionado", "bem-intencionados", "bem-lan"+u"\u00E7"+"ada", "bem-lan"+u"\u00E7"+"adas",
"bem-lan"+u"\u00E7"+"ado", "bem-lan"+u"\u00E7"+"ados", "bem-mandada", "bem-mandadas",
"bem-mandado", "bem-mandados", "bem-merecer", "bem-merecida",
"bem-merecidas", "bem-merecido", "bem-merecidos", "bem-nado",
"bem-nados", "bem-nascida", "bem-nascidas", "bem-nascido",
"bem-nascidos", "bem-ordenada", "bem-ordenadas", "bem-ordenado",
"bem-ordenados", "bem-ouvida", "bem-ouvidas", "bem-ouvido",
"bem-ouvidos", "bem-parada", "bem-paradas", "bem-parado",
"bem-parados", "bem-pensante", "bem-pensantes", "bem-posta",
"bem-postas", "bem-posto", "bem-postos", "bem-procedida",
"bem-procedidas", "bem-procedido", "bem-procedidos", "bem-proporcionada",
"bem-proporcionadas", "bem-proporcionado", "bem-proporcionados", "bem-queren"+u"\u00E7"+"a",
"bem-queren"+u"\u00E7"+"as", "bem-querente", "bem-querentes", "bem-querer",
"bem-sabida", "bem-sabidas", "bem-sabido", "bem-sabidos",
"bem-soante", "bem-soantes", "bem-son"+u"\u00E2"+"ncia", "bem-son"+u"\u00E2"+"ncias",
"bem-sonante", "bem-sonantes", "bem-sucedida", "bem-sucedidas",
"bem-sucedido", "bem-sucedidos", "bem-talhada", "bem-talhadas",
"bem-talhado", "bem-talhados", "bem-temente", "bem-tementes",
"bem-vestida", "bem-vestidas", "bem-vestido", "bem-vestidos",
"bem-vestir", "bem-vinda", "Bem-vinda", "bem-vindas",
"Bem-vindas", "bem-vindo", "Bem-vindo", "bem-vindos",
"Bem-vindos", "bem-vista", "bem-vistas", "bem-visto",
"bem-vistos", "best-seller", "best-sellers", "besta-fera",
"besta-quadrada", "bestas-feras", "Blu-ray",
"Blu-Ray", "blue-jeans", "bossa-nova",
"boia-fria", "boias-frias",
"boto-amarelo", "boto-branco", "boto-cinza",
"boto-preto", "boto-tucuxi", "boto-vermelho",
"botos-amarelos", "botos-brancos", "botos-pretos",
"botos-vermelhos", "brejo-cruzense", "brejo-grandense",
"brejo-santense", "cabo-grandense", "cabo-guia",
"cabo-joanense", "cabo-verdense", "cabo-verdianismo",
"cabo-verdiano", "cabra-cega", "cabras-cegas",
"ca"+u"\u00E7"+"a-minas", "ca"+u"\u00E7"+"a-moscas", "ca"+u"\u00E7"+"a-n"+u"\u00ED"+"quel",
"ca"+u"\u00E7"+"a-palavra", "Ca"+u"\u00E7"+"a-Palavra", "ca"+u"\u00E7"+"a-palavras",
"caga-fogo", "caixa-forte", "caixa-prego",
"caixa-preta", "caixas-fortes", "caixas-pregos",
"caixas-pretas", "c"+u"\u00E1"+"lcio-fosfato", "caloria-grama",
"cama-beliche", "caminh"+u"\u00E3"+"o-ba"+u"\u00FA", "caminh"+u"\u00E3"+"o-betoneira",
"caminh"+u"\u00E3"+"o-bomba", "caminh"+u"\u00E3"+"o-fossa", "caminh"+u"\u00E3"+"o-pipa",
"caminh"+u"\u00E3"+"o-tanque", "caminh"+u"\u00F5"+"es-ba"+u"\u00FA", "can"+u"\u00E1"+"rio-assobio",
"can"+u"\u00E1"+"rio-assoviador", "can"+u"\u00E1"+"rio-baeta", "can"+u"\u00E1"+"rio-belga",
"can"+u"\u00E1"+"rio-pardinho", "can"+u"\u00E1"+"rio-pardo", "can"+u"\u00E1"+"rios-assoviadores",
"can"+u"\u00E1"+"rios-belgas", "can"+u"\u00E1"+"rios-pardinhos", "can"+u"\u00E1"+"rios-pardos",
"carro-bomba", "carro-chefe", "carro-conceito",
"carro-dormit"+u"\u00F3"+"rio", "carro-forte", "carro-guincho",
"carro-leito", "carro-madrinha", "carro-patrulha",
"carro-pipa", "carro-restaurante", "carro-sal"+u"\u00E3"+"o",
"carro-suporte", "carro-tanque", "carro-vassoura",
"carros-fortes", "carta-bilhete", "carta-bomba",  "carta-limpa",
"carta-partida", "carta-piloto", "carta-postal",
"carta-telegrama", "cart"+u"\u00E3"+"o-alimenta"+u"\u00E7"+u"\u00E3"+"o",
"Cart"+u"\u00E3"+"o-Alimenta"+u"\u00E7"+u"\u00E3"+"o", "cart"+u"\u00E3"+"o-ficha", "cart"+u"\u00E3"+"o-postal",
"cart"+u"\u00E3"+"o-resposta", "Cart"+u"\u00E3"+"o-Resposta", "cartas-postais",
"cart"+u"\u00F5"+"es-postais", "Cart"+u"\u00F5"+"es-Respostas", "cavalo-branco",
"cavalo-fantasma", "cavalo-hora", "cavalo-judeu",
"cavalo-marinho", "cavalo-mosca", "cavalo-reiuno",
"cavalo-rinch"+u"\u00E3"+"o", "cavalo-vapor", "caval"+u"\u00F5"+"es-negrais",
"cavalos-judeus", "cavalos-marinhos", "cavalos-rinch"+u"\u00F5"+"es",
"CD-ROM", "Cear"+u"\u00E1"+"-Mirim", "c"+u"\u00E9"+"lula-alvo",
"c"+u"\u00E9"+"lula-filha", "c"+u"\u00E9"+"lula-flama", "c"+u"\u00E9"+"lula-guarda",
"c"+u"\u00E9"+"lula-m"+u"\u00E3"+"e", "c"+u"\u00E9"+"lula-mestre", "c"+u"\u00E9"+"lula-ovo",
"c"+u"\u00E9"+"lula-semente", "c"+u"\u00E9"+"lula-tronco", "C"+u"\u00E9"+"lula-Tronco",
"c"+u"\u00E9"+"lulas-T", "centro-africano", "centro-americanismo",
"centro-american"+u"\u00ED"+"stico", "centro-americano", "centro-direita",
"centro-esquerda", "centro-esquerdista", "centro-europeu",
"centro-leste", "Centro-Leste", "centro-norte",
"centro-novense", "centro-oeste", "Centro-Oeste",
"centro-remate", "centro-sul", "ch"+u"\u00E1"+"-biriba",
"ch"+u"\u00E1"+"-bravo", "ch"+u"\u00E1"+"-dan"+u"\u00E7"+"ante", "ch"+u"\u00E1"+"-ingl"+u"\u00EA"+"s",
"ch"+u"\u00E1"+"-mate", "ch"+u"\u00E1"+"-mineiro", "ch"+u"\u00E1"+"-paulista",
"ch"+u"\u00E1"+"-pomonga", "ch"+u"\u00E1"+"-preto", "ch"+u"\u00E1"+"-verde",
"ch"+u"\u00E3"+"-grandense", "ch"+u"\u00E3"+"-pretense", "chave-inglesa",
"chave-mestra", "chaves-inglesas", "checo-eslovaco",
"chefe-executivo", "chefe-geral", "chefes-executivos",
"cidad"+u"\u00E3"+"o-cliente", "cidade-cogumelo", "cidade-dormit"+u"\u00F3"+"rio",
"cidade-estado", "cidade-Estado", "cidade-fantasma",
"cidade-gauchense", "cidade-irm"+u"\u00E3", "cidade-jardim",
"cidade-laborat"+u"\u00F3"+"rio", "cidade-linear", "cidade-luz",
"cidade-museu", "cidade-sat"+u"\u00E9"+"lite", "cidade-sede",
"Cidade-Sede", "cidade-universit"+u"\u00E1"+"ria", "cidades-irm"+u"\u00E3"+"s",
"cidades-lineares", "cidades-museu", "ci"+u"\u00EA"+"ncia-cidad"+u"\u00E3",
"cient"+u"\u00ED"+"fico-anal"+u"\u00ED"+"tico", "cient"+u"\u00ED"+"fico-cultural", "cient"+u"\u00ED"+"fico-filos"+u"\u00F3"+"fico",
"cient"+u"\u00ED"+"fico-natural", "cient"+u"\u00ED"+"fico-tecnol"+u"\u00F3"+"gico", "cientista-chefe",
"cientista-cidad"+u"\u00E3"+"o", "cientistas-chefes", "cientistas-cidad"+u"\u00E3"+"os",
"cliente-alvo", "cliente-cidad"+u"\u00E3"+"o", "cliente-fornecedor",
"cliente-servidor", "cliente-tipo", "clientes-fornecedores", "clim"+u"\u00E1"+"tico-ambiental",
"cl"+u"\u00ED"+"nico-cir"+u"\u00FA"+"rgica", "cl"+u"\u00ED"+"nico-cir"+u"\u00FA"+"rgicas",
"cl"+u"\u00ED"+"nico-cir"+u"\u00FA"+"rgico", "cl"+u"\u00ED"+"nico-cir"+u"\u00FA"+"rgicos",
"cl"+u"\u00ED"+"nico-grupal", "cl"+u"\u00ED"+"nico-patol"+u"\u00F3"+"gico",
"cl"+u"\u00ED"+"nico-psicol"+u"\u00F3"+"gico", "cl"+u"\u00ED"+"nico-radiol"+u"\u00F3"+"gica", "comandante-chefe",
"comandante-geral", "comandantes-chefes", "comandantes-gerais",
"comando-chefe", "comando-geral", "Comando-Geral",
"comandos-chefes", "comandos-gerais", "com"+u"\u00E9"+"dia-pastel"+u"\u00E3"+"o",
"comendador-mor", "comendadores-mores", "c"+u"\u00F4"+"nsul-geral",
"c"+u"\u00F4"+"nsules-gerais", "consultor-geral", "Consultor-Geral",
"consultoria-geral", "Consultoria-Geral", "Conta-Corrente",
"conta-corrente", "conta-correntista", "conta-corrent"+u"\u00ED"+"stico",
"conta-fios", "conta-giros", "conta-gotas",
"conta-passos", "conta-quil"+u"\u00F4"+"metros", "conta-rota"+u"\u00E7"+u"\u00F5"+"es",
"conta-sal"+u"\u00E1"+"rio", "conta-voltas", "cont"+u"\u00E1"+"bil-financeiro",
"contador-mor", "contadores-mores", "contas-correntes",
"coron"+u"\u00E9"+"is-aviadores", "coron"+u"\u00E9"+"is-generais", "coronel-aviador",
"coronel-barrense", "coronel-cardosense", "coronel-ezequielense",
"coronel-fabricianense", "coronel-freitense", "coronel-general",
"coronel-goulartense", "coronel-juvenciense", "coronel-macedense",
"coronel-maiense", "coronel-martinense", "coronel-moemense",
"coronel-muriciense", "coronel-pachequense", "coronel-pilarense",
"coronel-teixeirense", "corta-luz", "cr"+u"\u00E9"+"dito-pr"+u"\u00EA"+"mio",
"credo-cruz", "crepe-cetim", "crepe-marroquim",
"criado-mudo", "criados-mudos", "crian"+u"\u00E7"+"a-problema",
"crist"+u"\u00E3"+"-democrata", "Crist"+u"\u00E3"+"-Democrata", "crist"+u"\u00E3"+"o-democrata",
"Crist"+u"\u00E3"+"o-Democrata", "cruz-almense", "cruz-altense",
"cruz-altino", "cruz-credo", "cruz-diabo",
"cruz-gracense", "cruz-machadense", "cruz-maltense",
"cruz-maltino", "cruz-pedrense", "cruz-possense",
"cruz-serrano", "cruzado-novo", "cruzados-novos",
"cruzeiro-fortalezense", "cruzeiro-nortense", "cruzeiro-oestense",
"cruzeiro-peixotense", "curta-metragem", "curtas-metragens",
"curto-circuitar", "curto-circuitos", "curtos-circuitos",
"curtos-flexores", "custo-benef"+u"\u00ED"+"cio", "custo-hora",
"Custo-Hora", "custo-rendimento", "decreto-lei",
"Decreto-Lei", "decretos-leis", "dedo-duro",
"del-rei", "del-Rei", "Del-Rei",
"delegada-chefe", "delegada-corregedora", "delegada-geral",
"delegadas-chefes", "delegado-chefe", "delegado-corregedor",
"delegado-geral", "delegados-chefes", "demo-liberal",
"demo-liberalismo", "demo-socialismo", "democracia-crist"+u"\u00E3",
"democrata-crist"+u"\u00E3", "democrata-crist"+u"\u00E3"+"o", "Desembargador-Corregedor",
"desembargador-relator", "desembargadora-relatora", "desembargadoras-relatoras",
"desembargadores-relatores", "diabo-marinho", "diabos-marinhos",
"did"+u"\u00E1"+"tico-cient"+u"\u00ED"+"fico", "did"+u"\u00E1"+"tico-pedag"+u"\u00F3"+"gico", "diesel-el"+u"\u00E9"+"trico",
"diesel-hidr"+u"\u00E1"+"ulico", "digital-anal"+u"\u00F3"+"gico", "Digital-Anal"+u"\u00F3"+"gico",
"dire"+u"\u00E7"+u"\u00E3"+"o-geral", "Dire"+u"\u00E7"+u"\u00E3"+"o-Geral", "dire"+u"\u00E7"+u"\u00E3"+"o-regional",
"dire"+u"\u00E7"+u"\u00F5"+"es-gerais", "direita-esquerda", "direito-fun"+u"\u00E7"+u"\u00E3"+"o",
"diretor-assistente", "diretor-associado", "diretor-criativo",
"diretor-desportivo", "diretor-executivo", "diretor-geral",
"Diretor-Geral", "diretor-gerente", "diretor-presidente",
"Diretor-Presidente", "diretor-produtor", "diretor-regional",
"diretor-secret"+u"\u00E1"+"rio", "diretor-superintendente", "diretora-assistente",
"diretora-associada", "diretora-criativa", "diretora-executiva",
"diretora-geral", "diretora-presidente", "diretora-produtora",
"diretoras-assistentes", "diretoras-associadas", "diretoras-criativas",
"diretoras-executivas", "diretoras-presidentes", "diretoras-produtoras",
"diretores-assistentes", "diretores-associados", "diretores-criativos",
"diretores-gerais", "diretores-gerentes", "diretores-presidentes",
"diretores-produtores", "diretores-secret"+u"\u00E1"+"rios", "diretoria-geral",
"Diretoria-Geral", "diretorias-gerais", "dose-padr"+u"\u00E3"+"o",
"ecol"+u"\u00F3"+"gico-aqu"+u"\u00E1"+"tico", "ecol"+u"\u00F3"+"gico-econ"+u"\u00F4"+"mico", "Ecol"+u"\u00F3"+"gico-Econ"+u"\u00F4"+"mico",
"ecol"+u"\u00F3"+"gico-jur"+u"\u00ED"+"dico", "econ"+u"\u00F4"+"mico-comercial", "econ"+u"\u00F4"+"mico-contributivo",
"econ"+u"\u00F4"+"mico-ecol"+u"\u00F3"+"gico", "Econ"+u"\u00F4"+"mico-Ecol"+u"\u00F3"+"gico", "econ"+u"\u00F4"+"mico-financeiro",
"Econ"+u"\u00F4"+"mico-Financeiro", "econ"+u"\u00F4"+"mico-fiscal", "econ"+u"\u00F4"+"mico-social",
"economista-chefe", "edif"+u"\u00ED"+"cio-garagem", "edif"+u"\u00ED"+"cio-sede",
"Edif"+u"\u00ED"+"cio-Sede", "editor-assistente", "editor-chefe",
"Editor-Chefe", "editor-executivo", "editora-assistente",
"editora-chefe", "editora-executiva", "editoras-assistentes",
"editoras-executivas", "editores-assistentes", "editores-executivos",
"el"+u"\u00E9"+"ctron-volt", "el"+u"\u00E9"+"ctrons-volt", "elefante-marinho",
"elefantes-marinhos", "elemento-chave", "eletromagn"+u"\u00E9"+"tico-cir"+u"\u00FA"+"rgico",
"Eletromagn"+u"\u00E9"+"tico-Cir"+u"\u00FA"+"rgico", "el"+u"\u00E9"+"tron-Volt", "el"+u"\u00E9"+"tron-volt",
"eletr"+u"\u00F4"+"nico-biol"+u"\u00F3"+"gico", "eletr"+u"\u00F4"+"nico-qu"+u"\u00E2"+"ntico", "el"+u"\u00E9"+"trons-volt",
"escola-empresa", "Escola-Empresa", "escola-modelo",
"Estado-Juiz", "estado-maior", "Estado-Maior",
"Estado-Membro", "estado-membro", "Estado-membro",
"estado-na"+u"\u00E7"+u"\u00E3"+"o", "Estado-na"+u"\u00E7"+u"\u00E3"+"o", "Estado-Na"+u"\u00E7"+u"\u00E3"+"o",
"estado-unidense", "estados-membros", "Estados-membros",
"Estados-Membros", "estados-na"+u"\u00E7"+u"\u00F5"+"es", u"\u00E9"+"tico-jur"+u"\u00ED"+"dico",
u"\u00E9"+"tico-moral", u"\u00E9"+"tico-pol"+u"\u00ED"+"tico", u"\u00E9"+"tico-profissional",
u"\u00E9"+"tico-social", u"\u00E9"+"tnico-racial", u"\u00E9"+"tnico-religioso",
"euro-africano", "euro-americano", "euro-"+u"\u00E1"+"rabe",
"euro-asi"+u"\u00E1"+"tico", "euro-atl"+u"\u00E2"+"ntico", "euro-d"+u"\u00F3"+"lar",
"euro-mediterr"+u"\u00E2"+"nico", "euro-otimismo", "euro-regi"+u"\u00E3"+"o",
"euro-siberiano", "euro-tropical", "executiva-chefe",
"executivas-chefes", "executivo-chefe", "executivos-chefes",
"fax-modem", "faxes-modens", "fazenda-modelo",
"fazenda-novense", "fecho-"+u"\u00E9"+"clair", "fecho-ecler",
"feij"+u"\u00E3"+"o-alfanje", "feij"+u"\u00E3"+"o-amendoim", "feij"+u"\u00E3"+"o-an"+u"\u00E3"+"o",
"feij"+u"\u00E3"+"o-aspargo", "feij"+u"\u00E3"+"o-azeite", "feij"+u"\u00E3"+"o-bacamarte",
"feij"+u"\u00E3"+"o-baet"+u"\u00E3"+"o", "feij"+u"\u00E3"+"o-baru", "feij"+u"\u00E3"+"o-batata",
"feij"+u"\u00E3"+"o-besugo", "feij"+u"\u00E3"+"o-branco", "feij"+u"\u00E3"+"o-bravo",
"feij"+u"\u00E3"+"o-cabacinha", "feij"+u"\u00E3"+"o-caboclo", "feij"+u"\u00E3"+"o-caf"+u"\u00E9",
"feij"+u"\u00E3"+"o-cap"+u"\u00E3"+"o", "feij"+u"\u00E3"+"o-careta", "feij"+u"\u00E3"+"o-carioca",
"feij"+u"\u00E3"+"o-carito", "feij"+u"\u00E3"+"o-carrapatinho", "feij"+u"\u00E3"+"o-carrapato",
"feij"+u"\u00E3"+"o-carumb"+u"\u00E9", "feij"+u"\u00E3"+"o-casado", "feij"+u"\u00E3"+"o-castanho",
"feij"+u"\u00E3"+"o-catarino", "feij"+u"\u00E3"+"o-cavalinho", "feij"+u"\u00E3"+"o-chicote",
"feij"+u"\u00E3"+"o-china", "feij"+u"\u00E3"+"o-chin"+u"\u00EA"+"s", "feij"+u"\u00E3"+"o-chocolate",
"feij"+u"\u00E3"+"o-chumbinho", "feij"+u"\u00E3"+"o-coco", "feij"+u"\u00E3"+"o-colubrino",
"feij"+u"\u00E3"+"o-comum", "feij"+u"\u00E3"+"o-congo", "feij"+u"\u00E3"+"o-corda",
"feij"+u"\u00E3"+"o-covado", "feij"+u"\u00E3"+"o-cru", "feij"+u"\u00E3"+"o-enxofre",
"feij"+u"\u00E3"+"o-escarlate", "feij"+u"\u00E3"+"o-espada", "feij"+u"\u00E3"+"o-farinha",
"feij"+u"\u00E3"+"o-fava", "feij"+u"\u00E3"+"o-favona", "feij"+u"\u00E3"+"o-fidalgo",
"feij"+u"\u00E3"+"o-figueira", "feij"+u"\u00E3"+"o-flor", "feij"+u"\u00E3"+"o-forrageiro",
"feij"+u"\u00E3"+"o-frade", "feij"+u"\u00E3"+"o-fradinho", "feij"+u"\u00E3"+"o-gigante",
"feij"+u"\u00E3"+"o-gl"+u"\u00F3"+"ria", "feij"+u"\u00E3"+"o-grande", "feij"+u"\u00E3"+"o-guando",
"feij"+u"\u00E3"+"o-guandu", "feij"+u"\u00E3"+"o-gurutuba", "feij"+u"\u00E3"+"o-holand"+u"\u00EA"+"s",
"feij"+u"\u00E3"+"o-imperador", "feij"+u"\u00E3"+"o-ingl"+u"\u00EA"+"s", "feij"+u"\u00E3"+"o-japon"+u"\u00EA"+"s",
"feij"+u"\u00E3"+"o-jiritana", "feij"+u"\u00E3"+"o-lablabe", "feij"+u"\u00E3"+"o-lagartixa",
"feij"+u"\u00E3"+"o-laranjeiro", "feij"+u"\u00E3"+"o-mac"+u"\u00E1"+u"\u00E7"+"ar", "feij"+u"\u00E3"+"o-mac"+u"\u00FA"+"ndi",
"feij"+u"\u00E3"+"o-m"+u"\u00E3"+"ezinha", "feij"+u"\u00E3"+"o-mancanha", "feij"+u"\u00E3"+"o-manteiga",
"feij"+u"\u00E3"+"o-mexido", "feij"+u"\u00E3"+"o-mineiro", "feij"+u"\u00E3"+"o-miraculoso",
"feij"+u"\u00E3"+"o-miudinho", "feij"+u"\u00E3"+"o-mi"+u"\u00FA"+"do", "feij"+u"\u00E3"+"o-mucuna",
"feij"+u"\u00E3"+"o-mula", "feij"+u"\u00E3"+"o-mulatinho", "feij"+u"\u00E3"+"o-mungo",
"feij"+u"\u00E3"+"o-or"+u"\u00F3", "feij"+u"\u00E3"+"o-papa", "feij"+u"\u00E3"+"o-peludo",
"feij"+u"\u00E3"+"o-pical", "feij"+u"\u00E3"+"o-pinheirinho", "feij"+u"\u00E3"+"o-pombinha",
"feij"+u"\u00E3"+"o-preto", "feij"+u"\u00E3"+"o-rajado", "feij"+u"\u00E3"+"o-rasteiro",
"feij"+u"\u00E3"+"o-sapota", "feij"+u"\u00E3"+"o-selim", "feij"+u"\u00E3"+"o-soja",
"feij"+u"\u00E3"+"o-telefone", "feij"+u"\u00E3"+"o-tonca", "feij"+u"\u00E3"+"o-transl"+u"\u00FA"+"cido",
"feij"+u"\u00E3"+"o-trepador", "feij"+u"\u00E3"+"o-tropeiro", "feij"+u"\u00E3"+"o-vassoura",
"feij"+u"\u00E3"+"o-veludo", "feij"+u"\u00E3"+"o-verde", "feij"+u"\u00E3"+"o-vermelho",
"feij"+u"\u00E3"+"o-vinha", "feij"+u"\u00E3"+"o-virado", "feij"+u"\u00E3"+"o-zebrado",
"feij"+u"\u00E3"+"ozinho-bravo", "feij"+u"\u00F5"+"es-an"+u"\u00F5"+"es", "feij"+u"\u00F5"+"es-baru",
"feij"+u"\u00F5"+"es-brancos", "feij"+u"\u00F5"+"es-bravos", "feij"+u"\u00F5"+"es-caboclos",
"feij"+u"\u00F5"+"es-cariocas", "feij"+u"\u00F5"+"es-casados", "feij"+u"\u00F5"+"es-castanhos",
"feij"+u"\u00F5"+"es-chineses", "feij"+u"\u00F5"+"es-colubrinos", "feij"+u"\u00F5"+"es-comuns",
"feij"+u"\u00F5"+"es-covados", "feij"+u"\u00F5"+"es-crus", "feij"+u"\u00F5"+"es-escarlates",
"feij"+u"\u00F5"+"es-forrageiros", "feij"+u"\u00F5"+"es-gigantes", "feij"+u"\u00F5"+"es-grandes",
"feij"+u"\u00F5"+"es-holandeses", "feij"+u"\u00F5"+"es-ingleses", "feij"+u"\u00F5"+"es-japoneses",
"feij"+u"\u00F5"+"es-laranjeiros", "feij"+u"\u00F5"+"es-mexidos", "feij"+u"\u00F5"+"es-mineiros",
"feij"+u"\u00F5"+"es-miraculosos", "feij"+u"\u00F5"+"es-miudinhos", "feij"+u"\u00F5"+"es-mi"+u"\u00FA"+"dos",
"feij"+u"\u00F5"+"es-moles", "feij"+u"\u00F5"+"es-mulatinhos", "feij"+u"\u00F5"+"es-peludos",
"feij"+u"\u00F5"+"es-pretos", "feij"+u"\u00F5"+"es-rajados", "feij"+u"\u00F5"+"es-rasteiros",
"feij"+u"\u00F5"+"es-transl"+u"\u00FA"+"cidos", "feij"+u"\u00F5"+"es-trepadores", "feij"+u"\u00F5"+"es-verdes",
"feij"+u"\u00F5"+"es-vermelhos", "feij"+u"\u00F5"+"es-virados", "feij"+u"\u00F5"+"es-zebrados",
"feira-grandense", "feira-novense", "fernando-noronhense",
"fernando-prestense", "ferro-gusa", "figura-chave",
"filme-cat"+u"\u00E1"+"strofe", "filme-evento", "filme-fam"+u"\u00ED"+"lia",
"filme-piloto", "fina-flor", "financeiro-cont"+u"\u00E1"+"bil",
"financeiro-or"+u"\u00E7"+"ament"+u"\u00E1"+"rio", "firma-membro", "Firma-Membro",
"f"+u"\u00ED"+"sico-ambiental", "f"+u"\u00ED"+"sico-econ"+u"\u00F4"+"mico", "f"+u"\u00ED"+"sico-educativo",
"f"+u"\u00ED"+"sico-financeiro", "F"+u"\u00ED"+"sico-Financeiro", "f"+u"\u00ED"+"sico-geogr"+u"\u00E1"+"fico",
"f"+u"\u00ED"+"sico-ge"+u"\u00F3"+"grafo", "f"+u"\u00ED"+"sico-matem"+u"\u00E1"+"tico", "f"+u"\u00ED"+"sico-mec"+u"\u00E2"+"nico",
"f"+u"\u00ED"+"sico-mor", "f"+u"\u00ED"+"sico-operacional", "f"+u"\u00ED"+"sico-ps"+u"\u00ED"+"quico",
"f"+u"\u00ED"+"sico-qu"+u"\u00ED"+"mico", "F"+u"\u00ED"+"sico-Qu"+u"\u00ED"+"mico", "folha-corrida",
"foz-igua"+u"\u00E7"+"uense", "francisco-alvense", "francisco-saense",
"francisco-santense", "franco-alem"+u"\u00E3", "franco-alem"+u"\u00E3"+"o",
"franco-americano", "franco-"+u"\u00E1"+"rabe", "franco-argentino",
"franco-atirador", "franco-belga", "franco-bordo",
"franco-brasileiro", "franco-brit"+u"\u00E2"+"nico", "franco-canadense",
"franco-colombiano", "franco-espanhol", "franco-espanhola",
"franco-ingl"+u"\u00EA"+"s", "franco-iraniano", "Franco-Iraniano",
"franco-ma"+u"\u00E7"+"om", "franco-ma"+u"\u00E7"+"onaria", "franco-ma"+u"\u00E7"+u"\u00F4"+"nico",
"franco-portugu"+u"\u00EA"+"s", "franco-proven"+u"\u00E7"+"al", "franco-su"+u"\u00ED"+u"\u00E7"+"o",
"francos-bordos", "geogr"+u"\u00E1"+"fico-lingu"+u"\u00ED"+"stico", "geol"+u"\u00F3"+"gico-geot"+u"\u00E9"+"cnico",
"Geosfera-Biosfera", "Ger"+u"\u00EA"+"ncia-Geral", "gerente-executivo",
"gerente-geral", "Gerente-Geral", "gerentes-executivos",
"gerentes-gerais", "germano-catolicismo", "germano-cat"+u"\u00F3"+"lico",
"germano-celta", "germano-crist"+u"\u00E3"+"o", "germano-cristianismo",
"germano-h"+u"\u00FA"+"ngaro", "germano-latino", "germano-otomano",
"germano-polaco", "germano-portugu"+u"\u00EA"+"s", "germano-sovi"+u"\u00E9"+"tico",
"germano-turco", "gira-mundo", "governador-geral",
"governadores-gerais", "gr"+u"\u00E3"+"-besta", "Gr"+u"\u00E3"+"-Bretanha",
"gr"+u"\u00E3"+"-cruz", "gr"+u"\u00E3"+"-ducado", "gr"+u"\u00E3"+"-ducal",
"gr"+u"\u00E3"+"-duque", "gr"+u"\u00E3"+"-duquesa", "gr"+u"\u00E3"+"-duquesa",
"gr"+u"\u00E3"+"-fino", "gr"+u"\u00E3"+"-lama", "gr"+u"\u00E3"+"-mestrado",
"gr"+u"\u00E3"+"-mestre", "gr"+u"\u00E3"+"-rabino", "gr"+u"\u00E3"+"-sacerdote",
"gr"+u"\u00E3"+"-tinhoso", "gr"+u"\u00E3"+"-turco", "gr"+u"\u00E3"+"-vizir",
"gr"+u"\u00E3"+"o-cruz", "gr"+u"\u00E3"+"o-ducado", "Gr"+u"\u00E3"+"o-Ducado",
"gr"+u"\u00E3"+"o-ducal", "gr"+u"\u00E3"+"o-duque", "gr"+u"\u00E3"+"o-mestrado",
"gr"+u"\u00E3"+"o-mestre", "Gr"+u"\u00E3"+"o-Mestre", "gr"+u"\u00E3"+"o-mogol",
"Gr"+u"\u00E3"+"o-Mongol", "gr"+u"\u00E3"+"o-rabino", "gr"+u"\u00E3"+"o-sacerdote",
"gr"+u"\u00E3"+"o-tinhoso", "gr"+u"\u00E3"+"o-turco", "gr"+u"\u00E3"+"o-vizir",
"gr"+u"\u00E3"+"os-ducados", "gr"+u"\u00E3"+"os-ducais", "gr"+u"\u00E3"+"os-lamas",
"gr"+u"\u00E3"+"os-mestrados", "gr"+u"\u00E3"+"os-mog"+u"\u00F3"+"is", "guarda-arn"+u"\u00EA"+"s",
"guarda-balizas", "guarda-bando", "guarda-barreira",
"guarda-barro", "guarda-bra"+u"\u00E7"+"o", "guarda-cabe"+u"\u00E7"+"a",
"guarda-cabras", "guarda-ca"+u"\u00E7"+"a", "guarda-cadeira",
"guarda-calhas", "guarda-cama", "guarda-cancela",
"guarda-c"+u"\u00E1"+"psula", "guarda-cara", "guarda-cartucho",
"guarda-casacas", "guarda-cascos", "guarda-catarro",
"guarda-chapim", "guarda-chave", "guarda-choque",
"guarda-chuva", "guarda-chuvada", "guarda-civil",
"guarda-comida", "guarda-corpo", "Guarda-Corpo",
"guarda-c"+u"\u00F3"+"s", "guarda-costas", "guarda-espelho",
"guarda-faceira", "guarda-fechos", "guarda-fio",
"guarda-fiscal", "guarda-florestal", "guarda-fogo",
"guarda-freio", "guarda-infantes", "guarda-joias",
"guarda-lama", "guarda-l"+u"\u00E2"+"mina", "guarda-leme",
"guarda-linha", "guarda-livros", "guarda-loi"+u"\u00E7"+"a",
"guarda-loros", "guarda-lou"+u"\u00E7"+"a", "guarda-lume",
"guarda-maior", "guarda-mancebo", "guarda-m"+u"\u00E3"+"o",
"guarda-marinha", "guarda-mato", "guarda-menor",
"guarda-meta", "guarda-mor", "Guarda-Mor",
"guarda-morense", "guarda-morr"+u"\u00E3"+"o", "guarda-m"+u"\u00F3"+"veis",
"guarda-noturno", "guarda-nuca", "guarda-olhos",
"guarda-patr"+u"\u00E3"+"o", "guarda-peito", "guarda-porta",
"guarda-port"+u"\u00E3"+"o", "guarda-prata", "guarda-quedas",
"guarda-queixo", "guarda-raios", "guarda-rede",
"guarda-republicano", "guarda-rios", "guarda-rodas",
"guarda-roupa", "guarda-roupeiro", "guarda-saias",
"guarda-selos", "guarda-sexo", "guarda-sol",
"guarda-solaria", "guarda-tufo", "guarda-vala",
"guarda-vassoiras", "guarda-vassouras", "guarda-vento",
"guarda-ventre", "guarda-vestidos", "guarda-vida",
"guarda-vinho", "guarda-vista", "guarda-volante",
"guarda-volume", "guarda-voz", "guardas-civis",
"guardas-faceiras", "guardas-florestais", "guardas-maiores",
"guardas-menores", "guardas-mores", "guardas-noturnas",
"guardas-noturnos", "hipot"+u"\u00E9"+"tico-dedutivo", "hipot"+u"\u00E9"+"tico-indutivo",
"hispano-americanismo", "hispano-americano", "hispano-"+u"\u00E1"+"rabe",
"hispano-argentino", "hispano-fen"+u"\u00ED"+"cio", "hispano-godo",
"hispano-luso", "hispano-marroquino", "hispano-portugu"+u"\u00EA"+"s",
"hispano-romano", "hist"+u"\u00F3"+"rico-antropol"+u"\u00F3"+"gico", "hist"+u"\u00F3"+"rico-cient"+u"\u00ED"+"fico",
"hist"+u"\u00F3"+"rico-comparativo", "hist"+u"\u00F3"+"rico-cr"+u"\u00ED"+"tico", "Hist"+u"\u00F3"+"rico-Cultural",
"hist"+u"\u00F3"+"rico-cultural", "hist"+u"\u00F3"+"rico-econ"+u"\u00F4"+"mico", "hist"+u"\u00F3"+"rico-etimol"+u"\u00F3"+"gico",
"hist"+u"\u00F3"+"rico-filos"+u"\u00F3"+"fico", "hist"+u"\u00F3"+"rico-fon"+u"\u00E9"+"tico", "hist"+u"\u00F3"+"rico-geneal"+u"\u00F3"+"gico",
"hist"+u"\u00F3"+"rico-geogr"+u"\u00E1"+"fico", "hist"+u"\u00F3"+"rico-lingu"+u"\u00ED"+"stico", "hist"+u"\u00F3"+"rico-natural",
"hist"+u"\u00F3"+"rico-social", "hist"+u"\u00F3"+"rico-teol"+u"\u00F3"+"gico", "Hist"+u"\u00F3"+"rico-Teol"+u"\u00F3"+"gico",
"homem-animal", "homem-aranha", "Homem-Aranha",
"Homem-Areia", "homem-bom", "homem-bomba",
"Homem-Borracha", "homem-borracha", "homem-chave",
"homem-deus", "homem-el"+u"\u00E1"+"stico", "homem-feito",
"homem-fera", "Homem-Fera", "homem-foguete",
"Homem-Foguete", "homem-formiga", "Homem-Formiga",
"homem-forte", "homem-galinha", "homem-gol",
"homem-hora", "Homem-M"+u"\u00E1"+"quina",
"homem-m"+u"\u00E1"+"quina", "homem-massa", "homem-morcego",
"Homem-Morcego", "homem-morto", "homem-mosca",
"homem-mulher", "homem-natureza", "homem-orquestra",
"homem-padr"+u"\u00E3"+"o", "homem-p"+u"\u00E1"+"ssaro", "Homem-P"+u"\u00E1"+"ssaro",
"homem-peixe", "Homem-Peixe", "homem-placa",
"Homem-Radioativo", "homem-radioativo", "homem-rob"+u"\u00F4",
"homem-sandu"+u"\u00ED"+"che", "homens-bons", "homens-deuses",
"homens-feitos", "homens-formigas", "homens-grandes",
"homens-horas", "homens-mortos", "homens-rob"+u"\u00F4"+"s",
"hortel"+u"\u00E3"+"s-silvestres", "hospitais-col"+u"\u00F4"+"nias", "hospital-col"+u"\u00F4"+"nia",
"hospital-escola", "IGP-M", "Ibero-americana", "ibero-americana", "Ibero-americanas",
"ibero-americanas", "Ibero-americano", "ibero-americano", "Ibero-americanos", "ibero-americanos",
"indo-abiss"+u"\u00ED"+"nio", "indo-afeg"+u"\u00E3"+"o", "indo-africano", "indo-"+u"\u00E1"+"rabe",
"indo-ariano", "indo-"+u"\u00E1"+"rico", "indo-australiano",
"Indo-Australiano", "indo-brit"+u"\u00E2"+"nico", "indo-c"+u"\u00E9"+"ltico",
"indo-chin"+u"\u00EA"+"s", "indo-cita", "indo-c"+u"\u00ED"+"tico",
"indo-europe"+u"\u00ED"+"sta", "indo-europeiza"+u"\u00E7"+u"\u00E3"+"o", "indo-europeizado",
"indo-europeu", "indo-gang"+u"\u00E9"+"tico", "indo-germ"+u"\u00E2"+"nico",
"indo-grego", "indo-hel"+u"\u00EA"+"nico", "indo-hitita",
"indo-ingl"+u"\u00EA"+"s", "indo-iraniano", "indo-ir"+u"\u00E2"+"nico",
"indo-malaio", "indo-mal"+u"\u00E1"+"sio", "indo-mel"+u"\u00E2"+"nico",
"indo-mu"+u"\u00E7"+"ulmano", "indo-oce"+u"\u00E2"+"nico", "indo-pac"+u"\u00ED"+"fico",
"indo-paquistan"+u"\u00EA"+"s", "indo-persa", "indo-portugu"+u"\u00EA"+"s",
"indo-russo", "indo-teut"+u"\u00F4"+"nico",
"infecto-contagiosa", "infecto-contagiosas", "infecto-contagioso", "infecto-contagiosos",
"inquisidor-geral",
"inquisidor-mor", "inquisidores-gerais", "inquisidores-mores",
"inspe"+u"\u00E7"+u"\u00E3"+"o-geral", "inspe"+u"\u00E7"+u"\u00F5"+"es-gerais", "inspetor-geral",
"inspetor-orientador", "inspetora-geral", "inspetores-gerais",
"inspetoria-geral", "inspetorias-gerais", "Inspetorias-Gerais",
"IPC-FIFE", "IPC-S", "ip"+u"\u00EA"+"-a"+u"\u00E7"+"u",
"ip"+u"\u00EA"+"-amarelo", "ip"+u"\u00EA"+"-batata", "ip"+u"\u00EA"+"-boia",
"ip"+u"\u00EA"+"-branco", "ip"+u"\u00EA"+"-cabeludo", "ip"+u"\u00EA"+"-caboclo",
"ip"+u"\u00EA"+"-claro", "ip"+u"\u00EA"+"-escuro", "ip"+u"\u00EA"+"-falso",
"ip"+u"\u00EA"+"-felpudo", "ip"+u"\u00EA"+"-mamono", "ip"+u"\u00EA"+"-mandioca",
"ip"+u"\u00EA"+"-mirim", "ip"+u"\u00EA"+"-pardo", "ip"+u"\u00EA"+"-peroba",
"ip"+u"\u00EA"+"-piranga", "ip"+u"\u00EA"+"-preto", "ip"+u"\u00EA"+"-rosa",
"ip"+u"\u00EA"+"-roxo", "ip"+u"\u00EA"+"-tabaco", "ip"+u"\u00EA"+"-verdadeiro",
"ip"+u"\u00EA"+"s-a"+u"\u00E7"+"us", "ip"+u"\u00EA"+"s-amarelos", "ip"+u"\u00EA"+"s-brancos",
"ip"+u"\u00EA"+"s-cabeludos", "ip"+u"\u00EA"+"s-caboclos", "ip"+u"\u00EA"+"s-claros",
"ip"+u"\u00EA"+"s-escuros", "ip"+u"\u00EA"+"s-falsos", "ip"+u"\u00EA"+"s-felpudos",
"ip"+u"\u00EA"+"s-mamonos", "ip"+u"\u00EA"+"s-mirins", "ip"+u"\u00EA"+"s-pardos",
"ip"+u"\u00EA"+"s-perobas", "ip"+u"\u00EA"+"s-pirangas", "ip"+u"\u00EA"+"s-pretos",
"ip"+u"\u00EA"+"s-rosas", "ip"+u"\u00EA"+"s-roxos", "ip"+u"\u00EA"+"s-verdadeiros",
"israelo-africano", "israelo-americano", "israelo-"+u"\u00E1"+"rabe",
"israelo-asi"+u"\u00E1"+"tico", "israelo-brasileiro", "israelo-eg"+u"\u00ED"+"pcio",
"israelo-franc"+u"\u00EA"+"s", "israelo-ingl"+u"\u00EA"+"s", "israelo-jordaniano",
"israelo-jord"+u"\u00E2"+"nio", "israelo-liban"+u"\u00EA"+"s", "israelo-palestiniano",
"israelo-sina"+u"\u00ED"+"tico", "israelo-s"+u"\u00ED"+"rio", "israelo-sovi"+u"\u00E9"+"tico",
u"\u00ED"+"talo-abexim", u"\u00ED"+"talo-abiss"+u"\u00ED"+"nio", u"\u00ED"+"talo-africano",
u"\u00ED"+"talo-alban"+u"\u00EA"+"s", u"\u00ED"+"talo-alem"+u"\u00E3"+"o", u"\u00ED"+"talo-americano",
u"\u00ED"+"talo-"+u"\u00E1"+"rabe", u"\u00ED"+"talo-argelino", u"\u00ED"+"talo-argentino",
u"\u00ED"+"talo-asi"+u"\u00E1"+"tico", u"\u00ED"+"talo-australiano", u"\u00ED"+"talo-austr"+u"\u00ED"+"aco",
u"\u00ED"+"talo-brasileiro", u"\u00ED"+"talo-brit"+u"\u00E2"+"nico", u"\u00ED"+"talo-canadense",
u"\u00ED"+"talo-celta", u"\u00ED"+"talo-c"+u"\u00E9"+"ltico", u"\u00ED"+"talo-colombiano",
u"\u00ED"+"talo-espanhol", u"\u00ED"+"talo-et"+u"\u00ED"+"ope", u"\u00ED"+"talo-eti"+u"\u00F3"+"pico",
u"\u00ED"+"talo-franc"+u"\u00EA"+"s", u"\u00ED"+"talo-gaul"+u"\u00EA"+"s", u"\u00ED"+"talo-germ"+u"\u00E2"+"nico",
u"\u00ED"+"talo-g"+u"\u00F3"+"tico", u"\u00ED"+"talo-grego", u"\u00ED"+"talo-ingl"+u"\u00EA"+"s",
u"\u00ED"+"talo-iugoslavo", u"\u00ED"+"talo-japon"+u"\u00EA"+"s", u"\u00ED"+"talo-luso",
u"\u00ED"+"talo-mexicano", u"\u00ED"+"talo-paulista", u"\u00ED"+"talo-portugu"+u"\u00EA"+"s",
u"\u00ED"+"talo-russo", u"\u00ED"+"talo-sovi"+u"\u00E9"+"tico", u"\u00ED"+"talo-su"+u"\u00ED"+u"\u00E7"+"o",
u"\u00ED"+"talo-turco", "jiu-jitsu", "judaico-crist"+u"\u00E3",
"judaico-crist"+u"\u00E3"+"o", "judaico-crist"+u"\u00E3"+"os", "judaico-russo",
"judiciais-penais", "judicial-penal", "juiz-conselheiro",
"juiz-forense", "juiz-presidente", "Ju"+u"\u00ED"+"za-Corregedora",
"jur"+u"\u00ED"+"dico-administrativo", "jur"+u"\u00ED"+"dico-agr"+u"\u00E1"+"rio", "jur"+u"\u00ED"+"dico-ambiental",
"jur"+u"\u00ED"+"dico-constitucional", "jur"+u"\u00ED"+"dico-cont"+u"\u00E1"+"bil", "jur"+u"\u00ED"+"dico-formal",
"jur"+u"\u00ED"+"dico-legal", "jur"+u"\u00ED"+"dico-normativo", "jur"+u"\u00ED"+"dico-obrigacional",
"jur"+u"\u00ED"+"dico-penal", "Jur"+u"\u00ED"+"dico-Penal", "jur"+u"\u00ED"+"dico-pol"+u"\u00ED"+"tico",
"jur"+u"\u00ED"+"dico-positivista", "jur"+u"\u00ED"+"dico-processual", "jur"+u"\u00ED"+"dico-trabalhista",
"jur"+u"\u00ED"+"dico-tribut"+u"\u00E1"+"rio", "Jur"+u"\u00ED"+"dico-Tribut"+u"\u00E1"+"rio", "juriti-azul",
"juriti-carregadeira", "juriti-gemedeira", "juriti-grande",
"juriti-pupu", "juriti-roxa", "juriti-verdadeira",
"juriti-vermelha", "juritis-azuis", "juritis-carregadeiras",
"juritis-gemedeiras", "juritis-grandes", "juritis-pupu",
"juritis-roxas", "juritis-verdadeiras", "juritis-vermelhas",
"juruti-azul", "juruti-pequena", "juruti-verdadeira",
"juruti-vermelha", "jurutis-azuis", "jurutis-pequenas",
"jurutis-verdadeiras", "jurutis-vermelhas", "justi"+u"\u00E7"+"a-maior",
"justi"+u"\u00E7"+"as-maiores", "juta-azul", "juta-indiana",
"juta-marrom", "juta-nacional", "juta-paulista",
"juta"+u"\u00ED"+"-caf"+u"\u00E9", "juta"+u"\u00ED"+"-catinga", "juta"+u"\u00ED"+"-grande",
"juta"+u"\u00ED"+"-mirim", "juta"+u"\u00ED"+"-pequeno", "juta"+u"\u00ED"+"-pororoca",
"juta"+u"\u00ED"+"-roxo", "juta"+u"\u00ED"+"s-grandes", "juta"+u"\u00ED"+"s-mirins",
"juta"+u"\u00ED"+"s-pequenos", "juta"+u"\u00ED"+"s-roxos", "jutas-azuis",
"jutas-indianas", "jutas-marrons", "jutas-nacionais",
"jutas-paulistas", "kilovolt-ampere", "kilowatt-hora",
"lago-junquense", "lago-pedrense", "lagoa-arrozense",
"Lagoa-Barra", "lagoa-barrense", "lagoa-bonitense",
"lagoa-branquense", "lagoa-canoense", "lagoa-cantense",
"lagoa-capinense", "lagoa-cercadense", "lagoa-clarense",
"lagoa-cruzense", "lagoa-enganense", "lagoa-formosense",
"lagoa-fundense", "lagoa-gatense", "lagoa-grandense",
"lagoa-lapinhense", "lagoa-matense", "lagoa-novense",
"lagoa-patense", "lagoa-pedrense", "lagoa-pratense",
"lagoa-pretense", "lagoa-realense", "lagoa-salgadense",
"lagoa-santense", "lagoa-sequense", "lagoa-sousense",
"lagoa-vermelhense", "lan"+u"\u00E7"+"a-bombas", "lan"+u"\u00E7"+"a-cabos",
"lan"+u"\u00E7"+"a-chamas", "lan"+u"\u00E7"+"a-foguetes", "lan"+u"\u00E7"+"a-gases",
"lan"+u"\u00E7"+"a-granadas", "lan"+u"\u00E7"+"a-minas", "lan"+u"\u00E7"+"a-m"+u"\u00ED"+"sseis",
"lan"+u"\u00E7"+"a-perfume", "lan"+u"\u00E7"+"a-perfumes", "lan"+u"\u00E7"+"a-pratos",
"lan"+u"\u00E7"+"a-roj"+u"\u00F5"+"es", "lan"+u"\u00E7"+"a-sat"+u"\u00E9"+"lites", "lan"+u"\u00E7"+"a-torpedeiro",
"lan"+u"\u00E7"+"a-torpedos",
"latino-americana", "latino-americanas", "latino-americano", "latino-americanos",
"lesa-p"+u"\u00E1"+"tria", "leste-europeu",
"leste-oeste", "Leste-Oeste", "leste-sudeste",
"Leste-Sudeste", "limpa-fossa", "limpa-fossas",
"lingu"+u"\u00ED"+"stico-comunicativa", "lingu"+u"\u00ED"+"stico-comunicativas",
"lingu"+u"\u00ED"+"stico-comunicativo", "lingu"+u"\u00ED"+"stico-comunicativos",
"l"+u"\u00F3"+"gico-formal", "l"+u"\u00F3"+"gico-jur"+u"\u00ED"+"dico", "l"+u"\u00F3"+"gico-matem"+u"\u00E1"+"tico",
"l"+u"\u00F3"+"gico-sem"+u"\u00E2"+"ntico", "lugar-comum", "luso-africano", "luso-americano",
"Luso-Americano", "luso-andaluz", "luso-"+u"\u00E1"+"rabe",
"luso-asiaticismo", "luso-asi"+u"\u00E1"+"tico", "luso-brasileirismo",
"luso-brasileiro", "Luso-Brasileiro", "luso-bras"+u"\u00ED"+"lico",
"luso-brasilidade", "luso-brit"+u"\u00E2"+"nico", "luso-canadiano",
"luso-castelhano", "luso-chin"+u"\u00EA"+"s", "luso-cultura",
"luso-descendente", "luso-espanhol", "luso-fen"+u"\u00ED"+"cio",
"luso-franc"+u"\u00EA"+"s", "luso-galaico", "luso-germ"+u"\u00E2"+"nico",
"luso-hisp"+u"\u00E2"+"nico", "luso-indianismo", "luso-indiano",
"luso-ingl"+u"\u00EA"+"s", "luso-italiano", "luso-japon"+u"\u00EA"+"s",
"luso-tropicalismo", "m"+u"\u00E1"+"-cria"+u"\u00E7"+u"\u00E3"+"o", "m"+u"\u00E1"+"-firma",
"m"+u"\u00E1"+"-l"+u"\u00ED"+"ngua", "m"+u"\u00E3"+"e-p"+u"\u00E1"+"tria", "m"+u"\u00E3"+"es-bentas",
"m"+u"\u00E3"+"es-joanas", "m"+u"\u00E3"+"es-p"+u"\u00E1"+"trias", "marxismo-leninismo",
"marxismos-leninismos", "marxista-leninista", "marxistas-leninistas",
"m"+u"\u00E1"+"s-f"+u"\u00E9"+"s", "m"+u"\u00E1"+"s-l"+u"\u00ED"+"nguas", "m"+u"\u00E9"+"dio-ligeiro",
"m"+u"\u00E9"+"dios-ligeiros", "mega-hertz", "megael"+u"\u00E9"+"ctron-volt",
"megael"+u"\u00E9"+"ctrons-volts", "megael"+u"\u00E9"+"tron-volt", "megael"+u"\u00E9"+"trons-volts",
"megavolt-ampere", "m"+u"\u00EA"+"s-luz", "meses-luz",
"mestre-caetanense", "mestre-cantor", "mestre-capoeira",
"mestre-china", "mestre-cirurgi"+u"\u00E3"+"o", "mestre-cuco",
"mestre-curandeiro", "mestre-domingos", "mestre-empada",
"mestre-escama", "mestre-escola", "mestre-escolado",
"mestre-impressor", "mestre-sala", "mestre-terreiro",
"mestre-vinagre", "mestres-cantores", "mestres-capoeiras",
"mestres-chinas", "mestres-cirurgi"+u"\u00F5"+"es", "mestres-cucas",
"mestres-cucos", "mestres-curandeiros", "mestres-domingos",
"mestres-escolados", "mestres-impressores", "mestres-salas",
"meta-an"+u"\u00E1"+"lise", "meta-aprendizagem", "meta-arenito",
"meta-arquivo", "meta-arter"+u"\u00ED"+"ola", "meta-"+u"\u00E1"+"tomo",
"meta-halloysita", "meta-heinrichita", "meta-heur"+u"\u00ED"+"stico",
"meta-hist"+u"\u00F3"+"ria", "meta-hohmannita", "meta-humano",
"metro-newton", "ministra-assistente", "ministra-candidata",
"ministra-chefe", "Ministra-Chefe", "ministra-relatora",
"ministras-assistentes", "ministras-candidatas", "ministro-assistente",
"ministro-candidato", "ministro-chefe", "Ministro-Chefe",
"ministro-presidente", "ministro-relator", "ministros-assistentes",
"ministros-candidatos", "modo-temporal", "m"+u"\u00F3"+"dulo-fonte",
"m"+u"\u00F3"+"dulo-objeto", "m"+u"\u00F3"+"dulos-fontes", "m"+u"\u00F3"+"dulos-objetos",
"monta-cargas", "monta-correias", "monta-livros",
"montanha-russa", "montanhas-russas", "monte-alegrense",
"monte-altense", "monte-aprazibilense", "monte-aprazivelense",
"monte-branquense", "monte-carmense", "monte-castelense",
"monte-cruzeirense", "monte-morense", "monte-pascoalense",
"monte-pedrense", "monte-realense", "monte-santense",
"monte-verdense", "morto-vivo", "mortos-vivos",
"Mortos-Vivos", "moto-cont"+u"\u00ED"+"nuo", "moto-perp"+u"\u00E9"+"tuo",
"moto-pr"+u"\u00F3"+"prio", "motor-bomba", "Motor-Bomba",
"motor-canh"+u"\u00E3"+"o", "motor-compressor", "motor-foguete",
"motor-gerador", "motor-ocular", "motores-bombas",
"motos-cont"+u"\u00ED"+"nuos", "motos-perp"+u"\u00E9"+"tuos", "motos-pr"+u"\u00F3"+"prios",
"mulher-aranha", "Mulher-Aranha", "mulher-dama",
"mulher-gato", "Mulher-Gato", "mulher-homem",
"mulher-macho", "mulher-objeto", "na"+u"\u00E7"+u"\u00E3"+"o-l"+u"\u00ED"+"der",
"nacional-sindicalismo", "nacional-sindical"+u"\u00ED"+"stico", "nacional-socialismo",
"Nacional-Socialismo", "Nacional-Socialista", "nacional-social"+u"\u00ED"+"stico",
"nave-irm"+u"\u00E3", "nave-m"+u"\u00E3"+"e", "nave-rob"+u"\u00F4",
"naves-irm"+u"\u00E3"+"s", "naves-m"+u"\u00E3"+"es", "naves-rob"+u"\u00F4"+"s",
"nipo-brasileira", "nipo-brasileiras", "nipo-brasileiro", "nipo-brasileiros",
"no-break", "n"+u"\u00F3"+"-cego", "norma-padr"+u"\u00E3"+"o",
"norma-princ"+u"\u00ED"+"pio", "normas-princ"+u"\u00ED"+"pios", "norte-africanismo",
"norte-africano", "Norte-Am"+u"\u00E9"+"rica", "norte-americanismo",
"norte-americanizar", "norte-americano", "Norte-Americano",
"norte-asi"+u"\u00E1"+"tico", "norte-coreano", "norte-cruzeirense",
"norte-diamantense", "norte-europeu", "norte-fluminense",
"Norte-Fluminense", "norte-irland"+u"\u00EA"+"s", "norte-nordeste",
"Norte-Nordeste", "norte-ocidental", "norte-oriental",
"norte-paraisense", "norte-sul", "norte-vietnamita",
"n"+u"\u00F3"+"s-cegos", "not"+u"\u00E1"+"rios-mores", "not"+u"\u00ED"+"cia-crime",
"not"+u"\u00ED"+"cias-crimes", "nova-alegriense", "nova-aliancense",
"nova-almeidense", "nova-ameliense", "nova-americano",
"nova-andradinense", "nova-ara"+u"\u00E7"+"aense", "nova-aurorense",
"nova-bassanense", "nova-belenense", "nova-betaniense",
"nova-brasiliense", "nova-bresciense", "nova-canaanense",
"nova-castilhense", "nova-catuense", "nova-concordiense",
"nova-conquistense", "nova-cruzense", "nova-erexinense",
"nova-esperancense", "nova-estrelense", "nova-fatimense",
"nova-florestense", "nova-friburguense", "nova-granadense",
"nova-ibiaense", "nova-igua"+u"\u00E7"+"uense", "nova-iorquino",
"nova-ipiraense", "nova-itaranense", "nova-lajense",
"nova-laranjeirense", "nova-londrinense", "nova-lusitaniense",
"nova-maripaense", "nova-milanense", "nova-mocajubense",
"nova-olimpiense", "nova-olindense", "nova-orlean"+u"\u00EA"+"s",
"nova-paduense", "nova-palmeirense", "nova-palmense",
"nova-palmirense", "nova-petropolitano", "nova-pontense",
"nova-pratense", "nova-riquezense", "nova-serranense",
"nova-serrinhense", "nova-sintrense", "nova-sourense",
"nova-tebense", "nova-teutoniense", "nova-timboteuense",
"nova-trentino", "nova-trevisense", "nova-veneciano",
"nova-venezense", "nova-xavantinense", "novo-acrense",
"novo-apuaense", "novo-arealense", "novo-cravinhense",
"novo-cruzeirense", "novo-hamburguense", "novo-horizontense",
"novo-horizontino", "novo-nilense", "novo-orientense",
"novo-paradisense", "novo-paraisense", "novo-rico",
"novo-sarandiense", "novo-testamental", "novo-testamentar",
"novo-testament"+u"\u00E1"+"rio", "novo-triunfense", "novos-ricos",
"noz-moscada", "nozes-moscadas", "obra-prima",
"obras-primas", "obsessivo-compulsivo", "on"+u"\u00E7"+"a-parda",
"on"+u"\u00E7"+"a-pintada", "on"+u"\u00E7"+"as-pintadas", "pa"+u"\u00ED"+"ses-irm"+u"\u00E3"+"os",
"pa"+u"\u00ED"+"ses-membros", "pa"+u"\u00ED"+"ses-parceiros", "palavra-chave",
"palavras-cruzadas", "pan-africanismo", "pan-african"+u"\u00ED"+"stico",
"pan-africano", "pan-aglutina"+u"\u00E7"+u"\u00E3"+"o", "pan-aglutinina",
"pan-amaz"+u"\u00F4"+"nia", "pan-amaz"+u"\u00F4"+"nico", "pan-americanismo",
"pan-american"+u"\u00ED"+"stico", "pan-americaniza"+u"\u00E7"+u"\u00E3"+"o", "pan-americano",
"Pan-Americano", "pan-"+u"\u00E1"+"rabe", "pan-ar"+u"\u00E1"+"bico",
"pan-arabismo", "pan-arab"+u"\u00ED"+"stico", "pan-arterite",
"pan-artrite", "pan-artr"+u"\u00ED"+"tico", "pan-asi"+u"\u00E1"+"tico",
"pan-asiatismo", "pan-asiat"+u"\u00ED"+"stico", "pan-astenia",
"pan-ast"+u"\u00EA"+"nico", "pan-atrofia", "pan-atr"+u"\u00F3"+"fico",
"pan-esl"+u"\u00E1"+"vico", "pan-eslavismo", "pan-eslavista",
"pan-eslav"+u"\u00ED"+"stico", "pan-europeu", "pan-harm"+u"\u00F4"+"nico",
"pan-harm"+u"\u00F4"+"nio", "pan-hel"+u"\u00EA"+"nico", "pan-heleno",
"pan-hematopenia", "pan-hematopo"+u"\u00E9"+"tico", "pan-hidr"+u"\u00F4"+"metro",
"pan-hipopituitarismo", "pan-hisp"+u"\u00E2"+"nico", "pan-hispanismo",
"pan-histerectomia", "pan-histerect"+u"\u00F4"+"mico", "pan-ib"+u"\u00E9"+"rico",
"pan-iconografia", "pan-iconogr"+u"\u00E1"+"fico", "pan-idiom"+u"\u00F3"+"rfico",
"pan-i"+u"\u00F4"+"nico", "pan-isl"+u"\u00E2"+"mico", "pan-islamismo",
"pan-islam"+u"\u00ED"+"stico", "pan-islamita", "pan-m"+u"\u00E1"+"gico",
"pan-mastite", "pan-mast"+u"\u00ED"+"tico", "pan-mielofisite",
"pan-m"+u"\u00ED"+"tico", "pan-mixia", "pan-naturalista",
"pan-negrismo", "pan-negritude", "pan-negro",
"pan-oftalmia", "pan-oft"+u"\u00E1"+"lmico", "pan-oftalmite",
"pan-oftalm"+u"\u00ED"+"tico", "pan-"+u"\u00F3"+"ptico", "pan-"+u"\u00F3"+"rg"+u"\u00E3"+"o",
"pan-or"+u"\u00F3"+"grafo", "pan-ortodoxo", "pan-oste"+u"\u00ED"+"te",
"pan-oste"+u"\u00ED"+"tico", "pan-"+u"\u00F3"+"tico", "pantera-negra",
"panteras-negras", "p"+u"\u00E3"+"o-bengala", "p"+u"\u00E3"+"o-duro",
"p"+u"\u00E3"+"o-petr"+u"\u00F3"+"polis", "papa-defunto", "Papa-Defunto",
"papa-l"+u"\u00E9"+"guas", "pap"+u"\u00E9"+"is-cont"+u"\u00ED"+"nuos", "papel-alum"+u"\u00ED"+"nio",
"papel-arroz", "papel-carbono", "papel-celofane",
"papel-chave", "papel-cont"+u"\u00ED"+"nuo", "papel-filtro",
"papel-jornal", "papel-manteiga", "papel-moeda",
"papel-of"+u"\u00ED"+"cio", "papel-pergaminho", "papel-pigmento",
"papel-porcelana", "papel-registro", "papel-rel"+u"\u00E2"+"mpago",
"papel-tabaco", "papel-t"+u"\u00ED"+"tulo", "par"+u"\u00E1"+"-minense",
"partido-alto", "Passa-Quatro", "peixe-agulha",
"peixe-aipim", "peixe-alecrim", "peixe-anjo",
"peixe-aranha", "peixe-armado", "peixe-azeite",
"peixe-barroso", "peixe-beijador", "peixe-beta",
"peixe-boi", "Peixe-Boi", "peixe-boto",
"peixe-branco", "peixe-bravense", "peixe-briga",
"peixe-bruxa", "peixe-cabe"+u"\u00E7"+"udo", "peixe-cabra",
"peixe-cachimbo", "peixe-cachorro", "peixe-cadela",
"peixe-cana", "peixe-canga", "peixe-c"+u"\u00E3"+"o",
"peixe-capim", "peixe-carago", "peixe-carlim",
"peixe-carta", "peixe-cavalo", "peixe-charuto",
"peixe-chato", "peixe-cigarra", "peixe-cobra",
"peixe-cobrelo", "peixe-coelho", "peixe-coiro",
"peixe-comum", "peixe-congo", "peixe-corcunda",
"peixe-corneta", "peixe-corno", "peixe-cornudo",
"peixe-correio", "peixe-couro", "peixe-cravo",
"peixe-curvo", "peixe-diabo", "peixe-disco",
"peixe-doirado", "peixe-dourado", "peixe-doutor",
"peixe-el"+u"\u00E1"+"stico", "peixe-elefante", "peixe-el"+u"\u00E9"+"trico",
"peixe-escama", "peixe-escolar", "peixe-espada",
"peixe-espinho", "peixe-fila", "peixe-flor",
"peixe-folha", "peixe-frade", "peixe-frito",
"peixe-galo", "peixe-gato", "peixe-homem",
"peixe-japon"+u"\u00EA"+"s", "peixe-judeu", "peixe-lagarto",
"peixe-le"+u"\u00E3"+"o", "peixe-lenha", "peixe-leque",
"peixe-lima", "peixe-lixa", "peixe-lobo",
"peixe-lua", "peixe-macaco", "peixe-machado",
"peixe-madama", "peixe-martelo", "peixe-moela",
"peixe-morcego", "peixe-mulher", "peixe-naire",
"peixe-negro", "peixe-palha"+u"\u00E7"+"o", "peixe-palmito",
"peixe-para"+u"\u00ED"+"so", "peixe-pau", "peixe-pedra",
"peixe-pegador", "peixe-pena", "peixe-p"+u"\u00E9"+"rsico",
"peixe-pica", "peixe-pi"+u"\u00E7"+"a", "peixe-piloto",
"peixe-pimenta", "peixe-piolho", "peixe-pombo",
"peixe-porco", "peixe-prata", "peixe-prego",
"peixe-preto", "peixe-rabo", "peixe-raposo",
"peixe-rato", "peixe-rei", "peixe-roda",
"peixe-sab"+u"\u00E3"+"o", "peixe-sapo", "peixe-serra",
"peixe-sol", "peixe-soldado", "peixe-sono",
"peixe-tigre", "peixe-tordo", "peixe-vaca",
"peixe-vampiro", "peixe-vela", "peixe-ventosa",
"peixe-verde", "peixe-voador", "peixe-zebra",
"peixe-zorro", "personagem-t"+u"\u00ED"+"tulo", "peso-galo",
"peso-leve", "peso-mosca", "peso-pena",
"peso-pesado", "peso-pluma", "pH-metro",
"pica-pau", "placa-m"+u"\u00E3"+"e", "pobre-diabo",
"pobres-diabos", "pol"+u"\u00ED"+"tico-administrativo", "pol"+u"\u00ED"+"tico-criminal",
"pol"+u"\u00ED"+"tico-cultural", "pol"+u"\u00ED"+"tico-decis"+u"\u00F3"+"rio", "pol"+u"\u00ED"+"tico-democr"+u"\u00E1"+"tico",
"pol"+u"\u00ED"+"tico-diplom"+u"\u00E1"+"tico", "pol"+u"\u00ED"+"tico-econ"+u"\u00F4"+"mico", "pol"+u"\u00ED"+"tico-eleitoral",
"pol"+u"\u00ED"+"tico-financeiro", "pol"+u"\u00ED"+"tico-ideol"+u"\u00F3"+"gico", "pol"+u"\u00ED"+"tico-institucional",
"Pol"+u"\u00ED"+"tico-Institucional", "pol"+u"\u00ED"+"tico-jur"+u"\u00ED"+"dico", "pol"+u"\u00ED"+"tico-militar",
"pol"+u"\u00ED"+"tico-partid"+u"\u00E1"+"rio", "pol"+u"\u00ED"+"tico-pedag"+u"\u00F3"+"gico", "pol"+u"\u00ED"+"tico-policial",
"pol"+u"\u00ED"+"tico-regulat"+u"\u00F3"+"rio", "pol"+u"\u00ED"+"tico-religioso", "pol"+u"\u00ED"+"tico-sindical",
"pol"+u"\u00ED"+"tico-social", "porta-acumulador", "porta-"+u"\u00E1"+"guia",
"porta-agulhas", "porta-algod"+u"\u00E3"+"o", "porta-aljava",
"porta-am"+u"\u00E1"+"lgama", "porta-arcabuz", "porta-avi"+u"\u00F5"+"es",
"porta-bagagem", "porta-balaio", "porta-bandeira",
"porta-baquetas", "porta-batel", "porta-bibel"+u"\u00F4"+"s",
"porta-bituca", "porta-bombas", "porta-broca",
"porta-cabos", "porta-cachimbos", "porta-caixa",
"porta-canecos", "porta-canetas", "porta-canh"+u"\u00E3"+"o",
"porta-capote", "porta-card"+u"\u00E1"+"pio", "porta-cartas",
"porta-cart"+u"\u00F5"+"es", "porta-c"+u"\u00E1"+"ustico", "porta-caut"+u"\u00E9"+"rio",
"porta-chapas", "porta-chap"+u"\u00E9"+"us", "porta-chaves",
"porta-cigarros", "porta-cilindros", "porta-clava",
"porta-clavina", "porta-cocheira", "porta-colo",
"porta-contentores", "porta-copos", "porta-cossinete",
"porta-cruz", "porta-discos", "porta-documentos",
"porta-electr"+u"\u00F3"+"dio", "porta-emendas", "porta-enxerto",
"porta-escovas", "porta-espa"+u"\u00E7"+"os", "porta-espada",
"porta-estandarte", "porta-ferramenta", "porta-ferro",
"porta-fios", "porta-fl"+u"\u00E2"+"mula", "porta-flores",
"porta-fogo", "porta-f"+u"\u00F3"+"lio", "porta-frasco",
"porta-fresa", "porta-fronha", "porta-fus"+u"\u00ED"+"vel",
"porta-garrafas", "porta-guardanapos", "porta-gui"+u"\u00E3"+"o",
"porta-helic"+u"\u00F3"+"pteros", "porta-hipofis"+u"\u00E1"+"rio", "porta-hip"+u"\u00F3"+"fise",
"porta-ins"+u"\u00ED"+"gnia", "porta-isca", "porta-janela",
"porta-joias", "porta-jornais", "porta-j"+u"\u00FA"+"bilos",
"porta-la"+u"\u00E7"+"os", "porta-l"+u"\u00E2"+"mpadas", "porta-lan"+u"\u00E7"+"a",
"porta-lanterna", "porta-l"+u"\u00E1"+"pis", "porta-len"+u"\u00E7"+"os",
"porta-leque", "porta-lira", "porta-livros",
"porta-luvas", "porta-luz", "porta-ma"+u"\u00E7"+"a",
"porta-machado", "porta-malas", "porta-manta",
"porta-mant"+u"\u00E9"+"us", "porta-marmita", "porta-mecha",
"porta-microfone", "porta-mira", "porta-misto",
"porta-mitra", "porta-moedas", "porta-morr"+u"\u00E3"+"o",
"porta-negativo", "porta-nitrato", "porta-notas",
"porta-novas", "porta-objeto", "porta-ordens",
"porta-original", "porta-ovos", "porta-p"+u"\u00E1"+"gina",
"porta-papel", "porta-paqu"+u"\u00EA", "porta-pastas",
"porta-paz", "porta-pedra", "porta-penas",
"porta-peso", "porta-pneum"+u"\u00E1"+"tico", "porta-pontas",
"porta-proj"+u"\u00E9"+"teis", "porta-rama", "porta-recado",
"porta-rede", "porta-rel"+u"\u00F3"+"gio", "porta-resist"+u"\u00EA"+"ncia",
"porta-ret"+u"\u00ED"+"cula", "porta-retratos", "porta-revistas",
"porta-roda", "porta-rolos", "porta-sabre",
"porta-saco", "porta-seios", "porta-sela",
"porta-sementes", "porta-serra", "porta-sonda",
"porta-talas", "porta-talheres", "porta-tenaz",
"porta-tirso", "porta-toalhas", "porta-torpedos",
"porta-treco", "porta-v"+u"\u00E1"+"lvula", "porta-vento",
"porta-voz", "Porta-Voz", "porto-abrigo",
"porto-acrense", "porto-acriano", "porto-alegrense",
"Porto-Alegrense", "porto-amazonense", "porto-andiraense",
"porto-biliar", "porto-brasiliense", "porto-camarguense",
"porto-estrelense", "porto-felicense", "porto-ferreirense",
"porto-firmense", "porto-florense", "porto-franco",
"porto-franquense", "porto-grandense", "porto-lucenense",
"porto-martinense", "porto-mauaense", "porto-mendense",
"porto-monicense", "porto-murtinhense", "porto-novense",
"porto-pedrense", "porto-realense", "porto-riquenho",
"porto-salvense", "porto-santanense", "porto-santense",
"porto-segurense", "porto-unionense", "porto-vitoriense",
"porto-xavierense", "posi"+u"\u00E7"+u"\u00E3"+"o-chave", "pouca-vergonha",
"pouco-caso", "pouso-alegrense", "pouso-altense",
"pouso-campense", "pouso-grandense", "pouso-novense",
"pouso-pindaibense", "pouso-redondense", "praia-mar",
"prainha-andradense", "pr"+u"\u00E1"+"tico-profissional", "pr"+u"\u00E9"+"-aborto",
"pr"+u"\u00E9"+"-acordo", "pr"+u"\u00E9"+"-Acordo", "pr"+u"\u00E9"+"-adamita",
"pr"+u"\u00E9"+"-admissional", "pr"+u"\u00E9"+"-ajustado", "pr"+u"\u00E9"+"-alfab"+u"\u00E9"+"tico",
"Pr"+u"\u00E9"+"-Amaz"+u"\u00F4"+"nia", "pr"+u"\u00E9"+"-amplificador", "pr"+u"\u00E9"+"-anal"+u"\u00ED"+"tico",
"pr"+u"\u00E9"+"-antepen"+u"\u00FA"+"ltimo", "pr"+u"\u00E9"+"-anunciador", "pr"+u"\u00E9"+"-apical",
"pr"+u"\u00E9"+"-arr"+u"\u00E1"+"bido", "pr"+u"\u00E9"+"-atendimento", "Pr"+u"\u00E9"+"-Atendimento",
"pr"+u"\u00E9"+"-banho", "pr"+u"\u00E9"+"-biol"+u"\u00F3"+"gico", "pr"+u"\u00E9"+"-bi"+u"\u00F4"+"nico",
"pr"+u"\u00E9"+"-cadastramento", "pr"+u"\u00E9"+"-cadastro", "pr"+u"\u00E9"+"-c"+u"\u00E2"+"mara",
"pr"+u"\u00E9"+"-candidato", "pr"+u"\u00E9"+"-candidatura", "pr"+u"\u00E9"+"-capitalismo",
"pr"+u"\u00E9"+"-carga", "pr"+u"\u00E9"+"-censura", "pr"+u"\u00E9"+"-cer"+u"\u00E2"+"mico",
"pr"+u"\u00E9"+"-citado", "pr"+u"\u00E9"+"-civilizado", "Pr"+u"\u00E9"+"-Classifica"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00E9"+"-clora"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-coagulado", "pr"+u"\u00E9"+"-codificado",
"pr"+u"\u00E9"+"-coito", "pr"+u"\u00E9"+"-coma", "pr"+u"\u00E9"+"-combinado",
"pr"+u"\u00E9"+"-comercial", "pr"+u"\u00E9"+"-competitivo", "pr"+u"\u00E9"+"-comprado",
"pr"+u"\u00E9"+"-compromissado", "pr"+u"\u00E9"+"-compromisso", "pr"+u"\u00E9"+"-condi"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00E9"+"-conflito", "pr"+u"\u00E9"+"-constitu"+u"\u00ED"+"do", "pr"+u"\u00E9"+"-contratado",
"pr"+u"\u00E9"+"-contratante", "pr"+u"\u00E9"+"-contrato", "pr"+u"\u00E9"+"-conven"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00E9"+"-Copa", "Pr"+u"\u00E9"+"-Copa", "pr"+u"\u00E9"+"-cordilheira",
"pr"+u"\u00E9"+"-cozido", "pr"+u"\u00E9"+"-cozimento", "pr"+u"\u00E9"+"-cozinhado",
"pr"+u"\u00E9"+"-cr"+u"\u00E9"+"dito", "pr"+u"\u00E9"+"-crise", "pr"+u"\u00E9"+"-cronograma",
"pr"+u"\u00E9"+"-curvamento", "pr"+u"\u00E9"+"-datado", "pr"+u"\u00E9"+"-definido",
"pr"+u"\u00E9"+"-demarcado", "pr"+u"\u00E9"+"-demarcador", "pr"+u"\u00E9"+"-designado",
"pr"+u"\u00E9"+"-determinado", "pr"+u"\u00E9"+"-diabetes", "pr"+u"\u00E9"+"-di"+u"\u00E1"+"lise",
"pr"+u"\u00E9"+"-digerido", "pr"+u"\u00E9"+"-eliminat"+u"\u00F3"+"rio", "pr"+u"\u00E9"+"-emiss"+u"\u00E3"+"o",
"pr"+u"\u00E9"+"-empenho", "pr"+u"\u00E9"+"-emprego", "pr"+u"\u00E9"+"-encolhido",
"Pr"+u"\u00E9"+"-Enem", "pr"+u"\u00E9"+"-equipado", "pr"+u"\u00E9"+"-esclerose",
"pr"+u"\u00E9"+"-escola", "Pr"+u"\u00E9"+"-Escola", "Pr"+u"\u00E9"+"-Escolar",
"pr"+u"\u00E9"+"-escolaridade", "pr"+u"\u00E9"+"-escolhido", "pr"+u"\u00E9"+"-esfor"+u"\u00E7"+"ado",
"pr"+u"\u00E9"+"-esfor"+u"\u00E7"+"o", "pr"+u"\u00E9"+"-estreia", "pr"+u"\u00E9"+"-executividade",
"pr"+u"\u00E9"+"-exposi"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-fabricado", "pr"+u"\u00E9"+"-fala",
"pr"+u"\u00E9"+"-falha", "pr"+u"\u00E9"+"-faturamento", "pr"+u"\u00E9"+"-filtro",
"pr"+u"\u00E9"+"-financiamento", "pr"+u"\u00E9"+"-fissuramento", "pr"+u"\u00E9"+"-fixado",
"pr"+u"\u00E9"+"-folha", "pr"+u"\u00E9"+"-formado", "pr"+u"\u00E9"+"-formante",
"pr"+u"\u00E9"+"-formula"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-franqueado", "pr"+u"\u00E9"+"-gabaritado",
"pr"+u"\u00E9"+"-gravado", "pr"+u"\u00E9"+"-helenismo", "pr"+u"\u00E9"+"-hidratado",
"pr"+u"\u00E9"+"-hisp"+u"\u00E2"+"nico", "Pr"+u"\u00E9"+"-Hist"+u"\u00F3"+"ria", "pr"+u"\u00E9"+"-historiador",
"Pr"+u"\u00E9"+"-Hist"+u"\u00F3"+"rico", "Pr"+u"\u00E9"+"-homogeneiza"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-iluminar",
"pr"+u"\u00E9"+"-impresso", "pr"+u"\u00E9"+"-inaugural", "pr"+u"\u00E9"+"-inca",
"pr"+u"\u00E9"+"-independente", "pr"+u"\u00E9"+"-industrial", "pr"+u"\u00E9"+"-inicia"+u"\u00E7"+u"\u00E3"+"o",
"Pr"+u"\u00E9"+"-Inicia"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-insular", "pr"+u"\u00E9"+"-jantar",
"pr"+u"\u00E9"+"-lan"+u"\u00E7"+"amento", "pr"+u"\u00E9"+"-letrado", "pr"+u"\u00E9"+"-levantamento",
"pr"+u"\u00E9"+"-ligado", "pr"+u"\u00E9"+"-limitado", "pr"+u"\u00E9"+"-lista",
"pr"+u"\u00E9"+"-logismo", "pr"+u"\u00E9"+"-maligno",
"pr"+u"\u00E9"+"-marxista", "pr"+u"\u00E9"+"-matrimonial", "Pr"+u"\u00E9"+"-Menstrual",
"pr"+u"\u00E9"+"-milenarismo", "pr"+u"\u00E9"+"-misturado", "Pr"+u"\u00E9"+"-Misturado",
"pr"+u"\u00E9"+"-modernismo", "pr"+u"\u00E9"+"-modernista", "pr"+u"\u00E9"+"-moderno",
"pr"+u"\u00E9"+"-moldado", "pr"+u"\u00E9"+"-montado", "pr"+u"\u00E9"+"-montagem",
"pr"+u"\u00E9"+"-motor", "pr"+u"\u00E9"+"-nasalizado", "pr"+u"\u00E9"+"-nupciais",
"pr"+u"\u00E9"+"-obeso", "pr"+u"\u00E9"+"-objetais", "pr"+u"\u00E9"+"-obliterado",
"pr"+u"\u00E9"+"-ocupado", "pr"+u"\u00E9"+"-ocupante", "pr"+u"\u00E9"+"-opera"+u"\u00E7"+u"\u00E3"+"o",
"Pr"+u"\u00E9"+"-Opera"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-operacional", "pr"+u"\u00E9"+"-operar",
"pr"+u"\u00E9"+"-oral", "pr"+u"\u00E9"+"-oxidante", "pr"+u"\u00E9"+"-pagamento",
"pr"+u"\u00E9"+"-pago", "pr"+u"\u00E9"+"-parametrizado", "pr"+u"\u00E9"+"-prensado",
"pr"+u"\u00E9"+"-prensador", "pr"+u"\u00E9"+"-prensamento", "pr"+u"\u00E9"+"-prensante",
"pr"+u"\u00E9"+"-prim"+u"\u00E1"+"rio", "pr"+u"\u00E9"+"-processamento", "pr"+u"\u00E9"+"-produ"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00E9"+"-projeto", "pr"+u"\u00E9"+"-proparox"+u"\u00ED"+"tono", "pr"+u"\u00E9"+"-pupa",
"pr"+u"\u00E9"+"-queixa", "pr"+u"\u00E9"+"-questionamento", "pr"+u"\u00E9"+"-qu"+u"\u00ED"+"mico",
"pr"+u"\u00E9"+"-rafaelita", "pr"+u"\u00E9"+"-reflexivo", "pr"+u"\u00E9"+"-reforma",
"pr"+u"\u00E9"+"-reformista", "pr"+u"\u00E9"+"-refrigerador", "pr"+u"\u00E9"+"-registo",
"pr"+u"\u00E9"+"-republicano", "pr"+u"\u00E9"+"-requisitado", "pr"+u"\u00E9"+"-requisito",
"Pr"+u"\u00E9"+"-Requisito", "pr"+u"\u00E9"+"-retal", "pr"+u"\u00E9"+"-r"+u"\u00E9"+"veillon",
"pr"+u"\u00E9"+"-R"+u"\u00E9"+"veillon", "pr"+u"\u00E9"+"-revolu"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00E9"+"-Revolu"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00E9"+"-sal", "Pr"+u"\u00E9"+"-Sal", "pr"+u"\u00E9"+"-salto",
"pr"+u"\u00E9"+"-santificado", "pr"+u"\u00E9"+"-senilidade", "pr"+u"\u00E9"+"-sensibilizado",
"pr"+u"\u00E9"+"-simbolismo", "pr"+u"\u00E9"+"-sin"+u"\u00E1"+"ptico", "pr"+u"\u00E9"+"-sincronizado",
"pr"+u"\u00E9"+"-social", "pr"+u"\u00E9"+"-socialismo", "Pr"+u"\u00E9"+"-Socr"+u"\u00E1"+"tico",
"pr"+u"\u00E9"+"-terminais", "pr"+u"\u00E9"+"-termo", "pr"+u"\u00E9"+"-teste",
"pr"+u"\u00E9"+"-tiroideu", "pr"+u"\u00E9"+"-t"+u"\u00F4"+"nico", "pr"+u"\u00E9"+"-tra"+u"\u00E7"+"ado",
"pr"+u"\u00E9"+"-tratado", "pr"+u"\u00E9"+"-tratamento", "Pr"+u"\u00E9"+"-Tratamento",
"pr"+u"\u00E9"+"-tumoral", "pr"+u"\u00E9"+"-v"+u"\u00E1"+"cuo", "pr"+u"\u00E9"+"-venda",
"pr"+u"\u00E9"+"-vesical", "Pr"+u"\u00E9"+"-Vestibular", "pr"+u"\u00E9"+"-visualiza"+u"\u00E7"+u"\u00E3"+"o",
"preia-mar", "prensa-pasta", "prep"+u"\u00F3"+"sito-geral",
"prep"+u"\u00F3"+"sitos-gerais", "presb"+u"\u00ED"+"tero-assistente", "presb"+u"\u00ED"+"teros-assistentes",
"presidente-alvense", "presidente-bernardense", "presidente-dutrense",
"presidente-epitaciense", "presidente-executivo", "presidente-getuliense",
"presidente-juscelinense", "presidente-olegariense", "presidente-prudentense",
"presidente-soarense", "presidente-venceslauense", "presidentes-executivos",
"prime-time", "primeira-dama", "Primeira-Dama",
"primeira-ministra", "Primeira-Ministra", "primeira-secret"+u"\u00E1"+"ria",
"primeiras-damas", "primeiro-cabo", "primeiro-cadete",
"primeiro-ministro", "Primeiro-Ministro", "primeiro-sargento",
"primeiro-secret"+u"\u00E1"+"rio", "primeiro-tenente", "primeiros-cabos",
"primeiros-cadetes", "primeiros-ministros", "primeiros-sargentos",
"primeiros-tenentes", "primo-division"+u"\u00E1"+"rio", "primo-infec"+u"\u00E7"+u"\u00E3"+"o",
"primo-infeccioso", "primo-irm"+u"\u00E3"+"o", "primo-vacina"+u"\u00E7"+u"\u00E3"+"o",
"primo-vacinar", "primo-vacinat"+u"\u00F3"+"rio", "primo-vacin"+u"\u00E1"+"vel",
"primos-irm"+u"\u00E3"+"os", "pris"+u"\u00E3"+"o-albergue", "prisma-objetiva",
"prismas-objetivas", "pr"+u"\u00F3"+"-aborto", "pr"+u"\u00F3"+"-academia",
"Pr"+u"\u00F3"+"-Academia", "pr"+u"\u00F3"+"-AGLP", "Pr"+u"\u00F3"+"-AGLP",
"pr"+u"\u00F3"+"-americano", "pr"+u"\u00F3"+"-an"+u"\u00E1"+"lise", "pr"+u"\u00F3"+"-anorexia",
"pr"+u"\u00F3"+"-arte", "pr"+u"\u00F3"+"-ativo", "pr"+u"\u00F3"+"-baleia",
"pr"+u"\u00F3"+"-brit"+u"\u00E2"+"nico", "pr"+u"\u00F3"+"-crian"+u"\u00E7"+"a", "pr"+u"\u00F3"+"-cubano",
"pr"+u"\u00F3"+"-democracia", "pr"+u"\u00F3"+"-desarmamento", "pr"+u"\u00F3"+"-descobrimento",
"pr"+u"\u00F3"+"-detonante", "pr"+u"\u00F3"+"-efici"+u"\u00EA"+"ncia", "pr"+u"\u00F3"+"-eficiente",
"pr"+u"\u00F3"+"-escolha", "pr"+u"\u00F3"+"-estatista", "pr"+u"\u00F3"+"-europeu",
"pr"+u"\u00F3"+"-expurgo", "pr"+u"\u00F3"+"-extradi"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00F3"+"-fam"+u"\u00ED"+"lia",
"pr"+u"\u00F3"+"-forma", "pr"+u"\u00F3"+"-gen"+u"\u00E9"+"tico", "pr"+u"\u00F3"+"-germ"+u"\u00E2"+"nico",
"pr"+u"\u00F3"+"-germanismo", "pr"+u"\u00F3"+"-german"+u"\u00ED"+"stico", "pr"+u"\u00F3"+"-governamental",
"pr"+u"\u00F3"+"-governo", "pr"+u"\u00F3"+"-hidrotropia", "pr"+u"\u00F3"+"-hidrotr"+u"\u00F3"+"pico",
"pr"+u"\u00F3"+"-hidrotropismo", "pr"+u"\u00F3"+"-homem", "pr"+u"\u00F3"+"-hom"+u"\u00ED"+"nida",
"pr"+u"\u00F3"+"-homin"+u"\u00ED"+"deo", "pr"+u"\u00F3"+"-imagem", "pr"+u"\u00F3"+"-inclus"+u"\u00E3"+"o",
"pr"+u"\u00F3"+"-independ"+u"\u00EA"+"ncia", "pr"+u"\u00F3"+"-inf"+u"\u00E2"+"ncia", "pr"+u"\u00F3"+"-israelita",
"pr"+u"\u00F3"+"-labore", "pr"+u"\u00F3"+"-mata", "pr"+u"\u00F3"+"-mem"+u"\u00F3"+"ria",
"pr"+u"\u00F3"+"-mulher", "pr"+u"\u00F3"+"-ocidental", "pr"+u"\u00F3"+"-oposi"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00F3"+"-oriental", "pr"+u"\u00F3"+"-reforma", "Pr"+u"\u00F3"+"-Reitor",
"pr"+u"\u00F3"+"-reitor", "pr"+u"\u00F3"+"-reitoria", "Pr"+u"\u00F3"+"-Reitoria",
"pr"+u"\u00F3"+"-republicano", "pr"+u"\u00F3"+"-revolu"+u"\u00E7"+u"\u00E3"+"o", "pr"+u"\u00F3"+"-Revolu"+u"\u00E7"+u"\u00E3"+"o",
"pr"+u"\u00F3"+"-russo", "pr"+u"\u00F3"+"-sa"+u"\u00FA"+"de", "pr"+u"\u00F3"+"-serenismo",
"pr"+u"\u00F3"+"-s"+u"\u00ED"+"rio", "pr"+u"\u00F3"+"-social", "pr"+u"\u00F3"+"-socialista",
"pr"+u"\u00F3"+"-sovi"+u"\u00E9"+"tico", "pr"+u"\u00F3"+"-universalismo", "pr"+u"\u00F3"+"-vegetarianismo",
"pr"+u"\u00F3"+"-vida", "processo-chave", "processo-crime",
"procurador-chefe", "Procurador-Chefe", "procurador-geral",
"Procurador-Geral", "procuradora-chefe", "Procuradora-Chefe",
"procuradora-geral", "Procuradora-Geral", "procuradores-gerais",
"procuradoria-geral", "Procuradoria-Geral", "procuradorias-gerais",
"produ"+u"\u00E7"+u"\u00E3"+"o-executiva", "produ"+u"\u00E7"+u"\u00E3"+"o-fam"+u"\u00ED"+"lia", "produtor-chefe",
"produtor-executivo", "produtora-chefe", "produtora-executiva",
"produtoras-chefes", "produtoras-executivas", "produtores-chefes",
"produtores-executivos", "professor-aluno", "professor-assistente",
"professor-associado", "professor-fundador", "professor-orientador",
"professor-praticante", "professor-tutor", "professora-adjunta",
"professora-assistente", "professora-associada", "professora-fundadora",
"professora-orientadora", "professora-praticante", "professora-tutora",
"professoras-adjuntas", "professoras-assistentes", "professoras-associadas",
"professoras-fundadoras", "professoras-orientadoras", "professoras-praticantes",
"professoras-tutoras", "professores-adjuntos", "professores-assistentes",
"professores-associados", "professores-fundadores", "professores-orientadores",
"professores-praticantes", "professores-tutores", "profissional-padr"+u"\u00E3"+"o",
"programa-fonte", "programa-objeto", "projeto-lei",
"projeto-piloto", "proto-hematoblasto", "proto-herdeiro",
"proto-hist"+u"\u00F3"+"ria", "proto-historiador", "proto-hist"+u"\u00F3"+"rico",
"proto-hitita", "proto-homem", "proto-homin"+u"\u00ED"+"deo",
"proto-homo", "proto-h"+u"\u00F3"+"rtico", "p"+u"\u00FA"+"blico-alvo",
"quadro-alvo", "quadro-negro", "quadros-negros",
"quartas-feiras", "Quartas-Feiras", "quart"+u"\u00E9"+"is-generais",
"quart"+u"\u00E9"+"is-mestres", "quartel-general", "quartel-mestre",
"que-diga", "que-farte", "quebra-anzol",
"quebra-arado", "quebra-bumba", "quebra-bunda",
"quebra-cabe"+u"\u00E7"+"a", "Quebra-Cabe"+u"\u00E7"+"a", "quebra-cacos",
"quebra-cana", "quebra-cangalhas", "Quebra-Carga",
"quebra-cavaco", "quebra-chamas", "quebra-chifre",
"quebra-chiquinha", "quebra-costas", "Quebra-Costas",
"quebra-costela", "quebra-dedos", "quebra-dentes",
"Quebra-Dentes", "quebra-engui"+u"\u00E7"+"o", "quebra-esquinas",
"quebra-fac"+u"\u00E3"+"o", "quebra-febre", "quebra-fogo",
"quebra-foice", "quebra-fouce", "quebra-frasco",
"quebra-freio", "quebra-galho", "quebra-gelo",
"quebra-goela", "quebra-jejum", "quebra-lan"+u"\u00E7"+"as",
"quebra-largado", "quebra-lou"+u"\u00E7"+"as", "quebra-luz",
"quebra-machado", "quebra-mar", "quebra-molas",
"quebra-munheca", "quebra-nozes", "Quebra-Nozes",
"quebra-panela", "quebra-pau", "quebra-pedra",
"quebra-pote", "quebra-queixo", "quebra-quilos",
"quebra-rabicho", "quebra-resguardo", "quebra-urnas",
"quebra-ventos", "quebra-verso", "quebra-vista",
"queixa-crime", "queixas-crime", "queixas-crimes",
"quilo-henry", "quilo-hertz", "quilo-ohm",
"quiloel"+u"\u00E9"+"tron-volt", "quiloel"+u"\u00E9"+"trons-volt", "quilo"+u"\u00E9"+"letrons-volts",
"quilograma-for"+u"\u00E7"+"a", "quilograma-metro", "quilograma-padr"+u"\u00E3"+"o",
"quilograma-peso", "quil"+u"\u00F4"+"metro-passageiro", "quilovar-hora",
"quilov"+u"\u00E1"+"tio-hora", "quilovolt-ampere", "quilovolt-amp\u00E8re",
"quilovolts-amperes", "quilowatt-hora", "qu"+u"\u00ED"+"mico-analista",
"qu"+u"\u00ED"+"mico-anal"+u"\u00ED"+"tico", "qu"+u"\u00ED"+"mico-bacteriol"+u"\u00F3"+"gico", "qu"+u"\u00ED"+"mico-bacteriologista",
"qu"+u"\u00ED"+"mico-biol"+u"\u00F3"+"gico", "qu"+u"\u00ED"+"mico-bromatol"+u"\u00F3"+"gico", "qu"+u"\u00ED"+"mico-farmac"+u"\u00EA"+"utico",
"qu"+u"\u00ED"+"mico-farmacol"+u"\u00F3"+"gico", "qu"+u"\u00ED"+"mico-f"+u"\u00ED"+"sico", "qu"+u"\u00ED"+"mico-fisiol"+u"\u00F3"+"gico",
"qu"+u"\u00ED"+"mico-laboratorial", "qu"+u"\u00ED"+"mico-legal", "quintas-feiras",
"Quintas-Feiras", "rainha-isabelense", "rainha-m"+u"\u00E3"+"e",
"rainha-margarida", "raio-trator", "raio-X",
"Raio-X", "raios-X", "Raios-X",
"rapa-tachos", "rebenta-boi", "rec"+u"\u00E9"+"m-aberto",
"rec"+u"\u00E9"+"m-assinante", "rec"+u"\u00E9"+"m-beato", "rec"+u"\u00E9"+"m-chegar",
"rec"+u"\u00E9"+"m-ciclista", "rec"+u"\u00E9"+"m-doutor", "rec"+u"\u00E9"+"m-eleito",
"rec"+u"\u00E9"+"m-entregue", "rec"+u"\u00E9"+"m-feito", "rec"+u"\u00E9"+"m-ferir",
"rec"+u"\u00E9"+"m-findo", "rec"+u"\u00E9"+"m-morto", "rec"+u"\u00E9"+"m-motorista",
"rec"+u"\u00E9"+"m-namoro", "rec"+u"\u00E9"+"m-nato", "rec"+u"\u00E9"+"m-nobre",
"rec"+u"\u00E9"+"m-nomea"+u"\u00E7"+u"\u00E3"+"o", "rec"+u"\u00E9"+"m-piloto", "rec"+u"\u00E9"+"m-poeta",
"rec"+u"\u00E9"+"m-sa"+u"\u00ED"+"do", "rec"+u"\u00E9"+"m-servo", "rec"+u"\u00E9"+"m-vencedor",
"rec"+u"\u00E9"+"m-vigente", "recepcionista-chefe", "recepcionistas-chefes",
"redator-chefe", "redatora-chefe", "relacionamento-rel"+u"\u00E2"+"mpago",
"rela"+u"\u00E7"+u"\u00F5"+"es-p"+u"\u00FA"+"blicas", "relator-geral", "relatora-geral",
"relatoras-gerais", "relatores-gerais", "religioso-cient"+u"\u00ED"+"fico",
"restritivo-objetiva", "reticulado-venoso", "reto-uretral",
"ret"+u"\u00F3"+"rico-discursivo", "retorta-moirisca", "retorta-mourisca",
"retortas-moiriscas", "retortas-mouriscas", "retrato-rob"+u"\u00F4",
"retro-oclus"+u"\u00E3"+"o", "retro-ocular", "retro-operar",
"retro-orbital", "retro-ov"+u"\u00E1"+"rico", "retro-ov"+u"\u00E1"+"rio",
"reuni"+u"\u00E3"+"o-almo"+u"\u00E7"+"o", "revira-olho", "riacho-almense",
"riacho-areense", "riacho-cedrense", "riacho-cruzense",
"riacho-fundense", "riacho-garimpense", "riacho-grandense",
"riacho-machadense", "riacho-matense", "riacho-meense",
"riacho-melense", "riacho-mendense", "riacho-nortense",
"riacho-oncense", "riacho-paulense", "riacho-pedrense",
"riacho-pintense", "riacho-queimadense", "riacho-santanense",
"riacho-sequense", "riacho-sertanense", "riacho-sobradense",
"riacho-verdense", "ribeira-bravense", "ribeira-grandense",
"ribeir"+u"\u00E3"+"o-amarelense", "ribeir"+u"\u00E3"+"o-bonitense", "ribeir"+u"\u00E3"+"o-branquense",
"ribeir"+u"\u00E3"+"o-clarense", "ribeir"+u"\u00E3"+"o-correntense", "ribeir"+u"\u00E3"+"o-correntino",
"ribeir"+u"\u00E3"+"o-cruzeirense", "ribeir"+u"\u00E3"+"o-fundense", "ribeir"+u"\u00E3"+"o-grandense",
"ribeir"+u"\u00E3"+"o-indiense", "ribeir"+u"\u00E3"+"o-maiense", "ribeir"+u"\u00E3"+"o-meense",
"ribeir"+u"\u00E3"+"o-nevense", "ribeir"+u"\u00E3"+"o-pedrense", "ribeir"+u"\u00E3"+"o-pinhalense",
"ribeir"+u"\u00E3"+"o-piresino", "ribeir"+u"\u00E3"+"o-pratense", "ribeir"+u"\u00E3"+"o-pretano",
"ribeir"+u"\u00E3"+"o-pretense", "ribeir"+u"\u00E3"+"o-saltense", "ribeir"+u"\u00E3"+"o-serrense",
"ribeir"+u"\u00E3"+"o-vermelhense", "ribeiro-gon"+u"\u00E7"+"alvense", "ribeiro-gon"+u"\u00E7"+"alvino",
"ribeiro-junqueirense", "ribeiro-pinhalense", "ribeiro-pretano",
"rio-alegrense", "rio-almense", "rio-angelense",
"rio-antense", "rio-antinhense", "rio-areense",
"rio-bananalense", "rio-barrense", "rio-bonense",
"rio-bonitense", "rio-branquense", "rio-brilhantense",
"rio-bugrense", "rio-campense", "rio-carolinense",
"rio-casquense", "rio-cedrense", "rio-clarense",
"rio-concepcionense", "rio-correntense", "rio-esperense",
"rio-florense", "rio-formosense", "rio-fortunense",
"rio-fundense", "rio-furnense", "rio-grandense",
"rio-larguense", "rio-mansense", "rio-mar",
"rio-mariano", "rio-matense", "rio-meense",
"rio-negrense", "rio-novense", "rio-oestense",
"rio-ostrense", "rio-palmense", "rio-paranaibano",
"rio-paranaibense", "rio-pardinhense", "rio-parnaibano",
"rio-pedrense", "rio-peixense", "rio-pinhalense",
"rio-piracicabense", "rio-pirense", "rio-platense",
"rio-pombense", "rio-pousense", "rio-pradense",
"rio-pratense", "rio-pretano", "rio-pretense",
"rio-realense", "rio-salense", "rio-saltense",
"rio-santense", "Rio-Santos", "rio-sulino",
"rio-sulista", "rio-telhense", "rio-unense",
"rio-varzeense", "rio-verdense", "rio-vermelhense",
"romance-rel"+u"\u00E2"+"mpago", "romano-"+u"\u00E1"+"rabe", "romano-bizantino",
"romano-cat"+u"\u00F3"+"lico", "romano-germ"+u"\u00E2"+"nico", "Romano-Germ"+u"\u00E2"+"nico",
"rubro-negro", "Rubro-Negro", "sal"+u"\u00E1"+"rio-base",
"sal"+u"\u00E1"+"rio-fam"+u"\u00ED"+"lia", "Sal"+u"\u00E1"+"rio-Fam"+u"\u00ED"+"lia", "sal"+u"\u00E1"+"rio-hora",
"sal"+u"\u00E1"+"rio-maternidade", "sal"+u"\u00E1"+"rio-m"+u"\u00ED"+"nimo", "sal"+u"\u00E1"+"rios-m"+u"\u00ED"+"nimos",
"salto-mortal", "santa-adelaidense", "santa-albertinense",
"santa-ameliense", "santa-angelicense", "santa-aparecidense",
"santa-barbarense", "santa-branquense", "santa-brigidense",
"santa-catarinense", "santa-clarense", "santa-cristinense",
"santa-cruzense", "santa-emiliense", "santa-ernestinense",
"santa-filomenense", "santa-gertrudense", "santa-helenense",
"santa-inesense", "santa-isabelense", "santa-joanense",
"santa-julianense", "santa-juliense", "santa-leopoldinense",
"santa-lidiense", "santa-luisense", "santa-luziense",
"santa-marcelinense", "santa-margaridense", "santa-marianense",
"santa-mariense", "santa-mercedense", "santa-monicense",
"santa-quiteriense", "santa-saletense", "santa-silvanense",
"santa-silveriense", "santa-teresense", "santa-teresinhense",
"santa-unionense", "santa-virginense", "santa-vitoriense",
"santos-reis", "s"+u"\u00E3"+"o-beneditense", "s"+u"\u00E3"+"o-bernardense",
"s"+u"\u00E3"+"o-bonifaciense", "s"+u"\u00E3"+"o-brasense", "s"+u"\u00E3"+"o-caetanense",
"s"+u"\u00E3"+"o-cristovense", "s"+u"\u00E3"+"o-desideriense", "s"+u"\u00E3"+"o-dominguense",
"s"+u"\u00E3"+"o-felicense", "s"+u"\u00E3"+"o-filipense", "s"+u"\u00E3"+"o-firminense",
"s"+u"\u00E3"+"o-franciscano", "s"+u"\u00E3"+"o-francisquense", "s"+u"\u00E3"+"o-gabrielense",
"s"+u"\u00E3"+"o-gon"+u"\u00E7"+"alense", "s"+u"\u00E3"+"o-gotardense", "s"+u"\u00E3"+"o-gregoriense",
"s"+u"\u00E3"+"o-jeronimense", "s"+u"\u00E3"+"o-joanense", "s"+u"\u00E3"+"o-joaquinense",
"s"+u"\u00E3"+"o-jorgense", "s"+u"\u00E3"+"o-joseense", "s"+u"\u00E3"+"o-julianense",
"s"+u"\u00E3"+"o-leopoldense", "s"+u"\u00E3"+"o-lourencense", "s"+u"\u00E3"+"o-lourenciano",
"s"+u"\u00E3"+"o-luisense", "s"+u"\u00E3"+"o-luquense", "s"+u"\u00E3"+"o-mamedense",
"s"+u"\u00E3"+"o-marcelinense", "s"+u"\u00E3"+"o-marcense", "s"+u"\u00E3"+"o-marinense",
"s"+u"\u00E3"+"o-marquense", "s"+u"\u00E3"+"o-martinhense", "s"+u"\u00E3"+"o-miguelense",
"s"+u"\u00E3"+"o-murtinhense", "s"+u"\u00E3"+"o-nicolauense", "s"+u"\u00E3"+"o-paulense",
"s"+u"\u00E3"+"o-pedrense", "s"+u"\u00E3"+"o-piense", "s"+u"\u00E3"+"o-rafaelense",
"s"+u"\u00E3"+"o-raimundense", "s"+u"\u00E3"+"o-robertense", "s"+u"\u00E3"+"o-roquense",
"s"+u"\u00E3"+"o-salvadorense", "s"+u"\u00E3"+"o-sebastianense", "s"+u"\u00E3"+"o-silvanense",
"s"+u"\u00E3"+"o-silvestrense", "s"+u"\u00E3"+"o-simonense", "s"+u"\u00E3"+"o-tiaguense",
"s"+u"\u00E3"+"o-tirsense", "s"+u"\u00E3"+"o-valentinense", "s"+u"\u00E3"+"o-valeriense",
"s"+u"\u00E3"+"o-vicentense", "sapo-boi", "sapo-cururu",
"sapos-cururus", "sargento-ajudante", "sargento-aspirante",
"sargento-chefe", "sargento-maior", "sargento-mor",
"sargentos-ajudantes", "sargentos-aspirantes", "sargentos-maiores",
"sargentos-mores", "sax-alto", "sax-baixo",
"sax-bar"+u"\u00ED"+"tono", "sax-tenor", "secretaria-executiva",
"Secretaria-Executiva", "secretaria-geral", "Secretaria-Geral",
"secret"+u"\u00E1"+"ria-adjunta", "secret"+u"\u00E1"+"ria-assistente", "secret"+u"\u00E1"+"ria-chefe",
"secret"+u"\u00E1"+"ria-executiva", "secret"+u"\u00E1"+"ria-geral", "Secret"+u"\u00E1"+"ria-Geral",
"secretariado-geral", "secretariados-gerais", "secretarias-gerais",
"Secretarias-Gerais", "secret"+u"\u00E1"+"rias-assistentes", "secret"+u"\u00E1"+"rias-chefes",
"secret"+u"\u00E1"+"rio-assistente", "secret"+u"\u00E1"+"rio-chefe", "secret"+u"\u00E1"+"rio-executivo",
"Secret"+u"\u00E1"+"rio-Executivo", "secret"+u"\u00E1"+"rio-geral", "Secret"+u"\u00E1"+"rio-Geral",
"secret"+u"\u00E1"+"rio-t"+u"\u00E9"+"cnico", "secret"+u"\u00E1"+"rios-assistentes", "secret"+u"\u00E1"+"rios-chefes",
"Secret"+u"\u00E1"+"rios-Executivos", "secret"+u"\u00E1"+"rios-gerais", "segundas-feiras",
"Segundas-Feiras", "seguro-democracia", "seguro-desemprego",
"Seguro-Desemprego", "seguro-garantia", "Seguro-Garantia",
"seguro-maternidade", "seguro-sa"+u"\u00FA"+"de", "sem-terra",
"Sem-Terra", "senhores-zorrilhos", "sensitiva-mansa",
"sensitivas-mansas", "sensitivo-glandular", "sensitivo-motor",
"sensitivo-muscular", "sensitivo-sensorial", "sensitivo-vascular",
"sensorial-motora", "Sens"+u"\u00F3"+"rio-Motor", "sens"+u"\u00F3"+"rio-motor",
"sens"+u"\u00F3"+"rio-motriz", "sens"+u"\u00F3"+"rio-t"+u"\u00E1"+"til", "servi"+u"\u00E7"+"o-chave",
"s"+u"\u00E9"+"rvio-croata", "servo-austr"+u"\u00ED"+"aco", "servo-b"+u"\u00F3"+"snio",
"servo-b"+u"\u00FA"+"lgaro", "servo-cro"+u"\u00E1"+"cio", "servo-croata",
"servo-cro"+u"\u00E1"+"tico", "servo-grego", "servo-h"+u"\u00FA"+"ngaro",
"servo-romeno", "servo-russo", "setor-alvo",
"setor-chave", "sextas-feiras", "sextas-rimas",
"sinh"+u"\u00E1"+"-mo"+u"\u00E7"+"a", "sino-brasileiro", "Sino-Brasileiro",
"sino-canadense", "sino-coreano", "sino-cubano",
"sino-franc"+u"\u00EA"+"s", "sino-hindu", "sino-indiano",
"sino-ingl"+u"\u00EA"+"s", "sino-japon"+u"\u00EA"+"s", "sino-portugu"+u"\u00EA"+"s",
"sino-russo", "sino-salom"+u"\u00E3"+"o", "sino-siberiano",
"sino-tibetano", "sintoma-chave", "siri-azul",
"siri-ba"+u"\u00FA", "siri-chita", "siri-mole",
"siri-patola", "S"+u"\u00ED"+"rio-Liban"+u"\u00EA"+"s", "s"+u"\u00ED"+"rio-liban"+u"\u00EA"+"s",
"situa"+u"\u00E7"+u"\u00E3"+"o-limite", "situa"+u"\u00E7"+u"\u00E3"+"o-problema", "Situa"+u"\u00E7"+u"\u00E3"+"o-Problema",
"sobrinha-neta", "sobrinho-neto", "sobrinhos-netos",
"s"+u"\u00F3"+"cia-administradora", "S"+u"\u00F3"+"cia-Administradora", "s"+u"\u00F3"+"cia-diretora",
"s"+u"\u00F3"+"cia-fundadora", "S"+u"\u00F3"+"cia-Fundadora", "s"+u"\u00F3"+"cia-laranja",
"s"+u"\u00F3"+"cia-propriet"+u"\u00E1"+"ria", "S"+u"\u00F3"+"cia-Propriet"+u"\u00E1"+"ria", "sociais-democracias",
"sociais-democratas", "sociais-democr"+u"\u00E1"+"ticas", "sociais-democr"+u"\u00E1"+"ticos",
"social-crist"+u"\u00E3"+"o", "social-democracia", "Social-Democracia",
"social-democrata", "Social-Democrata", "social-democr"+u"\u00E1"+"tico",
"social-econ"+u"\u00F4"+"mico", "social-hist"+u"\u00F3"+"rico", "social-patriota",
"social-patri"+u"\u00F3"+"tico", "s"+u"\u00F3"+"cias-diretoras", "s"+u"\u00F3"+"cias-fundadoras",
"s"+u"\u00F3"+"cias-laranjas", "s"+u"\u00F3"+"cias-propriet"+u"\u00E1"+"rias", "S"+u"\u00F3"+"cias-Propriet"+u"\u00E1"+"rias",
"s"+u"\u00F3"+"cio-administrador", "S"+u"\u00F3"+"cio-Administrador", "s"+u"\u00F3"+"cio-correspondente",
"s"+u"\u00F3"+"cio-diretor", "s"+u"\u00F3"+"cio-fundador", "S"+u"\u00F3"+"cio-Fundador",
"s"+u"\u00F3"+"cio-gerente", "S"+u"\u00F3"+"cio-Gerente", "s"+u"\u00F3"+"cio-hist"+u"\u00F3"+"rico",
"s"+u"\u00F3"+"cio-humanit"+u"\u00E1"+"rio", "s"+u"\u00F3"+"cio-laranja", "s"+u"\u00F3"+"cio-l"+u"\u00ED"+"der",
"s"+u"\u00F3"+"cio-propriet"+u"\u00E1"+"rio", "S"+u"\u00F3"+"cio-Propriet"+u"\u00E1"+"rio", "s"+u"\u00F3"+"cio-s"+u"\u00EA"+"nior",
"socioecon"+u"\u00F4"+"mico-cultural", "sociopol"+u"\u00ED"+"tico-econ"+u"\u00F4"+"mico", "s"+u"\u00F3"+"cios-diretores",
"s"+u"\u00F3"+"cios-fundadores", "s"+u"\u00F3"+"cios-gerentes", "s"+u"\u00F3"+"cios-laranjas",
"s"+u"\u00F3"+"cios-l"+u"\u00ED"+"deres", "s"+u"\u00F3"+"cios-propriet"+u"\u00E1"+"rios", "S"+u"\u00F3"+"cios-Propriet"+u"\u00E1"+"rios",
"sof"+u"\u00E1"+"-cama", "sossega-le"+u"\u00E3"+"o", "sota-almirante",
"sota-capit"+u"\u00E2"+"nia", "sota-capit"+u"\u00E3"+"o", "sota-cocheiro",
"sota-comitre", "sota-embaixador", "sota-estribeiro",
"sota-general", "sota-mestre", "sota-ministro",
"sota-patr"+u"\u00E3"+"o", "sota-piloto", "sota-proa",
"sota-ventar", "sota-voga", "soto-almirante",
"soto-cacheiro", "soto-capit"+u"\u00E3"+"o", "soto-comitre",
"soto-embaixador", "soto-estribeiro", "soto-general",
"soto-mestre", "soto-ministro", "soto-patr"+u"\u00E3"+"o",
"soto-piloto", "soto-p"+u"\u00F4"+"r", "soto-posto",
"soto-soberania", "stat-coulomb", "strip-tease",
"sueco-dinamarqu"+u"\u00EA"+"s", "sueco-finland"+u"\u00EA"+"s", "sueco-g"+u"\u00F3"+"tico",
"sueco-noruegu"+u"\u00EA"+"s", "sueco-russo", "su"+u"\u00ED"+u"\u00E7"+"o-franc"+u"\u00EA"+"s",
"sul-africanismo", "sul-africano", "sul-alentejano",
"sul-alvoradense", "sul-americana", "sul-americanas",
"sul-americanismo", "sul-americano", "Sul-Americano",
"sul-americanos", "sul-ar"+u"\u00E1"+"bico", "sul-asi"+u"\u00E1"+"tico",
"sul-bocaiuvense", "sul-ca"+u"\u00E7"+"apavense", "sul-cachoeirense",
"sul-caxiense", "sul-centenariense", "sul-continental",
"sul-coreano", "sul-cruzense", "sul-encruzilhadense",
"sul-esperancense", "sul-estrelense", "sul-europeu",
"sul-iemenita", "sul-oriental", "sul-vietnamita",
"sum"+u"\u00E9"+"rio-"+u"\u00E1"+"cade", "superintendente-chefe", "superior-geral",
"s"+u"\u00FA"+"pero-anterior", "s"+u"\u00FA"+"pero-exterior", "s"+u"\u00FA"+"pero-interior",
"s"+u"\u00FA"+"pero-lateral", "s"+u"\u00FA"+"pero-locular", "s"+u"\u00FA"+"pero-palmar",
"s"+u"\u00FA"+"pero-posterior", "surdo-mudez", "surdo-mudo",
"surdos-mudos", "Surdos-Mudos", "tarifa-pr"+u"\u00EA"+"mio",
"tatu-bola", "tatu-canastra", "tatu-cascudo",
"tatu-folha", "tatu-galinha", "tatu-milheiro",
"tatu-mulita", "tatu-peludo", "tatu-tapuia",
"tatu-veado", "tatu-vespa", "tatus-milheiros",
"tatus-tapuias", "t"+u"\u00E1"+"vola-redonda", "T"+u"\u00E1"+"vola-redonda",
"t"+u"\u00E1"+"xi-a"+u"\u00E9"+"reo", "t"+u"\u00E1"+"xis-a"+u"\u00E9"+"reos",
"t"+u"\u00E9"+"cnico-legislativa", "t"+u"\u00E9"+"cnico-legislativas", "t"+u"\u00E9"+"cnico-legislativo", "t"+u"\u00E9"+"cnico-legislativos",
"tecnol"+u"\u00F3"+"gico-eletr"+u"\u00F4"+"nica", "tecnol"+u"\u00F3"+"gico-eletr"+u"\u00F4"+"nicas",
 "tecnol"+u"\u00F3"+"gico-eletr"+u"\u00F4"+"nico", "tecnol"+u"\u00F3"+"gico-eletr"+u"\u00F4"+"nicos",
"tecnol"+u"\u00F3"+"gico-rob"+u"\u00F3"+"tica", "tecnol"+u"\u00F3"+"gico-rob"+u"\u00F3"+"ticas",
 "tecnol"+u"\u00F3"+"gico-rob"+u"\u00F3"+"tico", "tecnol"+u"\u00F3"+"gico-rob"+u"\u00F3"+"ticos",
"tecnocr"+u"\u00E1"+"tico-burocr"+u"\u00E1"+"tico",
"tele-educa"+u"\u00E7"+u"\u00E3"+"o", "tele-homenagem", "tema-chave",
"tema-t"+u"\u00ED"+"tulo", "t"+u"\u00EA"+"mporo-occipital", "tenente-aviador",
"tenente-brigadeiro", "tenente-coronel", "tenente-general",
"tenente-portelense", "tenentes-aviadores", "tenentes-brigadeiros",
"tenentes-coron"+u"\u00E9"+"is", "tenentes-generais", "teol"+u"\u00F3"+"gico-jur"+u"\u00ED"+"dico",
"teol"+u"\u00F3"+"gico-pol"+u"\u00ED"+"tico", "teoria-M", "terael"+u"\u00E9"+"tron-volt",
"ter"+u"\u00E7"+"a-feira", "ter"+u"\u00E7"+"as-feiras", "teresa-cristinense",
"terra-altense", "terra-areense", "terra-bonense",
"terra-branquense", "terra-firmense", "terra-forense",
"terra-grandense", "terra-m"+u"\u00E3"+"e", "terra-novense",
"terra-pretense", "terra-santense", "terra-tenente",
"terra-vermelhense", "tia-av"+u"\u00F3", "tia-bisav"+u"\u00F3",
"tia-rosa", "tias-av"+u"\u00F3"+"s", "tias-bisav"+u"\u00F3"+"s",
"Timor-Leste", "tio-av"+u"\u00F4", "tios-av"+u"\u00F4"+"s",
"tique-taque", "t"+u"\u00ED"+"quete-alimenta"+u"\u00E7"+u"\u00E3"+"o", "t"+u"\u00ED"+"quete-refei"+u"\u00E7"+u"\u00E3"+"o",
"t"+u"\u00ED"+"quete-restaurante", "tomate-bravo", "tomate-cabacinha",
"tomate-caj"+u"\u00E1", "tomate-cereja", "tomate-chimango",
"tomate-groselha", "tomate-pera", "tomate-piriforme",
"tomates-peras", "tomates-piriformes", "tonelada-for"+u"\u00E7"+"a",
"tonelada-milha", "tonelada-quil"+u"\u00F4"+"metro", "torquato-severense",
"torre-branquense", "torre-pedrense", "toxi-infec"+u"\u00E7"+u"\u00E3"+"o",
"toxina-anatoxina", "toxinas-anatoxinas", "trade-unionismo",
"trade-union"+u"\u00ED"+"stico", "tradutor-lexic"+u"\u00F3"+"grafo", "tradutora-lexic"+u"\u00F3"+"grafa",
"tradutoras-lexic"+u"\u00F3"+"grafas", "tradutores-lexic"+u"\u00F3"+"grafos", "traficante-viciada",
"traficante-viciado", "traficantes-viciadas", "traficantes-viciados",
"trave-mestra", "travessa-cal"+u"\u00E7"+"o", "trem-bala",
"trem-conceito", "trem-hospital", "truta-salmonada",
"tubar"+u"\u00E3"+"o-anequim", "tubar"+u"\u00E3"+"o-azul", "tubar"+u"\u00E3"+"o-baleia",
"tubar"+u"\u00E3"+"o-bicudo", "tubar"+u"\u00E3"+"o-branco", "tubar"+u"\u00E3"+"o-elefante",
"tubar"+u"\u00E3"+"o-grelha", "tubar"+u"\u00E3"+"o-jaguara", "tubar"+u"\u00E3"+"o-lim"+u"\u00E3"+"o",
"tubar"+u"\u00E3"+"o-martelo", "tubar"+u"\u00E3"+"o-raposa", "tubar"+u"\u00E3"+"o-sombreiro",
"tubar"+u"\u00E3"+"o-tigre", "tubar"+u"\u00E3"+"o-tintureira", "tubar"+u"\u00E3"+"o-vitam"+u"\u00ED"+"nico",
"tubar"+u"\u00F5"+"es-azuis", "tubar"+u"\u00F5"+"es-bicudos", "tubar"+u"\u00F5"+"es-brancos",
"tubar"+u"\u00F5"+"es-sombreiros", "tubo-ov"+u"\u00E1"+"rio", "tubo-ovarite",
"tucano-boi", "tucano-cachorrinho", "tucano-pacova",
"tucum-bravo", "tucum-maraj"+u"\u00E1", "tucum-mirim",
"tucum-preto", "tucum-rasteiro", "tucum-vermelho",
"tucum"+u"\u00E3"+"-a"+u"\u00E7"+"u", "tucum"+u"\u00E3"+"-arara", "tucum"+u"\u00E3"+"-piranga",
"tucum"+u"\u00E3"+"-piririca", "tucum"+u"\u00E3"+"-purupuru", "tucum"+u"\u00E3"+"s-purupuru",
"tucunar"+u"\u00E9"+"-cip"+u"\u00F3", "tucunar"+u"\u00E9"+"-comum", "tucunar"+u"\u00E9"+"-embira",
"tucunar"+u"\u00E9"+"-pinima", "tucunar"+u"\u00E9"+"s-comuns", "tucunar"+u"\u00E9"+"s-embiras",
"tucunar"+u"\u00E9"+"s-pinimas", "tungst"+u"\u00EA"+"nio-tit"+u"\u00E2"+"nio", "tupi-guarani",
"tupi-paulistense", "tupi-silveirense", "turbo-oxigenador",
"turco-"+u"\u00E1"+"rabe", "turco-asi"+u"\u00E1"+"tico", "turco-balc"+u"\u00E2"+"nico",
"turco-b"+u"\u00FA"+"lgaro", "turco-europeu", "turista-padr"+u"\u00E3"+"o",
"turma-piloto", "uni"+u"\u00E3"+"o-oestense", "uni"+u"\u00E3"+"o-paulista",
"uni"+u"\u00E3"+"o-vitoriense", "unidade-base", "universo-bolha",
"universo-ilha", "urso-formigueiro", "urtiga-branca",
"urtiga-morta", "urubu-ca"+u"\u00E7"+"ador", "usina-piloto",
"usina-prot"+u"\u00F3"+"tipo", "usu"+u"\u00E1"+"rio-chave", "usu"+u"\u00E1"+"rio-produtor",
"usu"+u"\u00E1"+"rios-produtores", "uto-asteca", "utr"+u"\u00ED"+"culo-acinoso",
"utr"+u"\u00ED"+"culo-auditivo", "uva-bastarda", "uva-c"+u"\u00E3"+"o",
"uva-espim", "uva-isabel", "uva-ma"+u"\u00E7"+u"\u00E3",
"uva-mijona", "uva-passa", "uva-rei",
"uva-tinta", "uva-ursina", "uvas-bastardas",
"uvas-bravas", "uvas-mijonas", "uvas-passas",
"uvas-ursinas", "v"+u"\u00E1"+"cuo-press"+u"\u00E3"+"o", "vaga-lume",
"vag"+u"\u00E3"+"o-cama", "vag"+u"\u00E3"+"o-cavalari"+u"\u00E7"+"a", "vag"+u"\u00E3"+"o-cisterna",
"vag"+u"\u00E3"+"o-dinam"+u"\u00F4"+"metro", "vag"+u"\u00E3"+"o-dormit"+u"\u00F3"+"rio", "vag"+u"\u00E3"+"o-frigor"+u"\u00ED"+"fico",
"vag"+u"\u00E3"+"o-leito", "vag"+u"\u00E3"+"o-pipa", "vag"+u"\u00E3"+"o-postal",
"vag"+u"\u00E3"+"o-restaurante", "vag"+u"\u00E3"+"o-sal"+u"\u00E3"+"o", "vag"+u"\u00E3"+"o-tanque",
"vag"+u"\u00E3"+"o-tremonha", "vag"+u"\u00F5"+"es-dinam"+u"\u00F4"+"metros", "vag"+u"\u00F5"+"es-dormit"+u"\u00F3"+"rios",
"vag"+u"\u00F5"+"es-frigor"+u"\u00ED"+"ficos", "vag"+u"\u00F5"+"es-postais", "vai-volta",
"vai-voltense", "vala-sousense", "valor-base",
"valor-oiro", "valor-ouro", "valor-papel",
"v"+u"\u00E3"+"o-livre", "v"+u"\u00E3"+"os-livres", "var-hora",
"var-hor"+u"\u00ED"+"metro", "vara-preta", "varas-pretas",
"vargem-alegrense", "vargem-altense", "vargem-bonitense",
"vargem-cedrense", "vargem-formosense", "vargem-grandense",
"vargem-lajense", "vargem-meense", "vargem-santanense",
"vari"+u"\u00E1"+"vel-chave", "varre-saiense", "v"+u"\u00E1"+"rzea-alegrense",
"v"+u"\u00E1"+"rzea-branquense", "v"+u"\u00E1"+"rzea-caldense", "v"+u"\u00E1"+"rzea-cedrense",
"v"+u"\u00E1"+"rzea-forense", "v"+u"\u00E1"+"rzea-fundense", "v"+u"\u00E1"+"rzea-grandense",
"v"+u"\u00E1"+"rzea-jenipapeirense", "v"+u"\u00E1"+"rzea-novense", "v"+u"\u00E1"+"rzea-palmense",
"v"+u"\u00E1"+"rzea-paulistense", "v"+u"\u00E1"+"rzea-pocense", "v"+u"\u00E1"+"tio-hora",
"vau-a"+u"\u00E7"+"uense", "vau-novense", "vaza-barris",
"Vaza-Barris", "vaza-mar"+u"\u00E9", "veadeiro-mestre",
"veadeiros-mestres", "veado-almiscareiro", "veado-boror"+u"\u00F3",
"veado-branco", "veado-camocica", "veado-campeiro",
"veado-canela", "veado-caracu", "veado-cariacu",
"veado-catingueiro", "veado-foboca", "veado-galheiro",
"veado-garapu", "veado-mateiro", "veado-negro",
"veado-pardo", "veado-roxo", "veado-sua"+u"\u00E7"+"utinga",
"veado-vermelho", "veado-vir"+u"\u00E1", "veados-brancos",
"veados-campeiros", "veados-caracu", "veados-cariacu",
"veados-catingueiros", "veados-galheiros", "veados-garapu",
"veados-mateiros", "veados-negros", "veados-pardos",
"veados-roxos", "veados-vermelhos", "v"+u"\u00E9"+"geto-animal",
"v"+u"\u00E9"+"geto-mineral", "ve"+u"\u00ED"+"culo-prot"+u"\u00F3"+"tipo", "ve"+u"\u00ED"+"culos-prot"+u"\u00F3"+"tipos",
"vera-cruzense", "verde-abacate", "verde-"+u"\u00E1"+"gua",
"verde-alface", "verde-alga", "verde-alvo",
"verde-amarelado", "verde-amarelo", "verde-azeitona",
"verde-azul", "verde-bandeira", "verde-bexiga",
"verde-bronze", "verde-cinza", "verde-claro",
"verde-escuro", "verde-esmeralda", "verde-flavo",
"verde-gaio", "verde-garrafa", "verde-imperial",
"verde-jade", "verde-limo", "verde-limoso",
"verde-malva", "verde-mar", "verde-mineral",
"verde-montanha", "verde-musgo", "verde-negro",
"verde-oliva", "verde-piscina", "verde-rubro",
"verde-salsa", "verde-velho", "verde-verdete",
"vermelho-cereja", "vermelho-escuro", "vermelho-salm"+u"\u00E3"+"o",
"via-cr"+u"\u00FA"+"cis", "via-f"+u"\u00E9"+"rrea", "via-sacra",
"vias-cr"+u"\u00FA"+"cis", "vias-sacras", "vice-almirantado",
"vice-almirante", "vice-bailio", "vice-campe"+u"\u00E3"+"o",
"vice-campeonato", "vice-candidato", "Vice-Candidato",
"vice-chanceler", "vice-chefe", "vice-chefia",
"vice-coletor", "vice-comandante", "vice-comiss"+u"\u00E1"+"rio",
"vice-comodoro", "vice-c"+u"\u00F4"+"nsul", "vice-consulado",
"vice-consular", "vice-c"+u"\u00F4"+"nsules", "vice-consulesa",
"vice-coordenador", "vice-decano", "vice-diretor",
"vice-d"+u"\u00F4"+"mino", "vice-ger"+u"\u00EA"+"ncia", "vice-gerente",
"vice-gestor", "vice-governador", "Vice-Governador",
"vice-governan"+u"\u00E7"+"a", "vice-governo", "vice-imperador",
"vice-intend"+u"\u00EA"+"ncia", "vice-intendente", "vice-lanterna",
"vice-lega"+u"\u00E7"+u"\u00E3"+"o", "vice-legado", "vice-l"+u"\u00ED"+"der",
"vice-lideran"+u"\u00E7"+"a", "vice-maestro", "vice-marechal",
"vice-ministro", "vice-mordomia", "vice-mordomo",
"vice-morte", "vice-penitenci"+u"\u00E1"+"rio", "vice-postulador",
"vice-prefeito", "vice-prefeitura", "vice-premi"+u"\u00EA",
"vice-presid"+u"\u00EA"+"ncia", "vice-Presid"+u"\u00EA"+"ncia", "Vice-Presid"+u"\u00EA"+"ncia",
"vice-presidencial", "vice-presidenta", "Vice-Presidenta",
"vice-presidente", "Vice-Presidente", "vice-procurador",
"vice-promotor", "Vice-Promotor", "vice-provedor",
"vice-prov"+u"\u00ED"+"ncia", "vice-provincial", "vice-provincialado",
"vice-provincialato", "vice-questor", "vice-questura",
"vice-rainha", "vice-real", "vice-realeza",
"vice-rei", "Vice-Rei", "vice-reinado",
"vice-reinar", "vice-reino", "Vice-Reitor",
"vice-reitor", "vice-reitorado", "vice-reitoral",
"vice-reitoria", "vice-secretaria", "vice-secretariado",
"vice-secret"+u"\u00E1"+"rio", "Vice-Secret"+u"\u00E1"+"rio", "vice-senescal",
"vice-senescalado", "vice-senescalia", "vice-tesouraria",
"vice-tesoureiro", "vice-tutela", "vice-tutor",
"vice-versa", "vicente-carvalhense", "vicente-dutrense",
"vicentino-ferrense", "victor-graefense", "vida-longa",
"vida-novense", "vida-sequense", "vidal-ramense",
"vieira-machadense", "vig"+u"\u00E1"+"rio-geral", "vig"+u"\u00E1"+"rios-gerais",
"Villa-Lobos", "vira-casaca", "vira-folhas",
"vira-lata", "visual-motor", "vit"+u"\u00F3"+"ria-r"+u"\u00E9"+"gia",
"vit"+u"\u00F3"+"ria-regina", "vit"+u"\u00F3"+"ria-unionense", "vit"+u"\u00F3"+"rias-r"+u"\u00E9"+"gias",
"vi"+u"\u00FA"+"va-alegre", "vi"+u"\u00FA"+"va-azul", "vi"+u"\u00FA"+"va-mo"+u"\u00E7"+"a",
"vi"+u"\u00FA"+"va-negra", "vi"+u"\u00FA"+"vas-alegres", "vi"+u"\u00FA"+"vas-azuis",
"vi"+u"\u00FA"+"vas-mo"+u"\u00E7"+"as", "vi"+u"\u00FA"+"vas-negras", "volt-ampere",
"volt-amp\u00E8re", "volt-amper"+u"\u00ED"+"metro", "volt-coulomb",
"volt-el"+u"\u00E9"+"ctron", "volt-el"+u"\u00E9"+"tron", "volt-segundo",
"volta-alegrense", "volta-freitense", "volta-grandense",
"volta-redondense", "volts-amperes", "volts-amp\u00E8res",
"volts-coulombs", "volts-el"+u"\u00E9"+"ctrons", "volts-el"+u"\u00E9"+"trons",
"volts-segundos", "voto-vista", "watt-hora",
"watt-hor"+u"\u00ED"+"metro", "watt-minuto", "watt-segundo",
"z"+u"\u00E1"+"s-tr"+u"\u00E1"+"s", "z"+u"\u00E9"+"-cuecas", "z"+u"\u00E9"+"-godes",
"z"+u"\u00E9"+"-ningu"+u"\u00E9"+"m", "z"+u"\u00E9"+"-pereira", "z"+u"\u00E9"+"-povinho",
"z"+u"\u00E9"+"-povo", "zen-budismo", "zero-quil"+u"\u00F4"+"metro",
"z"+u"\u00E9"+"s-pereiras", "z"+u"\u00E9"+"s-povinhos", "z"+u"\u00E9"+"s-povos",
"zona-alvo", "zona-zoster", "zoo-hematina", "zoo-hematita"])


# Termos compostos com duas palavras com -de- no meio: cor-de-rosa, abelha-de-cupim  
tcptoMde = set(["abelha-cupim", "abelha-fogo", "abelha-ouro", "abelha-pau",
"abelha-purga", "abelha-reino", "abelha-sapo", "abelhas-cachorro",
"abelhas-ch"+u"\u00E3"+"o", "abelhas-cupim", "abelhas-fogo", "abelhas-ouro",
"abelhas-pau", "abelhas-purga", "abelhas-reino", "abelhas-sapo",
"ab"+u"\u00F3"+"bora-coroa", "abra"+u"\u00E7"+"o-vide", "abra"+u"\u00E7"+"os-vide", "abric"+u"\u00F3"+"-macaco",
"abric"+u"\u00F3"+"s-macaco", "abutua-batata", "abutuas-batata", "a"+u"\u00E7"+"afate-oiro",
"a"+u"\u00E7"+"afate-ouro", "a"+u"\u00E7"+"afate-prata", "a"+u"\u00E7"+"afates-oiro", "a"+u"\u00E7"+"afates-ouro",
"a"+u"\u00E7"+"afates-prata", "a"+u"\u00E7"+"afroeira-pernambuco", "a"+u"\u00E7"+"afroeiras-pernambuco", "a"+u"\u00E7"+"a"+u"\u00ED"+"-caatinga",
"a"+u"\u00E7"+"a"+u"\u00ED"+"s-caatinga", "acar"+u"\u00E1"+"-v"+u"\u00E9"+"u", "acar"+u"\u00E1"+"s-v"+u"\u00E9"+"u", u"\u00E1"+"caro-galinha",
u"\u00E1"+"caros-galinha", "agr"+u"\u00ED"+"lica-rama", "agr"+u"\u00ED"+"licas-rama", u"\u00E1"+"gua-col"+u"\u00F4"+"nia",
u"\u00E1"+"guas-col"+u"\u00F4"+"nia", u"\u00E1"+"guia-poupa", u"\u00E1"+"guias-poupa", "agulha-marear",
"agulha-pastor", "agulh"+u"\u00E3"+"o-prata", "agulh"+u"\u00E3"+"o-trombeta", "agulh"+u"\u00E3"+"o-vela",
"agulhas-marear", "agulhas-pastor", "agulh"+u"\u00F5"+"es-prata", "agulh"+u"\u00F5"+"es-trombeta",
"agulh"+u"\u00F5"+"es-vela", "aipo-montevid"+u"\u00E9"+"u", "aipos-montevid"+u"\u00E9"+"u", "alamanda-jacobina",
"alamandas-jacobina", "albacora-laje", "albacoras-laje", "albatroz-sobrancelha",
"albatrozes-sobrancelha", "alcachofra-jerusal"+u"\u00E9"+"m", "alcachofras-jerusal"+u"\u00E9"+"m", "alecrim-angola",
"alecrim-campina", "alecrim-cheiro", "alecrins-angola", "alecrins-campina",
"alecrins-cheiro", "aletria-pau", "aletrias-pau", "alface-alger",
"alface-c"+u"\u00E3"+"o", "alface-cordeiro", "alface-porco", "alfaces-alger",
"alfaces-c"+u"\u00E3"+"o", "alfaces-cordeiro", "alfaces-porco", "alfafa-proven"+u"\u00E7"+"a",
"alfafas-proven"+u"\u00E7"+"a", "alfavaca-caboclo", "alfavaca-cheiro", "alfavaca-cobra",
"alfavaca-vaqueiro", "alfavacas-caboclo", "alfavacas-cheiro", "alfavacas-cobra",
"alfavacas-vaqueiro", "alfazema-caboclo", "alfazemas-caboclo", "alfinete-toucar",
"alfinetes-dama", "alfinetes-toucar", "algibeira-dama", "algibeiras-dama",
"algodoeiro-pernambuco", "algodoeiros-pernambuco", "alho-cheiro", "alho-espanha",
"alhos-cheiro", "alhos-espanha", "alma-biafada", "alma-caboclo",
"alma-ca"+u"\u00E7"+"ador", "alma-c"+u"\u00E2"+"ntaro", "alma-cavalo", "alma-chicharro",
"alma-gato", "alma-pombo", "alma-tapuio", "almas-biafada",
"almas-caboclo", "almas-ca"+u"\u00E7"+"ador", "almas-c"+u"\u00E2"+"ntaro", "almas-cavalo",
"almas-chicharro", "almas-gato", "almas-mestre", "almas-pombo",
"almas-tapuio", "almecegueira-cheiro", "almecegueira-minas", "almecegueiras-cheiro",
"almecegueiras-minas", "ameixa-espinho", "ameixa-madagascar", "ameixas-espinho",
"ameixas-madagascar", "am"+u"\u00EA"+"ndoa-coco", "am"+u"\u00EA"+"ndoa-espinho", "am"+u"\u00EA"+"ndoas-coco",
"am"+u"\u00EA"+"ndoas-espinho", "amendoeira-coco", "amendoeiras-coco", "amendoim-"+u"\u00E1"+"rvore",
"amendoim-veado", "amendoins-"+u"\u00E1"+"rvore", "amendoins-veado", "amor-burro",
"amor-hortel"+u"\u00E3"+"o", "amor-mo"+u"\u00E7"+"a", "amor-vaqueiro", "amoreira-"+u"\u00E1"+"rvore",
"amoreira-espinho", "amoreiras-"+u"\u00E1"+"rvore", "amoreiras-espinho", "amores-burro",
"amores-hortel"+u"\u00E3"+"o", "amores-mo"+u"\u00E7"+"a", "amores-vaqueiro", "anamb"+u"\u00E9"+"-capuz",
"anamb"+u"\u00E9"+"s-capuz", "anan"+u"\u00E1"+"s-agulha", "anan"+u"\u00E1"+"s-caraguat"+u"\u00E1", "ananases-agulha",
"ananases-caraguat"+u"\u00E1", "andorinha-bando", "andorinha-casa", "andorinhas-bando",
"andorinhas-casa", "ang"+u"\u00E9"+"lica-rama", "ang"+u"\u00E9"+"licas-rama", "angelim-espinho",
"angelim-morcego", "angelins-espinho", "angelins-morcego", "angico-banhado",
"angico-curtume", "angico-minas", "angico-montes", "angicos-banhado",
"angicos-curtume", "angicos-minas", "angicos-montes", "aninga-espinho",
"aninga-macaco", "aningas-espinho", "aningas-macaco", "anu-enchente",
"anum-enchente", "anuns-enchente", "anus-enchente", "anz"+u"\u00F3"+"is-lontra",
"anzol-lontra", "ara"+u"\u00E7"+u"\u00E1"+"-anta", "ara"+u"\u00E7"+u"\u00E1"+"-cheiro", "ara"+u"\u00E7"+u"\u00E1"+"-comer",
"ara"+u"\u00E7"+u"\u00E1"+"-coroa", "ara"+u"\u00E7"+u"\u00E1"+"-minas", "ara"+u"\u00E7"+u"\u00E1"+"-pomba", "ara"+u"\u00E7"+u"\u00E1"+"-tinguijar",
"ara"+u"\u00E7"+u"\u00E1"+"-umbigo", "ara"+u"\u00E7"+u"\u00E1"+"-veado", "ara"+u"\u00E7"+"ari-minhoca", "ara"+u"\u00E7"+"aris-minhoca",
"ara"+u"\u00E7"+u"\u00E1"+"s-anta", "ara"+u"\u00E7"+u"\u00E1"+"s-cheiro", "ara"+u"\u00E7"+u"\u00E1"+"s-comer", "ara"+u"\u00E7"+u"\u00E1"+"s-coroa",
"ara"+u"\u00E7"+u"\u00E1"+"s-minas", "ara"+u"\u00E7"+u"\u00E1"+"s-pomba", "ara"+u"\u00E7"+u"\u00E1"+"s-tinguijar", "ara"+u"\u00E7"+u"\u00E1"+"s-umbigo",
"ara"+u"\u00E7"+u"\u00E1"+"s-veado", "aranha-"+u"\u00E1"+"gua", "aranha-coco", "aranhas-"+u"\u00E1"+"gua",
"aranhas-coco", "arapoca-cheiro", "arapocas-cheiro", "araruta-porco",
"ararutas-porco", "araticum-boi", "araticum-boia", "araticum-cheiro",
"araticum-espinho", "araticum-jangada", "araticum-paca", "araticuns-boi",
"araticuns-boia", "araticuns-cheiro", "araticuns-espinho", "araticuns-jangada",
"araticuns-paca", "arco-pipa", "arcos-pipa", "arlequim-caiena",
"arlequins-caiena", "aroeira-bugre", "aroeira-capoeira", "aroeira-goi"+u"\u00E1"+"s",
"aroeira-minas", "aroeiras-bugre", "aroeiras-capoeira", "aroeiras-goi"+u"\u00E1"+"s",
"aroeiras-minas", "arraia-coroa", "arraia-fogo", "arraias-coroa",
"arraias-fogo", "arrozes-rato", "arrozes-telhado", "arruda-campinas",
"arrudas-campinas", u"\u00E1"+"rvore-anoz", u"\u00E1"+"rvore-arroz", u"\u00E1"+"rvore-b"+u"\u00E1"+"lsamo",
u"\u00E1"+"rvore-chocalho", u"\u00E1"+"rvore-coral", u"\u00E1"+"rvore-cuia", u"\u00E1"+"rvore-gralha",
u"\u00E1"+"rvore-graxa", u"\u00E1"+"rvore-incenso", u"\u00E1"+"rvore-judas", u"\u00E1"+"rvore-l"+u"\u00E3",
u"\u00E1"+"rvore-leite", u"\u00E1"+"rvore-l"+u"\u00F3"+"tus", u"\u00E1"+"rvore-natal", u"\u00E1"+"rvore-orqu"+u"\u00ED"+"dea",
u"\u00E1"+"rvore-pagode", u"\u00E1"+"rvore-rancho", u"\u00E1"+"rvore-ranho", u"\u00E1"+"rvore-sebo",
u"\u00E1"+"rvore-velas", u"\u00E1"+"rvores-anoz", u"\u00E1"+"rvores-arroz", u"\u00E1"+"rvores-b"+u"\u00E1"+"lsamo",
u"\u00E1"+"rvores-chocalho", u"\u00E1"+"rvores-coral", u"\u00E1"+"rvores-cuia", u"\u00E1"+"rvores-gralha",
u"\u00E1"+"rvores-graxa", u"\u00E1"+"rvores-incenso", u"\u00E1"+"rvores-judas", u"\u00E1"+"rvores-l"+u"\u00E3",
u"\u00E1"+"rvores-leite", u"\u00E1"+"rvores-l"+u"\u00F3"+"tus", u"\u00E1"+"rvores-natal", u"\u00E1"+"rvores-orqu"+u"\u00ED"+"dea",
u"\u00E1"+"rvores-pagode", u"\u00E1"+"rvores-rancho", u"\u00E1"+"rvores-ranho", u"\u00E1"+"rvores-sebo",
u"\u00E1"+"rvores-velas", "asa-barata", "asa-palha", "asa-papagaio",
"asa-sabre", "asa-telha", "asas-barata", "asas-palha",
"asas-papagaio", "asas-sabre", "asas-telha", "aspargo-jardim",
"aspargo-sala", "aspargos-jardim", "aspargos-sala", "assobio-cobra",
"assobio-folha", "assobios-cobra", "assobios-folha", "assovio-cobra",
"assovio-folha", "assovios-cobra", "assovios-folha", "atum-galha",
"atuns-galha", "ave-crocodilo", "avenca-espiga", "avenca-minas",
"avencas-espiga", "avencas-minas", "aves-crocodilo", "azedas-ovelha",
"azedinha-corumb"+u"\u00E1", "azedinha-goi"+u"\u00E1"+"s", "azedinhas-corumb"+u"\u00E1", "azedinhas-goi"+u"\u00E1"+"s",
"baba-boi", "baba-camelo", "baba-sapo", "babas-boi",
"babas-sapo", "babosa-"+u"\u00E1"+"rvore", "babosa-espiga", "babosa-pau",
"babosas-"+u"\u00E1"+"rvore", "babosas-espiga", "babosas-pau", "bacaba-azeite",
"bacaba-leque", "bacabas-azeite", "bacabas-leque", "bacu-pedra",
"bacupari-capoeira", "bacuparis-capoeira", "bacurau-bando", "bacurau-lajeado",
"bacuraus-bando", "bacuraus-lajeado", "bacuri-cerca", "bacuris-cerca",
"bacus-pedra", "badejo-lista", "badejos-lista", "badiana-cheiro",
"badianas-cheiro", "baga-caboclo", "baga-louro", "baga-pombo",
"baga-praia", "baga-tucano", "bagas-caboclo", "bagas-louro",
"bagas-pombo", "bagas-praia", "bagas-tucano", "bago-chumbo",
"bagos-chumbo", "bagre-areia", "bagre-lagoa", "bagre-mangue",
"bagre-manta", "bagre-penacho", "bagres-areia", "bagres-lagoa",
"bagres-mangue", "bagres-manta", "bagres-penacho", "baiacu-chifre",
"baiacu-espinho", "baiacus-chifre", "baiacus-espinho", "bainha-espada",
"bainha-faca", "bainhas-espada", "bainhas-faca", "baleia-barbatana",
"baleia-bico", "baleia-corcova", "baleia-gomo", "baleias-barbatana",
"baleias-bico", "baleias-corcova", "baleias-gomo", "b"+u"\u00E1"+"lsamo-arceu",
"b"+u"\u00E1"+"lsamo-canudo", "b"+u"\u00E1"+"lsamo-cartagena", "b"+u"\u00E1"+"lsamo-cheiro", "b"+u"\u00E1"+"lsamo-enxofre",
"b"+u"\u00E1"+"lsamo-tolu", "b"+u"\u00E1"+"lsamos-arceu", "b"+u"\u00E1"+"lsamos-canudo", "b"+u"\u00E1"+"lsamos-cartagena",
"b"+u"\u00E1"+"lsamos-cheiro", "b"+u"\u00E1"+"lsamos-enxofre", "b"+u"\u00E1"+"lsamos-tolu", "bambu-cani"+u"\u00E7"+"o",
"bambu-espinho", "bambu-mob"+u"\u00ED"+"lia", "bambu-pescador", "bambus-cani"+u"\u00E7"+"o",
"bambus-espinho", "bambus-mob"+u"\u00ED"+"lia", "bambus-pescador", "banana-imb"+u"\u00E9",
"banana-macaco", "banana-morcego", "banana-papagaio", "banana-sancho",
"banana-sementes", "bananas-imb"+u"\u00E9", "bananas-macaco", "bananas-morcego",
"bananas-papagaio", "bananas-sancho", "bananas-sementes", "bananeira-corda",
"bananeira-flor", "bananeira-italiano", "bananeira-jardim", "bananeira-leque",
"bananeira-madagascar", "bananeira-madag"+u"\u00E1"+"scar", "bananeira-papagaio", "bananeira-sementes",
"bananeiras-corda", "bananeiras-flor", "bananeiras-italiano", "bananeiras-jardim",
"bananeiras-leque", "bananeiras-madagascar", "bananeiras-madag"+u"\u00E1"+"scar", "bananeiras-papagaio",
"bananeiras-sementes", "bananeirinha-flor", "bananeirinha-sal"+u"\u00E3"+"o", "bananeirinha-touceira",
"bananeirinhas-flor", "bananeirinhas-sal"+u"\u00E3"+"o", "bananeirinhas-touceira", "barba-chibo",
"barba-lagoa", "barba-paca", "barba-pau", "barba-tim"+u"\u00E3"+"o",
"barbas-boi", "barbas-chibo", "barbas-lagoa", "barbas-pau",
"barbas-tim"+u"\u00E3"+"o", "barrete-cl"+u"\u00E9"+"rigo", "barrete-eleitor", "barrete-padre",
"barretes-cl"+u"\u00E9"+"rigo", "barretes-eleitor", "barretes-padre", "barriga-freira",
"barrigas-freira", "barriguda-espinho", "barrigudas-espinho", "bastardo-roma",
"bastardos-roma", "basti"+u"\u00E3"+"o-arruda", "basti"+u"\u00F5"+"es-arruda", "bast"+u"\u00F5"+"es-velho",
"batata-amaro", "batata-angola", "batata-arroba", "batata-bainha",
"batata-branco", "batata-bugio", "batata-caboclo", "batata-escamas",
"batata-pedra", "batata-perdiz", "batata-porco", "batata-purga",
"batata-puri", "batata-rama", "batata-taiui"+u"\u00E1", "batata-ti"+u"\u00FA",
"batata-veado", "batatas-amaro", "batatas-angola", "batatas-arroba",
"batatas-bainha", "batatas-branco", "batatas-bugio", "batatas-caboclo",
"batatas-escamas", "batatas-pedra", "batatas-perdiz", "batatas-porco",
"batatas-purga", "batatas-puri", "batatas-rama", "batatas-taiui"+u"\u00E1",
"batatas-ti"+u"\u00FA", "batatas-veado", "batatinha-cobra", "batatinhas-cobra",
"ba"+u"\u00FA"+"na-fogo", "ba"+u"\u00FA"+"nas-fogo", "baunilha-auacuri", "baunilhas-auacuri",
"beiju-pot"+u"\u00F3", "beijus-pot"+u"\u00F3", "bela-felgueira", "bela-felgueiras",
"belas-felgueira", "belas-felgueiras", "beldroega-cuba", "beldroega-inverno",
"beldroegas-cuba", "beldroegas-inverno", "bengala-camar"+u"\u00E3"+"o", "bengalas-camar"+u"\u00E3"+"o",
"besouro-"+u"\u00E1"+"gua", "besouro-chifre", "besouro-limeira", "besouro-maio",
"besouros-"+u"\u00E1"+"gua", "besouros-chifre", "besouros-limeira", "besouros-maio",
"besugo-ovas", "besugos-ovas", "bet"+u"\u00F4"+"nica-"+u"\u00E1"+"gua", "bet"+u"\u00F4"+"nicas-"+u"\u00E1"+"gua",
"bicheiro-conta", "bicheiros-conta", "bicho-buraco", "bicho-cachorro",
"bicho-carpinteiro", "bicho-cesto", "bicho-chifre", "bicho-coco",
"bicho-conta", "bicho-esterco", "bicho-fogo", "bicho-frade",
"bicho-galinha", "bicho-mato", "bicho-medran"+u"\u00E7"+"a", "bicho-mosca",
"bicho-ouvido", "bicho-parede", "bicho-pau", "bicho-p"+u"\u00E9",
"bicho-pena", "bicho-porco", "bicho-rumo", "bicho-seda",
"bicho-taquara", "bicho-unha", "bicho-vareja", "bicho-veludo",
"bichos-buraco", "bichos-cachorro", "bichos-canastro", "bichos-carpinteiro",
"bichos-cesto", "bichos-chifre", "bichos-coco", "bichos-conta",
"bichos-esterco", "bichos-fogo", "bichos-frade", "bichos-galinha",
"bichos-medran"+u"\u00E7"+"a", "bichos-mosca", "bichos-ouvido", "bichos-parede",
"bichos-pau", "bichos-p"+u"\u00E9", "bichos-pena", "bichos-porco",
"bichos-rumo", "bichos-seda", "bichos-taquara", "bichos-unha",
"bichos-vareja", "bichos-veludo", "bicuda-corso", "bicudas-corso",
"birib"+u"\u00E1"+"-pernambuco", "birib"+u"\u00E1"+"s-pernambuco", "bitelo-gente", "bitelos-gente",
"boais-alicante", "boal-alicante", "bofe-burro", "bofes-burro",
"boi-carro", "boi-deus", "boi-guar"+u"\u00E1", "bois-carro",
"bois-deus", "bois-guar"+u"\u00E1", "boita-bigodes", "boitas-bigodes",
"borboleta-alface", "borboleta-amoreira", "borboleta-bando", "borboleta-carvalho",
"borboleta-g"+u"\u00E1"+"s", "borboleta-p"+u"\u00EA"+"ssego", "borboleta-piracema", "borboletas-alface",
"borboletas-amoreira", "borboletas-bando", "borboletas-carvalho", "borboletas-g"+u"\u00E1"+"s",
"borboletas-p"+u"\u00EA"+"ssego", "borboletas-piracema", "bord"+u"\u00E3"+"o-macaco", "bord"+u"\u00E3"+"o-santiago",
"bord"+u"\u00F5"+"es-macaco", "bord"+u"\u00F5"+"es-santiago", "botos-"+u"\u00F3"+"culos", "branco-barita",
"brancos-barita", "bredo-cabe"+u"\u00E7"+"a", "bredo-espiga", "bredo-espinho",
"bredo-jardim", "bredo-muro", "bredo-namorado", "bredo-namoro",
"bredo-porco", "bredo-veado", "bredos-cabe"+u"\u00E7"+"a", "bredos-espiga",
"bredos-espinho", "bredos-jardim", "bredos-muro", "bredos-namorado",
"bredos-namoro", "bredos-porco", "bredos-veado", "brinco-passarinho",
"brinco-sagui", "brinco-saguim", "brinco-sauim", "brincos-passarinho",
"brincos-sagui", "brincos-saguim", "brincos-sauim", "brincos-surubim",
"broca-raiz", "brocas-raiz", "bruco-salvaterra", "brucos-salvaterra",
"bucha-purga", "buchas-purga", "bucho-boi", "bucho-r"+u"\u00E3",
"bucho-veado", "buchos-boi", "buchos-r"+u"\u00E3", "buchos-veado",
"buti"+u"\u00E1"+"-vinagre", "buti"+u"\u00E1"+"s-vinagre", "butua-corvo", "butuas-corvo",
"buxo-holanda", "buxos-holanda", "caavurana-cunh"+u"\u00E3", "caavuranas-cunh"+u"\u00E3",
"caba-igreja", "caba-ladr"+u"\u00E3"+"o", "caba-peixe", "caba"+u"\u00E7"+"a-trombeta",
"caba"+u"\u00E7"+"as-trombeta", "cabacinha-cobra", "cabacinhas-cobra", "cabas-igreja",
"cabas-ladr"+u"\u00E3"+"o", "cabas-peixe", "cabumbo-azeite", "cabumbos-azeite",
"cabur"+u"\u00E9"+"-orelha", "cabur"+u"\u00E9"+"s-orelha", "cacau-caiena", "cacau-caracas",
"cacau-mico", "cacaus-caiena", "cacaus-caracas", "cacaus-mico",
"cachimbo-jabuti", "cachimbo-macaco", "cachimbo-turco", "cachimbos-jabuti",
"cachimbos-macaco", "cachimbos-turco", "cacto-cabe"+u"\u00E7"+"a", "cactos-cabe"+u"\u00E7"+"a",
"caf"+u"\u00E9"+"-bagueio", "caf"+u"\u00E9"+"-bugre", "caf"+u"\u00E9"+"s-bagueio", "caf"+u"\u00E9"+"s-bugre",
"camale"+u"\u00E3"+"o-asas", "camale"+u"\u00E3"+"o-pedreira", "camale"+u"\u00F5"+"es-asas", "camale"+u"\u00F5"+"es-pedreira",
"camar"+u"\u00E1"+"-bilro", "camar"+u"\u00E1"+"-boi", "camar"+u"\u00E1"+"-cavalo", "camar"+u"\u00E1"+"-cheiro",
"camar"+u"\u00E1"+"-espinho", "camar"+u"\u00E1"+"-flecha", "camar"+u"\u00E3"+"o-areia", "camar"+u"\u00E3"+"o-estalo",
"camar"+u"\u00E3"+"o-pedra", "camar"+u"\u00E3"+"o-penedo", "camar"+u"\u00E1"+"s-bilro", "camar"+u"\u00E1"+"s-boi",
"camar"+u"\u00E1"+"s-cavalo", "camar"+u"\u00E1"+"s-cheiro", "camar"+u"\u00E1"+"s-espinho", "camar"+u"\u00E1"+"s-flecha",
"camar"+u"\u00F5"+"es-areia", "camar"+u"\u00F5"+"es-estalo", "camar"+u"\u00F5"+"es-pedra", "camar"+u"\u00F5"+"es-penedo",
"cambar"+u"\u00E1"+"-capoeira", "cambar"+u"\u00E1"+"-cheiro", "cambar"+u"\u00E1"+"-chumbo", "cambar"+u"\u00E1"+"-espinho",
"cambar"+u"\u00E1"+"-lixa", "cambar"+u"\u00E1"+"s-capoeira", "cambar"+u"\u00E1"+"s-cheiro", "cambar"+u"\u00E1"+"s-chumbo",
"cambar"+u"\u00E1"+"s-espinho", "cambar"+u"\u00E1"+"s-lixa", "camboat"+u"\u00E3"+"-leite", "camboat"+u"\u00E3"+"s-leite",
"camuru-cheiro", "camurus-cheiro", "cana-a"+u"\u00E7"+u"\u00FA"+"car", "Cana-A"+u"\u00E7"+u"\u00FA"+"car",
"cana-burro", "cana-elefante", "cana-frecha", "cana-imb"+u"\u00E9",
"cana-jacar"+u"\u00E9", "cana-macaco", "cana-passarinho", "cana-roca",
"cana-urubu", "cana-vassoira", "cana-vassoura", "cana-v"+u"\u00ED"+"bora",
"canaf"+u"\u00ED"+"stula-boi", "canaf"+u"\u00ED"+"stula-igap"+u"\u00F3", "canaf"+u"\u00ED"+"stula-lagoa", "canaf"+u"\u00ED"+"stulas-boi",
"canaf"+u"\u00ED"+"stulas-igap"+u"\u00F3", "canaf"+u"\u00ED"+"stulas-lagoa", "can"+u"\u00E1"+"rio-fran"+u"\u00E7"+"a", "can"+u"\u00E1"+"rios-fran"+u"\u00E7"+"a",
"canas-a"+u"\u00E7"+u"\u00FA"+"car", "canas-burro", "canas-cheiro", "canas-elefante",
"canas-frecha", "canas-imb"+u"\u00E9", "canas-jacar"+u"\u00E9", "canas-macaco",
"canas-passarinho", "canas-roca", "canas-urubu", "canas-vassoira",
"canas-vassoura", "canas-v"+u"\u00ED"+"bora", "canela-capoeira", "canela-catarro",
"canela-cutia", "canela-ema", "canela-gar"+u"\u00E7"+"a", "canela-goi"+u"\u00E1"+"s",
"canela-jacamim", "canela-papagaio", "canela-vargem", "canela-veado",
"canelas-capoeira", "canelas-catarro", "canelas-ceil"+u"\u00E3"+"o", "canelas-cheiro",
"canelas-cutia", "canelas-ema", "canelas-gar"+u"\u00E7"+"a", "canelas-goi"+u"\u00E1"+"s",
"canelas-jacamim", "canelas-papagaio", "canelas-vargem", "canelas-veado",
"canelas-velho", "caneleira-cheiro", "caneleira-ema", "caneleiras-cheiro",
"caneleiras-ema", "c"+u"\u00E2"+"nhamo-manila", "cani"+u"\u00E7"+"o-"+u"\u00E1"+"gua", "cani"+u"\u00E7"+"os-"+u"\u00E1"+"gua",
"cano"+u"\u00E9"+"-bot"+u"\u00E3"+"o", "cano"+u"\u00E9"+"s-bot"+u"\u00E3"+"o", "cansan"+u"\u00E7"+u"\u00E3"+"o-leite", "cansan"+u"\u00E7"+u"\u00F5"+"es-leite",
"canudo-cachimbo", "canudo-lagoa", "canudos-cachimbo", "canudos-lagoa",
"capela-vi"+u"\u00FA"+"va", "capelas-vi"+u"\u00FA"+"va", "capim-a"+u"\u00E7"+"ude", "capim-andar",
"capim-angola", "capim-bandeira", "capim-batatais", "capim-bezerro",
"capim-bode", "capim-bolota", "capim-bota", "capim-bot"+u"\u00E3"+"o",
"capim-bucha", "capim-burro", "capim-cabra", "capim-capivara",
"capim-carneiro", "capim-cavalo", "capim-cheiro", "capim-coco",
"capim-contas", "capim-corte", "capim-cuba", "capim-deus",
"capim-diamante", "capim-"+u"\u00E9"+"gua", "capim-esteira", "capim-feixe",
"capim-flecha", "capim-fogo", "capim-forquilha", "capim-itu",
"capim-lastro", "capim-manada", "capim-marreca", "capim-mula",
"capim-natal", "capim-pasto", "capim-pernambuco", "capim-planta",
"capim-pombo", "capim-raiz", "capim-rebanho", "capim-rodes",
"capim-rola", "capim-rosa", "capim-soca", "capim-tartaruga",
"capim-tenerife", "capim-teso", "capim-touceira", "capins-a"+u"\u00E7"+"ude",
"capins-andar", "capins-angola", "capins-bandeira", "capins-batatais",
"capins-bezerro", "capins-bode", "capins-bolota", "capins-bota",
"capins-bot"+u"\u00E3"+"o", "capins-bucha", "capins-burro", "capins-cabra",
"capins-capivara", "capins-carneiro", "capins-cavalo", "capins-cheiro",
"capins-coco", "capins-contas", "capins-corte", "capins-cuba",
"capins-diamante", "capins-"+u"\u00E9"+"gua", "capins-esteira", "capins-feixe",
"capins-flecha", "capins-fogo", "capins-forquilha", "capins-itu",
"capins-lastro", "capins-manada", "capins-marreca", "capins-mula",
"capins-natal", "capins-pasto", "capins-pernambuco", "capins-planta",
"capins-pombo", "capins-raiz", "capins-rebanho", "capins-rodes",
"capins-rola", "capins-rosa", "capins-sapo", "capins-soca",
"capins-tartaruga", "capins-tenerife", "capins-teso", "capins-touceira",
"capit"+u"\u00E3"+"es-sa"+u"\u00ED"+"ra", "capit"+u"\u00E3"+"es-sala", "capit"+u"\u00E3"+"o-sa"+u"\u00ED"+"ra", "capit"+u"\u00E3"+"o-sala",
"capixingui-bicho", "capixinguis-bicho", "car"+u"\u00E1"+"-"+u"\u00E1"+"gua", "car"+u"\u00E1"+"-caboclo",
"car"+u"\u00E1"+"-espinho", "car"+u"\u00E1"+"-jardim", "car"+u"\u00E1"+"-pedra", "car"+u"\u00E1"+"-sapateiro",
"caraguat"+u"\u00E1"+"-jardim", "caraguat"+u"\u00E1"+"s-jardim", "caranguejo-pedra", "caranguejos-pedra",
"caranha-toco", "caranha-viveiro", "caranhas-rio", "caranhas-toco",
"caranhas-viveiro", "car"+u"\u00E1"+"s-"+u"\u00E1"+"gua", "car"+u"\u00E1"+"s-angola", "car"+u"\u00E1"+"s-caboclo",
"car"+u"\u00E1"+"s-espinho", "car"+u"\u00E1"+"s-jardim", "car"+u"\u00E1"+"s-pedra", "car"+u"\u00E1"+"s-sapateiro",
"cardeais-poupa", "cardeal-poupa", "cardo-burro", "cardo-coalho",
"cardo-cochonilha", "cardo-comer", "cardo-isca", "cardo-ouro",
"cardo-visco", "cardo-visgo", "cardos-burro", "cardos-coalho",
"cardos-cochonilha", "cardos-comer", "cardos-isca", "cardos-ouro",
"cardos-visco", "cardos-visgo", "carmim-c"+u"\u00E1"+"rtamo", "carnes-donzela",
"carrapato-boi", "carrapato-cavalo", "carrapato-galinha", "carrapato-passarinho",
"carrapato-peixe", "carrapato-sapo", "carrapatos-boi", "carrapatos-cavalo",
"carrapatos-galinha", "carrapatos-passarinho", "carrapatos-peixe", "carrapatos-sapo",
"carrapicho-agulha", "carrapicho-boi", "carrapicho-cal"+u"\u00E7"+"ada", "carrapicho-carneiro",
"carrapicho-cavalo", "carrapicho-cigana", "carrapicho-l"+u"\u00E3", "carrapicho-linho",
"carrapicho-ovelha", "carrapichos-agulha", "carrapichos-boi", "carrapichos-cal"+u"\u00E7"+"ada",
"carrapichos-carneiro", "carrapichos-cavalo", "carrapichos-cigana", "carrapichos-l"+u"\u00E3",
"carrapichos-linho", "carrapichos-ovelha", "caruat"+u"\u00E1"+"-pau", "caruat"+u"\u00E1"+"s-pau",
"caruru-cacho", "caruru-espiga", "caruru-espinho", "caruru-pomba",
"caruru-porco", "caruru-sapo", "caruru-soldado", "caruru-veado",
"carurus-cacho", "carurus-espiga", "carurus-espinho", "carurus-pomba",
"carurus-porco", "carurus-sapo", "carurus-soldado", "carurus-veado",
"carv"+u"\u00E3"+"o-ferreiro", "carv"+u"\u00F5"+"es-ferreiro", "casca-carvalho", "casca-jacar"+u"\u00E9",
"casca-noz", "cascas-anta", "cascas-carvalho", "cascas-jacar"+u"\u00E9",
"casco-cavalo", "casco-jabuti", "casco-tatu", "cascos-cavalo",
"cascos-jabuti", "cascos-tatu", "cascudo-aranha", "cascudo-enfeite",
"cascudos-aranha", "cascudos-enfeite", "casta-correr", "castanha-"+u"\u00E1"+"frica",
"castanha-"+u"\u00E1"+"gua", "castanha-anta", "castanha-arara", "castanha-bugre",
"castanha-caiat"+u"\u00E9", "castanha-cutia", "castanha-jatob"+u"\u00E1", "castanha-macaco",
"castanha-minas", "castanha-mo"+u"\u00E7"+"ambique", "castanha-peixe", "castanha-puri",
"castanhas-"+u"\u00E1"+"frica", "castanhas-"+u"\u00E1"+"gua", "castanhas-anta", "castanhas-arara",
"castanhas-bugre", "castanhas-caiat"+u"\u00E9", "castanhas-cutia", "castanhas-macaco",
"castanhas-minas", "castanhas-mo"+u"\u00E7"+"ambique", "castanhas-peixe", "castanhas-puri",
"castanheiro-minas", "castanheiros-minas", "castas-correr", "castor-montanha",
"castores-montanha", "catingueiro-porco", "catingueiros-porco", "catota-espinho",
"catotas-espinho", "cavalinho-c"+u"\u00E3"+"o", "cavalinho-deus", "cavalinho-judeu",
"cavalinhos-judeu", "cavalo-c"+u"\u00E3"+"o", "cebola-lobo", "cebolas-cheiro",
"cebolas-lobo", "ceboleta-fran"+u"\u00E7"+"a", "ceboletas-fran"+u"\u00E7"+"a", "cebolinha-cheiro",
"cebolinhas-cheiro", "cenoura-creta", "cenouras-creta", "cent"+u"\u00E1"+"urea-jardim",
"cent"+u"\u00E1"+"ureas-jardim", "cereja-caiena", "cereja-purga", "cerejas-caiena",
"cerejas-purga", "cerejeira-purga", "cerejeiras-purga", "cerveja-pobre",
"cervejas-pobre", "cevada-jardim", "cevadas-jardim", "chaga-bauru",
"chagas-bauru", "chagas-jesus", "chasco-leque", "chascos-leque",
"chup"+u"\u00E3"+"o-arroz", "chup"+u"\u00F5"+"es-arroz", "chuva-imbu", "chuvas-imbu",
"cip"+u"\u00F3"+"-"+u"\u00E1"+"gua", "cip"+u"\u00F3"+"-agulha", "cip"+u"\u00F3"+"-alca"+u"\u00E7"+"uz", "cip"+u"\u00F3"+"-alho",
"cip"+u"\u00F3"+"-amarrar", "cip"+u"\u00F3"+"-aracu"+u"\u00E3"+"o", "cip"+u"\u00F3"+"-arco", "cip"+u"\u00F3"+"-bamburral",
"cip"+u"\u00F3"+"-barril", "cip"+u"\u00F3"+"-batata", "cip"+u"\u00F3"+"-boi", "cip"+u"\u00F3"+"-breu",
"cip"+u"\u00F3"+"-caba"+u"\u00E7"+"a", "cip"+u"\u00F3"+"-caboclo", "cip"+u"\u00F3"+"-canoa", "cip"+u"\u00F3"+"-carij"+u"\u00F3",
"cip"+u"\u00F3"+"-carneiro", "cip"+u"\u00F3"+"-cesto", "cip"+u"\u00F3"+"-chagas", "cip"+u"\u00F3"+"-chumbo",
"cip"+u"\u00F3"+"-cobra", "cip"+u"\u00F3"+"-cola", "cip"+u"\u00F3"+"-cora"+u"\u00E7"+u"\u00E3"+"o", "cip"+u"\u00F3"+"-corda",
"cip"+u"\u00F3"+"-cuman"+u"\u00E3", "cip"+u"\u00F3"+"-cunan"+u"\u00E3", "cip"+u"\u00F3"+"-embiri", "cip"+u"\u00F3"+"-escada",
"cip"+u"\u00F3"+"-fogo", "cip"+u"\u00F3"+"-gato", "cip"+u"\u00F3"+"-gota", "cip"+u"\u00F3"+"-imb"+u"\u00E9",
"cip"+u"\u00F3"+"-impingem", "cip"+u"\u00F3"+"-jabota", "cip"+u"\u00F3"+"-jabut"+u"\u00E1", "cip"+u"\u00F3"+"-jabuti",
"cip"+u"\u00F3"+"-jarrinha", "cip"+u"\u00F3"+"-junta", "cip"+u"\u00F3"+"-lavadeira", "cip"+u"\u00F3"+"-leite",
"cip"+u"\u00F3"+"-macaco", "cip"+u"\u00F3"+"-mainibu", "cip"+u"\u00F3"+"-mofumbo", "cip"+u"\u00F3"+"-morcego",
"cip"+u"\u00F3"+"-mucuna", "cip"+u"\u00F3"+"-paina", "cip"+u"\u00F3"+"-palma", "cip"+u"\u00F3"+"-penas",
"cip"+u"\u00F3"+"-poita", "cip"+u"\u00F3"+"-quati", "cip"+u"\u00F3"+"-rego", "cip"+u"\u00F3"+"-sapo",
"cip"+u"\u00F3"+"-seda", "cip"+u"\u00F3"+"-tamandu"+u"\u00E1", "cip"+u"\u00F3"+"-timb"+u"\u00F3", "cip"+u"\u00F3"+"-tucunar"+u"\u00E9",
"cip"+u"\u00F3"+"-vaqueiro", "cip"+u"\u00F3"+"-vi"+u"\u00FA"+"va", "cip"+u"\u00F3"+"s-"+u"\u00E1"+"gua", "cip"+u"\u00F3"+"s-agulha",
"cip"+u"\u00F3"+"s-alca"+u"\u00E7"+"uz", "cip"+u"\u00F3"+"s-alho", "cip"+u"\u00F3"+"s-amarrar", "cip"+u"\u00F3"+"s-aracu"+u"\u00E3"+"o",
"cip"+u"\u00F3"+"s-arco", "cip"+u"\u00F3"+"s-bamburral", "cip"+u"\u00F3"+"s-barril", "cip"+u"\u00F3"+"s-batata",
"cip"+u"\u00F3"+"s-boi", "cip"+u"\u00F3"+"s-breu", "cip"+u"\u00F3"+"s-caba"+u"\u00E7"+"a", "cip"+u"\u00F3"+"s-caboclo",
"cip"+u"\u00F3"+"s-canoa", "cip"+u"\u00F3"+"s-carij"+u"\u00F3", "cip"+u"\u00F3"+"s-carneiro", "cip"+u"\u00F3"+"s-cesto",
"cip"+u"\u00F3"+"s-chagas", "cip"+u"\u00F3"+"s-cobra", "cip"+u"\u00F3"+"s-cola", "cip"+u"\u00F3"+"s-cora"+u"\u00E7"+u"\u00E3"+"o",
"cip"+u"\u00F3"+"s-corda", "cip"+u"\u00F3"+"s-cuman"+u"\u00E3", "cip"+u"\u00F3"+"s-cunan"+u"\u00E3", "cip"+u"\u00F3"+"s-embiri",
"cip"+u"\u00F3"+"s-escada", "cip"+u"\u00F3"+"s-fogo", "cip"+u"\u00F3"+"s-gato", "cip"+u"\u00F3"+"s-gota",
"cip"+u"\u00F3"+"s-imb"+u"\u00E9", "cip"+u"\u00F3"+"s-impingem", "cip"+u"\u00F3"+"s-jabota", "cip"+u"\u00F3"+"s-jabut"+u"\u00E1",
"cip"+u"\u00F3"+"s-jabuti", "cip"+u"\u00F3"+"s-jarrinha", "cip"+u"\u00F3"+"s-junta", "cip"+u"\u00F3"+"s-lavadeira",
"cip"+u"\u00F3"+"s-leite", "cip"+u"\u00F3"+"s-macaco", "cip"+u"\u00F3"+"s-mainibu", "cip"+u"\u00F3"+"s-mofumbo",
"cip"+u"\u00F3"+"s-morcego", "cip"+u"\u00F3"+"s-mucuna", "cip"+u"\u00F3"+"s-paina", "cip"+u"\u00F3"+"s-palma",
"cip"+u"\u00F3"+"s-penas", "cip"+u"\u00F3"+"s-poita", "cip"+u"\u00F3"+"s-quati", "cip"+u"\u00F3"+"s-rego",
"cip"+u"\u00F3"+"s-sapo", "cip"+u"\u00F3"+"s-seda", "cip"+u"\u00F3"+"s-tamandu"+u"\u00E1", "cip"+u"\u00F3"+"s-timb"+u"\u00F3",
"cip"+u"\u00F3"+"s-tucunar"+u"\u00E9", "cip"+u"\u00F3"+"s-vaqueiro", "cip"+u"\u00F3"+"s-vi"+u"\u00FA"+"va", "cobra-"+u"\u00E1"+"gua",
"cobra-ar", "cobra-areia", "cobra-asa", "cobra-barata",
"cobra-cabelo", "cobra-caju", "cobra-capelo", "cobra-capim",
"cobra-cascavel", "cobra-cip"+u"\u00F3", "cobra-colchete", "cobra-coral",
"cobra-escada", "cobra-espada", "cobra-ferradura", "cobra-lixo",
"cobra-oco", "cobra-pernas", "cobra-rato", "cobra-ratos",
"cobra-veado", "cobra-vidro", "cobras-"+u"\u00E1"+"gua", "cobras-ar",
"cobras-areia", "cobras-asa", "cobras-barata", "cobras-cabelo",
"cobras-caju", "cobras-capelo", "cobras-capim", "cobras-cascavel",
"cobras-cip"+u"\u00F3", "cobras-colchete", "cobras-coral", "cobras-espada",
"cobras-ferradura", "cobras-lixo", "cobras-oco", "cobras-pernas",
"cobras-rato", "cobras-ratos", "cobras-veado", "cobras-vidro",
"coca-"+u"\u00E1"+"gua", "cocas-"+u"\u00E1"+"gua", "cochonilha-cera", "cochonilhas-cera",
"coco-airi", "coco-bacaia"+u"\u00FA"+"ba", "coco-bacaiuba", "coco-bocaiuva",
"coco-cachorro", "coco-catarro", "coco-cigano", "coco-colher",
"coco-dend"+u"\u00EA", "coco-espinho", "coco-guriri", "coco-indai"+u"\u00E1",
"coco-iri", "coco-macaco", "coco-nai"+u"\u00E1", "coco-natal",
"coco-oitava", "coco-"+u"\u00F3"+"leo", "coco-palmeira", "coco-pindoba",
"coco-purga", "coco-quaresma", "coco-quarta", "coco-ros"+u"\u00E1"+"rio",
"coco-vaqueiro", "coco-vassoura", "coco-veado", "coco-vinagre",
"cocos-bacaia"+u"\u00FA"+"ba", "cocos-bacaiuba", "cocos-bocaiuva", "cocos-cachorro",
"cocos-catarro", "cocos-cigano", "cocos-colher", "cocos-dend"+u"\u00EA",
"cocos-espinho", "cocos-guriri", "cocos-indai"+u"\u00E1", "cocos-iri",
"cocos-macaco", "cocos-nai"+u"\u00E1", "cocos-natal", "cocos-oitava",
"cocos-"+u"\u00F3"+"leo", "cocos-palmeira", "cocos-pindoba", "cocos-purga",
"cocos-quaresma", "cocos-quarta", "cocos-ros"+u"\u00E1"+"rio", "cocos-vaqueiro",
"cocos-vassoura", "cocos-veado", "cocos-vinagre", "coentro-caboclo",
"coentros-caboclos", "coerana-caravelas", "coerana-minas", "coerana-pernambuco",
"coeranas-caravelas", "coeranas-minas", "coeranas-pernambuco", "cogumelo-caboclo",
"cogumelo-chap"+u"\u00E9"+"u", "cogumelo-leite", "cogumelo-paris", "cogumelo-sangue",
"cogumelos-caboclo", "cogumelos-chap"+u"\u00E9"+"u", "cogumelos-leite", "cogumelos-paris",
"cogumelos-sangue", "coirana-caravelas", "coirana-minas", "coirana-pernambuco",
"coiranas-caravelas", "coiranas-minas", "coiranas-pernambuco", "cola-cavalo",
"cola-peixe", "cola-sapateiro", "cola-zorro", "colas-cavalo",
"colas-peixe", "colas-zorro", "coleira-sap"+u"\u00E9", "coleiras-sap"+u"\u00E9",
"coleiro-bando", "coleiro-choco", "coleiro-sap"+u"\u00E9", "coleiros-bando",
"coleiros-choco", "coleiros-sap"+u"\u00E9", "comer-arara", "comeres-arara",
"conduru-sangue", "condurus-sangue", "congonha-bugre", "congonha-caixeta",
"congonha-goi"+u"\u00E1"+"s", "congonhas-bugre", "congonhas-caixeta", "congonhas-goi"+u"\u00E1"+"s",
"conta-cabra", "contas-cabra", "copaibeira-minas", "copaibeiras-minas",
"coqueiro-dend"+u"\u00EA", "coqueiro-vassoura", "coqueiros-dend"+u"\u00EA", "coqueiros-vassoura",
"cor-rosa", "Cor-Rosa", "cord"+u"\u00E3"+"o-frade", "cord"+u"\u00F5"+"es-frade",
"correia-inverno", "correias-inverno", "corti"+u"\u00E7"+"a-montanha", "corti"+u"\u00E7"+"as-montanha",
"coruja-igreja", "coruj"+u"\u00E3"+"o-igreja", "corujas-igreja", "corvina-corso",
"corvina-linha", "corvinas-corso", "corvinas-linha", "cotovia-poupa",
"cotovias-poupa", "couve-adorno", "couve-"+u"\u00E1"+"gua", "couve-areia",
"couve-cortar", "couve-saboia", "couves-adorno", "couves-"+u"\u00E1"+"gua",
"couves-areia", "couves-cortar", "couves-saboia", "coxa-dama",
"coxa-dona", "coxa-freira", "coxas-dama",
"coxas-dona", "coxas-freira", "cravina-"+u"\u00E1"+"gua",
"cravina-lagartixa", "cravina-pau", "cravina-tunes", "cravinas-"+u"\u00E1"+"gua",
"cravinas-ambr"+u"\u00F3"+"sio", "cravinas-lagartixa", "cravinas-pau", "cravinas-tunes",
"cravinho-defunto", "cravinho-lagartixa", "cravinhos-campina", "cravinhos-defunto",
"cravinhos-lagartixa", "cravo-amor", "cravo-bast"+u"\u00E3"+"o", "cravo-bouba",
"cravo-burro", "cravo-cabe"+u"\u00E7"+"a", "cravo-cabecinha", "cravo-defunto",
"cravo-esperan"+u"\u00E7"+"a", "cravo-pau", "cravo-poeta", "cravo-seara",
"cravo-tunes", "cravo-urubu", "cravos-amor", "cravos-bast"+u"\u00E3"+"o",
"cravos-bouba", "cravos-burro", "cravos-cabe"+u"\u00E7"+"a", "cravos-cabecinha",
"cravos-defunto", "cravos-esperan"+u"\u00E7"+"a", "cravos-pau", "cravos-poeta",
"cravos-seara", "cravos-tunes", "cravos-urubu", "crista-galinha",
"crista-galo", "crista-mutum", "crista-peru", "cristais-rocha",
"cristal-rocha", "cristas-galinha", "cristas-galo", "cristas-mutum",
"cristas-peru", "cuia-macaco", "cuias-macaco", "cumbuca-macaco",
"cumbucas-macaco", "cutia-pau", "cutia-rabo", "cutias-pau",
"cutias-rabo", u"\u00E9"+"bano-zanzibar", u"\u00E9"+"banos-zanzibar", "embira-ca"+u"\u00E7"+"ador",
"embira-carrapato", "embira-porco", "embira-sapo", "embiras-ca"+u"\u00E7"+"ador",
"embiras-carrapato", "embiras-porco", "embiras-sapo", "enredadeira-borla",
"enredadeiras-borla", "enxerto-passarinho", "enxertos-passarinho", "erva-alfinete",
"erva-alho", "erva-alm"+u"\u00ED"+"scar", "erva-amor", "erva-andorinha",
"erva-anil", "erva-anta", "erva-b"+u"\u00E1"+"lsamos", "erva-basculho",
"erva-besteiros", "erva-bicho", "erva-bot"+u"\u00E3"+"o", "erva-bugre",
"erva-cabra", "erva-cabrita", "erva-c"+u"\u00E3"+"o", "erva-chumbo",
"erva-cobra", "erva-esteira", "erva-febra", "erva-fio",
"erva-fogo", "erva-fumo", "erva-gelo", "erva-goiabeira",
"erva-goma", "erva-gota", "erva-guin"+u"\u00E9", "erva-impingem",
"erva-jabuti", "erva-jararaca", "erva-laca", "erva-lagarto",
"erva-lavadeira", "erva-louco", "erva-maca"+u"\u00E9", "erva-malaca",
"erva-maleitas", "erva-mendigo", "erva-moira", "erva-moura",
"erva-mulher", "erva-muro", "erva-ouro", "erva-paina",
"erva-p"+u"\u00E2"+"ntano", "erva-parida", "erva-passarinho", "erva-piolho",
"erva-pontada", "erva-rato", "erva-roberto", "erva-salepo",
"erva-sangue", "erva-santana", "erva-santiago", "erva-sapo",
"ervas-alfinete", "ervas-alho", "ervas-amor", "ervas-andorinha",
"ervas-anil", "ervas-anjinho", "ervas-anta", "ervas-b"+u"\u00E1"+"lsamos",
"ervas-basculho", "ervas-besteiros", "ervas-bicho", "ervas-bot"+u"\u00E3"+"o",
"ervas-bugre", "ervas-cabra", "ervas-cabrita", "ervas-cheiro",
"ervas-chumbo", "ervas-cobra", "ervas-esteira", "ervas-febra",
"ervas-fio", "ervas-fogo", "ervas-fumo", "ervas-gelo",
"ervas-goiabeira", "ervas-goma", "ervas-gota", "ervas-guin"+u"\u00E9",
"ervas-impingem", "ervas-jabuti", "ervas-jararaca", "ervas-laca",
"ervas-lagarto", "ervas-lavadeira", "ervas-louco", "ervas-maca"+u"\u00E9",
"ervas-malaca", "ervas-maleitas", "ervas-mendigo", "ervas-moira",
"ervas-moura", "ervas-mulher", "ervas-muro", "ervas-ouro",
"ervas-paina", "ervas-p"+u"\u00E2"+"ntano", "ervas-parida", "ervas-passarinho",
"ervas-piolho", "ervas-pontada", "ervas-rato", "ervas-roberto",
"ervas-salepo", "ervas-sangue", "ervas-santana", "ervas-santiago",
"ervas-sapo", "ervilha-angola", "ervilha-"+u"\u00E1"+"rvore", "ervilha-cheiro",
"ervilha-lebre", "ervilha-pombo", "ervilha-vaca", "ervilhas-angola",
"ervilhas-"+u"\u00E1"+"rvore", "ervilhas-cheiro", "ervilhas-lebre", "ervilhas-pombo",
"ervilhas-vaca", "ervinha-parida", "ervinhas-parida", "escada-jabuti",
"escadas-jabuti", "escam"+u"\u00F4"+"nea-alepo", "escam"+u"\u00F4"+"neas-alepo", "escova-macaco",
"escovas-macaco", "escuma-sangue", "escumas-sangue", "espiga-"+u"\u00E1"+"gua",
"espiga-ferrugem", "espiga-leite", "espiga-sangue", "espigas-"+u"\u00E1"+"gua",
"espigas-ferrugem", "espigas-leite", "espigas-sangue", "espinafre-cuba",
"espinafres-cuba", "espinha-carneiro", "espinha-veado", "espinhas-carneiro",
"espinhas-veado", "espinheiro-agulha", "espinheiro-ameixa", "espinheiro-caiena",
"espinheiro-carneiro", "espinheiro-cerca", "espinheiro-cristo", "espinheiro-jerusal"+u"\u00E9"+"m",
"espinheiro-judeu", "espinheiro-vint"+u"\u00E9"+"m", "espinheiros-agulha", "espinheiros-ameixa",
"espinheiros-caiena", "espinheiros-carneiro", "espinheiros-cerca", "espinheiros-cristo",
"espinheiros-jerusal"+u"\u00E9"+"m", "espinheiros-judeu", "espinheiros-vint"+u"\u00E9"+"m", "espinho-agulha",
"espinho-bananeira", "espinho-ca"+u"\u00E7"+"ada", "espinho-cachorro", "espinho-carneiro",
"espinho-cerca", "espinho-cristo", "espinho-cruz", "espinho-deus",
"espinho-jerusal"+u"\u00E9"+"m", "espinho-judeu", "espinho-ladr"+u"\u00E3"+"o", "espinho-mariana",
"espinho-maric"+u"\u00E1", "espinho-roseta", "espinho-touro", "espinho-urso",
"espinho-vint"+u"\u00E9"+"m", "espinhos-agulha", "espinhos-bananeira", "espinhos-ca"+u"\u00E7"+"ada",
"espinhos-cachorro", "espinhos-carneiro", "espinhos-cerca", "espinhos-cristo",
"espinhos-cruz", "espinhos-deus", "espinhos-jerusal"+u"\u00E9"+"m", "espinhos-judeu",
"espinhos-ladr"+u"\u00E3"+"o", "espinhos-mariana", "espinhos-maric"+u"\u00E1", "espinhos-roseta",
"espinhos-touro", "espinhos-urso", "espinhos-vint"+u"\u00E9"+"m", "esponja-raiz",
"esponjas-raiz", "esterco-jurema", "estercos-jurema", "falc"+u"\u00E3"+"o-coleira",
"falc"+u"\u00F5"+"es-coleira", "fava-angola", "fava-arara", "fava-bel"+u"\u00E9"+"m",
"fava-besouro", "fava-bolacha", "fava-bolota", "fava-calabar",
"fava-cavalo", "fava-chapa", "fava-cobra", "fava-holanda",
"fava-impigem", "fava-impingem", "fava-lima", "fava-lobo",
"fava-malaca", "fava-ord"+u"\u00E1"+"lio", "fava-paca", "fava-porco",
"fava-quebranto", "fava-rama", "fava-rosca", "fava-sucupira",
"fava-tambaqui", "fava-tonca", "fava-vaca", "favas-angola",
"favas-arara", "favas-bel"+u"\u00E9"+"m", "favas-besouro", "favas-bolacha",
"favas-bolota", "favas-calabar", "favas-cavalo", "favas-chapa",
"favas-cheiro", "favas-cobra", "favas-engenho", "favas-holanda",
"favas-impigem", "favas-impingem", "favas-l"+u"\u00E1"+"zaro", "favas-lima",
"favas-lobo", "favas-malaca", "favas-ord"+u"\u00E1"+"lio", "favas-paca",
"favas-porco", "favas-quebranto", "favas-rama", "favas-rosca",
"favas-sucupira", "favas-tambaqui", "favas-tonca", "favas-vaca",
"faveira-berloque", "faveira-impigem", "faveira-impingem", "faveiras-berloque",
"faveiras-impigem", "faveiras-impingem", "favinha-capoeira", "favinhas-capoeira",
"feij"+u"\u00E3"+"o-"+u"\u00E1"+"rvore", "feij"+u"\u00E3"+"o-azeite", "feij"+u"\u00E3"+"o-boi", "feij"+u"\u00E3"+"o-cavalo",
"feij"+u"\u00E3"+"o-cera", "feij"+u"\u00E3"+"o-corda", "feij"+u"\u00E3"+"o-cuba", "feij"+u"\u00E3"+"o-frade",
"feij"+u"\u00E3"+"o-gado", "feij"+u"\u00E3"+"o-guando", "feij"+u"\u00E3"+"o-guizos", "feij"+u"\u00E3"+"o-java",
"feij"+u"\u00E3"+"o-leite", "feij"+u"\u00E3"+"o-lima", "feij"+u"\u00E3"+"o-lisboa", "feij"+u"\u00E3"+"o-mac"+u"\u00E1"+u"\u00E7"+"ar",
"feij"+u"\u00E3"+"o-metro", "feij"+u"\u00E3"+"o-pombinha", "feij"+u"\u00E3"+"o-porco", "feij"+u"\u00E3"+"o-rola",
"feij"+u"\u00E3"+"o-soja", "feij"+u"\u00E3"+"o-tropeiro", "feij"+u"\u00E3"+"o-vaca", "feij"+u"\u00E3"+"o-vagem",
"feij"+u"\u00E3"+"o-vara", "feij"+u"\u00E3"+"ozinho-capoeira", "feij"+u"\u00F5"+"es-"+u"\u00E1"+"rvore", "feij"+u"\u00F5"+"es-azeite",
"feij"+u"\u00F5"+"es-boi", "feij"+u"\u00F5"+"es-cavalo", "feij"+u"\u00F5"+"es-cera", "feij"+u"\u00F5"+"es-cobra",
"feij"+u"\u00F5"+"es-corda", "feij"+u"\u00F5"+"es-cuba", "feij"+u"\u00F5"+"es-frade", "feij"+u"\u00F5"+"es-gado",
"feij"+u"\u00F5"+"es-guando", "feij"+u"\u00F5"+"es-guizos", "feij"+u"\u00F5"+"es-java", "feij"+u"\u00F5"+"es-leite",
"feij"+u"\u00F5"+"es-lima", "feij"+u"\u00F5"+"es-lisboa", "feij"+u"\u00F5"+"es-mac"+u"\u00E1"+u"\u00E7"+"ar", "feij"+u"\u00F5"+"es-metro",
"feij"+u"\u00F5"+"es-pombinha", "feij"+u"\u00F5"+"es-porco", "feij"+u"\u00F5"+"es-rola", "feij"+u"\u00F5"+"es-soja",
"feij"+u"\u00F5"+"es-tropeiro", "feij"+u"\u00F5"+"es-vaca", "feij"+u"\u00F5"+"es-vagem", "feij"+u"\u00F5"+"es-vara",
"feij"+u"\u00F5"+"ezinhos-capoeira", "f"+u"\u00E9"+"is-gentio", "fel-gentio", "feles-gentio",
"feno-cheiro", "fenos-cheiro", "feto-bot"+u"\u00E3"+"o", "fetos-bot"+u"\u00E3"+"o",
"fiandeira-algod"+u"\u00E3"+"o", "fiandeiras-algod"+u"\u00E3"+"o", "figo-banana", "figo-figueira",
"figo-recheio", "figo-rocha", "figo-tordo", "figo-ver"+u"\u00E3"+"o",
"figos-banana", "figos-figueira", "figos-recheio", "figos-rocha",
"figos-tordo", "figos-ver"+u"\u00E3"+"o", "figueira-baco", "figueira-bengala",
"figueira-goa", "figueira-jardim", "figueira-lombrigueira", "figueira-pita",
"figueira-tocar", "figueira-toque", "figueiras-baco", "figueiras-bengala",
"figueiras-goa", "figueiras-jardim", "figueiras-lombrigueira", "figueiras-pita",
"figueiras-tocar", "figueiras-toque", "fil"+u"\u00E1"+"ria-guin"+u"\u00E9", "fil"+u"\u00E1"+"ria-medina",
"fil"+u"\u00E1"+"rias-guin"+u"\u00E9", "fil"+u"\u00E1"+"rias-medina", "flor-abril", "flor-"+u"\u00E1"+"gua",
"flor-algod"+u"\u00E3"+"o", "flor-anjinho", "flor-aranha", "flor-azar",
"flor-babado", "flor-babeiro", "flor-baile", "flor-barbeiro",
"flor-baunilha", "flor-besoiro", "flor-besouro", "flor-caboclo",
"flor-cachimbo", "flor-cal", "flor-camal", "flor-cardeal",
"flor-carnaval", "flor-casamento", "flor-cera", "flor-cetim",
"flor-chagas", "flor-ci"+u"\u00FA"+"me", "flor-cobra", "flor-coco",
"flor-coelho", "flor-coiro", "flor-contas", "flor-coral",
"flor-couro", "flor-cuco", "flor-diana", "flor-enxofre",
"flor-esp"+u"\u00ED"+"rito", "flor-fogo", "flor-gelo", "flor-grama",
"flor-graxa", "flor-"+u"\u00ED"+"ndio", "flor-inverno", "flor-jesus",
"flor-l"+u"\u00E3", "flor-lagartixa", "flor-lis", "flor-madeira",
"flor-maio", "flor-maracuj"+u"\u00E1", "flor-mariposa", "flor-merenda",
"flor-mico", "flor-moscada", "flor-musa", "flor-natal",
"flor-nespereira", "flor-noiva", "flor-padre", "flor-palha",
"flor-papagaio", "flor-para"+u"\u00ED"+"so", "flor-passarinho", "flor-pau",
"flor-pav"+u"\u00E3"+"o", "flor-pedra", "flor-pelicano", "flor-p"+u"\u00E9"+"rolas",
"flor-quaresma", "flor-sangue", "flor-sapato", "flor-sapo",
"flor-seda", "flor-sola", "flor-tanino", "flor-trombeta",
"flor-vaca", "flor-veado", "flor-veludo", "flor-vi"+u"\u00FA"+"va",
"flores-abril", "flores-"+u"\u00E1"+"gua", "flores-algod"+u"\u00E3"+"o", "flores-anjinho",
"flores-aranha", "flores-azar", "flores-babado", "flores-babeiro",
"flores-baile", "flores-barbeiro", "flores-baunilha", "flores-besoiro",
"flores-besouro", "flores-caboclo", "flores-cachimbo", "flores-cal",
"flores-camal", "flores-cardeal", "flores-carnaval", "flores-casamento",
"flores-cera", "flores-cetim", "flores-chagas", "flores-ci"+u"\u00FA"+"me",
"flores-cobra", "flores-coco", "flores-coelho", "flores-coiro",
"flores-contas", "flores-coral", "flores-couro", "flores-cuco",
"flores-diana", "flores-enxofre", "flores-esp"+u"\u00ED"+"rito", "flores-fogo",
"flores-gelo", "flores-grama", "flores-graxa", "flores-"+u"\u00ED"+"ndio",
"flores-inverno", "flores-jesus", "flores-l"+u"\u00E3", "flores-lagartixa",
"flores-lis", "flores-lisado", "flores-madeira", "flores-maio",
"flores-maracuj"+u"\u00E1", "flores-mariposa", "flores-merenda", "flores-mico",
"flores-moscada", "flores-musa", "flores-natal", "flores-nespereira",
"flores-noiva", "flores-padre", "flores-palha", "flores-papagaio",
"flores-para"+u"\u00ED"+"so", "flores-passarinho", "flores-pau", "flores-pav"+u"\u00E3"+"o",
"flores-pedra", "flores-pelicano", "flores-p"+u"\u00E9"+"rolas", "flores-quaresma",
"flores-sangue", "flores-sapato", "flores-sapo", "flores-seda",
"flores-sola", "flores-tanino", "flores-trombeta", "flores-vaca",
"flores-veado", "flores-veludo", "flores-vi"+u"\u00FA"+"va", "foca-capuz",
"focas-capuz", "folha-boldo", "folha-bolo", "folha-comich"+u"\u00E3"+"o",
"folha-f"+u"\u00ED"+"gado", "folha-figueira", "folha-fogo", "folha-fronte",
"folha-gelo", "folha-hera", "folha-lan"+u"\u00E7"+"a", "folha-leite",
"folha-louco", "folha-mangue", "folha-padre", "folha-paj"+u"\u00E9",
"folha-papagaio", "folha-pirarucu", "folha-sab"+u"\u00E3"+"o", "folha-sangue",
"folha-santana", "folha-serra", "folha-tara", "folha-urubu",
"folha-zebra", "folhas-boldo", "folhas-bolo", "folhas-comich"+u"\u00E3"+"o",
"folhas-f"+u"\u00ED"+"gado", "folhas-figueira", "folhas-fogo", "folhas-fronte",
"folhas-gelo", "folhas-hera", "folhas-lan"+u"\u00E7"+"a", "folhas-leite",
"folhas-louco", "folhas-mangue", "folhas-padre", "folhas-paj"+u"\u00E9",
"folhas-papagaio", "folhas-pirarucu", "folhas-sab"+u"\u00E3"+"o", "folhas-sangue",
"folhas-santana", "folhas-seda", "folhas-serra", "folhas-tara",
"folhas-urubu", "folhas-zebra", "formiga-bentinho", "formiga-bode",
"formiga-cemit"+u"\u00E9"+"rio", "formiga-cupim", "formiga-defunto", "formiga-febre",
"formiga-ferr"+u"\u00E3"+"o", "formiga-fogo", "formiga-imba"+u"\u00FA"+"va", "formiga-mandioca",
"formiga-monte", "formiga-n"+u"\u00F3"+"s", "formiga-novato", "formiga-on"+u"\u00E7"+"a",
"formiga-rabo", "formiga-raspa", "formiga-ro"+u"\u00E7"+"a", "formigas-bentinho",
"formigas-bode", "formigas-cemit"+u"\u00E9"+"rio", "formigas-cupim", "formigas-defunto",
"formigas-febre", "formigas-ferr"+u"\u00E3"+"o", "formigas-fogo", "formigas-imba"+u"\u00FA"+"va",
"formigas-mandioca", "formigas-monte", "formigas-n"+u"\u00F3"+"s", "formigas-novato",
"formigas-on"+u"\u00E7"+"a", "formigas-rabo", "formigas-raspa", "formigas-ro"+u"\u00E7"+"a",
"formosa-besteiros", "formosa-darei", "formosas-besteiros", "formosas-darei",
"forno-ja"+u"\u00E7"+"an"+u"\u00E3", "forno-jacar"+u"\u00E9", "fornos-ja"+u"\u00E7"+"an"+u"\u00E3", "fornos-jacar"+u"\u00E9",
"fruta-anamb"+u"\u00E9", "fruta-anel", "fruta-arara", "fruta-"+u"\u00E1"+"rvore",
"fruta-babado", "fruta-burro", "fruta-cachorro", "fruta-caiap"+u"\u00F3",
"fruta-cascavel", "fruta-codorna", "fruta-conde", "fruta-condessa",
"fruta-conta", "fruta-coruja", "fruta-cutia", "fruta-ema",
"fruta-fara"+u"\u00F3", "fruta-galo", "fruta-gentio", "fruta-grude",
"fruta-guar"+u"\u00E1", "fruta-guariba", "fruta-jacar"+u"\u00E9", "fruta-jacu",
"fruta-lepra", "fruta-lobo", "fruta-macaco", "fruta-manila",
"fruta-manteiga", "fruta-morcego", "fruta-p"+u"\u00E3"+"o", "fruta-papagaio",
"fruta-pav"+u"\u00E3"+"o", "fruta-perdiz", "fruta-pombo", "fruta-sab"+u"\u00E3"+"o",
"fruta-sabi"+u"\u00E1", "fruta-sa"+u"\u00ED"+"ra", "fruta-sapucainha", "fruta-tatu",
"fruta-tucano", "fruta-veado", "fruta-via", "fruta-v"+u"\u00ED"+"bora",
"frutas-anamb"+u"\u00E9", "frutas-anel", "frutas-arara", "frutas-"+u"\u00E1"+"rvore",
"frutas-babado", "frutas-burro", "frutas-cachorro", "frutas-caiap"+u"\u00F3",
"frutas-cascavel", "frutas-codorna", "frutas-conde", "frutas-condessa",
"frutas-conta", "frutas-coruja", "frutas-cutia", "frutas-ema",
"frutas-fara"+u"\u00F3", "frutas-galo", "frutas-gentio", "frutas-grude",
"frutas-guar"+u"\u00E1", "frutas-guariba", "frutas-jacar"+u"\u00E9", "frutas-jacu",
"frutas-lepra", "frutas-lobo", "frutas-macaco", "frutas-manila",
"frutas-manteiga", "frutas-morcego", "frutas-p"+u"\u00E3"+"o", "frutas-papagaio",
"frutas-pav"+u"\u00E3"+"o", "frutas-perdiz", "frutas-pombo", "frutas-sab"+u"\u00E3"+"o",
"frutas-sabi"+u"\u00E1", "frutas-sa"+u"\u00ED"+"ra", "frutas-sapucainha", "frutas-tatu",
"frutas-tucano", "frutas-veado", "frutas-via", "frutas-v"+u"\u00ED"+"bora",
"fruteira-arara", "fruteira-burro", "fruteira-cachorro", "fruteira-conde",
"fruteira-condessa", "fruteira-cutia", "fruteira-fara"+u"\u00F3", "fruteira-jacar"+u"\u00E9",
"fruteira-jacu", "fruteira-lobo", "fruteira-macaco", "fruteira-pav"+u"\u00E3"+"o",
"fruteira-perdiz", "fruteira-pombo", "fruteira-tucano", "fruteiras-arara",
"fruteiras-burro", "fruteiras-cachorro", "fruteiras-conde", "fruteiras-condessa",
"fruteiras-cutia", "fruteiras-fara"+u"\u00F3", "fruteiras-jacar"+u"\u00E9", "fruteiras-jacu",
"fruteiras-lobo", "fruteiras-macaco", "fruteiras-pav"+u"\u00E3"+"o", "fruteiras-perdiz",
"fruteiras-pombo", "fruteiras-tucano", "fruto-burro", "fruto-cachorro",
"fruto-caxinguel"+u"\u00EA", "fruto-cobra", "fruto-conta", "fruto-imb"+u"\u00E9",
"fruto-macaco", "fruto-manteiga", "fruto-morcego", "fruto-papagaio",
"fruto-perdiz", "fruto-sabi"+u"\u00E1", "frutos-burro", "frutos-cachorro",
"frutos-caxinguel"+u"\u00EA", "frutos-cobra", "frutos-conta", "frutos-imb"+u"\u00E9",
"frutos-macaco", "frutos-manteiga", "frutos-morcego", "frutos-papagaio",
"frutos-perdiz", "frutos-sabi"+u"\u00E1", "fumo-corda", "fumo-folha",
"fumo-jardim", "fumo-paisano", "fumo-para"+u"\u00ED"+"so", "fumo-raposa",
"fumos-corda", "fumos-folha", "fumos-jardim", "fumos-paisano",
"fumos-para"+u"\u00ED"+"so", "fumos-raposa", "funcho-"+u"\u00E1"+"gua", "funcho-floren"+u"\u00E7"+"a",
"funcho-porco", "funchos-"+u"\u00E1"+"gua", "funchos-floren"+u"\u00E7"+"a", "funchos-porco",
"gafanhoto-arriba"+u"\u00E7"+u"\u00E3"+"o", "gafanhoto-coqueiro", "gafanhoto-jurema", "gafanhoto-marmeleiro",
"gafanhoto-praga", "gafanhotos-arriba"+u"\u00E7"+u"\u00E3"+"o", "gafanhotos-coqueiro", "gafanhotos-jurema",
"gafanhotos-marmeleiro", "gafanhotos-praga", "galinha-"+u"\u00E1"+"gua", "galinha-bugre",
"galinha-fara"+u"\u00F3", "galinhas-"+u"\u00E1"+"gua", "galinhas-bugre", "galinhas-fara"+u"\u00F3",
"galo-bando", "galo-briga", "galo-eb"+u"\u00F3",
"galo-fita", "galo-penacho", "galo-pluma", "galo-rebanho",
"galos-bando", "galos-briga", "galos-eb"+u"\u00F3",
"galos-fita", "galos-penacho", "galos-pluma", "galos-rebanho",
"gameleira-cansa"+u"\u00E7"+"o", "gameleira-lombrigueira", "gameleira-pinga", "gameleira-purga",
"gameleira-veneno", "gameleiras-cansa"+u"\u00E7"+"o", "gameleiras-lombrigueiras", "gameleiras-pinga",
"gameleiras-purga", "gameleiras-veneno", "gargantas-ferro", "garoupa-segunda",
"garoupas-segunda", "gato-alg"+u"\u00E1"+"lia", "gato-madag"+u"\u00E1"+"scar", "gatos-alg"+u"\u00E1"+"lia",
"gavi"+u"\u00E3"+"o-anta", "gavi"+u"\u00E3"+"o-aru"+u"\u00E1", "gavi"+u"\u00E3"+"o-coleira", "gavi"+u"\u00E3"+"o-penacho",
"gavi"+u"\u00E3"+"o-queimada", "gavi"+u"\u00E3"+"o-serra", "gavi"+u"\u00F5"+"es-anta", "gavi"+u"\u00F5"+"es-aru"+u"\u00E1",
"gavi"+u"\u00F5"+"es-coleira", "gavi"+u"\u00F5"+"es-penacho", "gavi"+u"\u00F5"+"es-queimada", "gavi"+u"\u00F5"+"es-serra",
"gengibre-dourar", "gengibres-dourar", "gergelim-laguna", "gergelins-laguna",
"girass"+u"\u00F3"+"is-batatas", "girassol-batatas", "goela-lobo", "goelas-lobo",
"gog"+u"\u00F3"+"-guariba", "gog"+u"\u00F3"+"s-guariba", "goiaba-anta", "goiaba-espinho",
"goiaba-macaco", "goiabas-anta", "goiabas-espinho", "goiabas-macaco",
"goiabeira-cutia", "goiabeira-paca", "goiabeiras-cutia", "goiabeiras-paca",
"gomeiro-minas", "gomeiros-minas", "gralha-crista", "gralhas-crista",
"grama-adorno", "grama-cidade", "grama-coradouro", "grama-forquilha",
"grama-jacobina", "grama-jardim", "grama-maca"+u"\u00E9", "grama-maraj"+u"\u00F3",
"grama-pasto", "grama-pernambuco", "grama-ponta", "grama-sananduva",
"gramas-adorno", "gramas-cidade", "gramas-coradouro", "gramas-forquilha",
"gramas-jacobina", "gramas-jardim", "gramas-maca"+u"\u00E9", "gramas-maraj"+u"\u00F3",
"gramas-pasto", "gramas-pernambuco", "gramas-ponta", "gramas-sananduva",
"graminha-campinas", "graminha-jacobina", "graminha-raiz", "graminhas-campinas",
"graminhas-jacobina", "graminhas-raiz", "gr"+u"\u00E3"+"o-bico", "Gr"+u"\u00E3"+"o-Bico",
"gr"+u"\u00E3"+"o-cavalo", "gr"+u"\u00E3"+"o-galo", "gr"+u"\u00E3"+"o-maluco", "gr"+u"\u00E3"+"o-porco",
"gr"+u"\u00E3"+"o-pulha", "gr"+u"\u00E3"+"os-bico", "gr"+u"\u00E3"+"os-cavalo", "gr"+u"\u00E3"+"os-chumbo",
"gr"+u"\u00E3"+"os-galo", "gr"+u"\u00E3"+"os-maluco", "gr"+u"\u00E3"+"os-porco", "gr"+u"\u00E3"+"os-pulha",
"gr"+u"\u00E3"+"ozinho-galo", "gr"+u"\u00E3"+"ozinhos-galo", "gravat"+u"\u00E1"+"-agulha", "gravat"+u"\u00E1"+"-"+u"\u00E1"+"rvore",
"gravat"+u"\u00E1"+"-cerca", "gravat"+u"\u00E1"+"-coiro", "gravat"+u"\u00E1"+"-couro", "gravat"+u"\u00E1"+"-gancho",
"gravat"+u"\u00E1"+"-lajedo", "gravat"+u"\u00E1"+"-moqueca", "gravat"+u"\u00E1"+"-raposa", "gravat"+u"\u00E1"+"-rede",
"gravat"+u"\u00E1"+"-tingir", "gravat"+u"\u00E1"+"s-agulha", "gravat"+u"\u00E1"+"s-"+u"\u00E1"+"rvore", "gravat"+u"\u00E1"+"s-cerca",
"gravat"+u"\u00E1"+"s-coiro", "gravat"+u"\u00E1"+"s-couro", "gravat"+u"\u00E1"+"s-gancho", "gravat"+u"\u00E1"+"s-lajedo",
"gravat"+u"\u00E1"+"s-moqueca", "gravat"+u"\u00E1"+"s-raposa", "gravat"+u"\u00E1"+"s-rede", "gravat"+u"\u00E1"+"s-tingir",
"grinalda-vi"+u"\u00FA"+"va", "grinaldas-vi"+u"\u00FA"+"va", "guabiroba-cachorro", "guabiroba-minas",
"guabirobas-cachorro", "guabirobas-minas", "guabirobeira-cachorro", "guabirobeira-minas",
"guabirobeiras-cachorro", "guabirobeiras-minas", "guaricanga-bengala", "guaricangas-bengala",
"guirat"+u"\u00E3"+"-coqueiro", "guirat"+u"\u00E3"+"s-coqueiro", "hidra-"+u"\u00E1"+"gua", "hidras-"+u"\u00E1"+"gua",
"hortel"+u"\u00E3"+"-boi", "hortel"+u"\u00E3"+"-burro", "hortel"+u"\u00E3"+"-campina", "hortel"+u"\u00E3"+"-cavalo",
"hortel"+u"\u00E3"+"-cheiro", "hortel"+u"\u00E3"+"-leite", "hortel"+u"\u00E3"+"s-boi", "hortel"+u"\u00E3"+"s-burro",
"hortel"+u"\u00E3"+"s-campina", "hortel"+u"\u00E3"+"s-cavalo", "hortel"+u"\u00E3"+"s-cheiro", "hortel"+u"\u00E3"+"s-leite",
"imba"+u"\u00FA"+"ba-cheiro", "imba"+u"\u00FA"+"ba-vinho", "imba"+u"\u00FA"+"bas-cheiro", "imba"+u"\u00FA"+"bas-vinho",
"imb"+u"\u00E9"+"-amarrar", "imb"+u"\u00E9"+"-comer", "imb"+u"\u00E9"+"s-amarrar", "imb"+u"\u00E9"+"s-comer",
"imburana-camb"+u"\u00E3"+"o", "imburana-espinho", "imburanas-camb"+u"\u00E3"+"o", "imburanas-cheiro",
"imburanas-espinho", "imburi-cachorro", "imburis-cachorro", "incenso-caiena",
"incensos-caiena", "ing"+u"\u00E1"+"-fogo", "ing"+u"\u00E1"+"s-fogo", "inha"+u"\u00ED"+"ba-rego",
"inha"+u"\u00ED"+"bas-rego", "inhame-benim", "inhame-c"+u"\u00E3"+"o", "inhame-coriol"+u"\u00E1",
"inhame-enxerto", "inhame-lagartixa", "inhames-benim", "inhames-c"+u"\u00E3"+"o",
"inhames-coriol"+u"\u00E1", "inhames-enxerto", "inhames-lagartixa", "ip"+u"\u00EA"+"-impingem",
"ipeca-cuiab"+u"\u00E1", "ipecas-cuiab"+u"\u00E1", "ip"+u"\u00EA"+"s-impingem", "isca-sola",
"iscas-sola", "ixoras-cheiro", "jabuticaba-cip"+u"\u00F3", "jabuticabas-cip"+u"\u00F3",
"jabuticabeira-campinas", "jabuticabeiras-campinas", "jaca-pobre", "jacarand"+u"\u00E1"+"-campinas",
"jacarand"+u"\u00E1"+"-espinho", "jacarand"+u"\u00E1"+"-sangue", "jacarand"+u"\u00E1"+"s-campinas", "jacarand"+u"\u00E1"+"s-espinho",
"jacarand"+u"\u00E1"+"s-sangue", "jacar"+u"\u00E9"+"-"+u"\u00F3"+"culos", "jacar"+u"\u00E9"+"s-"+u"\u00F3"+"culos", "jacas-pobre",
"jacatir"+u"\u00E3"+"o-capote", "jacatir"+u"\u00F5"+"es-capote", "jalapa-lisboa", "jalapas-lisboa",
"jambeiro-malaca", "jambeiros-malaca", "jambo-malaca", "jambos-malaca",
"japu"+u"\u00E7"+u"\u00E1"+"-coleira", "jararaca-agosto", "jararacas-agosto", "jarrinha-franja",
"jarrinhas-franja", "jasmim-cachorro", "jasmim-caiena", "jasmim-cerca",
"jasmim-leite", "jasmim-soldado", "jasmins-cachorro", "jasmins-caiena",
"jasmins-cerca", "jasmins-leite", "jasmins-soldado", "jatob"+u"\u00E1"+"-anta",
"jatob"+u"\u00E1"+"-porco", "jatob"+u"\u00E1"+"s-anta", "jatob"+u"\u00E1"+"s-porco", "jequitib"+u"\u00E1"+"-agulheiro",
"jequitib"+u"\u00E1"+"s-agulheiro", "jeta"+u"\u00ED"+"-pernambuco", "jeta"+u"\u00ED"+"s-pernambuco", "jiquerana-goi"+u"\u00E1"+"s",
"jiqueranas-goi"+u"\u00E1"+"s", "jitirana-leite", "jitiranas-leite", "jo"+u"\u00E3"+"o-barro",
"jo"+u"\u00E3"+"o-leite", "jo"+u"\u00E3"+"o-pu"+u"\u00E7"+"a", "jo"+u"\u00E3"+"o-santar"+u"\u00E9"+"m", "jo"+u"\u00F5"+"es-barro",
"jo"+u"\u00F5"+"es-barros", "jo"+u"\u00F5"+"es-leite", "jo"+u"\u00F5"+"es-pu"+u"\u00E7"+"a", "jo"+u"\u00F5"+"es-santar"+u"\u00E9"+"m",
"ju"+u"\u00E1"+"-capote", "ju"+u"\u00E1"+"-comer", "ju"+u"\u00E1"+"s-capote", "ju"+u"\u00E1"+"s-comer",
"juciri-comer", "juciris-comer", "jun"+u"\u00E7"+"as-cheiro", "jun"+u"\u00E7"+"as-conta",
"junco-banhado", "junco-cabra", "junco-cangalha", "junco-cobra",
"juncos-banhado", "juncos-cabra", "juncos-cangalha", "juncos-cobra",
"junta-calangro", "junta-cobra", "juntas-calangro", "juntas-cobra",
"juqui-cerca", "juquis-cerca", "jurubeba-espinho", "jurubebas-espinho",
"lagarta-fogo", "lagarta-vidro", "lagartas-fogo", "lagartas-vidro",
"lagarto-"+u"\u00E1"+"gua", "lagartos-"+u"\u00E1"+"gua", "lagosta-espinho", "lagostas-espinho",
"lal"+u"\u00E1"+"-cintura", "lal"+u"\u00E1"+"s-cintura", "laranja-on"+u"\u00E7"+"a", "laranja-umbigo",
"laranjas-on"+u"\u00E7"+"a", "laranjas-terra", "laranjas-umbigo", "laranjeira-vaqueiro",
"laranjeiras-vaqueiro", "leite-cachorro", "leite-galinha", "leite-gameleira",
"leiteira-espinho", "leiteiras-espinho", "leites-cachorro", "leites-galinha",
"leites-gameleira", "leituga-burro", "leitugas-burro", "lesma-conchinha",
"lima-bico", "lima-umbigo", "lim"+u"\u00E3"+"o-caiena", "lim"+u"\u00E3"+"o-cheiro",
"lim"+u"\u00E3"+"o-galinha", "limas-bico", "limas-cheiro", "limas-umbigo",
"limeira-umbigo", "limeiras-umbigo", "limo-manta", "lim"+u"\u00F5"+"es-caiena",
"lim"+u"\u00F5"+"es-cheiro", "lim"+u"\u00F5"+"es-galinha", "limos-manta", "linguado-rio",
"linguados-rio", "lingueir"+u"\u00E3"+"o-canudo", "lingueir"+u"\u00F5"+"es-canudo", "linho-cuco",
"linho-raposa", "linhos-cuco", "linhos-raposa", "l"+u"\u00ED"+"rio-petr"+u"\u00F3"+"polis",
"l"+u"\u00ED"+"rios-petr"+u"\u00F3"+"polis", "lixa-lei", "lixa-pau", "lixas-lei",
"lixas-pau", "lixinha-fundura", "lixinhas-fundura", "loireiro-apolo",
"loireiros-apolo", "loureiro-apolo", "loureiro-jardim", "loureiros-apolo",
"loureiros-jardim", "louro-cheiro", "louros-cheiro", "lutos-quaresma",
"luvas-pastora", "luzerna-sequeiro", "luzernas-sequeiro", "ma"+u"\u00E7"+u"\u00E3"+"-an"+u"\u00E1"+"fega",
"ma"+u"\u00E7"+u"\u00E3"+"-arr"+u"\u00E1"+"tel", "ma"+u"\u00E7"+u"\u00E3"+"-boi", "ma"+u"\u00E7"+u"\u00E3"+"-caco", "ma"+u"\u00E7"+u"\u00E3"+"-cipreste",
"ma"+u"\u00E7"+u"\u00E3"+"-cobra", "ma"+u"\u00E7"+u"\u00E3"+"-cravo", "ma"+u"\u00E7"+u"\u00E3"+"-cuco", "ma"+u"\u00E7"+u"\u00E3"+"-espelho",
"ma"+u"\u00E7"+u"\u00E3"+"-prato", "ma"+u"\u00E7"+u"\u00E3"+"-rosa", "ma"+u"\u00E7"+u"\u00E3"+"-vime", "macaco-bando",
"macaco-cheiro", "macaco-noite", "macaco-sab"+u"\u00E1", "macacos-bando",
"macacos-cheiro", "macacos-noite", "macacos-sab"+u"\u00E1", "macambira-flecha",
"macambira-pedra", "macambira-serrote", "macambiras-flecha", "macambiras-pedra",
"macambiras-serrote", "macaquinho-bamb"+u"\u00E1", "macaquinhos-bamb"+u"\u00E1", "ma"+u"\u00E7"+u"\u00E3"+"s-an"+u"\u00E1"+"fega",
"ma"+u"\u00E7"+u"\u00E3"+"s-arr"+u"\u00E1"+"tel", "ma"+u"\u00E7"+u"\u00E3"+"s-boi", "ma"+u"\u00E7"+u"\u00E3"+"s-caco", "ma"+u"\u00E7"+u"\u00E3"+"s-cipreste",
"ma"+u"\u00E7"+u"\u00E3"+"s-cobra", "ma"+u"\u00E7"+u"\u00E3"+"s-cravo", "ma"+u"\u00E7"+u"\u00E3"+"s-cuco", "ma"+u"\u00E7"+u"\u00E3"+"s-espelho",
"ma"+u"\u00E7"+u"\u00E3"+"s-prato", "ma"+u"\u00E7"+u"\u00E3"+"s-rosa", "ma"+u"\u00E7"+u"\u00E3"+"s-vime", "macela-bot"+u"\u00E3"+"o",
"macela-tabuleiro", "macelas-bot"+u"\u00E3"+"o", "macelas-tabuleiro", "macieira-an"+u"\u00E1"+"fega",
"macieira-boi", "macieira-coroa", "macieiras-an"+u"\u00E1"+"fega", "macieiras-boi",
"macieiras-coroa", "macucu-paca", "macucus-paca", "madeira-rei",
"madeiras-rei", "madressilva-cheiro", "madressilvas-cheiro", "maitaca-maximiliano",
"maitacas-maximiliano", "malva-maraj"+u"\u00F3", "malva-pend"+u"\u00E3"+"o", "malva"+u"\u00ED"+"sco-pernambuco",
"malva"+u"\u00ED"+"scos-pernambuco", "malvas-cheiro", "malvas-maraj"+u"\u00F3", "malvas-pend"+u"\u00E3"+"o",
"mam"+u"\u00E3"+"o-corda", "mamica-cachorra", "mamica-cadela", "mamica-porca",
"mamicas-cachorra", "mamicas-cadela", "mamicas-porca", "mamoeiro-corda",
"mamoeiro-espinho", "mamoeiros-corda", "mamoeiros-espinho", "mam"+u"\u00F5"+"es-corda",
"mandacaru-boi", "mandacarus-boi", "mandibi-juntas", "mandibis-juntas",
"mangue-bot"+u"\u00E3"+"o", "mangue-espeto", "mangue-ob"+u"\u00F3", "mangue-pend"+u"\u00E3"+"o",
"mangue-sapateiro", "mangues-bot"+u"\u00E3"+"o", "mangues-espeto", "mangues-ob"+u"\u00F3",
"mangues-pend"+u"\u00E3"+"o", "mangues-sapateiro", "manjeric"+u"\u00E3"+"o-ceil"+u"\u00E3"+"o", "manjeric"+u"\u00E3"+"o-molho",
"manjeric"+u"\u00F5"+"es-ceil"+u"\u00E3"+"o", "manjeric"+u"\u00F5"+"es-molho", "manta-bret"+u"\u00E3"+"o", "mantas-bret"+u"\u00E3"+"o",
"mantimento-araponga", "mantimento-pobre", "mantimentos-araponga", "mantimentos-pobre",
"mapar"+u"\u00E1"+"-camet"+u"\u00E1", "mapar"+u"\u00E1"+"s-camet"+u"\u00E1", "maracuj"+u"\u00E1"+"-alho", "maracuj"+u"\u00E1"+"-cacho",
"maracuj"+u"\u00E1"+"-caiena", "maracuj"+u"\u00E1"+"-cheiro", "maracuj"+u"\u00E1"+"-cobra", "maracuj"+u"\u00E1"+"-corti"+u"\u00E7"+"a",
"maracuj"+u"\u00E1"+"-estalo", "maracuj"+u"\u00E1"+"-estrada", "maracuj"+u"\u00E1"+"-garapa", "maracuj"+u"\u00E1"+"-gaveta",
"maracuj"+u"\u00E1"+"-mochila", "maracuj"+u"\u00E1"+"-pedra", "maracuj"+u"\u00E1"+"-periquito", "maracuj"+u"\u00E1"+"-rato",
"maracuj"+u"\u00E1"+"s-alho", "maracuj"+u"\u00E1"+"s-cacho", "maracuj"+u"\u00E1"+"s-caiena", "maracuj"+u"\u00E1"+"s-cheiro",
"maracuj"+u"\u00E1"+"s-cobra", "maracuj"+u"\u00E1"+"s-corti"+u"\u00E7"+"a", "maracuj"+u"\u00E1"+"s-estalo", "maracuj"+u"\u00E1"+"s-estrada",
"maracuj"+u"\u00E1"+"s-garapa", "maracuj"+u"\u00E1"+"s-gaveta", "maracuj"+u"\u00E1"+"s-mochila", "maracuj"+u"\u00E1"+"s-pedra",
"maracuj"+u"\u00E1"+"s-periquito", "maracuj"+u"\u00E1"+"s-rato", "marimbondo-chap"+u"\u00E9"+"u", "marimbondos-chap"+u"\u00E9"+"u",
"marmelada-cachorro", "marmelada-cavalo", "marmelada-inverno", "marmelada-ver"+u"\u00E3"+"o",
"marmeladas-cachorro", "marmeladas-cavalo", "marmeladas-inverno", "marmeladas-ver"+u"\u00E3"+"o",
"marmeleiro-bengala", "marmeleiros-bengala", "marmelo-bengala", "marmelos-bengala",
"marmita-macaco", "marmitas-macaco", "marquesa-belas", "marquesas-belas",
"marreco-pequim", "marreco-ru"+u"\u00E3"+"o", "marrecos-pequim", "marrecos-ru"+u"\u00E3"+"o",
"matinho-agulhas", "matinho-lisboa", "matinho-sargo", "matinhos-agulhas",
"matinhos-lisboa", "matinhos-sargo", "mato-engodo", "mato-salema",
"matos-engodo", "matos-salema", "melancia-cobra", "melancias-cobra",
"mel"+u"\u00E3"+"o-caboclo", "mel"+u"\u00E3"+"o-morcego", "mel"+u"\u00E3"+"o-soldado", "meloeiro-soldado",
"meloeiros-soldado", "mel"+u"\u00F5"+"es-caboclo", "mel"+u"\u00F5"+"es-morcego", "mel"+u"\u00F5"+"es-soldado",
"melro-"+u"\u00E1"+"gua", "melro-coleira", "melros-"+u"\u00E1"+"gua", "melros-coleira",
"mico-topete", "micos-topete", "mijo-cavalo", "mijos-cavalo",
"milh"+u"\u00E3"+"-pend"+u"\u00E3"+"o", "milh"+u"\u00E3"+"s-pend"+u"\u00E3"+"o", "milho-"+u"\u00E1"+"gua", "milhos-"+u"\u00E1"+"gua",
"milhos-cobra", "mimosa-vereda", "mimosas-vereda", "mimoso-cacho",
"mimosos-cacho", "mineira-petr"+u"\u00F3"+"polis", "mineiras-petr"+u"\u00F3"+"polis", "miolos-capim",
"mocho-orelhas", "mochos-orelhas", "mocita"+u"\u00ED"+"ba-leite", "mocita"+u"\u00ED"+"bas-leite",
"moela-ema", "moela-mutum", "moelas-ema", "moelas-mutum",
"molong"+u"\u00F3"+"-colher", "molong"+u"\u00F3"+"s-colher", "monco-peru", "moncos-peru",
"monstro-gila", "monstros-gila", "monte-ouro", "montes-ouro",
"moreia-mangue", "moreias-mangue", "mosca-baga"+u"\u00E7"+"o", "mosca-banheiro",
"mosca-bicheira", "mosca-bicho", "mosca-casa", "mosca-cavalo",
"mosca-cavalos", "mosca-elefante", "mosca-est"+u"\u00E1"+"bulo", "mosca-fogo",
"mosca-madeira", "mosca-ura", "moscas-baga"+u"\u00E7"+"o", "moscas-banheiro",
"moscas-bicheira", "moscas-bicho", "moscas-casa", "moscas-cavalo",
"moscas-cavalos", "moscas-elefante", "moscas-est"+u"\u00E1"+"bulo", "moscas-fogo",
"moscas-freira", "moscas-inverno", "moscas-madeira", "moscas-ura",
"moscat"+u"\u00E9"+"is-jesus", "moscat"+u"\u00E9"+"is-set"+u"\u00FA"+"bal", "moscatel-jesus", "moscatel-set"+u"\u00FA"+"bal",
"mosquito-parede", "mosquitos-parede", "mourisco-semente", "mouriscos-semente",
"mu"+u"\u00E7"+"amb"+u"\u00E9"+"-espinhos", "mu"+u"\u00E7"+"amb"+u"\u00E9"+"s-espinhos", "mungunz"+u"\u00E1"+"-cortar", "mungunz"+u"\u00E1"+"s-cortar",
"murici-lenha", "murici-porco", "murici-tabuleiro", "muricis-lenha",
"muricis-porco", "muricis-tabuleiro", "murta-parida", "murtas-cheiro",
"murtas-parida", "murur"+u"\u00E9"+"-canudo", "murur"+u"\u00E9"+"-paj"+u"\u00E9", "murur"+u"\u00E9"+"s-canudo",
"murur"+u"\u00E9"+"s-paj"+u"\u00E9"+"s", "musaranho-"+u"\u00E1"+"gua", "musaranhos-"+u"\u00E1"+"gua", "musgo-irlanda",
"musgo-perd"+u"\u00E3"+"o", "musgos-irlanda", "musgos-perd"+u"\u00E3"+"o", "mutum-assobio",
"mutum-assovio", "mutum-fava", "mutuns-assobio", "mutuns-assovio",
"mutuns-fava", "nan"+u"\u00E1"+"-raposa", "nan"+u"\u00E1"+"s-raposa", "narciso-cheiro",
"narciso-inverno", "narcisos-cheiro", "navalha-macaco", "navalhas-macaco",
"nen"+u"\u00E9"+"m-galinha", "nen"+u"\u00E9"+"ns-galinha", "niquim-areia", "niquins-areia",
"niquins-saco", "nogueira-cobra", "nogueira-pec"+u"\u00E3", "nogueiras-cobra",
"nogueiras-pec"+u"\u00E3", "n"+u"\u00F3"+"s-cachorro", "noz-areca", "noz-banda",
"noz-batau"+u"\u00E1", "noz-bugre", "noz-cobra", "noz-coco",
"noz-cola", "noz-galha", "nozes-areca", "nozes-banda",
"nozes-batau"+u"\u00E1", "nozes-bugre", "nozes-cobra", "nozes-cola",
"nozes-galha", "oiti-porco", "oitis-porco", u"\u00F3"+"leo-copa"+u"\u00ED"+"ba",
u"\u00F3"+"leos-copa"+u"\u00ED"+"ba", "oliveira-cheiro", "oliveira-marrocos", "oliveiras-cheiro",
"oliveiras-marrocos", "ostra-pobre", "ostras-pobre", "pacas-rabo",
"paco-golungo", "pacos-golungo", "pacova-macaco", "pacovas-macaco",
"p"+u"\u00E3"+"es-cuco", "p"+u"\u00E3"+"es-galinha", "p"+u"\u00E3"+"es-galo", "p"+u"\u00E3"+"es-leite",
"p"+u"\u00E3"+"es-p"+u"\u00E1"+"ssaros", "p"+u"\u00E3"+"es-pobre", "p"+u"\u00E3"+"es-porcino", "p"+u"\u00E3"+"es-porco",
"paina-arbusto", "paina-cuba", "paina-penas", "paina-sapo",
"paina-seda", "painas-arbusto", "painas-cuba", "painas-penas",
"painas-sapo", "painas-seda", "paineira-cip"+u"\u00F3", "paineira-cuba",
"paineira-leite", "paineiras-cip"+u"\u00F3", "paineiras-cuba", "paineiras-leite",
"palha-"+u"\u00E1"+"gua", "palha-cani"+u"\u00E7"+"o", "palha-guin"+u"\u00E9", "palha-penacho",
"palhas-"+u"\u00E1"+"gua", "palhas-cani"+u"\u00E7"+"o", "palhas-guin"+u"\u00E9", "palhas-penacho",
"palma-cacho", "palma-chicote", "palma-igreja", "palma-leque",
"palmas-cacho", "palmas-chicote", "palmas-igreja", "palmas-leque",
"palmat"+u"\u00F3"+"ria-espinho", "palmat"+u"\u00F3"+"rias-espinho", "palmeira-cera", "palmeira-dend"+u"\u00EA",
"palmeira-igreja", "palmeira-leque", "palmeira-madagascar", "palmeira-palmira",
"palmeira-tebas", "palmeira-vinho", "palmeiras-cera", "palmeiras-dend"+u"\u00EA",
"palmeiras-igreja", "palmeiras-leque", "palmeiras-madagascar", "palmeiras-palmira",
"palmeiras-tebas", "palmeiras-vinho", "palmeirinha-petr"+u"\u00F3"+"polis", "palmeirinhas-petr"+u"\u00F3"+"polis",
"palmilha-papa", "palmilhas-papa", "palmito-ferr"+u"\u00E3"+"o", "palmitos-ferr"+u"\u00E3"+"o",
"palos-calenturas", "palos-maria", "panela-bugio", "panelas-bugio",
"p"+u"\u00E3"+"o-cuco", "p"+u"\u00E3"+"o-galinha", "p"+u"\u00E3"+"o-galo", "p"+u"\u00E3"+"o-p"+u"\u00E1"+"ssaros",
"p"+u"\u00E3"+"o-porcino", "p"+u"\u00E3"+"o-porco", "papagaio-coleira", "papagaio-colete",
"papagaios-coleira", "papagaios-colete", "papoila-espinho", "papoila-holanda",
"papoilas-espinho", "papoilas-holanda", "papoula-espinho", "papoula-holanda",
"papoulas-espinho", "papoulas-holanda", "paracuuba-lei", "paracuuba-leite",
"paracuubas-lei", "paracuubas-leite", "parasita-samambaia"+u"\u00E7"+"u", "parasitas-samambaia"+u"\u00E7"+"u",
"paratudo-praia", "paratudos-praia", "pargo-c"+u"\u00F3"+"tula", "pargo-mitra",
"pargo-morro", "pargos-c"+u"\u00F3"+"tula", "pargos-mitra", "pargos-morro",
"paric"+u"\u00E1"+"-curtume", "paric"+u"\u00E1"+"-esponjas", "paricarana-espinhos", "paricaranas-espinhos",
"paric"+u"\u00E1"+"s-curtume", "paric"+u"\u00E1"+"s-esponjas", "passarinho-arriba"+u"\u00E7"+u"\u00E3"+"o", "passarinho-ver"+u"\u00E3"+"o",
"passarinhos-arriba"+u"\u00E7"+u"\u00E3"+"o", "passarinhos-ver"+u"\u00E3"+"o", "p"+u"\u00E1"+"ssaro-fandango", "p"+u"\u00E1"+"ssaro-figo",
"p"+u"\u00E1"+"ssaros-fandango", "p"+u"\u00E1"+"ssaros-figo", "pati-goi"+u"\u00E1"+"s", "patinho-igap"+u"\u00F3",
"patinhos-igap"+u"\u00F3", "patis-goi"+u"\u00E1"+"s", "pato-caiena", "pato-car"+u"\u00FA"+"ncula",
"pato-crista", "patos-caiena", "patos-car"+u"\u00FA"+"ncula", "patos-crista",
"pau-ab"+u"\u00F3"+"bora", "pau-"+u"\u00E1"+"gua", "pau-alho", "pau-alo"+u"\u00E9",
"pau-angola", "pau-arara", "pau-arco", "pau-b"+u"\u00E1"+"lsamo",
"pau-balso", "pau-barbas", "pau-bicho", "pau-bico",
"pau-bilros", "pau-boia", "pau-bolo", "pau-boto",
"pau-breu", "pau-brincos", "pau-bugre", "pau-bujarrona",
"pau-burro", "pau-caca", "pau-cachimbo", "pau-caibro",
"pau-caixa", "pau-camar"+u"\u00E3"+"o", "pau-campeche", "pau-candeeiro",
"pau-cangalha", "pau-caninana", "pau-canoa", "pau-canoas",
"pau-cantil", "pau-capit"+u"\u00E3"+"o", "pau-carga", "pau-carne",
"pau-carrapato", "pau-castor", "pau-cera", "pau-cerca",
"pau-chanca", "pau-chapada", "pau-cheiro", "pau-chicle",
"pau-chifre", "pau-cinzas", "pau-cobre", "pau-colher",
"pau-concha", "pau-conta", "pau-coral", "pau-corti"+u"\u00E7"+"a",
"pau-cruz", "pau-cubi"+u"\u00FA", "pau-cunan"+u"\u00E3", "pau-curtir",
"pau-curtume", "pau-cutia", "pau-dedal", "pau-digest"+u"\u00E3"+"o",
"pau-elefante", "pau-embira", "pau-erva", "pau-ervilha",
"pau-espeto", "pau-espinho", "pau-esteira", "pau-faia",
"pau-farinha", "pau-fava", "pau-febre", "pau-ferro",
"pau-f"+u"\u00ED"+"gado", "pau-flecha", "pau-flechas", "pau-flor",
"pau-formiga", "pau-fumo", "pau-fuso", "pau-gasolina",
"pau-goma", "pau-guin"+u"\u00E9", "pau-guitarra", "pau-incenso",
"pau-jangada", "pau-jerimu", "pau-jerimum", "pau-judeu",
"pau-lacre", "pau-lagarto", "pau-lagoa", "pau-l"+u"\u00E1"+"grima",
"pau-lan"+u"\u00E7"+"a", "pau-lebre", "pau-leite", "pau-lepra",
"pau-letras", "pau-livros", "pau-lixa", "pau-macaco",
"pau-malho", "pau-manjerioba", "pau-manteiga", "pau-maria",
"pau-mastro", "pau-merda", "pau-moc"+u"\u00F3", "pau-monjolo",
"pau-moqu"+u"\u00E9"+"m", "pau-morcego", "pau-mutamba", "pau-navalha",
"pau-novato", "pau-olaria", "pau-"+u"\u00F3"+"leo", "pau-orvalho",
"pau-pente", "pau-pereira", "pau-pernambuco", "pau-pil"+u"\u00E3"+"o",
"pau-piranha", "pau-p"+u"\u00F3"+"lvora", "pau-pombo", "pau-porco",
"pau-praga", "pau-prego", "pau-pregui"+u"\u00E7"+"a", "pau-quiabo",
"pau-quina", "pau-rainha", "pau-rato", "pau-rede",
"pau-rego", "pau-reino", "pau-remo", "pau-renda",
"pau-resposta", "pau-rosa", "pau-rosas", "pau-roseira",
"pau-sab"+u"\u00E3"+"o", "pau-sabi"+u"\u00E1", "pau-salsa", "pau-sangue",
"pau-santana", "pau-santo", "pau-sapateiro", "pau-sapo",
"pau-sassafr"+u"\u00E1"+"s", "pau-sebo", "pau-seda", "pau-serra",
"pau-solar", "pau-surriola", "pau-tamanco", "pau-tartaruga",
"pau-tingui", "pau-tinta", "pau-tucano", "pau-vaca",
"pau-veado", "pau-veia", "pau-vinho", "pau-vint"+u"\u00E9"+"m",
"pau-viola", "pau-zebra", "paus-ab"+u"\u00F3"+"bora", "paus-"+u"\u00E1"+"gua",
"paus-alho", "paus-alo"+u"\u00E9", "paus-angola", "paus-arara",
"paus-arco", "paus-b"+u"\u00E1"+"lsamo", "paus-balso", "paus-barbas",
"paus-bicho", "paus-bico", "paus-bilros", "paus-boia",
"paus-bolo", "paus-boto", "paus-breu", "paus-brincos",
"paus-bugre", "paus-bujarrona", "paus-burro", "paus-caca",
"paus-cachimbo", "paus-caibro", "paus-caixa", "paus-camar"+u"\u00E3"+"o",
"paus-campeche", "paus-candeeiro", "paus-cangalha", "paus-caninana",
"paus-canoa", "paus-canoas", "paus-cantil", "paus-capit"+u"\u00E3"+"o",
"paus-carga", "paus-carne", "paus-carrapato", "paus-castor",
"paus-cera", "paus-cerca", "paus-chanca", "paus-chapada",
"paus-cheiro", "paus-chicle", "paus-chifre", "paus-cinzas",
"paus-cobre", "paus-colher", "paus-concha", "paus-conta",
"paus-coral", "paus-corti"+u"\u00E7"+"a", "paus-cruz", "paus-cubi"+u"\u00FA",
"paus-cunan"+u"\u00E3", "paus-curtir", "paus-curtume", "paus-cutia",
"paus-dedal", "paus-digest"+u"\u00E3"+"o", "paus-elefante", "paus-embira",
"paus-erva", "paus-ervilha", "paus-espeto", "paus-espinho",
"paus-esteira", "paus-faia", "paus-farinha", "paus-fava",
"paus-febre", "paus-ferro", "paus-f"+u"\u00ED"+"gado", "paus-flecha",
"paus-flechas", "paus-flor", "paus-formiga", "paus-fumo",
"paus-fuso", "paus-gasolina", "paus-goma", "paus-guin"+u"\u00E9",
"paus-guitarra", "paus-incenso", "paus-jangada", "paus-jerimu",
"paus-jerimum", "paus-judeu", "paus-lacre", "paus-lagarto",
"paus-lagoa", "paus-l"+u"\u00E1"+"grima", "paus-lan"+u"\u00E7"+"a", "paus-lebre",
"paus-leite", "paus-lepra", "paus-letras", "paus-livros",
"paus-lixa", "paus-macaco", "paus-malho", "paus-manjerioba",
"paus-manteiga", "paus-maria", "paus-mastro", "paus-merda",
"paus-moc"+u"\u00F3", "paus-monjolo", "paus-moqu"+u"\u00E9"+"m", "paus-morcego",
"paus-mutamba", "paus-navalha", "paus-novato", "paus-olaria",
"paus-"+u"\u00F3"+"leo", "paus-orvalho", "paus-pente", "paus-pereira",
"paus-pernambuco", "paus-pil"+u"\u00E3"+"o", "paus-piranha", "paus-p"+u"\u00F3"+"lvora",
"paus-pombo", "paus-porco", "paus-praga", "paus-prego",
"paus-pregui"+u"\u00E7"+"a", "paus-quiabo", "paus-quina", "paus-rainha",
"paus-rato", "paus-rede", "paus-rego", "paus-reino",
"paus-remo", "paus-renda", "paus-resposta", "paus-rosa",
"paus-rosas", "paus-roseira", "paus-sab"+u"\u00E3"+"o", "paus-sabi"+u"\u00E1",
"paus-salsa", "paus-sangue", "paus-santana", "paus-santo",
"paus-sapateiro", "paus-sapo", "paus-sassafr"+u"\u00E1"+"s", "paus-sebo",
"paus-seda", "paus-serra", "paus-solar", "paus-surriola",
"paus-tamanco", "paus-tartaruga", "paus-tingui", "paus-tinta",
"paus-tucano", "paus-vaca", "paus-veado", "paus-veia",
"paus-vinho", "paus-vint"+u"\u00E9"+"m", "paus-viola", "paus-zebra",
"pauxi-pedra", "pauxis-pedra", "pav"+u"\u00E3"+"o-java", "pav"+u"\u00F5"+"es-java",
"penacho-capim", "penachos-capim", "penas-avestruz", "pepino-burro",
"pepino-papagaio", "pepinos-burro", "pepinos-papagaio", "pequi"+u"\u00E1"+"-pedra",
"pequi"+u"\u00E1"+"s-pedra", "pera-advogado", "pera-"+u"\u00E1"+"gua", "pera-aguieira",
"pera-almeida", "pera-cristo", "pera-jesus", "pera-refego",
"pera-rei", "pera-rosa", "pera-unto", "peras-advogado",
"peras-"+u"\u00E1"+"gua", "peras-aguieira", "peras-almeida", "peras-cristo",
"peras-jesus", "peras-refego", "peras-rei", "peras-rosa",
"peras-unto", "percevejo-cama", "percevejo-casa", "percevejo-galinha",
"percevejos-cama", "percevejos-casa", "percevejos-galinha", "perdiz-alqueive",
"perdizes-alqueive", "periquitinho-vassoura", "periquitinhos-vassoura", "periquito-anta",
"periquitos-anta", "periquitos-campina", "peroba-campos", "peroba-cantagalo",
"peroba-goi"+u"\u00E1"+"s", "peroba-gomo", "peroba-minas", "peroba-pernambuco",
"peroba-rego", "perobas-campos", "perobas-cantagalo", "perobas-goi"+u"\u00E1"+"s",
"perobas-gomo", "perobas-minas", "perobas-pernambuco", "perobas-rego",
"peru-sol", "perus-sol", "p"+u"\u00EA"+"ssego-abrir", "p"+u"\u00EA"+"ssegos-abrir",
"pessegueiro-abrir", "pessegueiros-abrir", "pi"+u"\u00E3"+"o-purga", "pic"+u"\u00E3"+"o-tropeiro",
"pic"+u"\u00F5"+"es-trepar", "pic"+u"\u00F5"+"es-tropeiro", "pimenta-"+u"\u00E1"+"gua", "pimenta-bugre",
"pimenta-buta", "pimenta-caiena", "pimenta-cheiro", "pimenta-coroa",
"pimenta-galinha", "pimenta-gentio", "pimenta-macaco", "pimenta-queimar",
"pimenta-rabo", "pimenta-rato", "piment"+u"\u00E3"+"o-caiena", "piment"+u"\u00E3"+"o-cheiro",
"pimentas-"+u"\u00E1"+"gua", "pimentas-bugre", "pimentas-buta", "pimentas-caiena",
"pimentas-cheiro", "pimentas-coroa", "pimentas-galinha", "pimentas-gentio",
"pimentas-macaco", "pimentas-queimar", "pimentas-rabo", "pimentas-rato",
"piment"+u"\u00F5"+"es-caiena", "piment"+u"\u00F5"+"es-cheiro", "pinha-raiz", "pinh"+u"\u00E3"+"o-cerca",
"pinh"+u"\u00E3"+"o-madagascar", "pinh"+u"\u00E3"+"o-purga", "pinh"+u"\u00E3"+"o-rato", "pinhas-raiz",
"pinheirinho-jardim", "pinheirinho-sala", "pinheirinhos-jardim", "pinheirinhos-sala",
"pinheiro-alepo", "pinheiro-casquinha", "pinheiro-jerusal"+u"\u00E9"+"m", "pinheiro-purga",
"pinheiro-riga", "pinheiros-alepo", "pinheiros-casquinha", "pinheiros-jerusal"+u"\u00E9"+"m",
"pinheiros-purga", "pinheiros-riga", "pinho-riga", "pinh"+u"\u00F5"+"es-cerca",
"pinh"+u"\u00F5"+"es-madagascar", "pinh"+u"\u00F5"+"es-purga", "pinh"+u"\u00F5"+"es-rato", "pinh"+u"\u00F5"+"es-ratos",
"pinhos-flandres", "pinhos-riga", "piolho-baleia", "piolho-ca"+u"\u00E7"+u"\u00E3"+"o",
"piolho-cobra", "piolho-fara"+u"\u00F3", "piolho-galinha", "piolho-on"+u"\u00E7"+"a",
"piolho-padre", "piolho-planta", "piolho-soldado", "piolho-tubar"+u"\u00E3"+"o",
"piolho-urubu", "piolhos-baleia", "piolhos-ca"+u"\u00E7"+u"\u00E3"+"o", "piolhos-cobra",
"piolhos-fara"+u"\u00F3", "piolhos-galinha", "piolhos-on"+u"\u00E7"+"a", "piolhos-padre",
"piolhos-planta", "piolhos-soldado", "piolhos-tubar"+u"\u00E3"+"o", "piolhos-urubu",
"pipira-m"+u"\u00E1"+"scara", "pipira-prata", "pipiras-m"+u"\u00E1"+"scara", "pipiras-prata",
"piqui"+u"\u00E1"+"-pedra", "piqui"+u"\u00E1"+"s-pedra", "pira"+u"\u00ED"+"ba-pele", "pira"+u"\u00ED"+"bas-pele",
"pita-erva", "pitanga-cachorro", "pitangas-cachorro", "pitangueira-cachorro",
"pitangueira-jardim", "pitangueiras-cachorro", "pitangueiras-jardim", "pitas-erva",
"piteira-sinal", "piteiras-sinal", "pito-"+u"\u00E1"+"gua", "pito-saci",
"pitomba-leite", "pitomba-macaco", "pitombas-leite", "pitombas-macaco",
"pitombeira-maraj"+u"\u00F3", "pitombeiras-maraj"+u"\u00F3", "pitos-"+u"\u00E1"+"gua", "pitos-saci",
"pixirica-bolas", "pixiricas-bolas", "planta-neve", "plantas-neve",
"pl"+u"\u00E1"+"tano-g"+u"\u00EA"+"nio", "pl"+u"\u00E1"+"tanos-g"+u"\u00EA"+"nio", "pluma-capim", "pluma-pr"+u"\u00ED"+"ncipe",
"plumas-capim", "plumas-pr"+u"\u00ED"+"ncipe", "poaia-cip"+u"\u00F3", "poaia-minas",
"poaias-cip"+u"\u00F3", "poaias-minas", "pomba-arriba"+u"\u00E7"+u"\u00E3"+"o", "pomba-bando",
"pomba-espelho", "pomba-mulata", "pomba-sert"+u"\u00E3"+"o", "pombas-arriba"+u"\u00E7"+u"\u00E3"+"o",
"pombas-bando", "pombas-espelho", "pombas-mulata", "pombas-sert"+u"\u00E3"+"o",
"pombo-leque", "pombo-montanha", "pombos-leque", "pombos-montanha",
"porco-ferro", "porco-verrugas", "porcos-ferro", "porcos-verrugas",
"pregui"+u"\u00E7"+"a-bentinho", "pregui"+u"\u00E7"+"a-coleira", "pregui"+u"\u00E7"+"as-bentinho", "pregui"+u"\u00E7"+"as-coleira",
"primavera-caiena", "primaveras-caiena", "pulga-"+u"\u00E1"+"gua", "pulga-anta",
"pulga-areia", "pulga-bicho", "pulga-galinha", "pulg"+u"\u00E3"+"o-planta",
"pulgas-"+u"\u00E1"+"gua", "pulgas-anta", "pulgas-areia", "pulgas-bicho",
"pulgas-galinha", "pulg"+u"\u00F5"+"es-planta", "purga-caboclo", "purga-caiap"+u"\u00F3",
"purga-caitit"+u"\u00E9", "purga-carij"+u"\u00F3", "purga-cavalo", "purga-cereja",
"purga-gentio", "purga-marinheiro", "purga-nabi"+u"\u00E7"+"a", "purga-pastor",
"purga-paulista", "purga-veado", "purga-vento", "purgas-caboclo",
"purgas-caiap"+u"\u00F3", "purgas-caitit"+u"\u00E9", "purgas-carij"+u"\u00F3", "purgas-cavalo",
"purgas-cereja", "purgas-gentio", "purgas-marinheiro", "purgas-nabi"+u"\u00E7"+"a",
"purgas-pastor", "purgas-paulista", "purgas-veado", "purgas-vento",
"qu"+u"\u00E1"+"ssia-caiena", "qu"+u"\u00E1"+"ssias-caiena", "quati-bando", "quati-vara",
"quatis-bando", "quatis-vara", "quiabeiro-angola", "quiabeiros-angola",
"quiabo-angola", "quiabo-caiena", "quiabo-cheiro", "quiabo-cip"+u"\u00F3",
"quiabo-quina", "quiaborana-espinho", "quiaboranas-espinho", "quiabos-angola",
"quiabos-caiena", "quiabos-cheiro", "quiabos-cip"+u"\u00F3", "quiabos-quina",
"quibondo-"+u"\u00E1"+"gua", "quibondos-"+u"\u00E1"+"gua", "quigomb"+u"\u00F3"+"-cheiro", "quigomb"+u"\u00F3"+"s-cheiro",
"quina-caiena", "quina-cerrado", "quina-cip"+u"\u00F3", "quina-goi"+u"\u00E1"+"s",
"quina-periquito", "quina-pernambuco", "quina-raiz", "quina-rego",
"quina-rem"+u"\u00ED"+"gio", "quina-veado", "quinas-caiena", "quinas-cerrado",
"quinas-cip"+u"\u00F3", "quinas-goi"+u"\u00E1"+"s", "quinas-periquito", "quinas-pernambuco",
"quinas-raiz", "quinas-rego", "quinas-rem"+u"\u00ED"+"gio", "quinas-veado",
"quingomb"+u"\u00F3"+"-cheiro", "quingomb"+u"\u00F3"+"-espinho", "quingomb"+u"\u00F3"+"s-cheiro", "quingomb"+u"\u00F3"+"s-espinho",
"quito-pernambuco", "rabanete-cavalo", "rabanetes-cavalo", "r"+u"\u00E1"+"bano-cavalo",
"r"+u"\u00E1"+"banos-cavalo", "rabugem-cachorro", "rabugens-cachorro", "raiz-ang"+u"\u00E9"+"lica",
"raiz-babeiro", "raiz-barbeiro", "raiz-brand"+u"\u00E3"+"o", "raiz-bugre",
"raiz-c"+u"\u00E2"+"maras", "raiz-cana", "raiz-cedro", "raiz-ch"+u"\u00E1",
"raiz-cheiro", "raiz-cobra", "raiz-corvo", "raiz-curvo",
"raiz-fel", "raiz-frade", "raiz-guin"+u"\u00E9", "raiz-lagarto",
"raiz-lopes", "raiz-mostarda", "raiz-pipi", "raiz-quina",
"raiz-sol", "raiz-solteira", "raiz-ti"+u"\u00FA", "raiz-tucano",
"ra"+u"\u00ED"+"zes-ang"+u"\u00E9"+"lica", "ra"+u"\u00ED"+"zes-babeiro", "ra"+u"\u00ED"+"zes-barbeiro", "ra"+u"\u00ED"+"zes-brand"+u"\u00E3"+"o",
"ra"+u"\u00ED"+"zes-bugre", "ra"+u"\u00ED"+"zes-c"+u"\u00E2"+"maras", "ra"+u"\u00ED"+"zes-cana", "ra"+u"\u00ED"+"zes-cedro",
"ra"+u"\u00ED"+"zes-ch"+u"\u00E1", "ra"+u"\u00ED"+"zes-cheiro", "ra"+u"\u00ED"+"zes-cobra", "ra"+u"\u00ED"+"zes-corvo",
"ra"+u"\u00ED"+"zes-curvo", "ra"+u"\u00ED"+"zes-fel", "ra"+u"\u00ED"+"zes-frade", "ra"+u"\u00ED"+"zes-guin"+u"\u00E9",
"ra"+u"\u00ED"+"zes-lagarto", "ra"+u"\u00ED"+"zes-lopes", "ra"+u"\u00ED"+"zes-mostarda", "ra"+u"\u00ED"+"zes-pipi",
"ra"+u"\u00ED"+"zes-quina", "ra"+u"\u00ED"+"zes-sol", "ra"+u"\u00ED"+"zes-solteira", "ra"+u"\u00ED"+"zes-ti"+u"\u00FA",
"ra"+u"\u00ED"+"zes-tucano", "rama-bezerro", "ramas-bezerro", "ramo-seda",
"ramos-seda", "rap"+u"\u00E9"+"-saci", "rap"+u"\u00E9"+"s-saci", "rato-"+u"\u00E1"+"gua",
"rato-algod"+u"\u00E3"+"o", "rato-bambu", "rato-casa", "rato-couro",
"rato-esgoto", "rato-espinho", "rato-fara"+u"\u00F3", "rato-fava",
"rato-paiol", "rato-palmat"+u"\u00F3"+"ria", "rato-pentes", "rato-praga",
"rato-taquara", "rato-tromba", "ratos-"+u"\u00E1"+"gua", "ratos-algod"+u"\u00E3"+"o",
"ratos-bambu", "ratos-casa", "ratos-couro", "ratos-esgoto",
"ratos-espinho", "ratos-fara"+u"\u00F3", "ratos-fava", "ratos-paiol",
"ratos-palmat"+u"\u00F3"+"ria", "ratos-pentes", "ratos-praga", "ratos-taquara",
"ratos-tromba", "rede-le"+u"\u00E3"+"o", "redes-le"+u"\u00E3"+"o", "resed"+u"\u00E1"+"-cheiro",
"resed"+u"\u00E1"+"s-cheiro", "rinoceronte-Java", "rinocerontes-Java", "roaz-bandeira",
"roazes-bandeira", "robalo-areia", "robalo-galha", "robalos-areia",
"robalos-galha", "roca-eva", "roca-flores", "rocas-eva",
"rocas-flores", "rosa-bobo", "rosa-caboclo", "rosa-cachorro",
"rosa-c"+u"\u00E3"+"o", "rosa-ch"+u"\u00E3"+"o", "rosa-defunto", "rosa-jeric"+u"\u00F3",
"rosa-lobo", "rosa-musgo", "rosa-oiro", "rosa-ouro",
"rosa-p"+u"\u00E1"+"scoa", "rosa-pau", "rosa-toucar", "ros"+u"\u00E1"+"rio-if"+u"\u00E1",
"ros"+u"\u00E1"+"rio-jambo", "ros"+u"\u00E1"+"rio-jambu", "ros"+u"\u00E1"+"rios-if"+u"\u00E1", "ros"+u"\u00E1"+"rios-jambo",
"ros"+u"\u00E1"+"rios-jambu", "rosas-bobo", "rosas-caboclo", "rosas-cachorro",
"rosas-c"+u"\u00E3"+"o", "rosas-ch"+u"\u00E3"+"o", "rosas-defunto", "rosas-jeric"+u"\u00F3",
"rosas-lobo", "rosas-musgo", "rosas-oiro", "rosas-ouro",
"rosas-p"+u"\u00E1"+"scoa", "rosas-toucar", "roseira-damasco", "roseiras-damasco",
"roseta-pernambuco", "rosetas-pernambuco", "rouxin"+u"\u00F3"+"is-cani"+u"\u00E7"+"os", "rouxin"+u"\u00F3"+"is-espadana",
"rouxin"+u"\u00F3"+"is-java", "rouxin"+u"\u00F3"+"is-manaus", "rouxin"+u"\u00F3"+"is-muralha", "rouxinol-cani"+u"\u00E7"+"os",
"rouxinol-espadana", "rouxinol-java", "rouxinol-manaus", "rouxinol-muralha",
"sabacu-coroa", "sabacus-coroa", "sagui-bigode", "saguim-bigode",
"saguins-bigode", "saguis-bigode", "salamandra-"+u"\u00E1"+"gua", "salamandras-"+u"\u00E1"+"gua",
"salsa-"+u"\u00E1"+"gua", "salsa-burro", "salsa-castanheiro", "salsa-cavalo",
"salsa-cavalos", "salsa-cupim", "salsa-rocha", "salsaparrilha-lisboa",
"salsaparrilhas-lisboa", "salsas-"+u"\u00E1"+"gua", "salsas-burro", "salsas-castanheiro",
"salsas-cavalos", "salsas-cheiro", "salsas-cupim", "salsas-rocha",
"salva-maraj"+u"\u00F3", "salva-pernambuco", "salvas-maraj"+u"\u00F3", "salvas-pernambuco",
"samambaia-penacho", "samambaias-penacho", "san"+u"\u00E3"+"-samambaia", "san"+u"\u00E3"+"s-samambaia",
"sanha"+u"\u00E7"+"o-coqueiro", "sanha"+u"\u00E7"+"o-encontros", "sanha"+u"\u00E7"+"o-fogo", "sanha"+u"\u00E7"+"o-mamoeiro",
"sanha"+u"\u00E7"+"os-coqueiro", "sanha"+u"\u00E7"+"os-encontros", "sanha"+u"\u00E7"+"os-fogo", "sanha"+u"\u00E7"+"os-mamoeiro",
"sanha"+u"\u00E7"+"u-encontro", "sanha"+u"\u00E7"+"u-mamoeiro", "sanha"+u"\u00E7"+"us-encontro", "sanha"+u"\u00E7"+"us-mamoeiro",
"sap"+u"\u00E9"+"-capoeira", "sap"+u"\u00E9"+"s-capoeira", "sapo-chifre", "sapo-chifres",
"sapos-chifre", "sapucaia-castanha", "sapucaias-castanha", "sara-pita",
"saramago-"+u"\u00E1"+"gua", "saramagos-"+u"\u00E1"+"gua", "sarandi-caranguejo", "sarandi-espinho",
"sarandi-gargarejo", "sarandis-caranguejo", "sarandis-espinho", "sarandis-gargarejo",
"saras-pita", "sardinha-galha", "sardinha-gato", "sardinha-laje",
"sardinhas-galha", "sardinhas-gato", "sardinhas-laje", "sargo-bei"+u"\u00E7"+"o",
"sargo-dente", "sargos-bei"+u"\u00E7"+"o", "sargos-dente", "sarro-pito",
"sarros-pito", "savacu-coroa", "savacus-coroa", "siri-coral",
"siris-coral", "sorgo-alepo", "sorgo-espiga", "sorgo-pincel",
"sorgo-vassoura", "sorgos-alepo", "sorgos-espiga", "sorgos-pincel",
"sorgos-vassoura", "sumagre-proven"+u"\u00E7"+"a", "sumagres-proven"+u"\u00E7"+"a", "sumar"+u"\u00E9"+"-pedras",
"sumar"+u"\u00E9"+"s-pedras", "sumaumeira-macaco", "sumaumeiras-macaco", "surucucu-fogo",
"surucucu-patioba", "surucucu-pindoba", "surucucus-fogo", "surucucus-patioba",
"surucucus-pindoba", "tabaco-c"+u"\u00E3"+"o", "tabacos-c"+u"\u00E3"+"o", "taboca-maraj"+u"\u00F3",
"tabocas-maraj"+u"\u00F3", "tainha-corso", "tainha-rio", "tainhas-corso",
"tainhas-rio", "taiui"+u"\u00E1"+"-cip"+u"\u00F3", "taiui"+u"\u00E1"+"-comer", "taiui"+u"\u00E1"+"-goi"+u"\u00E1"+"s",
"taiui"+u"\u00E1"+"-jardim", "taiui"+u"\u00E1"+"-pimenta", "taiui"+u"\u00E1"+"-quiabo", "taiui"+u"\u00E1"+"s-cip"+u"\u00F3",
"taiui"+u"\u00E1"+"s-comer", "taiui"+u"\u00E1"+"s-goi"+u"\u00E1"+"s", "taiui"+u"\u00E1"+"s-jardim", "taiui"+u"\u00E1"+"s-pimenta",
"taiui"+u"\u00E1"+"s-quiabo", "taj"+u"\u00E1"+"-cobra", "taj"+u"\u00E1"+"s-cobra", "tajuj"+u"\u00E1"+"-cabacinho",
"tajuj"+u"\u00E1"+"-quiabo", "tajuj"+u"\u00E1"+"s-cabacinho", "tajuj"+u"\u00E1"+"s-quiabo", "tamanqueira-leite",
"tamanqueiras-leite", "tambetaru-espinho", "tambetarus-espinho", "tamboril-bravo",
"tamboris-bravo", "taquari-cavalo", "taquaris-cavalo", "tartaruga-couro",
"tartaruga-pente", "tartarugas-couro", "tartarugas-pente", "tarum"+u"\u00E3"+"-espinho",
"tarum"+u"\u00E3"+"-espinhos", "tarum"+u"\u00E3"+"s-espinho", "tarum"+u"\u00E3"+"s-espinhos", "tatu-folha",
"tatus-folha", "taxizeiro-tinta", "taxizeiros-tinta", "tembetaru-espinho",
"tembetarus-espinho", "tempor"+u"\u00E3"+"o-coruche", "tempor"+u"\u00E3"+"os-coruche", "terebintina-quio",
"tigre-bengala", "tigres-bengala", "timb"+u"\u00F3"+"-botic"+u"\u00E1"+"rio", "timb"+u"\u00F3"+"-caiena",
"timb"+u"\u00F3"+"-jacar"+u"\u00E9", "timb"+u"\u00F3"+"-peixe", "timb"+u"\u00F3"+"-raiz", "timb"+u"\u00F3"+"s-botic"+u"\u00E1"+"rio",
"timb"+u"\u00F3"+"s-caiena", "timb"+u"\u00F3"+"s-jacar"+u"\u00E9", "timb"+u"\u00F3"+"s-peixe", "timb"+u"\u00F3"+"s-raiz",
"tingui-caiena", "tingui-cip"+u"\u00F3", "tingui-cola", "tingui-leite",
"tingui-peixe", "tinguis-caiena", "tinguis-cip"+u"\u00F3", "tinguis-cola",
"tinguis-leite", "tinguis-peixe", "tinhor"+u"\u00E3"+"o-lombriga", "tinhor"+u"\u00F5"+"es-lombriga",
"toadas-couro", "tomate-"+u"\u00E1"+"rvore", "tomate-princesa", "tomates-"+u"\u00E1"+"rvore",
"tomates-princesa", "tomatinho-capucho", "tomatinhos-capucho", "tomilho-creta",
"tomilhos-creta", "topete-cardeal", "topetes-cardeal", "tor"+u"\u00F3"+"-espinho",
"tor"+u"\u00F3"+"s-espinho", "tra"+u"\u00E7"+"a-biblioteca", "tra"+u"\u00E7"+"as-biblioteca", "tremo"+u"\u00E7"+"o-cheiro",
"tremo"+u"\u00E7"+"o-jardim", "tremo"+u"\u00E7"+"o-minas", "tremo"+u"\u00E7"+"os-cheiro", "tremo"+u"\u00E7"+"os-jardim",
"tremo"+u"\u00E7"+"os-minas", "tr"+u"\u00E9"+"pano-coroa", "tr"+u"\u00E9"+"panos-coroa", "trevo-"+u"\u00E1"+"gua",
"trevo-carretilha", "trevo-carvalho", "trevo-cheiro", "trevo-seara",
"trevo-serra", "trevos-"+u"\u00E1"+"gua", "trevos-carretilha", "trevos-carvalho",
"trevos-cheiro", "trevos-seara", "trevos-serra", "tricolino-chifre",
"tricolino-crista", "tricolinos-chifre", "tricolinos-crista", "trigo-israel",
"trigo-milagre", "trigo-perdiz", "trigo-prioste", "trigo-ver"+u"\u00E3"+"o",
"trigos-israel", "trigos-milagre", "trigos-perdiz", "trigos-prioste",
"trigos-ver"+u"\u00E3"+"o", "trincais-pau", "trincal-pau", "tromba-elefante",
"trombas-elefante", "trombeta-arauto", "trombetas-arauto", "truta-lago",
"trutas-lago", "tubar"+u"\u00E3"+"o-focinho", "tubar"+u"\u00F5"+"es-focinho", "t"+u"\u00FA"+"bera-inverno",
"t"+u"\u00FA"+"bera-ver"+u"\u00E3"+"o", "t"+u"\u00FA"+"beras-inverno", "t"+u"\u00FA"+"beras-ver"+u"\u00E3"+"o", "tucano-cinta",
"tucanos-cinta", "tucum-carna"+u"\u00FA"+"ba", "tucum-redes", "tucuns-carna"+u"\u00FA"+"ba",
"tucuns-redes", "tuia-areia", "tuias-areia", "tulipa-jardim",
"tulipas-jardim", "uapu"+u"\u00E7"+u"\u00E1"+"-coleira", "uapu"+u"\u00E7"+u"\u00E1"+"s-coleira", "ub"+u"\u00E1"+"-facho",
"ub"+u"\u00E1"+"s-facho", "uirapuru-bando", "uirapurus-bando", "umbarana-abelha",
"umbaranas-abelha", "umiri-cheiro", "umiris-cheiro", "untu"+u"\u00E9"+"-ob"+u"\u00F3",
"untu"+u"\u00E9"+"s-ob"+u"\u00F3", "uquet"+u"\u00E9"+"-"+u"\u00E1"+"gua", "uquet"+u"\u00E9"+"-ob"+u"\u00F3", "uquet"+u"\u00E9"+"s-"+u"\u00E1"+"gua",
"uquet"+u"\u00E9"+"s-ob"+u"\u00F3", "uva-c"+u"\u00E3"+"o", "uva-espinho", "uva-facho",
"uva-galo", "uva-gentio", "uva-ob"+u"\u00F3", "uva-praia",
"uva-rei", "uva-urso", "uvas-c"+u"\u00E3"+"o", "uvas-cheiro",
"uvas-espinho", "uvas-facho", "uvas-galo", "uvas-gentio",
"uvas-ob"+u"\u00F3", "uvas-praia", "uvas-rato", "uvas-rei",
"uvas-urso", "uxi-morcego", "uxis-morcego", "vale-arinto",
"vales-arinto", "vara-oiro", "vara-ouro", "varas-canoa",
"varas-foguete", "varas-oiro", "varas-ouro", "vassoira-bruxa",
"vassoira-feiticeira", "vassoiras-bruxa", "vassoiras-feiticeira", "vassoiras-ferro",
"vassourinha-bot"+u"\u00E3"+"o", "vassourinha-rel"+u"\u00F3"+"gio", "vassourinha-varrer", "vassourinhas-bot"+u"\u00E3"+"o",
"vassourinhas-rel"+u"\u00F3"+"gio", "vassourinhas-varrer", "veado-virg"+u"\u00ED"+"nia", "veados-virg"+u"\u00ED"+"nia",
"velame-cheiro", "velames-cheiro", "veludo-penca", "veludos-penca",
"veneno-porco", "venenos-porco", "ver"+u"\u00E7"+"a-c"+u"\u00E3"+"o", "ver"+u"\u00E7"+"as-c"+u"\u00E3"+"o",
"verga-jabuti", "vergas-jabuti", "vergonha-estudante", "vergonhas-estudante",
"vermelhinha-galho", "vermelhinhas-galho", "vespa-cobra", "vespa-rodeio",
"vespas-cobra", "vespas-rodeio", "videira-enforcado", "videiras-enforcado",
"vidoeiro-papel", "vidoeiros-papel", "vinh"+u"\u00E1"+"tico-espinho", "vinh"+u"\u00E1"+"ticos-espinho",
"violeta-dama", "violeta-parma", "violetas-dama", "violetas-parma",
"voador-pedra", "voadores-pedra", "xex"+u"\u00E9"+"u-bananeira"])





#PREFIXO CONTRA - exceto A, H, R, S
pfxContrajto = set(["baixista", "baixistas", "baixo", "baixos",
"balan"+u"\u00E7"+"ar", "balancear", "baluarte", "baluartes",
"bandar", "bandeada", "bandeadas", "bandeado",
"bandeados", "bandear", "bandismo", "bandismos",
"bandista", "bandistas", "bando", "bandos",
"bater", "bordo", "bordos", "bracear",
"cadaste", "cadastes", "caibro", "caibros",
"caixilho", "caixilhos", "cambiar", "c"+u"\u00E2"+"mbio",
"c"+u"\u00E2"+"mbios", "campanha", "campanhas", "campo",
"campos", "canto", "cantos", "capa",
"capas", "carril", "carris", "cava",
"cavas", "c"+u"\u00E9"+"dula", "c"+u"\u00E9"+"dulas", "cena", "chapa",
"chapas", "chaveta", "chavetas", "cheque",
"cheques", "choque", "choques", "c"+u"\u00ED"+"clica", "c"+u"\u00ED"+"clicas",
"c"+u"\u00ED"+"clico", "c"+u"\u00ED"+"clicos", "ciclo", "ciclos", "cifra",
"cifras", "colar", "concep"+u"\u00E7"+u"\u00E3"+"o", "concep"+u"\u00E7"+u"\u00F5"+"es",
"corrente", "correntes", "costa", "costas",
"cr"+u"\u00ED"+"tica", "cr"+u"\u00ED"+"ticas", "cultura", "culturais",
"cultural", "culturas", "cunhar", "curva",
"curvas", "dan"+u"\u00E7"+"a", "dan"+u"\u00E7"+"ar", "declara"+u"\u00E7"+u"\u00E3"+"o",
"declara"+u"\u00E7"+u"\u00F5"+"es",
"declarar", "di"+u"\u00E7"+u"\u00E3"+"o", "di"+u"\u00E7"+u"\u00F5"+"es", "dique",
"diques",  "disse", "distinguir", "ditar", "dit"+u"\u00E1"+"veis",
"dit"+u"\u00E1"+"vel", "dita", "ditas", "dito", "ditos", "diz", "dizem", "dizente", "dizentes", "dizer",
"documento", "documentos", "domina"+u"\u00E7"+u"\u00E3"+"o", "domina"+u"\u00E7"+u"\u00F5"+"es",
"domin"+u"\u00E2"+"ncia", "domin"+u"\u00E2"+"ncias", "dom"+u"\u00ED"+"nio", "dom"+u"\u00ED"+"nios",
"dormentes", "economia", "economias", "econ"+u"\u00F4"+"mica",
"econ"+u"\u00F4"+"micas", "econ"+u"\u00F4"+"mico", "econ"+u"\u00F4"+"micos", u"\u00E9"+"dita",
u"\u00E9"+"ditas", u"\u00E9"+"dito", u"\u00E9"+"ditos", "emboscada",
"emboscadas", "emergente", "emergentes", "ente",
"entes", "erva", "ervas", "escarpa",
"escarpas", "escota", "escotas", "escritura",
"escrituras", "espi"+u"\u00E3"+"o", "espionagem", "espionagens", "estai",
"estais", "estimula"+u"\u00E7"+u"\u00E3"+"o", "estimula"+u"\u00E7"+u"\u00F5"+"es", "estimulante",
"estimulantes", "estimular", "estimulismo", "estimulismos",
"est"+u"\u00ED"+"mulo", "est"+u"\u00ED"+"mulos", "exemplo", "exemplos",
"explos"+u"\u00E3"+"o", "explos"+u"\u00F5"+"es", "fa"+u"\u00E7"+u"\u00E3"+"o", "fa"+u"\u00E7"+u"\u00F5"+"es",
"factuais", "factual", "fagote", "fagotes",
"faixa", "faixas", "fator", "fatores", "fazer",
"f"+u"\u00E9", "fecho", "fechos", "fei"+u"\u00E7"+u"\u00E3"+"o",
"fei"+u"\u00E7"+u"\u00F5"+"es", "feito", "feitos", "feiti"+u"\u00E7"+"o",
"feiti"+u"\u00E7"+"os", "ferro", "ferros", "f"+u"\u00E9"+"s", "figura", "figuras",
"fila", "filas", "fil"+u"\u00E9", "fileira",
"fileiras", "fil"+u"\u00E9"+"s", "fio", "fios",
"flanco", "flancos", "flange", "flanges", "fluxo", "fluxos",
"folheado", "folheados", "for"+u"\u00E7"+"a", "for"+u"\u00E7"+"as", "forte", "fortes",
"fossa", "fossas", "fosso", "fossos",
"fuga", "fugas", "fundo", "fundos",
"garantia", "garantias", "golpe", "golpear", "golpes",
"gosto", "gostos", "guerrilha", "guerrilhas",
"indica"+u"\u00E7"+u"\u00E3"+"o", "indica"+u"\u00E7"+u"\u00F5"+"es",
"indicada", "indicadas", "indicado", "indicados",
"indicar", "informa"+u"\u00E7"+u"\u00E3"+"o", "informa"+u"\u00E7"+u"\u00F5"+"es", "informar",
"insurg"+u"\u00EA"+"ncia", "insurg"+u"\u00EA"+"ncias", "insurgente", "insurgentes",
"intelig"+u"\u00EA"+"ncia", "intelig"+u"\u00EA"+"ncias", "interpela"+u"\u00E7"+u"\u00E3"+"o",
"interpela"+u"\u00E7"+u"\u00F5"+"es",
"interrogat"+u"\u00F3"+"rio", "interrogat"+u"\u00F3"+"rios", "jampear", "laterais",
"lateral", "liga", "ligas", "luz",
"luzes", "malhar", "mandar", "manga",
"mangas", "manifesta"+u"\u00E7"+u"\u00E3"+"o", "manifesta"+u"\u00E7"+u"\u00F5"+"es", "manifestante",
"manifestantes", "manifestar", "manobra", "manobrar",
"manobras", "m"+u"\u00E3"+"o", "m"+u"\u00E3"+"os", "marca"+u"\u00E7"+u"\u00E3"+"o",
"marca"+u"\u00E7"+u"\u00F5"+"es", "marcar", "marcha", "marchar", "marchas", "marco",  "marcos",
"mar"+u"\u00E9", "mar"+u"\u00E9"+"s", "margem", "margens",
"medida", "medidas", "mensagem", "mensagens", "mestra", "mestras", "mestre", "mestres",
"metade", "metades", "mezena", "mezenas",
"minar", "minutar", "miss"+u"\u00E3"+"o", "miss"+u"\u00F5"+"es",
"moldagem", "moldagens", "moldar", "molde", "moldes", "mon"+u"\u00E7"+u"\u00E3"+"o",
"mon"+u"\u00E7"+u"\u00F5"+"es", "motivo", "motivos", "movimento",
"movimentos", "muralha", "muralhas", "murar",
"natura", "naturais", "naturais", "natural",
"naturas", "nivelar", "nota", "notas",
"ofensiva", "ofensivas", "ofensivo", "ofensivos",
"oferta", "ofertas", "oitava", "oitavas",
"ordem", "ordena"+u"\u00E7"+u"\u00E3"+"o", "ordena"+u"\u00E7"+u"\u00F5"+"es", "ordenar",
"ordens", "pancada", "pancadas", "parente",
"parentes", "parte", "partes", "partida",
"partidas", "passo", "passos", "p"+u"\u00E9",
"pe"+u"\u00E7"+"onha", "pe"+u"\u00E7"+"onhas", "pedais", "pedal",
"pedalar", "pelo", "pelos", "p"+u"\u00E9"+"s",
"pesar", "peso", "pesos", "pilastra", "pilastras",
"pin"+u"\u00E1"+"zio", "pin"+u"\u00E1"+"zios", "piso", "pisos",
"placado", "placar", "placares", "poder",
"poderes", "pontar", "pontear", "pontista",
"pontistas", "ponto", "pontos", "pontualmente", "por",
"porca", "porcas", "posi"+u"\u00E7"+u"\u00E3"+"o", "posi"+u"\u00E7"+u"\u00F5"+"es",  "posto",  "postos",
"prepara"+u"\u00E7"+u"\u00E3"+"o", "prepara"+u"\u00E7"+u"\u00F5"+"es", "press"+u"\u00E3"+"o", "press"+u"\u00F5"+"es",
"presta"+u"\u00E7"+u"\u00E3"+"o", "presta"+u"\u00E7"+u"\u00F5"+"es", "producente", "producentes",
"produtiva", "produtivas", "produtivo", "produtivos",
"produzir", "programa", "programas", "projeto",
"projetos", "propaganda", "propagandas", "propor", "proposta",  "propostas",
"protestar", "protestos", "prova", "provar", "provas", "pun"+u"\u00E7"+"oar",
"punho", "punhos", "quarteado", "quarteados",
"quart"+u"\u00E9"+"is", "quartel", "quilha", "quilhas",
"telar", "tempo", "tempos", "tenor",
"tenores", "terrorismo", "terrorismos", "terrorista",
"terroristas", "testemunhar", "tirar", "torpedeiro",
"torpedeiros", "transfer"+u"\u00EA"+"ncia", "transfer"+u"\u00EA"+"ncias",
"tura", "turas", "valar", "vale", "vales",
"vapor", "vapores", "variante", "variantes",
"veneno", "venenos", "ventar", "vento", "ventor", "ventos",
"ventora", "ventoras", "ventores", "ventos",
"verdade", "verdades", "verga", "vergas",
"vers"+u"\u00E3"+"o", "vers"+u"\u00F5"+"es", "verter", "vinda",
"vindas", "vir", "volta", "voltas",
"voto", "votos"])

#PREFIXO CONTRA - A, H
pfxContraah = set(["abertura", "aberturas", "abside", "absides",
"acusa"+u"\u00E7"+u"\u00E3"+"o", "acusa"+u"\u00E7"+u"\u00F5"+"es", "acusar", "acus"+u"\u00E1"+"veis",
"acus"+u"\u00E1"+"vel", "aleia", "aleias", "alisado",
"alisados", "al"+u"\u00ED"+"sio", "al"+u"\u00ED"+"sios", "almeida",
"almeidas", "almirantado", "almirantados", "almirante",
"almirantes", "alquebrar", "amantilho", "amura",
"amuras", "an"+u"\u00E1"+"lise", "an"+u"\u00E1"+"lises", "antena",
"antenas", "anuncia"+u"\u00E7"+u"\u00E3"+"o", "anuncia"+u"\u00E7"+u"\u00F5"+"es", "anunciar",
"anunci"+u"\u00E1"+"veis", "anunci"+u"\u00E1"+"vel", "an"+u"\u00FA"+"ncio", "an"+u"\u00FA"+"ncios",
"apela"+u"\u00E7"+u"\u00E3"+"o", "apela"+u"\u00E7"+u"\u00F5"+"es", "apelar", "aproche",
"aproches", "arcada", "arcadas", "arcadura",
"arcaduras", "arco", "arcos", "argumenta"+u"\u00E7"+u"\u00E3"+"o",
"argumenta"+u"\u00E7"+u"\u00F5"+"es", "argumentais", "argumental", "argumentar",
"argumentativa", "argumentativas", "argumentativo", "argumentativos",
"argumento", "argumentos", "arminhado", "arminho",
"arminhos", "arrazoar", "arrazo"+u"\u00E1"+"veis", "arrazo"+u"\u00E1"+"vel",
"arrestar", "arrest"+u"\u00E1"+"veis", "arrest"+u"\u00E1"+"vel", "arresto",
"arrestos", "asa", "asas", "assalto",
"assaltos", "assembleia", "assembleias", "assinar",
"assin"+u"\u00E1"+"veis", "assin"+u"\u00E1"+"vel", "ast"+u"\u00FA"+"cia", "ast"+u"\u00FA"+"cias",
"atacante", "atacantes", "atacar", "atac"+u"\u00E1"+"veis",
"atac"+u"\u00E1"+"vel", "ataque", "ataques", "avisar",
"avis"+u"\u00E1"+"veis", "avis"+u"\u00E1"+"vel", "aviso", "avisos",
"habituais", "habitual", "habitualidade", "habitualidades",
"harmonia", "harmonias", "harm"+u"\u00F4"+"nico", "harm"+u"\u00F4"+"nicos",
"haste", "hastes", "homon"+u"\u00ED"+"mia", "homon"+u"\u00ED"+"mias",
"homon"+u"\u00ED"+"mico", "homon"+u"\u00ED"+"micos", "hom"+u"\u00F4"+"nimo", "hom"+u"\u00F4"+"nimos",
"homonomia", "homonomias", "homon"+u"\u00F4"+"mico", "homon"+u"\u00F4"+"micos",
"hom"+u"\u00F4"+"nomo", "hom"+u"\u00F4"+"nomos"])


#PREFIXO CONTRA -  R, S
pfxContrars = set(["raciocinar", "raciocin"+u"\u00E1"+"vel",
"racioc"+u"\u00ED"+"nio", "racioc"+u"\u00ED"+"nios",
"racionais", "racional", "rainha", "rainhas",
"rampa", "rampante", "rampantes", "rampas",
"rancho", "ranchos", "rapante", "rapantes",
"raz"+u"\u00E3"+"o", "raz"+u"\u00F5"+"es", "rea"+u"\u00E7"+u"\u00E3"+"o", "rea"+u"\u00E7"+u"\u00F5"+"es",
"reagente", "reagentes", "reagir", "rebite",
"rebites", "recibo", "recibos", "reforma", "reformar", "reformas",
"reform"+u"\u00E1"+"vel", "reformismo", "reformismos", "reformista",
"reformistas", "regra", "regrar", "regras", "regr"+u"\u00E1"+"vel", "regular",
"rel"+u"\u00F3"+"gio", "rel"+u"\u00F3"+"gios", "reparo", "reparos",
"r"+u"\u00E9"+"plica", "replicar", "r"+u"\u00E9"+"plicas", "replic"+u"\u00E1"+"vel",
"reptar", "respondente", "respondentes", "responder",
"respond"+u"\u00ED"+"veis", "respond"+u"\u00ED"+"vel", "resposta", "respostas",
"ret"+u"\u00E1"+"bulo", "ret"+u"\u00E1"+"bulos", "revoltar", "revolt"+u"\u00E1"+"vel",
"revolu"+u"\u00E7"+u"\u00E3"+"o", "revolucionar", "revolucion"+u"\u00E1"+"ria", "revolucion"+u"\u00E1"+"rias",
"revolucion"+u"\u00E1"+"rio", "revolucion"+u"\u00E1"+"rios", "revolu"+u"\u00E7"+u"\u00F5"+"es", "roda",
"rodas", "rolda", "roldas", "romper",
"romp"+u"\u00ED"+"veis", "romp"+u"\u00ED"+"vel", "ronda", "rondas",
"roquete", "roquetes", "rota"+u"\u00E7"+u"\u00E3"+"o", "rota"+u"\u00E7"+u"\u00F5"+"es",
"rotura", "roturas", "ruptura", "rupturas",
"rutura", "ruturas", "safra", "safras",
"salvar", "sauda"+u"\u00E7"+u"\u00E3"+"o", "sauda"+u"\u00E7"+u"\u00F5"+"es", "saudar",
"saud"+u"\u00E1"+"vel", "segurar", "segur"+u"\u00E1"+"vel", "seguros",
"selar", "selo", "selos", "senha",
"senhas", "senso", "sensos", "significa"+u"\u00E7"+u"\u00E3"+"o",
"significa"+u"\u00E7"+u"\u00F5"+"es", "sinais", "sinal", "sistema",
"sistemas", "sistem"+u"\u00E1"+"tica", "sistem"+u"\u00E1"+"ticas", "sistem"+u"\u00E1"+"tico",
"sistem"+u"\u00E1"+"ticos", "sist"+u"\u00EA"+"mica", "sist"+u"\u00EA"+"micas", "sist"+u"\u00EA"+"mico",
"sist"+u"\u00EA"+"micos", "soca", "socas", "sortida",
"sortidas", "sortido", "sortidos", "sugerir",
"suger"+u"\u00ED"+"veis", "suger"+u"\u00ED"+"vel", "sugest"+u"\u00E3"+"o", "sugestiva",
"sugestivas", "sugestivo", "sugestivos", "sugest"+u"\u00F5"+"es",
"sujeito", "sujeitos"])


# ccvbFsvo = Concordancia verbal no futuro do subjuntivo  
ccvbFsvo = {"antepor": "antepuser",
"autossatisfazer": "autossatisfizer",
"bendizer": "bendisser",
"benfazer": "benfizer",
"caber": "couber",
"compor": "compuser",
"condizer": "condisser",
"conter": "contiver",
"contradizer": "contradisser",
"contrapor": "contrapuser",
"depor": "depuser",
"descompor": "descompuser",
"desdizer": "desdisser",
"desfazer": "desfizer",
"despor": "despuser",
"dessatisfazer": "dessatisfizer",
"dispor": "dispuser",
"dizer": "disser",
"entredizer": "entredisser",
"entrefazer": "entrefizer",
"entrepor": "entrepuser",
"expor": "expuser",
"fazer": "fizer",
"impor": "impuser",
"impor": "impuser",
"indispor": "indispuser",
"interpor": "interpuser",
"justapor": "justapuser",
"liquefazer": "liquefizer",
"maldizer": "maldisser",
"malfazer": "malfizer",
"opor": "opuser",
"perfazer": "perfizer",
"poder": "puder",
"p"+u"\u00F4"+"r": "puser",
"predispor": "predispuser",
"pressupor": "pressupuser",
"propor": "propuser",
"querer": "quiser",
"recompor": "recompuser",
"redispor": "redispuser",
"redizer": "redisser",
"reexpor": "reexpuser",
"refazer": "refizer",
"reimpor": "reimpuser",
"reimpor": "reimpuser",
"repor": "repuser",
"repropor": "repropuser",
"reter": "retiver",
"saber": "souber",
"satisfazer": "satisfizer",
"sobrepor": "sobrepuser",
"subpor": "subpuser",
"superpor": "superpuser",
"supor": "supuser",
"ter": "tiver",
"torrefazer": "torrefizer",
"transpor": "transpuser"}

#Concordancia verbal no futuro do subjuntivo  
aavbFsvo = set(["antepor", "autossatisfazer", "bendizer", "benfazer", "caber", "compor",
"condizer", "conter", "contradizer", "contrapor", "depor", "descompor",
"desdizer", "desfazer", "despor", "dessatisfazer", "dispor", "dizer",
"entredizer", "entrefazer", "entrepor", "expor", "fazer", "impor",
"impor", "indispor", "interpor", "justapor", "liquefazer", "maldizer",
"malfazer", "opor", "perfazer", "p"+u"\u00F4"+"r", "predispor", "pressupor",
"propor", "querer", "recompor", "redispor", "redizer", "reexpor",
"refazer", "reimpor", "reimpor", "repor", "propor", "reter", "saber",
"satisfazer", "sobrepor", "subpor", "superpor", "supor",
"ter", "torrefazer", "transpor"])

# VERBOS DE LIGACAO

# Verbos de ligacao terceira pessoa do SINGULAR
aaVblgtps = {"continua": "continua",
"continuais": "continua",
"continuam": "continua",
"continuamos": "continua",
"continuar": "continuar",
"continuara": "continuara",
"continuara": "continuara",
"continuar"+u"\u00E1": "continuar"+u"\u00E1",
"continuaram": "continuou",
"continu"+u"\u00E1"+"ramos": "continuara",
"continuar"+u"\u00E3"+"o": "continuar"+u"\u00E1",
"continuaras": "continuara",
"continuar"+u"\u00E1"+"s": "continuar"+u"\u00E1",
"continuardes": "continuar",
"continuarei": "continuar"+u"\u00E1",
"continuareis": "continuar"+u"\u00E1",
"continu"+u"\u00E1"+"reis": "continuara",
"continuarem": "continuar",
"continuaremos": "continuar"+u"\u00E1",
"continuares": "continuar",
"continuaria": "continuaria",
"continuariam": "continuaria",
"continuar"+u"\u00ED"+"amos": "continuaria",
"continuarias": "continuaria",
"continuar"+u"\u00ED"+"eis": "continuaria",
"continuarmos": "continuar",
"continuas": "continua",
"continuasse": "continuasse",
"continu"+u"\u00E1"+"sseis": "continuasse",
"continuassem": "continuasse",
"continu"+u"\u00E1"+"ssemos": "continuasse",
"continuasses": "continuasse",
"continuaste": "continuou",
"continuastes": "continuou",
"continuava": "continuava",
"continuavam": "continuava",
"continu"+u"\u00E1"+"vamos": "continuava",
"continuavas": "continuava",
"continu"+u"\u00E1"+"veis": "continuava",
"continue": "continue",
"continuei": "continuou",
"continueis": "continue",
"continuem": "continue",
"continuemos": "continue",
"continues": "continue",
"continuo": "continua",
"continuou": "continuou",
u"\u00E9": u"\u00E9",
"era": "era",
"eram": "era",
u"\u00E9"+"ramos": "era",
"eras": "era",
u"\u00E9"+"reis": "era",
u"\u00E9"+"s": u"\u00E9",
"est"+u"\u00E1": "est"+u"\u00E1",
"estais": "est"+u"\u00E1",
"estamos": "est"+u"\u00E1",
"est"+u"\u00E3"+"o": "est"+u"\u00E1",
"estar"+u"\u00E1": "estar"+u"\u00E1",
"estar"+u"\u00E3"+"o": "estar"+u"\u00E1",
"estar"+u"\u00E1"+"s": "estar"+u"\u00E1",
"estarei": "estar"+u"\u00E1",
"estareis": "estar"+u"\u00E1",
"estaremos": "estar"+u"\u00E1",
"estaria": "estaria",
"estariam": "estaria",
"estar"+u"\u00ED"+"amos": "estaria",
"estarias": "estaria",
"estar"+u"\u00ED"+"eis": "estaria",
"est"+u"\u00E1"+"s": "est"+u"\u00E1",
"estava": "estava",
"estavam": "estava",
"est"+u"\u00E1"+"vamos": "estava",
"estavas": "estava",
"est"+u"\u00E1"+"veis": "estava",
"esteja": "esteja",
"estejais": "esteja",
"estejam": "esteja",
"estejamos": "esteja",
"estejas": "esteja",
"esteve": "esteve",
"estive": "esteve",
"estivemos": "esteve",
"estiver": "estiver",
"estivera": "estivera",
"estivera": "estivera",
"estiveram": "esteve",
"estiv"+u"\u00E9"+"ramos": "estivera",
"estiveras": "estivera",
"estiverdes": "estiver",
"estiv"+u"\u00E9"+"reis": "estivera",
"estiverem": "estiver",
"estiveres": "estiver",
"estivermos": "estiver",
"estivesse": "estivesse",
"estiv"+u"\u00E9"+"sseis": "estivesse",
"estivessem": "estivesse",
"estiv"+u"\u00E9"+"ssemos": "estivesse",
"estivesses": "estivesse",
"estiveste": "esteve",
"estivestes": "esteve",
"estou": "est"+u"\u00E1",
"fica": "fica",
"ficais": "fica",
"ficam": "fica",
"ficamos": "ficou",
"ficar": "ficar",
"ficara": "ficara",
"ficar"+u"\u00E1": "ficar"+u"\u00E1",
"ficaram": "ficou",
"fic"+u"\u00E1"+"ramos": "ficara",
"ficar"+u"\u00E3"+"o": "ficar"+u"\u00E1",
"ficaras": "ficara",
"ficar"+u"\u00E1"+"s": "ficar"+u"\u00E1",
"ficardes": "ficar",
"ficarei": "ficar"+u"\u00E1",
"ficareis": "ficar"+u"\u00E1",
"fic"+u"\u00E1"+"reis": "ficara",
"ficarem": "ficar",
"ficaremos": "ficar"+u"\u00E1",
"ficares": "ficar",
"ficaria": "ficaria",
"ficariam": "ficaria",
"ficar"+u"\u00ED"+"amos": "ficaria",
"ficarias": "ficaria",
"ficar"+u"\u00ED"+"eis": "ficaria",
"ficarmos": "ficar",
"ficas": "fica",
"ficasse": "ficasse",
"fic"+u"\u00E1"+"sseis": "ficasse",
"ficassem": "ficasse",
"fic"+u"\u00E1"+"ssemos": "ficasse",
"ficasses": "ficasse",
"ficaste": "ficou",
"ficastes": "ficou",
"ficava": "ficava",
"ficavam": "ficava",
"fic"+u"\u00E1"+"vamos": "ficava",
"ficavas": "ficava",
"fic"+u"\u00E1"+"veis": "ficava",
"fico": "fica",
"ficou": "ficou",
"fique": "fique",
"fiquei": "ficou",
"fiqueis": "fique",
"fiquem": "fique",
"fiquemos": "fique",
"fiques": "fique",
"foi": "foi",
"fomos": "foi",
"for": "for",
"fora": "fora",
"foram": "foi",
"f"+u"\u00F4"+"ramos": "fora",
"foras": "fora",
"fordes": "for",
"f"+u"\u00F4"+"reis": "fora",
"forem": "for",
"fores": "for",
"formos": "for",
"fosse": "fosse",
"f"+u"\u00F4"+"sseis": "fosse",
"fossem": "fosse",
"f"+u"\u00F4"+"ssemos": "fosse",
"fosses": "fosse",
"foste": "foi",
"fostes": "foi",
"fui": "foi",
"pare"+u"\u00E7"+"a": "pare"+u"\u00E7"+"a",
"pare"+u"\u00E7"+"ais": "pare"+u"\u00E7"+"a",
"pare"+u"\u00E7"+"am": "pare"+u"\u00E7"+"a",
"pare"+u"\u00E7"+"amos": "pare"+u"\u00E7"+"a",
"pare"+u"\u00E7"+"as": "pare"+u"\u00E7"+"a",
"parece": "parece",
"pareceis": "parece",
"parecem": "parece",
"parecemos": "pareceu",
"parecer": "parecer",
"parecera": "parecera",
"parecer"+u"\u00E1": "parecer"+u"\u00E1",
"pareceram": "pareceu",
"parec"+u"\u00EA"+"ramos": "parecera",
"parecer"+u"\u00E3"+"o": "parecer"+u"\u00E1",
"pareceras": "parecera",
"parecer"+u"\u00E1"+"s": "parecer"+u"\u00E1",
"parecerdes": "parecer",
"parecerei": "parecer"+u"\u00E1",
"parecereis": "parecer"+u"\u00E1",
"parec"+u"\u00EA"+"reis": "parecera",
"parecerem": "parecer",
"pareceremos": "parecer"+u"\u00E1",
"pareceres": "parecer",
"pareceria": "pareceria",
"pareceriam": "pareceria",
"parecer"+u"\u00ED"+"amos": "parec